"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RenderQueue = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path_1 = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_ecs_patterns_1 = require("@aws-cdk/aws-ecs-patterns");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const core_2 = require("../../core");
const deployment_instance_1 = require("../../core/lib/deployment-instance");
const runtime_info_1 = require("../../core/lib/runtime-info");
const rq_connection_1 = require("./rq-connection");
const secrets_management_1 = require("./secrets-management");
const version_1 = require("./version");
const wait_for_stable_service_1 = require("./wait-for-stable-service");
/**
 * Base class for Render Queue providers
 */
class RenderQueueBase extends core_1.Construct {
}
/**
 * The RenderQueue construct deploys an Elastic Container Service (ECS) service that serves Deadline's REST HTTP API to Deadline Clients.
 *
 * Most Deadline clients will connect to a Deadline render farm via the the RenderQueue. The API provides Deadline
 * clients access to Deadline's database and repository file-system in a way that is secure, performant, and scalable.
 *
 * ![architecture diagram](/diagrams/deadline/RenderQueue.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An Amazon Elastic Container Service (ECS) cluster.
 * - An AWS EC2 auto-scaling group that provides the instances that host the ECS service.
 * - An ECS service with a task definition that deploys the Deadline Remote Connetion Server (RCS) in a container.
 * - A Amazon CloudWatch log group for streaming logs from the Deadline RCS.
 * - An application load balancer, listener and target group that balance incoming traffic among the RCS containers.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - Care must be taken to secure what can connect to the RenderQueue. The RenderQueue does not authenticate API
 *    requests made against it. You must limit access to the RenderQueue endpoint to only trusted hosts. Those hosts
 *    should be governed carefully, as malicious software could use the API to remotely execute code across the entire render farm.
 * - The RenderQueue can be deployed with network encryption through Transport Layer Security (TLS) or without it. Unencrypted
 *    network communications can be eavesdropped upon or modified in transit. We strongly recommend deploying the RenderQueue
 *    with TLS enabled in production environments and it is configured to be on by default.
 *
 * @stability stable
 */
class RenderQueue extends RenderQueueBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0;
        super(scope, id);
        this.props = props;
        /**
         * Whether SEP policies have been added
         */
        this.haveAddedSEPPolicies = false;
        /**
         * Whether Resource Tracker policies have been added
         */
        this.haveAddedResourceTrackerPolicies = false;
        this.repository = props.repository;
        this.renderQueueSize = (_b = props === null || props === void 0 ? void 0 : props.renderQueueSize) !== null && _b !== void 0 ? _b : { min: 1, max: 1 };
        if (props.version.isLessThan(RenderQueue.MINIMUM_LOAD_BALANCING_VERSION)) {
            // Deadline versions earlier than 10.1.10 do not support horizontal scaling behind a load-balancer, so we limit to at most one instance
            if (((_c = this.renderQueueSize.min) !== null && _c !== void 0 ? _c : 0) > 1) {
                throw new Error(`renderQueueSize.min for Deadline version less than ${RenderQueue.MINIMUM_LOAD_BALANCING_VERSION.toString()} cannot be greater than 1 - got ${this.renderQueueSize.min}`);
            }
            if (((_d = this.renderQueueSize.desired) !== null && _d !== void 0 ? _d : 0) > 1) {
                throw new Error(`renderQueueSize.desired for Deadline version less than ${RenderQueue.MINIMUM_LOAD_BALANCING_VERSION.toString()} cannot be greater than 1 - got ${this.renderQueueSize.desired}`);
            }
            if (((_e = this.renderQueueSize.max) !== null && _e !== void 0 ? _e : 0) > 1) {
                throw new Error(`renderQueueSize.max for Deadline version less than ${RenderQueue.MINIMUM_LOAD_BALANCING_VERSION.toString()} cannot be greater than 1 - got ${this.renderQueueSize.max}`);
            }
        }
        this.version = props === null || props === void 0 ? void 0 : props.version;
        const externalProtocol = ((_g = (_f = props.trafficEncryption) === null || _f === void 0 ? void 0 : _f.externalTLS) === null || _g === void 0 ? void 0 : _g.enabled) === false ? aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
        let loadBalancerFQDN;
        let domainZone;
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            const tlsInfo = this.getOrCreateTlsInfo(props);
            this.certChain = tlsInfo.certChain;
            this.clientCert = tlsInfo.serverCert;
            loadBalancerFQDN = tlsInfo.fullyQualifiedDomainName;
            domainZone = tlsInfo.domainZone;
        }
        else {
            if (props.hostname) {
                loadBalancerFQDN = this.generateFullyQualifiedDomainName(props.hostname.zone, props.hostname.hostname);
                domainZone = props.hostname.zone;
            }
        }
        this.version = props.version;
        const internalProtocol = (_j = (_h = props.trafficEncryption) === null || _h === void 0 ? void 0 : _h.internalProtocol) !== null && _j !== void 0 ? _j : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
        this.cluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            vpc: props.vpc,
        });
        const minCapacity = (_l = (_k = props.renderQueueSize) === null || _k === void 0 ? void 0 : _k.min) !== null && _l !== void 0 ? _l : 1;
        if (minCapacity < 1) {
            throw new Error(`renderQueueSize.min capacity must be at least 1: got ${minCapacity}`);
        }
        const maxCapacity = (_m = this.renderQueueSize.max) !== null && _m !== void 0 ? _m : (_o = this.renderQueueSize) === null || _o === void 0 ? void 0 : _o.desired;
        if (((_p = this.renderQueueSize) === null || _p === void 0 ? void 0 : _p.desired) && maxCapacity && ((_q = this.renderQueueSize) === null || _q === void 0 ? void 0 : _q.desired) > maxCapacity) {
            throw new Error(`renderQueueSize.desired capacity cannot be more than ${maxCapacity}: got ${this.renderQueueSize.desired}`);
        }
        this.asg = this.cluster.addCapacity('RCS Capacity', {
            vpcSubnets: (_r = props.vpcSubnets) !== null && _r !== void 0 ? _r : RenderQueue.DEFAULT_VPC_SUBNETS_OTHER,
            instanceType: (_s = props.instanceType) !== null && _s !== void 0 ? _s : new aws_ec2_1.InstanceType('c5.large'),
            minCapacity,
            desiredCapacity: (_t = this.renderQueueSize) === null || _t === void 0 ? void 0 : _t.desired,
            maxCapacity,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    // See: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-ami-storage-config.html
                    // We want the volume to be encrypted. The default AMI size is 30-GiB.
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(30, { encrypted: true }),
                }],
            updateType: undefined,
            updatePolicy: aws_autoscaling_1.UpdatePolicy.rollingUpdate(),
            // addCapacity doesn't specifically take a securityGroup, but it passes on its properties to the ASG it creates,
            // so this security group will get applied there
            // @ts-ignore
            securityGroup: (_u = props.securityGroups) === null || _u === void 0 ? void 0 : _u.backend,
        });
        this.backendConnections = this.asg.connections;
        /**
         * The ECS-optimized AMI that is defaulted to when adding capacity to a cluster does not include the awscli or unzip
         * packages as is the case with the standard Amazon Linux AMI. These are required by RFDK scripts to configure the
         * direct connection on the host container instances.
         */
        this.asg.userData.addCommands('yum install -yq awscli unzip');
        if ((_v = props.enableLocalFileCaching) !== null && _v !== void 0 ? _v : false) {
            // Has to be done before any filesystems mount.
            this.enableFilecaching(this.asg);
        }
        const externalPortNumber = RenderQueue.RCS_PROTO_PORTS[externalProtocol];
        const internalPortNumber = RenderQueue.RCS_PROTO_PORTS[internalProtocol];
        this.logGroup = core_2.LogGroupFactory.createOrFetch(this, 'LogGroupWrapper', id, {
            logGroupPrefix: '/renderfarm/',
            ...props.logGroupProps,
        });
        this.logGroup.grantWrite(this.asg);
        if (props.repository.secretsManagementSettings.enabled) {
            const errors = [];
            if (props.version.isLessThan(version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION)) {
                errors.push(`The supplied Deadline version (${props.version.versionString}) does not support Deadline Secrets Management in RFDK. Either upgrade Deadline to the minimum required version (${version_1.Version.MINIMUM_SECRETS_MANAGEMENT_VERSION.versionString}) or disable the feature in the Repository's construct properties.`);
            }
            if (props.repository.secretsManagementSettings.credentials === undefined) {
                errors.push('The Repository does not have Secrets Management credentials');
            }
            if (internalProtocol !== aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
                errors.push('The internal protocol on the Render Queue is not HTTPS.');
            }
            if (externalProtocol !== aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
                errors.push('External TLS on the Render Queue is not enabled.');
            }
            if (errors.length > 0) {
                throw new Error(`Deadline Secrets Management is enabled on the supplied Repository but cannot be enabled on the Render Queue for the following reasons:\n${errors.join('\n')}`);
            }
        }
        const taskDefinition = this.createTaskDefinition({
            image: props.images.remoteConnectionServer,
            portNumber: internalPortNumber,
            protocol: internalProtocol,
            repository: props.repository,
            runAsUser: RenderQueue.RCS_USER,
            secretsManagementOptions: props.repository.secretsManagementSettings.enabled ? {
                credentials: props.repository.secretsManagementSettings.credentials,
                posixUsername: RenderQueue.RCS_USER.username,
            } : undefined,
        });
        this.taskDefinition = taskDefinition;
        // The fully-qualified domain name to use for the ALB
        const loadBalancer = new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, 'LB', {
            vpc: this.cluster.vpc,
            vpcSubnets: (_w = props.vpcSubnetsAlb) !== null && _w !== void 0 ? _w : RenderQueue.DEFAULT_VPC_SUBNETS_ALB,
            internetFacing: false,
            deletionProtection: (_x = props.deletionProtection) !== null && _x !== void 0 ? _x : true,
            securityGroup: (_y = props.securityGroups) === null || _y === void 0 ? void 0 : _y.frontend,
        });
        this.pattern = new aws_ecs_patterns_1.ApplicationLoadBalancedEc2Service(this, 'AlbEc2ServicePattern', {
            certificate: this.clientCert,
            cluster: this.cluster,
            desiredCount: (_z = this.renderQueueSize) === null || _z === void 0 ? void 0 : _z.desired,
            domainZone,
            domainName: loadBalancerFQDN,
            listenerPort: externalPortNumber,
            loadBalancer,
            protocol: externalProtocol,
            taskDefinition,
            // This is required to right-size our host capacity and not have the ECS service block on updates. We set a memory
            // reservation, but no memory limit on the container. This allows the container's memory usage to grow unbounded.
            // We want 1:1 container to container instances to not over-spend, but this comes at the price of down-time during
            // cloudformation updates.
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
            // This is required to ensure that the ALB listener's security group does not allow any ingress by default.
            openListener: false,
        });
        // An explicit dependency is required from the Service to the Client certificate
        // Otherwise cloud formation will try to remove the cert before the ALB using it is disposed.
        if (this.clientCert) {
            this.pattern.node.addDependency(this.clientCert);
        }
        // An explicit dependency is required from the service to the ASG providing its capacity.
        // See: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html
        this.pattern.service.node.addDependency(this.asg);
        this.loadBalancer = this.pattern.loadBalancer;
        // Enabling dropping of invalid HTTP header fields on the load balancer to prevent http smuggling attacks.
        this.loadBalancer.setAttribute('routing.http.drop_invalid_header_fields.enabled', 'true');
        if (props.accessLogs) {
            const accessLogsBucket = props.accessLogs.destinationBucket;
            // Policies are applied according to
            // https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-access-logs.html
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:PutObject'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [`${accessLogsBucket.bucketArn}/*`],
                conditions: {
                    StringEquals: {
                        's3:x-amz-acl': 'bucket-owner-full-control',
                    },
                },
            }));
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:GetBucketAcl'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [accessLogsBucket.bucketArn],
            }));
            this.loadBalancer.logAccessLogs(accessLogsBucket, props.accessLogs.prefix);
        }
        // Ensure tasks are run on separate container instances
        this.pattern.service.addPlacementConstraints(aws_ecs_1.PlacementConstraint.distinctInstances());
        /**
         * Uses an escape-hatch to set the target group protocol to HTTPS. We cannot configure server certificate
         * validation, but at least traffic is encrypted and terminated at the application layer.
         */
        const listener = this.loadBalancer.node.findChild('PublicListener');
        this.listener = listener;
        const targetGroup = listener.node.findChild('ECSGroup');
        const targetGroupResource = targetGroup.node.defaultChild;
        targetGroupResource.protocol = aws_elasticloadbalancingv2_1.ApplicationProtocol[internalProtocol];
        targetGroupResource.port = internalPortNumber;
        this.grantPrincipal = taskDefinition.taskRole;
        this.connections = new aws_ec2_1.Connections({
            defaultPort: aws_ec2_1.Port.tcp(externalPortNumber),
            securityGroups: this.pattern.loadBalancer.connections.securityGroups,
        });
        this.endpoint = new core_2.ConnectableApplicationEndpoint({
            address: loadBalancerFQDN !== null && loadBalancerFQDN !== void 0 ? loadBalancerFQDN : this.pattern.loadBalancer.loadBalancerDnsName,
            port: externalPortNumber,
            connections: this.connections,
            protocol: externalProtocol,
        });
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP) {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttp({
                endpoint: this.endpoint,
            });
        }
        else {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttps({
                endpoint: this.endpoint,
                caCert: this.certChain,
            });
        }
        (_0 = props.repository.secretsManagementSettings.credentials) === null || _0 === void 0 ? void 0 : _0.grantRead(this);
        this.ecsServiceStabilized = new wait_for_stable_service_1.WaitForStableService(this, 'WaitForStableService', {
            service: this.pattern.service,
        });
        this.node.defaultChild = taskDefinition;
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    onValidate() {
        const validationErrors = [];
        // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
        // a resource in another stack is referencing it.
        if (this.version instanceof _1.VersionQuery) {
            const versionStack = core_1.Stack.of(this.version);
            const thisStack = core_1.Stack.of(this);
            if (versionStack != thisStack) {
                validationErrors.push('A VersionQuery can not be supplied from a different stack');
            }
        }
        return validationErrors;
    }
    /**
     * Configures an ECS cluster to be able to connect to a RenderQueue.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientECS(param) {
        param.hosts.forEach(host => this.addChildDependency(host));
        return this.rqConnection.configureClientECS(param);
    }
    /**
     * Configure an Instance/Autoscaling group to connect to a RenderQueue.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientInstance(param) {
        this.addChildDependency(param.host);
        this.rqConnection.configureClientInstance(param);
    }
    /**
     * Configure a rule to automatically register all Deadline Secrets Management identities connecting from a given subnet to a specified role and status.
     *
     * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/secrets-management/deadline-secrets-management.html#identity-management-registration-settings-ref-label
     * for details.
     *
     * All RFDK constructs that require Deadline Secrets Management identity registration call this method internally.
     * End-users of RFDK should not need to use this method unless they have a special need and understand its inner
     * workings.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureSecretsManagementAutoRegistration(props) {
        if (!this.repository.secretsManagementSettings.enabled) {
            // Secrets management is not enabled, so do nothing
            return;
        }
        this.identityRegistrationSettings.addSubnetIdentityRegistrationSetting(props);
    }
    /**
     * Adds AWS Managed Policies to the Render Queue so it is able to control Deadline's Spot Event Plugin.
     *
     * See: https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html for additonal information.
     *
     * @param includeResourceTracker Whether or not the Resource tracker admin policy should also be added (Default: True).
     * @stability stable
     */
    addSEPPolicies(includeResourceTracker = true) {
        if (!this.haveAddedSEPPolicies) {
            const sepPolicy = aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineSpotEventPluginAdminPolicy');
            this.taskDefinition.taskRole.addManagedPolicy(sepPolicy);
            this.haveAddedSEPPolicies = true;
        }
        if (!this.haveAddedResourceTrackerPolicies) {
            if (includeResourceTracker) {
                const rtPolicy = aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AWSThinkboxDeadlineResourceTrackerAdminPolicy');
                this.taskDefinition.taskRole.addManagedPolicy(rtPolicy);
                this.haveAddedResourceTrackerPolicies = true;
            }
        }
    }
    /**
     * Add an ordering dependency to another Construct.
     *
     * All constructs in the child's scope will be deployed after the RenderQueue has been deployed and is ready to recieve traffic.
     *
     * This can be used to ensure that the RenderQueue is fully up and serving queries before a client attempts to connect to it.
     *
     * @param child The child to make dependent upon this RenderQueue.
     * @stability stable
     */
    addChildDependency(child) {
        // Narrowly define the dependencies to reduce the probability of cycles
        // ex: cycles that involve the security group of the RenderQueue & child.
        child.node.addDependency(this.listener);
        child.node.addDependency(this.taskDefinition);
        child.node.addDependency(this.pattern.service);
        child.node.addDependency(this.ecsServiceStabilized);
    }
    /**
     * Adds security groups to the frontend of the Render Queue, which is its load balancer.
     *
     * @param securityGroups The security groups to add.
     * @stability stable
     */
    addFrontendSecurityGroups(...securityGroups) {
        securityGroups.forEach(securityGroup => this.loadBalancer.addSecurityGroup(securityGroup));
    }
    /**
     * Adds security groups to the backend components of the Render Queue, which consists of the AutoScalingGroup for the Deadline RCS.
     *
     * @param securityGroups The security groups to add.
     * @stability stable
     */
    addBackendSecurityGroups(...securityGroups) {
        securityGroups.forEach(securityGroup => this.asg.addSecurityGroup(securityGroup));
    }
    enableFilecaching(asg) {
        const script = core_2.ScriptAsset.fromPathConvention(this, 'FilecachingScript', {
            osType: asg.osType,
            baseName: 'enableCacheFilesd',
            rootDir: path_1.join(__dirname, '..', 'scripts'),
        });
        // A comment in userData to make this easier to test.
        asg.userData.addCommands('# RenderQueue file caching enabled');
        script.executeOn({
            host: asg,
        });
    }
    createTaskDefinition(props) {
        const { image, portNumber, protocol, repository } = props;
        const taskDefinition = new aws_ecs_1.Ec2TaskDefinition(this, 'RCSTask');
        // Mount the repo filesystem to RenderQueue.HOST_REPO_FS_MOUNT_PATH
        const connection = repository.configureClientECS({
            containerInstances: {
                hosts: [this.asg],
            },
            containers: {
                taskDefinition,
            },
        });
        const environment = connection.containerEnvironment;
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            // Generate a self-signed X509 certificate, private key and passphrase for use by the RCS containers.
            // Note: the Application Load Balancer does not validate the certificate in any way.
            const rcsCertPem = new core_2.X509CertificatePem(this, 'TlsCaCertPem', {
                subject: {
                    cn: 'renderfarm.local',
                },
            });
            const rcsCertPkcs = new core_2.X509CertificatePkcs12(this, 'TlsRcsCertBundle', {
                sourceCertificate: rcsCertPem,
            });
            [rcsCertPem.cert, rcsCertPkcs.cert, rcsCertPkcs.passphrase].forEach(secret => {
                secret.grantRead(taskDefinition.taskRole);
            });
            environment.RCS_TLS_CA_CERT_URI = rcsCertPem.cert.secretArn;
            environment.RCS_TLS_CERT_URI = rcsCertPkcs.cert.secretArn;
            environment.RCS_TLS_CERT_PASSPHRASE_URI = rcsCertPkcs.passphrase.secretArn;
            environment.RCS_TLS_REQUIRE_CLIENT_CERT = 'no';
        }
        if (props.secretsManagementOptions !== undefined) {
            environment.RCS_SM_CREDENTIALS_URI = props.secretsManagementOptions.credentials.secretArn;
        }
        // We can ignore this in test coverage because we always use RenderQueue.RCS_USER
        /* istanbul ignore next */
        const user = props.runAsUser ? `${props.runAsUser.uid}:${props.runAsUser.gid}` : undefined;
        const containerDefinition = taskDefinition.addContainer('ContainerDefinition', {
            image,
            memoryReservationMiB: 2048,
            environment,
            logging: aws_ecs_1.LogDriver.awsLogs({
                logGroup: this.logGroup,
                streamPrefix: 'RCS',
            }),
            user,
        });
        containerDefinition.addMountPoints(connection.readWriteMountPoint);
        if (props.secretsManagementOptions !== undefined) {
            // Create volume to persist the RSA keypairs generated by Deadline between ECS tasks
            // This makes it so subsequent ECS tasks use the same initial Secrets Management identity
            const volumeName = 'deadline-user-keypairs';
            taskDefinition.addVolume({
                name: volumeName,
                dockerVolumeConfiguration: {
                    scope: aws_ecs_1.Scope.SHARED,
                    autoprovision: true,
                    driver: 'local',
                },
            });
            // Mount the volume into the container at the location where Deadline expects it
            containerDefinition.addMountPoints({
                readOnly: false,
                sourceVolume: volumeName,
                containerPath: `/home/${props.secretsManagementOptions.posixUsername}/.config/.mono/keypairs`,
            });
        }
        // Increase ulimits
        containerDefinition.addUlimits({
            name: aws_ecs_1.UlimitName.NOFILE,
            softLimit: 200000,
            hardLimit: 200000,
        }, {
            name: aws_ecs_1.UlimitName.NPROC,
            softLimit: 64000,
            hardLimit: 64000,
        });
        containerDefinition.addPortMappings({
            containerPort: portNumber,
            hostPort: portNumber,
        });
        return taskDefinition;
    }
    /**
     * Checks if the user supplied any certificate to use for TLS and uses them, or creates defaults to use.
     * @param props
     * @returns TlsInfo either based on input to the render queue, or the created defaults
     */
    getOrCreateTlsInfo(props) {
        var _b, _c, _d, _e;
        if ((((_c = (_b = props.trafficEncryption) === null || _b === void 0 ? void 0 : _b.externalTLS) === null || _c === void 0 ? void 0 : _c.acmCertificate) !== undefined) ||
            (((_e = (_d = props.trafficEncryption) === null || _d === void 0 ? void 0 : _d.externalTLS) === null || _e === void 0 ? void 0 : _e.rfdkCertificate) !== undefined)) {
            if (props.hostname === undefined) {
                throw new Error('The hostname for the render queue must be defined if supplying your own certificates.');
            }
            return this.getTlsInfoFromUserProps(props.trafficEncryption.externalTLS, props.hostname);
        }
        return this.createDefaultTlsInfo(props.vpc, props.hostname);
    }
    /**
     * Creates a default certificate to use for TLS and a PrivateHostedZone to put the load balancer in.
     * @param vpc
     * @param hostname
     * @returns default TlsInfo
     */
    createDefaultTlsInfo(vpc, hostname) {
        var _b;
        const domainZone = (_b = hostname === null || hostname === void 0 ? void 0 : hostname.zone) !== null && _b !== void 0 ? _b : new aws_route53_1.PrivateHostedZone(this, 'DnsZone', {
            vpc: vpc,
            zoneName: RenderQueue.DEFAULT_DOMAIN_NAME,
        });
        const fullyQualifiedDomainName = this.generateFullyQualifiedDomainName(domainZone, hostname === null || hostname === void 0 ? void 0 : hostname.hostname);
        const rootCa = new core_2.X509CertificatePem(this, 'RootCA', {
            subject: {
                cn: 'RenderQueueRootCA',
            },
        });
        const rfdkCert = new core_2.X509CertificatePem(this, 'RenderQueuePemCert', {
            subject: {
                cn: fullyQualifiedDomainName,
            },
            signingCertificate: rootCa,
        });
        const serverCert = new core_2.ImportedAcmCertificate(this, 'AcmCert', rfdkCert);
        const certChain = rfdkCert.certChain;
        return {
            domainZone,
            fullyQualifiedDomainName,
            serverCert,
            certChain,
        };
    }
    /**
     * Gets the certificate and PrivateHostedZone provided in the Render Queue's construct props.
     * @param externalTLS
     * @param hostname
     * @returns The provided certificate and domain info
     */
    getTlsInfoFromUserProps(externalTLS, hostname) {
        let serverCert;
        let certChain;
        if ((externalTLS.acmCertificate !== undefined) &&
            (externalTLS.rfdkCertificate !== undefined)) {
            throw new Error('Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS.');
        }
        if (!hostname.hostname) {
            throw new Error('A hostname must be supplied if a certificate is supplied, '
                + 'with the common name of the certificate matching the hostname + domain name.');
        }
        const fullyQualifiedDomainName = this.generateFullyQualifiedDomainName(hostname.zone, hostname.hostname);
        if (externalTLS.acmCertificate) {
            if (externalTLS.acmCertificateChain === undefined) {
                throw new Error('externalTLS.acmCertificateChain must be provided when using externalTLS.acmCertificate.');
            }
            serverCert = externalTLS.acmCertificate;
            certChain = externalTLS.acmCertificateChain;
        }
        else { // Using externalTLS.rfdkCertificate
            if (externalTLS.rfdkCertificate.certChain === undefined) {
                throw new Error('Provided rfdkCertificate does not contain a certificate chain.');
            }
            serverCert = new core_2.ImportedAcmCertificate(this, 'AcmCert', externalTLS.rfdkCertificate);
            certChain = externalTLS.rfdkCertificate.certChain;
        }
        return {
            domainZone: hostname.zone,
            fullyQualifiedDomainName,
            serverCert,
            certChain,
        };
    }
    /**
     * Helper method to create the fully qualified domain name for the given hostname and PrivateHostedZone.
     * @param hostname
     * @param zone
     * @returns The fully qualified domain name
     */
    generateFullyQualifiedDomainName(zone, hostname = RenderQueue.DEFAULT_HOSTNAME) {
        if (!RenderQueue.RE_VALID_HOSTNAME.test(hostname)) {
            throw new Error(`Invalid RenderQueue hostname: ${hostname}`);
        }
        return `${hostname}.${zone.zoneName}`;
    }
    /**
     * The instance that runs commands during the deployment.
     */
    get deploymentInstance() {
        var _b;
        const CONFIGURE_REPOSITORY_CONSTRUCT_ID = 'ConfigureRepository';
        const deploymentInstanceNode = this.node.tryFindChild(CONFIGURE_REPOSITORY_CONSTRUCT_ID);
        if (deploymentInstanceNode === undefined) {
            return new deployment_instance_1.DeploymentInstance(this, CONFIGURE_REPOSITORY_CONSTRUCT_ID, {
                securityGroup: this.backendConnections.securityGroups[0],
                vpc: this.props.vpc,
                vpcSubnets: (_b = this.props.vpcSubnets) !== null && _b !== void 0 ? _b : RenderQueue.DEFAULT_VPC_SUBNETS_OTHER,
            });
        }
        else if (deploymentInstanceNode instanceof deployment_instance_1.DeploymentInstance) {
            return deploymentInstanceNode;
        }
        else {
            throw new Error(`Unexpected type for ${deploymentInstanceNode.node.path}. Expected ${deployment_instance_1.DeploymentInstance.name}, but found ${typeof (deploymentInstanceNode)}.`);
        }
    }
    /**
     * The construct that manages Deadline Secrets Management identity registration settings
     */
    get identityRegistrationSettings() {
        var _b;
        const IDENTITY_REGISTRATION_CONSTRUCT_ID = 'SecretsManagementIdentityRegistration';
        const secretsManagementIdentityRegistration = this.node.tryFindChild(IDENTITY_REGISTRATION_CONSTRUCT_ID);
        if (!secretsManagementIdentityRegistration) {
            return new secrets_management_1.SecretsManagementIdentityRegistration(this, IDENTITY_REGISTRATION_CONSTRUCT_ID, {
                deploymentInstance: this.deploymentInstance,
                repository: this.repository,
                renderQueueSubnets: this.props.vpc.selectSubnets((_b = this.props.vpcSubnetsAlb) !== null && _b !== void 0 ? _b : RenderQueue.DEFAULT_VPC_SUBNETS_ALB),
                version: this.props.version,
            });
        }
        else if (secretsManagementIdentityRegistration instanceof secrets_management_1.SecretsManagementIdentityRegistration) {
            return secretsManagementIdentityRegistration;
        }
        else {
            throw new Error(`Unexpected type for ${secretsManagementIdentityRegistration.node.path}. Expected ${secrets_management_1.SecretsManagementIdentityRegistration.name}, but found ${typeof (secretsManagementIdentityRegistration)}.`);
        }
    }
}
exports.RenderQueue = RenderQueue;
_a = JSII_RTTI_SYMBOL_1;
RenderQueue[_a] = { fqn: "aws-rfdk.deadline.RenderQueue", version: "0.40.0" };
/**
 * Container listening ports for each protocol.
 */
RenderQueue.RCS_PROTO_PORTS = {
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP]: 8080,
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS]: 4433,
};
RenderQueue.DEFAULT_HOSTNAME = 'renderqueue';
RenderQueue.DEFAULT_DOMAIN_NAME = 'aws-rfdk.com';
RenderQueue.DEFAULT_VPC_SUBNETS_ALB = { subnetType: aws_ec2_1.SubnetType.PRIVATE, onePerAz: true };
RenderQueue.DEFAULT_VPC_SUBNETS_OTHER = { subnetType: aws_ec2_1.SubnetType.PRIVATE };
/**
* The minimum Deadline version required for the Remote Connection Server to support load-balancing
*/
RenderQueue.MINIMUM_LOAD_BALANCING_VERSION = new version_1.Version([10, 1, 10, 0]);
/**
 * Regular expression that validates a hostname (portion in front of the subdomain).
 */
RenderQueue.RE_VALID_HOSTNAME = /^[a-z](?:[a-z0-9-]{0,61}[a-z0-9])?$/i;
/**
 * Information for the RCS user.
 */
RenderQueue.RCS_USER = { uid: 1000, gid: 1000, username: 'ec2-user' };
//# sourceMappingURL=data:application/json;base64,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