"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigureSpotEventPlugin = exports.SpotEventPluginDisplayInstanceStatus = exports.SpotEventPluginPreJobTaskMode = exports.SpotEventPluginLoggingLevel = exports.SpotEventPluginState = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const render_queue_1 = require("./render-queue");
const secrets_management_ref_1 = require("./secrets-management-ref");
const spot_event_plugin_fleet_ref_1 = require("./spot-event-plugin-fleet-ref");
const version_1 = require("./version");
/**
 * How the event plug-in should respond to events.
 *
 * @stability stable
 */
var SpotEventPluginState;
(function (SpotEventPluginState) {
    SpotEventPluginState["GLOBAL_ENABLED"] = "Global Enabled";
    SpotEventPluginState["DISABLED"] = "Disabled";
})(SpotEventPluginState = exports.SpotEventPluginState || (exports.SpotEventPluginState = {}));
/**
 * Logging verbosity levels for the Spot Event Plugin.
 *
 * @stability stable
 */
var SpotEventPluginLoggingLevel;
(function (SpotEventPluginLoggingLevel) {
    SpotEventPluginLoggingLevel["STANDARD"] = "Standard";
    SpotEventPluginLoggingLevel["VERBOSE"] = "Verbose";
    SpotEventPluginLoggingLevel["DEBUG"] = "Debug";
    SpotEventPluginLoggingLevel["OFF"] = "Off";
})(SpotEventPluginLoggingLevel = exports.SpotEventPluginLoggingLevel || (exports.SpotEventPluginLoggingLevel = {}));
/**
 * How the Spot Event Plugin should handle Pre Job Tasks.
 *
 * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/job-scripts.html
 *
 * @stability stable
 */
var SpotEventPluginPreJobTaskMode;
(function (SpotEventPluginPreJobTaskMode) {
    SpotEventPluginPreJobTaskMode["CONSERVATIVE"] = "Conservative";
    SpotEventPluginPreJobTaskMode["IGNORE"] = "Ignore";
    SpotEventPluginPreJobTaskMode["NORMAL"] = "Normal";
})(SpotEventPluginPreJobTaskMode = exports.SpotEventPluginPreJobTaskMode || (exports.SpotEventPluginPreJobTaskMode = {}));
/**
 * The Worker Extra Info column to be used to display AWS Instance Status if the instance has been marked to be stopped or terminated by EC2 or Spot Event Plugin.
 *
 * See "AWS Instance Status" option at https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html#event-plugin-configuration-options
 * and https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/worker-config.html#extra-info
 *
 * @stability stable
 */
var SpotEventPluginDisplayInstanceStatus;
(function (SpotEventPluginDisplayInstanceStatus) {
    SpotEventPluginDisplayInstanceStatus["DISABLED"] = "Disabled";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_0"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_1"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_2"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_3"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_4"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_5"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_6"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_7"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_8"] = "ExtraInfo0";
    SpotEventPluginDisplayInstanceStatus["EXTRA_INFO_9"] = "ExtraInfo0";
})(SpotEventPluginDisplayInstanceStatus = exports.SpotEventPluginDisplayInstanceStatus || (exports.SpotEventPluginDisplayInstanceStatus = {}));
/**
 * This construct configures the Deadline Spot Event Plugin to deploy and auto-scale one or more spot fleets.
 *
 * For example, to configure the Spot Event Plugin with one spot fleet:
 *
 * ```ts
 * import { App, Stack, Vpc } from '@aws-rfdk/core';
 * import { InstanceClass, InstanceSize, InstanceType } from '@aws-cdk/aws-ec2';
 * import { AwsThinkboxEulaAcceptance, ConfigureSpotEventPlugin, RenderQueue, Repository, SpotEventPluginFleet, ThinkboxDockerImages, VersionQuery } from '@aws-rfdk/deadline';
 * const app = new App();
 * const stack = new Stack(app, 'Stack');
 * const vpc = new Vpc(stack, 'Vpc');
 * const version = new VersionQuery(stack, 'Version', {
 *    version: '10.1.12',
 * });
 * const images = new ThinkboxDockerImages(stack, 'Image', {
 *    version,
 *    // Change this to AwsThinkboxEulaAcceptance.USER_ACCEPTS_AWS_THINKBOX_EULA to accept the terms
 *    // of the AWS Thinkbox End User License Agreement
 *    userAwsThinkboxEulaAcceptance: AwsThinkboxEulaAcceptance.USER_REJECTS_AWS_THINKBOX_EULA,
 * });
 * const repository = new Repository(stack, 'Repository', {
 *    vpc,
 *    version,
 * });
 * const renderQueue = new RenderQueue(stack, 'RenderQueue', {
 *    vpc,
 *    images: images.forRenderQueue(),
 *    repository: repository,
 * });
 *
 * const fleet = new SpotEventPluginFleet(this, 'SpotEventPluginFleet', {
 *    vpc,
 *    renderQueue,
 *    deadlineGroups: ['group_name'],
 *    instanceTypes: [InstanceType.of(InstanceClass.T3, InstanceSize.LARGE)],
 *    workerMachineImage: new GenericLinuxImage({'us-west-2': 'ami-039f0c1faba28b015'}),
 *    naxCapacity: 1,
 * });
 *
 * const spotEventPluginConfig = new ConfigureSpotEventPlugin(this, 'ConfigureSpotEventPlugin', {
 *    vpc,
 *    renderQueue: renderQueue,
 *    spotFleets: [fleet],
 *    configuration: {
 *      enableResourceTracker: true,
 *    },
 * });
 * ```
 *
 * To provide this functionality, this construct will create an AWS Lambda function that is granted the ability
 * to connect to the render queue. This lambda is run automatically when you deploy or update the stack containing this construct.
 * Logs for all AWS Lambdas are automatically recorded in Amazon CloudWatch.
 *
 * This construct will configure the Spot Event Plugin, but the Spot Fleet Requests will not be created unless you:
 * - Submit the job with the assigned Deadline Group and Deadline Pool. See [Deadline Documentation](https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/job-submitting.html#submitting-jobs).
 *
 * Important: Disable 'Allow Workers to Perform House Cleaning If Pulse is not Running' in the 'Configure Repository Options'
 * when using Spot Event Plugin.
 * See https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html#prerequisites
 *
 * Important: Any resources created by the Spot Event Plugin will not be deleted with 'cdk destroy'.
 * Make sure that all such resources (e.g. Spot Fleet Request or Fleet Instances) are cleaned up, before destroying the stacks.
 * Disable the Spot Event Plugin by setting 'state' property to 'SpotEventPluginState.DISABLED' or via Deadline Monitor,
 * ensure you shutdown all Pulse instances and then terminate any Spot Fleet Requests in the AWS EC2 Instance Console.
 *
 * Note that this construct adds additional policies to the Render Queue's role
 * required to run the Spot Event Plugin and launch a Resource Tracker:
 *   - AWSThinkboxDeadlineSpotEventPluginAdminPolicy
 *   - AWSThinkboxDeadlineResourceTrackerAdminPolicy
 *   - A policy to pass a fleet and instance role
 *   - A policy to create tags for spot fleet requests
 *
 * The Spot Fleet Requests that this construct configures Deadline to create will always use the latest version of the
 * corresponding EC2 Launch Template that was created for them.
 *
 * ![architecture diagram](/diagrams/deadline/ConfigureSpotEventPlugin.svg)
 *
 * Resources Deployed
 * ------------------------
 * - An AWS Lambda that is used to connect to the render queue, and save Spot Event Plugin configurations.
 * - A CloudFormation Custom Resource that triggers execution of the Lambda on stack deployment, update, and deletion.
 * - An Amazon CloudWatch log group that records history of the AWS Lambda's execution.
 * - An IAM Policy attached to Render Queue's Role.
 * - EC2 Launch Templates for each Spot Event Plugin fleet.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda function that is created by this resource has access to both the certificates used to connect to the render queue,
 *    and the render queue port. An attacker that can find a way to modify and execute this lambda could use it to
 *    execute any requets against the render queue. You should not grant any additional actors/principals the ability to modify
 *    or execute this Lambda.
 *
 * @stability stable
 */
class ConfigureSpotEventPlugin extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4;
        super(scope, id);
        if (ConfigureSpotEventPlugin.uniqueRenderQueues.has(props.renderQueue)) {
            throw new Error('Only one ConfigureSpotEventPlugin construct is allowed per render queue.');
        }
        else {
            ConfigureSpotEventPlugin.uniqueRenderQueues.add(props.renderQueue);
        }
        if (props.renderQueue instanceof render_queue_1.RenderQueue) {
            // We do not check the patch version, so it's set to 0.
            const minimumVersion = new version_1.Version([10, 1, 12, 0]);
            if (props.renderQueue.version.isLessThan(minimumVersion)) {
                throw new Error(`Minimum supported Deadline version for ${this.constructor.name} is ` +
                    `${minimumVersion.versionString}. ` +
                    `Received: ${props.renderQueue.version.versionString}.`);
            }
            if (props.spotFleets && props.spotFleets.length !== 0) {
                // Always add Resource Tracker admin policy, even if props.configuration?.enableResourceTracker is false.
                // This improves usability, as customers won't need to add this policy manually, if they
                // enable Resource Tracker later in the Spot Event Plugin configuration (e.g., using Deadline Monitor and not RFDK).
                props.renderQueue.addSEPPolicies(true);
                const fleetRoles = props.spotFleets.map(sf => sf.fleetRole.roleArn);
                const fleetInstanceRoles = props.spotFleets.map(sf => sf.fleetInstanceRole.roleArn);
                new aws_iam_1.Policy(this, 'SpotEventPluginPolicy', {
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                'iam:PassRole',
                            ],
                            resources: [...fleetRoles, ...fleetInstanceRoles],
                            conditions: {
                                StringLike: {
                                    'iam:PassedToService': 'ec2.amazonaws.com',
                                },
                            },
                        }),
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                'ec2:CreateTags',
                            ],
                            resources: [
                                'arn:aws:ec2:*:*:spot-fleet-request/*',
                                'arn:aws:ec2:*:*:volume/*',
                            ],
                        }),
                    ],
                    roles: [
                        props.renderQueue.grantPrincipal,
                    ],
                });
            }
        }
        else {
            throw new Error('The provided render queue is not an instance of RenderQueue class. Some functionality is not supported.');
        }
        const region = core_1.Construct.isConstruct(props.renderQueue) ? core_1.Stack.of(props.renderQueue).region : core_1.Stack.of(this).region;
        const timeoutMins = 15;
        const configurator = new aws_lambda_1.Function(this, 'Configurator', {
            vpc: props.vpc,
            vpcSubnets: (_b = props.vpcSubnets) !== null && _b !== void 0 ? _b : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            description: `Used by a ConfigureSpotEventPlugin ${this.node.addr} to perform configuration of Deadline Spot Event Plugin`,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs'), {}),
            environment: {
                DEBUG: 'false',
                LAMBDA_TIMEOUT_MINS: timeoutMins.toString(),
            },
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'configure-spot-event-plugin.configureSEP',
            timeout: core_1.Duration.minutes(timeoutMins),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        configurator.connections.allowToDefaultPort(props.renderQueue);
        (_c = props.renderQueue.certChain) === null || _c === void 0 ? void 0 : _c.grantRead(configurator.grantPrincipal);
        const pluginConfig = {
            AWSInstanceStatus: (_e = (_d = props.configuration) === null || _d === void 0 ? void 0 : _d.awsInstanceStatus) !== null && _e !== void 0 ? _e : SpotEventPluginDisplayInstanceStatus.DISABLED,
            DeleteInterruptedSlaves: (_g = (_f = props.configuration) === null || _f === void 0 ? void 0 : _f.deleteEC2SpotInterruptedWorkers) !== null && _g !== void 0 ? _g : false,
            DeleteTerminatedSlaves: (_j = (_h = props.configuration) === null || _h === void 0 ? void 0 : _h.deleteSEPTerminatedWorkers) !== null && _j !== void 0 ? _j : false,
            IdleShutdown: (_m = (_l = (_k = props.configuration) === null || _k === void 0 ? void 0 : _k.idleShutdown) === null || _l === void 0 ? void 0 : _l.toMinutes({ integral: true })) !== null && _m !== void 0 ? _m : 10,
            Logging: (_p = (_o = props.configuration) === null || _o === void 0 ? void 0 : _o.loggingLevel) !== null && _p !== void 0 ? _p : SpotEventPluginLoggingLevel.STANDARD,
            PreJobTaskMode: (_r = (_q = props.configuration) === null || _q === void 0 ? void 0 : _q.preJobTaskMode) !== null && _r !== void 0 ? _r : SpotEventPluginPreJobTaskMode.CONSERVATIVE,
            Region: (_t = (_s = props.configuration) === null || _s === void 0 ? void 0 : _s.region) !== null && _t !== void 0 ? _t : region,
            ResourceTracker: (_v = (_u = props.configuration) === null || _u === void 0 ? void 0 : _u.enableResourceTracker) !== null && _v !== void 0 ? _v : true,
            StaggerInstances: (_x = (_w = props.configuration) === null || _w === void 0 ? void 0 : _w.maximumInstancesStartedPerCycle) !== null && _x !== void 0 ? _x : 50,
            State: (_z = (_y = props.configuration) === null || _y === void 0 ? void 0 : _y.state) !== null && _z !== void 0 ? _z : SpotEventPluginState.GLOBAL_ENABLED,
            StrictHardCap: (_1 = (_0 = props.configuration) === null || _0 === void 0 ? void 0 : _0.strictHardCap) !== null && _1 !== void 0 ? _1 : false,
        };
        const spotFleetRequestConfigs = this.mergeSpotFleetRequestConfigs(props.spotFleets);
        const deadlineGroups = Array.from(new Set((_2 = props.spotFleets) === null || _2 === void 0 ? void 0 : _2.map(fleet => fleet.deadlineGroups).reduce((p, c) => p.concat(c), [])));
        const deadlinePools = Array.from(new Set((_3 = props.spotFleets) === null || _3 === void 0 ? void 0 : _3.map(fleet => fleet.deadlinePools).reduce((p, c) => p === null || p === void 0 ? void 0 : p.concat(c !== null && c !== void 0 ? c : []), [])));
        const properties = {
            connection: {
                hostname: props.renderQueue.endpoint.hostname,
                port: props.renderQueue.endpoint.portAsString(),
                protocol: props.renderQueue.endpoint.applicationProtocol,
                caCertificateArn: (_4 = props.renderQueue.certChain) === null || _4 === void 0 ? void 0 : _4.secretArn,
            },
            spotFleetRequestConfigurations: spotFleetRequestConfigs,
            spotPluginConfigurations: pluginConfig,
            deadlineGroups,
            deadlinePools,
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: configurator.functionArn,
            resourceType: 'Custom::RFDK_ConfigureSpotEventPlugin',
            properties,
        });
        // Prevents a race during a stack-update.
        resource.node.addDependency(configurator.role);
        // We need to add this dependency to avoid failures while deleting a Custom Resource:
        // 'Custom Resource failed to stabilize in expected time. If you are using the Python cfn-response module,
        // you may need to update your Lambda function code so that CloudFormation can attach the updated version.'.
        // This happens, because Route Table Associations are deleted before the Custom Resource and we
        // don't get a response from 'doDelete()'.
        // Ideally, we would only want to add dependency on 'internetConnectivityEstablished' as shown below,
        // but it seems that CDK misses dependencies on Route Table Associations in that case:
        // const { internetConnectivityEstablished } = props.vpc.selectSubnets(props.vpcSubnets);
        // resource.node.addDependency(internetConnectivityEstablished);
        resource.node.addDependency(props.vpc);
        // /* istanbul ignore next */
        // Add a dependency on the render queue to ensure that
        // it is running before we try to send requests to it.
        resource.node.addDependency(props.renderQueue);
        if (props.spotFleets && props.renderQueue.repository.secretsManagementSettings.enabled) {
            props.spotFleets.forEach(spotFleet => {
                if (spotFleet.defaultSubnets) {
                    core_1.Annotations.of(spotFleet).addWarning('Deadline Secrets Management is enabled on the Repository and VPC subnets have not been supplied. Using dedicated subnets is recommended. See https://github.com/aws/aws-rfdk/blobs/release/packages/aws-rfdk/lib/deadline/README.md#using-dedicated-subnets-for-deadline-components');
                }
                props.renderQueue.configureSecretsManagementAutoRegistration({
                    dependent: resource,
                    role: secrets_management_ref_1.SecretsManagementRole.CLIENT,
                    registrationStatus: secrets_management_ref_1.SecretsManagementRegistrationStatus.REGISTERED,
                    vpc: props.vpc,
                    vpcSubnets: spotFleet.subnets,
                });
            });
        }
        this.node.defaultChild = resource;
    }
    tagSpecifications(fleet, resourceType) {
        return core_1.Lazy.any({
            produce: () => {
                if (fleet.tags.hasTags()) {
                    const tagSpecification = {
                        ResourceType: resourceType,
                        Tags: fleet.tags.renderTags(),
                    };
                    return [tagSpecification];
                }
                return undefined;
            },
        });
    }
    /**
     * Construct Spot Fleet Configurations from the provided fleet.
     * Each configuration is a mapping between one Deadline Group and one Spot Fleet Request Configuration.
     */
    generateSpotFleetRequestConfig(fleet) {
        var _b;
        const spotFleetRequestTagsToken = this.tagSpecifications(fleet, spot_event_plugin_fleet_ref_1.SpotFleetResourceType.SPOT_FLEET_REQUEST);
        const spotFleetRequestProps = {
            AllocationStrategy: fleet.allocationStrategy,
            IamFleetRole: fleet.fleetRole.roleArn,
            LaunchTemplateConfigs: fleet._launchTemplateConfigs,
            ReplaceUnhealthyInstances: true,
            // In order to work with Deadline, the 'Target Capacity' of the Spot fleet Request is
            // the maximum number of Workers that Deadline will start.
            TargetCapacity: fleet.maxCapacity,
            TerminateInstancesWithExpiration: true,
            // In order to work with Deadline, Spot Fleets Requests must be set to maintain.
            Type: spot_event_plugin_fleet_ref_1.SpotFleetRequestType.MAINTAIN,
            ValidUntil: (_b = fleet.validUntil) === null || _b === void 0 ? void 0 : _b.date.toISOString(),
            // Need to convert from IResolvable to bypass TypeScript
            TagSpecifications: spotFleetRequestTagsToken,
        };
        const spotFleetRequestConfigurations = fleet.deadlineGroups.map(group => {
            const spotFleetRequestConfiguration = {
                [group.toLowerCase()]: spotFleetRequestProps,
            };
            return spotFleetRequestConfiguration;
        });
        return spotFleetRequestConfigurations;
    }
    mergeSpotFleetRequestConfigs(spotFleets) {
        if (!spotFleets || spotFleets.length === 0) {
            return undefined;
        }
        const fullSpotFleetRequestConfiguration = {};
        spotFleets.map(fleet => {
            const spotFleetRequestConfigurations = this.generateSpotFleetRequestConfig(fleet);
            spotFleetRequestConfigurations.map(configuration => {
                for (const [key, value] of Object.entries(configuration)) {
                    if (key in fullSpotFleetRequestConfiguration) {
                        throw new Error(`Bad Group Name: ${key}. Group names in Spot Fleet Request Configurations should be unique.`);
                    }
                    fullSpotFleetRequestConfiguration[key] = value;
                }
            });
        });
        return fullSpotFleetRequestConfiguration;
    }
}
exports.ConfigureSpotEventPlugin = ConfigureSpotEventPlugin;
_a = JSII_RTTI_SYMBOL_1;
ConfigureSpotEventPlugin[_a] = { fqn: "aws-rfdk.deadline.ConfigureSpotEventPlugin", version: "0.40.0" };
/**
 * Only one Spot Event Plugin Configuration is allowed per render queue / repository.
 */
ConfigureSpotEventPlugin.uniqueRenderQueues = new Set();
//# sourceMappingURL=data:application/json;base64,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