"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const fsx = require("@aws-cdk/aws-fsx");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const token_regex_helpers_1 = require("./token-regex-helpers");
describe('MountableFsxLustre', () => {
    let app;
    let stack;
    let vpc;
    let fs;
    let fsSecurityGroup;
    let instance;
    let instanceSecurityGroup;
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app);
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        fsSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'FSxLSecurityGroup', {
            vpc,
        });
        fs = new fsx.LustreFileSystem(stack, 'FSxL', {
            vpc,
            vpcSubnet: vpc.privateSubnets[0],
            lustreConfiguration: {
                deploymentType: fsx.LustreDeploymentType.SCRATCH_1,
            },
            storageCapacityGiB: 1200,
            securityGroup: fsSecurityGroup,
        });
        instanceSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'InstanceSecurityGroup', {
            vpc,
        });
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            securityGroup: instanceSecurityGroup,
        });
    });
    test('mounts with defaults', () => {
        // GIVEN
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        // Make sure the instance has been granted ingress to the FSxL's security group
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: 988,
            ToPort: 1023,
            SourceSecurityGroupId: stack.resolve(instanceSecurityGroup.securityGroupId),
            GroupId: stack.resolve(fsSecurityGroup.securityGroupId),
        }));
        // Make sure we download the mountFsxLustre script asset bundle
        const s3Copy = 'aws s3 cp \'s3://${Token[TOKEN.\\d+]}/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \'/tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\'';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy)));
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('unzip /tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}')));
        // Make sure we install the Lustre client
        expect(userData).toMatch('bash ./installLustreClient.sh');
        // Make sure we execute the script with the correct args
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('bash ./mountFsxLustre.sh ${Token[TOKEN.\\d+]} /mnt/fsx/fs1 ${Token[TOKEN.\\d+]} rw')));
    });
    test('assert Linux-only', () => {
        // GIVEN
        const windowsInstance = new aws_ec2_1.Instance(stack, 'WindowsInstance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_SQL_2017_STANDARD),
        });
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        // THEN
        expect(() => {
            mount.mountToLinuxInstance(windowsInstance, {
                location: '/mnt/fsx/fs1',
                permissions: lib_1.MountPermissions.READONLY,
            });
        }).toThrowError('Target instance must be Linux.');
    });
    test('readonly mount', () => {
        // GIVEN
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
            permissions: lib_1.MountPermissions.READONLY,
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('mountFsxLustre.sh ${Token[TOKEN.\\d+]} /mnt/fsx/fs1 ${Token[TOKEN.\\d+]} r')));
    });
    test('extra mount options', () => {
        // GIVEN
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
            extraMountOptions: [
                'option1',
                'option2',
            ],
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('mountFsxLustre.sh ${Token[TOKEN.\\d+]} /mnt/fsx/fs1 ${Token[TOKEN.\\d+]} rw,option1,option2')));
    });
    test('asset is singleton', () => {
        var _a;
        // GIVEN
        const mount1 = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        const mount2 = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        // WHEN
        mount1.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
        });
        mount2.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
        });
        const userData = instance.userData.render();
        const s3Copy = 'aws s3 cp \'s3://${Token[TOKEN.\\d+]}/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\'';
        const regex = new RegExp(token_regex_helpers_1.escapeTokenRegex(s3Copy), 'g');
        const matches = (_a = userData.match(regex)) !== null && _a !== void 0 ? _a : [];
        // THEN
        // The source of the asset copy should be identical from mount1 & mount2
        expect(matches).toHaveLength(2);
        expect(matches[0]).toBe(matches[1]);
    });
    test('applies Lustre fileset', () => {
        // GIVEN
        const fileset = 'fileset';
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
            fileset,
        });
        // WHEN
        mount.mountToLinuxInstance(instance, {
            location: '/mnt/fsx/fs1',
        });
        const userData = instance.userData.render();
        // THEN
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(`bash ./mountFsxLustre.sh \${Token[TOKEN.\\d+]} /mnt/fsx/fs1 \${Token[TOKEN.\\d+]}/${fileset} rw`)));
    });
    test('.usesUserPosixPermissions() is true', () => {
        // GIVEN
        const mount = new lib_1.MountableFsxLustre(fs, {
            filesystem: fs,
        });
        // WHEN
        const usesUserPosixPermissions = mount.usesUserPosixPermissions();
        // THEN
        expect(usesUserPosixPermissions).toEqual(true);
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibW91bnRhYmxlLWZzeC1sdXN0cmUudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIm1vdW50YWJsZS1mc3gtbHVzdHJlLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7R0FHRzs7QUFFSCw0Q0FHeUI7QUFDekIsOENBUTBCO0FBQzFCLHdDQUF3QztBQUN4Qyx3Q0FHdUI7QUFFdkIsZ0NBR2dCO0FBRWhCLCtEQUUrQjtBQUUvQixRQUFRLENBQUMsb0JBQW9CLEVBQUUsR0FBRyxFQUFFO0lBQ2xDLElBQUksR0FBUSxDQUFDO0lBQ2IsSUFBSSxLQUFZLENBQUM7SUFDakIsSUFBSSxHQUFRLENBQUM7SUFDYixJQUFJLEVBQXdCLENBQUM7SUFDN0IsSUFBSSxlQUE4QixDQUFDO0lBQ25DLElBQUksUUFBa0IsQ0FBQztJQUN2QixJQUFJLHFCQUFvQyxDQUFDO0lBRXpDLFVBQVUsQ0FBQyxHQUFHLEVBQUU7UUFDZCxHQUFHLEdBQUcsSUFBSSxVQUFHLEVBQUUsQ0FBQztRQUNoQixLQUFLLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDdkIsR0FBRyxHQUFHLElBQUksYUFBRyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztRQUM1QixlQUFlLEdBQUcsSUFBSSx1QkFBYSxDQUFDLEtBQUssRUFBRSxtQkFBbUIsRUFBRTtZQUM5RCxHQUFHO1NBQ0osQ0FBQyxDQUFDO1FBQ0gsRUFBRSxHQUFHLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7WUFDM0MsR0FBRztZQUNILFNBQVMsRUFBRSxHQUFHLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQztZQUNoQyxtQkFBbUIsRUFBRTtnQkFDbkIsY0FBYyxFQUFFLEdBQUcsQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTO2FBQ25EO1lBQ0Qsa0JBQWtCLEVBQUUsSUFBSTtZQUN4QixhQUFhLEVBQUUsZUFBZTtTQUMvQixDQUFDLENBQUM7UUFDSCxxQkFBcUIsR0FBRyxJQUFJLHVCQUFhLENBQUMsS0FBSyxFQUFFLHVCQUF1QixFQUFFO1lBQ3hFLEdBQUc7U0FDSixDQUFDLENBQUM7UUFDSCxRQUFRLEdBQUcsSUFBSSxrQkFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDekMsR0FBRztZQUNILFlBQVksRUFBRSxJQUFJLHNCQUFZLENBQUMsVUFBVSxDQUFDO1lBQzFDLFlBQVksRUFBRSxzQkFBWSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsVUFBVSxFQUFFLCtCQUFxQixDQUFDLGNBQWMsRUFBRSxDQUFDO1lBQ2xHLGFBQWEsRUFBRSxxQkFBcUI7U0FDckMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsc0JBQXNCLEVBQUUsR0FBRyxFQUFFO1FBQ2hDLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLHdCQUFrQixDQUFDLEVBQUUsRUFBRTtZQUN2QyxVQUFVLEVBQUUsRUFBRTtTQUNmLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxLQUFLLENBQUMsb0JBQW9CLENBQUMsUUFBUSxFQUFFO1lBQ25DLFFBQVEsRUFBRSxjQUFjO1NBQ3pCLENBQUMsQ0FBQztRQUNILE1BQU0sUUFBUSxHQUFHLFFBQVEsQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUM7UUFFNUMsT0FBTztRQUNQLCtFQUErRTtRQUMvRSxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdDQUFnQyxFQUFFO1lBQ3JFLFVBQVUsRUFBRSxLQUFLO1lBQ2pCLFFBQVEsRUFBRSxHQUFHO1lBQ2IsTUFBTSxFQUFFLElBQUk7WUFDWixxQkFBcUIsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLHFCQUFxQixDQUFDLGVBQWUsQ0FBQztZQUMzRSxPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxlQUFlLENBQUMsZUFBZSxDQUFDO1NBQ3hELENBQUMsQ0FBQyxDQUFDO1FBQ0osK0RBQStEO1FBQy9ELE1BQU0sTUFBTSxHQUFHLG9JQUFvSSxDQUFDO1FBQ3BKLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxNQUFNLENBQUMsc0NBQWdCLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQy9ELE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxNQUFNLENBQUMsc0NBQWdCLENBQUMscURBQXFELENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDOUcseUNBQXlDO1FBQ3pDLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxPQUFPLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUMxRCx3REFBd0Q7UUFDeEQsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxJQUFJLE1BQU0sQ0FBQyxzQ0FBZ0IsQ0FBQyxvRkFBb0YsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUMvSSxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxtQkFBbUIsRUFBRSxHQUFHLEVBQUU7UUFDN0IsUUFBUTtRQUNSLE1BQU0sZUFBZSxHQUFHLElBQUksa0JBQVEsQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUU7WUFDN0QsR0FBRztZQUNILFlBQVksRUFBRSxJQUFJLHNCQUFZLENBQUMsVUFBVSxDQUFDO1lBQzFDLFlBQVksRUFBRSxzQkFBWSxDQUFDLGFBQWEsQ0FBQyx3QkFBYyxDQUFDLGtEQUFrRCxDQUFDO1NBQzVHLENBQUMsQ0FBQztRQUNILE1BQU0sS0FBSyxHQUFHLElBQUksd0JBQWtCLENBQUMsRUFBRSxFQUFFO1lBQ3ZDLFVBQVUsRUFBRSxFQUFFO1NBQ2YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDVixLQUFLLENBQUMsb0JBQW9CLENBQUMsZUFBZSxFQUFFO2dCQUMxQyxRQUFRLEVBQUUsY0FBYztnQkFDeEIsV0FBVyxFQUFFLHNCQUFnQixDQUFDLFFBQVE7YUFDdkMsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7SUFDcEQsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsR0FBRyxFQUFFO1FBQzFCLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLHdCQUFrQixDQUFDLEVBQUUsRUFBRTtZQUN2QyxVQUFVLEVBQUUsRUFBRTtTQUNmLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxLQUFLLENBQUMsb0JBQW9CLENBQUMsUUFBUSxFQUFFO1lBQ25DLFFBQVEsRUFBRSxjQUFjO1lBQ3hCLFdBQVcsRUFBRSxzQkFBZ0IsQ0FBQyxRQUFRO1NBQ3ZDLENBQUMsQ0FBQztRQUNILE1BQU0sUUFBUSxHQUFHLFFBQVEsQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUM7UUFFNUMsT0FBTztRQUNQLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxNQUFNLENBQUMsc0NBQWdCLENBQUMsNEVBQTRFLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDdkksQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMscUJBQXFCLEVBQUUsR0FBRyxFQUFFO1FBQy9CLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLHdCQUFrQixDQUFDLEVBQUUsRUFBRTtZQUN2QyxVQUFVLEVBQUUsRUFBRTtZQUNkLGlCQUFpQixFQUFFO2dCQUNqQixTQUFTO2dCQUNULFNBQVM7YUFDVjtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxLQUFLLENBQUMsb0JBQW9CLENBQUMsUUFBUSxFQUFFO1lBQ25DLFFBQVEsRUFBRSxjQUFjO1NBQ3pCLENBQUMsQ0FBQztRQUNILE1BQU0sUUFBUSxHQUFHLFFBQVEsQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUM7UUFFNUMsT0FBTztRQUNQLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxNQUFNLENBQUMsc0NBQWdCLENBQUMsNkZBQTZGLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDeEosQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsb0JBQW9CLEVBQUUsR0FBRyxFQUFFOztRQUM5QixRQUFRO1FBQ1IsTUFBTSxNQUFNLEdBQUcsSUFBSSx3QkFBa0IsQ0FBQyxFQUFFLEVBQUU7WUFDeEMsVUFBVSxFQUFFLEVBQUU7U0FDZixDQUFDLENBQUM7UUFDSCxNQUFNLE1BQU0sR0FBRyxJQUFJLHdCQUFrQixDQUFDLEVBQUUsRUFBRTtZQUN4QyxVQUFVLEVBQUUsRUFBRTtTQUNmLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsb0JBQW9CLENBQUMsUUFBUSxFQUFFO1lBQ3BDLFFBQVEsRUFBRSxjQUFjO1NBQ3pCLENBQUMsQ0FBQztRQUNILE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxRQUFRLEVBQUU7WUFDcEMsUUFBUSxFQUFFLGNBQWM7U0FDekIsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxRQUFRLEdBQUcsUUFBUSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUM1QyxNQUFNLE1BQU0sR0FBRyxrRkFBa0YsQ0FBQztRQUNsRyxNQUFNLEtBQUssR0FBRyxJQUFJLE1BQU0sQ0FBQyxzQ0FBZ0IsQ0FBQyxNQUFNLENBQUMsRUFBRSxHQUFHLENBQUMsQ0FBQztRQUN4RCxNQUFNLE9BQU8sU0FBRyxRQUFRLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxtQ0FBSSxFQUFFLENBQUM7UUFFNUMsT0FBTztRQUNQLHdFQUF3RTtRQUN4RSxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ2hDLE1BQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDdEMsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsd0JBQXdCLEVBQUUsR0FBRyxFQUFFO1FBQ2xDLFFBQVE7UUFDUixNQUFNLE9BQU8sR0FBRyxTQUFTLENBQUM7UUFDMUIsTUFBTSxLQUFLLEdBQUcsSUFBSSx3QkFBa0IsQ0FBQyxFQUFFLEVBQUU7WUFDdkMsVUFBVSxFQUFFLEVBQUU7WUFDZCxPQUFPO1NBQ1IsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxRQUFRLEVBQUU7WUFDbkMsUUFBUSxFQUFFLGNBQWM7U0FDekIsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxRQUFRLEdBQUcsUUFBUSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUU1QyxPQUFPO1FBQ1AsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxJQUFJLE1BQU0sQ0FBQyxzQ0FBZ0IsQ0FBQyxxRkFBcUYsT0FBTyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDNUosQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMscUNBQXFDLEVBQUUsR0FBRyxFQUFFO1FBQy9DLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLHdCQUFrQixDQUFDLEVBQUUsRUFBRTtZQUN2QyxVQUFVLEVBQUUsRUFBRTtTQUNmLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLHdCQUF3QixHQUFHLEtBQUssQ0FBQyx3QkFBd0IsRUFBRSxDQUFDO1FBRWxFLE9BQU87UUFDUCxNQUFNLENBQUMsd0JBQXdCLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDakQsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuICovXG5cbmltcG9ydCB7XG4gIGV4cGVjdCBhcyBjZGtFeHBlY3QsXG4gIGhhdmVSZXNvdXJjZUxpa2UsXG59IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQge1xuICBBbWF6b25MaW51eEdlbmVyYXRpb24sXG4gIEluc3RhbmNlLFxuICBJbnN0YW5jZVR5cGUsXG4gIE1hY2hpbmVJbWFnZSxcbiAgU2VjdXJpdHlHcm91cCxcbiAgVnBjLFxuICBXaW5kb3dzVmVyc2lvbixcbn0gZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBmc3ggZnJvbSAnQGF3cy1jZGsvYXdzLWZzeCc7XG5pbXBvcnQge1xuICBBcHAsXG4gIFN0YWNrLFxufSBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuaW1wb3J0IHtcbiAgTW91bnRhYmxlRnN4THVzdHJlLFxuICBNb3VudFBlcm1pc3Npb25zLFxufSBmcm9tICcuLi9saWInO1xuXG5pbXBvcnQge1xuICBlc2NhcGVUb2tlblJlZ2V4LFxufSBmcm9tICcuL3Rva2VuLXJlZ2V4LWhlbHBlcnMnO1xuXG5kZXNjcmliZSgnTW91bnRhYmxlRnN4THVzdHJlJywgKCkgPT4ge1xuICBsZXQgYXBwOiBBcHA7XG4gIGxldCBzdGFjazogU3RhY2s7XG4gIGxldCB2cGM6IFZwYztcbiAgbGV0IGZzOiBmc3guTHVzdHJlRmlsZVN5c3RlbTtcbiAgbGV0IGZzU2VjdXJpdHlHcm91cDogU2VjdXJpdHlHcm91cDtcbiAgbGV0IGluc3RhbmNlOiBJbnN0YW5jZTtcbiAgbGV0IGluc3RhbmNlU2VjdXJpdHlHcm91cDogU2VjdXJpdHlHcm91cDtcblxuICBiZWZvcmVFYWNoKCgpID0+IHtcbiAgICBhcHAgPSBuZXcgQXBwKCk7XG4gICAgc3RhY2sgPSBuZXcgU3RhY2soYXBwKTtcbiAgICB2cGMgPSBuZXcgVnBjKHN0YWNrLCAnVnBjJyk7XG4gICAgZnNTZWN1cml0eUdyb3VwID0gbmV3IFNlY3VyaXR5R3JvdXAoc3RhY2ssICdGU3hMU2VjdXJpdHlHcm91cCcsIHtcbiAgICAgIHZwYyxcbiAgICB9KTtcbiAgICBmcyA9IG5ldyBmc3guTHVzdHJlRmlsZVN5c3RlbShzdGFjaywgJ0ZTeEwnLCB7XG4gICAgICB2cGMsXG4gICAgICB2cGNTdWJuZXQ6IHZwYy5wcml2YXRlU3VibmV0c1swXSxcbiAgICAgIGx1c3RyZUNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgZGVwbG95bWVudFR5cGU6IGZzeC5MdXN0cmVEZXBsb3ltZW50VHlwZS5TQ1JBVENIXzEsXG4gICAgICB9LFxuICAgICAgc3RvcmFnZUNhcGFjaXR5R2lCOiAxMjAwLFxuICAgICAgc2VjdXJpdHlHcm91cDogZnNTZWN1cml0eUdyb3VwLFxuICAgIH0pO1xuICAgIGluc3RhbmNlU2VjdXJpdHlHcm91cCA9IG5ldyBTZWN1cml0eUdyb3VwKHN0YWNrLCAnSW5zdGFuY2VTZWN1cml0eUdyb3VwJywge1xuICAgICAgdnBjLFxuICAgIH0pO1xuICAgIGluc3RhbmNlID0gbmV3IEluc3RhbmNlKHN0YWNrLCAnSW5zdGFuY2UnLCB7XG4gICAgICB2cGMsXG4gICAgICBpbnN0YW5jZVR5cGU6IG5ldyBJbnN0YW5jZVR5cGUoJ3QzLnNtYWxsJyksXG4gICAgICBtYWNoaW5lSW1hZ2U6IE1hY2hpbmVJbWFnZS5sYXRlc3RBbWF6b25MaW51eCh7IGdlbmVyYXRpb246IEFtYXpvbkxpbnV4R2VuZXJhdGlvbi5BTUFaT05fTElOVVhfMiB9KSxcbiAgICAgIHNlY3VyaXR5R3JvdXA6IGluc3RhbmNlU2VjdXJpdHlHcm91cCxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnbW91bnRzIHdpdGggZGVmYXVsdHMnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBtb3VudCA9IG5ldyBNb3VudGFibGVGc3hMdXN0cmUoZnMsIHtcbiAgICAgIGZpbGVzeXN0ZW06IGZzLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIG1vdW50Lm1vdW50VG9MaW51eEluc3RhbmNlKGluc3RhbmNlLCB7XG4gICAgICBsb2NhdGlvbjogJy9tbnQvZnN4L2ZzMScsXG4gICAgfSk7XG4gICAgY29uc3QgdXNlckRhdGEgPSBpbnN0YW5jZS51c2VyRGF0YS5yZW5kZXIoKTtcblxuICAgIC8vIFRIRU5cbiAgICAvLyBNYWtlIHN1cmUgdGhlIGluc3RhbmNlIGhhcyBiZWVuIGdyYW50ZWQgaW5ncmVzcyB0byB0aGUgRlN4TCdzIHNlY3VyaXR5IGdyb3VwXG4gICAgY2RrRXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkVDMjo6U2VjdXJpdHlHcm91cEluZ3Jlc3MnLCB7XG4gICAgICBJcFByb3RvY29sOiAndGNwJyxcbiAgICAgIEZyb21Qb3J0OiA5ODgsXG4gICAgICBUb1BvcnQ6IDEwMjMsXG4gICAgICBTb3VyY2VTZWN1cml0eUdyb3VwSWQ6IHN0YWNrLnJlc29sdmUoaW5zdGFuY2VTZWN1cml0eUdyb3VwLnNlY3VyaXR5R3JvdXBJZCksXG4gICAgICBHcm91cElkOiBzdGFjay5yZXNvbHZlKGZzU2VjdXJpdHlHcm91cC5zZWN1cml0eUdyb3VwSWQpLFxuICAgIH0pKTtcbiAgICAvLyBNYWtlIHN1cmUgd2UgZG93bmxvYWQgdGhlIG1vdW50RnN4THVzdHJlIHNjcmlwdCBhc3NldCBidW5kbGVcbiAgICBjb25zdCBzM0NvcHkgPSAnYXdzIHMzIGNwIFxcJ3MzOi8vJHtUb2tlbltUT0tFTi5cXFxcZCtdfS8ke1Rva2VuW1RPS0VOLlxcXFxkK119JHtUb2tlbltUT0tFTi5cXFxcZCtdfVxcJyBcXCcvdG1wLyR7VG9rZW5bVE9LRU4uXFxcXGQrXX0ke1Rva2VuW1RPS0VOLlxcXFxkK119XFwnJztcbiAgICBleHBlY3QodXNlckRhdGEpLnRvTWF0Y2gobmV3IFJlZ0V4cChlc2NhcGVUb2tlblJlZ2V4KHMzQ29weSkpKTtcbiAgICBleHBlY3QodXNlckRhdGEpLnRvTWF0Y2gobmV3IFJlZ0V4cChlc2NhcGVUb2tlblJlZ2V4KCd1bnppcCAvdG1wLyR7VG9rZW5bVE9LRU4uXFxcXGQrXX0ke1Rva2VuW1RPS0VOLlxcXFxkK119JykpKTtcbiAgICAvLyBNYWtlIHN1cmUgd2UgaW5zdGFsbCB0aGUgTHVzdHJlIGNsaWVudFxuICAgIGV4cGVjdCh1c2VyRGF0YSkudG9NYXRjaCgnYmFzaCAuL2luc3RhbGxMdXN0cmVDbGllbnQuc2gnKTtcbiAgICAvLyBNYWtlIHN1cmUgd2UgZXhlY3V0ZSB0aGUgc2NyaXB0IHdpdGggdGhlIGNvcnJlY3QgYXJnc1xuICAgIGV4cGVjdCh1c2VyRGF0YSkudG9NYXRjaChuZXcgUmVnRXhwKGVzY2FwZVRva2VuUmVnZXgoJ2Jhc2ggLi9tb3VudEZzeEx1c3RyZS5zaCAke1Rva2VuW1RPS0VOLlxcXFxkK119IC9tbnQvZnN4L2ZzMSAke1Rva2VuW1RPS0VOLlxcXFxkK119IHJ3JykpKTtcbiAgfSk7XG5cbiAgdGVzdCgnYXNzZXJ0IExpbnV4LW9ubHknLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCB3aW5kb3dzSW5zdGFuY2UgPSBuZXcgSW5zdGFuY2Uoc3RhY2ssICdXaW5kb3dzSW5zdGFuY2UnLCB7XG4gICAgICB2cGMsXG4gICAgICBpbnN0YW5jZVR5cGU6IG5ldyBJbnN0YW5jZVR5cGUoJ3QzLnNtYWxsJyksXG4gICAgICBtYWNoaW5lSW1hZ2U6IE1hY2hpbmVJbWFnZS5sYXRlc3RXaW5kb3dzKFdpbmRvd3NWZXJzaW9uLldJTkRPV1NfU0VSVkVSXzIwMTlfRU5HTElTSF9GVUxMX1NRTF8yMDE3X1NUQU5EQVJEKSxcbiAgICB9KTtcbiAgICBjb25zdCBtb3VudCA9IG5ldyBNb3VudGFibGVGc3hMdXN0cmUoZnMsIHtcbiAgICAgIGZpbGVzeXN0ZW06IGZzLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICBtb3VudC5tb3VudFRvTGludXhJbnN0YW5jZSh3aW5kb3dzSW5zdGFuY2UsIHtcbiAgICAgICAgbG9jYXRpb246ICcvbW50L2ZzeC9mczEnLFxuICAgICAgICBwZXJtaXNzaW9uczogTW91bnRQZXJtaXNzaW9ucy5SRUFET05MWSxcbiAgICAgIH0pO1xuICAgIH0pLnRvVGhyb3dFcnJvcignVGFyZ2V0IGluc3RhbmNlIG11c3QgYmUgTGludXguJyk7XG4gIH0pO1xuXG4gIHRlc3QoJ3JlYWRvbmx5IG1vdW50JywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgbW91bnQgPSBuZXcgTW91bnRhYmxlRnN4THVzdHJlKGZzLCB7XG4gICAgICBmaWxlc3lzdGVtOiBmcyxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBtb3VudC5tb3VudFRvTGludXhJbnN0YW5jZShpbnN0YW5jZSwge1xuICAgICAgbG9jYXRpb246ICcvbW50L2ZzeC9mczEnLFxuICAgICAgcGVybWlzc2lvbnM6IE1vdW50UGVybWlzc2lvbnMuUkVBRE9OTFksXG4gICAgfSk7XG4gICAgY29uc3QgdXNlckRhdGEgPSBpbnN0YW5jZS51c2VyRGF0YS5yZW5kZXIoKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3QodXNlckRhdGEpLnRvTWF0Y2gobmV3IFJlZ0V4cChlc2NhcGVUb2tlblJlZ2V4KCdtb3VudEZzeEx1c3RyZS5zaCAke1Rva2VuW1RPS0VOLlxcXFxkK119IC9tbnQvZnN4L2ZzMSAke1Rva2VuW1RPS0VOLlxcXFxkK119IHInKSkpO1xuICB9KTtcblxuICB0ZXN0KCdleHRyYSBtb3VudCBvcHRpb25zJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgbW91bnQgPSBuZXcgTW91bnRhYmxlRnN4THVzdHJlKGZzLCB7XG4gICAgICBmaWxlc3lzdGVtOiBmcyxcbiAgICAgIGV4dHJhTW91bnRPcHRpb25zOiBbXG4gICAgICAgICdvcHRpb24xJyxcbiAgICAgICAgJ29wdGlvbjInLFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBtb3VudC5tb3VudFRvTGludXhJbnN0YW5jZShpbnN0YW5jZSwge1xuICAgICAgbG9jYXRpb246ICcvbW50L2ZzeC9mczEnLFxuICAgIH0pO1xuICAgIGNvbnN0IHVzZXJEYXRhID0gaW5zdGFuY2UudXNlckRhdGEucmVuZGVyKCk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHVzZXJEYXRhKS50b01hdGNoKG5ldyBSZWdFeHAoZXNjYXBlVG9rZW5SZWdleCgnbW91bnRGc3hMdXN0cmUuc2ggJHtUb2tlbltUT0tFTi5cXFxcZCtdfSAvbW50L2ZzeC9mczEgJHtUb2tlbltUT0tFTi5cXFxcZCtdfSBydyxvcHRpb24xLG9wdGlvbjInKSkpO1xuICB9KTtcblxuICB0ZXN0KCdhc3NldCBpcyBzaW5nbGV0b24nLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBtb3VudDEgPSBuZXcgTW91bnRhYmxlRnN4THVzdHJlKGZzLCB7XG4gICAgICBmaWxlc3lzdGVtOiBmcyxcbiAgICB9KTtcbiAgICBjb25zdCBtb3VudDIgPSBuZXcgTW91bnRhYmxlRnN4THVzdHJlKGZzLCB7XG4gICAgICBmaWxlc3lzdGVtOiBmcyxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBtb3VudDEubW91bnRUb0xpbnV4SW5zdGFuY2UoaW5zdGFuY2UsIHtcbiAgICAgIGxvY2F0aW9uOiAnL21udC9mc3gvZnMxJyxcbiAgICB9KTtcbiAgICBtb3VudDIubW91bnRUb0xpbnV4SW5zdGFuY2UoaW5zdGFuY2UsIHtcbiAgICAgIGxvY2F0aW9uOiAnL21udC9mc3gvZnMxJyxcbiAgICB9KTtcbiAgICBjb25zdCB1c2VyRGF0YSA9IGluc3RhbmNlLnVzZXJEYXRhLnJlbmRlcigpO1xuICAgIGNvbnN0IHMzQ29weSA9ICdhd3MgczMgY3AgXFwnczM6Ly8ke1Rva2VuW1RPS0VOLlxcXFxkK119LyR7VG9rZW5bVE9LRU4uXFxcXGQrXX0ke1Rva2VuW1RPS0VOLlxcXFxkK119XFwnJztcbiAgICBjb25zdCByZWdleCA9IG5ldyBSZWdFeHAoZXNjYXBlVG9rZW5SZWdleChzM0NvcHkpLCAnZycpO1xuICAgIGNvbnN0IG1hdGNoZXMgPSB1c2VyRGF0YS5tYXRjaChyZWdleCkgPz8gW107XG5cbiAgICAvLyBUSEVOXG4gICAgLy8gVGhlIHNvdXJjZSBvZiB0aGUgYXNzZXQgY29weSBzaG91bGQgYmUgaWRlbnRpY2FsIGZyb20gbW91bnQxICYgbW91bnQyXG4gICAgZXhwZWN0KG1hdGNoZXMpLnRvSGF2ZUxlbmd0aCgyKTtcbiAgICBleHBlY3QobWF0Y2hlc1swXSkudG9CZShtYXRjaGVzWzFdKTtcbiAgfSk7XG5cbiAgdGVzdCgnYXBwbGllcyBMdXN0cmUgZmlsZXNldCcsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGZpbGVzZXQgPSAnZmlsZXNldCc7XG4gICAgY29uc3QgbW91bnQgPSBuZXcgTW91bnRhYmxlRnN4THVzdHJlKGZzLCB7XG4gICAgICBmaWxlc3lzdGVtOiBmcyxcbiAgICAgIGZpbGVzZXQsXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgbW91bnQubW91bnRUb0xpbnV4SW5zdGFuY2UoaW5zdGFuY2UsIHtcbiAgICAgIGxvY2F0aW9uOiAnL21udC9mc3gvZnMxJyxcbiAgICB9KTtcbiAgICBjb25zdCB1c2VyRGF0YSA9IGluc3RhbmNlLnVzZXJEYXRhLnJlbmRlcigpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdCh1c2VyRGF0YSkudG9NYXRjaChuZXcgUmVnRXhwKGVzY2FwZVRva2VuUmVnZXgoYGJhc2ggLi9tb3VudEZzeEx1c3RyZS5zaCBcXCR7VG9rZW5bVE9LRU4uXFxcXGQrXX0gL21udC9mc3gvZnMxIFxcJHtUb2tlbltUT0tFTi5cXFxcZCtdfS8ke2ZpbGVzZXR9IHJ3YCkpKTtcbiAgfSk7XG5cbiAgdGVzdCgnLnVzZXNVc2VyUG9zaXhQZXJtaXNzaW9ucygpIGlzIHRydWUnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBtb3VudCA9IG5ldyBNb3VudGFibGVGc3hMdXN0cmUoZnMsIHtcbiAgICAgIGZpbGVzeXN0ZW06IGZzLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHVzZXNVc2VyUG9zaXhQZXJtaXNzaW9ucyA9IG1vdW50LnVzZXNVc2VyUG9zaXhQZXJtaXNzaW9ucygpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdCh1c2VzVXNlclBvc2l4UGVybWlzc2lvbnMpLnRvRXF1YWwodHJ1ZSk7XG4gIH0pO1xufSk7XG4iXX0=