"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.X509CertificatePkcs12 = exports.X509CertificatePem = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto = require("crypto");
const path_1 = require("path");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const lambdaLayerVersionArns_1 = require("../../lambdas/lambdaLayerVersionArns");
class X509CertificateBase extends core_1.Construct {
    constructor(scope, id, props) {
        var _c;
        super(scope, id);
        this.database = new aws_dynamodb_1.Table(this, 'Table', {
            partitionKey: { name: 'PhysicalId', type: aws_dynamodb_1.AttributeType.STRING },
            sortKey: { name: 'CustomResource', type: aws_dynamodb_1.AttributeType.STRING },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            encryption: aws_dynamodb_1.TableEncryption.AWS_MANAGED,
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
        });
        this.passphrase = new aws_secretsmanager_1.Secret(this, 'Passphrase', {
            description: `Passphrase for the private key of the X509Certificate ${core_1.Names.uniqueId(this)}`,
            encryptionKey: props.encryptionKey,
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
            },
        });
        const region = core_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        /*
         * We cannot make this a singleton function; doing so would create circular references in the lambda role (to sign
         * a cert we need a cert that this lambda generated).
         */
        this.lambdaFunc = new aws_lambda_1.Function(this, 'Generator', {
            description: `Used by a X509Certificate ${core_1.Names.uniqueId(this)} to generate certificates.`,
            code: props.lambdaCode,
            environment: {
                DATABASE: this.database.tableName,
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            layers: [openSslLayer],
            handler: props.lambdaHandler,
            timeout: core_1.Duration.seconds(90),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        this.database.grantReadWriteData(this.lambdaFunc);
        this.database.grant(this.lambdaFunc, 'dynamodb:DescribeTable');
        (_c = props.encryptionKey) === null || _c === void 0 ? void 0 : _c.grantEncrypt(this.lambdaFunc);
        this.passphrase.grantRead(this.lambdaFunc);
        const uniqueValue = crypto.createHash('md5').update(core_1.Names.uniqueId(this)).digest('hex');
        this.uniqueTag = new core_1.Tag(`X509SecretGrant-${uniqueValue.slice(0, 8).toUpperCase()}`, uniqueValue);
        const tagCondition = {};
        tagCondition[`secretsmanager:ResourceTag/${this.uniqueTag.key}`] = this.uniqueTag.value;
        this.lambdaFunc.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'secretsmanager:CreateSecret',
                'secretsmanager:DeleteSecret',
                'secretsmanager:TagResource',
                'secretsmanager:PutSecretValue',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
    }
}
/**
 * A Construct that uses a Lambda to generate an X.509 certificate and then saves the certificate's components into Secrets. On an update, if any properties of the construct are changed, then a new certificate will be generated. When the Stack is destroyed or the Construct is removed, the Secrets will all be deleted. An X.509 certificate is comprised of the certificate, a certificate chain with the chain of signing certificates (if any), and a private key that is password protected by a randomly generated passphrase.
 *
 * Cost:
 * The cost of four AWS SecretsManager Secrets in the deployed region.
 * The other resources created by this construct have negligible ongoing costs.
 *
 * ![architecture diagram](/diagrams/core/X509CertificatePem.svg)
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the Custom Resource.
 * - Secrets - 4 in total, for the certificate, it's private key, the passphrase to the key, and the cert chain.
 * - Lambda Function, with role - Used to create/update/delete the Custom Resource
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - Access to the AWS SecretsManager Secrets that are created by this construct should be tightly restricted
 *    to only the principal(s) that require access.
 *
 * @stability stable
 */
class X509CertificatePem extends X509CertificateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        super(scope, id, {
            lambdaCode: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            lambdaHandler: 'x509-certificate.generate',
            encryptionKey: props.encryptionKey,
        });
        if (((_c = props.validFor) !== null && _c !== void 0 ? _c : 1) < 1 && !core_1.Token.isUnresolved(props.validFor)) {
            core_1.Annotations.of(this).addError('Certificates must be valid for at least one day.');
        }
        (_d = props.signingCertificate) === null || _d === void 0 ? void 0 : _d.cert.grantRead(this.lambdaFunc);
        (_e = props.signingCertificate) === null || _e === void 0 ? void 0 : _e.key.grantRead(this.lambdaFunc);
        (_f = props.signingCertificate) === null || _f === void 0 ? void 0 : _f.passphrase.grantRead(this.lambdaFunc);
        (_h = (_g = props.signingCertificate) === null || _g === void 0 ? void 0 : _g.certChain) === null || _h === void 0 ? void 0 : _h.grantRead(this.lambdaFunc);
        const signingCertificate = props.signingCertificate
            ? {
                Cert: props.signingCertificate.cert.secretArn,
                Key: props.signingCertificate.key.secretArn,
                Passphrase: props.signingCertificate.passphrase.secretArn,
                CertChain: props.signingCertificate.certChain ? props.signingCertificate.certChain.secretArn : '',
            }
            : undefined;
        const properties = {
            DistinguishedName: {
                CN: props.subject.cn,
                O: (_j = props.subject.o) !== null && _j !== void 0 ? _j : 'AWS',
                OU: (_k = props.subject.ou) !== null && _k !== void 0 ? _k : 'Thinkbox',
            },
            Passphrase: this.passphrase.secretArn,
            Secret: {
                NamePrefix: this.node.path,
                Description: this.node.path,
                EncryptionKey: (_l = props.encryptionKey) === null || _l === void 0 ? void 0 : _l.keyArn,
                Tags: [
                    {
                        Key: this.uniqueTag.key,
                        Value: this.uniqueTag.value,
                    },
                ],
            },
            SigningCertificate: signingCertificate,
            CertificateValidFor: (_m = props.validFor) === null || _m === void 0 ? void 0 : _m.toString(),
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: this.lambdaFunc.functionArn,
            properties,
            resourceType: 'Custom::RFDK_X509Generator',
        });
        if (this.lambdaFunc.role) {
            // There's a race on update where this resource could execute before the role has updated.
            resource.node.addDependency(this.lambdaFunc.role);
        }
        this.cert = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Cert', {
            secretCompleteArn: core_1.Token.asString(resource.getAtt('Cert')),
            encryptionKey: props.encryptionKey,
        });
        this.key = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Key', {
            secretCompleteArn: core_1.Token.asString(resource.getAtt('Key')),
            encryptionKey: props.encryptionKey,
        });
        // We'll only have a chain if we used a ca to sign this cert. We cannot check for certChainResource being an empty
        // string because it is an unresolved token at this point.
        if (signingCertificate) {
            const certChainResource = resource.getAtt('CertChain');
            this.certChain = certChainResource
                ? aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'CertChain', {
                    secretCompleteArn: core_1.Token.asString(certChainResource),
                    encryptionKey: props.encryptionKey,
                })
                : undefined;
        }
    }
    /**
     * Grant read permissions for the certificate.
     *
     * @stability stable
     */
    grantCertRead(grantee) {
        var _c;
        const result = this.cert.grantRead(grantee);
        (_c = this.certChain) === null || _c === void 0 ? void 0 : _c.grantRead(grantee);
        return result;
    }
    /**
     * Grant read permissions for the certificate, key, and passphrase.
     *
     * @stability stable
     */
    grantFullRead(grantee) {
        var _c;
        const result = this.cert.grantRead(grantee);
        (_c = this.certChain) === null || _c === void 0 ? void 0 : _c.grantRead(grantee);
        this.key.grantRead(grantee);
        this.passphrase.grantRead(grantee);
        return result;
    }
}
exports.X509CertificatePem = X509CertificatePem;
_a = JSII_RTTI_SYMBOL_1;
X509CertificatePem[_a] = { fqn: "aws-rfdk.X509CertificatePem", version: "0.40.0" };
/**
 * This Construct will generate a PKCS #12 file from an X.509 certificate in PEM format. The PEM certificate must be provided through an instance of the X509CertificatePem Construct. A Lambda Function is used to do the conversion and the result is stored in a Secret. The PKCS #12 file is password protected with a passphrase that is randomly generated and stored in a Secret.
 *
 * ![architecture diagram](/diagrams/core/X509CertificatePkcs12.svg)
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the CustomResource.
 * - Secrets - 2 in total, The binary of the PKCS #12 certificate and its passphrase.
 * - Lambda Function, with role - Used to create/update/delete the CustomResource.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - Access to the AWS SecretsManager Secrets that are created by this construct should be tightly restricted
 *    to only the principal(s) that require access.
 *
 * @stability stable
 */
class X509CertificatePkcs12 extends X509CertificateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c;
        super(scope, id, {
            lambdaCode: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            lambdaHandler: 'x509-certificate.convert',
            encryptionKey: props.encryptionKey,
        });
        props.sourceCertificate.grantFullRead(this.lambdaFunc);
        const properties = {
            Passphrase: this.passphrase.secretArn,
            Secret: {
                NamePrefix: this.node.path,
                Description: this.node.path,
                EncryptionKey: (_c = props.encryptionKey) === null || _c === void 0 ? void 0 : _c.keyArn,
                Tags: [
                    {
                        Key: this.uniqueTag.key,
                        Value: this.uniqueTag.value,
                    },
                ],
            },
            Certificate: {
                Cert: props.sourceCertificate.cert.secretArn,
                CertChain: props.sourceCertificate.certChain ? props.sourceCertificate.certChain.secretArn : '',
                Key: props.sourceCertificate.key.secretArn,
                Passphrase: props.sourceCertificate.passphrase.secretArn,
            },
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: this.lambdaFunc.functionArn,
            properties,
            resourceType: 'Custom::RFDK_X509_PKCS12',
        });
        this.cert = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Cert', {
            secretCompleteArn: core_1.Token.asString(resource.getAtt('Cert')),
            encryptionKey: props.encryptionKey,
        });
    }
}
exports.X509CertificatePkcs12 = X509CertificatePkcs12;
_b = JSII_RTTI_SYMBOL_1;
X509CertificatePkcs12[_b] = { fqn: "aws-rfdk.X509CertificatePkcs12", version: "0.40.0" };
//# sourceMappingURL=data:application/json;base64,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