"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountableBlockVolume = exports.BlockVolumeFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const mount_permissions_helper_1 = require("./mount-permissions-helper");
/**
 * Block format options for formatting a blank/new BlockVolume.
 *
 * @stability stable
 */
var BlockVolumeFormat;
(function (BlockVolumeFormat) {
    BlockVolumeFormat["EXT3"] = "ext3";
    BlockVolumeFormat["EXT4"] = "ext4";
    BlockVolumeFormat["XFS"] = "xfs";
})(BlockVolumeFormat = exports.BlockVolumeFormat || (exports.BlockVolumeFormat = {}));
/**
 * This class encapsulates scripting that can be used by an instance to mount, format, and resize an Amazon Elastic Block Store (EBS) Volume to itself when it is launched.
 *
 * The scripting is added to
 * the instance's UserData to be run when the instance is first launched.
 *
 * The script that is employed by this class will:
 * 1) Attach the volume to this instance if it is not already attached;
 * 2) Format the block volume to the filesystem format that's passed as an argument to this script but,
 *    **ONLY IF** the filesystem has no current format;
 * 3) Mount the volume to the given mount point with the given mount options; and
 * 4) Resize the filesystem on the volume if the volume is larger than the formatted filesystem size.
 *
 * Note: This does **NOT** support multiple partitions on the EBS Volume; the script will exit with a failure code
 * when it detects multiple partitions on the device. It is expected that the whole block device is a single partition.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 * @remark If using this script with an instance within an AWS Auto Scaling Group (ASG) and you resize
 * the EBS volume, then you can terminate the instance to let the ASG replace the instance and benefit
 * from the larger volume size when this script resizes the filesystem on instance launch.
 */
class MountableBlockVolume {
    /**
     * @stability stable
     */
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
    }
    /**
     * Mount the filesystem to the given instance at instance startup.
     *
     * This is accomplished by
     * adding scripting to the UserData of the instance to mount the filesystem on startup.
     * If required, the instance's security group is granted ingress to the filesystem's security
     * group on the required ports.
     *
     * @stability stable
     * @inheritdoc true
     */
    mountToLinuxInstance(target, mount) {
        var _b;
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        this.grantRequiredPermissions(target);
        const mountScriptAsset = this.mountAssetSingleton(target);
        mountScriptAsset.grantRead(target.grantPrincipal);
        const mountScriptZip = target.userData.addS3DownloadCommand({
            bucket: mountScriptAsset.bucket,
            bucketKey: mountScriptAsset.s3ObjectKey,
        });
        const mountDir = path.posix.normalize(mount.location);
        const mountOptions = [mount_permissions_helper_1.MountPermissionsHelper.toLinuxMountOption(mount.permissions)];
        if (this.props.extraMountOptions) {
            mountOptions.push(...this.props.extraMountOptions);
        }
        const mountOptionsStr = mountOptions.join(',');
        const volumeFormat = (_b = this.props.volumeFormat) !== null && _b !== void 0 ? _b : BlockVolumeFormat.XFS;
        target.userData.addCommands('TMPDIR=$(mktemp -d)', 'pushd "$TMPDIR"', `unzip ${mountScriptZip}`, `bash ./mountEbsBlockVolume.sh ${this.props.blockVolume.volumeId} ${volumeFormat} ${mountDir} ${mountOptionsStr} ""`, 'popd', `rm -f ${mountScriptZip}`);
    }
    /**
     * Returns whether the mounted file-system evaluates the UID/GID of the system user accessing the file-system.
     *
     * Some network file-systems provide features to fix a UID/GID for all access to the mounted file-system and ignore
     * the system user accessing the file. If this is the case, an implementing class must indicate this in the return
     * value.
     *
     * @stability stable
     * @inheritdoc true
     */
    usesUserPosixPermissions() {
        return true;
    }
    /**
     * Grant required permissions to the target.
     *
     * The mounting script requires two permissions:
     * 1) Permission to describe the volume
     * 2) Permission to attach the volume
     *
     * @stability stable
     */
    grantRequiredPermissions(target) {
        // Volume.grantAttachVolumeByResourceTag() requires that the target be a construct; it adds a tag to the construct.
        // So, we fail if we're given something that is not compatible.
        if (!core_1.Construct.isConstruct(target)) {
            throw new Error('Target instance must be a construct. It cannot be constructed from attributes.');
        }
        // See: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html
        // Accessed July 2020
        // ec2:DescribeVolumes does not support resource or condition constraints.
        target.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'ec2:DescribeVolumes',
            ],
            resources: ['*'],
        }));
        this.props.blockVolume.grantAttachVolumeByResourceTag(target.grantPrincipal, [target]);
    }
    /**
     * Fetch the Asset singleton for the Volume mounting scripts, or generate it if needed.
     *
     * @stability stable
     */
    mountAssetSingleton(scope) {
        var _b;
        const stack = core_1.Stack.of(scope);
        const uuid = '01ca4aa6-d440-4f83-84d8-80a5a21fd0e3';
        const uniqueId = 'MountableBlockVolumeAsset' + uuid.replace(/[-]/g, '');
        return (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'bash'),
            exclude: ['**/*', '!mountEbsBlockVolume.sh', '!metadataUtilities.sh', '!ec2-certificates.crt'],
        });
    }
}
exports.MountableBlockVolume = MountableBlockVolume;
_a = JSII_RTTI_SYMBOL_1;
MountableBlockVolume[_a] = { fqn: "aws-rfdk.MountableBlockVolume", version: "0.40.0" };
//# sourceMappingURL=data:application/json;base64,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