"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbInstaller = exports.MongoDbSsplLicenseAcceptance = exports.MongoDbVersion = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
/**
 * Versions of MongoDB Community Edition that the {@link MongoDbInstaller} is
 * able to install.
 */
var MongoDbVersion;
(function (MongoDbVersion) {
    MongoDbVersion["COMMUNITY_3_6"] = "3.6";
})(MongoDbVersion = exports.MongoDbVersion || (exports.MongoDbVersion = {}));
/**
 * Choices for signifying the user's stance on the terms of the SSPL.
 *
 * See: https://www.mongodb.com/licensing/server-side-public-license
 *
 * @stability stable
 */
var MongoDbSsplLicenseAcceptance;
(function (MongoDbSsplLicenseAcceptance) {
    MongoDbSsplLicenseAcceptance[MongoDbSsplLicenseAcceptance["USER_REJECTS_SSPL"] = 0] = "USER_REJECTS_SSPL";
    MongoDbSsplLicenseAcceptance[MongoDbSsplLicenseAcceptance["USER_ACCEPTS_SSPL"] = 1] = "USER_ACCEPTS_SSPL";
})(MongoDbSsplLicenseAcceptance = exports.MongoDbSsplLicenseAcceptance || (exports.MongoDbSsplLicenseAcceptance = {}));
/**
 * This class provides a mechanism to install a version of MongoDB Community Edition during the initial launch of an instance.
 *
 * MongoDB is installed from the official sources using the system
 * package manger (yum). It installs the mongodb-org metapackage which will install the following packages:
 * 1) mongodb-org-mongos;
 * 2) mongodb-org-server;
 * 3) mongodb-org-shell; and
 * 4) mongodb-org-tools.
 *
 * Successful installation of MongoDB with this class requires:
 * 1) Explicit acceptance of the terms of the SSPL license, under which MongoDB is distributed; and
 * 2) The instance on which the installation is being performed is in a subnet that can access
 * the official MongoDB sites: https://repo.mongodb.org/ and https://www.mongodb.org
 *
 * Resources Deployed
 * ------------------------
 * - A CDK Asset package containing the installation scripts is deployed to your CDK staging bucket.
 *
 * Security Considerations
 * ------------------------
 * - Since this class installs MongoDB from official sources dynamically during instance start-up, it is succeptable
 *    to an attacker compromising the official MongoDB Inc. distribution channel for MongoDB. Such a compromise may
 *    result in the installation of unauthorized MongoDB binaries. Executing this attack would require an attacker
 *    compromise both the official installation packages and the MongoDB Inc. gpg key with which they are signed.
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 */
class MongoDbInstaller {
    /**
     * @stability stable
     */
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
        // Legal requirement: Users of this class must agree to the terms of the SSPL, without exception.
        // Developer note: It is a legal requirement that the default be USER_REJECTS_SSPL, so this check
        // must throw an error for every value except USER_ACCEPTS_SSPL.
        if (props.userSsplAcceptance !== MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL) {
            throw new Error(MongoDbInstaller.SSPL_ACCEPT_MESSAGE);
        }
    }
    /**
     * Install MongoDB to the given instance at instance startup.
     *
     * This is accomplished by
     * adding scripting to the instance's UserData to install MongoDB.
     *
     * Notes:
     * 1) The instance on which the installation is being performed must be in a subnet that can access
     * the official MongoDB sites: https://repo.mongodb.org/ and https://www.mongodb.org; and
     * 2) At this time, this method only supports installation onto instances that are running an operating system
     * that is compatible with x86-64 RedHat 7 -- this includes Amazon Linux 2, RedHat 7, and CentOS 7.
     *
     * @param target The target instance onto which to install MongoDB.
     * @stability stable
     */
    installOnLinuxInstance(target) {
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        const installerScriptAsset = this.installerAssetSingleton();
        installerScriptAsset.grantRead(target.grantPrincipal);
        const installerScript = target.userData.addS3DownloadCommand({
            bucket: installerScriptAsset.bucket,
            bucketKey: installerScriptAsset.s3ObjectKey,
        });
        target.userData.addCommands(`bash ${installerScript}`);
    }
    /**
     * Fetch the Asset singleton for the installation script, or generate it if needed.
     *
     * @stability stable
     */
    installerAssetSingleton() {
        var _b;
        const stack = core_1.Stack.of(this.scope);
        const uuid = '5b141ac9-fde5-45d8-9961-b7108fb3b18a';
        const uniqueId = 'MongoDbInstallerAsset' + uuid.replace(/[-]/g, '');
        return (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'mongodb', this.props.version, 'installMongoDb.sh'),
        });
    }
}
exports.MongoDbInstaller = MongoDbInstaller;
_a = JSII_RTTI_SYMBOL_1;
MongoDbInstaller[_a] = { fqn: "aws-rfdk.MongoDbInstaller", version: "0.40.0" };
/**
 * The SSPL licensing message that is presented to the user if they create an instance of
 * this class without explicitly accepting the SSPL.
 *
 * Note to developers: The text of this string is a legal requirement, and must not be altered
 * witout approval.
 */
MongoDbInstaller.SSPL_ACCEPT_MESSAGE = `
The MongoDbInstaller will install MongoDB Community Edition onto one or more EC2 instances.

MongoDB is provided by MongoDB Inc. under the SSPL license. By installing MongoDB, you are agreeing to the terms of this license.
Follow the link below to read the terms of the SSPL license.
https://www.mongodb.com/licensing/server-side-public-license

By using the MongoDbInstaller to install MongoDB you agree to the terms of the SSPL license.

Please set the userSsplAcceptance property to USER_ACCEPTS_SSPL to signify your acceptance of the terms of the SSPL license.
`;
//# sourceMappingURL=data:application/json;base64,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