"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ImportedAcmCertificate = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto = require("crypto");
const path_1 = require("path");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lambdaLayerVersionArns_1 = require("../../lambdas/lambdaLayerVersionArns");
/**
 * A Construct that creates an AWS CloudFormation Custom Resource that models a certificate that is imported into AWS Certificate Manager (ACM).
 *
 * It uses an AWS Lambda Function to extract the certificate from Secrets in AWS SecretsManager
 * and then import it into ACM. The interface is intended to be used with the {@link X509CertificatePem} Construct.
 *
 * ![architecture diagram](/diagrams/core/ImportedAcmCertificate.svg)
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the Custom Resource.
 * - An AWS Lambda Function, with IAM Role - Used to create/update/delete the Custom Resource.
 * - AWS Certificate Manager Certificate - Created by the Custom Resource.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda for this construct also has broad IAM permissions to delete any Certificate that is stored
 *    in AWS Certificate Manager. You should not grant any additional actors/principals the ability to modify or
 *    execute this Lambda.
 *
 * @stability stable
 */
class ImportedAcmCertificate extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        this.stack = core_1.Stack.of(this);
        this.env = {
            account: this.stack.account,
            region: this.stack.region,
        };
        this.database = new aws_dynamodb_1.Table(this, 'Table', {
            partitionKey: { name: 'PhysicalId', type: aws_dynamodb_1.AttributeType.STRING },
            sortKey: { name: 'CustomResource', type: aws_dynamodb_1.AttributeType.STRING },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            encryption: aws_dynamodb_1.TableEncryption.AWS_MANAGED,
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
        });
        const region = core_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        const lambda = new aws_lambda_1.SingletonFunction(this, 'AcmImporter', {
            uuid: ImportedAcmCertificate.IMPORTER_UUID,
            code: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            handler: 'x509-certificate.importCert',
            environment: {
                DATABASE: this.database.tableName,
                DEBUG: 'false',
            },
            layers: [openSslLayer],
            retryAttempts: 0,
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            timeout: core_1.Duration.minutes(5),
        });
        this.database.grantReadWriteData(lambda);
        this.database.grant(lambda, 'dynamodb:DescribeTable');
        props.cert.grantRead(lambda);
        props.key.grantRead(lambda);
        props.passphrase.grantRead(lambda);
        (_b = props.certChain) === null || _b === void 0 ? void 0 : _b.grantRead(lambda);
        (_c = props.encryptionKey) === null || _c === void 0 ? void 0 : _c.grantDecrypt(lambda);
        const uniqueValue = crypto.createHash('md5').update(core_1.Names.uniqueId(this)).digest('hex');
        this.uniqueTag = new core_1.Tag(`AcmCertImport-${uniqueValue.slice(0, 8).toUpperCase()}`, uniqueValue);
        const tagCondition = {};
        tagCondition[`aws:RequestTag/${this.uniqueTag.key}`] = this.uniqueTag.value;
        lambda.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'acm:AddTagsToCertificate',
                'acm:ImportCertificate',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // GetCertificate and DeleteCertificate don't currently support and conditions, so we have to give a broader policy
        // on them for now.
        // See https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanager.html#awscertificatemanager-aws_TagKeys
        // for the condition keys currently available on ACM actions.
        lambda.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'acm:DeleteCertificate',
                'acm:DescribeCertificate',
                'acm:GetCertificate',
            ],
            resources: ['*'],
        }));
        const properties = {
            X509CertificatePem: {
                Cert: props.cert.secretArn,
                Key: props.key.secretArn,
                Passphrase: props.passphrase.secretArn,
                CertChain: props.certChain ? props.certChain.secretArn : '',
            },
            Tags: [
                {
                    Key: this.uniqueTag.key,
                    Value: this.uniqueTag.value,
                },
                { Key: 'Name',
                    Value: this.uniqueTag.value,
                },
            ],
        };
        this.resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: lambda.functionArn,
            properties,
            resourceType: 'Custom::RFDK_AcmImportedCertificate',
        });
        this.certificateArn = core_1.Token.asString(this.resource.getAtt('CertificateArn'));
    }
    /**
     * Apply a removal policy to the custom resource that represents the certificate imported into ACM.
     *
     * @stability stable
     */
    applyRemovalPolicy(policy) {
        this.resource.applyRemovalPolicy(policy);
    }
    /**
     * Return the DaysToExpiry metric for this AWS Certificate Manager Certificate.
     *
     * By default, this is the minimum value over 1 day.
     *
     * This metric is no longer emitted once the certificate has effectively
     * expired, so alarms configured on this metric should probably treat missing
     * data as "breaching".
     *
     * @stability stable
     * @inheritdoc true
     */
    metricDaysToExpiry(props) {
        const certLookupNode = this.node.tryFindChild(ImportedAcmCertificate.CERT_LOOKUP_CONSTRUCT_ID);
        let certLookup;
        /* istanbul ignore next */
        if (certLookupNode) {
            certLookup = certLookupNode;
        }
        else {
            certLookup = aws_certificatemanager_1.Certificate.fromCertificateArn(this, ImportedAcmCertificate.CERT_LOOKUP_CONSTRUCT_ID, this.certificateArn);
        }
        return certLookup.metricDaysToExpiry(props);
    }
}
exports.ImportedAcmCertificate = ImportedAcmCertificate;
_a = JSII_RTTI_SYMBOL_1;
ImportedAcmCertificate[_a] = { fqn: "aws-rfdk.ImportedAcmCertificate", version: "0.40.0" };
ImportedAcmCertificate.IMPORTER_UUID = '2d20d8f2-7b84-444e-b738-c75b499a9eaa';
ImportedAcmCertificate.CERT_LOOKUP_CONSTRUCT_ID = 'CertificateLookup';
//# sourceMappingURL=data:application/json;base64,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