"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeploymentInstance = void 0;
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const runtime_info_1 = require("./runtime-info");
/**
 * Deploys an instance that runs its user data on deployment, waits for that user data to succeed, and optionally
 * terminates itself afterwards.
 *
 * Resources Deployed
 * ------------------------
 * - Auto Scaling Group (ASG) with max capacity of 1 instance.
 * - IAM instance profile, IAM role, and IAM policy
 * - An Amazon CloudWatch log group that contains the instance cloud-init logs
 * - A Lambda Function to fetch and existing Log Group or create a new one
 * - IAM role and policy for the Lambda Function
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *   is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *   performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *   bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *   environments.
 */
class DeploymentInstance extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id);
        this.asg = new aws_autoscaling_1.AutoScalingGroup(this, 'ASG', {
            instanceType: (_a = props.instanceType) !== null && _a !== void 0 ? _a : aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.SMALL),
            keyName: props.keyName,
            machineImage: (_b = props.machineImage) !== null && _b !== void 0 ? _b : aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            minCapacity: 1,
            maxCapacity: 1,
            securityGroup: props.securityGroup,
            signals: aws_autoscaling_1.Signals.waitForAll({
                timeout: (_c = props.executionTimeout) !== null && _c !== void 0 ? _c : DeploymentInstance.DEFAULT_EXECUTION_TIMEOUT,
            }),
            updatePolicy: aws_autoscaling_1.UpdatePolicy.replacingUpdate(),
            vpc: props.vpc,
            vpcSubnets: (_d = props.vpcSubnets) !== null && _d !== void 0 ? _d : {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
        });
        this.node.defaultChild = this.asg;
        this.connections = this.asg.connections;
        const logGroupName = (_e = props.logGroupName) !== null && _e !== void 0 ? _e : id;
        this.configureCloudWatchAgent(this.asg, logGroupName, props.logGroupProps);
        if ((_f = props.selfTerminate) !== null && _f !== void 0 ? _f : true) {
            this.configureSelfTermination();
        }
        this.asg.userData.addSignalOnExitCommand(this.asg);
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Make the execution of the instance dependent upon another construct
     *
     * @param dependency The construct that should be dependended upon
     */
    addExecutionDependency(dependency) {
        if (core_1.Construct.isConstruct(dependency)) {
            this.asg.node.defaultChild.node.addDependency(dependency);
        }
    }
    /**
     * @inheritdoc
     */
    get osType() {
        return this.asg.osType;
    }
    /**
     * @inheritdoc
     */
    get userData() {
        return this.asg.userData;
    }
    /**
     * @inheritdoc
     */
    get grantPrincipal() {
        return this.asg.grantPrincipal;
    }
    /**
     * Adds UserData commands to configure the CloudWatch Agent running on the deployment instance.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *   - The cloud-init log
     *
     * @param asg The auto-scaling group
     * @param groupName The name of the Log Group, or suffix of the Log Group if `logGroupProps.logGroupPrefix` is
     *                  specified
     * @param logGroupProps The properties for LogGroupFactory to create or fetch the log group
     */
    configureCloudWatchAgent(asg, groupName, logGroupProps) {
        var _a;
        const prefix = (_a = logGroupProps === null || logGroupProps === void 0 ? void 0 : logGroupProps.logGroupPrefix) !== null && _a !== void 0 ? _a : DeploymentInstance.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = _1.LogGroupFactory.createOrFetch(this, 'DeploymentInstanceLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(asg);
        const cloudWatchConfigurationBuilder = new _1.CloudWatchConfigBuilder(DeploymentInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        new _1.CloudWatchAgent(this, 'CloudWatchAgent', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: asg,
        });
    }
    configureSelfTermination() {
        // Add a policy to the ASG that allows it to modify itself. We cannot add the ASG name in resources as it will cause
        // cyclic dependency. Hence, using Condition Keys
        const tagCondition = {};
        tagCondition[`autoscaling:ResourceTag/${DeploymentInstance.ASG_TAG_KEY}`] = core_1.Names.uniqueId(this);
        core_1.Tags.of(this.asg).add(DeploymentInstance.ASG_TAG_KEY, core_1.Names.uniqueId(this));
        this.asg.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // Following policy is required to read the aws tags within the instance
        this.asg.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ec2:DescribeTags',
            ],
            resources: ['*'],
        }));
        // wait for the log flush interval to make sure that all the logs gets flushed.
        // this wait can be avoided in future by using a life-cycle-hook on 'TERMINATING' state.
        const terminationDelay = Math.ceil(DeploymentInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL.toMinutes({ integral: false }));
        this.asg.userData.addOnExitCommands(`sleep ${terminationDelay}m`);
        // fetching the instance id and ASG name and then setting its capacity to 0
        this.asg.userData.addOnExitCommands('TOKEN=$(curl -X PUT "http://169.254.169.254/latest/api/token" -H "X-aws-ec2-metadata-token-ttl-seconds: 30" 2> /dev/null)', 'INSTANCE="$(curl -s -H "X-aws-ec2-metadata-token: $TOKEN" http://169.254.169.254/latest/meta-data/instance-id  2> /dev/null)"', 'ASG="$(aws --region ' + core_1.Stack.of(this).region + ' ec2 describe-tags --filters "Name=resource-id,Values=${INSTANCE}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags[0].Value" --output text)"', 'aws --region ' + core_1.Stack.of(this).region + ' autoscaling update-auto-scaling-group --auto-scaling-group-name ${ASG} --min-size 0 --max-size 0 --desired-capacity 0');
    }
}
exports.DeploymentInstance = DeploymentInstance;
/**
 * The tag key name used as an IAM condition to restrict autoscaling API grants
 */
DeploymentInstance.ASG_TAG_KEY = 'resourceLogicalId';
/**
 * How often the CloudWatch agent will flush its log files to CloudWatch
 */
DeploymentInstance.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
/**
 * The default timeout to wait for CloudFormation success signals before failing the resource create/update
 */
DeploymentInstance.DEFAULT_EXECUTION_TIMEOUT = core_1.Duration.minutes(15);
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
DeploymentInstance.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
//# sourceMappingURL=data:application/json;base64,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