"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const certificatemanager = require("@aws-cdk/aws-certificatemanager");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
var HttpVersion;
(function (HttpVersion) {
    HttpVersion["HTTP1_1"] = "http1.1";
    HttpVersion["HTTP2"] = "http2";
})(HttpVersion = exports.HttpVersion || (exports.HttpVersion = {}));
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 */
var PriceClass;
(function (PriceClass) {
    PriceClass["PRICE_CLASS_100"] = "PriceClass_100";
    PriceClass["PRICE_CLASS_200"] = "PriceClass_200";
    PriceClass["PRICE_CLASS_ALL"] = "PriceClass_All";
})(PriceClass = exports.PriceClass || (exports.PriceClass = {}));
/**
 * How HTTPs should be handled with your distribution.
 */
var ViewerProtocolPolicy;
(function (ViewerProtocolPolicy) {
    ViewerProtocolPolicy["HTTPS_ONLY"] = "https-only";
    ViewerProtocolPolicy["REDIRECT_TO_HTTPS"] = "redirect-to-https";
    ViewerProtocolPolicy["ALLOW_ALL"] = "allow-all";
})(ViewerProtocolPolicy = exports.ViewerProtocolPolicy || (exports.ViewerProtocolPolicy = {}));
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
var SSLMethod;
(function (SSLMethod) {
    SSLMethod["SNI"] = "sni-only";
    SSLMethod["VIP"] = "vip";
})(SSLMethod = exports.SSLMethod || (exports.SSLMethod = {}));
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
var SecurityPolicyProtocol;
(function (SecurityPolicyProtocol) {
    SecurityPolicyProtocol["SSL_V3"] = "SSLv3";
    SecurityPolicyProtocol["TLS_V1"] = "TLSv1";
    SecurityPolicyProtocol["TLS_V1_2016"] = "TLSv1_2016";
    SecurityPolicyProtocol["TLS_V1_1_2016"] = "TLSv1.1_2016";
    SecurityPolicyProtocol["TLS_V1_2_2018"] = "TLSv1.2_2018";
})(SecurityPolicyProtocol = exports.SecurityPolicyProtocol || (exports.SecurityPolicyProtocol = {}));
var OriginSslPolicy;
(function (OriginSslPolicy) {
    OriginSslPolicy["SSL_V3"] = "SSLv3";
    OriginSslPolicy["TLS_V1"] = "TLSv1";
    OriginSslPolicy["TLS_V1_1"] = "TLSv1.1";
    OriginSslPolicy["TLS_V1_2"] = "TLSv1.2";
})(OriginSslPolicy = exports.OriginSslPolicy || (exports.OriginSslPolicy = {}));
var OriginProtocolPolicy;
(function (OriginProtocolPolicy) {
    OriginProtocolPolicy["HTTP_ONLY"] = "http-only";
    OriginProtocolPolicy["MATCH_VIEWER"] = "match-viewer";
    OriginProtocolPolicy["HTTPS_ONLY"] = "https-only";
})(OriginProtocolPolicy = exports.OriginProtocolPolicy || (exports.OriginProtocolPolicy = {}));
/**
 * An enum for the supported methods to a CloudFront distribution.
 */
var CloudFrontAllowedMethods;
(function (CloudFrontAllowedMethods) {
    CloudFrontAllowedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedMethods["GET_HEAD_OPTIONS"] = "GHO";
    CloudFrontAllowedMethods["ALL"] = "ALL";
})(CloudFrontAllowedMethods = exports.CloudFrontAllowedMethods || (exports.CloudFrontAllowedMethods = {}));
/**
 * Enums for the methods CloudFront can cache.
 */
var CloudFrontAllowedCachedMethods;
(function (CloudFrontAllowedCachedMethods) {
    CloudFrontAllowedCachedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedCachedMethods["GET_HEAD_OPTIONS"] = "GHO";
})(CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedCachedMethods || (exports.CloudFrontAllowedCachedMethods = {}));
var LambdaEdgeEventType;
(function (LambdaEdgeEventType) {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_REQUEST"] = "origin-request";
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_RESPONSE"] = "origin-response";
    /**
     * The viewer-request specifies the incoming request
     */
    LambdaEdgeEventType["VIEWER_REQUEST"] = "viewer-request";
    /**
     * The viewer-response specifies the outgoing reponse
     */
    LambdaEdgeEventType["VIEWER_RESPONSE"] = "viewer-response";
})(LambdaEdgeEventType = exports.LambdaEdgeEventType || (exports.LambdaEdgeEventType = {}));
/**
 * Viewer certificate configuration class
 */
class ViewerCertificate {
    constructor(props, aliases = []) {
        this.props = props;
        this.aliases = aliases;
    }
    /**
     * Generate an AWS Certificate Manager (ACM) viewer certificate configuration
     *
     * @param certificate AWS Certificate Manager (ACM) certificate.
     *                    Your certificate must be located in the us-east-1 (US East (N. Virginia)) region to be accessed by CloudFront
     * @param options certificate configuration options
     */
    static fromAcmCertificate(certificate, options = {}) {
        const { sslMethod: sslSupportMethod = SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            acmCertificateArn: certificate.certificateArn, sslSupportMethod, minimumProtocolVersion
        }, aliases);
    }
    /**
     * Generate an IAM viewer certificate configuration
     *
     * @param iamCertificateId Identifier of the IAM certificate
     * @param options certificate configuration options
     */
    static fromIamCertificate(iamCertificateId, options = {}) {
        const { sslMethod: sslSupportMethod = SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            iamCertificateId, sslSupportMethod, minimumProtocolVersion
        }, aliases);
    }
    /**
     * Generate a viewer certifcate configuration using
     * the CloudFront default certificate (e.g. d111111abcdef8.cloudfront.net)
     * and a {@link SecurityPolicyProtocol.TLS_V1} security policy.
     *
     * @param aliases Alternative CNAME aliases
     *                You also must create a CNAME record with your DNS service to route queries
     */
    static fromCloudFrontDefaultCertificate(...aliases) {
        return new ViewerCertificate({ cloudFrontDefaultCertificate: true }, aliases);
    }
}
exports.ViewerCertificate = ViewerCertificate;
/**
 * Amazon CloudFront is a global content delivery network (CDN) service that securely delivers data, videos,
 * applications, and APIs to your viewers with low latency and high transfer speeds.
 * CloudFront fronts user provided content and caches it at edge locations across the world.
 *
 * Here's how you can use this construct:
 *
 * ```ts
 * import { CloudFrontWebDistribution } from '@aws-cdk/aws-cloudfront'
 *
 * const sourceBucket = new Bucket(this, 'Bucket');
 *
 * const distribution = new CloudFrontWebDistribution(this, 'MyDistribution', {
 *  originConfigs: [
 *    {
 *      s3OriginSource: {
 *      s3BucketSource: sourceBucket
 *      },
 *      behaviors : [ {isDefaultBehavior: true}]
 *    }
 *  ]
 * });
 * ```
 *
 * This will create a CloudFront distribution that uses your S3Bucket as it's origin.
 *
 * You can customize the distribution using additional properties from the CloudFrontWebDistributionProps interface.
 *
 *
 */
class CloudFrontWebDistribution extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Maps our methods to the string arrays they are
         */
        this.METHOD_LOOKUP_MAP = {
            GH: ["GET", "HEAD"],
            GHO: ["GET", "HEAD", "OPTIONS"],
            ALL: ["DELETE", "GET", "HEAD", "OPTIONS", "PATCH", "POST", "PUT"],
        };
        /**
         * Maps for which SecurityPolicyProtocol are available to which SSLMethods
         */
        this.VALID_SSL_PROTOCOLS = {
            [SSLMethod.SNI]: [
                SecurityPolicyProtocol.TLS_V1, SecurityPolicyProtocol.TLS_V1_1_2016,
                SecurityPolicyProtocol.TLS_V1_2016, SecurityPolicyProtocol.TLS_V1_2_2018
            ],
            [SSLMethod.VIP]: [SecurityPolicyProtocol.SSL_V3, SecurityPolicyProtocol.TLS_V1],
        };
        let distributionConfig = {
            comment: props.comment,
            enabled: true,
            defaultRootObject: props.defaultRootObject !== undefined ? props.defaultRootObject : "index.html",
            httpVersion: props.httpVersion || HttpVersion.HTTP2,
            priceClass: props.priceClass || PriceClass.PRICE_CLASS_100,
            ipv6Enabled: (props.enableIpV6 !== undefined) ? props.enableIpV6 : true,
            // tslint:disable-next-line:max-line-length
            customErrorResponses: props.errorConfigurations,
            webAclId: props.webACLId,
        };
        const behaviors = [];
        const origins = [];
        let originIndex = 1;
        for (const originConfig of props.originConfigs) {
            const originId = `origin${originIndex}`;
            if (!originConfig.s3OriginSource && !originConfig.customOriginSource) {
                throw new Error("There must be at least one origin source - either an s3OriginSource or a customOriginSource");
            }
            if (originConfig.customOriginSource && originConfig.s3OriginSource) {
                throw new Error("There cannot be both an s3OriginSource and a customOriginSource in the same SourceConfiguration.");
            }
            const originHeaders = [];
            if (originConfig.originHeaders) {
                Object.keys(originConfig.originHeaders).forEach(key => {
                    const oHeader = {
                        headerName: key,
                        headerValue: originConfig.originHeaders[key]
                    };
                    originHeaders.push(oHeader);
                });
            }
            let s3OriginConfig;
            if (originConfig.s3OriginSource) {
                // first case for backwards compatibility
                if (originConfig.s3OriginSource.originAccessIdentity) {
                    // grant CloudFront OriginAccessIdentity read access to S3 bucket
                    originConfig.s3OriginSource.s3BucketSource.grantRead(originConfig.s3OriginSource.originAccessIdentity);
                    s3OriginConfig = {
                        originAccessIdentity: `origin-access-identity/cloudfront/${originConfig.s3OriginSource.originAccessIdentity.originAccessIdentityName}`
                    };
                }
                else {
                    s3OriginConfig = {};
                }
            }
            const originProperty = {
                id: originId,
                domainName: originConfig.s3OriginSource
                    ? originConfig.s3OriginSource.s3BucketSource.bucketRegionalDomainName
                    : originConfig.customOriginSource.domainName,
                originPath: originConfig.originPath,
                originCustomHeaders: originHeaders.length > 0 ? originHeaders : undefined,
                s3OriginConfig,
                customOriginConfig: originConfig.customOriginSource
                    ? {
                        httpPort: originConfig.customOriginSource.httpPort || 80,
                        httpsPort: originConfig.customOriginSource.httpsPort || 443,
                        originKeepaliveTimeout: originConfig.customOriginSource.originKeepaliveTimeout
                            && originConfig.customOriginSource.originKeepaliveTimeout.toSeconds() || 5,
                        originReadTimeout: originConfig.customOriginSource.originReadTimeout
                            && originConfig.customOriginSource.originReadTimeout.toSeconds() || 30,
                        originProtocolPolicy: originConfig.customOriginSource.originProtocolPolicy || OriginProtocolPolicy.HTTPS_ONLY,
                        originSslProtocols: originConfig.customOriginSource.allowedOriginSSLVersions || [OriginSslPolicy.TLS_V1_2]
                    }
                    : undefined
            };
            for (const behavior of originConfig.behaviors) {
                behaviors.push({ ...behavior, targetOriginId: originId });
            }
            origins.push(originProperty);
            originIndex++;
        }
        origins.forEach(origin => {
            if (!origin.s3OriginConfig && !origin.customOriginConfig) {
                throw new Error(`Origin ${origin.domainName} is missing either S3OriginConfig or CustomOriginConfig. At least 1 must be specified.`);
            }
        });
        distributionConfig = {
            ...distributionConfig,
            origins
        };
        const defaultBehaviors = behaviors.filter(behavior => behavior.isDefaultBehavior);
        if (defaultBehaviors.length !== 1) {
            throw new Error("There can only be one default behavior across all sources. [ One default behavior per distribution ].");
        }
        distributionConfig = { ...distributionConfig, defaultCacheBehavior: this.toBehavior(defaultBehaviors[0], props.viewerProtocolPolicy) };
        const otherBehaviors = [];
        for (const behavior of behaviors.filter(b => !b.isDefaultBehavior)) {
            if (!behavior.pathPattern) {
                throw new Error("pathPattern is required for all non-default behaviors");
            }
            otherBehaviors.push(this.toBehavior(behavior, props.viewerProtocolPolicy));
        }
        distributionConfig = { ...distributionConfig, cacheBehaviors: otherBehaviors.length > 0 ? otherBehaviors : undefined };
        if (props.aliasConfiguration && props.viewerCertificate) {
            throw new Error([
                'You cannot set both aliasConfiguration and viewerCertificate properties.',
                'Please only use viewerCertificate, as aliasConfiguration is deprecated.'
            ].join(' '));
        }
        let _viewerCertificate = props.viewerCertificate;
        if (props.aliasConfiguration) {
            const { acmCertRef, securityPolicy, sslMethod, names: aliases } = props.aliasConfiguration;
            _viewerCertificate = ViewerCertificate.fromAcmCertificate(certificatemanager.Certificate.fromCertificateArn(this, 'AliasConfigurationCert', acmCertRef), { securityPolicy, sslMethod, aliases });
        }
        if (_viewerCertificate) {
            const { props: viewerCertificate, aliases } = _viewerCertificate;
            Object.assign(distributionConfig, { aliases, viewerCertificate });
            const { minimumProtocolVersion, sslSupportMethod } = viewerCertificate;
            if (minimumProtocolVersion != null && sslSupportMethod != null) {
                const validProtocols = this.VALID_SSL_PROTOCOLS[sslSupportMethod];
                if (validProtocols.indexOf(minimumProtocolVersion.toString()) === -1) {
                    // tslint:disable-next-line:max-line-length
                    throw new Error(`${minimumProtocolVersion} is not compabtible with sslMethod ${sslSupportMethod}.\n\tValid Protocols are: ${validProtocols.join(", ")}`);
                }
            }
        }
        else {
            distributionConfig = { ...distributionConfig,
                viewerCertificate: { cloudFrontDefaultCertificate: true }
            };
        }
        if (props.loggingConfig) {
            this.loggingBucket = props.loggingConfig.bucket || new s3.Bucket(this, `LoggingBucket`);
            distributionConfig = {
                ...distributionConfig,
                logging: {
                    bucket: this.loggingBucket.bucketRegionalDomainName,
                    includeCookies: props.loggingConfig.includeCookies || false,
                    prefix: props.loggingConfig.prefix
                }
            };
        }
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig });
        this.node.defaultChild = distribution;
        this.domainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    toBehavior(input, protoPolicy) {
        let toReturn = {
            allowedMethods: this.METHOD_LOOKUP_MAP[input.allowedMethods || CloudFrontAllowedMethods.GET_HEAD],
            cachedMethods: this.METHOD_LOOKUP_MAP[input.cachedMethods || CloudFrontAllowedCachedMethods.GET_HEAD],
            compress: input.compress !== false,
            defaultTtl: input.defaultTtl && input.defaultTtl.toSeconds(),
            forwardedValues: input.forwardedValues || { queryString: false, cookies: { forward: "none" } },
            maxTtl: input.maxTtl && input.maxTtl.toSeconds(),
            minTtl: input.minTtl && input.minTtl.toSeconds(),
            trustedSigners: input.trustedSigners,
            targetOriginId: input.targetOriginId,
            viewerProtocolPolicy: protoPolicy || ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        if (!input.isDefaultBehavior) {
            toReturn = Object.assign(toReturn, { pathPattern: input.pathPattern });
        }
        if (input.lambdaFunctionAssociations) {
            toReturn = Object.assign(toReturn, {
                lambdaFunctionAssociations: input.lambdaFunctionAssociations
                    .map(fna => ({
                    eventType: fna.eventType,
                    lambdaFunctionArn: fna.lambdaFunction && fna.lambdaFunction.functionArn,
                }))
            });
            // allow edgelambda.amazonaws.com to assume the functions' execution role.
            for (const a of input.lambdaFunctionAssociations) {
                if (a.lambdaFunction.role && a.lambdaFunction.role instanceof iam.Role && a.lambdaFunction.role.assumeRolePolicy) {
                    a.lambdaFunction.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                        actions: ['sts:AssumeRole'],
                        principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')]
                    }));
                }
            }
        }
        return toReturn;
    }
}
exports.CloudFrontWebDistribution = CloudFrontWebDistribution;
//# sourceMappingURL=data:application/json;base64,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