"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../core/lib");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const lib_2 = require("../lib");
const asset_constants_1 = require("./asset-constants");
let app;
let stack;
let wfstack;
let vpc;
let renderQueue;
let rcsImage;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'infraStack', {
        env: {
            region: 'us-east-1',
        },
    });
    vpc = new aws_ec2_1.Vpc(stack, 'VPC');
    rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
    const version = lib_2.VersionQuery.exact(stack, 'Version', {
        majorVersion: 10,
        minorVersion: 0,
        releaseVersion: 0,
        patchVersion: 0,
    });
    renderQueue = new lib_2.RenderQueue(stack, 'RQ', {
        version,
        vpc,
        images: { remoteConnectionServer: rcsImage },
        repository: new lib_2.Repository(stack, 'Repository', {
            vpc,
            version,
        }),
    });
    wfstack = new core_1.Stack(app, 'workerFleetStack', {
        env: {
            region: 'us-east-1',
        },
    });
});
test('default worker fleet is created correctly', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
    });
    // THEN
    assert_1.expect(wfstack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
    assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't2.large',
        spotPrice: assert_1.ABSENT,
    }));
    assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        ToPort: parseInt(renderQueue.endpoint.portAsString(), 10),
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                stack.getLogicalId(fleet.fleet.connections.securityGroups[0].node.defaultChild),
                'GroupId',
            ],
        },
        GroupId: {
            'Fn::ImportValue': 'infraStack:ExportsOutputFnGetAttRQLBSecurityGroupAC643AEDGroupId8F9F7830',
        },
    }));
    assert_1.expect(wfstack).to(assert_1.haveResource('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: '/renderfarm/workerFleet',
    }));
    expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
    expect(fleet.node.metadata[0].data).toMatch('being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
    expect(fleet.node.metadata[1].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
    expect(fleet.node.metadata[1].data).toContain('being created without a health monitor attached to it. This means that the fleet will not automatically scale-in to 0 if the workers are unhealthy');
});
test('security group is added to fleet after its creation', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
    });
    fleet.addSecurityGroup(aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
        allowAllOutbound: false,
    }));
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        SecurityGroups: [
            {
                'Fn::GetAtt': [
                    stack.getLogicalId(fleet.fleet.connections.securityGroups[0].node.defaultChild),
                    'GroupId',
                ],
            },
            'sg-123456789',
        ],
    }));
});
test('WorkerFleet uses given security group', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
        securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
            allowAllOutbound: false,
        }),
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        SecurityGroups: [
            'sg-123456789',
        ],
    }));
});
test('default worker fleet is created correctly with linux image', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration'));
});
test('default worker fleet is created correctly with spot config', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        spotPrice: 2.5,
    });
    // THEN
    assert_1.expect(wfstack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
    assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        SpotPrice: '2.5',
    }));
});
test('default worker fleet is not created with incorrect spot config', () => {
    // WHEN
    expect(() => {
        new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            spotPrice: lib_2.WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT + 1,
        });
    }).toThrowError(/Invalid value: 256 for property 'spotPrice'. Valid values can be any decimal between 0.001 and 255./);
    // WHEN
    expect(() => {
        new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet2', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            spotPrice: lib_2.WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT / 2,
        });
    }).toThrowError(/Invalid value: 0.0005 for property 'spotPrice'. Valid values can be any decimal between 0.001 and 255./);
});
test('default worker fleet is created correctly custom Instance type', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't2.medium',
    }));
});
test('default worker fleet is created correctly with custom LogGroup prefix', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        logGroupProps: { logGroupPrefix: 'test-prefix' },
    });
    assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: 'test-prefixworkerFleet',
    }));
});
test('default worker fleet is created correctly custom subnet values', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // WHEN
    const workers = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
        vpcSubnets: {
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        },
        healthCheckConfig: {
            port: 6161,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        VPCZoneIdentifier: [{
                Ref: 'VPC1AzPublicSubnet1Subnet9649CC17',
            }],
    }));
    const userData = stack.resolve(workers.fleet.userData.render());
    expect(userData).toStrictEqual({
        'Fn::Join': [
            '',
            [
                '#!/bin/bash\nfunction exitTrap(){\nexitCode=$?\n/opt/aws/bin/cfn-signal --stack infraStack --resource workerFleetASG25520D69 --region us-east-1 -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n}\ntrap exitTrap EXIT\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\')\naws s3 cp \'s3://',
                { Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\' \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\'\n' +
                    'set -e\n' +
                    'chmod +x \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                {
                                    Ref: asset_constants_1.CWA_ASSET_LINUX.Key,
                                },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\'\n\'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\' ',
                { Ref: 'workerFleetStringParameterE88827AB' },
                '\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\')\naws s3 cp \'s3://',
                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\'\n' +
                    'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                    '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                    'fi\n' +
                    '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '" --render-queue "http://',
                {
                    'Fn::GetAtt': [
                        'RQLB3B7B1CBC',
                        'DNSName',
                    ],
                },
                ':8080" \n' +
                    'rm -f "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\"\n' +
                    'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                    '  service deadline10launcher restart\n' +
                    'fi\n' +
                    "mkdir -p $(dirname '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "')\naws s3 cp 's3://",
                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\nset -e\nchmod +x '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\n'/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '6161' '' '' ''",
            ],
        ],
    });
});
test('default worker fleet is created correctly with groups, pools and region', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // WHEN
    const workers = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
        vpcSubnets: {
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        },
        groups: ['A', 'B'],
        pools: ['C', 'D'],
        region: 'E',
    });
    // THEN
    const userData = stack.resolve(workers.fleet.userData.render());
    expect(userData).toStrictEqual({
        'Fn::Join': ['', [
                '#!/bin/bash\nfunction exitTrap(){\nexitCode=$?\n/opt/aws/bin/cfn-signal --stack infraStack --resource workerFleetASG25520D69 --region us-east-1 -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n}\ntrap exitTrap EXIT\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "')\naws s3 cp 's3://",
                { Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\nset -e\nchmod +x '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\n'/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' ",
                { Ref: 'workerFleetStringParameterE88827AB' },
                '\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\')\naws s3 cp \'s3://',
                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\'\n' +
                    'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                    '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                    'fi\n' +
                    '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '" --render-queue "http://',
                {
                    'Fn::GetAtt': [
                        'RQLB3B7B1CBC',
                        'DNSName',
                    ],
                },
                ':8080" \n' +
                    'rm -f "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\"\n' +
                    'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                    '  service deadline10launcher restart\n' +
                    'fi\n' +
                    "mkdir -p $(dirname '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "')\naws s3 cp 's3://",
                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\nset -e\nchmod +x '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\n'/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '63415' 'a,b' 'c,d' 'E'",
            ]],
    });
});
test('worker fleet does validation correctly with groups, pools and region', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // group name as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            groups: ['A', 'none'],
        });
    }).toThrowError();
    // group name with whitespace
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet1', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            groups: ['A', 'no ne'],
        });
    }).toThrowError(/Invalid value: no ne for property 'groups'/);
    // pool name with whitespace
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet2', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            pools: ['A', 'none'],
        });
    }).toThrowError(/Invalid value: none for property 'pools'/);
    // pool name as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet3', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            pools: ['A', 'none'],
        });
    }).toThrowError(/Invalid value: none for property 'pools'/);
    // region as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet4', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none',
        });
    }).toThrowError(/Invalid value: none for property 'region'/);
    // region as 'all'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet5', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'all',
        });
    }).toThrowError(/Invalid value: all for property 'region'/);
    // region as 'unrecognized'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet6', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'unrecognized',
        });
    }).toThrowError(/Invalid value: unrecognized for property 'region'/);
    // region with invalid characters
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet7', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none@123',
        });
    }).toThrowError(/Invalid value: none@123 for property 'region'/);
    // region with reserved name as substring
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet8', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none123',
        });
    }).not.toThrowError();
    // region with case-insensitive name
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet9', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'None',
        });
    }).toThrowError(/Invalid value: None for property 'region'/);
});
describe('Block Device Tests', () => {
    let healthMonitor;
    beforeEach(() => {
        // create a health monitor so it does not trigger warnings
        healthMonitor = new lib_1.HealthMonitor(wfstack, 'healthMonitor', {
            vpc,
        });
    });
    test('Warning if no BlockDevices provided', () => {
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
        });
        expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
        expect(fleet.node.metadata[0].data).toMatch('being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
    });
    test('No Warnings if Encrypted BlockDevices Provided', () => {
        const VOLUME_SIZE = 50;
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: true }),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        Encrypted: true,
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        }));
        expect(fleet.node.metadata).toHaveLength(0);
    });
    test('Warnings if non-Encrypted BlockDevices Provided', () => {
        const VOLUME_SIZE = 50;
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: false }),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        Encrypted: false,
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        }));
        expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
        expect(fleet.node.metadata[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the worker-fleet workerFleet is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
    });
    test('Warnings for BlockDevices without encryption specified', () => {
        const VOLUME_SIZE = 50;
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        }));
        expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
        expect(fleet.node.metadata[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the worker-fleet workerFleet is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
    });
    test('No warnings for Ephemeral blockDeviceVolumes', () => {
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ephemeral(0),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    DeviceName: DEVICE_NAME,
                    VirtualName: 'ephemeral0',
                },
            ],
        }));
        expect(fleet.node.metadata).toHaveLength(0);
    });
    test('No warnings for Suppressed blockDeviceVolumes', () => {
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.noDevice(),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    DeviceName: DEVICE_NAME,
                },
            ],
        }));
        expect(fleet.node.metadata).toHaveLength(0);
    });
});
describe('tagging', () => {
    tag_helpers_1.testConstructTags({
        constructName: 'WorkerInstanceFleet',
        createConstruct: () => {
            // GIVEN
            const healthMonitorStack = new core_1.Stack(app, 'HealthMonitorStack', {
                env: {
                    region: 'us-east-1',
                },
            });
            const healthMonitor = new lib_1.HealthMonitor(healthMonitorStack, 'healthMonitor', {
                vpc,
            });
            const deviceName = '/dev/xvda';
            // WHEN
            new lib_2.WorkerInstanceFleet(wfstack, 'WorkerFleet', {
                vpc,
                workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                    'us-east-1': 'ami-any',
                }),
                renderQueue,
                healthMonitor,
                blockDevices: [{
                        deviceName,
                        volume: aws_autoscaling_1.BlockDeviceVolume.noDevice(),
                    }],
            });
            return wfstack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::ElasticLoadBalancingV2::TargetGroup': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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