"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const lib_1 = require("../lib");
const env = {
    region: 'us-east-1',
};
let app;
let certificateSecret;
let deadlineVersion;
let dependencyStack;
let dockerContainer;
let images;
let lfCluster;
let licenses;
let rcsImage;
let renderQueue;
let stack;
let ubl;
let vpc;
let workerFleet;
describe('UsageBasedLicensing', () => {
    beforeEach(() => {
        // GIVEN
        app = new core_1.App();
        dependencyStack = new core_1.Stack(app, 'DependencyStack', { env });
        deadlineVersion = lib_1.VersionQuery.exact(dependencyStack, 'Version', {
            majorVersion: 10,
            minorVersion: 1,
            releaseVersion: 9,
            patchVersion: 1,
        });
        expect(deadlineVersion.linuxFullVersionString).toBeDefined();
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'VPC');
        rcsImage = aws_ecs_1.ContainerImage.fromDockerImageAsset(new aws_ecr_assets_1.DockerImageAsset(dependencyStack, 'Image', {
            directory: __dirname,
        }));
        renderQueue = new lib_1.RenderQueue(dependencyStack, 'RQ-NonDefaultPort', {
            version: deadlineVersion,
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_1.Repository(dependencyStack, 'RepositoryNonDefault', {
                vpc,
                version: deadlineVersion,
            }),
        });
        lfCluster = new aws_ecs_1.Cluster(dependencyStack, 'licenseForwarderCluster', {
            vpc,
        });
        certificateSecret = aws_secretsmanager_1.Secret.fromSecretArn(lfCluster, 'CertSecret', 'arn:aws:secretsmanager:us-west-2:675872700355:secret:CertSecret-j1kiFz');
        dockerContainer = new aws_ecr_assets_1.DockerImageAsset(lfCluster, 'license-forwarder', {
            directory: __dirname,
        });
        images = {
            licenseForwarder: aws_ecs_1.ContainerImage.fromDockerImageAsset(dockerContainer),
        };
        const workerStack = new core_1.Stack(app, 'WorkerStack', { env });
        workerFleet = new lib_1.WorkerInstanceFleet(workerStack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(dependencyStack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            }),
        });
        licenses = [lib_1.UsageBasedLicense.forMaya()];
        stack = new core_1.Stack(app, 'Stack', { env });
        // WHEN
        ubl = new lib_1.UsageBasedLicensing(stack, 'UBL', {
            certificateSecret,
            images,
            licenses,
            renderQueue,
            vpc,
        });
    });
    test('creates an ECS cluster', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Cluster'));
    });
    describe('creates an ASG', () => {
        test('defaults', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: '1',
                MaxSize: '1',
                VPCZoneIdentifier: [
                    {
                        'Fn::ImportValue': assert_1.stringLike(`${dependencyStack.stackName}:ExportsOutputRefVPCPrivateSubnet1Subnet*`),
                    },
                    {
                        'Fn::ImportValue': assert_1.stringLike(`${dependencyStack.stackName}:ExportsOutputRefVPCPrivateSubnet2Subnet*`),
                    },
                ],
            }));
        });
        test('capacity can be specified', () => {
            // WHEN
            const isolatedStack = new core_1.Stack(app, 'MyStack', { env });
            new lib_1.UsageBasedLicensing(isolatedStack, 'licenseForwarder', {
                certificateSecret,
                desiredCount: 2,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: '2',
                MaxSize: '2',
            }));
        });
        test('gives write access to log group', () => {
            // GIVEN
            const logGroup = ubl.node.findChild('UBLLogGroup');
            const asgRoleLogicalId = core_1.Stack.of(ubl).getLogicalId(ubl.asg.role.node.defaultChild);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: assert_1.arrayWith('logs:CreateLogStream', 'logs:PutLogEvents'),
                        Effect: 'Allow',
                        Resource: stack.resolve(logGroup.logGroupArn),
                    }),
                    Version: '2012-10-17',
                },
                Roles: assert_1.arrayWith({ Ref: asgRoleLogicalId }),
            }));
        });
    });
    describe('creates an ECS service', () => {
        test('associated with the cluster', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                Cluster: { Ref: stack.getLogicalId(ubl.cluster.node.defaultChild) },
            }));
        });
        describe('DesiredCount', () => {
            test('defaults to 1', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: 1,
                }));
            });
            test('can be specified', () => {
                // GIVEN
                const desiredCount = 2;
                const isolatedStack = new core_1.Stack(app, 'IsolatedStack', { env });
                // WHEN
                new lib_1.UsageBasedLicensing(isolatedStack, 'UBL', {
                    certificateSecret,
                    images,
                    licenses,
                    renderQueue,
                    vpc,
                    desiredCount,
                });
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: desiredCount,
                }));
            });
        });
        test('sets launch type to EC2', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                LaunchType: 'EC2',
            }));
        });
        test('sets distinct instance placement constraint', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                PlacementConstraints: assert_1.arrayWith({ Type: 'distinctInstance' }),
            }));
        });
        test('uses the task definition', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                TaskDefinition: { Ref: stack.getLogicalId(ubl.service.taskDefinition.node.defaultChild) },
            }));
        });
        test('with the correct deployment configuration', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                DeploymentConfiguration: {
                    MaximumPercent: 100,
                    MinimumHealthyPercent: 0,
                },
            }));
        });
    });
    describe('creates a task definition', () => {
        test('container name is LicenseForwarderContainer', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Name: 'LicenseForwarderContainer',
                    },
                ],
            }));
        });
        test('container is marked essential', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Essential: true,
                    },
                ],
            }));
        });
        test('with increased ulimits', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Ulimits: [
                            {
                                HardLimit: 200000,
                                Name: 'nofile',
                                SoftLimit: 200000,
                            },
                            {
                                HardLimit: 64000,
                                Name: 'nproc',
                                SoftLimit: 64000,
                            },
                        ],
                    },
                ],
            }));
        });
        test('with awslogs log driver', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        LogConfiguration: {
                            LogDriver: 'awslogs',
                            Options: {
                                'awslogs-group': {},
                                'awslogs-stream-prefix': 'LicenseForwarder',
                                'awslogs-region': env.region,
                            },
                        },
                    },
                ],
            }));
        });
        test('configures UBL certificates', () => {
            // GIVEN
            const taskRoleLogicalId = core_1.Stack.of(ubl).getLogicalId(ubl.service.taskDefinition.taskRole.node.defaultChild);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'UBL_CERTIFICATES_URI',
                            Value: certificateSecret.secretArn,
                        }),
                    },
                ],
                TaskRoleArn: {
                    'Fn::GetAtt': [
                        taskRoleLogicalId,
                        'Arn',
                    ],
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: certificateSecret.secretArn,
                        },
                    ],
                    Version: '2012-10-17',
                },
                Roles: [
                    { Ref: core_1.Stack.of(ubl).getLogicalId(ubl.service.taskDefinition.taskRole.node.defaultChild) },
                ],
            }));
        });
        test('uses host networking', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                NetworkMode: 'host',
            }));
        });
        test('is marked EC2 compatible only', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                RequiresCompatibilities: ['EC2'],
            }));
        });
    });
    test('License Forwarder subnet selection', () => {
        // GIVEN
        const publicSubnetIds = ['PublicSubnet1', 'PublicSubnet2'];
        const vpcFromAttributes = aws_ec2_1.Vpc.fromVpcAttributes(dependencyStack, 'AttrVpc', {
            availabilityZones: ['us-east-1a', 'us-east-1b'],
            vpcId: 'vpcid',
            publicSubnetIds,
        });
        stack = new core_1.Stack(app, 'IsolatedStack', { env });
        // WHEN
        new lib_1.UsageBasedLicensing(stack, 'licenseForwarder', {
            certificateSecret,
            images,
            licenses,
            renderQueue,
            vpc: vpcFromAttributes,
            vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PUBLIC },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
            VPCZoneIdentifier: publicSubnetIds,
        }));
    });
    describe('license limits', () => {
        test('multiple licenses with limits', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack', { env });
            // WHEN
            new lib_1.UsageBasedLicensing(isolatedStack, 'licenseForwarder', {
                vpc,
                images,
                certificateSecret,
                renderQueue,
                licenses: [
                    lib_1.UsageBasedLicense.forMaya(10),
                    lib_1.UsageBasedLicense.forVray(10),
                ],
            });
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'UBL_LIMITS',
                            Value: 'maya:10;vray:10',
                        }),
                    },
                ],
            }));
        });
        test.each([
            ['3dsMax', lib_1.UsageBasedLicense.for3dsMax(10), [27002]],
            ['Arnold', lib_1.UsageBasedLicense.forArnold(10), [5056, 7056]],
            ['Cinema4D', lib_1.UsageBasedLicense.forCinema4D(10), [5057, 7057]],
            ['Clarisse', lib_1.UsageBasedLicense.forClarisse(10), [40500]],
            ['Houdini', lib_1.UsageBasedLicense.forHoudini(10), [1715]],
            ['Katana', lib_1.UsageBasedLicense.forKatana(10), [4101, 6101]],
            ['KeyShot', lib_1.UsageBasedLicense.forKeyShot(10), [27003, 2703]],
            ['Krakatoa', lib_1.UsageBasedLicense.forKrakatoa(10), [27000, 2700]],
            ['Mantra', lib_1.UsageBasedLicense.forMantra(10), [1716]],
            ['Maxwell', lib_1.UsageBasedLicense.forMaxwell(10), [5055, 7055]],
            ['Maya', lib_1.UsageBasedLicense.forMaya(10), [27002, 2702]],
            ['Nuke', lib_1.UsageBasedLicense.forNuke(10), [4101, 6101]],
            ['RealFlow', lib_1.UsageBasedLicense.forRealFlow(10), [5055, 7055]],
            ['RedShift', lib_1.UsageBasedLicense.forRedShift(10), [5054, 7054]],
            ['Vray', lib_1.UsageBasedLicense.forVray(10), [30306]],
            ['Yeti', lib_1.UsageBasedLicense.forYeti(10), [5053, 7053]],
        ])('Test open port for license type %s', (_licenseName, license, ports) => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack', { env });
            // WHEN
            ubl = new lib_1.UsageBasedLicensing(isolatedStack, 'licenseForwarder', {
                vpc,
                certificateSecret,
                licenses: [
                    license,
                ],
                renderQueue,
                images,
            });
            ubl.grantPortAccess(workerFleet, [license]);
            // THEN
            ports.forEach(port => {
                const ublAsgSecurityGroup = ubl.asg.connections.securityGroups[0].node.defaultChild;
                const ublAsgSecurityGroupLogicalId = isolatedStack.getLogicalId(ublAsgSecurityGroup);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: port,
                    GroupId: {
                        'Fn::GetAtt': [
                            ublAsgSecurityGroupLogicalId,
                            'GroupId',
                        ],
                    },
                    SourceSecurityGroupId: 'sg-123456789',
                }));
            });
        });
        test('requires one usage based license', () => {
            // Without any licenses
            expect(() => {
                new lib_1.UsageBasedLicensing(dependencyStack, 'licenseForwarder', {
                    vpc,
                    images,
                    certificateSecret: certificateSecret,
                    licenses: [],
                    renderQueue,
                });
            }).toThrowError('Should be specified at least one license with defined limit.');
        });
    });
    describe('configures render queue', () => {
        test('adds ingress rule for asg', () => {
            const ublAsgSg = ubl.asg.connections.securityGroups[0].node.defaultChild;
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 8080,
                ToPort: 8080,
                GroupId: {
                    'Fn::ImportValue': assert_1.stringLike(`${core_1.Stack.of(renderQueue).stackName}:ExportsOutputFnGetAttRQNonDefaultPortLBSecurityGroup*`),
                },
                SourceSecurityGroupId: {
                    'Fn::GetAtt': [
                        core_1.Stack.of(ubl).getLogicalId(ublAsgSg),
                        'GroupId',
                    ],
                },
            }));
        });
        test('sets RENDER_QUEUE_URI environment variable', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'RENDER_QUEUE_URI',
                            Value: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'http://',
                                        {
                                            'Fn::ImportValue': assert_1.stringLike(`${core_1.Stack.of(renderQueue).stackName}:ExportsOutputFnGetAttRQNonDefaultPortLB*`),
                                        },
                                        ':8080',
                                    ],
                                ],
                            },
                        }),
                    },
                ],
            }));
        });
    });
    describe('tagging', () => {
        tag_helpers_1.testConstructTags({
            constructName: 'UsageBasedLicensing',
            createConstruct: () => {
                return stack;
            },
            resourceTypeCounts: {
                'AWS::ECS::Cluster': 1,
                'AWS::EC2::SecurityGroup': 1,
                'AWS::IAM::Role': 5,
                'AWS::AutoScaling::AutoScalingGroup': 1,
                'AWS::Lambda::Function': 1,
                'AWS::SNS::Topic': 1,
                'AWS::ECS::TaskDefinition': 1,
                'AWS::ECS::Service': 1,
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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