"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../core/lib");
const token_regex_helpers_1 = require("../../core/test/token-regex-helpers");
const lib_2 = require("../lib");
describe('DocumentDB', () => {
    let stack;
    let vpc;
    let database;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        database = new aws_docdb_1.DatabaseCluster(stack, 'DbCluster', {
            masterUser: {
                username: 'master',
            },
            instanceProps: {
                instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.XLARGE),
                vpc,
                vpcSubnets: {
                    onePerAz: true,
                    subnetType: aws_ec2_1.SubnetType.PRIVATE,
                },
            },
            backup: {
                retention: core_1.Duration.days(15),
            },
        });
        if (!database.secret) {
            throw new Error('secret cannot be null');
        }
    });
    test('Grants access to Document DB Secret', () => {
        // GIVEN
        const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // WHEN
        connection.grantRead(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [{
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: { Ref: 'DbClusterSecretAttachment4201A1ED' },
                    }],
            },
        }));
    });
    test('addInstallerDBArgs defines required elements', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addInstallerDBArgs(instance);
        const userData = instance.userData.render();
        // THEN
        expect(userData).toContain('configure_database_installation_args(){\n');
        expect(userData).toContain('\nexport -f configure_database_installation_args');
        expect(userData).toContain('\nINSTALLER_DB_ARGS=(');
    });
    test('allow connection', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'SG', {
            vpc,
        });
        // WHEN
        connection.allowConnectionsFrom(securityGroup);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: {
                'Fn::GetAtt': [
                    'DbCluster224236EF',
                    'Port',
                ],
            },
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'SGADB53937',
                    'GroupId',
                ],
            },
            ToPort: {
                'Fn::GetAtt': [
                    'DbCluster224236EF',
                    'Port',
                ],
            },
        }));
    });
    test('add child dependency', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addChildDependency(instance);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            DependsOn: [
                'DbClusterInstance155835CE5',
                'InstanceInstanceRoleE9785DE5',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    test('add child dependency to attributes', () => {
        // GIVEN
        const docdb = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'Database', {
            clusterEndpointAddress: 'addr',
            clusterIdentifier: 'identifier',
            instanceEndpointAddresses: ['addr'],
            instanceIdentifiers: ['identifier'],
            port: 3306,
            readerEndpointAddress: 'reader-address',
            securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            }),
        });
        const connection = lib_2.DatabaseConnection.forDocDB({ database: docdb, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addChildDependency(instance);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            DependsOn: [
                'InstanceInstanceRoleE9785DE5',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    test('add child dependency throws when cluster implementation changed', () => {
        // GIVEN
        const docdb = aws_docdb_1.DatabaseCluster.fromDatabaseClusterAttributes(stack, 'Database', {
            clusterEndpointAddress: 'addr',
            clusterIdentifier: 'identifier',
            instanceEndpointAddresses: ['addr'],
            instanceIdentifiers: ['identifier'],
            port: 3306,
            readerEndpointAddress: 'reader-address',
            securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            }),
        });
        const connection = lib_2.DatabaseConnection.forDocDB({ database: docdb, login: database.secret });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        docdb.node.defaultChild = database; // Trick addChildDependency() into thinking this is a real construct.
        // THEN
        expect(() => {
            connection.addChildDependency(instance);
        }).toThrowError(/The internal implementation of the AWS CDK's DocumentDB cluster construct may have changed./);
    });
    test('asserts linux-only', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forDocDB({ database, login: database.secret });
        // WHEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
        // THEN
        expect(() => {
            connection.addInstallerDBArgs(instance);
        }).toThrowError('Can only install Deadline from a Linux instance.');
        expect(() => {
            connection.addConnectionDBArgs(instance);
        }).toThrowError('Connecting to the Deadline Database is currently only supported for Linux.');
    });
});
describe('MongoDB', () => {
    let stack;
    let vpc;
    let database;
    let clientCert;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        const hostname = 'mongo';
        const zoneName = 'deadline.internal';
        const dnsZone = new aws_route53_1.PrivateHostedZone(stack, 'PrivateHostedZone', {
            vpc,
            zoneName,
        });
        const caCert = new lib_1.X509CertificatePem(stack, 'CaCert', {
            subject: {
                cn: 'DistinguishedName',
            },
        });
        const serverCert = new lib_1.X509CertificatePem(stack, 'ServerCert', {
            subject: {
                cn: `${hostname}.${zoneName}`,
            },
            signingCertificate: caCert,
        });
        clientCert = new lib_1.X509CertificatePem(stack, 'ClientCert', {
            subject: {
                cn: 'dbuser',
                ou: 'TestClient',
            },
            signingCertificate: caCert,
        });
        database = new lib_1.MongoDbInstance(stack, 'MongoDb', {
            vpc,
            mongoDb: {
                userSsplAcceptance: lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL,
                version: lib_1.MongoDbVersion.COMMUNITY_3_6,
                hostname,
                dnsZone,
                serverCertificate: serverCert,
            },
        });
    });
    test('Grants access to certificate Secrets', () => {
        // GIVEN
        const role = new aws_iam_1.Role(stack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // WHEN
        connection.grantRead(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Effect: 'Allow',
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            'Fn::GetAtt': [
                                'ClientCert',
                                'Cert',
                            ],
                        },
                    },
                    {
                        Effect: 'Allow',
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Resource: {
                            Ref: 'ClientCertPassphrase8A71E1E1',
                        },
                    },
                ],
            },
        }));
    });
    test('addInstallerDBArgs defines required elements', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addInstallerDBArgs(instance);
        const userData = instance.userData.render();
        // THEN
        const token = '${Token[TOKEN.\\d+]}';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('\'/tmp/' + token + token + '\' ' + token + ' /opt/Thinkbox/certs/mongo_client.pfx')));
        expect(userData).toContain('configure_database_installation_args(){\n');
        expect(userData).toContain('\nexport -f configure_database_installation_args');
        expect(userData).toContain('{ set +x; } 2>/dev/null');
        expect(userData).toContain('\nINSTALLER_DB_ARGS=( ["--dbssl"]=true ["--dbauth"]=true ["--dbsslauth"]=true');
        expect(userData).toContain('["--dbhost"]="mongo.deadline.internal"');
        expect(userData).toContain('["--dbport"]=27017');
        expect(userData).toContain('["--dbclientcert"]=');
        expect(userData).toContain('["--dbcertpass"]=$CERT_PASSWORD');
    });
    test('addConnectionDBArgs defines required elements', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addConnectionDBArgs(instance);
        const userData = instance.userData.render();
        // THEN
        const token = '${Token[TOKEN.\\d+]}';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('\'/tmp/' + token + token + '\' ' + token + ' /opt/Thinkbox/certs/mongo_client.pfx')));
        expect(userData).toContain('configure_deadline_database(){\n');
        expect(userData).toContain('\nexport -f configure_deadline_database');
        expect(userData).toContain('{ set +x; } 2>/dev/null');
        expect(userData).toContain('\nexport DB_CERT_FILE=');
        expect(userData).toContain('\nexport DB_CERT_PASSWORD=');
    });
    test('defines required container environment variables', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // THEN
        expect(connection.containerEnvironment).toHaveProperty('DB_TLS_CLIENT_CERT_URI');
        expect(connection.containerEnvironment).toHaveProperty('DB_TLS_CLIENT_CERT_PASSWORD_URI');
    });
    test('allow connection', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'SG', {
            vpc,
        });
        // WHEN
        connection.allowConnectionsFrom(securityGroup);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            FromPort: 27017,
            ToPort: 27017,
            SourceSecurityGroupId: {
                'Fn::GetAtt': [
                    'SGADB53937',
                    'GroupId',
                ],
            },
            GroupId: {
                'Fn::GetAtt': [
                    'MongoDbServerAsgInstanceSecurityGroupCE623335',
                    'GroupId',
                ],
            },
        }));
    });
    test('add child dependency', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        connection.addChildDependency(instance);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            DependsOn: [
                'InstanceInstanceRoleE9785DE5',
                'MongoDbServerAsgASG47B3D94E',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    test('asserts linux-only', () => {
        // GIVEN
        const connection = lib_2.DatabaseConnection.forMongoDbInstance({ database, clientCertificate: clientCert });
        // WHEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
        // THEN
        expect(() => {
            connection.addInstallerDBArgs(instance);
        }).toThrowError('Can only install Deadline from a Linux instance.');
        expect(() => {
            connection.addConnectionDBArgs(instance);
        }).toThrowError('Connecting to the Deadline Database is currently only supported for Linux.');
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGF0YWJhc2UtY29ubmVjdGlvbi50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiZGF0YWJhc2UtY29ubmVjdGlvbi50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7O0dBR0c7O0FBRUgsNENBS3lCO0FBQ3pCLGtEQUU0QjtBQUM1Qiw4Q0FXMEI7QUFDMUIsOENBRzBCO0FBQzFCLHNEQUU4QjtBQUM5Qix3Q0FHdUI7QUFFdkIsd0NBTXdCO0FBQ3hCLDZFQUU2QztBQUM3QyxnQ0FFZ0I7QUFFaEIsUUFBUSxDQUFDLFlBQVksRUFBRSxHQUFHLEVBQUU7SUFDMUIsSUFBSSxLQUFZLENBQUM7SUFDakIsSUFBSSxHQUFRLENBQUM7SUFDYixJQUFJLFFBQXlCLENBQUM7SUFFOUIsVUFBVSxDQUFDLEdBQUcsRUFBRTtRQUNkLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQ3BCLEdBQUcsR0FBRyxJQUFJLGFBQUcsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFNUIsUUFBUSxHQUFHLElBQUksMkJBQWUsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFO1lBQ2pELFVBQVUsRUFBRTtnQkFDVixRQUFRLEVBQUUsUUFBUTthQUNuQjtZQUNELGFBQWEsRUFBRTtnQkFDYixZQUFZLEVBQUUsc0JBQVksQ0FBQyxFQUFFLENBQzNCLHVCQUFhLENBQUMsRUFBRSxFQUNoQixzQkFBWSxDQUFDLE1BQU0sQ0FDcEI7Z0JBQ0QsR0FBRztnQkFDSCxVQUFVLEVBQUU7b0JBQ1YsUUFBUSxFQUFFLElBQUk7b0JBQ2QsVUFBVSxFQUFFLG9CQUFVLENBQUMsT0FBTztpQkFDL0I7YUFDRjtZQUNELE1BQU0sRUFBRTtnQkFDTixTQUFTLEVBQUUsZUFBUSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUM7YUFDN0I7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRTtZQUNwQixNQUFNLElBQUksS0FBSyxDQUFDLHVCQUF1QixDQUFDLENBQUM7U0FDMUM7SUFDSCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxxQ0FBcUMsRUFBRSxHQUFHLEVBQUU7UUFDL0MsUUFBUTtRQUNSLE1BQU0sSUFBSSxHQUFHLElBQUksY0FBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUUsRUFBQyxTQUFTLEVBQUUsSUFBSSw4QkFBb0IsRUFBRSxFQUFDLENBQUMsQ0FBQztRQUM5RSxNQUFNLFVBQVUsR0FBRyx3QkFBa0IsQ0FBQyxRQUFRLENBQUMsRUFBQyxRQUFRLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxNQUFPLEVBQUMsQ0FBQyxDQUFDO1FBRXBGLE9BQU87UUFDUCxVQUFVLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRTNCLE9BQU87UUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsa0JBQWtCLEVBQUU7WUFDbkQsY0FBYyxFQUFFO2dCQUNkLE9BQU8sRUFBRSxZQUFZO2dCQUNyQixTQUFTLEVBQUUsQ0FBQzt3QkFDVixNQUFNLEVBQUU7NEJBQ04sK0JBQStCOzRCQUMvQiwrQkFBK0I7eUJBQ2hDO3dCQUNELE1BQU0sRUFBRSxPQUFPO3dCQUNmLFFBQVEsRUFBRSxFQUFDLEdBQUcsRUFBRSxtQ0FBbUMsRUFBQztxQkFDckQsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDLENBQUM7SUFDTixDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyw4Q0FBOEMsRUFBRSxHQUFHLEVBQUU7UUFDeEQsUUFBUTtRQUNSLE1BQU0sVUFBVSxHQUFHLHdCQUFrQixDQUFDLFFBQVEsQ0FBQyxFQUFDLFFBQVEsRUFBRSxLQUFLLEVBQUUsUUFBUSxDQUFDLE1BQU8sRUFBQyxDQUFDLENBQUM7UUFDcEYsTUFBTSxRQUFRLEdBQUcsSUFBSSxrQkFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDL0MsR0FBRztZQUNILFlBQVksRUFBRSxJQUFJLHNCQUFZLENBQUMsVUFBVSxDQUFDO1lBQzFDLFlBQVksRUFBRSxzQkFBWSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsVUFBVSxFQUFFLCtCQUFxQixDQUFDLGNBQWMsRUFBRSxDQUFDO1NBQ25HLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxVQUFVLENBQUMsa0JBQWtCLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDeEMsTUFBTSxRQUFRLEdBQUcsUUFBUSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEVBQUUsQ0FBQztRQUU1QyxPQUFPO1FBQ1AsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLFNBQVMsQ0FBQywyQ0FBMkMsQ0FBQyxDQUFDO1FBQ3hFLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxTQUFTLENBQUMsa0RBQWtELENBQUMsQ0FBQztRQUMvRSxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsU0FBUyxDQUFDLHVCQUF1QixDQUFDLENBQUM7SUFDdEQsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsa0JBQWtCLEVBQUUsR0FBRyxFQUFFO1FBQzVCLFFBQVE7UUFDUixNQUFNLFVBQVUsR0FBRyx3QkFBa0IsQ0FBQyxRQUFRLENBQUMsRUFBQyxRQUFRLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxNQUFPLEVBQUMsQ0FBQyxDQUFDO1FBQ3BGLE1BQU0sYUFBYSxHQUFHLElBQUksdUJBQWEsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFO1lBQ25ELEdBQUc7U0FDSixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsVUFBVSxDQUFDLG9CQUFvQixDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBRS9DLE9BQU87UUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdDQUFnQyxFQUFFO1lBQ3JFLFVBQVUsRUFBRSxLQUFLO1lBQ2pCLFFBQVEsRUFBRTtnQkFDUixZQUFZLEVBQUU7b0JBQ1osbUJBQW1CO29CQUNuQixNQUFNO2lCQUNQO2FBQ0Y7WUFDRCxxQkFBcUIsRUFBRTtnQkFDckIsWUFBWSxFQUFFO29CQUNaLFlBQVk7b0JBQ1osU0FBUztpQkFDVjthQUNGO1lBQ0QsTUFBTSxFQUFFO2dCQUNOLFlBQVksRUFBRTtvQkFDWixtQkFBbUI7b0JBQ25CLE1BQU07aUJBQ1A7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsc0JBQXNCLEVBQUUsR0FBRyxFQUFFO1FBQ2hDLFFBQVE7UUFDUixNQUFNLFVBQVUsR0FBRyx3QkFBa0IsQ0FBQyxRQUFRLENBQUMsRUFBQyxRQUFRLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxNQUFPLEVBQUMsQ0FBQyxDQUFDO1FBQ3BGLE1BQU0sUUFBUSxHQUFHLElBQUksa0JBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQy9DLEdBQUc7WUFDSCxZQUFZLEVBQUUsSUFBSSxzQkFBWSxDQUFDLFVBQVUsQ0FBQztZQUMxQyxZQUFZLEVBQUUsc0JBQVksQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLFVBQVUsRUFBRSwrQkFBcUIsQ0FBQyxjQUFjLEVBQUUsQ0FBQztTQUNuRyxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsVUFBVSxDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRXhDLE9BQU87UUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLG9CQUFvQixFQUFFO1lBQ3pELFNBQVMsRUFBRTtnQkFDVCw0QkFBNEI7Z0JBQzVCLDhCQUE4QjthQUMvQjtTQUNGLEVBQUUscUJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7SUFDdkMsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsb0NBQW9DLEVBQUUsR0FBRyxFQUFFO1FBQzlDLFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRywyQkFBZSxDQUFDLDZCQUE2QixDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDN0Usc0JBQXNCLEVBQUUsTUFBTTtZQUM5QixpQkFBaUIsRUFBRSxZQUFZO1lBQy9CLHlCQUF5QixFQUFFLENBQUMsTUFBTSxDQUFDO1lBQ25DLG1CQUFtQixFQUFFLENBQUMsWUFBWSxDQUFDO1lBQ25DLElBQUksRUFBRSxJQUFJO1lBQ1YscUJBQXFCLEVBQUUsZ0JBQWdCO1lBQ3ZDLGFBQWEsRUFBRSx1QkFBYSxDQUFDLG1CQUFtQixDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUUsY0FBYyxFQUFFO2dCQUM1RSxnQkFBZ0IsRUFBRSxLQUFLO2FBQ3hCLENBQUM7U0FDSCxDQUFDLENBQUM7UUFDSCxNQUFNLFVBQVUsR0FBRyx3QkFBa0IsQ0FBQyxRQUFRLENBQUMsRUFBQyxRQUFRLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBRSxRQUFRLENBQUMsTUFBTyxFQUFDLENBQUMsQ0FBQztRQUMzRixNQUFNLFFBQVEsR0FBRyxJQUFJLGtCQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUMvQyxHQUFHO1lBQ0gsWUFBWSxFQUFFLElBQUksc0JBQVksQ0FBQyxVQUFVLENBQUM7WUFDMUMsWUFBWSxFQUFFLHNCQUFZLENBQUMsaUJBQWlCLENBQUMsRUFBRSxVQUFVLEVBQUUsK0JBQXFCLENBQUMsY0FBYyxFQUFFLENBQUM7U0FDbkcsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLFVBQVUsQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUV4QyxPQUFPO1FBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyxvQkFBb0IsRUFBRTtZQUN6RCxTQUFTLEVBQUU7Z0JBQ1QsOEJBQThCO2FBQy9CO1NBQ0YsRUFBRSxxQkFBWSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQztJQUN2QyxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxpRUFBaUUsRUFBRSxHQUFHLEVBQUU7UUFDM0UsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLDJCQUFlLENBQUMsNkJBQTZCLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUM3RSxzQkFBc0IsRUFBRSxNQUFNO1lBQzlCLGlCQUFpQixFQUFFLFlBQVk7WUFDL0IseUJBQXlCLEVBQUUsQ0FBQyxNQUFNLENBQUM7WUFDbkMsbUJBQW1CLEVBQUUsQ0FBQyxZQUFZLENBQUM7WUFDbkMsSUFBSSxFQUFFLElBQUk7WUFDVixxQkFBcUIsRUFBRSxnQkFBZ0I7WUFDdkMsYUFBYSxFQUFFLHVCQUFhLENBQUMsbUJBQW1CLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRSxjQUFjLEVBQUU7Z0JBQzVFLGdCQUFnQixFQUFFLEtBQUs7YUFDeEIsQ0FBQztTQUNILENBQUMsQ0FBQztRQUNILE1BQU0sVUFBVSxHQUFHLHdCQUFrQixDQUFDLFFBQVEsQ0FBQyxFQUFDLFFBQVEsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxNQUFPLEVBQUMsQ0FBQyxDQUFDO1FBQzNGLE1BQU0sUUFBUSxHQUFHLElBQUksa0JBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQy9DLEdBQUc7WUFDSCxZQUFZLEVBQUUsSUFBSSxzQkFBWSxDQUFDLFVBQVUsQ0FBQztZQUMxQyxZQUFZLEVBQUUsc0JBQVksQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLFVBQVUsRUFBRSwrQkFBcUIsQ0FBQyxjQUFjLEVBQUUsQ0FBQztTQUNuRyxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsS0FBSyxDQUFDLElBQUksQ0FBQyxZQUFZLEdBQUcsUUFBUSxDQUFDLENBQUMscUVBQXFFO1FBRXpHLE9BQU87UUFDUCxNQUFNLENBQUMsR0FBRyxFQUFFO1lBQ1YsVUFBVSxDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzFDLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQyw2RkFBNkYsQ0FBQyxDQUFDO0lBQ2pILENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLG9CQUFvQixFQUFFLEdBQUcsRUFBRTtRQUM5QixRQUFRO1FBQ1IsTUFBTSxVQUFVLEdBQUcsd0JBQWtCLENBQUMsUUFBUSxDQUFDLEVBQUMsUUFBUSxFQUFFLEtBQUssRUFBRSxRQUFRLENBQUMsTUFBTyxFQUFDLENBQUMsQ0FBQztRQUVwRixPQUFPO1FBQ1AsTUFBTSxRQUFRLEdBQUcsSUFBSSxrQkFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7WUFDL0MsR0FBRztZQUNILFlBQVksRUFBRSxJQUFJLHNCQUFZLENBQUMsVUFBVSxDQUFDO1lBQzFDLFlBQVksRUFBRSxzQkFBWSxDQUFDLGFBQWEsQ0FBQyx3QkFBYyxDQUFDLHFDQUFxQyxDQUFDO1NBQy9GLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsR0FBRyxFQUFFO1lBQ1YsVUFBVSxDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQzFDLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQyxrREFBa0QsQ0FBQyxDQUFDO1FBQ3BFLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDVixVQUFVLENBQUMsbUJBQW1CLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDM0MsQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLDRFQUE0RSxDQUFDLENBQUM7SUFDaEcsQ0FBQyxDQUFDLENBQUM7QUFFTCxDQUFDLENBQUMsQ0FBQztBQUVILFFBQVEsQ0FBQyxTQUFTLEVBQUUsR0FBRyxFQUFFO0lBQ3ZCLElBQUksS0FBWSxDQUFDO0lBQ2pCLElBQUksR0FBUSxDQUFDO0lBQ2IsSUFBSSxRQUFrQixDQUFDO0lBQ3ZCLElBQUksVUFBOEIsQ0FBQztJQUVuQyxVQUFVLENBQUMsR0FBRyxFQUFFO1FBQ2QsS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDcEIsR0FBRyxHQUFHLElBQUksYUFBRyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQztRQUM1QixNQUFNLFFBQVEsR0FBRyxPQUFPLENBQUM7UUFDekIsTUFBTSxRQUFRLEdBQUcsbUJBQW1CLENBQUM7UUFDckMsTUFBTSxPQUFPLEdBQUcsSUFBSSwrQkFBaUIsQ0FBQyxLQUFLLEVBQUUsbUJBQW1CLEVBQUU7WUFDaEUsR0FBRztZQUNILFFBQVE7U0FDVCxDQUFDLENBQUM7UUFDSCxNQUFNLE1BQU0sR0FBRyxJQUFJLHdCQUFrQixDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDckQsT0FBTyxFQUFFO2dCQUNQLEVBQUUsRUFBRSxtQkFBbUI7YUFDeEI7U0FDRixDQUFDLENBQUM7UUFDSCxNQUFNLFVBQVUsR0FBRyxJQUFJLHdCQUFrQixDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7WUFDN0QsT0FBTyxFQUFFO2dCQUNQLEVBQUUsRUFBRSxHQUFHLFFBQVEsSUFBSSxRQUFRLEVBQUU7YUFDOUI7WUFDRCxrQkFBa0IsRUFBRSxNQUFNO1NBQzNCLENBQUMsQ0FBQztRQUNILFVBQVUsR0FBRyxJQUFJLHdCQUFrQixDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7WUFDdkQsT0FBTyxFQUFFO2dCQUNQLEVBQUUsRUFBRSxRQUFRO2dCQUNaLEVBQUUsRUFBRSxZQUFZO2FBQ2pCO1lBQ0Qsa0JBQWtCLEVBQUUsTUFBTTtTQUMzQixDQUFDLENBQUM7UUFFSCxRQUFRLEdBQUcsSUFBSSxxQkFBZSxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7WUFDL0MsR0FBRztZQUNILE9BQU8sRUFBRTtnQkFDUCxrQkFBa0IsRUFBRSxrQ0FBNEIsQ0FBQyxpQkFBaUI7Z0JBQ2xFLE9BQU8sRUFBRSxvQkFBYyxDQUFDLGFBQWE7Z0JBQ3JDLFFBQVE7Z0JBQ1IsT0FBTztnQkFDUCxpQkFBaUIsRUFBRSxVQUFVO2FBQzlCO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsc0NBQXNDLEVBQUUsR0FBRyxFQUFFO1FBQ2hELFFBQVE7UUFDUixNQUFNLElBQUksR0FBRyxJQUFJLGNBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLEVBQUMsU0FBUyxFQUFFLElBQUksOEJBQW9CLEVBQUUsRUFBQyxDQUFDLENBQUM7UUFDOUUsTUFBTSxVQUFVLEdBQUcsd0JBQWtCLENBQUMsa0JBQWtCLENBQUMsRUFBQyxRQUFRLEVBQUUsaUJBQWlCLEVBQUUsVUFBVSxFQUFDLENBQUMsQ0FBQztRQUVwRyxPQUFPO1FBQ1AsVUFBVSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUUzQixPQUFPO1FBQ1AsZUFBUyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO1lBQ25ELGNBQWMsRUFBRTtnQkFDZCxPQUFPLEVBQUUsWUFBWTtnQkFDckIsU0FBUyxFQUFFO29CQUNUO3dCQUNFLE1BQU0sRUFBRSxPQUFPO3dCQUNmLE1BQU0sRUFBRTs0QkFDTiwrQkFBK0I7NEJBQy9CLCtCQUErQjt5QkFDaEM7d0JBQ0QsUUFBUSxFQUFFOzRCQUNSLFlBQVksRUFBRTtnQ0FDWixZQUFZO2dDQUNaLE1BQU07NkJBQ1A7eUJBQ0Y7cUJBQ0Y7b0JBQ0Q7d0JBQ0UsTUFBTSxFQUFFLE9BQU87d0JBQ2YsTUFBTSxFQUFFOzRCQUNOLCtCQUErQjs0QkFDL0IsK0JBQStCO3lCQUNoQzt3QkFDRCxRQUFRLEVBQUU7NEJBQ1IsR0FBRyxFQUFFLDhCQUE4Qjt5QkFDcEM7cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsOENBQThDLEVBQUUsR0FBRyxFQUFFO1FBQ3hELFFBQVE7UUFDUixNQUFNLFVBQVUsR0FBRyx3QkFBa0IsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFDLFFBQVEsRUFBRSxpQkFBaUIsRUFBRSxVQUFVLEVBQUMsQ0FBQyxDQUFDO1FBQ3BHLE1BQU0sUUFBUSxHQUFHLElBQUksa0JBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQy9DLEdBQUc7WUFDSCxZQUFZLEVBQUUsSUFBSSxzQkFBWSxDQUFDLFVBQVUsQ0FBQztZQUMxQyxZQUFZLEVBQUUsc0JBQVksQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLFVBQVUsRUFBRSwrQkFBcUIsQ0FBQyxjQUFjLEVBQUUsQ0FBQztTQUNuRyxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsVUFBVSxDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ3hDLE1BQU0sUUFBUSxHQUFHLFFBQVEsQ0FBQyxRQUFRLENBQUMsTUFBTSxFQUFFLENBQUM7UUFFNUMsT0FBTztRQUNQLE1BQU0sS0FBSyxHQUFHLHNCQUFzQixDQUFDO1FBQ3JDLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxNQUFNLENBQUMsc0NBQWdCLENBQUMsU0FBUyxHQUFHLEtBQUssR0FBRyxLQUFLLEdBQUcsS0FBSyxHQUFHLEtBQUssR0FBRyx1Q0FBdUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUM1SSxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsU0FBUyxDQUFDLDJDQUEyQyxDQUFDLENBQUM7UUFDeEUsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxrREFBa0QsQ0FBQyxDQUFDO1FBQy9FLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxTQUFTLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUN0RCxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsU0FBUyxDQUFDLCtFQUErRSxDQUFDLENBQUM7UUFDNUcsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLFNBQVMsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDO1FBQ3JFLE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxTQUFTLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUNqRCxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsU0FBUyxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDbEQsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO0lBQ2hFLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLCtDQUErQyxFQUFFLEdBQUcsRUFBRTtRQUN6RCxRQUFRO1FBQ1IsTUFBTSxVQUFVLEdBQUcsd0JBQWtCLENBQUMsa0JBQWtCLENBQUMsRUFBQyxRQUFRLEVBQUUsaUJBQWlCLEVBQUUsVUFBVSxFQUFDLENBQUMsQ0FBQztRQUNwRyxNQUFNLFFBQVEsR0FBRyxJQUFJLGtCQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUMvQyxHQUFHO1lBQ0gsWUFBWSxFQUFFLElBQUksc0JBQVksQ0FBQyxVQUFVLENBQUM7WUFDMUMsWUFBWSxFQUFFLHNCQUFZLENBQUMsaUJBQWlCLENBQUMsRUFBRSxVQUFVLEVBQUUsK0JBQXFCLENBQUMsY0FBYyxFQUFFLENBQUM7U0FDbkcsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUN6QyxNQUFNLFFBQVEsR0FBRyxRQUFRLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxDQUFDO1FBRTVDLE9BQU87UUFDUCxNQUFNLEtBQUssR0FBRyxzQkFBc0IsQ0FBQztRQUNyQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksTUFBTSxDQUFDLHNDQUFnQixDQUFDLFNBQVMsR0FBRyxLQUFLLEdBQUcsS0FBSyxHQUFHLEtBQUssR0FBRyxLQUFLLEdBQUcsdUNBQXVDLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDNUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxrQ0FBa0MsQ0FBQyxDQUFDO1FBQy9ELE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxTQUFTLENBQUMseUNBQXlDLENBQUMsQ0FBQztRQUN0RSxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsU0FBUyxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDdEQsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDLFNBQVMsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ3JELE1BQU0sQ0FBQyxRQUFRLENBQUMsQ0FBQyxTQUFTLENBQUMsNEJBQTRCLENBQUMsQ0FBQztJQUMzRCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxrREFBa0QsRUFBRSxHQUFHLEVBQUU7UUFDNUQsUUFBUTtRQUNSLE1BQU0sVUFBVSxHQUFHLHdCQUFrQixDQUFDLGtCQUFrQixDQUFDLEVBQUMsUUFBUSxFQUFFLGlCQUFpQixFQUFFLFVBQVUsRUFBQyxDQUFDLENBQUM7UUFFcEcsT0FBTztRQUNQLE1BQU0sQ0FBQyxVQUFVLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxjQUFjLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNqRixNQUFNLENBQUMsVUFBVSxDQUFDLG9CQUFvQixDQUFDLENBQUMsY0FBYyxDQUFDLGlDQUFpQyxDQUFDLENBQUM7SUFDNUYsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsa0JBQWtCLEVBQUUsR0FBRyxFQUFFO1FBQzVCLFFBQVE7UUFDUixNQUFNLFVBQVUsR0FBRyx3QkFBa0IsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFDLFFBQVEsRUFBRSxpQkFBaUIsRUFBRSxVQUFVLEVBQUMsQ0FBQyxDQUFDO1FBQ3BHLE1BQU0sYUFBYSxHQUFHLElBQUksdUJBQWEsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFO1lBQ25ELEdBQUc7U0FDSixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsVUFBVSxDQUFDLG9CQUFvQixDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBRS9DLE9BQU87UUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLGdDQUFnQyxFQUFFO1lBQ3JFLFVBQVUsRUFBRSxLQUFLO1lBQ2pCLFFBQVEsRUFBRSxLQUFLO1lBQ2YsTUFBTSxFQUFFLEtBQUs7WUFDYixxQkFBcUIsRUFBRTtnQkFDckIsWUFBWSxFQUFFO29CQUNaLFlBQVk7b0JBQ1osU0FBUztpQkFDVjthQUNGO1lBQ0QsT0FBTyxFQUFFO2dCQUNQLFlBQVksRUFBRTtvQkFDWiwrQ0FBK0M7b0JBQy9DLFNBQVM7aUJBQ1Y7YUFDRjtTQUNGLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsc0JBQXNCLEVBQUUsR0FBRyxFQUFFO1FBQ2hDLFFBQVE7UUFDUixNQUFNLFVBQVUsR0FBRyx3QkFBa0IsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFDLFFBQVEsRUFBRSxpQkFBaUIsRUFBRSxVQUFVLEVBQUMsQ0FBQyxDQUFDO1FBQ3BHLE1BQU0sUUFBUSxHQUFHLElBQUksa0JBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQy9DLEdBQUc7WUFDSCxZQUFZLEVBQUUsSUFBSSxzQkFBWSxDQUFDLFVBQVUsQ0FBQztZQUMxQyxZQUFZLEVBQUUsc0JBQVksQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLFVBQVUsRUFBRSwrQkFBcUIsQ0FBQyxjQUFjLEVBQUUsQ0FBQztTQUNuRyxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsVUFBVSxDQUFDLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRXhDLE9BQU87UUFDUCxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLG9CQUFvQixFQUFFO1lBQ3pELFNBQVMsRUFBRTtnQkFDVCw4QkFBOEI7Z0JBQzlCLDZCQUE2QjthQUM5QjtTQUNGLEVBQUUscUJBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7SUFDdkMsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsb0JBQW9CLEVBQUUsR0FBRyxFQUFFO1FBQzlCLFFBQVE7UUFDUixNQUFNLFVBQVUsR0FBRyx3QkFBa0IsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFDLFFBQVEsRUFBRSxpQkFBaUIsRUFBRSxVQUFVLEVBQUMsQ0FBQyxDQUFDO1FBRXBHLE9BQU87UUFDUCxNQUFNLFFBQVEsR0FBRyxJQUFJLGtCQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRTtZQUMvQyxHQUFHO1lBQ0gsWUFBWSxFQUFFLElBQUksc0JBQVksQ0FBQyxVQUFVLENBQUM7WUFDMUMsWUFBWSxFQUFFLHNCQUFZLENBQUMsYUFBYSxDQUFDLHdCQUFjLENBQUMscUNBQXFDLENBQUM7U0FDL0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxHQUFHLEVBQUU7WUFDVixVQUFVLENBQUMsa0JBQWtCLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDMUMsQ0FBQyxDQUFDLENBQUMsWUFBWSxDQUFDLGtEQUFrRCxDQUFDLENBQUM7UUFDcEUsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUNWLFVBQVUsQ0FBQyxtQkFBbUIsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUMzQyxDQUFDLENBQUMsQ0FBQyxZQUFZLENBQUMsNEVBQTRFLENBQUMsQ0FBQztJQUNoRyxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXG4gKi9cblxuaW1wb3J0IHtcbiAgZXhwZWN0IGFzIGV4cGVjdENESyxcbiAgaGF2ZVJlc291cmNlLFxuICBoYXZlUmVzb3VyY2VMaWtlLFxuICBSZXNvdXJjZVBhcnQsXG59IGZyb20gJ0Bhd3MtY2RrL2Fzc2VydCc7XG5pbXBvcnQge1xuICBEYXRhYmFzZUNsdXN0ZXIsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1kb2NkYic7XG5pbXBvcnQge1xuICBBbWF6b25MaW51eEdlbmVyYXRpb24sXG4gIEluc3RhbmNlLFxuICBJbnN0YW5jZUNsYXNzLFxuICBJbnN0YW5jZVNpemUsXG4gIEluc3RhbmNlVHlwZSxcbiAgTWFjaGluZUltYWdlLFxuICBTZWN1cml0eUdyb3VwLFxuICBTdWJuZXRUeXBlLFxuICBWcGMsXG4gIFdpbmRvd3NWZXJzaW9uLFxufSBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCB7XG4gIEFjY291bnRSb290UHJpbmNpcGFsLFxuICBSb2xlLFxufSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCB7XG4gIFByaXZhdGVIb3N0ZWRab25lLFxufSBmcm9tICdAYXdzLWNkay9hd3Mtcm91dGU1Myc7XG5pbXBvcnQge1xuICBEdXJhdGlvbixcbiAgU3RhY2ssXG59IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG5pbXBvcnQge1xuICBJTW9uZ29EYixcbiAgTW9uZ29EYkluc3RhbmNlLFxuICBNb25nb0RiU3NwbExpY2Vuc2VBY2NlcHRhbmNlLFxuICBNb25nb0RiVmVyc2lvbixcbiAgWDUwOUNlcnRpZmljYXRlUGVtLFxufSBmcm9tICcuLi8uLi9jb3JlL2xpYic7XG5pbXBvcnQge1xuICBlc2NhcGVUb2tlblJlZ2V4LFxufSBmcm9tICcuLi8uLi9jb3JlL3Rlc3QvdG9rZW4tcmVnZXgtaGVscGVycyc7XG5pbXBvcnQge1xuICBEYXRhYmFzZUNvbm5lY3Rpb24sXG59IGZyb20gJy4uL2xpYic7XG5cbmRlc2NyaWJlKCdEb2N1bWVudERCJywgKCkgPT4ge1xuICBsZXQgc3RhY2s6IFN0YWNrO1xuICBsZXQgdnBjOiBWcGM7XG4gIGxldCBkYXRhYmFzZTogRGF0YWJhc2VDbHVzdGVyO1xuXG4gIGJlZm9yZUVhY2goKCkgPT4ge1xuICAgIHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgdnBjID0gbmV3IFZwYyhzdGFjaywgJ1ZQQycpO1xuXG4gICAgZGF0YWJhc2UgPSBuZXcgRGF0YWJhc2VDbHVzdGVyKHN0YWNrLCAnRGJDbHVzdGVyJywge1xuICAgICAgbWFzdGVyVXNlcjoge1xuICAgICAgICB1c2VybmFtZTogJ21hc3RlcicsXG4gICAgICB9LFxuICAgICAgaW5zdGFuY2VQcm9wczoge1xuICAgICAgICBpbnN0YW5jZVR5cGU6IEluc3RhbmNlVHlwZS5vZihcbiAgICAgICAgICBJbnN0YW5jZUNsYXNzLlI1LFxuICAgICAgICAgIEluc3RhbmNlU2l6ZS5YTEFSR0UsXG4gICAgICAgICksXG4gICAgICAgIHZwYyxcbiAgICAgICAgdnBjU3VibmV0czoge1xuICAgICAgICAgIG9uZVBlckF6OiB0cnVlLFxuICAgICAgICAgIHN1Ym5ldFR5cGU6IFN1Ym5ldFR5cGUuUFJJVkFURSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICBiYWNrdXA6IHtcbiAgICAgICAgcmV0ZW50aW9uOiBEdXJhdGlvbi5kYXlzKDE1KSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBpZiAoIWRhdGFiYXNlLnNlY3JldCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdzZWNyZXQgY2Fubm90IGJlIG51bGwnKTtcbiAgICB9XG4gIH0pO1xuXG4gIHRlc3QoJ0dyYW50cyBhY2Nlc3MgdG8gRG9jdW1lbnQgREIgU2VjcmV0JywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgcm9sZSA9IG5ldyBSb2xlKHN0YWNrLCAnUm9sZScsIHthc3N1bWVkQnk6IG5ldyBBY2NvdW50Um9vdFByaW5jaXBhbCgpfSk7XG4gICAgY29uc3QgY29ubmVjdGlvbiA9IERhdGFiYXNlQ29ubmVjdGlvbi5mb3JEb2NEQih7ZGF0YWJhc2UsIGxvZ2luOiBkYXRhYmFzZS5zZWNyZXQhfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29ubmVjdGlvbi5ncmFudFJlYWQocm9sZSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycsXG4gICAgICAgIFN0YXRlbWVudDogW3tcbiAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgICdzZWNyZXRzbWFuYWdlcjpHZXRTZWNyZXRWYWx1ZScsXG4gICAgICAgICAgICAnc2VjcmV0c21hbmFnZXI6RGVzY3JpYmVTZWNyZXQnLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgIFJlc291cmNlOiB7UmVmOiAnRGJDbHVzdGVyU2VjcmV0QXR0YWNobWVudDQyMDFBMUVEJ30sXG4gICAgICAgIH1dLFxuICAgICAgfSxcbiAgICB9KSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2FkZEluc3RhbGxlckRCQXJncyBkZWZpbmVzIHJlcXVpcmVkIGVsZW1lbnRzJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgY29ubmVjdGlvbiA9IERhdGFiYXNlQ29ubmVjdGlvbi5mb3JEb2NEQih7ZGF0YWJhc2UsIGxvZ2luOiBkYXRhYmFzZS5zZWNyZXQhfSk7XG4gICAgY29uc3QgaW5zdGFuY2UgPSBuZXcgSW5zdGFuY2Uoc3RhY2ssICdJbnN0YW5jZScsIHtcbiAgICAgIHZwYyxcbiAgICAgIGluc3RhbmNlVHlwZTogbmV3IEluc3RhbmNlVHlwZSgndDMuc21hbGwnKSxcbiAgICAgIG1hY2hpbmVJbWFnZTogTWFjaGluZUltYWdlLmxhdGVzdEFtYXpvbkxpbnV4KHsgZ2VuZXJhdGlvbjogQW1hem9uTGludXhHZW5lcmF0aW9uLkFNQVpPTl9MSU5VWF8yIH0pLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbm5lY3Rpb24uYWRkSW5zdGFsbGVyREJBcmdzKGluc3RhbmNlKTtcbiAgICBjb25zdCB1c2VyRGF0YSA9IGluc3RhbmNlLnVzZXJEYXRhLnJlbmRlcigpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdCh1c2VyRGF0YSkudG9Db250YWluKCdjb25maWd1cmVfZGF0YWJhc2VfaW5zdGFsbGF0aW9uX2FyZ3MoKXtcXG4nKTtcbiAgICBleHBlY3QodXNlckRhdGEpLnRvQ29udGFpbignXFxuZXhwb3J0IC1mIGNvbmZpZ3VyZV9kYXRhYmFzZV9pbnN0YWxsYXRpb25fYXJncycpO1xuICAgIGV4cGVjdCh1c2VyRGF0YSkudG9Db250YWluKCdcXG5JTlNUQUxMRVJfREJfQVJHUz0oJyk7XG4gIH0pO1xuXG4gIHRlc3QoJ2FsbG93IGNvbm5lY3Rpb24nLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBjb25uZWN0aW9uID0gRGF0YWJhc2VDb25uZWN0aW9uLmZvckRvY0RCKHtkYXRhYmFzZSwgbG9naW46IGRhdGFiYXNlLnNlY3JldCF9KTtcbiAgICBjb25zdCBzZWN1cml0eUdyb3VwID0gbmV3IFNlY3VyaXR5R3JvdXAoc3RhY2ssICdTRycsIHtcbiAgICAgIHZwYyxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25uZWN0aW9uLmFsbG93Q29ubmVjdGlvbnNGcm9tKHNlY3VyaXR5R3JvdXApO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXBJbmdyZXNzJywge1xuICAgICAgSXBQcm90b2NvbDogJ3RjcCcsXG4gICAgICBGcm9tUG9ydDoge1xuICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAnRGJDbHVzdGVyMjI0MjM2RUYnLFxuICAgICAgICAgICdQb3J0JyxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgICBTb3VyY2VTZWN1cml0eUdyb3VwSWQ6IHtcbiAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgJ1NHQURCNTM5MzcnLFxuICAgICAgICAgICdHcm91cElkJyxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgICBUb1BvcnQ6IHtcbiAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgJ0RiQ2x1c3RlcjIyNDIzNkVGJyxcbiAgICAgICAgICAnUG9ydCcsXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pKTtcbiAgfSk7XG5cbiAgdGVzdCgnYWRkIGNoaWxkIGRlcGVuZGVuY3knLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBjb25uZWN0aW9uID0gRGF0YWJhc2VDb25uZWN0aW9uLmZvckRvY0RCKHtkYXRhYmFzZSwgbG9naW46IGRhdGFiYXNlLnNlY3JldCF9KTtcbiAgICBjb25zdCBpbnN0YW5jZSA9IG5ldyBJbnN0YW5jZShzdGFjaywgJ0luc3RhbmNlJywge1xuICAgICAgdnBjLFxuICAgICAgaW5zdGFuY2VUeXBlOiBuZXcgSW5zdGFuY2VUeXBlKCd0My5zbWFsbCcpLFxuICAgICAgbWFjaGluZUltYWdlOiBNYWNoaW5lSW1hZ2UubGF0ZXN0QW1hem9uTGludXgoeyBnZW5lcmF0aW9uOiBBbWF6b25MaW51eEdlbmVyYXRpb24uQU1BWk9OX0xJTlVYXzIgfSksXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29ubmVjdGlvbi5hZGRDaGlsZERlcGVuZGVuY3koaW5zdGFuY2UpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQzI6Okluc3RhbmNlJywge1xuICAgICAgRGVwZW5kc09uOiBbXG4gICAgICAgICdEYkNsdXN0ZXJJbnN0YW5jZTE1NTgzNUNFNScsXG4gICAgICAgICdJbnN0YW5jZUluc3RhbmNlUm9sZUU5Nzg1REU1JyxcbiAgICAgIF0sXG4gICAgfSwgUmVzb3VyY2VQYXJ0LkNvbXBsZXRlRGVmaW5pdGlvbikpO1xuICB9KTtcblxuICB0ZXN0KCdhZGQgY2hpbGQgZGVwZW5kZW5jeSB0byBhdHRyaWJ1dGVzJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgZG9jZGIgPSBEYXRhYmFzZUNsdXN0ZXIuZnJvbURhdGFiYXNlQ2x1c3RlckF0dHJpYnV0ZXMoc3RhY2ssICdEYXRhYmFzZScsIHtcbiAgICAgIGNsdXN0ZXJFbmRwb2ludEFkZHJlc3M6ICdhZGRyJyxcbiAgICAgIGNsdXN0ZXJJZGVudGlmaWVyOiAnaWRlbnRpZmllcicsXG4gICAgICBpbnN0YW5jZUVuZHBvaW50QWRkcmVzc2VzOiBbJ2FkZHInXSxcbiAgICAgIGluc3RhbmNlSWRlbnRpZmllcnM6IFsnaWRlbnRpZmllciddLFxuICAgICAgcG9ydDogMzMwNixcbiAgICAgIHJlYWRlckVuZHBvaW50QWRkcmVzczogJ3JlYWRlci1hZGRyZXNzJyxcbiAgICAgIHNlY3VyaXR5R3JvdXA6IFNlY3VyaXR5R3JvdXAuZnJvbVNlY3VyaXR5R3JvdXBJZChzdGFjaywgJ1NHJywgJ3NnLTEyMzQ1Njc4OScsIHtcbiAgICAgICAgYWxsb3dBbGxPdXRib3VuZDogZmFsc2UsXG4gICAgICB9KSxcbiAgICB9KTtcbiAgICBjb25zdCBjb25uZWN0aW9uID0gRGF0YWJhc2VDb25uZWN0aW9uLmZvckRvY0RCKHtkYXRhYmFzZTogZG9jZGIsIGxvZ2luOiBkYXRhYmFzZS5zZWNyZXQhfSk7XG4gICAgY29uc3QgaW5zdGFuY2UgPSBuZXcgSW5zdGFuY2Uoc3RhY2ssICdJbnN0YW5jZScsIHtcbiAgICAgIHZwYyxcbiAgICAgIGluc3RhbmNlVHlwZTogbmV3IEluc3RhbmNlVHlwZSgndDMuc21hbGwnKSxcbiAgICAgIG1hY2hpbmVJbWFnZTogTWFjaGluZUltYWdlLmxhdGVzdEFtYXpvbkxpbnV4KHsgZ2VuZXJhdGlvbjogQW1hem9uTGludXhHZW5lcmF0aW9uLkFNQVpPTl9MSU5VWF8yIH0pLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbm5lY3Rpb24uYWRkQ2hpbGREZXBlbmRlbmN5KGluc3RhbmNlKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUMyOjpJbnN0YW5jZScsIHtcbiAgICAgIERlcGVuZHNPbjogW1xuICAgICAgICAnSW5zdGFuY2VJbnN0YW5jZVJvbGVFOTc4NURFNScsXG4gICAgICBdLFxuICAgIH0sIFJlc291cmNlUGFydC5Db21wbGV0ZURlZmluaXRpb24pKTtcbiAgfSk7XG5cbiAgdGVzdCgnYWRkIGNoaWxkIGRlcGVuZGVuY3kgdGhyb3dzIHdoZW4gY2x1c3RlciBpbXBsZW1lbnRhdGlvbiBjaGFuZ2VkJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3QgZG9jZGIgPSBEYXRhYmFzZUNsdXN0ZXIuZnJvbURhdGFiYXNlQ2x1c3RlckF0dHJpYnV0ZXMoc3RhY2ssICdEYXRhYmFzZScsIHtcbiAgICAgIGNsdXN0ZXJFbmRwb2ludEFkZHJlc3M6ICdhZGRyJyxcbiAgICAgIGNsdXN0ZXJJZGVudGlmaWVyOiAnaWRlbnRpZmllcicsXG4gICAgICBpbnN0YW5jZUVuZHBvaW50QWRkcmVzc2VzOiBbJ2FkZHInXSxcbiAgICAgIGluc3RhbmNlSWRlbnRpZmllcnM6IFsnaWRlbnRpZmllciddLFxuICAgICAgcG9ydDogMzMwNixcbiAgICAgIHJlYWRlckVuZHBvaW50QWRkcmVzczogJ3JlYWRlci1hZGRyZXNzJyxcbiAgICAgIHNlY3VyaXR5R3JvdXA6IFNlY3VyaXR5R3JvdXAuZnJvbVNlY3VyaXR5R3JvdXBJZChzdGFjaywgJ1NHJywgJ3NnLTEyMzQ1Njc4OScsIHtcbiAgICAgICAgYWxsb3dBbGxPdXRib3VuZDogZmFsc2UsXG4gICAgICB9KSxcbiAgICB9KTtcbiAgICBjb25zdCBjb25uZWN0aW9uID0gRGF0YWJhc2VDb25uZWN0aW9uLmZvckRvY0RCKHtkYXRhYmFzZTogZG9jZGIsIGxvZ2luOiBkYXRhYmFzZS5zZWNyZXQhfSk7XG4gICAgY29uc3QgaW5zdGFuY2UgPSBuZXcgSW5zdGFuY2Uoc3RhY2ssICdJbnN0YW5jZScsIHtcbiAgICAgIHZwYyxcbiAgICAgIGluc3RhbmNlVHlwZTogbmV3IEluc3RhbmNlVHlwZSgndDMuc21hbGwnKSxcbiAgICAgIG1hY2hpbmVJbWFnZTogTWFjaGluZUltYWdlLmxhdGVzdEFtYXpvbkxpbnV4KHsgZ2VuZXJhdGlvbjogQW1hem9uTGludXhHZW5lcmF0aW9uLkFNQVpPTl9MSU5VWF8yIH0pLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGRvY2RiLm5vZGUuZGVmYXVsdENoaWxkID0gZGF0YWJhc2U7IC8vIFRyaWNrIGFkZENoaWxkRGVwZW5kZW5jeSgpIGludG8gdGhpbmtpbmcgdGhpcyBpcyBhIHJlYWwgY29uc3RydWN0LlxuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICBjb25uZWN0aW9uLmFkZENoaWxkRGVwZW5kZW5jeShpbnN0YW5jZSk7XG4gICAgfSkudG9UaHJvd0Vycm9yKC9UaGUgaW50ZXJuYWwgaW1wbGVtZW50YXRpb24gb2YgdGhlIEFXUyBDREsncyBEb2N1bWVudERCIGNsdXN0ZXIgY29uc3RydWN0IG1heSBoYXZlIGNoYW5nZWQuLyk7XG4gIH0pO1xuXG4gIHRlc3QoJ2Fzc2VydHMgbGludXgtb25seScsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGNvbm5lY3Rpb24gPSBEYXRhYmFzZUNvbm5lY3Rpb24uZm9yRG9jREIoe2RhdGFiYXNlLCBsb2dpbjogZGF0YWJhc2Uuc2VjcmV0IX0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGluc3RhbmNlID0gbmV3IEluc3RhbmNlKHN0YWNrLCAnSW5zdGFuY2UnLCB7XG4gICAgICB2cGMsXG4gICAgICBpbnN0YW5jZVR5cGU6IG5ldyBJbnN0YW5jZVR5cGUoJ3QzLnNtYWxsJyksXG4gICAgICBtYWNoaW5lSW1hZ2U6IE1hY2hpbmVJbWFnZS5sYXRlc3RXaW5kb3dzKFdpbmRvd3NWZXJzaW9uLldJTkRPV1NfU0VSVkVSXzIwMTlfRU5HTElTSF9GVUxMX0JBU0UpLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICBjb25uZWN0aW9uLmFkZEluc3RhbGxlckRCQXJncyhpbnN0YW5jZSk7XG4gICAgfSkudG9UaHJvd0Vycm9yKCdDYW4gb25seSBpbnN0YWxsIERlYWRsaW5lIGZyb20gYSBMaW51eCBpbnN0YW5jZS4nKTtcbiAgICBleHBlY3QoKCkgPT4ge1xuICAgICAgY29ubmVjdGlvbi5hZGRDb25uZWN0aW9uREJBcmdzKGluc3RhbmNlKTtcbiAgICB9KS50b1Rocm93RXJyb3IoJ0Nvbm5lY3RpbmcgdG8gdGhlIERlYWRsaW5lIERhdGFiYXNlIGlzIGN1cnJlbnRseSBvbmx5IHN1cHBvcnRlZCBmb3IgTGludXguJyk7XG4gIH0pO1xuXG59KTtcblxuZGVzY3JpYmUoJ01vbmdvREInLCAoKSA9PiB7XG4gIGxldCBzdGFjazogU3RhY2s7XG4gIGxldCB2cGM6IFZwYztcbiAgbGV0IGRhdGFiYXNlOiBJTW9uZ29EYjtcbiAgbGV0IGNsaWVudENlcnQ6IFg1MDlDZXJ0aWZpY2F0ZVBlbTtcblxuICBiZWZvcmVFYWNoKCgpID0+IHtcbiAgICBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIHZwYyA9IG5ldyBWcGMoc3RhY2ssICdWUEMnKTtcbiAgICBjb25zdCBob3N0bmFtZSA9ICdtb25nbyc7XG4gICAgY29uc3Qgem9uZU5hbWUgPSAnZGVhZGxpbmUuaW50ZXJuYWwnO1xuICAgIGNvbnN0IGRuc1pvbmUgPSBuZXcgUHJpdmF0ZUhvc3RlZFpvbmUoc3RhY2ssICdQcml2YXRlSG9zdGVkWm9uZScsIHtcbiAgICAgIHZwYyxcbiAgICAgIHpvbmVOYW1lLFxuICAgIH0pO1xuICAgIGNvbnN0IGNhQ2VydCA9IG5ldyBYNTA5Q2VydGlmaWNhdGVQZW0oc3RhY2ssICdDYUNlcnQnLCB7XG4gICAgICBzdWJqZWN0OiB7XG4gICAgICAgIGNuOiAnRGlzdGluZ3Vpc2hlZE5hbWUnLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICBjb25zdCBzZXJ2ZXJDZXJ0ID0gbmV3IFg1MDlDZXJ0aWZpY2F0ZVBlbShzdGFjaywgJ1NlcnZlckNlcnQnLCB7XG4gICAgICBzdWJqZWN0OiB7XG4gICAgICAgIGNuOiBgJHtob3N0bmFtZX0uJHt6b25lTmFtZX1gLFxuICAgICAgfSxcbiAgICAgIHNpZ25pbmdDZXJ0aWZpY2F0ZTogY2FDZXJ0LFxuICAgIH0pO1xuICAgIGNsaWVudENlcnQgPSBuZXcgWDUwOUNlcnRpZmljYXRlUGVtKHN0YWNrLCAnQ2xpZW50Q2VydCcsIHtcbiAgICAgIHN1YmplY3Q6IHtcbiAgICAgICAgY246ICdkYnVzZXInLFxuICAgICAgICBvdTogJ1Rlc3RDbGllbnQnLFxuICAgICAgfSxcbiAgICAgIHNpZ25pbmdDZXJ0aWZpY2F0ZTogY2FDZXJ0LFxuICAgIH0pO1xuXG4gICAgZGF0YWJhc2UgPSBuZXcgTW9uZ29EYkluc3RhbmNlKHN0YWNrLCAnTW9uZ29EYicsIHtcbiAgICAgIHZwYyxcbiAgICAgIG1vbmdvRGI6IHtcbiAgICAgICAgdXNlclNzcGxBY2NlcHRhbmNlOiBNb25nb0RiU3NwbExpY2Vuc2VBY2NlcHRhbmNlLlVTRVJfQUNDRVBUU19TU1BMLFxuICAgICAgICB2ZXJzaW9uOiBNb25nb0RiVmVyc2lvbi5DT01NVU5JVFlfM182LFxuICAgICAgICBob3N0bmFtZSxcbiAgICAgICAgZG5zWm9uZSxcbiAgICAgICAgc2VydmVyQ2VydGlmaWNhdGU6IHNlcnZlckNlcnQsXG4gICAgICB9LFxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdHcmFudHMgYWNjZXNzIHRvIGNlcnRpZmljYXRlIFNlY3JldHMnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCByb2xlID0gbmV3IFJvbGUoc3RhY2ssICdSb2xlJywge2Fzc3VtZWRCeTogbmV3IEFjY291bnRSb290UHJpbmNpcGFsKCl9KTtcbiAgICBjb25zdCBjb25uZWN0aW9uID0gRGF0YWJhc2VDb25uZWN0aW9uLmZvck1vbmdvRGJJbnN0YW5jZSh7ZGF0YWJhc2UsIGNsaWVudENlcnRpZmljYXRlOiBjbGllbnRDZXJ0fSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29ubmVjdGlvbi5ncmFudFJlYWQocm9sZSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0Q0RLKHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycsXG4gICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAnc2VjcmV0c21hbmFnZXI6R2V0U2VjcmV0VmFsdWUnLFxuICAgICAgICAgICAgICAnc2VjcmV0c21hbmFnZXI6RGVzY3JpYmVTZWNyZXQnLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICAgICdDbGllbnRDZXJ0JyxcbiAgICAgICAgICAgICAgICAnQ2VydCcsXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICdzZWNyZXRzbWFuYWdlcjpHZXRTZWNyZXRWYWx1ZScsXG4gICAgICAgICAgICAgICdzZWNyZXRzbWFuYWdlcjpEZXNjcmliZVNlY3JldCcsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgICAgUmVmOiAnQ2xpZW50Q2VydFBhc3NwaHJhc2U4QTcxRTFFMScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pKTtcbiAgfSk7XG5cbiAgdGVzdCgnYWRkSW5zdGFsbGVyREJBcmdzIGRlZmluZXMgcmVxdWlyZWQgZWxlbWVudHMnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBjb25uZWN0aW9uID0gRGF0YWJhc2VDb25uZWN0aW9uLmZvck1vbmdvRGJJbnN0YW5jZSh7ZGF0YWJhc2UsIGNsaWVudENlcnRpZmljYXRlOiBjbGllbnRDZXJ0fSk7XG4gICAgY29uc3QgaW5zdGFuY2UgPSBuZXcgSW5zdGFuY2Uoc3RhY2ssICdJbnN0YW5jZScsIHtcbiAgICAgIHZwYyxcbiAgICAgIGluc3RhbmNlVHlwZTogbmV3IEluc3RhbmNlVHlwZSgndDMuc21hbGwnKSxcbiAgICAgIG1hY2hpbmVJbWFnZTogTWFjaGluZUltYWdlLmxhdGVzdEFtYXpvbkxpbnV4KHsgZ2VuZXJhdGlvbjogQW1hem9uTGludXhHZW5lcmF0aW9uLkFNQVpPTl9MSU5VWF8yIH0pLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbm5lY3Rpb24uYWRkSW5zdGFsbGVyREJBcmdzKGluc3RhbmNlKTtcbiAgICBjb25zdCB1c2VyRGF0YSA9IGluc3RhbmNlLnVzZXJEYXRhLnJlbmRlcigpO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IHRva2VuID0gJyR7VG9rZW5bVE9LRU4uXFxcXGQrXX0nO1xuICAgIGV4cGVjdCh1c2VyRGF0YSkudG9NYXRjaChuZXcgUmVnRXhwKGVzY2FwZVRva2VuUmVnZXgoJ1xcJy90bXAvJyArIHRva2VuICsgdG9rZW4gKyAnXFwnICcgKyB0b2tlbiArICcgL29wdC9UaGlua2JveC9jZXJ0cy9tb25nb19jbGllbnQucGZ4JykpKTtcbiAgICBleHBlY3QodXNlckRhdGEpLnRvQ29udGFpbignY29uZmlndXJlX2RhdGFiYXNlX2luc3RhbGxhdGlvbl9hcmdzKCl7XFxuJyk7XG4gICAgZXhwZWN0KHVzZXJEYXRhKS50b0NvbnRhaW4oJ1xcbmV4cG9ydCAtZiBjb25maWd1cmVfZGF0YWJhc2VfaW5zdGFsbGF0aW9uX2FyZ3MnKTtcbiAgICBleHBlY3QodXNlckRhdGEpLnRvQ29udGFpbigneyBzZXQgK3g7IH0gMj4vZGV2L251bGwnKTtcbiAgICBleHBlY3QodXNlckRhdGEpLnRvQ29udGFpbignXFxuSU5TVEFMTEVSX0RCX0FSR1M9KCBbXCItLWRic3NsXCJdPXRydWUgW1wiLS1kYmF1dGhcIl09dHJ1ZSBbXCItLWRic3NsYXV0aFwiXT10cnVlJyk7XG4gICAgZXhwZWN0KHVzZXJEYXRhKS50b0NvbnRhaW4oJ1tcIi0tZGJob3N0XCJdPVwibW9uZ28uZGVhZGxpbmUuaW50ZXJuYWxcIicpO1xuICAgIGV4cGVjdCh1c2VyRGF0YSkudG9Db250YWluKCdbXCItLWRicG9ydFwiXT0yNzAxNycpO1xuICAgIGV4cGVjdCh1c2VyRGF0YSkudG9Db250YWluKCdbXCItLWRiY2xpZW50Y2VydFwiXT0nKTtcbiAgICBleHBlY3QodXNlckRhdGEpLnRvQ29udGFpbignW1wiLS1kYmNlcnRwYXNzXCJdPSRDRVJUX1BBU1NXT1JEJyk7XG4gIH0pO1xuXG4gIHRlc3QoJ2FkZENvbm5lY3Rpb25EQkFyZ3MgZGVmaW5lcyByZXF1aXJlZCBlbGVtZW50cycsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGNvbm5lY3Rpb24gPSBEYXRhYmFzZUNvbm5lY3Rpb24uZm9yTW9uZ29EYkluc3RhbmNlKHtkYXRhYmFzZSwgY2xpZW50Q2VydGlmaWNhdGU6IGNsaWVudENlcnR9KTtcbiAgICBjb25zdCBpbnN0YW5jZSA9IG5ldyBJbnN0YW5jZShzdGFjaywgJ0luc3RhbmNlJywge1xuICAgICAgdnBjLFxuICAgICAgaW5zdGFuY2VUeXBlOiBuZXcgSW5zdGFuY2VUeXBlKCd0My5zbWFsbCcpLFxuICAgICAgbWFjaGluZUltYWdlOiBNYWNoaW5lSW1hZ2UubGF0ZXN0QW1hem9uTGludXgoeyBnZW5lcmF0aW9uOiBBbWF6b25MaW51eEdlbmVyYXRpb24uQU1BWk9OX0xJTlVYXzIgfSksXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29ubmVjdGlvbi5hZGRDb25uZWN0aW9uREJBcmdzKGluc3RhbmNlKTtcbiAgICBjb25zdCB1c2VyRGF0YSA9IGluc3RhbmNlLnVzZXJEYXRhLnJlbmRlcigpO1xuXG4gICAgLy8gVEhFTlxuICAgIGNvbnN0IHRva2VuID0gJyR7VG9rZW5bVE9LRU4uXFxcXGQrXX0nO1xuICAgIGV4cGVjdCh1c2VyRGF0YSkudG9NYXRjaChuZXcgUmVnRXhwKGVzY2FwZVRva2VuUmVnZXgoJ1xcJy90bXAvJyArIHRva2VuICsgdG9rZW4gKyAnXFwnICcgKyB0b2tlbiArICcgL29wdC9UaGlua2JveC9jZXJ0cy9tb25nb19jbGllbnQucGZ4JykpKTtcbiAgICBleHBlY3QodXNlckRhdGEpLnRvQ29udGFpbignY29uZmlndXJlX2RlYWRsaW5lX2RhdGFiYXNlKCl7XFxuJyk7XG4gICAgZXhwZWN0KHVzZXJEYXRhKS50b0NvbnRhaW4oJ1xcbmV4cG9ydCAtZiBjb25maWd1cmVfZGVhZGxpbmVfZGF0YWJhc2UnKTtcbiAgICBleHBlY3QodXNlckRhdGEpLnRvQ29udGFpbigneyBzZXQgK3g7IH0gMj4vZGV2L251bGwnKTtcbiAgICBleHBlY3QodXNlckRhdGEpLnRvQ29udGFpbignXFxuZXhwb3J0IERCX0NFUlRfRklMRT0nKTtcbiAgICBleHBlY3QodXNlckRhdGEpLnRvQ29udGFpbignXFxuZXhwb3J0IERCX0NFUlRfUEFTU1dPUkQ9Jyk7XG4gIH0pO1xuXG4gIHRlc3QoJ2RlZmluZXMgcmVxdWlyZWQgY29udGFpbmVyIGVudmlyb25tZW50IHZhcmlhYmxlcycsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGNvbm5lY3Rpb24gPSBEYXRhYmFzZUNvbm5lY3Rpb24uZm9yTW9uZ29EYkluc3RhbmNlKHtkYXRhYmFzZSwgY2xpZW50Q2VydGlmaWNhdGU6IGNsaWVudENlcnR9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3QoY29ubmVjdGlvbi5jb250YWluZXJFbnZpcm9ubWVudCkudG9IYXZlUHJvcGVydHkoJ0RCX1RMU19DTElFTlRfQ0VSVF9VUkknKTtcbiAgICBleHBlY3QoY29ubmVjdGlvbi5jb250YWluZXJFbnZpcm9ubWVudCkudG9IYXZlUHJvcGVydHkoJ0RCX1RMU19DTElFTlRfQ0VSVF9QQVNTV09SRF9VUkknKTtcbiAgfSk7XG5cbiAgdGVzdCgnYWxsb3cgY29ubmVjdGlvbicsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGNvbm5lY3Rpb24gPSBEYXRhYmFzZUNvbm5lY3Rpb24uZm9yTW9uZ29EYkluc3RhbmNlKHtkYXRhYmFzZSwgY2xpZW50Q2VydGlmaWNhdGU6IGNsaWVudENlcnR9KTtcbiAgICBjb25zdCBzZWN1cml0eUdyb3VwID0gbmV3IFNlY3VyaXR5R3JvdXAoc3RhY2ssICdTRycsIHtcbiAgICAgIHZwYyxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25uZWN0aW9uLmFsbG93Q29ubmVjdGlvbnNGcm9tKHNlY3VyaXR5R3JvdXApO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdENESyhzdGFjaykudG8oaGF2ZVJlc291cmNlTGlrZSgnQVdTOjpFQzI6OlNlY3VyaXR5R3JvdXBJbmdyZXNzJywge1xuICAgICAgSXBQcm90b2NvbDogJ3RjcCcsXG4gICAgICBGcm9tUG9ydDogMjcwMTcsXG4gICAgICBUb1BvcnQ6IDI3MDE3LFxuICAgICAgU291cmNlU2VjdXJpdHlHcm91cElkOiB7XG4gICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICdTR0FEQjUzOTM3JyxcbiAgICAgICAgICAnR3JvdXBJZCcsXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgICAgR3JvdXBJZDoge1xuICAgICAgICAnRm46OkdldEF0dCc6IFtcbiAgICAgICAgICAnTW9uZ29EYlNlcnZlckFzZ0luc3RhbmNlU2VjdXJpdHlHcm91cENFNjIzMzM1JyxcbiAgICAgICAgICAnR3JvdXBJZCcsXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pKTtcbiAgfSk7XG5cbiAgdGVzdCgnYWRkIGNoaWxkIGRlcGVuZGVuY3knLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBjb25uZWN0aW9uID0gRGF0YWJhc2VDb25uZWN0aW9uLmZvck1vbmdvRGJJbnN0YW5jZSh7ZGF0YWJhc2UsIGNsaWVudENlcnRpZmljYXRlOiBjbGllbnRDZXJ0fSk7XG4gICAgY29uc3QgaW5zdGFuY2UgPSBuZXcgSW5zdGFuY2Uoc3RhY2ssICdJbnN0YW5jZScsIHtcbiAgICAgIHZwYyxcbiAgICAgIGluc3RhbmNlVHlwZTogbmV3IEluc3RhbmNlVHlwZSgndDMuc21hbGwnKSxcbiAgICAgIG1hY2hpbmVJbWFnZTogTWFjaGluZUltYWdlLmxhdGVzdEFtYXpvbkxpbnV4KHsgZ2VuZXJhdGlvbjogQW1hem9uTGludXhHZW5lcmF0aW9uLkFNQVpPTl9MSU5VWF8yIH0pLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbm5lY3Rpb24uYWRkQ2hpbGREZXBlbmRlbmN5KGluc3RhbmNlKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3RDREsoc3RhY2spLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6RUMyOjpJbnN0YW5jZScsIHtcbiAgICAgIERlcGVuZHNPbjogW1xuICAgICAgICAnSW5zdGFuY2VJbnN0YW5jZVJvbGVFOTc4NURFNScsXG4gICAgICAgICdNb25nb0RiU2VydmVyQXNnQVNHNDdCM0Q5NEUnLFxuICAgICAgXSxcbiAgICB9LCBSZXNvdXJjZVBhcnQuQ29tcGxldGVEZWZpbml0aW9uKSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2Fzc2VydHMgbGludXgtb25seScsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IGNvbm5lY3Rpb24gPSBEYXRhYmFzZUNvbm5lY3Rpb24uZm9yTW9uZ29EYkluc3RhbmNlKHtkYXRhYmFzZSwgY2xpZW50Q2VydGlmaWNhdGU6IGNsaWVudENlcnR9KTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCBpbnN0YW5jZSA9IG5ldyBJbnN0YW5jZShzdGFjaywgJ0luc3RhbmNlJywge1xuICAgICAgdnBjLFxuICAgICAgaW5zdGFuY2VUeXBlOiBuZXcgSW5zdGFuY2VUeXBlKCd0My5zbWFsbCcpLFxuICAgICAgbWFjaGluZUltYWdlOiBNYWNoaW5lSW1hZ2UubGF0ZXN0V2luZG93cyhXaW5kb3dzVmVyc2lvbi5XSU5ET1dTX1NFUlZFUl8yMDE5X0VOR0xJU0hfRlVMTF9CQVNFKSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3QoKCkgPT4ge1xuICAgICAgY29ubmVjdGlvbi5hZGRJbnN0YWxsZXJEQkFyZ3MoaW5zdGFuY2UpO1xuICAgIH0pLnRvVGhyb3dFcnJvcignQ2FuIG9ubHkgaW5zdGFsbCBEZWFkbGluZSBmcm9tIGEgTGludXggaW5zdGFuY2UuJyk7XG4gICAgZXhwZWN0KCgpID0+IHtcbiAgICAgIGNvbm5lY3Rpb24uYWRkQ29ubmVjdGlvbkRCQXJncyhpbnN0YW5jZSk7XG4gICAgfSkudG9UaHJvd0Vycm9yKCdDb25uZWN0aW5nIHRvIHRoZSBEZWFkbGluZSBEYXRhYmFzZSBpcyBjdXJyZW50bHkgb25seSBzdXBwb3J0ZWQgZm9yIExpbnV4LicpO1xuICB9KTtcbn0pO1xuIl19