"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkerInstanceFleet = void 0;
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
/**
 *  A new or imported Deadline Worker Fleet.
 */
class WorkerInstanceFleetBase extends core_1.Construct {
}
/**
 * This construct reperesents a fleet of Deadline Workers.
 *
 * The construct consists of an Auto Scaling Group (ASG) of instances using a provided AMI which has Deadline and any number
 * of render applications installed.  Whenever an instance in the ASG start it will connect Deadline to the desired render queue.
 *
 * When the worker fleet is deployed if it has been provided a HealthMonitor the Worker fleet will register itself against the Monitor
 * to ensure that the fleet remains healthy.
 *
 * Resources Deployed
 * ------------------------
 * 1) An AutoScalingGroup to maintain the number of instances;
 * 2) An Instance Role and corresponding IAM Policy;
 * 3) A script asset which is uploaded to your deployment bucket used to configure the worker so it can connect to the Render Queue
 * 4) An aws-rfdk.CloudWatchAgent to configure sending logs to cloudwatch.
 *
 * @ResourcesDeployed
 */
class WorkerInstanceFleet extends WorkerInstanceFleetBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.stack = core_1.Stack.of(scope);
        this.validateProps(props);
        // Launching the fleet with deadline workers.
        this.fleet = new aws_autoscaling_1.AutoScalingGroup(this, 'Default', {
            vpc: props.vpc,
            instanceType: (props.instanceType ? props.instanceType : aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE)),
            machineImage: props.workerMachineImage,
            keyName: props.keyName,
            vpcSubnets: props.vpcSubnets ? props.vpcSubnets : {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            securityGroup: props.securityGroup,
            minCapacity: props.minCapacity,
            maxCapacity: props.maxCapacity,
            desiredCapacity: props.desiredCapacity,
            resourceSignalTimeout: WorkerInstanceFleet.RESOURCE_SIGNAL_TIMEOUT,
            healthCheck: aws_autoscaling_1.HealthCheck.elb({
                grace: WorkerInstanceFleet.DEFAULT_HEALTH_CHECK_INTERVAL,
            }),
            role: props.role,
            spotPrice: (_a = props.spotPrice) === null || _a === void 0 ? void 0 : _a.toString(),
            blockDevices: props.blockDevices,
        });
        this.targetCapacity = parseInt(this.fleet.node.defaultChild.maxSize, 10);
        this.targetScope = this;
        this.targetToMonitor = this.fleet;
        this.targetCapacityMetric = new aws_cloudwatch_1.Metric({
            namespace: 'AWS/AutoScaling',
            metricName: 'GroupDesiredCapacity',
            dimensions: {
                AutoScalingGroupName: this.fleet.autoScalingGroupName,
            },
            label: 'GroupDesiredCapacity',
        });
        this.targetUpdatePolicy = new aws_iam_1.Policy(this, 'ASGUpdatePolicy', {
            statements: [new aws_iam_1.PolicyStatement({
                    actions: ['autoscaling:UpdateAutoScalingGroup'],
                    resources: [this.fleet.autoScalingGroupArn],
                })],
        });
        this.fleet.node.defaultChild.metricsCollection = [{
                granularity: '1Minute',
                metrics: ['GroupDesiredCapacity'],
            }];
        this.grantPrincipal = this.fleet.grantPrincipal;
        this.connections = this.fleet.connections;
        this.connections.allowToDefaultPort(props.renderQueue);
        let healthCheckPort = core_2.HealthMonitor.DEFAULT_HEALTH_CHECK_PORT;
        if (props.healthCheckConfig && props.healthCheckConfig.port) {
            healthCheckPort = props.healthCheckConfig.port;
        }
        // Configure the health monitoring if provided
        this.configureHealthMonitor(props, healthCheckPort);
        // Updating the user data with installation logs stream.
        this.configureCloudWatchLogStream(this.fleet, id, props.logGroupProps);
        props.renderQueue.configureClientInstance({
            host: this.fleet,
        });
        // Updating the user data with deadline repository installation commands.
        this.configureWorkerScript(this.fleet, props, healthCheckPort);
        // Updating the user data with successful cfn-signal commands.
        this.fleet.userData.addSignalOnExitCommand(this.fleet);
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Add the security group to all workers
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.fleet.addSecurityGroup(securityGroup);
    }
    configureCloudWatchLogStream(fleetInstance, id, logGroupProps) {
        const prefix = (logGroupProps === null || logGroupProps === void 0 ? void 0 : logGroupProps.logGroupPrefix) ? logGroupProps.logGroupPrefix : WorkerInstanceFleet.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, `${id}LogGroupWrapper`, `${id}`, defaultedLogGroupProps);
        logGroup.grantWrite(fleetInstance);
        const cloudWatchConfigurationBuilder = new core_2.CloudWatchConfigBuilder(core_1.Duration.seconds(15));
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'UserdataExecution', 'C:\\ProgramData\\Amazon\\EC2-Windows\\Launch\\Log\\UserdataExecution.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'WorkerLogs', 'C:\\ProgramData\\Thinkbox\\Deadline10\\logs\\deadlineslave*.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'LauncherLogs', 'C:\\ProgramData\\Thinkbox\\Deadline10\\logs\\deadlinelauncher*.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'WorkerLogs', '/var/log/Thinkbox/Deadline10/deadlineslave*.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'LauncherLogs', '/var/log/Thinkbox/Deadline10/deadlinelauncher*.log');
        new core_2.CloudWatchAgent(this, 'WorkerFleetLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: fleetInstance,
        });
    }
    configureWorkerScript(fleetInstance, props, healthCheckPort) {
        const configureWorkerScriptAsset = core_2.ScriptAsset.fromPathConvention(this, 'WorkerConfigurationScript', {
            osType: fleetInstance.osType,
            baseName: 'configureWorker',
            rootDir: path.join(__dirname, '..', 'scripts/'),
        });
        // Converting to lower case, as groups and pools are all stored in lower case in deadline.
        const groups = props.groups ? props.groups.map(val => val.toLowerCase()).join(',') : '';
        const pools = props.pools ? props.pools.map(val => val.toLowerCase()).join(',') : '';
        configureWorkerScriptAsset.executeOn({
            host: fleetInstance,
            args: [
                `'${healthCheckPort}'`,
                `'${groups}'`,
                `'${pools}'`,
                `'${props.region || ''}'`,
            ],
        });
    }
    validateProps(props) {
        this.validateSpotPrice(props.spotPrice);
        this.validateArrayGroupsPoolsSyntax(props.groups, /^(?!none$)[a-zA-Z0-9-_]+$/i, 'groups');
        this.validateArrayGroupsPoolsSyntax(props.pools, /^(?!none$)[a-zA-Z0-9-_]+$/i, 'pools');
        this.validateRegion(props.region, /^(?!none$|all$|unrecognized$)[a-zA-Z0-9-_]+$/i);
        this.validateBlockDevices(props.blockDevices);
    }
    validateSpotPrice(spotPrice) {
        if (spotPrice && !(spotPrice >= WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT && spotPrice <= WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT)) {
            throw new Error(`Invalid value: ${spotPrice} for property 'spotPrice'. Valid values can be any decimal between ${WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT} and ${WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT}.`);
        }
    }
    validateRegion(region, regex) {
        if (region && !regex.test(region)) {
            throw new Error(`Invalid value: ${region} for property 'region'. Valid characters are A-Z, a-z, 0-9, - and _. ‘All’, ‘none’ and ‘unrecognized’ are reserved names that cannot be used.`);
        }
    }
    validateArrayGroupsPoolsSyntax(array, regex, property) {
        if (array) {
            array.forEach(value => {
                if (!regex.test(value)) {
                    throw new Error(`Invalid value: ${value} for property '${property}'. Valid characters are A-Z, a-z, 0-9, - and _. Also, group 'none' is reserved as the default group.`);
                }
            });
        }
    }
    validateBlockDevices(blockDevices) {
        if (blockDevices === undefined) {
            this.node.addWarning(`The worker-fleet ${this.node.id} is being created without being provided any block devices so the Source AMI's devices will be used. ` +
                'Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
        }
        else {
            blockDevices.forEach(device => {
                if (device.volume.ebsDevice === undefined) {
                    // Suppressed or Ephemeral Block Device
                    return;
                }
                // encrypted is not exposed as part of ebsDeviceProps so we need to confirm it exists then access it via [].
                // eslint-disable-next-line dot-notation
                if (('encrypted' in device.volume.ebsDevice === false) || ('encrypted' in device.volume.ebsDevice && !device.volume.ebsDevice['encrypted'])) {
                    this.node.addWarning(`The BlockDevice "${device.deviceName}" on the worker-fleet ${this.node.id} is not encrypted. ` +
                        'Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.');
                }
            });
        }
    }
    configureHealthMonitor(props, healthCheckPort) {
        if (props.healthMonitor) {
            props.healthMonitor.registerFleet(this, props.healthCheckConfig || {
                port: healthCheckPort,
            });
        }
        else {
            this.node.addWarning(`The worker-fleet ${this.node.id} is being created without a health monitor attached to it. This means that the fleet will not automatically scale-in to 0 if the workers are unhealthy.`);
        }
    }
}
exports.WorkerInstanceFleet = WorkerInstanceFleet;
/**
 * The min limit for spot price.
 */
WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT = 0.001;
/**
 * The max limit for spot price.
 */
WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT = 255;
/**
 * This determines worker's health based on any hardware or software issues of EC2 instance.
 * Resource Tracker does deep ping every 5 minutes. These checks should be more frequent so
 * that any EC2 level issues are identified ASAP. Hence setting it to 1 minute.
 */
WorkerInstanceFleet.DEFAULT_HEALTH_CHECK_INTERVAL = core_1.Duration.minutes(1);
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
WorkerInstanceFleet.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * Setting the default signal timeout to 15 min. This is the max time, a single instance is expected
 * to take for launch and execute the user-data for deadline worker configuration. As we are setting
 * failure signals in the user-data, any failure will terminate deployment immediately.
 */
WorkerInstanceFleet.RESOURCE_SIGNAL_TIMEOUT = core_1.Duration.minutes(15);
//# sourceMappingURL=data:application/json;base64,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