"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.UsageBasedLicensing = exports.UsageBasedLicense = void 0;
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
/**
 * Instances of this class represent a usage-based license for a particular product.
 * It encapsulates all of the information specific to a product that the UsageBasedLicensing
 * construct requires to interoperate with that product.
 */
class UsageBasedLicense {
    constructor(props) {
        this.licenseName = props.licenseName;
        this.ports = props.ports;
        this.limit = props.limit;
    }
    /**
     * Method for 3dsMax license limit.
     *
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static for3dsMax(limit) {
        return new UsageBasedLicense({
            licenseName: 'max',
            ports: [aws_ec2_1.Port.tcp(27002)],
            limit,
        });
    }
    /**
     * Method for Arnold license limit
     *
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forArnold(limit) {
        return new UsageBasedLicense({
            licenseName: 'arnold',
            ports: [aws_ec2_1.Port.tcp(5056), aws_ec2_1.Port.tcp(7056)],
            limit,
        });
    }
    /**
     * Method for Cinema 4D license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forCinema4D(limit) {
        return new UsageBasedLicense({
            licenseName: 'cinema4d',
            ports: [aws_ec2_1.Port.tcp(5057), aws_ec2_1.Port.tcp(7057)],
            limit,
        });
    }
    /**
     * Method for Clarisse license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forClarisse(limit) {
        return new UsageBasedLicense({
            licenseName: 'clarisse',
            ports: [aws_ec2_1.Port.tcp(40500)],
            limit,
        });
    }
    /**
     * Method for Houdini license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forHoudini(limit) {
        return new UsageBasedLicense({
            licenseName: 'houdini',
            ports: [aws_ec2_1.Port.tcp(1715)],
            limit,
        });
    }
    /**
     * Method for Katana license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forKatana(limit) {
        return new UsageBasedLicense({
            licenseName: 'katana',
            ports: [aws_ec2_1.Port.tcp(4101), aws_ec2_1.Port.tcp(6101)],
            limit,
        });
    }
    /**
     * Method for KeyShot license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forKeyShot(limit) {
        return new UsageBasedLicense({
            licenseName: 'keyshot',
            ports: [aws_ec2_1.Port.tcp(27003), aws_ec2_1.Port.tcp(2703)],
            limit,
        });
    }
    /**
     * Method for krakatoa license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forKrakatoa(limit) {
        return new UsageBasedLicense({
            licenseName: 'krakatoa',
            ports: [aws_ec2_1.Port.tcp(27000), aws_ec2_1.Port.tcp(2700)],
            limit,
        });
    }
    /**
     * Method for Mantra license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forMantra(limit) {
        return new UsageBasedLicense({
            licenseName: 'mantra',
            ports: [aws_ec2_1.Port.tcp(1716)],
            limit,
        });
    }
    /**
     * Method for maxwell license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forMaxwell(limit) {
        return new UsageBasedLicense({
            licenseName: 'maxwell',
            ports: [aws_ec2_1.Port.tcp(5055), aws_ec2_1.Port.tcp(7055)],
            limit,
        });
    }
    /**
     * Method for Maya license limit
     *
     * @remark 3ds-Max usage-based licenses are not available with the UsageBasedLicensing
     * construct that deploys Deadline 10.1.9.
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forMaya(limit) {
        return new UsageBasedLicense({
            licenseName: 'maya',
            ports: [aws_ec2_1.Port.tcp(27002), aws_ec2_1.Port.tcp(2702)],
            limit,
        });
    }
    /**
     * Method for Nuke license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forNuke(limit) {
        return new UsageBasedLicense({
            licenseName: 'nuke',
            ports: [aws_ec2_1.Port.tcp(4101), aws_ec2_1.Port.tcp(6101)],
            limit,
        });
    }
    /**
     * Method for RealFlow license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forRealFlow(limit) {
        return new UsageBasedLicense({
            licenseName: 'realflow',
            ports: [aws_ec2_1.Port.tcp(5055), aws_ec2_1.Port.tcp(7055)],
            limit,
        });
    }
    /**
     * Method for RedShift license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forRedShift(limit) {
        return new UsageBasedLicense({
            licenseName: 'redshift',
            ports: [aws_ec2_1.Port.tcp(5054), aws_ec2_1.Port.tcp(7054)],
            limit,
        });
    }
    /**
     * Method for V-Ray license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forVray(limit) {
        return new UsageBasedLicense({
            licenseName: 'vray',
            ports: [aws_ec2_1.Port.tcp(30306)],
            limit,
        });
    }
    /**
     * Method for Yeti license limit
     *
     * @param limit - The maximum number of rendering tasks that can have this UBL license checked out at the same time.
     *
     * @default - limit will be set to unlimited
     */
    static forYeti(limit) {
        return new UsageBasedLicense({
            licenseName: 'yeti',
            ports: [aws_ec2_1.Port.tcp(5053), aws_ec2_1.Port.tcp(7053)],
            limit,
        });
    }
}
exports.UsageBasedLicense = UsageBasedLicense;
/**
 * Constant used to signify unlimited overage.
 */
UsageBasedLicense.UNLIMITED = 2147483647;
/**
 * This construct is an implementation of the Deadline component that is required for Usage-based Licensing (UBL)
 * (see: https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/licensing-usage-based.html )
 * in a render farm.
 *
 * Internally this is implemented as one or more instances of the Deadline License Forwarder application set up
 * to communicate to the render queue and Thinkbox’s licensing system, and to allow ingress connections
 * from the worker nodes so that they can acquire licenses as needed.
 *
 * The Deadline License Forwarder is set up to run within an AWS ECS task.
 *
 * Access to the running License Forwarder is gated by a security group that, by default, allows no ingress;
 * when a Deadline Worker requires access to licensing, then the RFDK constructs will grant that worker’s security group
 * ingress on TCP port 17004 as well as other ports as required by the specific licenses being used.
 *
 * Note: This construct does not currently implement the Deadline License Forwarder's Web Forwarding functionality.
 * This construct is not usable in any China region.
 *
 * Resources Deployed
 * ------------------------
 * 1) The Auto Scaling Group (ASG) added to the Amazon ECS cluster that is hosting the Deadline License Forwarder for UBL.
 *    By default, creates one instance. The default instance type is C5 Large.
 * 2) Elastic Block Store device(s) associated with the EC2 instance(s) in the ASG. The default volume size is 30 GiB.
 * 3) The LogGroup if it doesn't exist already.
 *
 * @ResourcesDeployed
 */
class UsageBasedLicensing extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        const usageBasedLicenses = new Array();
        props.licenses.forEach(license => {
            usageBasedLicenses.push(`${license.licenseName}:${license.limit ? license.limit : UsageBasedLicense.UNLIMITED}`);
        });
        if (usageBasedLicenses.length < 1) {
            throw new Error('Should be specified at least one license with defined limit.');
        }
        this.cluster = new aws_ecs_1.Cluster(this, 'Cluster', { vpc: props.vpc });
        this.asg = this.cluster.addCapacity('ASG', {
            vpcSubnets: (_a = props.vpcSubnets) !== null && _a !== void 0 ? _a : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            instanceType: props.instanceType ? props.instanceType : aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.C5, aws_ec2_1.InstanceSize.LARGE),
            minCapacity: (_b = props.desiredCount) !== null && _b !== void 0 ? _b : 1,
            maxCapacity: (_c = props.desiredCount) !== null && _c !== void 0 ? _c : 1,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(30, { encrypted: true }),
                }],
        });
        const taskDefinition = new aws_ecs_1.TaskDefinition(this, 'TaskDefinition', {
            compatibility: aws_ecs_1.Compatibility.EC2,
            networkMode: aws_ecs_1.NetworkMode.HOST,
        });
        this.grantPrincipal = taskDefinition.taskRole;
        const containerEnv = {
            UBL_CERTIFICATES_URI: '',
            UBL_LIMITS: usageBasedLicenses.join(';'),
            ...props.renderQueue.configureClientECS({
                hosts: [this.asg],
                grantee: this,
            }),
        };
        containerEnv.UBL_CERTIFICATES_URI = props.certificateSecret.secretArn;
        props.certificateSecret.grantRead(taskDefinition.taskRole);
        const prefix = ((_d = props.logGroupProps) === null || _d === void 0 ? void 0 : _d.logGroupPrefix) ? props.logGroupProps.logGroupPrefix : UsageBasedLicensing.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...props.logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, 'LogGroupWrapper', id, defaultedLogGroupProps);
        logGroup.grantWrite(this.asg);
        const container = taskDefinition.addContainer('LicenseForwarderContainer', {
            image: props.images.licenseForwarder,
            environment: containerEnv,
            memoryReservationMiB: 1024,
            logging: aws_ecs_1.LogDriver.awsLogs({
                logGroup,
                streamPrefix: 'LicenseForwarder',
            }),
        });
        // Increase ulimits
        container.addUlimits({
            name: aws_ecs_1.UlimitName.NOFILE,
            softLimit: 200000,
            hardLimit: 200000,
        }, {
            name: aws_ecs_1.UlimitName.NPROC,
            softLimit: 64000,
            hardLimit: 64000,
        });
        this.service = new aws_ecs_1.Ec2Service(this, 'Service', {
            cluster: this.cluster,
            taskDefinition,
            desiredCount: props.desiredCount,
            placementConstraints: [aws_ecs_1.PlacementConstraint.distinctInstances()],
            // This is required to right-size our host capacity and not have the ECS service block on updates. We set a memory
            // reservation, but no memory limit on the container. This allows the container's memory usage to grow unbounded.
            // We want 1:1 container to container instances to not over-spend, but this comes at the price of down-time during
            // cloudformation updates.
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
        });
        // An explicit dependency is required from the service to the ASG providing its capacity.
        // See: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html
        this.service.node.addDependency(this.asg);
        this.node.defaultChild = this.service;
        this.connections.allowToDefaultPort(props.renderQueue);
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * This method grant access of worker fleet to ports that required
     *
     * @param workerFleet - worker fleet
     * @param licenses - UBL licenses
     */
    grantPortAccess(workerFleet, licenses) {
        licenses.forEach(license => {
            license.ports.forEach(port => {
                this.connections.allowFrom(workerFleet, port);
            });
        });
        this.connections.allowFrom(workerFleet, aws_ec2_1.Port.tcp(UsageBasedLicensing.LF_PORT));
    }
    /**
     * The connections object that allows you to control network egress/ingress to the License Forwarder.
     */
    get connections() {
        return this.service.connections;
    }
}
exports.UsageBasedLicensing = UsageBasedLicensing;
/**
 * The port that the License Forwarder listens on
 */
UsageBasedLicensing.LF_PORT = 17004;
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
UsageBasedLicensing.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
//# sourceMappingURL=data:application/json;base64,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