"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Repository = void 0;
const path = require("path");
const url_1 = require("url");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const database_connection_1 = require("./database-connection");
/**
 * This construct represents the main Deadline Repository which contains the central database and file system
 * that Deadline requires.
 *
 * When deployed this construct will start up a single instance which will run the Deadline Repository installer to
 * initialize the file system and database, the logs of which will be forwarded to Cloudwatch via a CloudWatchAgent.
 * After the installation is complete the instance will be shutdown.
 *
 * Whenever the stack is updated if a change is detected in the installer a new instance will be started, which will perform
 * a check on the existing Deadline Repository.  If they are compatible with the new installer an update will be performed
 * and the deployment will continue, otherwise the the deployment will be cancelled.
 * In either case the instance will be cleaned up.
 *
 * Resources Deployed
 * ------------------------
 * 1) Encrypted EFS File System - If no IFileSystem is provided;
 * 2) DocumentDB and DatabaseConnection - If no database connection is provided;
 * 3) Auto Scaling Group (ASG) with min & max capacity of 1 instance;
 * 4) Instance Role and corresponding IAM Policy
 * 5) A Script Asset which is uploaded to your deployment bucket to run the installer
 * 6) An aws-rfdk.CloudWatchAgent to configure sending logs to cloudwatch.
 *
 * @ResourcesDeployed
 */
class Repository extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q;
        super(scope, id);
        if (props.database && ((_a = props.backupOptions) === null || _a === void 0 ? void 0 : _a.databaseRetention)) {
            this.node.addWarning('Backup retention for database will not be applied since a database is not being created by this construct');
        }
        if (props.fileSystem && ((_b = props.removalPolicy) === null || _b === void 0 ? void 0 : _b.filesystem)) {
            this.node.addWarning('RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct');
        }
        if (props.database && ((_c = props.removalPolicy) === null || _c === void 0 ? void 0 : _c.database)) {
            this.node.addWarning('RemovalPolicy for database will not be applied since a database is not being created by this construct');
        }
        this.version = props.version;
        // Set up the Filesystem and Database components of the repository
        this.fileSystem = (_d = props.fileSystem) !== null && _d !== void 0 ? _d : new core_2.MountableEfs(this, {
            filesystem: new aws_efs_1.FileSystem(this, 'FileSystem', {
                vpc: props.vpc,
                vpcSubnets: (_e = props.vpcSubnets) !== null && _e !== void 0 ? _e : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
                encrypted: true,
                lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_14_DAYS,
                removalPolicy: (_g = (_f = props.removalPolicy) === null || _f === void 0 ? void 0 : _f.filesystem) !== null && _g !== void 0 ? _g : core_1.RemovalPolicy.RETAIN,
            }),
        });
        if (props.database) {
            this.databaseConnection = props.database;
            if (props.databaseAuditLogging !== undefined) {
                this.node.addWarning(`The parameter databaseAuditLogging only has an effect when the Repository is creating its own database. 
        Please ensure that the Database provided is configured correctly.`);
            }
        }
        else {
            const databaseAuditLogging = (_h = props.databaseAuditLogging) !== null && _h !== void 0 ? _h : true;
            /**
             * This option is part of enabling audit logging for DocumentDB; the other required part is the enabling of the CloudWatch exports below.
             *
             * For more information about audit logging in DocumentDB, see:  https://docs.aws.amazon.com/documentdb/latest/developerguide/event-auditing.html
             */
            const parameterGroup = databaseAuditLogging ? new aws_docdb_1.ClusterParameterGroup(this, 'ParameterGroup', {
                description: 'DocDB cluster parameter group with enabled audit logs',
                family: 'docdb3.6',
                parameters: {
                    audit_logs: 'enabled',
                },
            }) : undefined;
            const instances = (_j = props.documentDbInstanceCount) !== null && _j !== void 0 ? _j : Repository.DEFAULT_NUM_DOCDB_INSTANCES;
            const dbCluster = new aws_docdb_1.DatabaseCluster(this, 'DocumentDatabase', {
                masterUser: { username: 'DocDBUser' },
                instanceProps: {
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.LARGE),
                    vpc: props.vpc,
                    vpcSubnets: (_k = props.vpcSubnets) !== null && _k !== void 0 ? _k : { subnetType: aws_ec2_1.SubnetType.PRIVATE, onePerAz: true },
                },
                instances,
                backup: {
                    retention: (_m = (_l = props.backupOptions) === null || _l === void 0 ? void 0 : _l.databaseRetention) !== null && _m !== void 0 ? _m : Repository.DEFAULT_DATABASE_RETENTION_PERIOD,
                },
                parameterGroup,
                removalPolicy: (_p = (_o = props.removalPolicy) === null || _o === void 0 ? void 0 : _o.database) !== null && _p !== void 0 ? _p : core_1.RemovalPolicy.RETAIN,
            });
            if (databaseAuditLogging) {
                /**
                 * This option enable export audit logs to Amazon CloudWatch.
                 * This is second options that required for enable audit log.
                 */
                const cfnDB = dbCluster.node.findChild('Resource');
                cfnDB.enableCloudwatchLogsExports = ['audit'];
            }
            /* istanbul ignore next */
            if (!dbCluster.secret) {
                /* istanbul ignore next */
                throw new Error('DBCluster failed to get set up properly -- missing login secret.');
            }
            // This is a workaround because of the bug in CDK implementation:
            // autoMinorVersionUpgrade should be true by default but it's not.
            // This code can be removed once fixed in CDK.
            for (let i = 1; i <= instances; i++) {
                const docdbInstance = dbCluster.node.tryFindChild(`Instance${i}`);
                docdbInstance.autoMinorVersionUpgrade = true;
            }
            this.databaseConnection = database_connection_1.DatabaseConnection.forDocDB({
                database: dbCluster,
                login: dbCluster.secret,
            });
        }
        // Launching the instance which installs the deadline repository in the stack.
        this.installerGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Installer', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc: props.vpc,
            vpcSubnets: (_q = props.vpcSubnets) !== null && _q !== void 0 ? _q : {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            minCapacity: 1,
            maxCapacity: 1,
            resourceSignalTimeout: (props.repositoryInstallationTimeout || core_1.Duration.minutes(15)),
            updateType: aws_autoscaling_1.UpdateType.REPLACING_UPDATE,
            replacingUpdateMinSuccessfulInstancesPercent: 100,
        });
        this.node.defaultChild = this.installerGroup;
        // Ensure the DB is serving before we try to connect to it.
        this.databaseConnection.addChildDependency(this.installerGroup);
        // Updating the user data with installation logs stream -- ALWAYS DO THIS FIRST.
        this.configureCloudWatchLogStream(this.installerGroup, `${id}`, props.logGroupProps);
        this.setupDirectConnect(this.installerGroup, Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH);
        this.rootPrefix = props.repositoryInstallationPrefix || Repository.DEFAULT_REPO_PREFIX;
        if (path.posix.isAbsolute(this.rootPrefix)) {
            // If the input path is absolute, then we make it relative (to the root of the repo file-system)
            this.rootPrefix = path.posix.relative(path.posix.sep, this.rootPrefix);
        }
        const repositoryInstallationPath = path.posix.normalize(path.posix.join(Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH, this.rootPrefix));
        // Updating the user data with deadline repository installation commands.
        this.configureRepositoryInstallerScript(this.installerGroup, repositoryInstallationPath, props.version);
        this.configureSelfTermination();
        // Updating the user data with successful cfn-signal commands.
        this.installerGroup.userData.addSignalOnExitCommand(this.installerGroup);
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * @inheritdoc
     */
    configureClientECS(props) {
        var _a, _b;
        const hostMountPoint = (_a = props.containerInstances.filesystemMountPoint) !== null && _a !== void 0 ? _a : '/mnt/repo';
        const containerMountPoint = (_b = props.containers.filesystemMountPoint) !== null && _b !== void 0 ? _b : `/opt/Thinkbox/DeadlineRepository${this.version.majorVersion}`;
        // Set up a direct connection on the host machine. This:
        //  - grants IAM permissions to the role associated with the instance profile access to
        //    - the file-system
        //    - the DB secret containing the credentials
        //  - adds a security group ingress rule to the DB cluster and file-system
        //  - adds userdata commands to mount the repository file-system on the host
        props.containerInstances.hosts.forEach(host => {
            this.setupDirectConnect(host, hostMountPoint);
        });
        // Build up a mapping of environment variables that are used to configure the container's direct connection to the
        // repository
        const containerEnvironment = {
            REPO_URI: url_1.pathToFileURL(containerMountPoint).toString(),
        };
        // The role associated with the task definition needs access to connect to the database
        this.databaseConnection.grantRead(props.containers.taskDefinition.taskRole);
        // Add any environment variables specified by the connection
        Object.entries(this.databaseConnection.containerEnvironment).forEach((entry) => {
            const [envVarName, envVarValue] = entry;
            containerEnvironment[envVarName] = envVarValue;
        });
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.containers.taskDefinition.node.addDependency(this);
        // Configure a named volume in the task-definition that points to the container host's mount-point of the repository
        // file-system
        props.containers.taskDefinition.addVolume({
            name: Repository.ECS_VOLUME_NAME,
            host: {
                sourcePath: path.posix.normalize(path.posix.join(hostMountPoint, this.rootPrefix)),
            },
        });
        // Return the container connection. This data structure contains all the pieces needed to create containers
        // that can directly connect to the repository.
        return {
            containerEnvironment,
            readOnlyMountPoint: {
                containerPath: containerMountPoint,
                readOnly: true,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
            readWriteMountPoint: {
                containerPath: containerMountPoint,
                readOnly: false,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
        };
    }
    /**
     * @inheritdoc
     */
    configureClientInstance(props) {
        var _a;
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.host.node.addDependency(this);
        this.setupDirectConnect(props.host, props.mountPoint);
        const stack = core_1.Stack.of(this);
        const uuid = 'f625e47b-7aed-4879-9861-513a72145525';
        const uniqueId = 'DeadlineRepository' + props.host.osType + uuid.replace(/[-]/g, '');
        const configureDirectConnect = (_a = stack.node.tryFindChild(uniqueId)) !== null && _a !== void 0 ? _a : core_2.ScriptAsset.fromPathConvention(stack, uniqueId, {
            osType: props.host.osType,
            baseName: 'configureRepositoryDirectConnect',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        configureDirectConnect.grantRead(props.host);
        this.databaseConnection.addConnectionDBArgs(props.host);
        const repoPath = path.posix.normalize(path.posix.join(props.mountPoint, this.rootPrefix));
        configureDirectConnect.executeOn({
            host: props.host,
            args: [`"${repoPath}"`],
        });
    }
    /**
     * Set up direct connect to this repo for the given host. Specifically:
     *  - IAM permissions & security group access to the database.
     *  - mounting the repository filesystem
     *
     * @param host Host to setup.
     * @param repositoryMountPoint Absolute directory at which to mount the repo filesystem.
     *
     * @remark Only allowable for Windows hosts.
     */
    setupDirectConnect(host, repositoryMountPoint) {
        if (host.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            throw new Error('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
        }
        this.databaseConnection.grantRead(host);
        this.databaseConnection.allowConnectionsFrom(host);
        this.fileSystem.mountToLinuxInstance(host, {
            location: repositoryMountPoint,
        });
    }
    /**
     * Adds UserData commands to configure the CloudWatch Agent running on the instance that performs the repository
     * installation.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *   - The cloud-init log
     *   - The Deadline Repo's installer log
     *
     * @param installerGroup The instance that performs the Deadline Repository installation
     * @param logGroupProps
     */
    configureCloudWatchLogStream(installerGroup, groupName, logGroupProps) {
        const prefix = (logGroupProps === null || logGroupProps === void 0 ? void 0 : logGroupProps.logGroupPrefix) ? logGroupProps.logGroupPrefix : Repository.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, 'RepositoryLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(installerGroup);
        const cloudWatchConfigurationBuilder = new core_2.CloudWatchConfigBuilder(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'deadlineRepositoryInstallationLogs', '/tmp/bitrock_installer.log');
        new core_2.CloudWatchAgent(this, 'RepositoryInstallerLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: installerGroup,
        });
    }
    configureSelfTermination() {
        const tagKey = 'resourceLogicalId';
        /*
        Add a policy to the ASG that allows it to modify itself. We cannot add the ASG name in resources
        as it will cause cyclic dependency. Hence, using Condition Keys
        */
        const tagCondition = {};
        tagCondition[`autoscaling:ResourceTag/${tagKey}`] = this.node.uniqueId;
        core_1.Tag.add(this.installerGroup, tagKey, this.node.uniqueId);
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // Following policy is required to read the aws tags within the instance
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ec2:DescribeTags',
            ],
            resources: ['*'],
        }));
        // wait for the log flush interval to make sure that all the logs gets flushed.
        // this wait can be avoided in future by using a life-cycle-hook on 'TERMINATING' state.
        const terminationDelay = Math.ceil(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL.toMinutes({ integral: false }));
        this.installerGroup.userData.addOnExitCommands(`sleep ${terminationDelay}m`);
        // fetching the instance id and asg name and then setting all the capacity to 0 to terminate the installer.
        this.installerGroup.userData.addOnExitCommands('INSTANCE="$(curl http://169.254.169.254/latest/meta-data/instance-id)"');
        this.installerGroup.userData.addOnExitCommands('ASG="$(aws --region ' + core_1.Stack.of(this).region + ' ec2 describe-tags --filters "Name=resource-id,Values=${INSTANCE}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags[0].Value" --output text)"');
        this.installerGroup.userData.addOnExitCommands('aws --region ' + core_1.Stack.of(this).region + ' autoscaling update-auto-scaling-group --auto-scaling-group-name ${ASG} --min-size 0 --max-size 0 --desired-capacity 0');
    }
    configureRepositoryInstallerScript(installerGroup, installPath, version) {
        var _a;
        const installerScriptAsset = core_2.ScriptAsset.fromPathConvention(this, 'DeadlineRepositoryInstallerScript', {
            osType: installerGroup.osType,
            baseName: 'installDeadlineRepository',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        this.databaseConnection.addInstallerDBArgs(installerGroup);
        if (!((_a = version.linuxInstallers) === null || _a === void 0 ? void 0 : _a.repository)) {
            throw new Error('Version given to Repository must provide a Linux Repository installer.');
        }
        const linuxVersionString = version.linuxFullVersionString();
        if (!linuxVersionString) {
            throw new Error('Version given to Repository must provide a full Linux version string.');
        }
        version.linuxInstallers.repository.s3Bucket.grantRead(installerGroup, version.linuxInstallers.repository.objectKey);
        installerScriptAsset.executeOn({
            host: installerGroup,
            args: [
                `"s3://${version.linuxInstallers.repository.s3Bucket.bucketName}/${version.linuxInstallers.repository.objectKey}"`,
                `"${installPath}"`,
                linuxVersionString,
            ],
        });
    }
}
exports.Repository = Repository;
/**
 * Default file system mount path for repository
 */
Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH = '/mnt/efs/fs1';
/**
 * Default installation prefix for deadline repository.
 */
Repository.DEFAULT_REPO_PREFIX = 'DeadlineRepository';
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
Repository.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * How often Cloudwatch logs will be flushed.
 */
Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
/**
 * The name of the volume used in ECS task definitions to mount the repository file-system mounted on EC2 hosts into
 * containers.
 */
Repository.ECS_VOLUME_NAME = 'RepositoryFilesystem';
/**
 * The default number of DocDB instances if one isn't provided in the props.
 */
Repository.DEFAULT_NUM_DOCDB_INSTANCES = 1;
/**
 * Default retention period for DocumentDB automated backups if one isn't provided in the props.
 */
Repository.DEFAULT_DATABASE_RETENTION_PERIOD = core_1.Duration.days(15);
//# sourceMappingURL=data:application/json;base64,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