/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { IVpc, SubnetSelection } from '@aws-cdk/aws-ec2';
import { MountPoint, TaskDefinition } from '@aws-cdk/aws-ecs';
import { Construct, Duration, IConstruct, RemovalPolicy } from '@aws-cdk/core';
import { IMountableLinuxFilesystem, LogGroupFactoryProps } from '../../core';
import { DatabaseConnection } from './database-connection';
import { IHost } from './host-ref';
import { IVersion } from './version-ref';
/**
 * Configuration interface for specifying ECS container instances to permit connecting hosted ECS tasks to the repository
 */
export interface ECSContainerInstanceProps {
    /**
     * The set of hosts that will be hosting the containers.
     *
     * This can be AutoScalingGroups that make up the capacity of an Amazon ECS cluster, or individual instances.
     */
    readonly hosts: IHost[];
    /**
     * The path where the repository file-system is mounted on the container hosts.
     *
     * @default "/mnt/repo"
     */
    readonly filesystemMountPoint?: string;
}
/**
 * Configuration interface to directly connect an ECS task to the repository.
 */
export interface ECSTaskProps {
    /**
     * The task definition to connect to the repository.
     *
     * [disable-awslint:ref-via-interface]
     */
    readonly taskDefinition: TaskDefinition;
    /**
     * The path where the repository file-system is mounted within the container.
     *
     * @default "/opt/Thinkbox/DeadlineRepository{MAJOR_VER}"
     */
    readonly filesystemMountPoint?: string;
}
/**
 * The properties used to configure Deadline running in an Amazon EC2 ECS task to directly connect to the repository.
 */
export interface ECSDirectConnectProps {
    /**
     * Configuration of ECS host instances to permit connecting hosted ECS tasks to the repository
     */
    readonly containerInstances: ECSContainerInstanceProps;
    /**
     * Configuration to directly connect an ECS task to the repository.
     */
    readonly containers: ECSTaskProps;
}
/**
 * Interface that can be used to configure a {@link @aws-cdk/aws-ecs#ContainerDefinition} definition to directly connect
 * to the repository.
 */
export interface IContainerDirectRepositoryConnection {
    /**
     * Environment variables that configure a direct connection to the repository.
     */
    readonly containerEnvironment: {
        [name: string]: string;
    };
    /**
     * A {@link MountPoint} that can be used to create a read/write mount the repository file-system from the task's
     * container instance into a container. This can be used with the `addMountPoint` method of the
     * {@link @aws-cdk/aws-ecs#ContainerDefinition} instance.
     */
    readonly readWriteMountPoint: MountPoint;
    /**
     * A {@link MountPoint} that can be used to create a read/write mount the repository file-system from the task's
     * container instance into a container. This can be used with the `addMountPoint` method of the
     * {@link @aws-cdk/aws-ecs#ContainerDefinition} instance.
     */
    readonly readOnlyMountPoint: MountPoint;
}
/**
 *  The Properties used to configure Deadline, that is running in an Amazon EC2 instance, a direct connection with a repository.
 */
export interface InstanceDirectConnectProps {
    /**
     * The Instance/UserData which will directly connect to the Repository
     */
    readonly host: IHost;
    /**
     * The location where the Repositories file system will be mounted on the instance.
     */
    readonly mountPoint: string;
}
/**
 * Interface for Deadline Repository.
 */
export interface IRepository extends IConstruct {
    /**
     * The path to the Deadline Repository directory.
     *
     * This is expressed as a relative path from the root of the Deadline Repository file-system.
     */
    readonly rootPrefix: string;
    /**
     * The version of Deadline for Linux that is installed on this Repository.
     */
    readonly version: IVersion;
    /**
     * Configures an ECS Container Instance and Task Definition for deploying a Deadline Client that directly connects to
     * this repository.
     *
     * This includes:
     *   - Ingress to database & filesystem Security Groups, as required.
     *   - IAM Permissions for database & filesystem, as required.
     *   - Mounts the Repository File System via UserData
     *
     * @param props The props used to configure the Deadline client.
     * @returns A mapping of environment variable names and their values to set in the container
     */
    configureClientECS(props: ECSDirectConnectProps): IContainerDirectRepositoryConnection;
    /**
     * Configure a Deadline Client, that is running in an Amazon EC2 instance, for direct connection to this repository.
     * This includes:
     *   - Ingress to database & filesystem Security Groups, as required.
     *   - IAM Permissions for database & filesystem, as required.
     *   - Mounts the Repository File System via UserData
     *   - Configures Deadline to direct-connect to the Repository.
     *
     * @param props The props used to configure the Deadline client.
     */
    configureClientInstance(props: InstanceDirectConnectProps): void;
}
/**
 * Properties for backups of resources that are created by the Repository.
 */
export interface RepositoryBackupOptions {
    /**
     * If this Repository is creating its own Amazon DocumentDB database, then this specifies the retention period to
     * use on the database. If the Repository is not creating a DocumentDB database, because one was given,
     * then this property is ignored.
     * Please visit https://aws.amazon.com/documentdb/pricing/ to learn more about DocumentDB backup storage pricing.
     *
     * @default Duration.days(15)
     */
    readonly databaseRetention?: Duration;
}
export interface RepositoryRemovalPolicies {
    /**
     * If this Repository is creating its own Amazon DocumentDB database, then this specifies the retention policy to
     * use on the database. If the Repository is not creating a DocumentDB database, because one was given,
     * then this property is ignored.
     *
     * @default RemovalPolicy.RETAIN
     */
    readonly database?: RemovalPolicy;
    /**
     * If this Repository is creating its own Amazon Elastic File System (EFS), then this specifies the retention policy to
     * use on the filesystem. If the Repository is not creating an EFS, because one was given, then this property is ignored.
     *
     * @default RemovalPolicy.RETAIN
     */
    readonly filesystem?: RemovalPolicy;
}
/**
 * Properties for the Deadline repository
 */
export interface RepositoryProps {
    /**
     * VPC to launch the Repository In
     */
    readonly vpc: IVpc;
    /**
     * Version property to specify the version of deadline repository to be installed.
     * This, in future, would be an optional property. If not passed, it should fetch
     * the latest version of deadline. The current implementation of Version construct
     * only supports importing it with static values, hence keeping it mandatory for now.
     */
    readonly version: IVersion;
    /**
     * Properties for setting up the Deadline Repository's LogGroup in CloudWatch
     * @default - LogGroup will be created with all properties' default values to the LogGroup: /renderfarm/<construct id>
     */
    readonly logGroupProps?: LogGroupFactoryProps;
    /**
     * The length of time to wait for the repository installation before considering it as failure.
     *
     * The maximum value is 43200 (12 hours).
     *
     * @default Duration.minutes(15)
     */
    readonly repositoryInstallationTimeout?: Duration;
    /**
     * Specify the file system where the deadline repository needs to be initialized.
     *
     * @default An Encrypted EFS File System will be created
     */
    readonly fileSystem?: IMountableLinuxFilesystem;
    /**
     * The prefix for the deadline repository installation path on the given file system.
     *
     * @default: "/DeadlineRepository/"
     */
    readonly repositoryInstallationPrefix?: string;
    /**
     * Specify the database where the deadline schema needs to be initialized.
     *
     * @default A Document DB Cluster will be created with a single db.r5.large instance.
     */
    readonly database?: DatabaseConnection;
    /**
     * Define the removal policies for the resources that this Repository creates. These define what happens
     * to the resoureces when the stack that defines them is destroyed.
     *
     * @default RemovalPolicy.RETAIN for all resources
     */
    readonly removalPolicy?: RepositoryRemovalPolicies;
    /**
     * If this Repository is creating its own DocumentDB database, then this specifies if audit logging will be enabled
     *
     * Audit logs are a security best-practice. They record connection, data definition language (DDL), user management,
     * and authorization events within the database, and are useful for post-incident auditing. That is, they can help you
     * figure out what an unauthorized user, who gained access to your database, has done with that access.
     *
     * @default true
     */
    readonly databaseAuditLogging?: boolean;
    /**
     * If this Repository is creating its own Amazon DocumentDB database, then this specifies the number of
     * compute instances to be created.
     *
     * @default 1
     */
    readonly documentDbInstanceCount?: number;
    /**
     * All resources that are created by this Repository will be deployed to these Subnets. This includes the
     * Auto Scaling Group that is created for running the Repository Installer. If this Repository is creating
     * an Amazon DocumentDB database and/or Amazon Elastic File System (EFS), then this specifies the subnets
     * to which they are deployed.
     *
     * @default: Private subnets in the VPC
     */
    readonly vpcSubnets?: SubnetSelection;
    /**
     * Define the backup options for the resources that this Repository creates.
     *
     * @default Duration.days(15) for the database
     */
    readonly backupOptions?: RepositoryBackupOptions;
}
/**
 * This construct represents the main Deadline Repository which contains the central database and file system
 * that Deadline requires.
 *
 * When deployed this construct will start up a single instance which will run the Deadline Repository installer to
 * initialize the file system and database, the logs of which will be forwarded to Cloudwatch via a CloudWatchAgent.
 * After the installation is complete the instance will be shutdown.
 *
 * Whenever the stack is updated if a change is detected in the installer a new instance will be started, which will perform
 * a check on the existing Deadline Repository.  If they are compatible with the new installer an update will be performed
 * and the deployment will continue, otherwise the the deployment will be cancelled.
 * In either case the instance will be cleaned up.
 *
 * Resources Deployed
 * ------------------------
 * 1) Encrypted EFS File System - If no IFileSystem is provided;
 * 2) DocumentDB and DatabaseConnection - If no database connection is provided;
 * 3) Auto Scaling Group (ASG) with min & max capacity of 1 instance;
 * 4) Instance Role and corresponding IAM Policy
 * 5) A Script Asset which is uploaded to your deployment bucket to run the installer
 * 6) An aws-rfdk.CloudWatchAgent to configure sending logs to cloudwatch.
 *
 * @ResourcesDeployed
 */
export declare class Repository extends Construct implements IRepository {
    /**
     * Default file system mount path for repository
     */
    private static DEFAULT_FILE_SYSTEM_MOUNT_PATH;
    /**
     * Default installation prefix for deadline repository.
     */
    private static DEFAULT_REPO_PREFIX;
    /**
     * Default prefix for a LogGroup if one isn't provided in the props.
     */
    private static DEFAULT_LOG_GROUP_PREFIX;
    /**
     * How often Cloudwatch logs will be flushed.
     */
    private static CLOUDWATCH_LOG_FLUSH_INTERVAL;
    /**
     * The name of the volume used in ECS task definitions to mount the repository file-system mounted on EC2 hosts into
     * containers.
     */
    private static ECS_VOLUME_NAME;
    /**
     * The default number of DocDB instances if one isn't provided in the props.
     */
    private static DEFAULT_NUM_DOCDB_INSTANCES;
    /**
     * Default retention period for DocumentDB automated backups if one isn't provided in the props.
     */
    private static DEFAULT_DATABASE_RETENTION_PERIOD;
    /**
     * @inheritdoc
     */
    readonly rootPrefix: string;
    /**
     * @inheritdoc
     */
    readonly version: IVersion;
    /**
     * Connection object for the database for this repository.
     */
    readonly databaseConnection: DatabaseConnection;
    /**
     * The Linux-mountable filesystem that will store the Deadline repository filesystem contents.
     */
    readonly fileSystem: IMountableLinuxFilesystem;
    /**
     * The autoscaling group for this repository's installer-running instance.
     */
    private readonly installerGroup;
    constructor(scope: Construct, id: string, props: RepositoryProps);
    /**
     * @inheritdoc
     */
    configureClientECS(props: ECSDirectConnectProps): IContainerDirectRepositoryConnection;
    /**
     * @inheritdoc
     */
    configureClientInstance(props: InstanceDirectConnectProps): void;
    /**
     * Set up direct connect to this repo for the given host. Specifically:
     *  - IAM permissions & security group access to the database.
     *  - mounting the repository filesystem
     *
     * @param host Host to setup.
     * @param repositoryMountPoint Absolute directory at which to mount the repo filesystem.
     *
     * @remark Only allowable for Windows hosts.
     */
    private setupDirectConnect;
    /**
     * Adds UserData commands to configure the CloudWatch Agent running on the instance that performs the repository
     * installation.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *   - The cloud-init log
     *   - The Deadline Repo's installer log
     *
     * @param installerGroup The instance that performs the Deadline Repository installation
     * @param logGroupProps
     */
    private configureCloudWatchLogStream;
    private configureSelfTermination;
    private configureRepositoryInstallerScript;
}
