"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.RenderQueue = void 0;
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_ecs_patterns_1 = require("@aws-cdk/aws-ecs-patterns");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const rq_connection_1 = require("./rq-connection");
/**
 * Base class for Render Queue providers
 */
class RenderQueueBase extends core_1.Construct {
}
/**
 * The RenderQueue construct deploys an Elastic Container Service (ECS) service that serves Deadline's REST HTTP API
 * to Deadline Clients.
 *
 * Most Deadline clients will connect to a Deadline render farm via the the RenderQueue. The API provides Deadline
 * clients access to Deadline's database and repository file-system in a way that is secure, performant, and scalable.
 *
 * Resources Deployed
 * ------------------------
 * 1) An ECS cluster
 * 2) An EC2 auto-scaling group that provides the EC2 container instances that host the ECS service
 * 3) An ECS service with a task definition that deploys the RCS container
 * 4) A CloudWatch bucket for streaming logs from the RCS container
 * 5) An application load balancer, listener and target group that balance incoming traffic among the RCS containers
 *
 * @ResourcesDeployed
 */
class RenderQueue extends RenderQueueBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        super(scope, id);
        // The RCS does not currently support horizontal scaling behind a load-balancer, so we limit to at most one instance
        if (props.renderQueueSize && props.renderQueueSize.min !== undefined && props.renderQueueSize.min > 1) {
            throw new Error(`renderQueueSize.min cannot be greater than 1 - got ${props.renderQueueSize.min}`);
        }
        if (props.renderQueueSize && props.renderQueueSize.desired !== undefined && props.renderQueueSize.desired > 1) {
            throw new Error(`renderQueueSize.desired cannot be greater than 1 - got ${props.renderQueueSize.desired}`);
        }
        let externalProtocol;
        if ((_a = props.trafficEncryption) === null || _a === void 0 ? void 0 : _a.externalTLS) {
            externalProtocol = aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
            if ((props.trafficEncryption.externalTLS.acmCertificate === undefined) ===
                (props.trafficEncryption.externalTLS.rfdkCertificate === undefined)) {
                throw new Error('Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS.');
            }
            else if (props.trafficEncryption.externalTLS.rfdkCertificate) {
                if (props.trafficEncryption.externalTLS.rfdkCertificate.certChain === undefined) {
                    throw new Error('Provided rfdkCertificate does not contain a certificate chain.');
                }
                this.clientCert = new core_2.ImportedAcmCertificate(this, 'AcmCert', props.trafficEncryption.externalTLS.rfdkCertificate);
                this.certChain = props.trafficEncryption.externalTLS.rfdkCertificate.certChain;
            }
            else {
                if (props.trafficEncryption.externalTLS.acmCertificateChain === undefined) {
                    throw new Error('externalTLS.acmCertificateChain must be provided when using externalTLS.acmCertificate.');
                }
                this.clientCert = props.trafficEncryption.externalTLS.acmCertificate;
                this.certChain = props.trafficEncryption.externalTLS.acmCertificateChain;
            }
        }
        else {
            externalProtocol = aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP;
        }
        const internalProtocol = (_c = (_b = props.trafficEncryption) === null || _b === void 0 ? void 0 : _b.internalProtocol) !== null && _c !== void 0 ? _c : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS && !props.hostname) {
            throw new Error('A hostname must be provided when the external protocol is HTTPS');
        }
        this.cluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            vpc: props.vpc,
        });
        const minCapacity = (_e = (_d = props.renderQueueSize) === null || _d === void 0 ? void 0 : _d.min) !== null && _e !== void 0 ? _e : 1;
        if (minCapacity < 1) {
            throw new Error(`renderQueueSize.min capacity must be at least 1: got ${minCapacity}`);
        }
        this.asg = this.cluster.addCapacity('RCS Capacity', {
            vpcSubnets: (_f = props.vpcSubnets) !== null && _f !== void 0 ? _f : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            instanceType: (_g = props.instanceType) !== null && _g !== void 0 ? _g : new aws_ec2_1.InstanceType('c5.large'),
            minCapacity,
            desiredCapacity: (_h = props.renderQueueSize) === null || _h === void 0 ? void 0 : _h.desired,
            maxCapacity: 1,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    // See: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-ami-storage-config.html
                    // We want the volume to be encrypted. The default AMI size is 30-GiB.
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(30, { encrypted: true }),
                }],
            updateType: aws_autoscaling_1.UpdateType.ROLLING_UPDATE,
        });
        /**
         * The ECS-optimized AMI that is defaulted to when adding capacity to a cluster does not include the awscli or unzip
         * packages as is the case with the standard Amazon Linux AMI. These are required by RFDK scripts to configure the
         * direct connection on the host container instances.
         */
        this.asg.userData.addCommands('yum install -yq awscli unzip');
        const externalPortNumber = RenderQueue.RCS_PROTO_PORTS[externalProtocol];
        const internalPortNumber = RenderQueue.RCS_PROTO_PORTS[internalProtocol];
        this.logGroup = core_2.LogGroupFactory.createOrFetch(this, 'LogGroupWrapper', id, {
            logGroupPrefix: '/renderfarm/',
            ...props.logGroupProps,
        });
        this.logGroup.grantWrite(this.asg);
        const taskDefinition = this.createTaskDefinition({
            image: props.images.remoteConnectionServer,
            portNumber: internalPortNumber,
            protocol: internalProtocol,
            repository: props.repository,
        });
        this.taskDefinition = taskDefinition;
        // The fully-qualified domain name to use for the ALB
        let loadBalancerFQDN;
        if (props.hostname) {
            const label = (_j = props.hostname.hostname) !== null && _j !== void 0 ? _j : 'renderqueue';
            if (props.hostname.hostname && !RenderQueue.RE_VALID_HOSTNAME.test(label)) {
                throw new Error(`Invalid RenderQueue hostname: ${label}`);
            }
            loadBalancerFQDN = `${label}.${props.hostname.zone.zoneName}`;
        }
        const loadBalancer = new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, 'LB', {
            vpc: this.cluster.vpc,
            internetFacing: false,
            deletionProtection: (_k = props.deletionProtection) !== null && _k !== void 0 ? _k : true,
        });
        this.pattern = new aws_ecs_patterns_1.ApplicationLoadBalancedEc2Service(this, 'AlbEc2ServicePattern', {
            certificate: this.clientCert,
            cluster: this.cluster,
            desiredCount: (_l = props.renderQueueSize) === null || _l === void 0 ? void 0 : _l.desired,
            domainZone: (_m = props.hostname) === null || _m === void 0 ? void 0 : _m.zone,
            domainName: loadBalancerFQDN,
            listenerPort: externalPortNumber,
            loadBalancer,
            protocol: externalProtocol,
            taskDefinition,
            // This is required to right-size our host capacity and not have the ECS service block on updates. We set a memory
            // reservation, but no memory limit on the container. This allows the container's memory usage to grow unbounded.
            // We want 1:1 container to container instances to not over-spend, but this comes at the price of down-time during
            // cloudformation updates.
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
            // This is required to ensure that the ALB listener's security group does not allow any ingress by default.
            openListener: false,
        });
        // An explicit dependency is required from the Service to the Client certificate
        // Otherwise cloud formation will try to remove the cert before the ALB using it is disposed.
        if (this.clientCert) {
            this.pattern.node.addDependency(this.clientCert);
        }
        // An explicit dependency is required from the service to the ASG providing its capacity.
        // See: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html
        this.pattern.service.node.addDependency(this.asg);
        this.loadBalancer = this.pattern.loadBalancer;
        // Enabling dropping of invalid HTTP header fields on the load balancer to prevent http smuggling attacks.
        this.loadBalancer.setAttribute('routing.http.drop_invalid_header_fields.enabled', 'true');
        if (props.accessLogs) {
            const accessLogsBucket = props.accessLogs.destinationBucket;
            // Policies are applied according to
            // https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-access-logs.html
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:PutObject'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [`${accessLogsBucket.bucketArn}/*`],
                conditions: {
                    StringEquals: {
                        's3:x-amz-acl': 'bucket-owner-full-control',
                    },
                },
            }));
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:GetBucketAcl'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [accessLogsBucket.bucketArn],
            }));
            this.loadBalancer.logAccessLogs(accessLogsBucket, props.accessLogs.prefix);
        }
        // Ensure tasks are run on separate container instances
        this.pattern.service.addPlacementConstraints(aws_ecs_1.PlacementConstraint.distinctInstances());
        /**
         * Uses an escape-hatch to set the target group protocol to HTTPS. We cannot configure server certificate
         * validation, but at least traffic is encrypted and terminated at the application layer.
         */
        const listener = this.loadBalancer.node.findChild('PublicListener');
        this.listener = listener;
        const targetGroup = listener.node.findChild('ECSGroup');
        const targetGroupResource = targetGroup.node.defaultChild;
        targetGroupResource.protocol = aws_elasticloadbalancingv2_1.ApplicationProtocol[internalProtocol];
        targetGroupResource.port = internalPortNumber;
        this.grantPrincipal = taskDefinition.taskRole;
        this.connections = new aws_ec2_1.Connections({
            defaultPort: aws_ec2_1.Port.tcp(externalPortNumber),
            securityGroups: this.pattern.loadBalancer.connections.securityGroups,
        });
        this.endpoint = new core_2.ConnectableApplicationEndpoint({
            address: this.pattern.loadBalancer.loadBalancerDnsName,
            port: externalPortNumber,
            connections: this.connections,
            protocol: externalProtocol,
        });
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP) {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttp({
                endpoint: this.endpoint,
            });
        }
        else {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttps({
                endpoint: this.endpoint,
                caCert: this.certChain,
            });
        }
        this.node.defaultChild = taskDefinition;
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * @inheritdoc
     */
    configureClientECS(param) {
        param.hosts.forEach(host => this.addChildDependency(host));
        return this.rqConnection.configureClientECS(param);
    }
    /**
     * @inheritdoc
     */
    configureClientInstance(param) {
        this.addChildDependency(param.host);
        this.rqConnection.configureClientInstance(param);
    }
    /**
     * Add an ordering dependency to another Construct.
     *
     * All constructs in the child's scope will be deployed after the RenderQueue has been deployed and is ready to recieve traffic.
     *
     * This can be used to ensure that the RenderQueue is fully up and serving queries before a client attempts to connect to it.
     *
     * @param child The child to make dependent upon this RenderQueue.
     */
    addChildDependency(child) {
        // Narrowly define the dependencies to reduce the probability of cycles
        // ex: cycles that involve the security group of the RenderQueue & child.
        child.node.addDependency(this.listener);
        child.node.addDependency(this.taskDefinition);
    }
    createTaskDefinition(props) {
        const { image, portNumber, protocol, repository } = props;
        const taskDefinition = new aws_ecs_1.Ec2TaskDefinition(this, 'RCSTask');
        // Mount the repo filesystem to RenderQueue.HOST_REPO_FS_MOUNT_PATH
        const connection = repository.configureClientECS({
            containerInstances: {
                hosts: [this.asg],
            },
            containers: {
                taskDefinition,
            },
        });
        const environment = connection.containerEnvironment;
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            // Generate a self-signed X509 certificate, private key and passphrase for use by the RCS containers.
            // Note: the Application Load Balancer does not validate the certificate in any way.
            const rcsCertPem = new core_2.X509CertificatePem(this, 'TlsCaCertPem', {
                subject: {
                    cn: 'renderfarm.local',
                },
            });
            const rcsCertPkcs = new core_2.X509CertificatePkcs12(this, 'TlsRcsCertBundle', {
                sourceCertificate: rcsCertPem,
            });
            [rcsCertPem.cert, rcsCertPkcs.cert, rcsCertPkcs.passphrase].forEach(secret => {
                secret.grantRead(taskDefinition.taskRole);
            });
            environment.RCS_TLS_CA_CERT_URI = rcsCertPem.cert.secretArn;
            environment.RCS_TLS_CERT_URI = rcsCertPkcs.cert.secretArn;
            environment.RCS_TLS_CERT_PASSPHRASE_URI = rcsCertPkcs.passphrase.secretArn;
            environment.RCS_TLS_REQUIRE_CLIENT_CERT = 'no';
        }
        const containerDefinition = taskDefinition.addContainer('ContainerDefinition', {
            image,
            memoryReservationMiB: 2048,
            environment,
            logging: aws_ecs_1.LogDriver.awsLogs({
                logGroup: this.logGroup,
                streamPrefix: 'RCS',
            }),
        });
        containerDefinition.addMountPoints(connection.readWriteMountPoint);
        // Increase ulimits
        containerDefinition.addUlimits({
            name: aws_ecs_1.UlimitName.NOFILE,
            softLimit: 200000,
            hardLimit: 200000,
        }, {
            name: aws_ecs_1.UlimitName.NPROC,
            softLimit: 64000,
            hardLimit: 64000,
        });
        containerDefinition.addPortMappings({
            containerPort: portNumber,
            hostPort: portNumber,
        });
        return taskDefinition;
    }
}
exports.RenderQueue = RenderQueue;
/**
 * Container listening ports for each protocol.
 */
RenderQueue.RCS_PROTO_PORTS = {
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP]: 8080,
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS]: 4433,
};
/**
 * Regular expression that validates a hostname (portion in front of the subdomain).
 */
RenderQueue.RE_VALID_HOSTNAME = /^[a-z](?:[a-z0-9-]{0,61}[a-z0-9])?$/i;
//# sourceMappingURL=data:application/json;base64,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