/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { AutoScalingGroup } from '@aws-cdk/aws-autoscaling';
import { Connections, IConnectable } from '@aws-cdk/aws-ec2';
import { Cluster } from '@aws-cdk/aws-ecs';
import { ApplicationLoadBalancer } from '@aws-cdk/aws-elasticloadbalancingv2';
import { IGrantable, IPrincipal } from '@aws-cdk/aws-iam';
import { Construct, IConstruct } from '@aws-cdk/core';
import { ECSConnectOptions, InstanceConnectOptions, RenderQueueProps } from '.';
import { ConnectableApplicationEndpoint } from '../../core';
/**
 * Interface for Deadline Render Queue.
 */
export interface IRenderQueue extends IConstruct, IConnectable {
    /**
     * The endpoint used to connect to the Render Queue
     */
    readonly endpoint: ConnectableApplicationEndpoint;
    /**
     * Configures an ECS cluster to be able to connect to a RenderQueue
     * @returns An environment mapping that is used to configure the Docker Images
     */
    configureClientECS(params: ECSConnectOptions): {
        [name: string]: string;
    };
    /**
     * Configure an Instance/Autoscaling group to connect to a RenderQueue
     */
    configureClientInstance(params: InstanceConnectOptions): void;
}
/**
 * Base class for Render Queue providers
 */
declare abstract class RenderQueueBase extends Construct implements IRenderQueue {
    /**
     * The endpoint that Deadline clients can use to connect to the Render Queue
     */
    abstract readonly endpoint: ConnectableApplicationEndpoint;
    /**
     * Allows specifying security group connections for the Render Queue.
     */
    abstract readonly connections: Connections;
    /**
     * Configures an ECS cluster to be able to connect to a RenderQueue
     * @returns An environment mapping that is used to configure the Docker Images
     */
    abstract configureClientECS(params: ECSConnectOptions): {
        [name: string]: string;
    };
    /**
     * Configure an Instance/Autoscaling group to connect to a RenderQueue
     */
    abstract configureClientInstance(params: InstanceConnectOptions): void;
}
/**
 * The RenderQueue construct deploys an Elastic Container Service (ECS) service that serves Deadline's REST HTTP API
 * to Deadline Clients.
 *
 * Most Deadline clients will connect to a Deadline render farm via the the RenderQueue. The API provides Deadline
 * clients access to Deadline's database and repository file-system in a way that is secure, performant, and scalable.
 *
 * Resources Deployed
 * ------------------------
 * 1) An ECS cluster
 * 2) An EC2 auto-scaling group that provides the EC2 container instances that host the ECS service
 * 3) An ECS service with a task definition that deploys the RCS container
 * 4) A CloudWatch bucket for streaming logs from the RCS container
 * 5) An application load balancer, listener and target group that balance incoming traffic among the RCS containers
 *
 * @ResourcesDeployed
 */
export declare class RenderQueue extends RenderQueueBase implements IGrantable {
    /**
     * Container listening ports for each protocol.
     */
    private static readonly RCS_PROTO_PORTS;
    /**
     * Regular expression that validates a hostname (portion in front of the subdomain).
     */
    private static readonly RE_VALID_HOSTNAME;
    /**
     * The principal to grant permissions to.
     */
    readonly grantPrincipal: IPrincipal;
    /**
     * The Amazon ECS cluster that is hosting the fleet of Deadline RCS applications.
     */
    readonly cluster: Cluster;
    /**
     * @inheritdoc
     */
    readonly connections: Connections;
    /**
     * @inheritdoc
     */
    readonly endpoint: ConnectableApplicationEndpoint;
    /**
     * The application load balancer that serves the traffic.
     */
    readonly loadBalancer: ApplicationLoadBalancer;
    /**
     * The Amazon EC2 Auto Scaling Group within the {@link RenderQueue.cluster}
     * that contains the Deadline RCS's instances.
     */
    readonly asg: AutoScalingGroup;
    /**
     * The log group where the RCS container will log to
     */
    private readonly logGroup;
    /**
     * Instance of the Application Load Balanced EC2 service pattern.
     */
    private readonly pattern;
    /**
     * The certificate used by the ALB for external Traffic
     */
    private readonly clientCert?;
    /**
     * The connection object that contains the logic for how clients can connect to the Render Queue.
     */
    private readonly rqConnection;
    /**
     * The secret containing the cert chain for external connections.
     */
    private readonly certChain?;
    /**
     * The listener on the ALB that is redirecting traffic to the RCS.
     */
    private readonly listener;
    /**
     * The ECS task for the RCS.
     */
    private readonly taskDefinition;
    constructor(scope: Construct, id: string, props: RenderQueueProps);
    /**
     * @inheritdoc
     */
    configureClientECS(param: ECSConnectOptions): {
        [name: string]: string;
    };
    /**
     * @inheritdoc
     */
    configureClientInstance(param: InstanceConnectOptions): void;
    /**
     * Add an ordering dependency to another Construct.
     *
     * All constructs in the child's scope will be deployed after the RenderQueue has been deployed and is ready to recieve traffic.
     *
     * This can be used to ensure that the RenderQueue is fully up and serving queries before a client attempts to connect to it.
     *
     * @param child The child to make dependent upon this RenderQueue.
     */
    addChildDependency(child: IConstruct): void;
    private createTaskDefinition;
}
export {};
