"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseConnection = void 0;
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
/**
 * Helper class for connecting Thinkbox's Deadline to a specific Database.
 */
class DatabaseConnection {
    /**
     * Creates a DatabaseConnection which allows Deadline to connect to Amazon DocumentDB.
     *
     * Resources Deployed
     * ------------------------
     * This construct does not deploy any resources
     *
     * @ResourcesDeployed
     */
    static forDocDB(options) {
        return new DocDBDatabaseConnection(options);
    }
    /**
     * Creates a DatabaseConnection which allows Deadline to connect to MongoDB.
     *
     * Resources Deployed
     * ------------------------
     * This construct does not deploy any resources
     *
     * @ResourcesDeployed
     */
    static forMongoDbInstance(options) {
        return new MongoDbInstanceDatabaseConnection(options);
    }
}
exports.DatabaseConnection = DatabaseConnection;
/**
 * Specialization of {@link DatabaseConnection} targetting Amazon DocumentDB.
 */
class DocDBDatabaseConnection extends DatabaseConnection {
    constructor(props) {
        super();
        this.props = props;
        this.containerEnvironment = {
            // The container must fetch the credentials from Secrets Manager
            DB_CREDENTIALS_URI: this.props.login.secretArn,
        };
    }
    /**
     * @inheritdoc
     */
    addInstallerDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Can only install Deadline from a Linux instance.');
        }
        host.userData.addCommands('configure_database_installation_args(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export SECRET_STRING=\`aws secretsmanager get-secret-value --secret-id ${this.props.login.secretArn} --region ${core_1.Stack.of(this.props.login).region} | getJsonVal 'SecretString'\``, "DB_USERNAME=`printenv SECRET_STRING | getJsonVal 'username'`", "DB_PASSWORD=`printenv SECRET_STRING | getJsonVal 'password'`", 'unset SECRET_STRING', `INSTALLER_DB_ARGS=( ["--dbuser"]=$DB_USERNAME ["--dbpassword"]=$DB_PASSWORD ["--dbhost"]=${this.props.database.clusterEndpoint.hostname}` +
            ` ["--dbport"]=${this.props.database.clusterEndpoint.portAsString()} ["--dbtype"]=DocumentDB )`, 'unset DB_USERNAME', 'unset DB_PASSWORD', 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_database_installation_args');
    }
    /**
     * @inheritdoc
     */
    addConnectionDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Connecting to the Deadline Database is currently only supported for Linux.');
        }
        host.userData.addCommands('configure_deadline_database(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export SECRET_STRING=\`aws secretsmanager get-secret-value --secret-id ${this.props.login.secretArn} --region ${core_1.Stack.of(this.props.login).region} | getJsonVal 'SecretString'\``, "DB_USERNAME=`printenv SECRET_STRING | getJsonVal 'username'`", "DB_PASSWORD=`printenv SECRET_STRING | getJsonVal 'password'`", 'unset SECRET_STRING', 'sudo -u ec2-user "${deadlinecommand}" -StoreDatabasecredentials "${DB_USERNAME}" "${DB_PASSWORD}"', 'unset DB_USERNAME', 'unset DB_PASSWORD', 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_deadline_database');
    }
    /**
     * @inheritdoc
     */
    allowConnectionsFrom(other) {
        other.connections.allowTo(this.props.database, this.props.database.connections.defaultPort);
    }
    /**
     * @inheritdoc
     */
    grantRead(grantee) {
        this.props.login.grantRead(grantee);
    }
    /**
     * @inheritdoc
     */
    addChildDependency(child) {
        // To depend on document DB it is not sufficient to depend on the Cluster. The instances are what serves data, so
        // we must add a dependency to an instance in the DocDB cluster.
        // The DocDB L2 does not expose any of its instances as properties, so we have to escape-hatch to gain access.
        const docdbInstance = this.props.database.node.tryFindChild('Instance1');
        // We won't find an instance in two situations:
        //  1) The DocDB Cluster was created from attributes. In this case, the DocDB pre-exists the stack and there's no need
        //     to depend on anything.
        //  2) The DocDB Cluster was constructed, but the internal name for the instance has been changed from 'Instance1'; this is
        //     unlikely, but not impossible.
        // We can differentiate cases (1) & (2) by looking for the defaultChild on the cluster. The version from attributes will not have one.
        if (docdbInstance) {
            child.node.addDependency(docdbInstance);
        }
        else if (this.props.database.node.defaultChild) {
            throw new Error('The internal implementation of the AWS CDK\'s DocumentDB cluster construct may have changed. Please update to a newer RFDK for an updated implementation, or file a ticket if this is the latest release.');
        }
    }
}
/**
 * Specialization of {@link DatabaseConnection} targetting MongoDB.
 */
class MongoDbInstanceDatabaseConnection extends DatabaseConnection {
    constructor(props) {
        super();
        this.props = props;
        this.containerEnvironment = {
            DB_TLS_CLIENT_CERT_URI: props.clientCertificate.cert.secretArn,
            DB_TLS_CLIENT_CERT_PASSWORD_URI: props.clientCertificate.passphrase.secretArn,
        };
    }
    /**
     * @inheritdoc
     */
    allowConnectionsFrom(other) {
        other.connections.allowTo(this.props.database, aws_ec2_1.Port.tcp(this.props.database.port));
    }
    /**
     * @inheritdoc
     */
    addInstallerDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Can only install Deadline from a Linux instance.');
        }
        this.downloadCertificate(host);
        const certPwSecret = this.props.clientCertificate.passphrase;
        host.userData.addCommands('configure_database_installation_args(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 
        // Suppress -x, so no secrets go to the logs
        'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `CERT_PASSWORD=$(aws secretsmanager get-secret-value --secret-id ${certPwSecret.secretArn} --region ${core_1.Stack.of(certPwSecret).region} | getJsonVal 'SecretString')`, 'INSTALLER_DB_ARGS=( ["--dbssl"]=true ["--dbauth"]=true ["--dbsslauth"]=true ' +
            `["--dbhost"]="${this.props.database.fullHostname}" ["--dbport"]=${this.props.database.port} ` +
            `["--dbclientcert"]="${MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION}" ["--dbcertpass"]=$CERT_PASSWORD )`, 'unset CERT_PASSWORD', 
        // Restore -x, if it was set.
        'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_database_installation_args');
    }
    /**
     * @inheritdoc
     */
    addConnectionDBArgs(host) {
        if (host.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Connecting to the Deadline Database is currently only supported for Linux.');
        }
        this.downloadCertificate(host);
        const certPwSecret = this.props.clientCertificate.passphrase;
        host.userData.addCommands('configure_deadline_database(){', 'getJsonVal(){ python -c \'import json,sys;obj=json.load(sys.stdin);print obj["\'$1\'"]\'; }', 'SET_X_IS_SET=$-', '{ set +x; } 2>/dev/null', `export DB_CERT_FILE="${MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION}"`, `export DB_CERT_PASSWORD=$(aws secretsmanager get-secret-value --secret-id ${certPwSecret.secretArn} --region ${core_1.Stack.of(certPwSecret).region} | getJsonVal 'SecretString')`, 'if [[ $SET_X_IS_SET =~ x ]]; then set -x; else set +x; fi', '}', 'export -f configure_deadline_database');
    }
    /**
     * @inheritdoc
     */
    grantRead(grantee) {
        this.props.clientCertificate.cert.grantRead(grantee);
        this.props.clientCertificate.passphrase.grantRead(grantee);
    }
    /**
     * @inheritdoc
     */
    addChildDependency(child) {
        if (this.props.database.hasOwnProperty('server')) {
            const db = this.props.database;
            child.node.addDependency(db.server.autoscalingGroup.node.defaultChild);
        }
    }
    /**
     * Download the client PKCS#12 certificate for authenticating to the MongoDB, and place it into
     * the path defined by: DB_CERT_LOCATION
     * @param host
     */
    downloadCertificate(host) {
        var _a;
        const stack = core_1.Stack.of(host);
        const uuid = 'e8125dd2-ab2c-4861-8ee4-998c26b30ee0';
        const uniqueId = 'GetSecretToFile' + host.osType + uuid.replace(/[-]/g, '');
        const getSecretsScript = (_a = stack.node.tryFindChild(uniqueId)) !== null && _a !== void 0 ? _a : core_2.ScriptAsset.fromPathConvention(stack, uniqueId, {
            osType: host.osType,
            baseName: 'getSecretToFile',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        getSecretsScript.executeOn({
            host,
            args: [
                this.props.clientCertificate.cert.secretArn,
                MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION,
            ],
        });
    }
}
MongoDbInstanceDatabaseConnection.DB_CERT_LOCATION = '/opt/Thinkbox/certs/mongo_client.pfx';
//# sourceMappingURL=data:application/json;base64,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