"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
const tag_helpers_1 = require("./tag-helpers");
const token_regex_helpers_1 = require("./token-regex-helpers");
describe('Test MongoDbInstance', () => {
    let app;
    let stack;
    let vpc;
    let dnsZone;
    let caCert;
    let serverCert;
    const hostname = 'hostname';
    const zoneName = 'testZone';
    const version = lib_1.MongoDbVersion.COMMUNITY_3_6;
    const userSsplAcceptance = lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL;
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app, 'Stack');
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        dnsZone = new aws_route53_1.PrivateHostedZone(stack, 'PrivateHostedZone', {
            vpc,
            zoneName,
        });
        caCert = new lib_1.X509CertificatePem(stack, 'CaCert', {
            subject: {
                cn: 'DistinguishedName',
            },
        });
        serverCert = new lib_1.X509CertificatePem(stack, 'ServerCert', {
            subject: {
                cn: `${hostname}.${zoneName}`,
            },
            signingCertificate: caCert,
        });
    });
    test('default mongodb instance is created correctly', () => {
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: 'r5.large',
            BlockDeviceMappings: assert_1.arrayWith(assert_1.objectLike({
                Ebs: assert_1.objectLike({
                    Encrypted: true,
                }),
            })),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Route53::RecordSet', {
            Name: hostname + '.' + zoneName + '.',
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SecretsManager::Secret', {
            Description: `Admin credentials for the MongoDB database ${instance.node.uniqueId}`,
            GenerateSecretString: {
                ExcludeCharacters: '\"()$\'',
                ExcludePunctuation: true,
                GenerateStringKey: 'password',
                IncludeSpace: false,
                PasswordLength: 24,
                RequireEachIncludedType: true,
                SecretStringTemplate: '{\"username\":\"admin\"}',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Encrypted: true,
            Tags: assert_1.arrayWith(assert_1.objectLike({
                Key: 'VolumeGrantAttach-dff922f1fb3c0287b3759d461a48c4b8',
                Value: 'b0381797ae9723313d35ad6f9aa815f2',
            })),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: assert_1.objectLike({
                Statement: assert_1.arrayWith({
                    Action: [
                        'logs:CreateLogStream',
                        'logs:PutLogEvents',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':logs:',
                                {
                                    Ref: 'AWS::Region',
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':log-group:',
                                {
                                    'Fn::GetAtt': [
                                        'MongoDbInstanceMongoDbInstanceLogGroupWrapperEAF733BB',
                                        'LogGroupName',
                                    ],
                                },
                                ':*',
                            ],
                        ],
                    },
                }, {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Resource: [
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        Ref: asset_constants_1.MONGODB_INSTANCE_3_6_SCRIPT.Bucket,
                                    },
                                ],
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        Ref: asset_constants_1.MONGODB_INSTANCE_3_6_SCRIPT.Bucket,
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'Cert',
                        ],
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'CertChain',
                        ],
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'Key',
                        ],
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'ServerCertPassphraseE4C3CB38',
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'MongoDbInstanceAdminUser54147F2B',
                    },
                }),
            }),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
            LogGroupName: '/renderfarm/MongoDbInstance',
        }));
        const cloudInitLogPath = '/var/log/cloud-init-output.log';
        const cloudInitLogPrefix = 'cloud-init-output';
        const mongoLogPath = '/var/log/mongodb/mongod.log';
        const mongoLogPrefix = 'MongoDB';
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SSM::Parameter', {
            Description: 'config file for Repository logs config',
            Value: assert_1.objectLike({
                'Fn::Join': assert_1.arrayWith(assert_1.arrayWith('\",\"log_stream_name\":\"' + cloudInitLogPrefix + '-{instance_id}\",\"file_path\":\"' + cloudInitLogPath + '\",' +
                    '\"timezone\":\"Local\"},{\"log_group_name\":\"', '\",\"log_stream_name\":\"' + mongoLogPrefix + '-{instance_id}\",\"file_path\":\"' + mongoLogPath + '\"' +
                    ',\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}')),
            }),
        }));
        const userData = instance.userData.render();
        const token = '${Token[TOKEN.\\d+]}';
        // Make sure we add signal on exit
        const exitTrap = '#!/bin/bash\n' +
            'function exitTrap(){\n' +
            'exitCode=$?\n' +
            '/opt/aws/bin/cfn-signal --stack Stack --resource MongoDbInstanceServerAsgASG2643AD1D --region ' + token +
            ' -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n' +
            'test \"${MONGO_SETUP_DIR} != \"\" && sudo umount \"${MONGO_SETUP_DIR}\n' +
            '}';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(exitTrap)));
        const callExitTrap = 'trap exitTrap EXIT';
        expect(userData).toMatch(new RegExp(callExitTrap));
        const settings = 'set -xefuo pipefail';
        expect(userData).toMatch(new RegExp(settings));
        const createTempDir = 'mkdir -p $\\(dirname \'/tmp/' + token + token + '\'\\)\n';
        const s3Copy = 'aws s3 cp \'s3://' + token + '/' + token + token + '\' \'/tmp/' + token + token + '\'\n';
        // CloudWatch Agent
        const setE = 'set -e\n';
        const setChmod = 'chmod \\+x \'/tmp/' + token + token + '\'\n';
        const execute = '\'/tmp/' + token + token + '\' ' + token + '\n';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + setE + setChmod + execute)));
        // Make sure we mount EBS volume
        const mount = 'TMPDIR=$\\(mktemp -d\\)\n' +
            'pushd \"$TMPDIR\"\n' +
            'unzip /tmp/' + token + token + '\n' +
            'bash ./mountEbsBlockVolume.sh ' + token + ' xfs /var/lib/mongo rw \"\"\n' +
            'popd\n' +
            'rm -f /tmp/' + token + token;
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + mount)));
        // install mongodb
        const bashCmd = 'bash /tmp/' + token + token;
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + bashCmd)));
        // configureMongoDb
        const monogdbConfig = 'which mongod && test -f /etc/mongod.conf\n' +
            'sudo service mongod stop\n' +
            'MONGO_SETUP_DIR=$\\(mktemp -d\\)\n' +
            'mkdir -p \"${MONGO_SETUP_DIR}\"\n' +
            'sudo mount -t tmpfs -o size=50M tmpfs \"${MONGO_SETUP_DIR}\"\n' +
            'pushd \"${MONGO_SETUP_DIR}\"\n' +
            'unzip /tmp/' + token + token + '\n' +
            'cp /etc/mongod.conf .';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + monogdbConfig)));
        // Getting the server certificate
        const serverCertCmd = 'bash serverCertFromSecrets.sh \\"' + token + '\\" \\"' + token + '\\" \\"' + token + '\\" \\"' + token + '\\"';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(serverCertCmd)));
        // set mongodb certificates and credentials
        const monogdbCredentials = 'sudo mkdir -p /etc/mongod_certs\n' +
            'sudo mv ./ca.crt ./key.pem /etc/mongod_certs\n' +
            'sudo chown root.mongod -R /etc/mongod_certs/\n' +
            'sudo chmod 640 -R /etc/mongod_certs/\n' +
            'sudo chmod 750 /etc/mongod_certs/\n' +
            'sudo chown mongod.mongod -R /var/lib/mongo\n' +
            'bash ./setMongoLimits.sh\n' +
            'bash ./setStoragePath.sh \"/var/lib/mongo\"\n' +
            'bash ./setMongoNoAuth.sh\n' +
            'sudo service mongod start\n' +
            'bash ./setAdminCredentials.sh \"' + token + '\"';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(monogdbCredentials)));
        // Setup for live deployment, and start mongod
        const startMongo = 'sudo service mongod stop\n' +
            'bash ./setLiveConfiguration.sh\n' +
            'sudo systemctl enable mongod\n' +
            'sudo service mongod start\n' +
            'popd';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(startMongo)));
        // Make sure all the required public members are set
        expect(instance.version).toBe(version);
        expect(instance.connections).toBeDefined();
        expect(instance.connections).toBe(instance.server.connections);
        expect(instance.grantPrincipal).toBeDefined();
        expect(instance.grantPrincipal).toBe(instance.server.grantPrincipal);
        expect(instance.port).toBeDefined();
        expect(instance.role).toBeDefined();
        expect(instance.role).toBe(instance.server.role);
        expect(instance.userData).toBeDefined();
        expect(instance.userData).toBe(instance.server.userData);
        expect(instance.fullHostname).toBeDefined();
    });
    test('throw exception when no available subnets', () => {
        // GIVEN
        const invalidSubnets = {
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
            availabilityZones: ['dummy zone'],
        };
        // THEN
        expect(() => {
            new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
                mongoDb: {
                    version,
                    dnsZone,
                    hostname,
                    serverCertificate: serverCert,
                    userSsplAcceptance,
                },
                vpc,
                vpcSubnets: invalidSubnets,
            });
        }).toThrowError(/Did not find any subnets matching/);
    });
    test('changing instance type works correctly', () => {
        // GIVEN
        const expectedInstanceType = 'm4.micro';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            instanceType: new aws_ec2_1.InstanceType(expectedInstanceType),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: expectedInstanceType,
        }));
    });
    test('allowing ssh connection with key name', () => {
        // GIVEN
        const expectedKeyName = 'someKeyName';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            keyName: expectedKeyName,
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            KeyName: expectedKeyName,
        }));
    });
    test('using custom admin user works correctly', () => {
        // GIVEN
        const expectedAdminUser = new aws_secretsmanager_1.Secret(stack, 'AdminUser', {
            description: 'Custom admin credentials for the MongoDB database',
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
                generateStringKey: 'test_password',
                secretStringTemplate: JSON.stringify({ username: 'admin' }),
            },
        });
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                adminUser: expectedAdminUser,
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        expect(instance.adminUser).toBe(expectedAdminUser);
    });
    test('setting security group works correctly', () => {
        // GIVEN
        const actualSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'SecurityGroup', {
            securityGroupName: 'CustomSecurityGroup',
            vpc,
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            securityGroup: actualSecurityGroup,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::SecurityGroup', 1));
    });
    test('setting role works correctly', () => {
        // GIVEN
        const expectedRole = new aws_iam_1.Role(stack, 'Role', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            roleName: 'CustomRole',
        });
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            role: expectedRole,
        });
        // THEN
        expect(instance.server.role).toBe(expectedRole);
        expect(instance.role).toBe(expectedRole);
    });
    test('setting custom data volume works correctly', () => {
        // GIVEN
        const actualVolume = new aws_ec2_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: core_1.Size.gibibytes(50),
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volume: actualVolume,
                },
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Volume', 1));
    });
    test('setting custom encryption key for data volume works correctly', () => {
        // GIVEN
        // KmsKeyId is Key961B73FD
        const actualEncryptionKey = new aws_kms_1.Key(stack, 'Key', {
            description: 'Key for testing',
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volumeProps: {
                        encryptionKey: actualEncryptionKey,
                    },
                },
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Encrypted: true,
            KmsKeyId: assert_1.objectLike({
                'Fn::GetAtt': assert_1.arrayWith('Key961B73FD'),
            }),
        }));
    });
    test('setting custom size for data volume works correctly', () => {
        // GIVEN
        const volumeSize = 123;
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volumeProps: {
                        size: core_1.Size.gibibytes(volumeSize),
                    },
                },
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Size: volumeSize,
        }));
    });
    test('setting LogGroup bucket name enables export to S3', () => {
        // GIVEN
        const bucketName = 'test-bucket';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                bucketName,
            },
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Targets: assert_1.arrayWith(assert_1.objectLike({
                Input: '{\"BucketName\":\"' + bucketName + '\",\"ExportFrequencyInHours\":1,\"LogGroupName\":\"/renderfarm/MongoDbInstance\",\"RetentionInHours\":72}',
            })),
        }));
    });
    test('is created with correct LogGroup prefix', () => {
        // GIVEN
        const logGroupPrefix = 'test-prefix/';
        const uniqueId = 'MongoDbInstance';
        // WHEN
        new lib_1.MongoDbInstance(stack, uniqueId, {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                logGroupPrefix,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            LogGroupName: logGroupPrefix + uniqueId,
        }));
    });
    test('not using default LogGroup prefix if prefix is empty', () => {
        // GIVEN
        const logGroupPrefix = '';
        const uniqueId = 'MongoDbInstance';
        const expectedLogGroupName = logGroupPrefix + uniqueId;
        const defaultLogGroupName = '/renderfarm/' + uniqueId;
        // WHEN
        new lib_1.MongoDbInstance(stack, uniqueId, {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                logGroupPrefix,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            LogGroupName: expectedLogGroupName,
        }));
        assert_1.expect(stack).notTo(assert_1.haveResource('Custom::LogRetention', {
            LogGroupName: defaultLogGroupName,
        }));
    });
    test('is created with correct LogGroup retention', () => {
        // GIVEN
        const retention = aws_logs_1.RetentionDays.ONE_DAY;
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                retention,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            RetentionInDays: retention,
        }));
    });
    tag_helpers_1.testConstructTags({
        constructName: 'MongoDbInstance',
        createConstruct: () => {
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            new lib_1.MongoDbInstance(isolatedStack, 'MongoDbInstance', {
                mongoDb: {
                    version,
                    dnsZone,
                    hostname,
                    serverCertificate: serverCert,
                    userSsplAcceptance,
                },
                vpc,
            });
            return isolatedStack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::EC2::NetworkInterface': 1,
            'AWS::SecretsManager::Secret': 1,
            'AWS::EC2::Volume': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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