"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.X509CertificatePkcs12 = exports.X509CertificatePem = void 0;
const crypto = require("crypto");
const path_1 = require("path");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const lambdaLayerVersionArns_1 = require("../lambdas/lambdaLayerVersionArns");
class X509CertificateBase extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.database = new aws_dynamodb_1.Table(this, 'Table', {
            partitionKey: { name: 'PhysicalId', type: aws_dynamodb_1.AttributeType.STRING },
            sortKey: { name: 'CustomResource', type: aws_dynamodb_1.AttributeType.STRING },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            serverSideEncryption: true,
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
        });
        this.passphrase = new aws_secretsmanager_1.Secret(this, 'Passphrase', {
            description: `Passphrase for the private key of the X509Certificate ${this.node.uniqueId}`,
            encryptionKey: props.encryptionKey,
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
            },
        });
        const region = core_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        /*
         * We cannot make this a singleton function; doing so would create circular references in the lambda role (to sign
         * a cert we need a cert that this lambda generated).
         */
        this.lambdaFunc = new aws_lambda_1.Function(this, 'Generator', {
            description: `Used by a X509Certificate ${this.node.uniqueId} to generate certificates.`,
            code: props.lambdaCode,
            environment: {
                DATABASE: this.database.tableName,
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            layers: [openSslLayer],
            handler: props.lambdaHandler,
            timeout: core_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        this.database.grantReadWriteData(this.lambdaFunc);
        this.database.grant(this.lambdaFunc, 'dynamodb:DescribeTable');
        (_a = props.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncrypt(this.lambdaFunc);
        this.passphrase.grantRead(this.lambdaFunc);
        const uniqueValue = crypto.createHash('md5').update(this.node.uniqueId).digest('hex');
        this.uniqueTag = new core_1.Tag(`X509SecretGrant-${uniqueValue.slice(0, 8).toUpperCase()}`, uniqueValue);
        const tagCondition = {};
        tagCondition[`secretsmanager:ResourceTag/${this.uniqueTag.key}`] = this.uniqueTag.value;
        this.lambdaFunc.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'secretsmanager:CreateSecret',
                'secretsmanager:DeleteSecret',
                'secretsmanager:TagResource',
                'secretsmanager:PutSecretValue',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
    }
}
/**
 * A Construct that uses a Lambda to generate an X.509 certificate and then saves the certificate's components into
 * Secrets. On an update, if any properties of the construct are changed, then a new certificate will be generated.
 * When the Stack is destroyed or the Construct is removed, the Secrets will all be deleted. An X.509 certificate is
 * comprised of the certificate, a certificate chain with the chain of signing certificates (if any), and a private key
 * that is password protected by a randomly generated passphrase.
 *
 * Cost:
 * The cost of four AWS SecretsManager Secrets in the deployed region.
 * The other resources created by this construct have negligible ongoing costs.
 *
 * Resources Deployed
 * ------------------------
 * 1) DynamoDB Table - Used for tracking resources created by the Custom Resource.
 * 2) Secrets - 4 in total, for the certificate, it's private key, the passphrase to the key, and the cert chain.
 * 3) Lambda Function, with role - Used to create/update/delete the Custom Resource
 *
 * @ResourcesDeployed
 */
class X509CertificatePem extends X509CertificateBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        super(scope, id, {
            lambdaCode: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', 'lambdas', 'nodejs')),
            lambdaHandler: 'x509-certificate.generate',
            encryptionKey: props.encryptionKey,
        });
        (_a = props.signingCertificate) === null || _a === void 0 ? void 0 : _a.cert.grantRead(this.lambdaFunc);
        (_b = props.signingCertificate) === null || _b === void 0 ? void 0 : _b.key.grantRead(this.lambdaFunc);
        (_c = props.signingCertificate) === null || _c === void 0 ? void 0 : _c.passphrase.grantRead(this.lambdaFunc);
        (_e = (_d = props.signingCertificate) === null || _d === void 0 ? void 0 : _d.certChain) === null || _e === void 0 ? void 0 : _e.grantRead(this.lambdaFunc);
        const signingCertificate = props.signingCertificate
            ? {
                Cert: props.signingCertificate.cert.secretArn,
                Key: props.signingCertificate.key.secretArn,
                Passphrase: props.signingCertificate.passphrase.secretArn,
                CertChain: props.signingCertificate.certChain ? props.signingCertificate.certChain.secretArn : '',
            }
            : undefined;
        const properties = {
            DistinguishedName: {
                CN: props.subject.cn,
                O: (_f = props.subject.o) !== null && _f !== void 0 ? _f : 'AWS',
                OU: (_g = props.subject.ou) !== null && _g !== void 0 ? _g : 'Thinkbox',
            },
            Passphrase: this.passphrase.secretArn,
            Secret: {
                NamePrefix: this.node.path,
                Description: this.node.path,
                EncryptionKey: (_h = props.encryptionKey) === null || _h === void 0 ? void 0 : _h.keyArn,
                Tags: [
                    {
                        Key: this.uniqueTag.key,
                        Value: this.uniqueTag.value,
                    },
                ],
            },
            SigningCertificate: signingCertificate,
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: this.lambdaFunc.functionArn,
            properties,
            resourceType: 'Custom::RFDK_X509Generator',
        });
        if (this.lambdaFunc.role) {
            // There's a race on update where this resource could execute before the role has updated.
            resource.node.addDependency(this.lambdaFunc.role);
        }
        this.cert = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Cert', {
            secretArn: core_1.Token.asString(resource.getAtt('Cert')),
            encryptionKey: props.encryptionKey,
        });
        this.key = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Key', {
            secretArn: core_1.Token.asString(resource.getAtt('Key')),
            encryptionKey: props.encryptionKey,
        });
        // We'll only have a chain if we used a ca to sign this cert. We cannot check for certChainResource being an empty
        // string because it is an unresolved token at this point.
        if (signingCertificate) {
            const certChainResource = resource.getAtt('CertChain');
            this.certChain = certChainResource
                ? aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'CertChain', {
                    secretArn: core_1.Token.asString(certChainResource),
                    encryptionKey: props.encryptionKey,
                })
                : undefined;
        }
    }
    /**
     * Grant read permissions for the certificate
     */
    grantCertRead(grantee) {
        var _a;
        const result = this.cert.grantRead(grantee);
        (_a = this.certChain) === null || _a === void 0 ? void 0 : _a.grantRead(grantee);
        return result;
    }
    /**
     * Grant read permissions for the certificate, key, and passphrase
     */
    grantFullRead(grantee) {
        var _a;
        const result = this.cert.grantRead(grantee);
        (_a = this.certChain) === null || _a === void 0 ? void 0 : _a.grantRead(grantee);
        this.key.grantRead(grantee);
        this.passphrase.grantRead(grantee);
        return result;
    }
}
exports.X509CertificatePem = X509CertificatePem;
/**
 * This Construct will generate a PKCS #12 file from an X.509 certificate in PEM format. The PEM certificate must be
 * provided through an instance of the X509CertificatePem Construct. A Lambda Function is used to do the conversion and
 * the result is stored in a Secret. The PKCS #12 file is password protected with a passphrase that is randomly
 * generated and stored in a Secret.
 *
 * Resources Deployed
 * ------------------------
 * 1) DynamoDB Table - Used for tracking resources created by the CustomResource.
 * 2) Secrets - 2 in total, The binary of the PKCS #12 certificate and its passphrase.
 * 3) Lambda Function, with role - Used to create/update/delete the CustomResource.
 *
 * @ResourcesDeployed
 */
class X509CertificatePkcs12 extends X509CertificateBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            lambdaCode: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', 'lambdas', 'nodejs')),
            lambdaHandler: 'x509-certificate.convert',
            encryptionKey: props.encryptionKey,
        });
        props.sourceCertificate.grantFullRead(this.lambdaFunc);
        const properties = {
            Passphrase: this.passphrase.secretArn,
            Secret: {
                NamePrefix: this.node.path,
                Description: this.node.path,
                EncryptionKey: (_a = props.encryptionKey) === null || _a === void 0 ? void 0 : _a.keyArn,
                Tags: [
                    {
                        Key: this.uniqueTag.key,
                        Value: this.uniqueTag.value,
                    },
                ],
            },
            Certificate: {
                Cert: props.sourceCertificate.cert.secretArn,
                CertChain: props.sourceCertificate.certChain ? props.sourceCertificate.certChain.secretArn : '',
                Key: props.sourceCertificate.key.secretArn,
                Passphrase: props.sourceCertificate.passphrase.secretArn,
            },
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: this.lambdaFunc.functionArn,
            properties,
            resourceType: 'Custom::RFDK_X509_PKCS12',
        });
        this.cert = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Cert', {
            secretArn: core_1.Token.asString(resource.getAtt('Cert')),
            encryptionKey: props.encryptionKey,
        });
    }
}
exports.X509CertificatePkcs12 = X509CertificatePkcs12;
//# sourceMappingURL=data:application/json;base64,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