/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { Table } from '@aws-cdk/aws-dynamodb';
import { Grant, IGrantable } from '@aws-cdk/aws-iam';
import { IKey } from '@aws-cdk/aws-kms';
import { Code, Function as LambdaFunction } from '@aws-cdk/aws-lambda';
import { ISecret } from '@aws-cdk/aws-secretsmanager';
import { Construct, IConstruct, Tag } from '@aws-cdk/core';
/**
 * The identification for a self-signed CA or Certificate.
 * These fields are industry standard, and can be found in rfc1779 (see: https://tools.ietf.org/html/rfc1779)
 * or the X.520 specification (see: ITU-T Rec.X.520)
 */
export interface DistinguishedName {
    /**
     * Common Name for the identity.
     *  a) For servers -- The fully qualified domain name (aka: fqdn) of the server.
     *  b) For clients, or as a self-signed CA -- Any name you would like to identify the certificate.
     */
    readonly cn: string;
    /**
     * Organization that is creating the identity. For example, your company name.
     * @default: AWS
     */
    readonly o?: string;
    /**
     * Organization Unit that is creating the identity. For example, the name of your group/unit.
     * @default: Thinkbox
     */
    readonly ou?: string;
}
/**
 * Properties for generating an X.509 certificate.
 */
export interface X509CertificatePemProps {
    /**
     * The subject, or identity, for the generated certificate.
     */
    readonly subject: DistinguishedName;
    /**
     * If provided, then this KMS is used to secure the cert, key, and passphrase Secrets created by the construct.
     * [disable-awslint:ref-via-interface]
     * @default: Uses the account's default CMK (the one named aws/secretsmanager). If a AWS KMS CMK with that name
     * doesn't yet exist, then Secrets Manager creates it for you automatically the first time it needs to encrypt a
     * version's SecretString or SecretBinary fields.
     */
    readonly encryptionKey?: IKey;
    /**
     * If provided, then use this certificate to sign the generated certificate forming a chain of trust.
     * @default: None. The generated certificate will be self-signed
     */
    readonly signingCertificate?: X509CertificatePem;
}
/**
 * Interface for fields found on an X509Certificate construct.
 */
export interface IX509CertificatePem extends IConstruct {
    /**
     * The public certificate chain for this X.509 Certificate encoded in
     * {@link https://en.wikipedia.org/wiki/Privacy-Enhanced_Mail|PEM format}. The text of the chain is stored in the
     * 'SecretString' of the given secret. To extract the public certificate simply copy the contents of the
     * SecretString to a file.
     */
    readonly cert: ISecret;
    /**
     * The private key for this X509Certificate encoded in
     * {@link https://en.wikipedia.org/wiki/Privacy-Enhanced_Mail|PEM format}. The text of the key is stored in the
     * 'SecretString' of the given secret. To extract the public certificate simply copy the contents of the
     * SecretString to a file.
     *
     * Note that the private key is encrypted. The passphrase is stored in the the passphrase Secret.
     *
     * If you need to decrypt the private key into an unencrypted form, then you can:
     * 0. Caution. Decrypting a private key adds a security risk by making it easier to obtain your private key.
     * 1. Copy the contents of the Secret to a file called 'encrypted.key'
     * 2. Run: openssl rsa -in encrypted.key -out decrypted.key
     * 3. Enter the passphrase at the prompt
     */
    readonly key: ISecret;
    /**
     * The encryption passphrase for the private key is stored in the 'SecretString' of this Secret.
     */
    readonly passphrase: ISecret;
    /**
     * A Secret that contains the chain of Certificates used to sign this Certificate
     * @default: No certificate chain is used, signifying a self-signed Certificate
     */
    readonly certChain?: ISecret;
}
interface X509CertificateBaseProps {
    readonly lambdaCode: Code;
    readonly lambdaHandler: string;
    readonly encryptionKey?: IKey;
}
declare abstract class X509CertificateBase extends Construct {
    /**
     * The encryption passphrase for the private key is in the 'SecretString' of this secret.
     */
    readonly passphrase: ISecret;
    protected database: Table;
    protected lambdaFunc: LambdaFunction;
    protected uniqueTag: Tag;
    constructor(scope: Construct, id: string, props: X509CertificateBaseProps);
}
/**
 * A Construct that uses a Lambda to generate an X.509 certificate and then saves the certificate's components into
 * Secrets. On an update, if any properties of the construct are changed, then a new certificate will be generated.
 * When the Stack is destroyed or the Construct is removed, the Secrets will all be deleted. An X.509 certificate is
 * comprised of the certificate, a certificate chain with the chain of signing certificates (if any), and a private key
 * that is password protected by a randomly generated passphrase.
 *
 * Cost:
 * The cost of four AWS SecretsManager Secrets in the deployed region.
 * The other resources created by this construct have negligible ongoing costs.
 *
 * Resources Deployed
 * ------------------------
 * 1) DynamoDB Table - Used for tracking resources created by the Custom Resource.
 * 2) Secrets - 4 in total, for the certificate, it's private key, the passphrase to the key, and the cert chain.
 * 3) Lambda Function, with role - Used to create/update/delete the Custom Resource
 *
 * @ResourcesDeployed
 */
export declare class X509CertificatePem extends X509CertificateBase implements IX509CertificatePem {
    readonly cert: ISecret;
    readonly key: ISecret;
    readonly certChain?: ISecret;
    constructor(scope: Construct, id: string, props: X509CertificatePemProps);
    /**
     * Grant read permissions for the certificate
     */
    grantCertRead(grantee: IGrantable): Grant;
    /**
     * Grant read permissions for the certificate, key, and passphrase
     */
    grantFullRead(grantee: IGrantable): Grant;
}
/**
 * Construct properties for generating a PKCS #12 file from an X.509 certificate.
 */
export interface X509CertificatePkcs12Props {
    /**
     * The source PEM certificiate for the PKCS #12 file.
     */
    readonly sourceCertificate: X509CertificatePem;
    /**
     * If provided, then this KMS is used to secure the cert, key, and passphrase Secrets created by the construct.
     * [disable-awslint:ref-via-interface]
     * @default: None
     */
    readonly encryptionKey?: IKey;
}
/**
 * Properties of an X.509 PKCS #12 file.
 */
export interface IX509CertificatePkcs12 extends IConstruct {
    /**
     * The PKCS #12 data is stored in the 'SecretBinary' of this Secret.
     */
    readonly cert: ISecret;
    /**
     * The encryption passphrase for the cert is stored in the 'SecretString' of this Secret.
     */
    readonly passphrase: ISecret;
}
/**
 * This Construct will generate a PKCS #12 file from an X.509 certificate in PEM format. The PEM certificate must be
 * provided through an instance of the X509CertificatePem Construct. A Lambda Function is used to do the conversion and
 * the result is stored in a Secret. The PKCS #12 file is password protected with a passphrase that is randomly
 * generated and stored in a Secret.
 *
 * Resources Deployed
 * ------------------------
 * 1) DynamoDB Table - Used for tracking resources created by the CustomResource.
 * 2) Secrets - 2 in total, The binary of the PKCS #12 certificate and its passphrase.
 * 3) Lambda Function, with role - Used to create/update/delete the CustomResource.
 *
 * @ResourcesDeployed
 */
export declare class X509CertificatePkcs12 extends X509CertificateBase implements IX509CertificatePkcs12 {
    readonly cert: ISecret;
    constructor(scope: Construct, id: string, props: X509CertificatePkcs12Props);
}
export {};
