/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { AutoScalingGroup, BlockDevice } from '@aws-cdk/aws-autoscaling';
import { CfnNetworkInterface, Connections, IConnectable, IMachineImage, InstanceType, ISecurityGroup, IVpc, OperatingSystemType, SubnetSelection, UserData } from '@aws-cdk/aws-ec2';
import { IGrantable, IPrincipal, IRole } from '@aws-cdk/aws-iam';
import { Function as LambdaFunction } from '@aws-cdk/aws-lambda';
import { Construct, Duration } from '@aws-cdk/core';
/**
 * Required and optional properties that define the construction of a {@link StaticPrivateIpServer}
 */
export interface StaticPrivateIpServerProps {
    /**
     * VPC in which to launch the instance.
     */
    readonly vpc: IVpc;
    /**
     * The type of instance to launch
     */
    readonly instanceType: InstanceType;
    /**
     * The AMI to launch the instance with.
     */
    readonly machineImage: IMachineImage;
    /**
     * Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes.
     *
     * Each instance that is launched has an associated root device volume, either an Amazon EBS volume or an instance store volume.
     * You can use block device mappings to specify additional EBS volumes or instance store volumes to attach to an instance when it is launched.
     *
     * @default Uses the block device mapping of the AMI.
     */
    readonly blockDevices?: BlockDevice[];
    /**
     * Name of the EC2 SSH keypair to grant access to the instance.
     *
     * @default No SSH access will be possible.
     */
    readonly keyName?: string;
    /**
     * The specific private IP address to assign to the Elastic Network Interface of this instance.
     *
     * @default An IP address is randomly assigned from the subnet.
     */
    readonly privateIpAddress?: string;
    /**
     * The length of time to wait for the instance to signal successful deployment
     * during the initial deployment, or update, of your stack.
     *
     * The maximum value is 12 hours.
     *
     * @default The deployment does not require a success signal from the instance.
     */
    readonly resourceSignalTimeout?: Duration;
    /**
     * An IAM role to associate with the instance profile that is assigned to this instance.
     * The role must be assumable by the service principal `ec2.amazonaws.com`
     *
     * @default A role will automatically be created, it can be accessed via the `role` property.
     */
    readonly role?: IRole;
    /**
     * The security group to assign to this instance.
     *
     * @default A new security group is created for this instance.
     */
    readonly securityGroup?: ISecurityGroup;
    /**
     * Specific UserData to use. UserData is a script that is run automatically by the instance the very first time that a new instance is started.
     *
     * The UserData may be mutated after creation.
     *
     * @default A UserData that is appropriate to the {@link machineImage}'s operating system is created.
     */
    readonly userData?: UserData;
    /**
     * Where to place the instance within the VPC.
     *
     * @default The instance is placed within a Private subnet.
     */
    readonly vpcSubnets?: SubnetSelection;
}
/**
 * This construct provides a single instance, provided by an Auto Scaling Group (ASG), that
 * has an attached Elastic Network Interface (ENI) that is providing a private ip address.
 * This ENI is automatically re-attached to the instance if the instance is replaced
 * by the ASG.
 *
 * The ENI provides an unchanging private IP address that can always be used to connect
 * to the instance regardless of how many times the instance has been replaced. Furthermore,
 * the ENI has a MAC address that remains unchanged unless the ENI is destroyed.
 *
 * Essentially, this provides an instance with an unchanging private IP address that will
 * automatically recover from termination. This instance is suitable for use as an application server,
 * such as a license server, that must always be reachable by the same IP address.
 *
 * Resources Deployed
 * ------------------------
 * 1) Auto Scaling Group (ASG) with min & max capacity of 1 instance;
 * 2) Elastic Network Interface (ENI);
 * 3) Security Group for the ASG;
 * 4) Instance Role and corresponding IAM Policy
 * 5) SNS Topic & Role for instance-launch lifecycle events -- max one of each per stack; and
 * 6) Lambda function, with role, to attach the ENI in response to instance-launch lifecycle events -- max one per stack.
 *
 * @ResourcesDeployed
 */
export declare class StaticPrivateIpServer extends Construct implements IConnectable, IGrantable {
    /**
     * The Auto Scaling Group that contains the instance this construct creates.
     */
    readonly autoscalingGroup: AutoScalingGroup;
    /**
     * Allows for providing security group connections to/from this instance.
     */
    readonly connections: Connections;
    /**
     * The principal to grant permission to. Granting permissions to this principal will grant
     * those permissions to the instance role.
     */
    readonly grantPrincipal: IPrincipal;
    /**
     * The type of operating system that the instance is running.
     */
    readonly osType: OperatingSystemType;
    /**
     * The Private IP address that has been assigned to the ENI.
     */
    readonly privateIpAddress: string;
    /**
     * The IAM role that is assumed by the instance.
     */
    readonly role: IRole;
    /**
     * The UserData for this instance.
     * UserData is a script that is run automatically by the instance the very first time that a new instance is started.
     */
    readonly userData: UserData;
    constructor(scope: Construct, id: string, props: StaticPrivateIpServerProps);
    /**
     * Set up an instance launch lifecycle action that will attach the eni to the single instance
     * in this construct's AutoScalingGroup when a new instance is launched.
     */
    protected attachEniLifecyleTarget(eni: CfnNetworkInterface): void;
    /**
     * Create, or fetch, the lambda function that will process instance-start lifecycle events from this construct.
     */
    protected setupLifecycleEventHandlerFunction(): LambdaFunction;
    /**
     * Create, or fetch, an SNS Topic to which we'll direct the ASG's instance-start lifecycle hook events. Also creates, or fetches,
     * the accompanying role that allows the lifecycle events to be published to the SNS Topic.
     * @param lambdaHandler The lambda singleton that will be processing the lifecycle events.
     * @returns { topic: Topic, role: Role }
     */
    protected setupLifecycleNotificationTopic(lambdaHandler: LambdaFunction): {
        [key: string]: any;
    };
    /**
     * Convert a UUID into a string that's usable in a construct id.
     */
    private removeHyphens;
}
