"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScriptAsset = void 0;
const path = require("path");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
var ScriptExtension;
(function (ScriptExtension) {
    ScriptExtension[ScriptExtension[".sh"] = 0] = ".sh";
    ScriptExtension[ScriptExtension[".ps1"] = 1] = ".ps1";
})(ScriptExtension || (ScriptExtension = {}));
var ScriptPathPrefix;
(function (ScriptPathPrefix) {
    ScriptPathPrefix[ScriptPathPrefix["bash"] = 0] = "bash";
    ScriptPathPrefix[ScriptPathPrefix["powershell"] = 1] = "powershell";
})(ScriptPathPrefix || (ScriptPathPrefix = {}));
/**
 * This method returns the path to a script based on RFDK conventional directory structure and the target
 * operating system of the script.
 *
 * The directory structure convention keeps linux scripts in `${scriptRoot}//bash/*.sh` and Windows scripts in
 * `${scriptRoot}/powershell/*.ps1`.
 *
 * @param osType
 * @param scriptName
 */
function getConventionalScriptPath(params) {
    const { rootDir: scriptDir, baseName: scriptName, osType } = params;
    return path.join(scriptDir, ScriptPathPrefix[osType], `${scriptName}${ScriptExtension[osType]}`);
}
/**
 * An S3 asset that contains a shell script intended to be executed through instance user data.
 *
 * This is used by other constructs to generalize the concept of a script
 * (bash or powershell) that executes on an instance.
 * It provides a wrapper around the CDK’s S3 Asset construct
 * ( https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-s3-assets.Asset.html )
 *
 * The script asset is placed into and fetched from the CDK bootstrap S3 bucket.
 *
 * Resources Deployed
 * ------------------------
 * 1) An Asset which is uploaded to the bootstrap S3 bucket.
 *
 * @ResourcesDeployed
 */
class ScriptAsset extends aws_s3_assets_1.Asset {
    /**
     * Returns a {@link ScriptAsset} instance by computing the path to the script using RFDK's script directory structure
     * convention.
     *
     * By convention, scripts are kept in a `scripts` directory in each `aws-rfdk/*` package. The scripts are organized
     * based on target shell (and implicitly target operating system). The directory structure looks like:
     *
     * ```
     * scripts/
     *   bash/
     *     script-one.sh
     *     script-two.sh
     *   powershell
     *     script-one.ps1
     *     script-one.ps1
     * ```
     *
     * @param scope The scope for the created {@link ScriptAsset}
     * @param id The construct id for the created {@link ScriptAsset}
     * @param scriptParams The parameters that are used to compute the conventional path to the script file
     */
    static fromPathConvention(scope, id, scriptParams) {
        const scriptPath = getConventionalScriptPath(scriptParams);
        return new ScriptAsset(scope, id, { path: scriptPath });
    }
    constructor(scope, id, props) {
        super(scope, id, props);
    }
    /**
     * Adds commands to the {@link IScriptHost} to download and execute the ScriptAsset.
     *
     * @param props The parameters for executing the script
     */
    executeOn(props) {
        const { host, args } = props;
        // Grant permission to fetch the script asset
        this.grantRead(host);
        // Add a command to the user data to download the script asset
        const instanceScriptPath = host.userData.addS3DownloadCommand({
            bucket: this.bucket,
            bucketKey: this.s3ObjectKey,
        });
        // Add a command to the user data to execute the downloaded script
        host.userData.addExecuteFileCommand({
            filePath: instanceScriptPath,
            arguments: (args || []).join(' '),
        });
    }
}
exports.ScriptAsset = ScriptAsset;
//# sourceMappingURL=data:application/json;base64,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