"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountableBlockVolume = exports.BlockVolumeFormat = void 0;
const crypto = require("crypto");
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const mount_permissions_helper_1 = require("./mount-permissions-helper");
/**
 * Block format options for formatting a blank/new BlockVolume
 */
var BlockVolumeFormat;
(function (BlockVolumeFormat) {
    /**
     * See: https://en.wikipedia.org/wiki/Ext3
     */
    BlockVolumeFormat["EXT3"] = "ext3";
    /**
     * See: https://en.wikipedia.org/wiki/Ext4
     */
    BlockVolumeFormat["EXT4"] = "ext4";
    /**
     * See: https://en.wikipedia.org/wiki/XFS
     */
    BlockVolumeFormat["XFS"] = "xfs";
})(BlockVolumeFormat = exports.BlockVolumeFormat || (exports.BlockVolumeFormat = {}));
/**
 * This class encapsulates scripting that can be used by an instance to mount, format, and resize an
 * Amazon Elastic Block Store (EBS) Volume to itself when it is launched. The scripting is added to
 * the instance's UserData to be run when the instance is first launched.
 *
 * The script that is employed by this class will:
 * 1) Attach the volume to this instance if it is not already attached;
 * 2) Format the block volume to the filesystem format that's passed as an argument to this script but,
 *   **ONLY IF** the filesystem has no current format;
 * 3) Mount the volume to the given mount point with the given mount options; and
 * 4) Resize the filesystem on the volume if the volume is larger than the formatted filesystem size.
 *
 * Note: This does **NOT** support multiple partitions on the EBS Volume; the script will exit with a failure code
 * when it detects multiple partitions on the device. It is expected that the whole block device is a single partition.
 *
 * @remark If using this script with an instance within an AWS Auto Scaling Group (ASG) and you resize
 * the EBS volume, then you can terminate the instance to let the ASG replace the instance and benefit
 * from the larger volume size when this script resizes the filesystem on instance launch.
 */
class MountableBlockVolume {
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
    }
    /**
     * @inheritdoc
     */
    mountToLinuxInstance(target, mount) {
        var _a;
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        this.grantRequiredPermissions(target);
        const mountScriptAsset = this.mountAssetSingleton();
        mountScriptAsset.grantRead(target.grantPrincipal);
        const mountScriptZip = target.userData.addS3DownloadCommand({
            bucket: mountScriptAsset.bucket,
            bucketKey: mountScriptAsset.s3ObjectKey,
        });
        const mountDir = path.posix.normalize(mount.location);
        const mountOptions = [mount_permissions_helper_1.MountPermissionsHelper.toLinuxMountOption(mount.permissions)];
        if (this.props.extraMountOptions) {
            mountOptions.push(...this.props.extraMountOptions);
        }
        const mountOptionsStr = mountOptions.join(',');
        const volumeFormat = (_a = this.props.volumeFormat) !== null && _a !== void 0 ? _a : BlockVolumeFormat.XFS;
        target.userData.addCommands('TMPDIR=$(mktemp -d)', 'pushd "$TMPDIR"', `unzip ${mountScriptZip}`, `bash ./mountEbsBlockVolume.sh ${this.props.blockVolume.volumeId} ${volumeFormat} ${mountDir} ${mountOptionsStr} ""`, 'popd', `rm -f ${mountScriptZip}`);
    }
    /**
     * Grant required permissions to the target. The mounting script requires two permissions:
     * 1) Permission to describe the volume
     * 2) Permission to attach the volume
     */
    grantRequiredPermissions(target) {
        // Volume.grantAttachVolumeByResourceTag() requires that the target be a construct; it adds a tag to the construct.
        // So, we fail if we're given something that is not compatible.
        if (!core_1.Construct.isConstruct(target)) {
            throw new Error('Target instance must be a construct. It cannot be constructed from attributes.');
        }
        // See: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html
        // Accessed July 2020
        // ec2:DescribeVolumes does not support resource or condition constraints.
        target.grantPrincipal.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'ec2:DescribeVolumes',
            ],
            resources: ['*'],
        }));
        // Work-around a bug in v1.49.1 of CDK.
        // To be able to mount the *same* volume to multiple instances we must provide a tag suffix to the permission grant
        // that is unique to this particular combination of volume + mount target.
        // We can remove this, if desired, once it is fixed in upstream CDK.
        function hashUniqueIds(resources) {
            const md5 = crypto.createHash('md5');
            resources.forEach(res => md5.update(res.node.uniqueId));
            return md5.digest('hex');
        }
        this.props.blockVolume.grantAttachVolumeByResourceTag(target.grantPrincipal, [target], hashUniqueIds([target, this.props.blockVolume]));
    }
    /**
     * Fetch the Asset singleton for the Volume mounting scripts, or generate it if needed.
     */
    mountAssetSingleton() {
        var _a;
        const stack = core_1.Stack.of(this.scope);
        const uuid = '01ca4aa6-d440-4f83-84d8-80a5a21fd0e3';
        const uniqueId = 'MountableBlockVolumeAsset' + uuid.replace(/[-]/g, '');
        return (_a = stack.node.tryFindChild(uniqueId)) !== null && _a !== void 0 ? _a : new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'bash'),
            exclude: ['**/*', '!mountEbsBlockVolume.sh', '!metadataUtilities.sh', '!ec2-certificates.crt'],
        });
    }
}
exports.MountableBlockVolume = MountableBlockVolume;
//# sourceMappingURL=data:application/json;base64,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