"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbPostInstallSetup = void 0;
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const lambdaLayerVersionArns_1 = require("../lambdas/lambdaLayerVersionArns");
/**
 * This construct performs post-installation setup on a MongoDB database by logging into the database, and
 * executing commands against it. To provide this functionality, this construct will create an AWS Lambda function
 * that is granted the ability to connect to the given MongoDB using its administrator credentials. This lambda
 * is run automatically when you deploy or update the stack containing this construct. Logs for all AWS Lambdas are
 * automatically recorded in Amazon CloudWatch.
 *
 * Presently, the only post-installation action that this construct can perform is creating users. There are two types
 * of users that it can create:
 * 1. Password-authenticated users -- these users will be created within the 'admin' database.
 * 2. X.509-authenticated users -- these users will be created within the '$external' database.
 *
 * Resources Deployed
 * ------------------------
 * - An AWS Lambda that is used to connect to the MongoDB application, and perform post-installation tasks.
 * - A CloudFormation Custom Resource that triggers execution of the Lambda on stack deployment, update, and deletion.
 * - An Amazon CloudWatch log group that records history of the AWS Lambda's execution.
 *
 * @ResourcesDeployed
 */
class MongoDbPostInstallSetup extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id);
        (_a = props.users.x509AuthUsers) === null || _a === void 0 ? void 0 : _a.forEach(user => {
            try {
                JSON.parse(user.roles);
            }
            catch (e) {
                throw new Error(`MongoDbPostInstallSetup: Could not parse JSON role for x509 user: ${user.roles}`);
            }
        });
        const region = core_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        const lamdbaFunc = new aws_lambda_1.Function(this, 'Lambda', {
            vpc: props.vpc,
            vpcSubnets: (_b = props.vpcSubnets) !== null && _b !== void 0 ? _b : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            description: `Used by a MongoDbPostInstallSetup ${this.node.uniqueId} to perform post-installation setup on a MongoDB`,
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', 'lambdas', 'nodejs'), {
            // Exclude commented out, for now, as a work-around for a CDK bug with at least CDK v1.49.1.
            // If we exclude files, then the asset hash is not calculated correctly and can result in updates to these
            // files not being picked up by the live system.
            // exclude: [
            //   '**/*',
            //   '!mongodb', '!mongodb/*',
            //   '!lib',
            //   '!lib/custom-resource', '!lib/custom-resource/*',
            //   '!lib/aws-lambda', '!lib/aws-lambda/*',
            //   '!lib/secrets-manager', '!lib/secrets-manager/*',
            //   '**/test',
            // ],
            }),
            environment: {
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'mongodb.configureMongo',
            layers: [openSslLayer],
            timeout: core_1.Duration.minutes(2),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        lamdbaFunc.connections.allowTo(props.mongoDb, aws_ec2_1.Port.tcp(props.mongoDb.port));
        props.mongoDb.certificateChain.grantRead(lamdbaFunc.grantPrincipal);
        props.mongoDb.adminUser.grantRead(lamdbaFunc.grantPrincipal);
        (_c = props.users.passwordAuthUsers) === null || _c === void 0 ? void 0 : _c.forEach(secret => secret.grantRead(lamdbaFunc));
        (_d = props.users.x509AuthUsers) === null || _d === void 0 ? void 0 : _d.forEach(user => user.certificate.grantRead(lamdbaFunc));
        const properties = {
            Connection: {
                Hostname: props.mongoDb.fullHostname,
                Port: props.mongoDb.port.toString(),
                CaCertificate: props.mongoDb.certificateChain.secretArn,
                Credentials: props.mongoDb.adminUser.secretArn,
            },
            PasswordAuthUsers: (_e = props.users.passwordAuthUsers) === null || _e === void 0 ? void 0 : _e.map(secret => secret.secretArn),
            X509AuthUsers: (_f = props.users.x509AuthUsers) === null || _f === void 0 ? void 0 : _f.map(user => ({ Certificate: user.certificate.secretArn, Roles: user.roles })),
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: lamdbaFunc.functionArn,
            resourceType: 'Custom::RFDK_MongoDbPostInstallSetup',
            properties,
        });
        // Prevents a race during a stack-update.
        resource.node.addDependency(lamdbaFunc.role);
        /* istanbul ignore next */
        if (props.mongoDb.node.defaultChild) {
            // Add a dependency on the ASG within the StaticPrivateIpServer to ensure that
            // mongo is running before we try to login to it.
            resource.node.addDependency(props.mongoDb.node.defaultChild.node.defaultChild);
        }
        this.node.defaultChild = resource;
    }
}
exports.MongoDbPostInstallSetup = MongoDbPostInstallSetup;
//# sourceMappingURL=data:application/json;base64,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