"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbInstaller = exports.MongoDbSsplLicenseAcceptance = exports.MongoDbVersion = void 0;
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
/**
 * Versions of MongoDB Community Edition that the {@link MongoDbInstaller} is
 * able to install.
 */
var MongoDbVersion;
(function (MongoDbVersion) {
    /**
     * MongoDB 3.6 Community Edition.
     * See: https://docs.mongodb.com/v3.6/introduction/
     */
    MongoDbVersion["COMMUNITY_3_6"] = "3.6";
})(MongoDbVersion = exports.MongoDbVersion || (exports.MongoDbVersion = {}));
/**
 * Choices for signifying the user's stance on the terms of the SSPL.
 * See: https://www.mongodb.com/licensing/server-side-public-license
 */
var MongoDbSsplLicenseAcceptance;
(function (MongoDbSsplLicenseAcceptance) {
    /**
     * The user signifies their explicit rejection of the tems of the SSPL.
     */
    MongoDbSsplLicenseAcceptance[MongoDbSsplLicenseAcceptance["USER_REJECTS_SSPL"] = 0] = "USER_REJECTS_SSPL";
    /**
     * The user signifies their explicit acceptance of the terms of the SSPL.
     */
    MongoDbSsplLicenseAcceptance[MongoDbSsplLicenseAcceptance["USER_ACCEPTS_SSPL"] = 1] = "USER_ACCEPTS_SSPL";
})(MongoDbSsplLicenseAcceptance = exports.MongoDbSsplLicenseAcceptance || (exports.MongoDbSsplLicenseAcceptance = {}));
/**
 * This class provides a mechanism to install a version of MongoDB Community Edition during the
 * initial launch of an instance. MongoDB is installed from the official sources using the system
 * package manger (yum). It installs the mongodb-org metapackage which will install the following packages:
 * 1) mongodb-org-mongos;
 * 2) mongodb-org-server;
 * 3) mongodb-org-shell; and
 * 4) mongodb-org-tools.
 *
 * Successful installation of MongoDB with this class requires:
 * 1) Explicit acceptance of the terms of the SSPL license, under which MongoDB is distributed; and
 * 2) The instance on which the installation is being performed is in a subnet that can access
 * the official MongoDB sites: https://repo.mongodb.org/ and https://www.mongodb.org
 *
 * Resources Deployed
 * ------------------------
 * - A CDK Asset package containing the installation scripts is deployed to your CDK staging bucket.
 *
 * @ResourcesDeployed
 */
class MongoDbInstaller {
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
        // Legal requirement: Users of this class must agree to the terms of the SSPL, without exception.
        // Developer note: It is a legal requirement that the default be USER_REJECTS_SSPL, so this check
        // must throw an error for every value except USER_ACCEPTS_SSPL.
        if (props.userSsplAcceptance !== MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL) {
            throw new Error(MongoDbInstaller.SSPL_ACCEPT_MESSAGE);
        }
    }
    /**
     * Install MongoDB to the given instance at instance startup. This is accomplished by
     * adding scripting to the instance's UserData to install MongoDB.
     *
     * Notes:
     * 1) The instance on which the installation is being performed must be in a subnet that can access
     * the official MongoDB sites: https://repo.mongodb.org/ and https://www.mongodb.org; and
     * 2) At this time, this method only supports installation onto instances that are running an operating system
     * that is compatible with x86-64 RedHat 7 -- this includes Amazon Linux 2, RedHat 7, and CentOS 7.
     *
     * @param target The target instance onto which to install MongoDB.
     */
    installOnLinuxInstance(target) {
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        const installerScriptAsset = this.installerAssetSingleton();
        installerScriptAsset.grantRead(target.grantPrincipal);
        const installerScript = target.userData.addS3DownloadCommand({
            bucket: installerScriptAsset.bucket,
            bucketKey: installerScriptAsset.s3ObjectKey,
        });
        target.userData.addCommands(`bash ${installerScript}`);
    }
    /**
     * Fetch the Asset singleton for the installation script, or generate it if needed.
     */
    installerAssetSingleton() {
        var _a;
        const stack = core_1.Stack.of(this.scope);
        const uuid = '5b141ac9-fde5-45d8-9961-b7108fb3b18a';
        const uniqueId = 'MongoDbInstallerAsset' + uuid.replace(/[-]/g, '');
        return (_a = stack.node.tryFindChild(uniqueId)) !== null && _a !== void 0 ? _a : new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'mongodb', this.props.version, 'installMongoDb.sh'),
        });
    }
}
exports.MongoDbInstaller = MongoDbInstaller;
/**
 * The SSPL licensing message that is presented to the user if they create an instance of
 * this class without explicitly accepting the SSPL.
 *
 * Note to developers: The text of this string is a legal requirement, and must not be altered
 * witout approval.
 */
MongoDbInstaller.SSPL_ACCEPT_MESSAGE = `
The MongoDbInstaller will install MongoDB Community Edition onto one or more EC2 instances.

MongoDB is provided by MongoDB Inc. under the SSPL license. By installing MongoDB, you are agreeing to the terms of this license.
Follow the link below to read the terms of the SSPL license.
https://www.mongodb.com/licensing/server-side-public-license

By using the MongoDbInstaller to install MongoDB you agree to the terms of the SSPL license.

Please set the userSsplAcceptance property to USER_ACCEPTS_SSPL to signify your acceptance of the terms of the SSPL license.
`;
//# sourceMappingURL=data:application/json;base64,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