/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { Asset } from '@aws-cdk/aws-s3-assets';
import { Construct } from '@aws-cdk/core';
import { IScriptHost } from './script-assets';
/**
 * Versions of MongoDB Community Edition that the {@link MongoDbInstaller} is
 * able to install.
 */
export declare enum MongoDbVersion {
    /**
     * MongoDB 3.6 Community Edition.
     * See: https://docs.mongodb.com/v3.6/introduction/
     */
    COMMUNITY_3_6 = "3.6"
}
/**
 * Choices for signifying the user's stance on the terms of the SSPL.
 * See: https://www.mongodb.com/licensing/server-side-public-license
 */
export declare enum MongoDbSsplLicenseAcceptance {
    /**
     * The user signifies their explicit rejection of the tems of the SSPL.
     */
    USER_REJECTS_SSPL = 0,
    /**
     * The user signifies their explicit acceptance of the terms of the SSPL.
     */
    USER_ACCEPTS_SSPL = 1
}
/**
 * Properties that are required to create a {@link MongoDbInstaller}.
 */
export interface MongoDbInstallerProps {
    /**
     * MongoDB Community edition is licensed under the terms of the SSPL (see: https://www.mongodb.com/licensing/server-side-public-license ).
     * Users of MongoDbInstaller must explicitly signify their acceptance of the terms of the SSPL through this
     * property before the {@link MongoDbInstaller} will be allowed to install MongoDB.
     *
     * @default MongoDbSsplLicenseAcceptance.USER_REJECTS_SSPL
     */
    readonly userSsplAcceptance?: MongoDbSsplLicenseAcceptance;
    /**
     * The version of MongoDB to install.
     */
    readonly version: MongoDbVersion;
}
/**
 * This class provides a mechanism to install a version of MongoDB Community Edition during the
 * initial launch of an instance. MongoDB is installed from the official sources using the system
 * package manger (yum). It installs the mongodb-org metapackage which will install the following packages:
 * 1) mongodb-org-mongos;
 * 2) mongodb-org-server;
 * 3) mongodb-org-shell; and
 * 4) mongodb-org-tools.
 *
 * Successful installation of MongoDB with this class requires:
 * 1) Explicit acceptance of the terms of the SSPL license, under which MongoDB is distributed; and
 * 2) The instance on which the installation is being performed is in a subnet that can access
 * the official MongoDB sites: https://repo.mongodb.org/ and https://www.mongodb.org
 *
 * Resources Deployed
 * ------------------------
 * - A CDK Asset package containing the installation scripts is deployed to your CDK staging bucket.
 *
 * @ResourcesDeployed
 */
export declare class MongoDbInstaller {
    protected readonly scope: Construct;
    protected readonly props: MongoDbInstallerProps;
    /**
     * The SSPL licensing message that is presented to the user if they create an instance of
     * this class without explicitly accepting the SSPL.
     *
     * Note to developers: The text of this string is a legal requirement, and must not be altered
     * witout approval.
     */
    private static readonly SSPL_ACCEPT_MESSAGE;
    constructor(scope: Construct, props: MongoDbInstallerProps);
    /**
     * Install MongoDB to the given instance at instance startup. This is accomplished by
     * adding scripting to the instance's UserData to install MongoDB.
     *
     * Notes:
     * 1) The instance on which the installation is being performed must be in a subnet that can access
     * the official MongoDB sites: https://repo.mongodb.org/ and https://www.mongodb.org; and
     * 2) At this time, this method only supports installation onto instances that are running an operating system
     * that is compatible with x86-64 RedHat 7 -- this includes Amazon Linux 2, RedHat 7, and CentOS 7.
     *
     * @param target The target instance onto which to install MongoDB.
     */
    installOnLinuxInstance(target: IScriptHost): void;
    /**
     * Fetch the Asset singleton for the installation script, or generate it if needed.
     */
    protected installerAssetSingleton(): Asset;
}
