"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.LogGroupFactory = void 0;
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const exporting_log_group_1 = require("./exporting-log-group");
/**
 * This factory will return an ILogGroup based on the configuration provided to it. The LogGroup will either be
 * wrapped in a LogRetention from the aws-lambda package that has the ability to look up and reuse an existing LogGroup
 * or an ExportingLogGroup that uses a LogRetention and adds additional functionality to export the logs to S3.
 */
class LogGroupFactory {
    /**
     * Either create a new LogGroup given the LogGroup name, or return the existing LogGroup.
     */
    static createOrFetch(scope, logWrapperId, logGroupName, props) {
        const fullLogGroupName = (props === null || props === void 0 ? void 0 : props.logGroupPrefix) ? `${props.logGroupPrefix}${logGroupName}` : logGroupName;
        const retention = (props === null || props === void 0 ? void 0 : props.retention) ? props.retention : LogGroupFactory.DEFAULT_LOG_RETENTION;
        return (props === null || props === void 0 ? void 0 : props.bucketName) ? new exporting_log_group_1.ExportingLogGroup(scope, logWrapperId, {
            bucketName: props.bucketName,
            logGroupName: fullLogGroupName,
            retention: props.retention,
        }).logGroup
            : aws_logs_1.LogGroup.fromLogGroupArn(scope, `${logGroupName}LogGroup`, new aws_lambda_1.LogRetention(scope, logWrapperId, {
                logGroupName: fullLogGroupName,
                retention,
            }).logGroupArn);
    }
}
exports.LogGroupFactory = LogGroupFactory;
/**
 * Default retention period to hold logs in CloudWatch for.
 */
LogGroupFactory.DEFAULT_LOG_RETENTION = aws_logs_1.RetentionDays.THREE_DAYS;
//# sourceMappingURL=data:application/json;base64,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