"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ImportedAcmCertificate = void 0;
const crypto = require("crypto");
const path_1 = require("path");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lambdaLayerVersionArns_1 = require("../lambdas/lambdaLayerVersionArns");
/**
 * A Construct that holds a Custom Resource modelling a certificate that was imported into ACM. It uses a Lambda
 * Function to extract the certificate from Secrets and then import it into ACM. It is intended to be used with the
 * X509CertificatePem Construct.
 *
 * Resources Deployed
 * ------------------------
 * 1) DynamoDB Table - Used for tracking resources created by the CustomResource.
 * 2) Lambda Function, with Role - Used to create/update/delete the CustomResource.
 * 3) ACM Certificate - Created by the CustomResource.
 *
 * @ResourcesDeployed
 */
class ImportedAcmCertificate extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.stack = core_1.Stack.of(this);
        this.database = new aws_dynamodb_1.Table(this, 'Table', {
            partitionKey: { name: 'PhysicalId', type: aws_dynamodb_1.AttributeType.STRING },
            sortKey: { name: 'CustomResource', type: aws_dynamodb_1.AttributeType.STRING },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            serverSideEncryption: true,
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
        });
        const region = core_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        const lambda = new aws_lambda_1.SingletonFunction(this, 'AcmImporter', {
            uuid: ImportedAcmCertificate.IMPORTER_UUID,
            code: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', 'lambdas', 'nodejs')),
            handler: 'x509-certificate.importCert',
            environment: {
                DATABASE: this.database.tableName,
                DEBUG: 'false',
            },
            layers: [openSslLayer],
            retryAttempts: 0,
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            timeout: core_1.Duration.seconds(30),
        });
        this.database.grantReadWriteData(lambda);
        this.database.grant(lambda, 'dynamodb:DescribeTable');
        props.cert.grantRead(lambda);
        props.key.grantRead(lambda);
        props.passphrase.grantRead(lambda);
        (_a = props.certChain) === null || _a === void 0 ? void 0 : _a.grantRead(lambda);
        (_b = props.encryptionKey) === null || _b === void 0 ? void 0 : _b.grantDecrypt(lambda);
        const uniqueValue = crypto.createHash('md5').update(this.node.uniqueId).digest('hex');
        this.uniqueTag = new core_1.Tag(`AcmCertImport-${uniqueValue.slice(0, 8).toUpperCase()}`, uniqueValue);
        const tagCondition = {};
        tagCondition[`aws:RequestTag/${this.uniqueTag.key}`] = this.uniqueTag.value;
        lambda.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'acm:AddTagsToCertificate',
                'acm:ImportCertificate',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // GetCertificate and DeleteCertificate don't currently support and conditions, so we have to give a broader policy
        // on them for now.
        // See https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscertificatemanager.html#awscertificatemanager-aws_TagKeys
        // for the condition keys currently available on ACM actions.
        lambda.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'acm:DeleteCertificate',
                'acm:DescribeCertificate',
                'acm:GetCertificate',
            ],
            resources: ['*'],
        }));
        const properties = {
            X509CertificatePem: {
                Cert: props.cert.secretArn,
                Key: props.key.secretArn,
                Passphrase: props.passphrase.secretArn,
                CertChain: props.certChain ? props.certChain.secretArn : '',
            },
            Tags: [
                {
                    Key: this.uniqueTag.key,
                    Value: this.uniqueTag.value,
                },
                { Key: 'Name',
                    Value: this.uniqueTag.value,
                },
            ],
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: lambda.functionArn,
            properties,
            resourceType: 'Custom::RFDK_AcmImportedCertificate',
        });
        this.certificateArn = core_1.Token.asString(resource.getAtt('CertificateArn'));
    }
}
exports.ImportedAcmCertificate = ImportedAcmCertificate;
ImportedAcmCertificate.IMPORTER_UUID = '2d20d8f2-7b84-444e-b738-c75b499a9eaa';
//# sourceMappingURL=data:application/json;base64,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