/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { ICertificate } from '@aws-cdk/aws-certificatemanager';
import { Table } from '@aws-cdk/aws-dynamodb';
import { IKey } from '@aws-cdk/aws-kms';
import { ISecret } from '@aws-cdk/aws-secretsmanager';
import { Construct, Stack, Tag } from '@aws-cdk/core';
/**
 * Properties for importing a Certificate from Secrets into ACM.
 */
export interface ImportedAcmCertificateProps {
    /**
     * A Secret that contains the Certificate data
     */
    readonly cert: ISecret;
    /**
     * A Secret that contains the encrypted Private Key data
     */
    readonly key: ISecret;
    /**
     * A Secret that contains the passphrase of the encrypted Private Key
     */
    readonly passphrase: ISecret;
    /**
     * A Secret that contains the chain of Certificates used to sign this Certificate
     * @default: No certificate chain is used, signifying a self-signed Certificate
     */
    readonly certChain?: ISecret;
    /**
     * The KMS Key used to encrypt the secrets. The Custom Resource to import the Certificate to ACM will be granted
     * permission to decrypt Secrets using this Key.
     * @default: If the account's default CMK was used to encrypt the Secrets, no special permissions need to be given
     */
    readonly encryptionKey?: IKey;
}
/**
 * A Construct that holds a Custom Resource modelling a certificate that was imported into ACM. It uses a Lambda
 * Function to extract the certificate from Secrets and then import it into ACM. It is intended to be used with the
 * X509CertificatePem Construct.
 *
 * Resources Deployed
 * ------------------------
 * 1) DynamoDB Table - Used for tracking resources created by the CustomResource.
 * 2) Lambda Function, with Role - Used to create/update/delete the CustomResource.
 * 3) ACM Certificate - Created by the CustomResource.
 *
 * @ResourcesDeployed
 */
export declare class ImportedAcmCertificate extends Construct implements ICertificate {
    private static IMPORTER_UUID;
    /**
     * The ARN for the Certificate that was imported into ACM
     */
    readonly certificateArn: string;
    readonly stack: Stack;
    protected readonly database: Table;
    protected readonly uniqueTag: Tag;
    constructor(scope: Construct, id: string, props: ImportedAcmCertificateProps);
}
