"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthMonitor = void 0;
const path = require("path");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_cloudwatch_actions_1 = require("@aws-cdk/aws-cloudwatch-actions");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sns_1 = require("@aws-cdk/aws-sns");
const aws_sns_subscriptions_1 = require("@aws-cdk/aws-sns-subscriptions");
const core_1 = require("@aws-cdk/core");
const load_balancer_manager_1 = require("./load-balancer-manager");
const runtime_info_1 = require("./runtime-info");
/**
 *  A new or imported Health Monitor.
 */
class HealthMonitorBase extends core_1.Construct {
}
/**
 * This construct is responsible for the deep health checks of compute instances.
 * It also replaces unhealthy instances and suspends unhealthy fleets.
 * Although, using this constructs adds up additional costs for monitoring,
 * it is highly recommended using this construct to help avoid / minimize runaway costs for compute instances.
 *
 * An instance is considered to be unhealthy when:
 *   1) Deadline client is not installed on it;
 *   2) Deadline client is installed but not running on it;
 *   3) RCS is not configured correctly for Deadline client;
 *   4) it is unable to connect to RCS due to any infrastructure issues;
 *   5) the health monitor is unable to reach it because of some infrastructure issues.
 *
 * A fleet is considered to be unhealthy when:
 *   1) at least 1 instance is unhealthy for the configured grace period;
 *   2) a percentage of unhealthy instances in the fleet is above a threshold at any given point of time.
 *
 * This internally creates an array of application load balancers and attaches
 * the worker-fleet (which internally is implemented as an Auto Scaling Group) to its listeners.
 * There is no load-balancing traffic on the load balancers,
 * it is only used for health checks.
 * Intention is to use the default properties of laod balancer health
 * checks which does HTTP pings at frequent intervals to all the
 * instances in the fleet and determines its health. If any of the
 * instance is found unhealthy, it is replaced. The target group
 * also publishes the unhealthy target count metric which is used
 * to identify the unhealthy fleet.
 *
 * Other than the default instance level protection, it also creates a lambda
 * which is responsible to set the fleet size to 0 in the event of a fleet
 * being sufficiently unhealthy to warrant termination.
 * This lambda is triggered by CloudWatch alarms via SNS (Simple Notification Service).
 *
 * Resources Deployed
 * ------------------------
 * 1) Application Load Balancer(s) doing frequent pings to the workers;
 * 2) KMS Key to encrypt SNS messages - If no encryption key is provided;
 * 3) SNS topic for all unhealthy fleet notifications;
 * 4) The Alarm if the node is unhealthy for a long period;
 * 5) The Alarm if the healthy host percentage if lower than allowed;
 * 4) A single Lambda function that sets fleet size to 0 when triggered.
 *
 * @ResourcesDeployed
 */
class HealthMonitor extends HealthMonitorBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.stack = core_1.Stack.of(scope);
        this.props = props;
        this.lbFactory = new load_balancer_manager_1.LoadBalancerFactory(this, props.vpc);
        const topicEncryptKey = props.encryptionKey || new aws_kms_1.Key(this, 'SNSEncryptionKey', {
            description: `This key is used to encrypt SNS messages for ${this.node.uniqueId}.`,
            enableKeyRotation: true,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            trustAccountIdentities: true,
        });
        // allow cloudwatch service to send encrypted messages
        topicEncryptKey.grant(new aws_iam_1.ServicePrincipal('cloudwatch.amazonaws.com'), 'kms:Decrypt', 'kms:GenerateDataKey');
        this.unhealthyFleetActionTopic = new aws_sns_1.Topic(this, 'UnhealthyFleetTopic', {
            masterKey: topicEncryptKey,
        });
        this.unhealthyFleetActionTopic.grantPublish(new aws_iam_1.ServicePrincipal('cloudwatch.amazonaws.com'));
        this.alarmTopicAction = new aws_cloudwatch_actions_1.SnsAction(this.unhealthyFleetActionTopic);
        this.unhealthyFleetActionLambda = new aws_lambda_1.SingletonFunction(this, 'UnhealthyFleetAction', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '../lambdas/nodejs/unhealthyFleetAction')),
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            lambdaPurpose: 'unhealthyFleetTermination',
            timeout: core_1.Duration.seconds(300),
            uuid: '28bccf6a-aa76-478c-9239-e2f5bcc0254c',
        });
        this.unhealthyFleetActionTopic.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(this.unhealthyFleetActionLambda));
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Attaches the load-balancing target to the ELB for instance-level
     * monitoring. The ELB does frequent pings to the workers and determines
     * if a worker node is unhealthy. If so, it replaces the instance.
     *
     * It also creates an Alarm for healthy host percent and suspends the
     * fleet if the given alarm is breaching. It sets the maxCapacity
     * property of the auto-scaling group to 0. This should be
     * reset manually after fixing the issue.
     *
     * @param monitorableFleet
     * @param healthCheckConfig
     */
    registerFleet(monitorableFleet, healthCheckConfig) {
        const { loadBalancer, targetGroup } = this.lbFactory.registerWorkerFleet(monitorableFleet, healthCheckConfig, this.props);
        this.createFleetAlarms(monitorableFleet, healthCheckConfig, loadBalancer, targetGroup);
    }
    createFleetAlarms(monitorableFleet, healthCheckConfig, loadBalancer, targetGroup) {
        monitorableFleet.connections.allowFrom(loadBalancer, aws_ec2_1.Port.tcp(healthCheckConfig.port || HealthMonitor.LOAD_BALANCER_LISTENING_PORT));
        const percentMetric = new aws_cloudwatch_1.MathExpression({
            label: 'UnhealthyHostPercent',
            expression: 'IF(fleetCapacity, 100*(unhealthyHostCount/fleetCapacity), 0)',
            usingMetrics: {
                unhealthyHostCount: targetGroup.metricUnhealthyHostCount({
                    statistic: 'max',
                }),
                fleetCapacity: monitorableFleet.targetCapacityMetric,
            },
            period: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_HARD,
        });
        // When unhealthy fleet is more than healthyFleetThresholdPercent or 35% at any given period of 5 minutes
        const immediateTerminationAlarm = percentMetric.createAlarm(monitorableFleet.targetScope, 'UnhealthyFleetTermination', {
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            threshold: 100 - (healthCheckConfig.healthyFleetThresholdPercent || HealthMonitor.DEFAULT_HEALTHY_FLEET_THRESHOLD_PERCENT_HARD),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD,
            datapointsToAlarm: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD,
            actionsEnabled: true,
        });
        immediateTerminationAlarm.addAlarmAction(this.alarmTopicAction);
        // When at least one node is unhealthy over a period of 2 hours
        const percentMetricGracePeriod = new aws_cloudwatch_1.MathExpression({
            label: 'UnhealthyHostPercent',
            expression: 'IF(fleetCapacity, 100*(unhealthyHostCount/fleetCapacity), 0)',
            usingMetrics: {
                unhealthyHostCount: targetGroup.metricUnhealthyHostCount({
                    statistic: 'max',
                }),
                fleetCapacity: monitorableFleet.targetCapacityMetric,
            },
            period: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_GRACE,
        });
        const gracePeriodTerminationAlarm = percentMetricGracePeriod.createAlarm(monitorableFleet.targetScope, 'UnhealthyFleetGracePeriod', {
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            threshold: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_THRESHOLD_PERCENT_GRACE,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE,
            datapointsToAlarm: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE,
            actionsEnabled: true,
        });
        gracePeriodTerminationAlarm.addAlarmAction(this.alarmTopicAction);
        monitorableFleet.targetUpdatePolicy.attachToRole(this.unhealthyFleetActionLambda.role);
    }
}
exports.HealthMonitor = HealthMonitor;
/**
 * Default health check listening port
 */
HealthMonitor.DEFAULT_HEALTH_CHECK_PORT = 63415;
/**
 * Resource Tracker in Deadline currently publish health status every 5 min, hence keeping this same
 */
HealthMonitor.DEFAULT_HEALTH_CHECK_INTERVAL = core_1.Duration.minutes(5);
/**
 * Resource Tracker in Deadline currently determines host unhealthy in 15 min, hence keeping this count
 */
HealthMonitor.DEFAULT_UNHEALTHY_HOST_THRESHOLD = 3;
/**
 * This is the minimum possible value of ALB health-check config, we want to mark worker healthy ASAP
 */
HealthMonitor.DEFAULT_HEALTHY_HOST_THRESHOLD = 2;
/**
 * Since we are not doing any load balancing, this port is just an arbitrary port.
 */
HealthMonitor.LOAD_BALANCER_LISTENING_PORT = 8081;
/**
 * This number is taken from Resource Tracker implementation. If a fleet's healthy percent
 * is less than this threshold at any given point of time, it is suspended.
 */
HealthMonitor.DEFAULT_HEALTHY_FLEET_THRESHOLD_PERCENT_HARD = 65;
/**
 * This number is taken from Resource Tracker implementation. If a fleet has at least 1
 * unhealthy host for a period of 2 hours, it is suspended.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_THRESHOLD_PERCENT_GRACE = 0;
/**
 * This number is taken from Resource Tracker implementation. We monitor unhealthy fleet for immediate
 * termination for a period fo 5 minutes.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_HARD = core_1.Duration.minutes(5);
/**
 * In Resource Tracker, we evaluate the fleet's health for determining the grace period over a period
 * of 5 minutes. For the first unhealthy signal, a instance can take upto 10min (max), hence we are
 * setting this period to be 15.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_GRACE = core_1.Duration.minutes(15);
/**
 * This number is taken from Resource Tracker implementation. Fleet is terminated immediately if it
 * has unhealthy host percent above the hard limit.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD = 1;
/**
 * This number is taken from Resource Tracker implementation. The grace period duration is 2 hours,
 * since the grace period is 15 minutes, we need continuous 8 data points crossing the threshold.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE = 8;
//# sourceMappingURL=data:application/json;base64,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