"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudWatchAgent = void 0;
const path = require("path");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_ssm_1 = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
const script_assets_1 = require("./script-assets");
/**
 * This construct is a thin wrapper that provides the ability to install and configure the CloudWatchAgent
 * ( https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Install-CloudWatch-Agent.html )
 * on one or more EC2 instances during instance startup.
 *
 * It accomplishes this by downloading and executing the configuration script on the instance.
 * The script will download the CloudWatch Agent installer,
 * optionally verify the installer, and finally install the CloudWatch Agent.
 * The installer is downloaded via the Amazon S3 API, thus, this construct can be used
 * on instances that have no access to the internet as long as the VPC contains
 * an VPC Gateway Endpoint for S3 ( https://docs.aws.amazon.com/vpc/latest/userguide/vpc-endpoints-s3.html ).
 *
 * {@link CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR} - Context variable to skip validation
 * of the downloaded CloudWatch Agent installer if set to 'TRUE'.
 * WARNING: Only use this if your deployments are failing due to a validation failure,
 * but you have verified that the failure is benign.
 *
 * Resources Deployed
 * ------------------------
 * 1) String SSM Parameter in Systems Manager Parameter Store to store the cloudwatch agent configuration;
 * 2) A script Asset which is uploaded to S3 bucket.
 *
 * @ResourcesDeployed
 */
class CloudWatchAgent extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Create the asset for the configuration script
        this.configurationScript = script_assets_1.ScriptAsset.fromPathConvention(scope, 'CloudWatchConfigurationScriptAsset', {
            osType: props.host.osType,
            baseName: 'configureCloudWatchAgent',
            rootDir: path.join(__dirname, '../scripts/'),
        });
        // Create a new SSM Parameter holding the json configuration
        this.ssmParameterForConfig = new aws_ssm_1.StringParameter(scope, 'StringParameter', {
            description: 'config file for Repository logs config',
            stringValue: props.cloudWatchConfig,
        });
        this.grantRead(props.host);
        this.configure(props.host, this.node.tryGetContext(CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR) === 'TRUE');
    }
    /**
     * Grants read permissions to the principal on the assets bucket and parameter store.
     */
    grantRead(grantee) {
        this.configurationScript.grantRead(grantee);
        this.ssmParameterForConfig.grantRead(grantee);
    }
    /**
     * Configures the CloudWatch Agent on the target host.
     *
     * This is done by adding UserData commands to the target host.
     *
     * @param host The host to configure the CloudWatch agent on
     * @param skipValidation Skips the validation of the CloudWatch agent installer if set to true.
     */
    configure(host, skipValidation) {
        // Grant access to the required CloudWatch Agent installer files
        const cloudWatchAgentBucket = aws_s3_1.Bucket.fromBucketArn(this, 'CloudWatchAgentBucket', 'arn:aws:s3:::amazoncloudwatch-agent');
        cloudWatchAgentBucket.grantRead(host);
        const scriptArgs = [];
        if (skipValidation) {
            // Flags must be set before positional arguments for some scripts
            scriptArgs.push(CloudWatchAgent.SKIP_CWAGENT_VALIDATION_FLAG);
        }
        scriptArgs.push(this.ssmParameterForConfig.parameterName);
        this.configurationScript.executeOn({
            host,
            args: scriptArgs,
        });
    }
}
exports.CloudWatchAgent = CloudWatchAgent;
/**
 * The context variable to indicate that CloudWatch agent installer validation should be skipped.
 */
CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR = 'RFDK_SKIP_CWAGENT_VALIDATION';
/**
 * The flag for configureCloudWatchAgent script to skip CloudWatch agent installer validation.
 */
CloudWatchAgent.SKIP_CWAGENT_VALIDATION_FLAG = '-s';
//# sourceMappingURL=data:application/json;base64,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