#!/usr/bin/env node
"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
const url = require("url");
const util_1 = require("util");
const args = process.argv.slice(2);
let deadlineInstallerURI = '';
let dockerRecipesURI = '';
let outputFolder = './stage';
let verbose = false;
// Parse command parameters
let n = 0;
while (n < args.length) {
    switch (args[n]) {
        case '-d':
        case '--deadlineInstallerURI':
            n++;
            deadlineInstallerURI = args[n];
            break;
        case '-c':
        case '--dockerRecipesURI':
            n++;
            dockerRecipesURI = args[n];
            break;
        case '-o':
        case '--output':
            n++;
            outputFolder = args[n];
            break;
        case '--verbose':
            verbose = true;
            break;
        default:
            console.error(`Unexpected command parameter ${args[n]}`);
            process.exit(1);
    }
    n++;
}
// Show help if URI for deadline installer or URI for docker  is not specified.
if (deadlineInstallerURI === '' || dockerRecipesURI === '') {
    usage(1);
}
const deadlineInstallerURL = new url.URL(deadlineInstallerURI);
const dockerRecipeURL = new url.URL(dockerRecipesURI);
if (deadlineInstallerURL.protocol !== 's3:') {
    console.error(`ERROR: Invalid URI protocol ("${deadlineInstallerURL.protocol}") for --deadlineInstallerURI. Only "s3:" URIs are supported.`);
    usage(1);
}
if (dockerRecipeURL.protocol !== 's3:') {
    console.error(`ERROR: Invalid URI protocol ("${dockerRecipeURL.protocol}") for --dockerRecipeURL. Only "s3:" URIs are supported.`);
    usage(1);
}
if (!validateBucketName(deadlineInstallerURL.hostname) || !validateBucketName(dockerRecipeURL.hostname)) {
    usage(1);
}
if (!fs.existsSync(outputFolder)) {
    fs.mkdirSync(outputFolder);
}
else if (fs.readdirSync(outputFolder).length > 0) {
    console.error('The target directory is not empty.');
    process.exit(1);
}
try {
    // Get Docker recipe
    getAndExtractArchive({
        uri: dockerRecipeURL,
        targetFolder: outputFolder,
        verbose,
        tarOptions: [`-x${verbose ? 'v' : ''}z`],
    });
    // Get Deadline client installer
    const deadlineInstallerPath = getFile({
        uri: deadlineInstallerURL,
        targetFolder: path.join(outputFolder, 'bin'),
        verbose,
    });
    // Make installer executable
    makeExecutable(deadlineInstallerPath);
}
catch (e) {
    let errorMsg;
    if (util_1.types.isNativeError(e)) {
        errorMsg = e.message;
    }
    else {
        errorMsg = e.toString();
    }
    console.error(`ERROR: ${errorMsg}`);
    process.exit(1);
}
/**
 * Attempts to add UNIX executable permission bits to a file.
 *
 * Any errors are caught and:
 *  - more user-friendly error message is displayed
 *
 * @param filePath Path to make executable
 */
function makeExecutable(filePath) {
    try {
        let mode = fs.statSync(filePath).mode;
        // eslint-disable-next-line no-bitwise
        mode = mode | fs.constants.S_IXUSR | fs.constants.S_IXGRP | fs.constants.S_IXOTH;
        fs.chmodSync(filePath, mode);
    }
    catch (e) {
        let errorDetail;
        if (e instanceof Error) {
            errorDetail = e.message;
        }
        else {
            errorDetail = e.toString();
        }
        throw new Error(`Could not add executable permission to Deadline installer: ${errorDetail}`);
    }
}
/**
 * Validate name of S3 bucket
 *
 * @param s3BucketName - name of S3 bucket
 */
function validateBucketName(s3BucketName) {
    const regExpForS3Bucket = new RegExp('^([a-z]|(\\d(?!\\d{0,2}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3})))\
([a-z\\d]|(\\.(?!(\\.|-)))|(-(?!\\.))){1,61}[a-z\\d]$');
    if (!regExpForS3Bucket.test(s3BucketName)) {
        console.error(`S3 bucket name '${s3BucketName}' has invalid format.\
    Please follow S3 bucket naming requirements:\
    https://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html`);
        return false;
    }
    return true;
}
/**
 * Spawns a shell then executes the command within that shell.
 * Method will not return until the child process has fully closed.
 *
 * @param command - The command to run.
 * @param showCommand - If true shows which command will be spawned.
 */
function spawnCommand(command, showCommand, commandArgs = []) {
    if (showCommand) {
        console.info(`Command: ${command} ${commandArgs.join(' ')}`);
    }
    const result = child_process_1.spawnSync(command, commandArgs, {
        stdio: 'inherit',
    });
    return result.status === 0;
}
/**
 * Show help string for usage and exit with error code.
 *
 * @param errorCode - THe code of error that will be returned.
 */
function usage(errorCode) {
    console.info(`
Stages a Deadline release for building Docker container images with RFDK.
This tool requires that tar and the AWS CLI are installed.
The AWS CLI must be configured to authenticate using your AWS account. This can be done by configuring your default profile or with environment.
See https://docs.aws.amazon.com/cli/latest/userguide/cli-chap-configure.html for documentation on how to configure the AWS CLI.

Usage: stage-deadline [--output <output_dir>] [--verbose]
                      -d <deadline_installer_uri>
                      -c <deadline_recipes_uri>

Arguments:
    -d, --deadlineInstallerURI <deadline_installer_uri>
        Specifies a URI pointing to the Deadline Linux Client installer. This currently supports S3 URIs of the form:

        s3://thinkbox-installers/Deadline/10.1.x.y/Linux/DeadlineClient-10.1.x.y-linux-x64-installer.run

    -c, --dockerRecipesURI <deadline_recipes_uri>
        Specifies a URI pointing to the Deadline Docker recipes. This currently supports S3 URIs of the form:

        s3://thinkbox-installers/DeadlineDocker/10.1.x.y/DeadlineDocker-10.1.x.y.tar.gz

Options:
    -o, --output <output_dir>
        Specifies a path to an output directory where Deadline will be staged. The default is to use a "stage"
        sub-directory of the working directory.

    --verbose
        Increases the verbosity of the output
  `.trimLeft());
    process.exit(errorCode);
}
/**
 * Gets a file from a specified URI.
 *
 * This is currently limited to obtaining objects from S3.
 *
 * @param props Properties for fetching the file
 * @returns The path to the fetched file
 */
function getFile(props) {
    if (!fs.existsSync(props.targetFolder)) {
        fs.mkdirSync(props.targetFolder);
    }
    const cmdArgs = ['s3', 'cp'];
    if (!props.verbose) {
        cmdArgs.push('--quiet');
    }
    cmdArgs.push(props.uri.href);
    cmdArgs.push(props.targetFolder);
    const success = spawnCommand('aws', props.verbose, cmdArgs);
    if (!success) {
        throw new Error(`Could not fetch ${props.uri.href}`);
    }
    return path.join(props.targetFolder, uriFilename(props.uri));
}
/**
 * Return the filename portion of the URL.
 *
 * This is the right-most component (separated by / characters) of a URI's path.
 *
 * @param uri The source URI
 */
function uriFilename(uri) {
    const pathParts = uri.pathname.split('/');
    const fileName = pathParts[pathParts.length - 1];
    return fileName;
}
/**
 * Obtain and extract a tar archive.
 *
 * @param props Properties to specify how to get and extract the archive
 */
function getAndExtractArchive(props) {
    const { targetFolder, tarOptions } = props;
    const tarPath = getFile(props);
    const filesExtracted = spawnCommand('tar', props.verbose, [
        '-C',
        targetFolder,
        ...tarOptions,
        '-f',
        tarPath,
    ]);
    if (fs.existsSync(tarPath)) {
        fs.unlinkSync(tarPath);
    }
    // Exit with error if recipe wasn't extracted.
    if (!filesExtracted) {
        throw new Error(`File ${tarPath} has not been extracted successfully.`);
    }
}
//# sourceMappingURL=data:application/json;base64,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