"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserTablePrivileges = void 0;
const cdk = require("aws-cdk-lib");
const table_1 = require("../table");
const database_query_1 = require("./database-query");
const handler_name_1 = require("./database-query-provider/handler-name");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const constructs_1 = require("constructs");
/**
 * Privileges granted to a Redshift user on Redshift tables.
 *
 * This construct is located in the `private` directory to ensure that it is not exported for direct public use. This
 * means that user privileges must be managed through the `Table.grant` method or the `User.addTablePrivileges`
 * method. Thus, each `User` will have at most one `UserTablePrivileges` construct to manage its privileges. For details
 * on why this is a Good Thing, see the README, under "Granting Privileges".
 */
class UserTablePrivileges extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.privileges = (_a = props.privileges) !== null && _a !== void 0 ? _a : [];
        new database_query_1.DatabaseQuery(this, 'Resource', {
            ...props,
            handler: handler_name_1.HandlerName.UserTablePrivileges,
            properties: {
                username: props.user.username,
                tablePrivileges: cdk.Lazy.any({
                    produce: () => {
                        const reducedPrivileges = this.privileges.reduce((privileges, { table, actions }) => {
                            const tableName = table.tableName;
                            if (!(tableName in privileges)) {
                                privileges[tableName] = [];
                            }
                            actions = actions.concat(privileges[tableName]);
                            if (actions.includes(table_1.TableAction.ALL)) {
                                actions = [table_1.TableAction.ALL];
                            }
                            if (actions.includes(table_1.TableAction.UPDATE) || actions.includes(table_1.TableAction.DELETE)) {
                                actions.push(table_1.TableAction.SELECT);
                            }
                            privileges[tableName] = Array.from(new Set(actions));
                            return privileges;
                        }, {});
                        const serializedPrivileges = Object.entries(reducedPrivileges).map(([tableName, actions]) => ({
                            tableName: tableName,
                            actions: actions.map(action => table_1.TableAction[action]),
                        }));
                        return serializedPrivileges;
                    },
                }),
            },
        });
    }
    /**
     * Grant this user additional privileges.
     */
    addPrivileges(table, ...actions) {
        this.privileges.push({ table, actions });
    }
}
exports.UserTablePrivileges = UserTablePrivileges;
//# sourceMappingURL=data:application/json;base64,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