"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const cf = require("@aws-cdk/aws-cloudfront");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
test('Cloudfront Distribution Included', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            CustomErrorResponses: [
                {
                    ErrorCode: 403,
                    ResponseCode: 200,
                    ResponsePagePath: '/index.html',
                },
                {
                    ErrorCode: 404,
                    ResponseCode: 200,
                    ResponsePagePath: '/index.html',
                },
            ],
            DefaultCacheBehavior: {
                ViewerProtocolPolicy: 'redirect-to-https',
            },
            DefaultRootObject: 'index.html',
            HttpVersion: 'http2',
            IPV6Enabled: true,
            PriceClass: 'PriceClass_100',
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true,
            },
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        CanonicalUser: {
                            'Fn::GetAtt': [
                                'spaDeployOriginAccessIdentityEDA4C19C',
                                'S3CanonicalUserId',
                            ],
                        },
                    },
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'spaDeployWebsiteBucket1E4C4442',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'spaDeployWebsiteBucket1E4C4442',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                }
            ],
        },
    }));
});
test('Cloudfront Distribution Included - with non-default error-doc cfg set', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            CustomErrorResponses: [
                {
                    ErrorCode: 403,
                    ResponseCode: 200,
                    ResponsePagePath: '/error.html',
                },
                {
                    ErrorCode: 404,
                    ResponseCode: 200,
                    ResponsePagePath: '/error.html',
                },
            ],
            DefaultCacheBehavior: {
                ViewerProtocolPolicy: 'redirect-to-https',
            },
            DefaultRootObject: 'index.html',
            HttpVersion: 'http2',
            IPV6Enabled: true,
            PriceClass: 'PriceClass_100',
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true,
            },
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        CanonicalUser: {
                            'Fn::GetAtt': [
                                'spaDeployOriginAccessIdentityEDA4C19C',
                                'S3CanonicalUserId',
                            ],
                        },
                    },
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'spaDeployWebsiteBucket1E4C4442',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'spaDeployWebsiteBucket1E4C4442',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                }
            ],
        },
    }));
});
test('Cloudfront With Custom Cert and Aliases', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com'],
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.test.com',
            ],
            ViewerCertificate: {
                AcmCertificateArn: 'arn:1234',
                SslSupportMethod: 'sni-only',
            },
        },
    }));
});
test('Basic Site Setup', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:GetObject',
                    Effect: 'Allow',
                    Principal: '*',
                }
            ],
        },
    }));
});
test('Basic Site Setup with Error Doc set', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createBasicSite({
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:GetObject',
                    Effect: 'Allow',
                    Principal: '*',
                }
            ],
        },
    }));
});
test('Basic Site Setup, Encrypted Bucket', () => {
    const stack = new core_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:GetObject',
                    Effect: 'Allow',
                    Principal: '*',
                }
            ],
        },
    }));
});
test('Cloudfront With Encrypted Bucket', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true });
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        certificateARN: 'arn:1234',
        cfAliases: ['www.test.com'],
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.test.com',
            ],
            ViewerCertificate: {
                AcmCertificateArn: 'arn:1234',
                SslSupportMethod: 'sni-only',
            },
        },
    }));
});
test('Cloudfront With Custom Defined Behaviors', () => {
    const stack = new core_1.Stack();
    // WHEN
    const deploy = new lib_1.SPADeploy(stack, 'spaDeploy');
    deploy.createSiteWithCloudfront({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        cfBehaviors: [
            {
                isDefaultBehavior: true,
                allowedMethods: cf.CloudFrontAllowedMethods.ALL,
                forwardedValues: {
                    queryString: true,
                    cookies: { forward: 'all' },
                    headers: ['*'],
                },
            },
            {
                pathPattern: '/virtual-path',
                allowedMethods: cf.CloudFrontAllowedMethods.GET_HEAD,
                cachedMethods: cf.CloudFrontAllowedCachedMethods.GET_HEAD,
            },
        ],
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            CacheBehaviors: [
                {
                    AllowedMethods: ['GET', 'HEAD'],
                    CachedMethods: ['GET', 'HEAD'],
                    PathPattern: '/virtual-path',
                },
            ],
            DefaultCacheBehavior: {
                AllowedMethods: ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT'],
                ForwardedValues: {
                    Cookies: { Forward: 'all' },
                    Headers: ['*'],
                    QueryString: true,
                },
                TargetOriginId: 'origin1'
            },
        },
    }));
});
test('Basic Site Setup, IP Filter', () => {
    const stack = new core_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true, ipFilter: true, ipList: ['1.1.1.1'] })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 's3:GetObject',
                    Condition: {
                        IpAddress: {
                            'aws:SourceIp': [
                                '1.1.1.1',
                            ],
                        },
                    },
                    Effect: 'Allow',
                    Principal: '*',
                }
            ],
        },
    }));
});
test('Create From Hosted Zone', () => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234',
        },
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.cdkspadeploy.com',
            ],
            ViewerCertificate: {
                SslSupportMethod: 'sni-only',
            },
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        CanonicalUser: {
                            'Fn::GetAtt': [
                                'spaDeployOriginAccessIdentityEDA4C19C',
                                'S3CanonicalUserId',
                            ],
                        },
                    },
                    Resource: [
                        {
                            'Fn::GetAtt': [
                                'spaDeployWebsiteBucket1E4C4442',
                                'Arn',
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'spaDeployWebsiteBucket1E4C4442',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                }
            ],
        },
    }));
});
test('Create From Hosted Zone with Error Bucket', () => {
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'testStack', {
        env: {
            region: 'us-east-1',
            account: '1234',
        },
    });
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { encryptBucket: true })
        .createSiteFromHostedZone({
        zoneName: 'cdkspadeploy.com',
        indexDoc: 'index.html',
        errorDoc: 'error.html',
        websiteFolder: 'website',
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: 'AES256',
                    },
                },
            ],
        },
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
            ErrorDocument: 'error.html',
        },
    }));
    assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            Aliases: [
                'www.cdkspadeploy.com',
            ],
            ViewerCertificate: {
                SslSupportMethod: 'sni-only',
            },
        },
    }));
});
test('Basic Site Setup, URL Output Enabled With Name', () => {
    const stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: true,
        exportWebsiteUrlName: exportName,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveOutput({
        exportName,
    }));
});
test('Basic Site Setup, URL Output Enabled With No Name', () => {
    const stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: true,
    });
    // THEN
    assert_1.expect(stack).notTo(assert_1.haveOutput({
        exportName,
    }));
});
test('Basic Site Setup, URL Output Not Enabled', () => {
    const stack = new core_1.Stack();
    const exportName = 'test-export-name';
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', {})
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        exportWebsiteUrlOutput: false,
    });
    // THEN
    assert_1.expect(stack).notTo(assert_1.haveOutput({
        exportName,
    }));
});
test('Basic Site Setup, Block Public Enabled', () => {
    const stack = new core_1.Stack();
    // WHEN
    new lib_1.SPADeploy(stack, 'spaDeploy', { ipFilter: true })
        .createBasicSite({
        indexDoc: 'index.html',
        websiteFolder: 'website',
        blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
        WebsiteConfiguration: {
            IndexDocument: 'index.html',
        },
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            BlockPublicPolicy: true,
            IgnorePublicAcls: true,
            RestrictPublicBuckets: true
        },
    }));
});
//# sourceMappingURL=data:application/json;base64,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