# Deep Feature Synthesis in Rust

This is a project to implement the [Deep Feature Synthesis](https://github.com/alteryx/featuretools/blob/main/featuretools/synthesis/dfs.py) algorithm in Rust.

## Running in Python

`pip install rust_dfs`

Using from python

```python
# Import Featuretools, rust_dfs, and some other utility functions
import featuretools as ft
from rust_dfs.utils import *
from rust_dfs import generate_features, compare_featuresets

# Generate a fake dataset with 2 Integer columns
df = generate_fake_dataframe(
    n_rows=10,
    col_defs=[
        ([1,2,3], 2),
        # ("Boolean", 2),
    ]
)

# pick some primitives
f_primitives = [
    ft.primitives.GreaterThan,
    # ft.primitives.LessThan
]

# or use all of them
# f_primitives = list(ft.primitives.utils.get_transform_primitives().values())

# convert datafame to an entityset
es = df_to_es(df)

# run dfs with features_only=True
ft_feats = ft.dfs(
    entityset=es, 
    target_dataframe_name="nums", 
    trans_primitives=f_primitives, 
    features_only=True,
    max_depth=1
)

# ft_feats = [<Feature: F_0>, <Feature: F_1>, <Feature: F_0 > F_1>, <Feature: F_1 > F_0>]

# Convert back into a format that we can use to compare with rust
c_feats = list(convert_features(ft_feats).values())

# Now run using Rust

# convert featuretools primitives to rust primitives
r_primitives = convert_primitives(f_primitives)

# convert dataframe to rust features
r_features = dataframe_to_features(es.dataframes[0])

# generate engineered features using Rust (create new features only)
r_derived_feats = generate_features(r_features, r_primitives)


a,b = compare_featuresets(c_feats, r_derived_feats)

print("=== Features generated by Featuretools, that were NOT generated by Rust ===")
print(a)
print()
print("=== Features generated by Rust, that were NOT generated by Featuretools ===")
print(b)
```

## Develop Guide

Create virtualenv
```
python -m venv .env
source .env/bin/activate
pip install maturin
pip install featuretools
pip install pandas
```

Ensure `Cargo.toml` is configured

```toml
[lib]
name = "rust_dfs"
crate-type = ["cdylib"]
```

Run `maturin`

```
maturin develop
```

## Run `main.rs`

To run as a rust binary:

```
cargo run --no-default-features
```