# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['secret_sdk',
 'secret_sdk.client',
 'secret_sdk.client.lcd',
 'secret_sdk.client.lcd.api',
 'secret_sdk.core',
 'secret_sdk.core.auth',
 'secret_sdk.core.auth.data',
 'secret_sdk.core.bank',
 'secret_sdk.core.distribution',
 'secret_sdk.core.staking',
 'secret_sdk.core.staking.data',
 'secret_sdk.core.wasm',
 'secret_sdk.key',
 'secret_sdk.util']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.7.3,<4.0.0',
 'attrs>=20.3.0,<21.0.0',
 'bech32>=1.2.0,<2.0.0',
 'bip32utils>=0.3.post4,<0.4',
 'boltons>=21.0.0,<22.0.0',
 'ecdsa>=0.16.1,<0.17.0',
 'furl>=2.1.3,<3.0.0',
 'miscreant>=0.3.0,<0.4.0',
 'mnemonic>=0.19,<0.20',
 'nest-asyncio>=1.5.1,<2.0.0',
 'protobuf>=3.17.3,<4.0.0',
 'wrapt>=1.12.1,<2.0.0']

setup_kwargs = {
    'name': 'secret-sdk',
    'version': '1.0',
    'description': 'The Python SDK for Secret',
    'long_description': '# <div  align="center"> <p > Python SDK for Secret Network </p>\n\n<br>\n<br>\nThe Secret Software Development Kit (SDK) in Python is a simple library toolkit for building software that can interact with the Secret blockchain and provides simple abstractions over core data structures, serialization, key management, and API request generation. The SDK is based on a fork of <a href="https://github.com/terra-money/terra.py">terra.py</a> on Terra \n\n## Features\n\n- Written in Python offering extensive support libraries\n- Versatile support for key management solutions\n- Exposes the Secret Rest API through LCDClient\n\n<br/>\n\n# Table of Contents\n- [API Reference](#api-reference)\n- [Getting Started](#getting-started)\n  - [Requirements](#requirements)\n  - [Installation](#installation)\n  - [Dependencies](#dependencies)\n  - [Tests](#tests)\n  - [Code Quality](#code-quality)\n- [Usage Examples](#usage-examples) \n  - [Getting Blockchain Information](#getting-blockchain-information)\n    - [Async Usage](#async-usage)\n  - [Building and Signing Transactions](#building-and-signing-transactions)\n      - [Example Using a Wallet](#example-using-a-wallet-recommended)\n- [Contributing](#contributing)\n  - [Reporting an Issue](#reporting-an-issue)\n  - [Requesting a Feature](#requesting-a-feature)\n  - [Contributing Code](#contributing-code)\n  - [Documentation Contributions](#documentation-contributions)\n- [License](#license)\n\n<br/>\n\n\n\n# API Reference\nAn intricate reference to the APIs on the Secret SDK can be found <a href="https://secretapi.io/#/">here</a>.\n\n<br/>\n\n# Getting Started\nA walk-through of the steps to get started with the Secret SDK alongside with a few use case examples are provided below. \n\n## Requirements\nSecret SDK requires <a href="https://www.python.org/downloads/">Python v3.7+</a>.\n\n## Installation\n\n<sub>**NOTE:** *All code starting with a `$` is meant to run on your terminal (a bash prompt). All code starting with a `>>>` is meant to run in a python interpreter, like <a href="https://pypi.org/project/ipython/">ipython</a>.*</sub>\n\nSecret SDK can be installed (preferably in a `virtual environment` from PyPI using `pip`) as follows:\n\n```\n$ pip install -U secret-sdk\n```\n<sub>*You might have `pip3` installed instead of `pip`; proceed according to your own setup.*<sub>\n\n## Dependencies\nSecret SDK uses <a href="https://python-poetry.org/">Poetry</a> to manage dependencies. To get set up with all the required dependencies, run:\n```\n$ pip install poetry\n$ poetry install\n```\n\n\n## Tests\nSecret SDK provides extensive tests for data classes and functions. To run them, after the steps in [Dependencies](#dependencies):\n```\n$ make test\n```\n\n<br/>\n\n# Usage Examples\nSecret SDK can help you read block data, query secret contracts, sign and send transactions, and many more.\nFollowing examples are provided to help get building started.\n\nIn order to interact with the Secret blockchain, you\'ll need a connection to a Secret node or an api. This can be done through setting up an LCDClient (The LCDClient is an object representing an HTTP connection to a Secret LCD node.):\n\n```\n>>> from secret_sdk.client.lcd import LCDClient\n>>> secret = LCDClient(chain_id="secret-4", url=node_rest_endpoint)\n```\n\n## Getting Blockchain Information\n\nOnce properly configured, the `LCDClient` instance will allow you to interact with the Secret blockchain. Try getting the latest block height:\n\n\n```\n>>> secret.tendermint.block_info()[\'block\'][\'header\'][\'height\']\n```\n\n`\'1687543\'`\n\n\n### Async Usage\n\nIf you want to make asynchronous, non-blocking LCD requests, you can use AsyncLCDClient. The interface is similar to LCDClient, except the module and wallet API functions must be awaited.\n\n\n<pre><code>\n>>> import asyncio \n>>> from secret_sdk.client.lcd import AsyncLCDClient\n\n>>> async def main():\n        <strong>async with AsyncLCDClient(url=node_rest_endpoint, chain_id="secret-4") as secret:</strong>\n            community_pool = await secret.distribution.community_pool()\n            print(community_pool)\n            <strong>await secret.session.close()  # you must close the session</strong>\n\n>>> asyncio.get_event_loop().run_until_complete(main())\n</code></pre>\n\n## Building and Signing Transactions\n\nIf you wish to perform a state-changing operation on the Secret blockchain such as sending tokens, swapping assets, withdrawing rewards, or even invoking functions on smart contracts, you must create a **transaction** and broadcast it to the network.\nSecret SDK provides functions that help create StdTx objects.\n\n### Example Using a Wallet (*recommended*)\n\nA `Wallet` allows you to create and sign a transaction in a single step by automatically fetching the latest information from the blockchain (chain ID, account number, sequence).\n\nUse `LCDClient.wallet()` to create a Wallet from any Key instance. The Key provided should correspond to the account you intend to sign the transaction with.\n\n\n```\n>>> from secret_sdk.client.lcd import LCDClient\n>>> from secret_sdk.key.mnemonic import MnemonicKey\n\n>>> mk = MnemonicKey(mnemonic=MNEMONIC) \n>>> secret = LCDClient(node_rest_endpoint, "secret-4")\n>>> wallet = secret.wallet(mk)\n```\n\nOnce you have your Wallet, you can create a StdTx using `Wallet.create_and_sign_tx` then broadcast it to the network with `secret.tx.broadcast` with your broadcast mode of choice (block, sync, async - see cosmos docs).\n\n```\n>>> from secret_sdk.core.auth import StdFee\n>>> from secret_sdk.core.bank import MsgSend\n\n>>> send_msg = MsgSend(\n            wallet.key.acc_address,\n            RECIPIENT,\n            "1000000uscrt"    # send 1 scrt\n        )\n>>> tx = wallet.create_and_sign_tx(\n        msgs=[send_msg],\n        memo="My first transaction!",\n        fee=StdFee(200000, "120000uscrt")\n    )\n>>> result = secret.tx.broadcast(tx)\n>>> print(result)\n```\n\nOr use the abstraction `wallet.send_tokens` (see `wallet.execute_tx` to execute a smart contract with `handle_msg`).\n\n```\n>>> tx = wallet.send_tokens(wallet.key.acc_address, RECIPIENT, "1000000uscrt")\n```\n\n<br/>\n\n# Contributing\n\nCommunity contribution, whether it\'s a new feature, correction, bug report, additional documentation, or any other feedback is always welcome. Please read through this section to ensure that your contribution is in the most suitable format for us to effectively process.\n\n<br/>\n\n## Reporting an Issue \nFirst things first: **Do NOT report security vulnerabilities in public issues!** Please disclose responsibly by letting the Secret SDK team know upfront (discord , telegram). We will assess the issue as soon as possible on a best-effort basis and will give you an estimate for when we have a fix and release available for an eventual public disclosure. </br>\nIf you encounter a different issue with the Python SDK, check first to see if there is an existing issue on the Issues page or a pull request on the Pull request page (both Open and Closed tabs) addressing the topic.\n\nIf there isn\'t a discussion on the topic there, you can file an issue. The ideal report includes:\n\n* A description of the problem / suggestion.\n* How to recreate the bug.\n* If relevant, including the versions of your:\n    * Python interpreter\n    * Secret SDK\n    * Optionally of the other dependencies involved\n* If possible, create a pull request with a (failing) test case demonstrating what\'s wrong. This makes the process for fixing bugs quicker & gets issues resolved sooner.\n</br>\n\n## Requesting a Feature\nIf you wish to request the addition of a feature, please first checkout the Issues page and the Pull requests page (both Open and Closed tabs). If you decide to continue with the request, think of the merits of the feature to convince the project\'s developers, and provide as much detail and context as possible in the form of filing an issue on the Issues page.\n\n\n<br/>\n\n## Contributing Code\nIf you wish to contribute to the repository in the form of patches, improvements, new features, etc., first scale the contribution. If it is a major development, like implementing a feature, it is recommended that you consult with the developers of the project before starting the development in order not to risk spending a lot of time working on a change that might not get merged into the project. Once confirmed, you are welcome to submit your pull request.\n</br>\n\n### For new contributors, here is a quick guide: \n\n1. Fork the repository.\n2. Build the project using the [Dependencies](#dependencies) and [Tests](#tests) steps.\n3. Install a <a href="https://virtualenv.pypa.io/en/latest/index.html">virtualenv</a>.\n4. Develop your code and test the changes using the [Tests](#tests) and [Code Quality](#code-quality) steps.\n5. Commit your changes\n6. Push your fork and submit a pull request to the repository\'s `main` branch to propose your code.\n   \n\nA good pull request:\n* is clear.\n* works across all supported versions of Python. (3.7+)\n* Follows the existing style of the code base (<a href="https://pypi.org/project/flake8/">`Flake8`</a>).\n* Has comments included as needed.\n* A test case that demonstrates the previous flaw that now passes with the included patch, or demonstrates the newly added feature.\n* If it adds / changes a public API, it must also include documentation for those changes.\n* Must be appropriately licensed (MIT License).\n</br>\n\n## Documentation Contributions\nDocumentation improvements are always welcome. The documentation files live in the [docs](./docs) directory of the repository and are written in <a href="https://docutils.sourceforge.io/rst.html">reStructuredText</a> and use <a href="https://www.sphinx-doc.org/en/master/">Sphinx</a> to create the full suite of documentation.\n</br>\nWhen contributing documentation, please do your best to follow the style of the documentation files. This means a soft-limit of 88 characters wide in your text files and a semi-formal, yet friendly and approachable, prose style. You can propose your imporvements by submiting a pull request as explained above.\n\n### Need more information on how to contribute?\nYou can give this <a href="https://opensource.guide/how-to-contribute/#how-to-submit-a-contribution">guide</a> read for more insight.\n\n\n<br/>\n\n# License\n\nThis software is licensed under the MIT license. See [LICENSE](./LICENSE) for full disclosure.\n\n\n',
    'author': 'SCRT LabRador, Secret analytics',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/stephanegg/secret-sdk-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
