# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['plotille']

package_data = \
{'': ['*']}

install_requires = \
['six>=1.12,<2.0']

setup_kwargs = {
    'name': 'plotille',
    'version': '3.7.1',
    'description': 'Plot in the terminal using braille dots.',
    'long_description': "# Plotille\n\n[![CircleCI](https://circleci.com/gh/tammoippen/plotille.svg?style=svg)](https://circleci.com/gh/tammoippen/plotille)\n[![Coverage Status](https://coveralls.io/repos/github/tammoippen/plotille/badge.svg?branch=master)](https://coveralls.io/github/tammoippen/plotille?branch=master)\n[![Language grade: Python](https://img.shields.io/lgtm/grade/python/g/tammoippen/plotille.svg?logo=lgtm&logoWidth=18)](https://lgtm.com/projects/g/tammoippen/plotille/context:python)\n[![Tested CPython Versions](https://img.shields.io/badge/cpython-2.7%2C%203.5%2C%203.6%2C%203.7%2C%203.8-brightgreen.svg)](https://img.shields.io/badge/cpython-2.7%2C%203.5%2C%203.6%2C%203.7%2C%203.8-brightgreen.svg)\n[![Tested PyPy Versions](https://img.shields.io/badge/pypy-2.7--7.3%2C%203.5--7.0.0%2C%203.6--7.3-brightgreen.svg)](https://img.shields.io/badge/pypy-2.7--7.3%2C%203.5--7.0.0%2C%203.6--7.3-brightgreen.svg)\n[![PyPi version](https://img.shields.io/pypi/v/plotille.svg)](https://pypi.python.org/pypi/plotille)\n[![PyPi license](https://img.shields.io/pypi/l/plotille.svg)](https://pypi.python.org/pypi/plotille)\n\nPlot, scatter plots and histograms in the terminal using braille dots, with (almost) no dependancies. Plot with color or make complex figures - similar to a very small sibling to matplotlib. Or use the canvas to plot dots and lines yourself.\n\nInstall:\n\n```sh\npip install plotille\n```\n\nSimilar to other libraries:\n\n* like [drawille](https://github.com/asciimoo/drawille), but focused on graphing – plus X/Y-axis.\n* like [termplot](https://github.com/justnoise/termplot), but with braille (finer dots), left to right histogram and linear interpolation for plotting function.\n* like [termgraph](https://github.com/sgeisler/termgraph) (not on pypi), but very different style.\n* like [terminalplot](https://github.com/kressi/terminalplot), but with braille, X/Y-axis, histogram, linear interpolation.\n\nBasic support for timeseries plotting is provided with release 3.2: for any `X` or `Y` values you can also add `datetime.datetime`, `pendulum.datetime` or `numpy.datetime64` values. Labels are generated respecting the difference of `x_limits` and `y_limits`.\n\n## Documentation\n\n```python\nIn [1]: import plotille\nIn [2]: import numpy as np\nIn [3]: X = np.sort(np.random.normal(size=1000))\n```\n\n### Figure\n\nTo construct plots the recomended way is to use a `Figure`:\n\n```python\nIn [12]: plotille.Figure?\nInit signature: plotille.Figure()\nDocstring:\nFigure class to compose multiple plots.\n\nWithin a Figure you can easily compose many plots, assign labels to plots\nand define the properties of the underlying Canvas. Possible properties that\ncan be defined are:\n\n    width, height: int    Define the number of characters in X / Y direction\n                          which are used for plotting.\n    x_limits: float       Define the X limits of the reference coordinate system,\n                          that will be plottered.\n    y_limits: float       Define the Y limits of the reference coordinate system,\n                          that will be plottered.\n    color_mode: str       Define the used color mode. See `plotille.color()`.\n    with_colors: bool     Define, whether to use colors at all.\n    background: multiple  Define the background color.\n    x_label, y_label: str Define the X / Y axis label.\n```\n\nBasically, you create a `Figure`, define the properties and add your plots. Using the `show()` function, the `Figure` generates the plot using a new canvas:\n\n```python\nIn [13] fig = plotille.Figure()\nIn [14] fig.width = 60\nIn [15] fig.height = 30\nIn [16] fig.set_x_limits(min_=-3, max_=3)\nIn [17] fig.set_y_limits(min_=-1, max_=1)\nIn [18] fig.color_mode = 'byte'\nIn [19] fig.plot([-0.5, 1], [-1, 1], lc=25, label='First line')\nIn [20] fig.scatter(X, np.sin(X), lc=100, label='sin')\nIn [21] fig.plot(X, (X+2)**2 , lc=200, label='square')\nIn [22] print(fig.show(legend=True))\n```\n![](https://github.com/tammoippen/plotille/raw/master/imgs/figure.png)\n\nThe available plotting functions are:\n```python\n# create a plot with linear interpolation between points\nFigure.plot(self, X, Y, lc=None, interp='linear', label=None)\n# create a scatter plot with no interpolation between points\nFigure.scatter(self, X, Y, lc=None, label=None)\n# create a histogram over X\nFigure.histogram(self, X, bins=160, lc=None)\n```\n\nOther interesting functions are:\n```python\n# remove all plots from the figure\nFigure.clear(self)\n# Create a canvas, plot the registered plots and return the string for displaying the plot\nFigure.show(self, legend=False)\n```\n\n### Graphing:\n\nThere are some utility functions for fast graphing of single plots.\n\n#### Plot:\n```python\nIn [4]: plotille.plot?\nSignature:\nplt.plot(\n    X,\n    Y,\n    width=80,\n    height=40,\n    X_label='X',\n    Y_label='Y',\n    linesep='\\n',\n    interp='linear',\n    x_min=None,\n    x_max=None,\n    y_min=None,\n    y_max=None,\n    lc=None,\n    bg=None,\n    color_mode='names',\n    origin=True,\n)\nDocstring:\nCreate plot with X , Y values and linear interpolation between points\n\nParameters:\n    X: List[float]         X values.\n    Y: List[float]         Y values. X and Y must have the same number of entries.\n    width: int             The number of characters for the width (columns) of the canvas.\n    hight: int             The number of characters for the hight (rows) of the canvas.\n    X_label: str           Label for X-axis.\n    Y_label: str           Label for Y-axis. max 8 characters.\n    linesep: str           The requested line seperator. default: os.linesep\n    interp: Optional[str]  Specify interpolation; values None, 'linear'\n    x_min, x_max: float    Limits for the displayed X values.\n    y_min, y_max: float    Limits for the displayed Y values.\n    lc: multiple           Give the line color.\n    bg: multiple           Give the background color.\n    color_mode: str        Specify color input mode; 'names' (default), 'byte' or 'rgb'\n                           see plotille.color.__docs__\n    origin: bool           Whether to print the origin. default: True\n\nReturns:\n    str: plot over `X`, `Y`.\n\nIn [5]: print(plotille.plot(X, np.sin(X), height=30, width=60))\n```\n![](https://github.com/tammoippen/plotille/raw/master/imgs/plot.png)\n\n#### Scatter:\n```python\nIn [6]: plotille.scatter?\nSignature:\nplt.scatter(\n    X,\n    Y,\n    width=80,\n    height=40,\n    X_label='X',\n    Y_label='Y',\n    linesep='\\n',\n    x_min=None,\n    x_max=None,\n    y_min=None,\n    y_max=None,\n    lc=None,\n    bg=None,\n    color_mode='names',\n    origin=True,\n)\nDocstring:\nCreate scatter plot with X , Y values\n\nBasically plotting without interpolation:\n    `plot(X, Y, ... , interp=None)`\n\nParameters:\n    X: List[float]       X values.\n    Y: List[float]       Y values. X and Y must have the same number of entries.\n    width: int           The number of characters for the width (columns) of the canvas.\n    hight: int           The number of characters for the hight (rows) of the canvas.\n    X_label: str         Label for X-axis.\n    Y_label: str         Label for Y-axis. max 8 characters.\n    linesep: str         The requested line seperator. default: os.linesep\n    x_min, x_max: float  Limits for the displayed X values.\n    y_min, y_max: float  Limits for the displayed Y values.\n    lc: multiple         Give the line color.\n    bg: multiple         Give the background color.\n    color_mode: str      Specify color input mode; 'names' (default), 'byte' or 'rgb'\n                         see plotille.color.__docs__\n    origin: bool         Whether to print the origin. default: True\n\nReturns:\n    str: scatter plot over `X`, `Y`.\n\nIn [7]: print(plotille.scatter(X, np.sin(X), height=30, width=60))\n```\n![](https://github.com/tammoippen/plotille/raw/master/imgs/scatter.png)\n\n#### Hist:\n\nInspired by [crappyhist](http://kevinastraight.x10host.com/2013/12/28/python-histograms-from-the-console/) (link is gone, but I made a [gist](https://gist.github.com/tammoippen/4474e838e969bf177155231ebba52386)).\n```python\nIn [8]: plotille.hist?\nSignature: plotille.hist(X, bins=40, width=80, log_scale=False, linesep='\\n', lc=None, bg=None, color_mode='names')\nDocstring:\nCreate histogram over `X` from left to right\n\nThe values on the left are the center of the bucket, i.e. `(bin[i] + bin[i+1]) / 2`.\nThe values on the right are the total counts of this bucket.\n\nParameters:\n    X: List[float]  The items to count over.\n    bins: int       The number of bins to put X entries in (rows).\n    width: int      The number of characters for the width (columns).\n    log_scale: bool Scale the histogram with `log` function.\n    linesep: str    The requested line seperator. default: os.linesep\n    lc: multiple         Give the line color.\n    bg: multiple         Give the background color.\n    color_mode: str      Specify color input mode; 'names' (default), 'byte' or 'rgb'\n                         see plotille.color.__docs__\n\nReturns:\n    str: histogram over `X` from left to right.\n\nIn [9]: print(plotille.hist(np.random.normal(size=10000)))\n```\n![](https://github.com/tammoippen/plotille/raw/master/imgs/hist.png)\n\n#### Histogram:\n\nThere is also another more 'usual' histogram function available:\n```python\nIn [10]: plotille.histogram?\nSignature: plotille.histogram(X, bins=160, width=80, height=40, X_label='X', Y_label='Counts', linesep='\\n', x_min=None, x_max=None, y_min=None, y_max=None, lc=None, bg=None, color_mode='names')\nDocstring:\nCreate histogram over `X`\n\nIn contrast to `hist`, this is the more `usual` histogram from bottom\nto up. The X-axis represents the values in `X` and the Y-axis is the\ncorresponding frequency.\n\nParameters:\n    X: List[float]  The items to count over.\n    bins: int       The number of bins to put X entries in (columns).\n    height: int     The number of characters for the height (rows).\n    X_label: str    Label for X-axis.\n    Y_label: str    Label for Y-axis. max 8 characters.\n    linesep: str    The requested line seperator. default: os.linesep\n    x_min, x_max: float  Limits for the displayed X values.\n    y_min, y_max: float  Limits for the displayed Y values.\n    lc: multiple         Give the line color.\n    bg: multiple         Give the background color.\n    color_mode: str      Specify color input mode; 'names' (default), 'byte' or 'rgb'\n                         see plotille.color.__docs__\n\nReturns:\n    str: histogram over `X`.\n\nIn [11]: print(plotille.histogram(np.random.normal(size=10000)))\n```\n![](https://github.com/tammoippen/plotille/raw/master/imgs/histogram.png)\n\n### Canvas:\n\nThe underlying plotting area is modeled as the `Canvas` class:\n```python\nIn [12]: plotille.Canvas?\nInit signature: plotille.Canvas(width, height, xmin=0, ymin=0, xmax=1, ymax=1, background=None, color_mode='names')\nDocstring:\nA canvas object for plotting braille dots\n\nA Canvas object has a `width` x `height` characters large canvas, in which it\ncan plot indivitual braille point, lines out of braille points, rectangles,...\nSince a full braille character has 2 x 4 dots (⣿), the canvas has `width` * 2, `height` * 4\ndots to plot into in total.\n\nIt maintains two coordinate systems: a reference system with the limits (xmin, ymin)\nin the lower left corner to (xmax, ymax) in the upper right corner is transformed\ninto the canvas discrete, i.e. dots, coordinate system (0, 0) to (`width` * 2, `height` * 4).\nIt does so transparently to clients of the Canvas, i.e. all plotting functions\nonly accept coordinates in the reference system. If the coordinates are outside\nthe reference system, they are not plotted.\nInit docstring:\nInitiate a Canvas object\n\nParameters:\n    width: int            The number of characters for the width (columns) of the canvas.\n    hight: int            The number of characters for the hight (rows) of the canvas.\n    xmin, ymin: float     Lower left corner of reference system.\n    xmax, ymax: float     Upper right corner of reference system.\n    background: multiple  Background color of the canvas.\n    color_mode: str       The color-mode for all colors of this canvas; either 'names' (default)\n                          'rgb' or 'byte'. See `plotille.color()`.\n\nReturns:\n    Canvas object\n```\n\nThe most interesting functions are:\n\n*point:*\n```python\nSignature: plotille.Canvas.point(self, x, y, set_=True, color=None)\nDocstring:\nPut a point into the canvas at (x, y) [reference coordinate system]\n\nParameters:\n    x: float         x-coordinate on reference system.\n    y: float         y-coordinate on reference system.\n    set_: bool       Whether to plot or remove the point.\n    color: multiple  Color of the point.\n```\n\n*line:*\n```python\nIn [14]: plotille.Canvas.line?\nSignature: plotille.Canvas.line(self, x0, y0, x1, y1, set_=True, color=None)\nDocstring:\nPlot line between point (x0, y0) and (x1, y1) [reference coordinate system].\n\nParameters:\n    x0, y0: float    Point 0\n    x1, y1: float    Point 1\n    set_: bool       Whether to plot or remove the line.\n    color: multiple  Color of the line.\n```\n\n*rect:*\n```python\nIn [15]: Canvas.rect?\nSignature: plotille.Canvas.rect(self, xmin, ymin, xmax, ymax, set_=True, color=None)\nDocstring:\nPlot rectangle with bbox (xmin, ymin) and (xmax, ymax) [reference coordinate system].\n\nParameters:\n    xmin, ymin: float  Lower left corner of rectangle.\n    xmax, ymax: float  Upper right corner of rectangle.\n    set_: bool         Whether to plot or remove the rect.\n    color: multiple    Color of the rect.\n```\n\n*plot:*\n```python\nIn [16]: Canvas.plot?\nSignature: plotille.Canvas.plot(self, x_axis=False, y_axis=False, y_label='Y', x_label='X', linesep='\\n')\nDocstring:\nTransform canvas into `print`-able string\n\nParameters:\n    x_axis: bool  Add a X-axis at the bottom.\n    y_axis: bool  Add a Y-axis to the left.\n    y_label: str  Label for Y-axis. max 8 characters.\n    x_label: str  Label for X-axis.\n    linesep: str  The requested line seperator. default: os.linesep\n\nReturns:\n    unicode: The cancas as a string.\n```\n\nYou can use it for example to plot a house in the terminal:\n```python\nIn [17]: c = Canvas(width=40, height=20)\nIn [18]: c.rect(0.1, 0.1, 0.6, 0.6)\nIn [19]: c.line(0.1, 0.1, 0.6, 0.6)\nIn [20]: c.line(0.1, 0.6, 0.6, 0.1)\nIn [21]: c.line(0.1, 0.6, 0.35, 0.8)\nIn [22]: c.line(0.35, 0.8, 0.6, 0.6)\nIn [23]: print(c.plot())\n```\n![](https://github.com/tammoippen/plotille/raw/master/imgs/house.png)\n\n\n## Stargazers over time\n\n[![Stargazers over time](https://starchart.cc/tammoippen/plotille.svg)](https://starchart.cc/tammoippen/plotille)\n",
    'author': 'Tammo Ippen',
    'author_email': 'tammo.ippen@posteo.de',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/tammoippen/plotille',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=2.7, !=3.0.*, !=3.1.*, !=3.2.*, !=3.3.*, !=3.4.*',
}


setup(**setup_kwargs)
