# RepRepBuild is the build tool for Reproducible Reporting.
# Copyright (C) 2023 Toon Verstraelen
#
# This file is part of RepRepBuild.
#
# RepRepBuild is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 3
# of the License, or (at your option) any later version.
#
# RepRepBuild is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, see <http://www.gnu.org/licenses/>
#
# --
"""Run BibTex on a LaTeX aux file."""


import argparse
import os
import subprocess
import sys

from .utils import write_dep


def main():
    """Main program."""
    args = parse_args()
    return run_bibtex(args.path_aux)


def parse_args():
    """Parse command-line arguments."""
    parser = argparse.ArgumentParser("rr-bibtex", description="Run bibtex and bibsane")
    parser.add_argument("path_aux", help="The top-level aux file.")
    return parser.parse_args()


def run_bibtex(path_aux):
    """Process the `.first.aux file`, run bibtex and bibsane."""
    if not path_aux.endswith(".first.aux"):
        print(f"BibTeX input must have a `.first.aux` extension. Got {path_aux}")
        return 2
    workdir, fn_aux = os.path.split(path_aux)
    prefix = fn_aux[:-10]

    args = ["bibtex", fn_aux]
    path_blg = os.path.join(workdir, prefix + ".blg")
    cp = subprocess.run(
        args,
        cwd=workdir,
        stdin=subprocess.DEVNULL,
        stdout=subprocess.DEVNULL,
        stderr=subprocess.DEVNULL,
    )
    if cp.returncode != 0:
        print(f"    Error running `bibtex {fn_aux}` in `{workdir}`.")
        with open(path_blg) as f:
            for line in f:
                print(line[:-1])
                if line.startswith("You've used "):
                    break
        return 1

    args = ["bibsane", fn_aux, "--config=../bibsane.yaml"]
    cp = subprocess.run(
        args,
        cwd=workdir,
        text=True,
        stdin=subprocess.DEVNULL,
        stdout=subprocess.PIPE,
        stderr=subprocess.STDOUT,
    )
    if cp.returncode != 0:
        print(f"    Error running `bibsane {fn_aux}` in `{workdir}`.")
        sys.stdout.write(cp.stdout)
        return 1

    # Rename files to those without the `first` interjection.
    for ext in "blg", "bbl":
        os.rename(
            os.path.join(workdir, f"{prefix}.first.{ext}"),
            os.path.join(workdir, f"{prefix}.{ext}"),
        )

    # Parse the blg file to get a list of used bib files.
    paths_bib = set()
    with open(path_blg) as f:
        for line in f:
            if line.startswith("Database file #"):
                paths_bib.add(os.path.join(workdir, line.split()[-1]))

    # Discard the bib files generated by LaTeX.
    path_fls = os.path.join(workdir, prefix + ".fls")
    with open(path_fls) as f:
        for line in f:
            if line.startswith("OUTPUT "):
                paths_bib.discard(os.path.join(workdir, os.path.normpath(line[7:].strip())))

    # Store the input bib files for dependency tracking
    path_bbl = os.path.join(workdir, prefix + ".bbl")
    path_dep = path_aux + ".d"
    write_dep(path_dep, [path_bbl], paths_bib)

    return 0


if __name__ == "__main__":
    main()
