'''
# Amazon EC2 Construct Library

<!--BEGIN STABILITY BANNER-->---


![cfn-resources: Stable](https://img.shields.io/badge/cfn--resources-stable-success.svg?style=for-the-badge)

![cdk-constructs: Stable](https://img.shields.io/badge/cdk--constructs-stable-success.svg?style=for-the-badge)

---
<!--END STABILITY BANNER-->

The `@aws-cdk/aws-ec2` package contains primitives for setting up networking and
instances.

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
import aws_cdk.aws_ec2 as ec2
```

## VPC

Most projects need a Virtual Private Cloud to provide security by means of
network partitioning. This is achieved by creating an instance of
`Vpc`:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
vpc = ec2.Vpc(self, "VPC")
```

All default constructs require EC2 instances to be launched inside a VPC, so
you should generally start by defining a VPC whenever you need to launch
instances for your project.

### Subnet Types

A VPC consists of one or more subnets that instances can be placed into. CDK
distinguishes three different subnet types:

* **Public** - public subnets connect directly to the Internet using an
  Internet Gateway. If you want your instances to have a public IP address
  and be directly reachable from the Internet, you must place them in a
  public subnet.
* **Private** - instances in private subnets are not directly routable from the
  Internet, and connect out to the Internet via a NAT gateway. By default, a
  NAT gateway is created in every public subnet for maximum availability. Be
  aware that you will be charged for NAT gateways.
* **Isolated** - isolated subnets do not route from or to the Internet, and
  as such do not require NAT gateways. They can only connect to or be
  connected to from other instances in the same VPC. A default VPC configuration
  will not include isolated subnets,

A default VPC configuration will create public and **private** subnets. However, if
`natGateways:0` **and** `subnetConfiguration` is undefined, default VPC configuration
will create public and **isolated** subnets. See [*Advanced Subnet Configuration*](#advanced-subnet-configuration)
below for information on how to change the default subnet configuration.

Constructs using the VPC will "launch instances" (or more accurately, create
Elastic Network Interfaces) into one or more of the subnets. They all accept
a property called `subnetSelection` (sometimes called `vpcSubnets`) to allow
you to select in what subnet to place the ENIs, usually defaulting to
*private* subnets if the property is omitted.

If you would like to save on the cost of NAT gateways, you can use
*isolated* subnets instead of *private* subnets (as described in Advanced
*Subnet Configuration*). If you need private instances to have
internet connectivity, another option is to reduce the number of NAT gateways
created by setting the `natGateways` property to a lower value (the default
is one NAT gateway per availability zone). Be aware that this may have
availability implications for your application.

[Read more about
subnets](https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/VPC_Subnets.html).

### Control over availability zones

By default, a VPC will spread over at most 3 Availability Zones available to
it. To change the number of Availability Zones that the VPC will spread over,
specify the `maxAzs` property when defining it.

The number of Availability Zones that are available depends on the *region*
and *account* of the Stack containing the VPC. If the [region and account are
specified](https://docs.aws.amazon.com/cdk/latest/guide/environments.html) on
the Stack, the CLI will [look up the existing Availability
Zones](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-regions-availability-zones.html#using-regions-availability-zones-describe)
and get an accurate count. If region and account are not specified, the stack
could be deployed anywhere and it will have to make a safe choice, limiting
itself to 2 Availability Zones.

Therefore, to get the VPC to spread over 3 or more availability zones, you
must specify the environment where the stack will be deployed.

You can gain full control over the availability zones selection strategy by overriding the Stack's [`get availabilityZones()`](https://github.com/aws/aws-cdk/blob/master/packages/@aws-cdk/core/lib/stack.ts) method:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
class MyStack(Stack): get availabilityZones(): string[] {
        return ['us-west-2a', 'us-west-2b'];
      }

    def __init__(self, scope, id, *, description=None, env=None, stackName=None, tags=None, synthesizer=None, terminationProtection=None, analyticsReporting=None):
        super().__init__(scope, id, description=description, env=env, stackName=stackName, tags=tags, synthesizer=synthesizer, terminationProtection=terminationProtection, analyticsReporting=analyticsReporting)
```

Note that overriding the `get availabilityZones()` method will override the default behavior for all constructs defined within the Stack.

### Choosing subnets for resources

When creating resources that create Elastic Network Interfaces (such as
databases or instances), there is an option to choose which subnets to place
them in. For example, a VPC endpoint by default is placed into a subnet in
every availability zone, but you can override which subnets to use. The property
is typically called one of `subnets`, `vpcSubnets` or `subnetSelection`.

The example below will place the endpoint into two AZs (`us-east-1a` and `us-east-1c`),
in Isolated subnets:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
InterfaceVpcEndpoint(stack, "VPC Endpoint",
    vpc=vpc,
    service=InterfaceVpcEndpointService("com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc", 443),
    subnets={
        "subnet_type": SubnetType.ISOLATED,
        "availability_zones": ["us-east-1a", "us-east-1c"]
    }
)
```

You can also specify specific subnet objects for granular control:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
InterfaceVpcEndpoint(stack, "VPC Endpoint",
    vpc=vpc,
    service=InterfaceVpcEndpointService("com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc", 443),
    subnets={
        "subnets": [subnet1, subnet2]
    }
)
```

Which subnets are selected is evaluated as follows:

* `subnets`: if specific subnet objects are supplied, these are selected, and no other
  logic is used.
* `subnetType`/`subnetGroupName`: otherwise, a set of subnets is selected by
  supplying either type or name:

  * `subnetType` will select all subnets of the given type.
  * `subnetGroupName` should be used to distinguish between multiple groups of subnets of
    the same type (for example, you may want to separate your application instances and your
    RDS instances into two distinct groups of Isolated subnets).
  * If neither are given, the first available subnet group of a given type that
    exists in the VPC will be used, in this order: Private, then Isolated, then Public.
    In short: by default ENIs will preferentially be placed in subnets not connected to
    the Internet.
* `availabilityZones`/`onePerAz`: finally, some availability-zone based filtering may be done.
  This filtering by availability zones will only be possible if the VPC has been created or
  looked up in a non-environment agnostic stack (so account and region have been set and
  availability zones have been looked up).

  * `availabilityZones`: only the specific subnets from the selected subnet groups that are
    in the given availability zones will be returned.
  * `onePerAz`: per availability zone, a maximum of one subnet will be returned (Useful for resource
    types that do not allow creating two ENIs in the same availability zone).
* `subnetFilters`: additional filtering on subnets using any number of user-provided filters which
  extend the SubnetFilter class.

### Using NAT instances

By default, the `Vpc` construct will create NAT *gateways* for you, which
are managed by AWS. If you would prefer to use your own managed NAT
*instances* instead, specify a different value for the `natGatewayProvider`
property, as follows:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
# Configure the `natGatewayProvider` when defining a Vpc
nat_gateway_provider = ec2.NatProvider.instance(
    instance_type=ec2.InstanceType("t3.small")
)

vpc = ec2.Vpc(self, "MyVpc",
    nat_gateway_provider=nat_gateway_provider,

    # The 'natGateways' parameter now controls the number of NAT instances
    nat_gateways=2
)
```

The construct will automatically search for the most recent NAT gateway AMI.
If you prefer to use a custom AMI, use `machineImage: MachineImage.genericLinux({ ... })` and configure the right AMI ID for the
regions you want to deploy to.

By default, the NAT instances will route all traffic. To control what traffic
gets routed, pass `allowAllTraffic: false` and access the
`NatInstanceProvider.connections` member after having passed it to the VPC:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
provider = NatProvider.instance(
    instance_type=,
    allow_all_traffic=False
)
Vpc(stack, "TheVPC",
    nat_gateway_provider=provider
)
provider.connections.allow_from(Peer.ipv4("1.2.3.4/8"), Port.tcp(80))
```

### Advanced Subnet Configuration

If the default VPC configuration (public and private subnets spanning the
size of the VPC) don't suffice for you, you can configure what subnets to
create by specifying the `subnetConfiguration` property. It allows you
to configure the number and size of all subnets. Specifying an advanced
subnet configuration could look like this:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
vpc = ec2.Vpc(self, "TheVPC",
    # 'cidr' configures the IP range and size of the entire VPC.
    # The IP space will be divided over the configured subnets.
    cidr="10.0.0.0/21",

    # 'maxAzs' configures the maximum number of availability zones to use
    max_azs=3,

    # 'subnetConfiguration' specifies the "subnet groups" to create.
    # Every subnet group will have a subnet for each AZ, so this
    # configuration will create `3 groups × 3 AZs = 9` subnets.
    subnet_configuration=[{
        # 'subnetType' controls Internet access, as described above.
        "subnet_type": ec2.SubnetType.PUBLIC,

        # 'name' is used to name this particular subnet group. You will have to
        # use the name for subnet selection if you have more than one subnet
        # group of the same type.
        "name": "Ingress",

        # 'cidrMask' specifies the IP addresses in the range of of individual
        # subnets in the group. Each of the subnets in this group will contain
        # `2^(32 address bits - 24 subnet bits) - 2 reserved addresses = 254`
        # usable IP addresses.
        #
        # If 'cidrMask' is left out the available address space is evenly
        # divided across the remaining subnet groups.
        "cidr_mask": 24
    }, {
        "cidr_mask": 24,
        "name": "Application",
        "subnet_type": ec2.SubnetType.PRIVATE
    }, {
        "cidr_mask": 28,
        "name": "Database",
        "subnet_type": ec2.SubnetType.ISOLATED,

        # 'reserved' can be used to reserve IP address space. No resources will
        # be created for this subnet, but the IP range will be kept available for
        # future creation of this subnet, or even for future subdivision.
        "reserved": True
    }
    ]
)
```

The example above is one possible configuration, but the user can use the
constructs above to implement many other network configurations.

The `Vpc` from the above configuration in a Region with three
availability zones will be the following:

Subnet Name       |Type      |IP Block      |AZ|Features
------------------|----------|--------------|--|--------
IngressSubnet1    |`PUBLIC`  |`10.0.0.0/24` |#1|NAT Gateway
IngressSubnet2    |`PUBLIC`  |`10.0.1.0/24` |#2|NAT Gateway
IngressSubnet3    |`PUBLIC`  |`10.0.2.0/24` |#3|NAT Gateway
ApplicationSubnet1|`PRIVATE` |`10.0.3.0/24` |#1|Route to NAT in IngressSubnet1
ApplicationSubnet2|`PRIVATE` |`10.0.4.0/24` |#2|Route to NAT in IngressSubnet2
ApplicationSubnet3|`PRIVATE` |`10.0.5.0/24` |#3|Route to NAT in IngressSubnet3
DatabaseSubnet1   |`ISOLATED`|`10.0.6.0/28` |#1|Only routes within the VPC
DatabaseSubnet2   |`ISOLATED`|`10.0.6.16/28`|#2|Only routes within the VPC
DatabaseSubnet3   |`ISOLATED`|`10.0.6.32/28`|#3|Only routes within the VPC

### Accessing the Internet Gateway

If you need access to the internet gateway, you can get its ID like so:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
igw_id = vpc.internet_gateway_id
```

For a VPC with only `ISOLATED` subnets, this value will be undefined.

This is only supported for VPCs created in the stack - currently you're
unable to get the ID for imported VPCs. To do that you'd have to specifically
look up the Internet Gateway by name, which would require knowing the name
beforehand.

This can be useful for configuring routing using a combination of gateways:
for more information see [Routing](#routing) below.

#### Routing

It's possible to add routes to any subnets using the `addRoute()` method. If for
example you want an isolated subnet to have a static route via the default
Internet Gateway created for the public subnet - perhaps for routing a VPN
connection - you can do so like this:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
vpc = ec2.Vpc(self, "VPC",
    subnet_configuration=[{
        "subnet_type": SubnetType.PUBLIC,
        "name": "Public"
    }, {
        "subnet_type": SubnetType.ISOLATED,
        "name": "Isolated"
    }]
)(vpc.isolatedSubnets[0]).add_route("StaticRoute",
    router_id=vpc.internet_gateway_id,
    router_type=RouterType.GATEWAY,
    destination_cidr_block="8.8.8.8/32"
)
```

*Note that we cast to `Subnet` here because the list of subnets only returns an
`ISubnet`.*

### Reserving subnet IP space

There are situations where the IP space for a subnet or number of subnets
will need to be reserved. This is useful in situations where subnets would
need to be added after the vpc is originally deployed, without causing IP
renumbering for existing subnets. The IP space for a subnet may be reserved
by setting the `reserved` subnetConfiguration property to true, as shown
below:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
vpc = ec2.Vpc(self, "TheVPC",
    nat_gateways=1,
    subnet_configuration=[{
        "cidr_mask": 26,
        "name": "Public",
        "subnet_type": ec2.SubnetType.PUBLIC
    }, {
        "cidr_mask": 26,
        "name": "Application1",
        "subnet_type": ec2.SubnetType.PRIVATE
    }, {
        "cidr_mask": 26,
        "name": "Application2",
        "subnet_type": ec2.SubnetType.PRIVATE,
        "reserved": True
    }, {
        "cidr_mask": 27,
        "name": "Database",
        "subnet_type": ec2.SubnetType.ISOLATED
    }
    ]
)
```

In the example above, the subnet for Application2 is not actually provisioned
but its IP space is still reserved. If in the future this subnet needs to be
provisioned, then the `reserved: true` property should be removed. Reserving
parts of the IP space prevents the other subnets from getting renumbered.

### Sharing VPCs between stacks

If you are creating multiple `Stack`s inside the same CDK application, you
can reuse a VPC defined in one Stack in another by simply passing the VPC
instance around:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
#
# Stack1 creates the VPC
#
class Stack1(cdk.Stack):

    def __init__(self, scope, id, *, description=None, env=None, stackName=None, tags=None, synthesizer=None, terminationProtection=None, analyticsReporting=None):
        super().__init__(scope, id, description=description, env=env, stackName=stackName, tags=tags, synthesizer=synthesizer, terminationProtection=terminationProtection, analyticsReporting=analyticsReporting)

        self.vpc = ec2.Vpc(self, "VPC")

#
# Stack2 consumes the VPC
#
class Stack2(cdk.Stack):
    def __init__(self, scope, id, *, vpc, description=None, env=None, stackName=None, tags=None, synthesizer=None, terminationProtection=None, analyticsReporting=None):
        super().__init__(scope, id, vpc=vpc, description=description, env=env, stackName=stackName, tags=tags, synthesizer=synthesizer, terminationProtection=terminationProtection, analyticsReporting=analyticsReporting)

        # Pass the VPC to a construct that needs it
        ConstructThatTakesAVpc(self, "Construct",
            vpc=vpc
        )

stack1 = Stack1(app, "Stack1")
stack2 = Stack2(app, "Stack2",
    vpc=stack1.vpc
)
```

### Importing an existing VPC

If your VPC is created outside your CDK app, you can use `Vpc.fromLookup()`.
The CDK CLI will search for the specified VPC in the the stack's region and
account, and import the subnet configuration. Looking up can be done by VPC
ID, but more flexibly by searching for a specific tag on the VPC.

Subnet types will be determined from the `aws-cdk:subnet-type` tag on the
subnet if it exists, or the presence of a route to an Internet Gateway
otherwise. Subnet names will be determined from the `aws-cdk:subnet-name` tag
on the subnet if it exists, or will mirror the subnet type otherwise (i.e.
a public subnet will have the name `"Public"`).

The result of the `Vpc.fromLookup()` operation will be written to a file
called `cdk.context.json`. You must commit this file to source control so
that the lookup values are available in non-privileged environments such
as CI build steps, and to ensure your template builds are repeatable.

Here's how `Vpc.fromLookup()` can be used:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
vpc = ec2.Vpc.from_lookup(stack, "VPC",
    # This imports the default VPC but you can also
    # specify a 'vpcName' or 'tags'.
    is_default=True
)
```

`Vpc.fromLookup` is the recommended way to import VPCs. If for whatever
reason you do not want to use the context mechanism to look up a VPC at
synthesis time, you can also use `Vpc.fromVpcAttributes`. This has the
following limitations:

* Every subnet group in the VPC must have a subnet in each availability zone
  (for example, each AZ must have both a public and private subnet). Asymmetric
  VPCs are not supported.
* All VpcId, SubnetId, RouteTableId, ... parameters must either be known at
  synthesis time, or they must come from deploy-time list parameters whose
  deploy-time lengths are known at synthesis time.

Using `Vpc.fromVpcAttributes()` looks like this:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
vpc = ec2.Vpc.from_vpc_attributes(stack, "VPC",
    vpc_id="vpc-1234",
    availability_zones=["us-east-1a", "us-east-1b"],

    # Either pass literals for all IDs
    public_subnet_ids=["s-12345", "s-67890"],

    # OR: import a list of known length
    private_subnet_ids=Fn.import_list_value("PrivateSubnetIds", 2),

    # OR: split an imported string to a list of known length
    isolated_subnet_ids=Fn.split(",", ssm.StringParameter.value_for_string_parameter(stack, "MyParameter"), 2)
)
```

## Allowing Connections

In AWS, all network traffic in and out of **Elastic Network Interfaces** (ENIs)
is controlled by **Security Groups**. You can think of Security Groups as a
firewall with a set of rules. By default, Security Groups allow no incoming
(ingress) traffic and all outgoing (egress) traffic. You can add ingress rules
to them to allow incoming traffic streams. To exert fine-grained control over
egress traffic, set `allowAllOutbound: false` on the `SecurityGroup`, after
which you can add egress traffic rules.

You can manipulate Security Groups directly:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
my_security_group = ec2.SecurityGroup(self, "SecurityGroup",
    vpc=vpc,
    description="Allow ssh access to ec2 instances",
    allow_all_outbound=True
)
my_security_group.add_ingress_rule(ec2.Peer.any_ipv4(), ec2.Port.tcp(22), "allow ssh access from the world")
```

All constructs that create ENIs on your behalf (typically constructs that create
EC2 instances or other VPC-connected resources) will all have security groups
automatically assigned. Those constructs have an attribute called
**connections**, which is an object that makes it convenient to update the
security groups. If you want to allow connections between two constructs that
have security groups, you have to add an **Egress** rule to one Security Group,
and an **Ingress** rule to the other. The connections object will automatically
take care of this for you:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
# Allow connections from anywhere
load_balancer.connections.allow_from_any_ipv4(ec2.Port.tcp(443), "Allow inbound HTTPS")

# The same, but an explicit IP address
load_balancer.connections.allow_from(ec2.Peer.ipv4("1.2.3.4/32"), ec2.Port.tcp(443), "Allow inbound HTTPS")

# Allow connection between AutoScalingGroups
app_fleet.connections.allow_to(db_fleet, ec2.Port.tcp(443), "App can call database")
```

### Connection Peers

There are various classes that implement the connection peer part:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
# Simple connection peers
peer = ec2.Peer.ipv4("10.0.0.0/16")
peer = ec2.Peer.any_ipv4()
peer = ec2.Peer.ipv6("::0/0")
peer = ec2.Peer.any_ipv6()
peer = ec2.Peer.prefix_list("pl-12345")
app_fleet.connections.allow_to(peer, ec2.Port.tcp(443), "Allow outbound HTTPS")
```

Any object that has a security group can itself be used as a connection peer:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
# These automatically create appropriate ingress and egress rules in both security groups
fleet1.connections.allow_to(fleet2, ec2.Port.tcp(80), "Allow between fleets")

app_fleet.connections.allow_from_any_ipv4(ec2.Port.tcp(80), "Allow from load balancer")
```

### Port Ranges

The connections that are allowed are specified by port ranges. A number of classes provide
the connection specifier:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
ec2.Port.tcp(80)
ec2.Port.tcp_range(60000, 65535)
ec2.Port.all_tcp()
ec2.Port.all_traffic()
```

> NOTE: This set is not complete yet; for example, there is no library support for ICMP at the moment.
> However, you can write your own classes to implement those.

### Default Ports

Some Constructs have default ports associated with them. For example, the
listener of a load balancer does (it's the public port), or instances of an
RDS database (it's the port the database is accepting connections on).

If the object you're calling the peering method on has a default port associated with it, you can call
`allowDefaultPortFrom()` and omit the port specifier. If the argument has an associated default port, call
`allowDefaultPortTo()`.

For example:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
# Port implicit in listener
listener.connections.allow_default_port_from_any_ipv4("Allow public")

# Port implicit in peer
app_fleet.connections.allow_default_port_to(rds_database, "Fleet can access database")
```

### Security group rules

By default, security group wills be added inline to the security group in the output cloud formation
template, if applicable.  This includes any static rules by ip address and port range.  This
optimization helps to minimize the size of the template.

In some environments this is not desirable, for example if your security group access is controlled
via tags. You can disable inline rules per security group or globally via the context key
`@aws-cdk/aws-ec2.securityGroupDisableInlineRules`.

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
my_security_group_without_inline_rules = ec2.SecurityGroup(self, "SecurityGroup",
    vpc=vpc,
    description="Allow ssh access to ec2 instances",
    allow_all_outbound=True,
    disable_inline_rules=True
)
# This will add the rule as an external cloud formation construct
my_security_group_without_inline_rules.add_ingress_rule(ec2.Peer.any_ipv4(), ec2.Port.tcp(22), "allow ssh access from the world")
```

## Machine Images (AMIs)

AMIs control the OS that gets launched when you start your EC2 instance. The EC2
library contains constructs to select the AMI you want to use.

Depending on the type of AMI, you select it a different way. Here are some
examples of things you might want to use:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
# Pick the right Amazon Linux edition. All arguments shown are optional
# and will default to these values when omitted.
amzn_linux = ec2.MachineImage.latest_amazon_linux(
    generation=ec2.AmazonLinuxGeneration.AMAZON_LINUX,
    edition=ec2.AmazonLinuxEdition.STANDARD,
    virtualization=ec2.AmazonLinuxVirt.HVM,
    storage=ec2.AmazonLinuxStorage.GENERAL_PURPOSE,
    cpu_type=ec2.AmazonLinuxCpuType.X86_64
)

# Pick a Windows edition to use
windows = ec2.MachineImage.latest_windows(ec2.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE)

# Read AMI id from SSM parameter store
ssm = ec2.MachineImage.from_sSMParameter("/my/ami", ec2.OperatingSystemType.LINUX)

# Look up the most recent image matching a set of AMI filters.
# In this case, look up the NAT instance AMI, by using a wildcard
# in the 'name' field:
nat_ami = ec2.MachineImage.lookup(
    name="amzn-ami-vpc-nat-*",
    owners=["amazon"]
)

# For other custom (Linux) images, instantiate a `GenericLinuxImage` with
# a map giving the AMI to in for each region:
linux = ec2.MachineImage.generic_linux({
    "us-east-1": "ami-97785bed",
    "eu-west-1": "ami-12345678"
})

# For other custom (Windows) images, instantiate a `GenericWindowsImage` with
# a map giving the AMI to in for each region:
generic_windows = ec2.MachineImage.generic_windows({
    "us-east-1": "ami-97785bed",
    "eu-west-1": "ami-12345678"
})
```

> NOTE: The AMIs selected by `MachineImage.lookup()` will be cached in
> `cdk.context.json`, so that your AutoScalingGroup instances aren't replaced while
> you are making unrelated changes to your CDK app.
>
> To query for the latest AMI again, remove the relevant cache entry from
> `cdk.context.json`, or use the `cdk context` command. For more information, see
> [Runtime Context](https://docs.aws.amazon.com/cdk/latest/guide/context.html) in the CDK
> developer guide.
>
> `MachineImage.genericLinux()`, `MachineImage.genericWindows()` will use `CfnMapping` in
> an agnostic stack.

## Special VPC configurations

### VPN connections to a VPC

Create your VPC with VPN connections by specifying the `vpnConnections` props (keys are construct `id`s):

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
vpc = ec2.Vpc(self, "MyVpc",
    vpn_connections={
        "dynamic": {# Dynamic routing (BGP)
            "ip": "1.2.3.4"},
        "static": {# Static routing
            "ip": "4.5.6.7",
            "static_routes": ["192.168.10.0/24", "192.168.20.0/24"
            ]}
    }
)
```

To create a VPC that can accept VPN connections, set `vpnGateway` to `true`:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
vpc = ec2.Vpc(self, "MyVpc",
    vpn_gateway=True
)
```

VPN connections can then be added:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
vpc.add_vpn_connection("Dynamic",
    ip="1.2.3.4"
)
```

By default, routes will be propagated on the route tables associated with the private subnets. If no
private subnets exists, isolated subnets are used. If no isolated subnets exists, public subnets are
used. Use the `Vpc` property `vpnRoutePropagation` to customize this behavior.

VPN connections expose [metrics (cloudwatch.Metric)](https://github.com/aws/aws-cdk/blob/master/packages/%40aws-cdk/aws-cloudwatch/README.md) across all tunnels in the account/region and per connection:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
# Across all tunnels in the account/region
all_data_out = ec2.VpnConnection.metric_all_tunnel_data_out()

# For a specific vpn connection
vpn_connection = vpc.add_vpn_connection("Dynamic",
    ip="1.2.3.4"
)
state = vpn_connection.metric_tunnel_state()
```

### VPC endpoints

A VPC endpoint enables you to privately connect your VPC to supported AWS services and VPC endpoint services powered by PrivateLink without requiring an internet gateway, NAT device, VPN connection, or AWS Direct Connect connection. Instances in your VPC do not require public IP addresses to communicate with resources in the service. Traffic between your VPC and the other service does not leave the Amazon network.

Endpoints are virtual devices. They are horizontally scaled, redundant, and highly available VPC components that allow communication between instances in your VPC and services without imposing availability risks or bandwidth constraints on your network traffic.

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
# Add gateway endpoints when creating the VPC
vpc = ec2.Vpc(self, "MyVpc",
    gateway_endpoints={
        "S3": GatewayVpcEndpointOptions(
            service=ec2.GatewayVpcEndpointAwsService.S3
        )
    }
)

# Alternatively gateway endpoints can be added on the VPC
dynamo_db_endpoint = vpc.add_gateway_endpoint("DynamoDbEndpoint",
    service=ec2.GatewayVpcEndpointAwsService.DYNAMODB
)

# This allows to customize the endpoint policy
dynamo_db_endpoint.add_to_policy(
    iam.PolicyStatement(# Restrict to listing and describing tables
        principals=[iam.AnyPrincipal()],
        actions=["dynamodb:DescribeTable", "dynamodb:ListTables"],
        resources=["*"]))

# Add an interface endpoint
vpc.add_interface_endpoint("EcrDockerEndpoint", {
    "service": ec2.InterfaceVpcEndpointAwsService.ECR_DOCKER
})
```

By default, CDK will place a VPC endpoint in one subnet per AZ. If you wish to override the AZs CDK places the VPC endpoint in,
use the `subnets` parameter as follows:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
InterfaceVpcEndpoint(stack, "VPC Endpoint",
    vpc=vpc,
    service=InterfaceVpcEndpointService("com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc", 443),
    # Choose which availability zones to place the VPC endpoint in, based on
    # available AZs
    subnets={
        "availability_zones": ["us-east-1a", "us-east-1c"]
    }
)
```

Per the [AWS documentation](https://aws.amazon.com/premiumsupport/knowledge-center/interface-endpoint-availability-zone/), not all
VPC endpoint services are available in all AZs. If you specify the parameter `lookupSupportedAzs`, CDK attempts to discover which
AZs an endpoint service is available in, and will ensure the VPC endpoint is not placed in a subnet that doesn't match those AZs.
These AZs will be stored in cdk.context.json.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
InterfaceVpcEndpoint(stack, "VPC Endpoint",
    vpc=vpc,
    service=InterfaceVpcEndpointService("com.amazonaws.vpce.us-east-1.vpce-svc-uuddlrlrbastrtsvc", 443),
    # Choose which availability zones to place the VPC endpoint in, based on
    # available AZs
    lookup_supported_azs=True
)
```

#### Security groups for interface VPC endpoints

By default, interface VPC endpoints create a new security group and traffic is **not**
automatically allowed from the VPC CIDR.

Use the `connections` object to allow traffic to flow to the endpoint:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
my_endpoint.connections.allow_default_port_from_any_ipv4()
```

Alternatively, existing security groups can be used by specifying the `securityGroups` prop.

### VPC endpoint services

A VPC endpoint service enables you to expose a Network Load Balancer(s) as a provider service to consumers, who connect to your service over a VPC endpoint. You can restrict access to your service via allowed principals (anything that extends ArnPrincipal), and require that new connections be manually accepted.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
VpcEndpointService(self, "EndpointService",
    vpc_endpoint_service_load_balancers=[network_load_balancer1, network_load_balancer2],
    acceptance_required=True,
    allowed_principals=[ArnPrincipal("arn:aws:iam::123456789012:root")]
)
```

Endpoint services support private DNS, which makes it easier for clients to connect to your service by automatically setting up DNS in their VPC.
You can enable private DNS on an endpoint service like so:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
from aws_cdk.aws_route53 import VpcEndpointServiceDomainName


VpcEndpointServiceDomainName(stack, "EndpointDomain",
    endpoint_service=vpces,
    domain_name="my-stuff.aws-cdk.dev",
    public_hosted_zone=zone
)
```

Note: The domain name must be owned (registered through Route53) by the account the endpoint service is in, or delegated to the account.
The VpcEndpointServiceDomainName will handle the AWS side of domain verification, the process for which can be found
[here](https://docs.aws.amazon.com/vpc/latest/userguide/endpoint-services-dns-validation.html)

### Client VPN endpoint

AWS Client VPN is a managed client-based VPN service that enables you to securely access your AWS
resources and resources in your on-premises network. With Client VPN, you can access your resources
from any location using an OpenVPN-based VPN client.

Use the `addClientVpnEndpoint()` method to add a client VPN endpoint to a VPC:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
vpc.add_client_vpn_endpoint("Endpoint",
    cidr="10.100.0.0/16",
    server_certificate_arn="arn:aws:acm:us-east-1:123456789012:certificate/server-certificate-id",
    # Mutual authentication
    client_certificate_arn="arn:aws:acm:us-east-1:123456789012:certificate/client-certificate-id",
    # User-based authentication
    user_based_authentication=ec2.ClientVpnUserBasedAuthentication.federated(saml_provider)
)
```

The endpoint must use at least one [authentication method](https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/client-authentication.html):

* Mutual authentication with a client certificate
* User-based authentication (directory or federated)

If user-based authentication is used, the [self-service portal URL](https://docs.aws.amazon.com/vpn/latest/clientvpn-user/self-service-portal.html)
is made available via a CloudFormation output.

By default, a new security group is created and logging is enabled. Moreover, a rule to
authorize all users to the VPC CIDR is created.

To customize authorization rules, set the `authorizeAllUsersToVpcCidr` prop to `false`
and use `addaddAuthorizationRule()`:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
endpoint = vpc.add_client_vpn_endpoint("Endpoint",
    cidr="10.100.0.0/16",
    server_certificate_arn="arn:aws:acm:us-east-1:123456789012:certificate/server-certificate-id",
    user_based_authentication=ec2.ClientVpnUserBasedAuthentication.federated(saml_provider),
    authorize_all_users_to_vpc_cidr=False
)

endpoint.add_authorization_rule("Rule",
    cidr="10.0.10.0/32",
    group_id="group-id"
)
```

Use `addRoute()` to configure network routes:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
endpoint = vpc.add_client_vpn_endpoint("Endpoint",
    cidr="10.100.0.0/16",
    server_certificate_arn="arn:aws:acm:us-east-1:123456789012:certificate/server-certificate-id",
    user_based_authentication=ec2.ClientVpnUserBasedAuthentication.federated(saml_provider)
)

# Client-to-client access
endpoint.add_route("Route",
    cidr="10.100.0.0/16",
    target=ec2.ClientVpnRouteTarget.local()
)
```

Use the `connections` object of the endpoint to allow traffic to other security groups.

## Instances

You can use the `Instance` class to start up a single EC2 instance. For production setups, we recommend
you use an `AutoScalingGroup` from the `aws-autoscaling` module instead, as AutoScalingGroups will take
care of restarting your instance if it ever fails.

### Configuring Instances using CloudFormation Init (cfn-init)

CloudFormation Init allows you to configure your instances by writing files to them, installing software
packages, starting services and running arbitrary commands. By default, if any of the instance setup
commands throw an error, the deployment will fail and roll back to the previously known good state.
The following documentation also applies to `AutoScalingGroup`s.

For the full set of capabilities of this system, see the documentation for
[`AWS::CloudFormation::Init`](https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-init.html).
Here is an example of applying some configuration to an instance:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
ec2.Instance(self, "Instance",
    # Showing the most complex setup, if you have simpler requirements
    # you can use `CloudFormationInit.fromElements()`.
    init=ec2.CloudFormationInit.from_config_sets(
        config_sets={
            # Applies the configs below in this order
            "default": ["yumPreinstall", "config"]
        },
        configs={
            "yum_preinstall": ec2.InitConfig([
                # Install an Amazon Linux package using yum
                ec2.InitPackage.yum("git")
            ]),
            "config": ec2.InitConfig([
                # Create a JSON file from tokens (can also create other files)
                ec2.InitFile.from_object("/etc/stack.json",
                    stack_id=stack.stack_id,
                    stack_name=stack.stack_name,
                    region=stack.region
                ),

                # Create a group and user
                ec2.InitGroup.from_name("my-group"),
                ec2.InitUser.from_name("my-user"),

                # Install an RPM from the internet
                ec2.InitPackage.rpm("http://mirrors.ukfast.co.uk/sites/dl.fedoraproject.org/pub/epel/8/Everything/x86_64/Packages/r/rubygem-git-1.5.0-2.el8.noarch.rpm")
            ])
        }
    ),
    init_options={
        # Optional, which configsets to activate (['default'] by default)
        "config_sets": ["default"],

        # Optional, how long the installation is expected to take (5 minutes by default)
        "timeout": Duration.minutes(30)
    }
)
```

You can have services restarted after the init process has made changes to the system.
To do that, instantiate an `InitServiceRestartHandle` and pass it to the config elements
that need to trigger the restart and the service itself. For example, the following
config writes a config file for nginx, extracts an archive to the root directory, and then
restarts nginx so that it picks up the new config and files:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
handle = ec2.InitServiceRestartHandle()

ec2.CloudFormationInit.from_elements(
    ec2.InitFile.from_string("/etc/nginx/nginx.conf", "...", service_restart_handles=[handle]),
    ec2.InitSource.from_bucket("/var/www/html", my_bucket, "html.zip", service_restart_handles=[handle]),
    ec2.InitService.enable("nginx",
        service_restart_handle=handle
    ))
```

### Bastion Hosts

A bastion host functions as an instance used to access servers and resources in a VPC without open up the complete VPC on a network level.
You can use bastion hosts using a standard SSH connection targeting port 22 on the host. As an alternative, you can connect the SSH connection
feature of AWS Systems Manager Session Manager, which does not need an opened security group. (https://aws.amazon.com/about-aws/whats-new/2019/07/session-manager-launches-tunneling-support-for-ssh-and-scp/)

A default bastion host for use via SSM can be configured like:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
host = ec2.BastionHostLinux(self, "BastionHost", vpc=vpc)
```

If you want to connect from the internet using SSH, you need to place the host into a public subnet. You can then configure allowed source hosts.

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
host = ec2.BastionHostLinux(self, "BastionHost",
    vpc=vpc,
    subnet_selection=SubnetSelection(subnet_type=ec2.SubnetType.PUBLIC)
)
host.allow_ssh_access_from(ec2.Peer.ipv4("1.2.3.4/32"))
```

As there are no SSH public keys deployed on this machine, you need to use [EC2 Instance Connect](https://aws.amazon.com/de/blogs/compute/new-using-amazon-ec2-instance-connect-for-ssh-access-to-your-ec2-instances/)
with the command `aws ec2-instance-connect send-ssh-public-key` to provide your SSH public key.

EBS volume for the bastion host can be encrypted like:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
host = ec2.BastionHostLinux(stack, "BastionHost",
    vpc=vpc,
    block_devices=[{
        "device_name": "EBSBastionHost",
        "volume": BlockDeviceVolume.ebs(10,
            encrypted=True
        )
    }]
)
```

### Block Devices

To add EBS block device mappings, specify the `blockDevices` property. The following example sets the EBS-backed
root device (`/dev/sda1`) size to 50 GiB, and adds another EBS-backed device mapped to `/dev/sdm` that is 100 GiB in
size:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
ec2.Instance(self, "Instance",
    # ...
    block_devices=[{
        "device_name": "/dev/sda1",
        "volume": ec2.BlockDeviceVolume.ebs(50)
    }, {
        "device_name": "/dev/sdm",
        "volume": ec2.BlockDeviceVolume.ebs(100)
    }
    ]
)
```

### Volumes

Whereas a `BlockDeviceVolume` is an EBS volume that is created and destroyed as part of the creation and destruction of a specific instance. A `Volume` is for when you want an EBS volume separate from any particular instance. A `Volume` is an EBS block device that can be attached to, or detached from, any instance at any time. Some types of `Volume`s can also be attached to multiple instances at the same time to allow you to have shared storage between those instances.

A notable restriction is that a Volume can only be attached to instances in the same availability zone as the Volume itself.

The following demonstrates how to create a 500 GiB encrypted Volume in the `us-west-2a` availability zone, and give a role the ability to attach that Volume to a specific instance:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
instance = ec2.Instance(self, "Instance")
role = iam.Role(stack, "SomeRole",
    assumed_by=iam.AccountRootPrincipal()
)
volume = ec2.Volume(self, "Volume",
    availability_zone="us-west-2a",
    size=cdk.Size.gibibytes(500),
    encrypted=True
)

volume.grant_attach_volume(role, [instance])
```

#### Instances Attaching Volumes to Themselves

If you need to grant an instance the ability to attach/detach an EBS volume to/from itself, then using `grantAttachVolume` and `grantDetachVolume` as outlined above
will lead to an unresolvable circular reference between the instance role and the instance. In this case, use `grantAttachVolumeByResourceTag` and `grantDetachVolumeByResourceTag` as follows:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
instance = ec2.Instance(self, "Instance")
volume = ec2.Volume(self, "Volume")

attach_grant = volume.grant_attach_volume_by_resource_tag(instance.grant_principal, [instance])
detach_grant = volume.grant_detach_volume_by_resource_tag(instance.grant_principal, [instance])
```

#### Attaching Volumes

The Amazon EC2 documentation for
[Linux Instances](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/AmazonEBS.html) and
[Windows Instances](https://docs.aws.amazon.com/AWSEC2/latest/WindowsGuide/ebs-volumes.html) contains information on how
to attach and detach your Volumes to/from instances, and how to format them for use.

The following is a sample skeleton of EC2 UserData that can be used to attach a Volume to the Linux instance that it is running on:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
volume = ec2.Volume(self, "Volume")
instance = ec2.Instance(self, "Instance")
volume.grant_attach_volume_by_resource_tag(instance.grant_principal, [instance])
target_device = "/dev/xvdz"
instance.user_data.add_commands("TOKEN=$(curl -SsfX PUT \"http://169.254.169.254/latest/api/token\" -H \"X-aws-ec2-metadata-token-ttl-seconds: 21600\")", "INSTANCE_ID=$(curl -SsfH \"X-aws-ec2-metadata-token: $TOKEN\" http://169.254.169.254/latest/meta-data/instance-id)", f"aws --region {Stack.of(this).region} ec2 attach-volume --volume-id {volume.volumeId} --instance-id $INSTANCE_ID --device {targetDevice}", f"while ! test -e {targetDevice}; do sleep 1; done")
```

## VPC Flow Logs

VPC Flow Logs is a feature that enables you to capture information about the IP traffic going to and from network interfaces in your VPC. Flow log data can be published to Amazon CloudWatch Logs and Amazon S3. After you've created a flow log, you can retrieve and view its data in the chosen destination. ([https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html](https://docs.aws.amazon.com/vpc/latest/userguide/flow-logs.html)).

By default a flow log will be created with CloudWatch Logs as the destination.

You can create a flow log like this:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
ec2.FlowLog(self, "FlowLog",
    resource_type=ec2.FlowLogResourceType.from_vpc(vpc)
)
```

Or you can add a Flow Log to a VPC by using the addFlowLog method like this:

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
vpc = ec2.Vpc(self, "Vpc")

vpc.add_flow_log("FlowLog")
```

You can also add multiple flow logs with different destinations.

```python
# Example automatically generated. See https://github.com/aws/jsii/issues/826
vpc = ec2.Vpc(self, "Vpc")

vpc.add_flow_log("FlowLogS3",
    destination=ec2.FlowLogDestination.to_s3()
)

vpc.add_flow_log("FlowLogCloudWatch",
    traffic_type=ec2.FlowLogTrafficType.REJECT
)
```

By default the CDK will create the necessary resources for the destination. For the CloudWatch Logs destination
it will create a CloudWatch Logs Log Group as well as the IAM role with the necessary permissions to publish to
the log group. In the case of an S3 destination, it will create the S3 bucket.

If you want to customize any of the destination resources you can provide your own as part of the `destination`.

*CloudWatch Logs*

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
log_group = logs.LogGroup(self, "MyCustomLogGroup")

role = iam.Role(self, "MyCustomRole",
    assumed_by=iam.ServicePrincipal("vpc-flow-logs.amazonaws.com")
)

ec2.FlowLog(self, "FlowLog",
    resource_type=ec2.FlowLogResourceType.from_vpc(vpc),
    destination=ec2.FlowLogDestination.to_cloud_watch_logs(log_group, role)
)
```

*S3*

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
bucket = s3.Bucket(self, "MyCustomBucket")

ec2.FlowLog(self, "FlowLog",
    resource_type=ec2.FlowLogResourceType.from_vpc(vpc),
    destination=ec2.FlowLogDestination.to_s3(bucket)
)

ec2.FlowLog(self, "FlowLogWithKeyPrefix",
    resource_type=ec2.FlowLogResourceType.from_vpc(vpc),
    destination=ec2.FlowLogDestination.to_s3(bucket, "prefix/")
)
```

## User Data

User data enables you to run a script when your instances start up.  In order to configure these scripts you can add commands directly to the script
or you can use the UserData's convenience functions to aid in the creation of your script.

A user data could be configured to run a script found in an asset through the following:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
asset = Asset(self, "Asset", path=path.join(__dirname, "configure.sh"))
instance = ec2.Instance(self, "Instance")
local_path = instance.user_data.add_s3_download_command(
    bucket=asset.bucket,
    bucket_key=asset.s3_object_key
)
instance.user_data.add_execute_file_command(
    file_path=local_path,
    arguments="--verbose -y"
)
asset.grant_read(instance.role)
```

### Multipart user data

In addition, to above the `MultipartUserData` can be used to change instance startup behavior. Multipart user data are composed
from separate parts forming archive. The most common parts are scripts executed during instance set-up. However, there are other
kinds, too.

The advantage of multipart archive is in flexibility when it's needed to add additional parts or to use specialized parts to
fine tune instance startup. Some services (like AWS Batch) supports only `MultipartUserData`.

The parts can be executed at different moment of instance start-up and can serve a different purposes. This is controlled by `contentType` property.
For common scripts, `text/x-shellscript; charset="utf-8"` can be used as content type.

In order to create archive the `MultipartUserData` has to be instantiated. Than, user can add parts to multipart archive using `addPart`. The `MultipartBody` contains methods supporting creation of body parts.

If the very custom part is required, it can be created using `MultipartUserData.fromRawBody`, in this case full control over content type,
transfer encoding, and body properties is given to the user.

Below is an example for creating multipart user data with single body part responsible for installing `awscli` and configuring maximum size
of storage used by Docker containers:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
boot_hook_conf = ec2.UserData.for_linux()
boot_hook_conf.add_commands("cloud-init-per once docker_options echo 'OPTIONS=\"${OPTIONS} --storage-opt dm.basesize=40G\"' >> /etc/sysconfig/docker")

setup_commands = ec2.UserData.for_linux()
setup_commands.add_commands("sudo yum install awscli && echo Packages installed らと > /var/tmp/setup")

multipart_user_data = ec2.MultipartUserData()
# The docker has to be configured at early stage, so content type is overridden to boothook
multipart_user_data.add_part(ec2.MultipartBody.from_user_data(boot_hook_conf, "text/cloud-boothook; charset=\"us-ascii\""))
# Execute the rest of setup
multipart_user_data.add_part(ec2.MultipartBody.from_user_data(setup_commands))

ec2.LaunchTemplate(stack, "",
    user_data=multipart_user_data,
    block_devices=[]
)
```

For more information see
[Specifying Multiple User Data Blocks Using a MIME Multi Part Archive](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#multi-part_user_data)

#### Using add*Command on MultipartUserData

To use the `add*Command` methods, that are inherited from the `UserData` interface, on `MultipartUserData` you must add a part
to the `MultipartUserData` and designate it as the reciever for these methods. This is accomplished by using the `addUserDataPart()`
method on `MultipartUserData` with the `makeDefault` argument set to `true`:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
multipart_user_data = ec2.MultipartUserData()
commands_user_data = ec2.UserData.for_linux()
multipart_user_data.add_user_data_part(commands_user_data, MultipartBody.SHELL_SCRIPT, True)

# Adding commands to the multipartUserData adds them to commandsUserData, and vice-versa.
multipart_user_data.add_commands("touch /root/multi.txt")
commands_user_data.add_commands("touch /root/userdata.txt")
```

When used on an EC2 instance, the above `multipartUserData` will create both `multi.txt` and `userdata.txt` in `/root`.

## Importing existing subnet

To import an existing Subnet, call `Subnet.fromSubnetAttributes()` or
`Subnet.fromSubnetId()`. Only if you supply the subnet's Availability Zone
and Route Table Ids when calling `Subnet.fromSubnetAttributes()` will you be
able to use the CDK features that use these values (such as selecting one
subnet per AZ).

Importing an existing subnet looks like this:

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
# Supply all properties
subnet = Subnet.from_subnet_attributes(self, "SubnetFromAttributes",
    subnet_id="s-1234",
    availability_zone="pub-az-4465",
    route_table_id="rt-145"
)

# Supply only subnet id
subnet = Subnet.from_subnet_id(self, "SubnetFromId", "s-1234")
```

## Launch Templates

A Launch Template is a standardized template that contains the configuration information to launch an instance.
They can be used when launching instances on their own, through Amazon EC2 Auto Scaling, EC2 Fleet, and Spot Fleet.
Launch templates enable you to store launch parameters so that you do not have to specify them every time you launch
an instance. For information on Launch Templates please see the
[official documentation](https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html).

The following demonstrates how to create a launch template with an Amazon Machine Image, and security group.

```python
# Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
vpc = ec2.Vpc(...)
# ...
template = ec2.LaunchTemplate(self, "LaunchTemplate",
    machine_image=ec2.AmazonMachineImage(),
    security_group=ec2.SecurityGroup(self, "LaunchTemplateSG",
        vpc=vpc
    )
)
```
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from ._jsii import *

import aws_cdk.assets
import aws_cdk.aws_cloudwatch
import aws_cdk.aws_iam
import aws_cdk.aws_kms
import aws_cdk.aws_logs
import aws_cdk.aws_s3
import aws_cdk.aws_s3_assets
import aws_cdk.core
import constructs


class AclCidr(metaclass=jsii.JSIIAbstractClass, jsii_type="@aws-cdk/aws-ec2.AclCidr"):
    '''Either an IPv4 or an IPv6 CIDR.'''

    def __init__(self) -> None:
        jsii.create(AclCidr, self, [])

    @jsii.member(jsii_name="anyIpv4") # type: ignore[misc]
    @builtins.classmethod
    def any_ipv4(cls) -> "AclCidr":
        '''The CIDR containing all IPv4 addresses (i.e., 0.0.0.0/0).'''
        return typing.cast("AclCidr", jsii.sinvoke(cls, "anyIpv4", []))

    @jsii.member(jsii_name="anyIpv6") # type: ignore[misc]
    @builtins.classmethod
    def any_ipv6(cls) -> "AclCidr":
        '''The CIDR containing all IPv6 addresses (i.e., ::/0).'''
        return typing.cast("AclCidr", jsii.sinvoke(cls, "anyIpv6", []))

    @jsii.member(jsii_name="ipv4") # type: ignore[misc]
    @builtins.classmethod
    def ipv4(cls, ipv4_cidr: builtins.str) -> "AclCidr":
        '''An IP network range in CIDR notation (for example, 172.16.0.0/24).

        :param ipv4_cidr: -
        '''
        return typing.cast("AclCidr", jsii.sinvoke(cls, "ipv4", [ipv4_cidr]))

    @jsii.member(jsii_name="ipv6") # type: ignore[misc]
    @builtins.classmethod
    def ipv6(cls, ipv6_cidr: builtins.str) -> "AclCidr":
        '''An IPv6 network range in CIDR notation (for example, 2001:db8::/48).

        :param ipv6_cidr: -
        '''
        return typing.cast("AclCidr", jsii.sinvoke(cls, "ipv6", [ipv6_cidr]))

    @jsii.member(jsii_name="toCidrConfig") # type: ignore[misc]
    @abc.abstractmethod
    def to_cidr_config(self) -> "AclCidrConfig":
        ...


class _AclCidrProxy(AclCidr):
    @jsii.member(jsii_name="toCidrConfig")
    def to_cidr_config(self) -> "AclCidrConfig":
        return typing.cast("AclCidrConfig", jsii.invoke(self, "toCidrConfig", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AclCidr).__jsii_proxy_class__ = lambda : _AclCidrProxy


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.AclCidrConfig",
    jsii_struct_bases=[],
    name_mapping={"cidr_block": "cidrBlock", "ipv6_cidr_block": "ipv6CidrBlock"},
)
class AclCidrConfig:
    def __init__(
        self,
        *,
        cidr_block: typing.Optional[builtins.str] = None,
        ipv6_cidr_block: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Acl Configuration for CIDR.

        :param cidr_block: Ipv4 CIDR.
        :param ipv6_cidr_block: Ipv6 CIDR.
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if cidr_block is not None:
            self._values["cidr_block"] = cidr_block
        if ipv6_cidr_block is not None:
            self._values["ipv6_cidr_block"] = ipv6_cidr_block

    @builtins.property
    def cidr_block(self) -> typing.Optional[builtins.str]:
        '''Ipv4 CIDR.'''
        result = self._values.get("cidr_block")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ipv6_cidr_block(self) -> typing.Optional[builtins.str]:
        '''Ipv6 CIDR.'''
        result = self._values.get("ipv6_cidr_block")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AclCidrConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.AclIcmp",
    jsii_struct_bases=[],
    name_mapping={"code": "code", "type": "type"},
)
class AclIcmp:
    def __init__(
        self,
        *,
        code: typing.Optional[jsii.Number] = None,
        type: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties to create Icmp.

        :param code: The Internet Control Message Protocol (ICMP) code. You can use -1 to specify all ICMP codes for the given ICMP type. Requirement is conditional: Required if you specify 1 (ICMP) for the protocol parameter.
        :param type: The Internet Control Message Protocol (ICMP) type. You can use -1 to specify all ICMP types. Conditional requirement: Required if you specify 1 (ICMP) for the CreateNetworkAclEntry protocol parameter.
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if code is not None:
            self._values["code"] = code
        if type is not None:
            self._values["type"] = type

    @builtins.property
    def code(self) -> typing.Optional[jsii.Number]:
        '''The Internet Control Message Protocol (ICMP) code.

        You can use -1 to specify all ICMP
        codes for the given ICMP type. Requirement is conditional: Required if you
        specify 1 (ICMP) for the protocol parameter.
        '''
        result = self._values.get("code")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def type(self) -> typing.Optional[jsii.Number]:
        '''The Internet Control Message Protocol (ICMP) type.

        You can use -1 to specify all ICMP types.
        Conditional requirement: Required if you specify 1 (ICMP) for the CreateNetworkAclEntry protocol parameter.
        '''
        result = self._values.get("type")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AclIcmp(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.AclPortRange",
    jsii_struct_bases=[],
    name_mapping={"from_": "from", "to": "to"},
)
class AclPortRange:
    def __init__(
        self,
        *,
        from_: typing.Optional[jsii.Number] = None,
        to: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties to create PortRange.

        :param from_: The first port in the range. Required if you specify 6 (TCP) or 17 (UDP) for the protocol parameter.
        :param to: The last port in the range. Required if you specify 6 (TCP) or 17 (UDP) for the protocol parameter.
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if from_ is not None:
            self._values["from_"] = from_
        if to is not None:
            self._values["to"] = to

    @builtins.property
    def from_(self) -> typing.Optional[jsii.Number]:
        '''The first port in the range.

        Required if you specify 6 (TCP) or 17 (UDP) for the protocol parameter.
        '''
        result = self._values.get("from_")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def to(self) -> typing.Optional[jsii.Number]:
        '''The last port in the range.

        Required if you specify 6 (TCP) or 17 (UDP) for the protocol parameter.
        '''
        result = self._values.get("to")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AclPortRange(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AclTraffic(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@aws-cdk/aws-ec2.AclTraffic",
):
    '''The traffic that is configured using a Network ACL entry.'''

    def __init__(self) -> None:
        jsii.create(AclTraffic, self, [])

    @jsii.member(jsii_name="allTraffic") # type: ignore[misc]
    @builtins.classmethod
    def all_traffic(cls) -> "AclTraffic":
        '''Apply the ACL entry to all traffic.'''
        return typing.cast("AclTraffic", jsii.sinvoke(cls, "allTraffic", []))

    @jsii.member(jsii_name="icmp") # type: ignore[misc]
    @builtins.classmethod
    def icmp(
        cls,
        *,
        code: typing.Optional[jsii.Number] = None,
        type: typing.Optional[jsii.Number] = None,
    ) -> "AclTraffic":
        '''Apply the ACL entry to ICMP traffic of given type and code.

        :param code: The Internet Control Message Protocol (ICMP) code. You can use -1 to specify all ICMP codes for the given ICMP type. Requirement is conditional: Required if you specify 1 (ICMP) for the protocol parameter.
        :param type: The Internet Control Message Protocol (ICMP) type. You can use -1 to specify all ICMP types. Conditional requirement: Required if you specify 1 (ICMP) for the CreateNetworkAclEntry protocol parameter.
        '''
        props = AclIcmp(code=code, type=type)

        return typing.cast("AclTraffic", jsii.sinvoke(cls, "icmp", [props]))

    @jsii.member(jsii_name="icmpv6") # type: ignore[misc]
    @builtins.classmethod
    def icmpv6(
        cls,
        *,
        code: typing.Optional[jsii.Number] = None,
        type: typing.Optional[jsii.Number] = None,
    ) -> "AclTraffic":
        '''Apply the ACL entry to ICMPv6 traffic of given type and code.

        Requires an IPv6 CIDR block.

        :param code: The Internet Control Message Protocol (ICMP) code. You can use -1 to specify all ICMP codes for the given ICMP type. Requirement is conditional: Required if you specify 1 (ICMP) for the protocol parameter.
        :param type: The Internet Control Message Protocol (ICMP) type. You can use -1 to specify all ICMP types. Conditional requirement: Required if you specify 1 (ICMP) for the CreateNetworkAclEntry protocol parameter.
        '''
        props = AclIcmp(code=code, type=type)

        return typing.cast("AclTraffic", jsii.sinvoke(cls, "icmpv6", [props]))

    @jsii.member(jsii_name="tcpPort") # type: ignore[misc]
    @builtins.classmethod
    def tcp_port(cls, port: jsii.Number) -> "AclTraffic":
        '''Apply the ACL entry to TCP traffic on a given port.

        :param port: -
        '''
        return typing.cast("AclTraffic", jsii.sinvoke(cls, "tcpPort", [port]))

    @jsii.member(jsii_name="tcpPortRange") # type: ignore[misc]
    @builtins.classmethod
    def tcp_port_range(
        cls,
        start_port: jsii.Number,
        end_port: jsii.Number,
    ) -> "AclTraffic":
        '''Apply the ACL entry to TCP traffic on a given port range.

        :param start_port: -
        :param end_port: -
        '''
        return typing.cast("AclTraffic", jsii.sinvoke(cls, "tcpPortRange", [start_port, end_port]))

    @jsii.member(jsii_name="udpPort") # type: ignore[misc]
    @builtins.classmethod
    def udp_port(cls, port: jsii.Number) -> "AclTraffic":
        '''Apply the ACL entry to UDP traffic on a given port.

        :param port: -
        '''
        return typing.cast("AclTraffic", jsii.sinvoke(cls, "udpPort", [port]))

    @jsii.member(jsii_name="udpPortRange") # type: ignore[misc]
    @builtins.classmethod
    def udp_port_range(
        cls,
        start_port: jsii.Number,
        end_port: jsii.Number,
    ) -> "AclTraffic":
        '''Apply the ACL entry to UDP traffic on a given port range.

        :param start_port: -
        :param end_port: -
        '''
        return typing.cast("AclTraffic", jsii.sinvoke(cls, "udpPortRange", [start_port, end_port]))

    @jsii.member(jsii_name="toTrafficConfig") # type: ignore[misc]
    @abc.abstractmethod
    def to_traffic_config(self) -> "AclTrafficConfig":
        ...


class _AclTrafficProxy(AclTraffic):
    @jsii.member(jsii_name="toTrafficConfig")
    def to_traffic_config(self) -> "AclTrafficConfig":
        return typing.cast("AclTrafficConfig", jsii.invoke(self, "toTrafficConfig", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AclTraffic).__jsii_proxy_class__ = lambda : _AclTrafficProxy


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.AclTrafficConfig",
    jsii_struct_bases=[],
    name_mapping={"protocol": "protocol", "icmp": "icmp", "port_range": "portRange"},
)
class AclTrafficConfig:
    def __init__(
        self,
        *,
        protocol: jsii.Number,
        icmp: typing.Optional[AclIcmp] = None,
        port_range: typing.Optional[AclPortRange] = None,
    ) -> None:
        '''Acl Configuration for traffic.

        :param protocol: The protocol number. A value of "-1" means all protocols. If you specify "-1" or a protocol number other than "6" (TCP), "17" (UDP), or "1" (ICMP), traffic on all ports is allowed, regardless of any ports or ICMP types or codes that you specify. If you specify protocol "58" (ICMPv6) and specify an IPv4 CIDR block, traffic for all ICMP types and codes allowed, regardless of any that you specify. If you specify protocol "58" (ICMPv6) and specify an IPv6 CIDR block, you must specify an ICMP type and code. Default: 17
        :param icmp: The Internet Control Message Protocol (ICMP) code and type. Default: - Required if specifying 1 (ICMP) for the protocol parameter.
        :param port_range: The range of port numbers for the UDP/TCP protocol. Default: - Required if specifying 6 (TCP) or 17 (UDP) for the protocol parameter
        '''
        if isinstance(icmp, dict):
            icmp = AclIcmp(**icmp)
        if isinstance(port_range, dict):
            port_range = AclPortRange(**port_range)
        self._values: typing.Dict[str, typing.Any] = {
            "protocol": protocol,
        }
        if icmp is not None:
            self._values["icmp"] = icmp
        if port_range is not None:
            self._values["port_range"] = port_range

    @builtins.property
    def protocol(self) -> jsii.Number:
        '''The protocol number.

        A value of "-1" means all protocols.

        If you specify "-1" or a protocol number other than "6" (TCP), "17" (UDP),
        or "1" (ICMP), traffic on all ports is allowed, regardless of any ports or
        ICMP types or codes that you specify.

        If you specify protocol "58" (ICMPv6) and specify an IPv4 CIDR
        block, traffic for all ICMP types and codes allowed, regardless of any that
        you specify. If you specify protocol "58" (ICMPv6) and specify an IPv6 CIDR
        block, you must specify an ICMP type and code.

        :default: 17
        '''
        result = self._values.get("protocol")
        assert result is not None, "Required property 'protocol' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def icmp(self) -> typing.Optional[AclIcmp]:
        '''The Internet Control Message Protocol (ICMP) code and type.

        :default: - Required if specifying 1 (ICMP) for the protocol parameter.
        '''
        result = self._values.get("icmp")
        return typing.cast(typing.Optional[AclIcmp], result)

    @builtins.property
    def port_range(self) -> typing.Optional[AclPortRange]:
        '''The range of port numbers for the UDP/TCP protocol.

        :default: - Required if specifying 6 (TCP) or 17 (UDP) for the protocol parameter
        '''
        result = self._values.get("port_range")
        return typing.cast(typing.Optional[AclPortRange], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AclTrafficConfig(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-ec2.Action")
class Action(enum.Enum):
    '''What action to apply to traffic matching the ACL.'''

    ALLOW = "ALLOW"
    '''Allow the traffic.'''
    DENY = "DENY"
    '''Deny the traffic.'''


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.AddRouteOptions",
    jsii_struct_bases=[],
    name_mapping={
        "router_id": "routerId",
        "router_type": "routerType",
        "destination_cidr_block": "destinationCidrBlock",
        "destination_ipv6_cidr_block": "destinationIpv6CidrBlock",
        "enables_internet_connectivity": "enablesInternetConnectivity",
    },
)
class AddRouteOptions:
    def __init__(
        self,
        *,
        router_id: builtins.str,
        router_type: "RouterType",
        destination_cidr_block: typing.Optional[builtins.str] = None,
        destination_ipv6_cidr_block: typing.Optional[builtins.str] = None,
        enables_internet_connectivity: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Options for adding a new route to a subnet.

        :param router_id: The ID of the router. Can be an instance ID, gateway ID, etc, depending on the router type.
        :param router_type: What type of router to route this traffic to.
        :param destination_cidr_block: IPv4 range this route applies to. Default: '0.0.0.0/0'
        :param destination_ipv6_cidr_block: IPv6 range this route applies to. Default: - Uses IPv6
        :param enables_internet_connectivity: Whether this route will enable internet connectivity. If true, this route will be added before any AWS resources that depend on internet connectivity in the VPC will be created. Default: false
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "router_id": router_id,
            "router_type": router_type,
        }
        if destination_cidr_block is not None:
            self._values["destination_cidr_block"] = destination_cidr_block
        if destination_ipv6_cidr_block is not None:
            self._values["destination_ipv6_cidr_block"] = destination_ipv6_cidr_block
        if enables_internet_connectivity is not None:
            self._values["enables_internet_connectivity"] = enables_internet_connectivity

    @builtins.property
    def router_id(self) -> builtins.str:
        '''The ID of the router.

        Can be an instance ID, gateway ID, etc, depending on the router type.
        '''
        result = self._values.get("router_id")
        assert result is not None, "Required property 'router_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def router_type(self) -> "RouterType":
        '''What type of router to route this traffic to.'''
        result = self._values.get("router_type")
        assert result is not None, "Required property 'router_type' is missing"
        return typing.cast("RouterType", result)

    @builtins.property
    def destination_cidr_block(self) -> typing.Optional[builtins.str]:
        '''IPv4 range this route applies to.

        :default: '0.0.0.0/0'
        '''
        result = self._values.get("destination_cidr_block")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def destination_ipv6_cidr_block(self) -> typing.Optional[builtins.str]:
        '''IPv6 range this route applies to.

        :default: - Uses IPv6
        '''
        result = self._values.get("destination_ipv6_cidr_block")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def enables_internet_connectivity(self) -> typing.Optional[builtins.bool]:
        '''Whether this route will enable internet connectivity.

        If true, this route will be added before any AWS resources that depend
        on internet connectivity in the VPC will be created.

        :default: false
        '''
        result = self._values.get("enables_internet_connectivity")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AddRouteOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-ec2.AmazonLinuxCpuType")
class AmazonLinuxCpuType(enum.Enum):
    '''CPU type.'''

    ARM_64 = "ARM_64"
    '''arm64 CPU type.'''
    X86_64 = "X86_64"
    '''x86_64 CPU type.'''


@jsii.enum(jsii_type="@aws-cdk/aws-ec2.AmazonLinuxEdition")
class AmazonLinuxEdition(enum.Enum):
    '''Amazon Linux edition.'''

    STANDARD = "STANDARD"
    '''Standard edition.'''
    MINIMAL = "MINIMAL"
    '''Minimal edition.'''


@jsii.enum(jsii_type="@aws-cdk/aws-ec2.AmazonLinuxGeneration")
class AmazonLinuxGeneration(enum.Enum):
    '''What generation of Amazon Linux to use.'''

    AMAZON_LINUX = "AMAZON_LINUX"
    '''Amazon Linux.'''
    AMAZON_LINUX_2 = "AMAZON_LINUX_2"
    '''Amazon Linux 2.'''


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.AmazonLinuxImageProps",
    jsii_struct_bases=[],
    name_mapping={
        "cpu_type": "cpuType",
        "edition": "edition",
        "generation": "generation",
        "storage": "storage",
        "user_data": "userData",
        "virtualization": "virtualization",
    },
)
class AmazonLinuxImageProps:
    def __init__(
        self,
        *,
        cpu_type: typing.Optional[AmazonLinuxCpuType] = None,
        edition: typing.Optional[AmazonLinuxEdition] = None,
        generation: typing.Optional[AmazonLinuxGeneration] = None,
        storage: typing.Optional["AmazonLinuxStorage"] = None,
        user_data: typing.Optional["UserData"] = None,
        virtualization: typing.Optional["AmazonLinuxVirt"] = None,
    ) -> None:
        '''Amazon Linux image properties.

        :param cpu_type: CPU Type. Default: X86_64
        :param edition: What edition of Amazon Linux to use. Default: Standard
        :param generation: What generation of Amazon Linux to use. Default: AmazonLinux
        :param storage: What storage backed image to use. Default: GeneralPurpose
        :param user_data: Initial user data. Default: - Empty UserData for Linux machines
        :param virtualization: Virtualization type. Default: HVM
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if cpu_type is not None:
            self._values["cpu_type"] = cpu_type
        if edition is not None:
            self._values["edition"] = edition
        if generation is not None:
            self._values["generation"] = generation
        if storage is not None:
            self._values["storage"] = storage
        if user_data is not None:
            self._values["user_data"] = user_data
        if virtualization is not None:
            self._values["virtualization"] = virtualization

    @builtins.property
    def cpu_type(self) -> typing.Optional[AmazonLinuxCpuType]:
        '''CPU Type.

        :default: X86_64
        '''
        result = self._values.get("cpu_type")
        return typing.cast(typing.Optional[AmazonLinuxCpuType], result)

    @builtins.property
    def edition(self) -> typing.Optional[AmazonLinuxEdition]:
        '''What edition of Amazon Linux to use.

        :default: Standard
        '''
        result = self._values.get("edition")
        return typing.cast(typing.Optional[AmazonLinuxEdition], result)

    @builtins.property
    def generation(self) -> typing.Optional[AmazonLinuxGeneration]:
        '''What generation of Amazon Linux to use.

        :default: AmazonLinux
        '''
        result = self._values.get("generation")
        return typing.cast(typing.Optional[AmazonLinuxGeneration], result)

    @builtins.property
    def storage(self) -> typing.Optional["AmazonLinuxStorage"]:
        '''What storage backed image to use.

        :default: GeneralPurpose
        '''
        result = self._values.get("storage")
        return typing.cast(typing.Optional["AmazonLinuxStorage"], result)

    @builtins.property
    def user_data(self) -> typing.Optional["UserData"]:
        '''Initial user data.

        :default: - Empty UserData for Linux machines
        '''
        result = self._values.get("user_data")
        return typing.cast(typing.Optional["UserData"], result)

    @builtins.property
    def virtualization(self) -> typing.Optional["AmazonLinuxVirt"]:
        '''Virtualization type.

        :default: HVM
        '''
        result = self._values.get("virtualization")
        return typing.cast(typing.Optional["AmazonLinuxVirt"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AmazonLinuxImageProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@aws-cdk/aws-ec2.AmazonLinuxStorage")
class AmazonLinuxStorage(enum.Enum):
    EBS = "EBS"
    '''EBS-backed storage.'''
    GENERAL_PURPOSE = "GENERAL_PURPOSE"
    '''General Purpose-based storage (recommended).'''


@jsii.enum(jsii_type="@aws-cdk/aws-ec2.AmazonLinuxVirt")
class AmazonLinuxVirt(enum.Enum):
    '''Virtualization type for Amazon Linux.'''

    HVM = "HVM"
    '''HVM virtualization (recommended).'''
    PV = "PV"
    '''PV virtualization.'''


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.ApplyCloudFormationInitOptions",
    jsii_struct_bases=[],
    name_mapping={
        "config_sets": "configSets",
        "embed_fingerprint": "embedFingerprint",
        "ignore_failures": "ignoreFailures",
        "print_log": "printLog",
        "timeout": "timeout",
    },
)
class ApplyCloudFormationInitOptions:
    def __init__(
        self,
        *,
        config_sets: typing.Optional[typing.Sequence[builtins.str]] = None,
        embed_fingerprint: typing.Optional[builtins.bool] = None,
        ignore_failures: typing.Optional[builtins.bool] = None,
        print_log: typing.Optional[builtins.bool] = None,
        timeout: typing.Optional[aws_cdk.core.Duration] = None,
    ) -> None:
        '''Options for applying CloudFormation init to an instance or instance group.

        :param config_sets: ConfigSet to activate. Default: ['default']
        :param embed_fingerprint: Force instance replacement by embedding a config fingerprint. If ``true`` (the default), a hash of the config will be embedded into the UserData, so that if the config changes, the UserData changes. - If the EC2 instance is instance-store backed or ``userDataCausesReplacement`` is set, this will cause the instance to be replaced and the new configuration to be applied. - If the instance is EBS-backed and ``userDataCausesReplacement`` is not set, the change of UserData will make the instance restart but not be replaced, and the configuration will not be applied automatically. If ``false``, no hash will be embedded, and if the CloudFormation Init config changes nothing will happen to the running instance. If a config update introduces errors, you will not notice until after the CloudFormation deployment successfully finishes and the next instance fails to launch. Default: true
        :param ignore_failures: Don't fail the instance creation when cfn-init fails. You can use this to prevent CloudFormation from rolling back when instances fail to start up, to help in debugging. Default: false
        :param print_log: Print the results of running cfn-init to the Instance System Log. By default, the output of running cfn-init is written to a log file on the instance. Set this to ``true`` to print it to the System Log (visible from the EC2 Console), ``false`` to not print it. (Be aware that the system log is refreshed at certain points in time of the instance life cycle, and successful execution may not always show up). Default: true
        :param timeout: Timeout waiting for the configuration to be applied. Default: Duration.minutes(5)
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if config_sets is not None:
            self._values["config_sets"] = config_sets
        if embed_fingerprint is not None:
            self._values["embed_fingerprint"] = embed_fingerprint
        if ignore_failures is not None:
            self._values["ignore_failures"] = ignore_failures
        if print_log is not None:
            self._values["print_log"] = print_log
        if timeout is not None:
            self._values["timeout"] = timeout

    @builtins.property
    def config_sets(self) -> typing.Optional[typing.List[builtins.str]]:
        '''ConfigSet to activate.

        :default: ['default']
        '''
        result = self._values.get("config_sets")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def embed_fingerprint(self) -> typing.Optional[builtins.bool]:
        '''Force instance replacement by embedding a config fingerprint.

        If ``true`` (the default), a hash of the config will be embedded into the
        UserData, so that if the config changes, the UserData changes.

        - If the EC2 instance is instance-store backed or
          ``userDataCausesReplacement`` is set, this will cause the instance to be
          replaced and the new configuration to be applied.
        - If the instance is EBS-backed and ``userDataCausesReplacement`` is not
          set, the change of UserData will make the instance restart but not be
          replaced, and the configuration will not be applied automatically.

        If ``false``, no hash will be embedded, and if the CloudFormation Init
        config changes nothing will happen to the running instance. If a
        config update introduces errors, you will not notice until after the
        CloudFormation deployment successfully finishes and the next instance
        fails to launch.

        :default: true
        '''
        result = self._values.get("embed_fingerprint")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def ignore_failures(self) -> typing.Optional[builtins.bool]:
        '''Don't fail the instance creation when cfn-init fails.

        You can use this to prevent CloudFormation from rolling back when
        instances fail to start up, to help in debugging.

        :default: false
        '''
        result = self._values.get("ignore_failures")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def print_log(self) -> typing.Optional[builtins.bool]:
        '''Print the results of running cfn-init to the Instance System Log.

        By default, the output of running cfn-init is written to a log file
        on the instance. Set this to ``true`` to print it to the System Log
        (visible from the EC2 Console), ``false`` to not print it.

        (Be aware that the system log is refreshed at certain points in
        time of the instance life cycle, and successful execution may
        not always show up).

        :default: true
        '''
        result = self._values.get("print_log")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def timeout(self) -> typing.Optional[aws_cdk.core.Duration]:
        '''Timeout waiting for the configuration to be applied.

        :default: Duration.minutes(5)
        '''
        result = self._values.get("timeout")
        return typing.cast(typing.Optional[aws_cdk.core.Duration], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApplyCloudFormationInitOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.AttachInitOptions",
    jsii_struct_bases=[],
    name_mapping={
        "instance_role": "instanceRole",
        "platform": "platform",
        "user_data": "userData",
        "config_sets": "configSets",
        "embed_fingerprint": "embedFingerprint",
        "ignore_failures": "ignoreFailures",
        "print_log": "printLog",
    },
)
class AttachInitOptions:
    def __init__(
        self,
        *,
        instance_role: aws_cdk.aws_iam.IRole,
        platform: "OperatingSystemType",
        user_data: "UserData",
        config_sets: typing.Optional[typing.Sequence[builtins.str]] = None,
        embed_fingerprint: typing.Optional[builtins.bool] = None,
        ignore_failures: typing.Optional[builtins.bool] = None,
        print_log: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Options for attaching a CloudFormationInit to a resource.

        :param instance_role: Instance role of the consuming instance or fleet.
        :param platform: OS Platform the init config will be used for.
        :param user_data: UserData to add commands to.
        :param config_sets: ConfigSet to activate. Default: ['default']
        :param embed_fingerprint: Whether to embed a hash into the userData. If ``true`` (the default), a hash of the config will be embedded into the UserData, so that if the config changes, the UserData changes and the instance will be replaced. If ``false``, no such hash will be embedded, and if the CloudFormation Init config changes nothing will happen to the running instance. Default: true
        :param ignore_failures: Don't fail the instance creation when cfn-init fails. You can use this to prevent CloudFormation from rolling back when instances fail to start up, to help in debugging. Default: false
        :param print_log: Print the results of running cfn-init to the Instance System Log. By default, the output of running cfn-init is written to a log file on the instance. Set this to ``true`` to print it to the System Log (visible from the EC2 Console), ``false`` to not print it. (Be aware that the system log is refreshed at certain points in time of the instance life cycle, and successful execution may not always show up). Default: true
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "instance_role": instance_role,
            "platform": platform,
            "user_data": user_data,
        }
        if config_sets is not None:
            self._values["config_sets"] = config_sets
        if embed_fingerprint is not None:
            self._values["embed_fingerprint"] = embed_fingerprint
        if ignore_failures is not None:
            self._values["ignore_failures"] = ignore_failures
        if print_log is not None:
            self._values["print_log"] = print_log

    @builtins.property
    def instance_role(self) -> aws_cdk.aws_iam.IRole:
        '''Instance role of the consuming instance or fleet.'''
        result = self._values.get("instance_role")
        assert result is not None, "Required property 'instance_role' is missing"
        return typing.cast(aws_cdk.aws_iam.IRole, result)

    @builtins.property
    def platform(self) -> "OperatingSystemType":
        '''OS Platform the init config will be used for.'''
        result = self._values.get("platform")
        assert result is not None, "Required property 'platform' is missing"
        return typing.cast("OperatingSystemType", result)

    @builtins.property
    def user_data(self) -> "UserData":
        '''UserData to add commands to.'''
        result = self._values.get("user_data")
        assert result is not None, "Required property 'user_data' is missing"
        return typing.cast("UserData", result)

    @builtins.property
    def config_sets(self) -> typing.Optional[typing.List[builtins.str]]:
        '''ConfigSet to activate.

        :default: ['default']
        '''
        result = self._values.get("config_sets")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def embed_fingerprint(self) -> typing.Optional[builtins.bool]:
        '''Whether to embed a hash into the userData.

        If ``true`` (the default), a hash of the config will be embedded into the
        UserData, so that if the config changes, the UserData changes and
        the instance will be replaced.

        If ``false``, no such hash will be embedded, and if the CloudFormation Init
        config changes nothing will happen to the running instance.

        :default: true
        '''
        result = self._values.get("embed_fingerprint")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def ignore_failures(self) -> typing.Optional[builtins.bool]:
        '''Don't fail the instance creation when cfn-init fails.

        You can use this to prevent CloudFormation from rolling back when
        instances fail to start up, to help in debugging.

        :default: false
        '''
        result = self._values.get("ignore_failures")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def print_log(self) -> typing.Optional[builtins.bool]:
        '''Print the results of running cfn-init to the Instance System Log.

        By default, the output of running cfn-init is written to a log file
        on the instance. Set this to ``true`` to print it to the System Log
        (visible from the EC2 Console), ``false`` to not print it.

        (Be aware that the system log is refreshed at certain points in
        time of the instance life cycle, and successful execution may
        not always show up).

        :default: true
        '''
        result = self._values.get("print_log")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AttachInitOptions(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.BastionHostLinuxProps",
    jsii_struct_bases=[],
    name_mapping={
        "vpc": "vpc",
        "availability_zone": "availabilityZone",
        "block_devices": "blockDevices",
        "instance_name": "instanceName",
        "instance_type": "instanceType",
        "machine_image": "machineImage",
        "security_group": "securityGroup",
        "subnet_selection": "subnetSelection",
    },
)
class BastionHostLinuxProps:
    def __init__(
        self,
        *,
        vpc: "IVpc",
        availability_zone: typing.Optional[builtins.str] = None,
        block_devices: typing.Optional[typing.Sequence["BlockDevice"]] = None,
        instance_name: typing.Optional[builtins.str] = None,
        instance_type: typing.Optional["InstanceType"] = None,
        machine_image: typing.Optional["IMachineImage"] = None,
        security_group: typing.Optional["ISecurityGroup"] = None,
        subnet_selection: typing.Optional["SubnetSelection"] = None,
    ) -> None:
        '''Properties of the bastion host.

        :param vpc: VPC to launch the instance in.
        :param availability_zone: In which AZ to place the instance within the VPC. Default: - Random zone.
        :param block_devices: Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes. Each instance that is launched has an associated root device volume, either an Amazon EBS volume or an instance store volume. You can use block device mappings to specify additional EBS volumes or instance store volumes to attach to an instance when it is launched. Default: - Uses the block device mapping of the AMI
        :param instance_name: The name of the instance. Default: 'BastionHost'
        :param instance_type: Type of instance to launch. Default: 't3.nano'
        :param machine_image: The machine image to use, assumed to have SSM Agent preinstalled. Default: - An Amazon Linux 2 image which is kept up-to-date automatically (the instance may be replaced on every deployment) and already has SSM Agent installed.
        :param security_group: Security Group to assign to this instance. Default: - create new security group with no inbound and all outbound traffic allowed
        :param subnet_selection: Select the subnets to run the bastion host in. Set this to PUBLIC if you need to connect to this instance via the internet and cannot use SSM. You have to allow port 22 manually by using the connections field Default: - private subnets of the supplied VPC
        '''
        if isinstance(subnet_selection, dict):
            subnet_selection = SubnetSelection(**subnet_selection)
        self._values: typing.Dict[str, typing.Any] = {
            "vpc": vpc,
        }
        if availability_zone is not None:
            self._values["availability_zone"] = availability_zone
        if block_devices is not None:
            self._values["block_devices"] = block_devices
        if instance_name is not None:
            self._values["instance_name"] = instance_name
        if instance_type is not None:
            self._values["instance_type"] = instance_type
        if machine_image is not None:
            self._values["machine_image"] = machine_image
        if security_group is not None:
            self._values["security_group"] = security_group
        if subnet_selection is not None:
            self._values["subnet_selection"] = subnet_selection

    @builtins.property
    def vpc(self) -> "IVpc":
        '''VPC to launch the instance in.'''
        result = self._values.get("vpc")
        assert result is not None, "Required property 'vpc' is missing"
        return typing.cast("IVpc", result)

    @builtins.property
    def availability_zone(self) -> typing.Optional[builtins.str]:
        '''In which AZ to place the instance within the VPC.

        :default: - Random zone.
        '''
        result = self._values.get("availability_zone")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def block_devices(self) -> typing.Optional[typing.List["BlockDevice"]]:
        '''Specifies how block devices are exposed to the instance. You can specify virtual devices and EBS volumes.

        Each instance that is launched has an associated root device volume,
        either an Amazon EBS volume or an instance store volume.
        You can use block device mappings to specify additional EBS volumes or
        instance store volumes to attach to an instance when it is launched.

        :default: - Uses the block device mapping of the AMI

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/block-device-mapping-concepts.html
        '''
        result = self._values.get("block_devices")
        return typing.cast(typing.Optional[typing.List["BlockDevice"]], result)

    @builtins.property
    def instance_name(self) -> typing.Optional[builtins.str]:
        '''The name of the instance.

        :default: 'BastionHost'
        '''
        result = self._values.get("instance_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_type(self) -> typing.Optional["InstanceType"]:
        '''Type of instance to launch.

        :default: 't3.nano'
        '''
        result = self._values.get("instance_type")
        return typing.cast(typing.Optional["InstanceType"], result)

    @builtins.property
    def machine_image(self) -> typing.Optional["IMachineImage"]:
        '''The machine image to use, assumed to have SSM Agent preinstalled.

        :default:

        - An Amazon Linux 2 image which is kept up-to-date automatically (the instance
        may be replaced on every deployment) and already has SSM Agent installed.
        '''
        result = self._values.get("machine_image")
        return typing.cast(typing.Optional["IMachineImage"], result)

    @builtins.property
    def security_group(self) -> typing.Optional["ISecurityGroup"]:
        '''Security Group to assign to this instance.

        :default: - create new security group with no inbound and all outbound traffic allowed
        '''
        result = self._values.get("security_group")
        return typing.cast(typing.Optional["ISecurityGroup"], result)

    @builtins.property
    def subnet_selection(self) -> typing.Optional["SubnetSelection"]:
        '''Select the subnets to run the bastion host in.

        Set this to PUBLIC if you need to connect to this instance via the internet and cannot use SSM.
        You have to allow port 22 manually by using the connections field

        :default: - private subnets of the supplied VPC
        '''
        result = self._values.get("subnet_selection")
        return typing.cast(typing.Optional["SubnetSelection"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BastionHostLinuxProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.BlockDevice",
    jsii_struct_bases=[],
    name_mapping={
        "device_name": "deviceName",
        "volume": "volume",
        "mapping_enabled": "mappingEnabled",
    },
)
class BlockDevice:
    def __init__(
        self,
        *,
        device_name: builtins.str,
        volume: "BlockDeviceVolume",
        mapping_enabled: typing.Optional[builtins.bool] = None,
    ) -> None:
        '''Block device.

        :param device_name: The device name exposed to the EC2 instance.
        :param volume: Defines the block device volume, to be either an Amazon EBS volume or an ephemeral instance store volume.
        :param mapping_enabled: If false, the device mapping will be suppressed. If set to false for the root device, the instance might fail the Amazon EC2 health check. Amazon EC2 Auto Scaling launches a replacement instance if the instance fails the health check. Default: true - device mapping is left untouched
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "device_name": device_name,
            "volume": volume,
        }
        if mapping_enabled is not None:
            self._values["mapping_enabled"] = mapping_enabled

    @builtins.property
    def device_name(self) -> builtins.str:
        '''The device name exposed to the EC2 instance.

        :see: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/device_naming.html

        Example::

            # Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
            "/dev/sdh" , "xvdh"
        '''
        result = self._values.get("device_name")
        assert result is not None, "Required property 'device_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def volume(self) -> "BlockDeviceVolume":
        '''Defines the block device volume, to be either an Amazon EBS volume or an ephemeral instance store volume.

        Example::

            # Example automatically generated without compilation. See https://github.com/aws/jsii/issues/826
            BlockDeviceVolume.ebs(15) , BlockDeviceVolume.ephemeral(0)
        '''
        result = self._values.get("volume")
        assert result is not None, "Required property 'volume' is missing"
        return typing.cast("BlockDeviceVolume", result)

    @builtins.property
    def mapping_enabled(self) -> typing.Optional[builtins.bool]:
        '''If false, the device mapping will be suppressed.

        If set to false for the root device, the instance might fail the Amazon EC2 health check.
        Amazon EC2 Auto Scaling launches a replacement instance if the instance fails the health check.

        :default: true - device mapping is left untouched
        '''
        result = self._values.get("mapping_enabled")
        return typing.cast(typing.Optional[builtins.bool], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BlockDevice(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class BlockDeviceVolume(
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.BlockDeviceVolume",
):
    '''Describes a block device mapping for an EC2 instance or Auto Scaling group.'''

    def __init__(
        self,
        ebs_device: typing.Optional["EbsDeviceProps"] = None,
        virtual_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param ebs_device: EBS device info.
        :param virtual_name: Virtual device name.
        '''
        jsii.create(BlockDeviceVolume, self, [ebs_device, virtual_name])

    @jsii.member(jsii_name="ebs") # type: ignore[misc]
    @builtins.classmethod
    def ebs(
        cls,
        volume_size: jsii.Number,
        *,
        encrypted: typing.Optional[builtins.bool] = None,
        delete_on_termination: typing.Optional[builtins.bool] = None,
        iops: typing.Optional[jsii.Number] = None,
        volume_type: typing.Optional["EbsDeviceVolumeType"] = None,
    ) -> "BlockDeviceVolume":
        '''Creates a new Elastic Block Storage device.

        :param volume_size: The volume size, in Gibibytes (GiB).
        :param encrypted: Specifies whether the EBS volume is encrypted. Encrypted EBS volumes can only be attached to instances that support Amazon EBS encryption Default: false
        :param delete_on_termination: Indicates whether to delete the volume when the instance is terminated. Default: - true for Amazon EC2 Auto Scaling, false otherwise (e.g. EBS)
        :param iops: The number of I/O operations per second (IOPS) to provision for the volume. Must only be set for {@link volumeType}: {@link EbsDeviceVolumeType.IO1} The maximum ratio of IOPS to volume size (in GiB) is 50:1, so for 5,000 provisioned IOPS, you need at least 100 GiB storage on the volume. Default: - none, required for {@link EbsDeviceVolumeType.IO1}
        :param volume_type: The EBS volume type. Default: {@link EbsDeviceVolumeType.GP2}
        '''
        options = EbsDeviceOptions(
            encrypted=encrypted,
            delete_on_termination=delete_on_termination,
            iops=iops,
            volume_type=volume_type,
        )

        return typing.cast("BlockDeviceVolume", jsii.sinvoke(cls, "ebs", [volume_size, options]))

    @jsii.member(jsii_name="ebsFromSnapshot") # type: ignore[misc]
    @builtins.classmethod
    def ebs_from_snapshot(
        cls,
        snapshot_id: builtins.str,
        *,
        volume_size: typing.Optional[jsii.Number] = None,
        delete_on_termination: typing.Optional[builtins.bool] = None,
        iops: typing.Optional[jsii.Number] = None,
        volume_type: typing.Optional["EbsDeviceVolumeType"] = None,
    ) -> "BlockDeviceVolume":
        '''Creates a new Elastic Block Storage device from an existing snapshot.

        :param snapshot_id: The snapshot ID of the volume to use.
        :param volume_size: The volume size, in Gibibytes (GiB). If you specify volumeSize, it must be equal or greater than the size of the snapshot. Default: - The snapshot size
        :param delete_on_termination: Indicates whether to delete the volume when the instance is terminated. Default: - true for Amazon EC2 Auto Scaling, false otherwise (e.g. EBS)
        :param iops: The number of I/O operations per second (IOPS) to provision for the volume. Must only be set for {@link volumeType}: {@link EbsDeviceVolumeType.IO1} The maximum ratio of IOPS to volume size (in GiB) is 50:1, so for 5,000 provisioned IOPS, you need at least 100 GiB storage on the volume. Default: - none, required for {@link EbsDeviceVolumeType.IO1}
        :param volume_type: The EBS volume type. Default: {@link EbsDeviceVolumeType.GP2}
        '''
        options = EbsDeviceSnapshotOptions(
            volume_size=volume_size,
            delete_on_termination=delete_on_termination,
            iops=iops,
            volume_type=volume_type,
        )

        return typing.cast("BlockDeviceVolume", jsii.sinvoke(cls, "ebsFromSnapshot", [snapshot_id, options]))

    @jsii.member(jsii_name="ephemeral") # type: ignore[misc]
    @builtins.classmethod
    def ephemeral(cls, volume_index: jsii.Number) -> "BlockDeviceVolume":
        '''Creates a virtual, ephemeral device.

        The name will be in the form ephemeral{volumeIndex}.

        :param volume_index: the volume index. Must be equal or greater than 0
        '''
        return typing.cast("BlockDeviceVolume", jsii.sinvoke(cls, "ephemeral", [volume_index]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ebsDevice")
    def ebs_device(self) -> typing.Optional["EbsDeviceProps"]:
        '''EBS device info.'''
        return typing.cast(typing.Optional["EbsDeviceProps"], jsii.get(self, "ebsDevice"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="virtualName")
    def virtual_name(self) -> typing.Optional[builtins.str]:
        '''Virtual device name.'''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "virtualName"))


@jsii.implements(aws_cdk.core.IInspectable)
class CfnCapacityReservation(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnCapacityReservation",
):
    '''A CloudFormation ``AWS::EC2::CapacityReservation``.

    :cloudformationResource: AWS::EC2::CapacityReservation
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        availability_zone: builtins.str,
        instance_count: jsii.Number,
        instance_platform: builtins.str,
        instance_type: builtins.str,
        ebs_optimized: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        end_date: typing.Optional[builtins.str] = None,
        end_date_type: typing.Optional[builtins.str] = None,
        ephemeral_storage: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        instance_match_criteria: typing.Optional[builtins.str] = None,
        tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnCapacityReservation.TagSpecificationProperty"]]]] = None,
        tenancy: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::CapacityReservation``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param availability_zone: ``AWS::EC2::CapacityReservation.AvailabilityZone``.
        :param instance_count: ``AWS::EC2::CapacityReservation.InstanceCount``.
        :param instance_platform: ``AWS::EC2::CapacityReservation.InstancePlatform``.
        :param instance_type: ``AWS::EC2::CapacityReservation.InstanceType``.
        :param ebs_optimized: ``AWS::EC2::CapacityReservation.EbsOptimized``.
        :param end_date: ``AWS::EC2::CapacityReservation.EndDate``.
        :param end_date_type: ``AWS::EC2::CapacityReservation.EndDateType``.
        :param ephemeral_storage: ``AWS::EC2::CapacityReservation.EphemeralStorage``.
        :param instance_match_criteria: ``AWS::EC2::CapacityReservation.InstanceMatchCriteria``.
        :param tag_specifications: ``AWS::EC2::CapacityReservation.TagSpecifications``.
        :param tenancy: ``AWS::EC2::CapacityReservation.Tenancy``.
        '''
        props = CfnCapacityReservationProps(
            availability_zone=availability_zone,
            instance_count=instance_count,
            instance_platform=instance_platform,
            instance_type=instance_type,
            ebs_optimized=ebs_optimized,
            end_date=end_date,
            end_date_type=end_date_type,
            ephemeral_storage=ephemeral_storage,
            instance_match_criteria=instance_match_criteria,
            tag_specifications=tag_specifications,
            tenancy=tenancy,
        )

        jsii.create(CfnCapacityReservation, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrAvailabilityZone")
    def attr_availability_zone(self) -> builtins.str:
        '''
        :cloudformationAttribute: AvailabilityZone
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAvailabilityZone"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrAvailableInstanceCount")
    def attr_available_instance_count(self) -> jsii.Number:
        '''
        :cloudformationAttribute: AvailableInstanceCount
        '''
        return typing.cast(jsii.Number, jsii.get(self, "attrAvailableInstanceCount"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrInstanceType")
    def attr_instance_type(self) -> builtins.str:
        '''
        :cloudformationAttribute: InstanceType
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrInstanceType"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrTenancy")
    def attr_tenancy(self) -> builtins.str:
        '''
        :cloudformationAttribute: Tenancy
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrTenancy"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrTotalInstanceCount")
    def attr_total_instance_count(self) -> jsii.Number:
        '''
        :cloudformationAttribute: TotalInstanceCount
        '''
        return typing.cast(jsii.Number, jsii.get(self, "attrTotalInstanceCount"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="availabilityZone")
    def availability_zone(self) -> builtins.str:
        '''``AWS::EC2::CapacityReservation.AvailabilityZone``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-availabilityzone
        '''
        return typing.cast(builtins.str, jsii.get(self, "availabilityZone"))

    @availability_zone.setter
    def availability_zone(self, value: builtins.str) -> None:
        jsii.set(self, "availabilityZone", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceCount")
    def instance_count(self) -> jsii.Number:
        '''``AWS::EC2::CapacityReservation.InstanceCount``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instancecount
        '''
        return typing.cast(jsii.Number, jsii.get(self, "instanceCount"))

    @instance_count.setter
    def instance_count(self, value: jsii.Number) -> None:
        jsii.set(self, "instanceCount", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instancePlatform")
    def instance_platform(self) -> builtins.str:
        '''``AWS::EC2::CapacityReservation.InstancePlatform``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instanceplatform
        '''
        return typing.cast(builtins.str, jsii.get(self, "instancePlatform"))

    @instance_platform.setter
    def instance_platform(self, value: builtins.str) -> None:
        jsii.set(self, "instancePlatform", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceType")
    def instance_type(self) -> builtins.str:
        '''``AWS::EC2::CapacityReservation.InstanceType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instancetype
        '''
        return typing.cast(builtins.str, jsii.get(self, "instanceType"))

    @instance_type.setter
    def instance_type(self, value: builtins.str) -> None:
        jsii.set(self, "instanceType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ebsOptimized")
    def ebs_optimized(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::CapacityReservation.EbsOptimized``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-ebsoptimized
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "ebsOptimized"))

    @ebs_optimized.setter
    def ebs_optimized(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "ebsOptimized", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="endDate")
    def end_date(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::CapacityReservation.EndDate``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-enddate
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "endDate"))

    @end_date.setter
    def end_date(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "endDate", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="endDateType")
    def end_date_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::CapacityReservation.EndDateType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-enddatetype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "endDateType"))

    @end_date_type.setter
    def end_date_type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "endDateType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ephemeralStorage")
    def ephemeral_storage(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::CapacityReservation.EphemeralStorage``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-ephemeralstorage
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "ephemeralStorage"))

    @ephemeral_storage.setter
    def ephemeral_storage(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "ephemeralStorage", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceMatchCriteria")
    def instance_match_criteria(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::CapacityReservation.InstanceMatchCriteria``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instancematchcriteria
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceMatchCriteria"))

    @instance_match_criteria.setter
    def instance_match_criteria(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "instanceMatchCriteria", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tagSpecifications")
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnCapacityReservation.TagSpecificationProperty"]]]]:
        '''``AWS::EC2::CapacityReservation.TagSpecifications``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-tagspecifications
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnCapacityReservation.TagSpecificationProperty"]]]], jsii.get(self, "tagSpecifications"))

    @tag_specifications.setter
    def tag_specifications(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnCapacityReservation.TagSpecificationProperty"]]]],
    ) -> None:
        jsii.set(self, "tagSpecifications", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tenancy")
    def tenancy(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::CapacityReservation.Tenancy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-tenancy
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "tenancy"))

    @tenancy.setter
    def tenancy(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "tenancy", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnCapacityReservation.TagSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"resource_type": "resourceType", "tags": "tags"},
    )
    class TagSpecificationProperty:
        def __init__(
            self,
            *,
            resource_type: typing.Optional[builtins.str] = None,
            tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
        ) -> None:
            '''
            :param resource_type: ``CfnCapacityReservation.TagSpecificationProperty.ResourceType``.
            :param tags: ``CfnCapacityReservation.TagSpecificationProperty.Tags``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservation-tagspecification.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if resource_type is not None:
                self._values["resource_type"] = resource_type
            if tags is not None:
                self._values["tags"] = tags

        @builtins.property
        def resource_type(self) -> typing.Optional[builtins.str]:
            '''``CfnCapacityReservation.TagSpecificationProperty.ResourceType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservation-tagspecification.html#cfn-ec2-capacityreservation-tagspecification-resourcetype
            '''
            result = self._values.get("resource_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
            '''``CfnCapacityReservation.TagSpecificationProperty.Tags``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-capacityreservation-tagspecification.html#cfn-ec2-capacityreservation-tagspecification-tags
            '''
            result = self._values.get("tags")
            return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnCapacityReservationProps",
    jsii_struct_bases=[],
    name_mapping={
        "availability_zone": "availabilityZone",
        "instance_count": "instanceCount",
        "instance_platform": "instancePlatform",
        "instance_type": "instanceType",
        "ebs_optimized": "ebsOptimized",
        "end_date": "endDate",
        "end_date_type": "endDateType",
        "ephemeral_storage": "ephemeralStorage",
        "instance_match_criteria": "instanceMatchCriteria",
        "tag_specifications": "tagSpecifications",
        "tenancy": "tenancy",
    },
)
class CfnCapacityReservationProps:
    def __init__(
        self,
        *,
        availability_zone: builtins.str,
        instance_count: jsii.Number,
        instance_platform: builtins.str,
        instance_type: builtins.str,
        ebs_optimized: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        end_date: typing.Optional[builtins.str] = None,
        end_date_type: typing.Optional[builtins.str] = None,
        ephemeral_storage: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        instance_match_criteria: typing.Optional[builtins.str] = None,
        tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnCapacityReservation.TagSpecificationProperty]]]] = None,
        tenancy: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::CapacityReservation``.

        :param availability_zone: ``AWS::EC2::CapacityReservation.AvailabilityZone``.
        :param instance_count: ``AWS::EC2::CapacityReservation.InstanceCount``.
        :param instance_platform: ``AWS::EC2::CapacityReservation.InstancePlatform``.
        :param instance_type: ``AWS::EC2::CapacityReservation.InstanceType``.
        :param ebs_optimized: ``AWS::EC2::CapacityReservation.EbsOptimized``.
        :param end_date: ``AWS::EC2::CapacityReservation.EndDate``.
        :param end_date_type: ``AWS::EC2::CapacityReservation.EndDateType``.
        :param ephemeral_storage: ``AWS::EC2::CapacityReservation.EphemeralStorage``.
        :param instance_match_criteria: ``AWS::EC2::CapacityReservation.InstanceMatchCriteria``.
        :param tag_specifications: ``AWS::EC2::CapacityReservation.TagSpecifications``.
        :param tenancy: ``AWS::EC2::CapacityReservation.Tenancy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "availability_zone": availability_zone,
            "instance_count": instance_count,
            "instance_platform": instance_platform,
            "instance_type": instance_type,
        }
        if ebs_optimized is not None:
            self._values["ebs_optimized"] = ebs_optimized
        if end_date is not None:
            self._values["end_date"] = end_date
        if end_date_type is not None:
            self._values["end_date_type"] = end_date_type
        if ephemeral_storage is not None:
            self._values["ephemeral_storage"] = ephemeral_storage
        if instance_match_criteria is not None:
            self._values["instance_match_criteria"] = instance_match_criteria
        if tag_specifications is not None:
            self._values["tag_specifications"] = tag_specifications
        if tenancy is not None:
            self._values["tenancy"] = tenancy

    @builtins.property
    def availability_zone(self) -> builtins.str:
        '''``AWS::EC2::CapacityReservation.AvailabilityZone``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-availabilityzone
        '''
        result = self._values.get("availability_zone")
        assert result is not None, "Required property 'availability_zone' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def instance_count(self) -> jsii.Number:
        '''``AWS::EC2::CapacityReservation.InstanceCount``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instancecount
        '''
        result = self._values.get("instance_count")
        assert result is not None, "Required property 'instance_count' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def instance_platform(self) -> builtins.str:
        '''``AWS::EC2::CapacityReservation.InstancePlatform``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instanceplatform
        '''
        result = self._values.get("instance_platform")
        assert result is not None, "Required property 'instance_platform' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def instance_type(self) -> builtins.str:
        '''``AWS::EC2::CapacityReservation.InstanceType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instancetype
        '''
        result = self._values.get("instance_type")
        assert result is not None, "Required property 'instance_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def ebs_optimized(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::CapacityReservation.EbsOptimized``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-ebsoptimized
        '''
        result = self._values.get("ebs_optimized")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def end_date(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::CapacityReservation.EndDate``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-enddate
        '''
        result = self._values.get("end_date")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def end_date_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::CapacityReservation.EndDateType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-enddatetype
        '''
        result = self._values.get("end_date_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ephemeral_storage(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::CapacityReservation.EphemeralStorage``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-ephemeralstorage
        '''
        result = self._values.get("ephemeral_storage")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def instance_match_criteria(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::CapacityReservation.InstanceMatchCriteria``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-instancematchcriteria
        '''
        result = self._values.get("instance_match_criteria")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnCapacityReservation.TagSpecificationProperty]]]]:
        '''``AWS::EC2::CapacityReservation.TagSpecifications``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-tagspecifications
        '''
        result = self._values.get("tag_specifications")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnCapacityReservation.TagSpecificationProperty]]]], result)

    @builtins.property
    def tenancy(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::CapacityReservation.Tenancy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-capacityreservation.html#cfn-ec2-capacityreservation-tenancy
        '''
        result = self._values.get("tenancy")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCapacityReservationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnCarrierGateway(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnCarrierGateway",
):
    '''A CloudFormation ``AWS::EC2::CarrierGateway``.

    :cloudformationResource: AWS::EC2::CarrierGateway
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-carriergateway.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        vpc_id: builtins.str,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::CarrierGateway``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param vpc_id: ``AWS::EC2::CarrierGateway.VpcId``.
        :param tags: ``AWS::EC2::CarrierGateway.Tags``.
        '''
        props = CfnCarrierGatewayProps(vpc_id=vpc_id, tags=tags)

        jsii.create(CfnCarrierGateway, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrCarrierGatewayId")
    def attr_carrier_gateway_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: CarrierGatewayId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCarrierGatewayId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrOwnerId")
    def attr_owner_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: OwnerId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrOwnerId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrState")
    def attr_state(self) -> builtins.str:
        '''
        :cloudformationAttribute: State
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrState"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''``AWS::EC2::CarrierGateway.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-carriergateway.html#cfn-ec2-carriergateway-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="vpcId")
    def vpc_id(self) -> builtins.str:
        '''``AWS::EC2::CarrierGateway.VpcId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-carriergateway.html#cfn-ec2-carriergateway-vpcid
        '''
        return typing.cast(builtins.str, jsii.get(self, "vpcId"))

    @vpc_id.setter
    def vpc_id(self, value: builtins.str) -> None:
        jsii.set(self, "vpcId", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnCarrierGatewayProps",
    jsii_struct_bases=[],
    name_mapping={"vpc_id": "vpcId", "tags": "tags"},
)
class CfnCarrierGatewayProps:
    def __init__(
        self,
        *,
        vpc_id: builtins.str,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::CarrierGateway``.

        :param vpc_id: ``AWS::EC2::CarrierGateway.VpcId``.
        :param tags: ``AWS::EC2::CarrierGateway.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-carriergateway.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "vpc_id": vpc_id,
        }
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def vpc_id(self) -> builtins.str:
        '''``AWS::EC2::CarrierGateway.VpcId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-carriergateway.html#cfn-ec2-carriergateway-vpcid
        '''
        result = self._values.get("vpc_id")
        assert result is not None, "Required property 'vpc_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''``AWS::EC2::CarrierGateway.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-carriergateway.html#cfn-ec2-carriergateway-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCarrierGatewayProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnClientVpnAuthorizationRule(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnClientVpnAuthorizationRule",
):
    '''A CloudFormation ``AWS::EC2::ClientVpnAuthorizationRule``.

    :cloudformationResource: AWS::EC2::ClientVpnAuthorizationRule
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        client_vpn_endpoint_id: builtins.str,
        target_network_cidr: builtins.str,
        access_group_id: typing.Optional[builtins.str] = None,
        authorize_all_groups: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::ClientVpnAuthorizationRule``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param client_vpn_endpoint_id: ``AWS::EC2::ClientVpnAuthorizationRule.ClientVpnEndpointId``.
        :param target_network_cidr: ``AWS::EC2::ClientVpnAuthorizationRule.TargetNetworkCidr``.
        :param access_group_id: ``AWS::EC2::ClientVpnAuthorizationRule.AccessGroupId``.
        :param authorize_all_groups: ``AWS::EC2::ClientVpnAuthorizationRule.AuthorizeAllGroups``.
        :param description: ``AWS::EC2::ClientVpnAuthorizationRule.Description``.
        '''
        props = CfnClientVpnAuthorizationRuleProps(
            client_vpn_endpoint_id=client_vpn_endpoint_id,
            target_network_cidr=target_network_cidr,
            access_group_id=access_group_id,
            authorize_all_groups=authorize_all_groups,
            description=description,
        )

        jsii.create(CfnClientVpnAuthorizationRule, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clientVpnEndpointId")
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnAuthorizationRule.ClientVpnEndpointId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-clientvpnendpointid
        '''
        return typing.cast(builtins.str, jsii.get(self, "clientVpnEndpointId"))

    @client_vpn_endpoint_id.setter
    def client_vpn_endpoint_id(self, value: builtins.str) -> None:
        jsii.set(self, "clientVpnEndpointId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="targetNetworkCidr")
    def target_network_cidr(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnAuthorizationRule.TargetNetworkCidr``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-targetnetworkcidr
        '''
        return typing.cast(builtins.str, jsii.get(self, "targetNetworkCidr"))

    @target_network_cidr.setter
    def target_network_cidr(self, value: builtins.str) -> None:
        jsii.set(self, "targetNetworkCidr", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="accessGroupId")
    def access_group_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::ClientVpnAuthorizationRule.AccessGroupId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-accessgroupid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "accessGroupId"))

    @access_group_id.setter
    def access_group_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "accessGroupId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="authorizeAllGroups")
    def authorize_all_groups(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::ClientVpnAuthorizationRule.AuthorizeAllGroups``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-authorizeallgroups
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "authorizeAllGroups"))

    @authorize_all_groups.setter
    def authorize_all_groups(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "authorizeAllGroups", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::ClientVpnAuthorizationRule.Description``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "description", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnClientVpnAuthorizationRuleProps",
    jsii_struct_bases=[],
    name_mapping={
        "client_vpn_endpoint_id": "clientVpnEndpointId",
        "target_network_cidr": "targetNetworkCidr",
        "access_group_id": "accessGroupId",
        "authorize_all_groups": "authorizeAllGroups",
        "description": "description",
    },
)
class CfnClientVpnAuthorizationRuleProps:
    def __init__(
        self,
        *,
        client_vpn_endpoint_id: builtins.str,
        target_network_cidr: builtins.str,
        access_group_id: typing.Optional[builtins.str] = None,
        authorize_all_groups: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::ClientVpnAuthorizationRule``.

        :param client_vpn_endpoint_id: ``AWS::EC2::ClientVpnAuthorizationRule.ClientVpnEndpointId``.
        :param target_network_cidr: ``AWS::EC2::ClientVpnAuthorizationRule.TargetNetworkCidr``.
        :param access_group_id: ``AWS::EC2::ClientVpnAuthorizationRule.AccessGroupId``.
        :param authorize_all_groups: ``AWS::EC2::ClientVpnAuthorizationRule.AuthorizeAllGroups``.
        :param description: ``AWS::EC2::ClientVpnAuthorizationRule.Description``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "client_vpn_endpoint_id": client_vpn_endpoint_id,
            "target_network_cidr": target_network_cidr,
        }
        if access_group_id is not None:
            self._values["access_group_id"] = access_group_id
        if authorize_all_groups is not None:
            self._values["authorize_all_groups"] = authorize_all_groups
        if description is not None:
            self._values["description"] = description

    @builtins.property
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnAuthorizationRule.ClientVpnEndpointId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-clientvpnendpointid
        '''
        result = self._values.get("client_vpn_endpoint_id")
        assert result is not None, "Required property 'client_vpn_endpoint_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def target_network_cidr(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnAuthorizationRule.TargetNetworkCidr``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-targetnetworkcidr
        '''
        result = self._values.get("target_network_cidr")
        assert result is not None, "Required property 'target_network_cidr' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def access_group_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::ClientVpnAuthorizationRule.AccessGroupId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-accessgroupid
        '''
        result = self._values.get("access_group_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def authorize_all_groups(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::ClientVpnAuthorizationRule.AuthorizeAllGroups``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-authorizeallgroups
        '''
        result = self._values.get("authorize_all_groups")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::ClientVpnAuthorizationRule.Description``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnauthorizationrule.html#cfn-ec2-clientvpnauthorizationrule-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClientVpnAuthorizationRuleProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnClientVpnEndpoint(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpoint",
):
    '''A CloudFormation ``AWS::EC2::ClientVpnEndpoint``.

    :cloudformationResource: AWS::EC2::ClientVpnEndpoint
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        authentication_options: typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union["CfnClientVpnEndpoint.ClientAuthenticationRequestProperty", aws_cdk.core.IResolvable]]],
        client_cidr_block: builtins.str,
        connection_log_options: typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.ConnectionLogOptionsProperty"],
        server_certificate_arn: builtins.str,
        client_connect_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.ClientConnectOptionsProperty"]] = None,
        description: typing.Optional[builtins.str] = None,
        dns_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        self_service_portal: typing.Optional[builtins.str] = None,
        split_tunnel: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.TagSpecificationProperty"]]]] = None,
        transport_protocol: typing.Optional[builtins.str] = None,
        vpc_id: typing.Optional[builtins.str] = None,
        vpn_port: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::ClientVpnEndpoint``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param authentication_options: ``AWS::EC2::ClientVpnEndpoint.AuthenticationOptions``.
        :param client_cidr_block: ``AWS::EC2::ClientVpnEndpoint.ClientCidrBlock``.
        :param connection_log_options: ``AWS::EC2::ClientVpnEndpoint.ConnectionLogOptions``.
        :param server_certificate_arn: ``AWS::EC2::ClientVpnEndpoint.ServerCertificateArn``.
        :param client_connect_options: ``AWS::EC2::ClientVpnEndpoint.ClientConnectOptions``.
        :param description: ``AWS::EC2::ClientVpnEndpoint.Description``.
        :param dns_servers: ``AWS::EC2::ClientVpnEndpoint.DnsServers``.
        :param security_group_ids: ``AWS::EC2::ClientVpnEndpoint.SecurityGroupIds``.
        :param self_service_portal: ``AWS::EC2::ClientVpnEndpoint.SelfServicePortal``.
        :param split_tunnel: ``AWS::EC2::ClientVpnEndpoint.SplitTunnel``.
        :param tag_specifications: ``AWS::EC2::ClientVpnEndpoint.TagSpecifications``.
        :param transport_protocol: ``AWS::EC2::ClientVpnEndpoint.TransportProtocol``.
        :param vpc_id: ``AWS::EC2::ClientVpnEndpoint.VpcId``.
        :param vpn_port: ``AWS::EC2::ClientVpnEndpoint.VpnPort``.
        '''
        props = CfnClientVpnEndpointProps(
            authentication_options=authentication_options,
            client_cidr_block=client_cidr_block,
            connection_log_options=connection_log_options,
            server_certificate_arn=server_certificate_arn,
            client_connect_options=client_connect_options,
            description=description,
            dns_servers=dns_servers,
            security_group_ids=security_group_ids,
            self_service_portal=self_service_portal,
            split_tunnel=split_tunnel,
            tag_specifications=tag_specifications,
            transport_protocol=transport_protocol,
            vpc_id=vpc_id,
            vpn_port=vpn_port,
        )

        jsii.create(CfnClientVpnEndpoint, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="authenticationOptions")
    def authentication_options(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union["CfnClientVpnEndpoint.ClientAuthenticationRequestProperty", aws_cdk.core.IResolvable]]]:
        '''``AWS::EC2::ClientVpnEndpoint.AuthenticationOptions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-authenticationoptions
        '''
        return typing.cast(typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union["CfnClientVpnEndpoint.ClientAuthenticationRequestProperty", aws_cdk.core.IResolvable]]], jsii.get(self, "authenticationOptions"))

    @authentication_options.setter
    def authentication_options(
        self,
        value: typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union["CfnClientVpnEndpoint.ClientAuthenticationRequestProperty", aws_cdk.core.IResolvable]]],
    ) -> None:
        jsii.set(self, "authenticationOptions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clientCidrBlock")
    def client_cidr_block(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnEndpoint.ClientCidrBlock``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-clientcidrblock
        '''
        return typing.cast(builtins.str, jsii.get(self, "clientCidrBlock"))

    @client_cidr_block.setter
    def client_cidr_block(self, value: builtins.str) -> None:
        jsii.set(self, "clientCidrBlock", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="connectionLogOptions")
    def connection_log_options(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.ConnectionLogOptionsProperty"]:
        '''``AWS::EC2::ClientVpnEndpoint.ConnectionLogOptions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-connectionlogoptions
        '''
        return typing.cast(typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.ConnectionLogOptionsProperty"], jsii.get(self, "connectionLogOptions"))

    @connection_log_options.setter
    def connection_log_options(
        self,
        value: typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.ConnectionLogOptionsProperty"],
    ) -> None:
        jsii.set(self, "connectionLogOptions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="serverCertificateArn")
    def server_certificate_arn(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnEndpoint.ServerCertificateArn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-servercertificatearn
        '''
        return typing.cast(builtins.str, jsii.get(self, "serverCertificateArn"))

    @server_certificate_arn.setter
    def server_certificate_arn(self, value: builtins.str) -> None:
        jsii.set(self, "serverCertificateArn", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clientConnectOptions")
    def client_connect_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.ClientConnectOptionsProperty"]]:
        '''``AWS::EC2::ClientVpnEndpoint.ClientConnectOptions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-clientconnectoptions
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.ClientConnectOptionsProperty"]], jsii.get(self, "clientConnectOptions"))

    @client_connect_options.setter
    def client_connect_options(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.ClientConnectOptionsProperty"]],
    ) -> None:
        jsii.set(self, "clientConnectOptions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::ClientVpnEndpoint.Description``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "description", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="dnsServers")
    def dns_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::ClientVpnEndpoint.DnsServers``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-dnsservers
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "dnsServers"))

    @dns_servers.setter
    def dns_servers(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        jsii.set(self, "dnsServers", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="securityGroupIds")
    def security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::ClientVpnEndpoint.SecurityGroupIds``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-securitygroupids
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "securityGroupIds"))

    @security_group_ids.setter
    def security_group_ids(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "securityGroupIds", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="selfServicePortal")
    def self_service_portal(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::ClientVpnEndpoint.SelfServicePortal``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-selfserviceportal
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "selfServicePortal"))

    @self_service_portal.setter
    def self_service_portal(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "selfServicePortal", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="splitTunnel")
    def split_tunnel(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::ClientVpnEndpoint.SplitTunnel``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-splittunnel
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "splitTunnel"))

    @split_tunnel.setter
    def split_tunnel(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "splitTunnel", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tagSpecifications")
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.TagSpecificationProperty"]]]]:
        '''``AWS::EC2::ClientVpnEndpoint.TagSpecifications``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-tagspecifications
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.TagSpecificationProperty"]]]], jsii.get(self, "tagSpecifications"))

    @tag_specifications.setter
    def tag_specifications(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.TagSpecificationProperty"]]]],
    ) -> None:
        jsii.set(self, "tagSpecifications", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="transportProtocol")
    def transport_protocol(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::ClientVpnEndpoint.TransportProtocol``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-transportprotocol
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "transportProtocol"))

    @transport_protocol.setter
    def transport_protocol(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "transportProtocol", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="vpcId")
    def vpc_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::ClientVpnEndpoint.VpcId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-vpcid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "vpcId"))

    @vpc_id.setter
    def vpc_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "vpcId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="vpnPort")
    def vpn_port(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::ClientVpnEndpoint.VpnPort``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-vpnport
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "vpnPort"))

    @vpn_port.setter
    def vpn_port(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "vpnPort", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "client_root_certificate_chain_arn": "clientRootCertificateChainArn",
        },
    )
    class CertificateAuthenticationRequestProperty:
        def __init__(self, *, client_root_certificate_chain_arn: builtins.str) -> None:
            '''
            :param client_root_certificate_chain_arn: ``CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty.ClientRootCertificateChainArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-certificateauthenticationrequest.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "client_root_certificate_chain_arn": client_root_certificate_chain_arn,
            }

        @builtins.property
        def client_root_certificate_chain_arn(self) -> builtins.str:
            '''``CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty.ClientRootCertificateChainArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-certificateauthenticationrequest.html#cfn-ec2-clientvpnendpoint-certificateauthenticationrequest-clientrootcertificatechainarn
            '''
            result = self._values.get("client_root_certificate_chain_arn")
            assert result is not None, "Required property 'client_root_certificate_chain_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CertificateAuthenticationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpoint.ClientAuthenticationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "type": "type",
            "active_directory": "activeDirectory",
            "federated_authentication": "federatedAuthentication",
            "mutual_authentication": "mutualAuthentication",
        },
    )
    class ClientAuthenticationRequestProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            active_directory: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty"]] = None,
            federated_authentication: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty"]] = None,
            mutual_authentication: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty"]] = None,
        ) -> None:
            '''
            :param type: ``CfnClientVpnEndpoint.ClientAuthenticationRequestProperty.Type``.
            :param active_directory: ``CfnClientVpnEndpoint.ClientAuthenticationRequestProperty.ActiveDirectory``.
            :param federated_authentication: ``CfnClientVpnEndpoint.ClientAuthenticationRequestProperty.FederatedAuthentication``.
            :param mutual_authentication: ``CfnClientVpnEndpoint.ClientAuthenticationRequestProperty.MutualAuthentication``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientauthenticationrequest.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "type": type,
            }
            if active_directory is not None:
                self._values["active_directory"] = active_directory
            if federated_authentication is not None:
                self._values["federated_authentication"] = federated_authentication
            if mutual_authentication is not None:
                self._values["mutual_authentication"] = mutual_authentication

        @builtins.property
        def type(self) -> builtins.str:
            '''``CfnClientVpnEndpoint.ClientAuthenticationRequestProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientauthenticationrequest.html#cfn-ec2-clientvpnendpoint-clientauthenticationrequest-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def active_directory(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty"]]:
            '''``CfnClientVpnEndpoint.ClientAuthenticationRequestProperty.ActiveDirectory``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientauthenticationrequest.html#cfn-ec2-clientvpnendpoint-clientauthenticationrequest-activedirectory
            '''
            result = self._values.get("active_directory")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty"]], result)

        @builtins.property
        def federated_authentication(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty"]]:
            '''``CfnClientVpnEndpoint.ClientAuthenticationRequestProperty.FederatedAuthentication``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientauthenticationrequest.html#cfn-ec2-clientvpnendpoint-clientauthenticationrequest-federatedauthentication
            '''
            result = self._values.get("federated_authentication")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty"]], result)

        @builtins.property
        def mutual_authentication(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty"]]:
            '''``CfnClientVpnEndpoint.ClientAuthenticationRequestProperty.MutualAuthentication``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientauthenticationrequest.html#cfn-ec2-clientvpnendpoint-clientauthenticationrequest-mutualauthentication
            '''
            result = self._values.get("mutual_authentication")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnClientVpnEndpoint.CertificateAuthenticationRequestProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClientAuthenticationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpoint.ClientConnectOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "enabled": "enabled",
            "lambda_function_arn": "lambdaFunctionArn",
        },
    )
    class ClientConnectOptionsProperty:
        def __init__(
            self,
            *,
            enabled: typing.Union[builtins.bool, aws_cdk.core.IResolvable],
            lambda_function_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param enabled: ``CfnClientVpnEndpoint.ClientConnectOptionsProperty.Enabled``.
            :param lambda_function_arn: ``CfnClientVpnEndpoint.ClientConnectOptionsProperty.LambdaFunctionArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientconnectoptions.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "enabled": enabled,
            }
            if lambda_function_arn is not None:
                self._values["lambda_function_arn"] = lambda_function_arn

        @builtins.property
        def enabled(self) -> typing.Union[builtins.bool, aws_cdk.core.IResolvable]:
            '''``CfnClientVpnEndpoint.ClientConnectOptionsProperty.Enabled``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientconnectoptions.html#cfn-ec2-clientvpnendpoint-clientconnectoptions-enabled
            '''
            result = self._values.get("enabled")
            assert result is not None, "Required property 'enabled' is missing"
            return typing.cast(typing.Union[builtins.bool, aws_cdk.core.IResolvable], result)

        @builtins.property
        def lambda_function_arn(self) -> typing.Optional[builtins.str]:
            '''``CfnClientVpnEndpoint.ClientConnectOptionsProperty.LambdaFunctionArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-clientconnectoptions.html#cfn-ec2-clientvpnendpoint-clientconnectoptions-lambdafunctionarn
            '''
            result = self._values.get("lambda_function_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClientConnectOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpoint.ConnectionLogOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "enabled": "enabled",
            "cloudwatch_log_group": "cloudwatchLogGroup",
            "cloudwatch_log_stream": "cloudwatchLogStream",
        },
    )
    class ConnectionLogOptionsProperty:
        def __init__(
            self,
            *,
            enabled: typing.Union[builtins.bool, aws_cdk.core.IResolvable],
            cloudwatch_log_group: typing.Optional[builtins.str] = None,
            cloudwatch_log_stream: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param enabled: ``CfnClientVpnEndpoint.ConnectionLogOptionsProperty.Enabled``.
            :param cloudwatch_log_group: ``CfnClientVpnEndpoint.ConnectionLogOptionsProperty.CloudwatchLogGroup``.
            :param cloudwatch_log_stream: ``CfnClientVpnEndpoint.ConnectionLogOptionsProperty.CloudwatchLogStream``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-connectionlogoptions.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "enabled": enabled,
            }
            if cloudwatch_log_group is not None:
                self._values["cloudwatch_log_group"] = cloudwatch_log_group
            if cloudwatch_log_stream is not None:
                self._values["cloudwatch_log_stream"] = cloudwatch_log_stream

        @builtins.property
        def enabled(self) -> typing.Union[builtins.bool, aws_cdk.core.IResolvable]:
            '''``CfnClientVpnEndpoint.ConnectionLogOptionsProperty.Enabled``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-connectionlogoptions.html#cfn-ec2-clientvpnendpoint-connectionlogoptions-enabled
            '''
            result = self._values.get("enabled")
            assert result is not None, "Required property 'enabled' is missing"
            return typing.cast(typing.Union[builtins.bool, aws_cdk.core.IResolvable], result)

        @builtins.property
        def cloudwatch_log_group(self) -> typing.Optional[builtins.str]:
            '''``CfnClientVpnEndpoint.ConnectionLogOptionsProperty.CloudwatchLogGroup``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-connectionlogoptions.html#cfn-ec2-clientvpnendpoint-connectionlogoptions-cloudwatchloggroup
            '''
            result = self._values.get("cloudwatch_log_group")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def cloudwatch_log_stream(self) -> typing.Optional[builtins.str]:
            '''``CfnClientVpnEndpoint.ConnectionLogOptionsProperty.CloudwatchLogStream``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-connectionlogoptions.html#cfn-ec2-clientvpnendpoint-connectionlogoptions-cloudwatchlogstream
            '''
            result = self._values.get("cloudwatch_log_stream")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ConnectionLogOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"directory_id": "directoryId"},
    )
    class DirectoryServiceAuthenticationRequestProperty:
        def __init__(self, *, directory_id: builtins.str) -> None:
            '''
            :param directory_id: ``CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty.DirectoryId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-directoryserviceauthenticationrequest.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "directory_id": directory_id,
            }

        @builtins.property
        def directory_id(self) -> builtins.str:
            '''``CfnClientVpnEndpoint.DirectoryServiceAuthenticationRequestProperty.DirectoryId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-directoryserviceauthenticationrequest.html#cfn-ec2-clientvpnendpoint-directoryserviceauthenticationrequest-directoryid
            '''
            result = self._values.get("directory_id")
            assert result is not None, "Required property 'directory_id' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "DirectoryServiceAuthenticationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "saml_provider_arn": "samlProviderArn",
            "self_service_saml_provider_arn": "selfServiceSamlProviderArn",
        },
    )
    class FederatedAuthenticationRequestProperty:
        def __init__(
            self,
            *,
            saml_provider_arn: builtins.str,
            self_service_saml_provider_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param saml_provider_arn: ``CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty.SAMLProviderArn``.
            :param self_service_saml_provider_arn: ``CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty.SelfServiceSAMLProviderArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-federatedauthenticationrequest.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "saml_provider_arn": saml_provider_arn,
            }
            if self_service_saml_provider_arn is not None:
                self._values["self_service_saml_provider_arn"] = self_service_saml_provider_arn

        @builtins.property
        def saml_provider_arn(self) -> builtins.str:
            '''``CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty.SAMLProviderArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-federatedauthenticationrequest.html#cfn-ec2-clientvpnendpoint-federatedauthenticationrequest-samlproviderarn
            '''
            result = self._values.get("saml_provider_arn")
            assert result is not None, "Required property 'saml_provider_arn' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def self_service_saml_provider_arn(self) -> typing.Optional[builtins.str]:
            '''``CfnClientVpnEndpoint.FederatedAuthenticationRequestProperty.SelfServiceSAMLProviderArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-federatedauthenticationrequest.html#cfn-ec2-clientvpnendpoint-federatedauthenticationrequest-selfservicesamlproviderarn
            '''
            result = self._values.get("self_service_saml_provider_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FederatedAuthenticationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpoint.TagSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"resource_type": "resourceType", "tags": "tags"},
    )
    class TagSpecificationProperty:
        def __init__(
            self,
            *,
            resource_type: builtins.str,
            tags: typing.Sequence[aws_cdk.core.CfnTag],
        ) -> None:
            '''
            :param resource_type: ``CfnClientVpnEndpoint.TagSpecificationProperty.ResourceType``.
            :param tags: ``CfnClientVpnEndpoint.TagSpecificationProperty.Tags``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-tagspecification.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "resource_type": resource_type,
                "tags": tags,
            }

        @builtins.property
        def resource_type(self) -> builtins.str:
            '''``CfnClientVpnEndpoint.TagSpecificationProperty.ResourceType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-tagspecification.html#cfn-ec2-clientvpnendpoint-tagspecification-resourcetype
            '''
            result = self._values.get("resource_type")
            assert result is not None, "Required property 'resource_type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def tags(self) -> typing.List[aws_cdk.core.CfnTag]:
            '''``CfnClientVpnEndpoint.TagSpecificationProperty.Tags``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-clientvpnendpoint-tagspecification.html#cfn-ec2-clientvpnendpoint-tagspecification-tags
            '''
            result = self._values.get("tags")
            assert result is not None, "Required property 'tags' is missing"
            return typing.cast(typing.List[aws_cdk.core.CfnTag], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnClientVpnEndpointProps",
    jsii_struct_bases=[],
    name_mapping={
        "authentication_options": "authenticationOptions",
        "client_cidr_block": "clientCidrBlock",
        "connection_log_options": "connectionLogOptions",
        "server_certificate_arn": "serverCertificateArn",
        "client_connect_options": "clientConnectOptions",
        "description": "description",
        "dns_servers": "dnsServers",
        "security_group_ids": "securityGroupIds",
        "self_service_portal": "selfServicePortal",
        "split_tunnel": "splitTunnel",
        "tag_specifications": "tagSpecifications",
        "transport_protocol": "transportProtocol",
        "vpc_id": "vpcId",
        "vpn_port": "vpnPort",
    },
)
class CfnClientVpnEndpointProps:
    def __init__(
        self,
        *,
        authentication_options: typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[CfnClientVpnEndpoint.ClientAuthenticationRequestProperty, aws_cdk.core.IResolvable]]],
        client_cidr_block: builtins.str,
        connection_log_options: typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.ConnectionLogOptionsProperty],
        server_certificate_arn: builtins.str,
        client_connect_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.ClientConnectOptionsProperty]] = None,
        description: typing.Optional[builtins.str] = None,
        dns_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        self_service_portal: typing.Optional[builtins.str] = None,
        split_tunnel: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.TagSpecificationProperty]]]] = None,
        transport_protocol: typing.Optional[builtins.str] = None,
        vpc_id: typing.Optional[builtins.str] = None,
        vpn_port: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::ClientVpnEndpoint``.

        :param authentication_options: ``AWS::EC2::ClientVpnEndpoint.AuthenticationOptions``.
        :param client_cidr_block: ``AWS::EC2::ClientVpnEndpoint.ClientCidrBlock``.
        :param connection_log_options: ``AWS::EC2::ClientVpnEndpoint.ConnectionLogOptions``.
        :param server_certificate_arn: ``AWS::EC2::ClientVpnEndpoint.ServerCertificateArn``.
        :param client_connect_options: ``AWS::EC2::ClientVpnEndpoint.ClientConnectOptions``.
        :param description: ``AWS::EC2::ClientVpnEndpoint.Description``.
        :param dns_servers: ``AWS::EC2::ClientVpnEndpoint.DnsServers``.
        :param security_group_ids: ``AWS::EC2::ClientVpnEndpoint.SecurityGroupIds``.
        :param self_service_portal: ``AWS::EC2::ClientVpnEndpoint.SelfServicePortal``.
        :param split_tunnel: ``AWS::EC2::ClientVpnEndpoint.SplitTunnel``.
        :param tag_specifications: ``AWS::EC2::ClientVpnEndpoint.TagSpecifications``.
        :param transport_protocol: ``AWS::EC2::ClientVpnEndpoint.TransportProtocol``.
        :param vpc_id: ``AWS::EC2::ClientVpnEndpoint.VpcId``.
        :param vpn_port: ``AWS::EC2::ClientVpnEndpoint.VpnPort``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "authentication_options": authentication_options,
            "client_cidr_block": client_cidr_block,
            "connection_log_options": connection_log_options,
            "server_certificate_arn": server_certificate_arn,
        }
        if client_connect_options is not None:
            self._values["client_connect_options"] = client_connect_options
        if description is not None:
            self._values["description"] = description
        if dns_servers is not None:
            self._values["dns_servers"] = dns_servers
        if security_group_ids is not None:
            self._values["security_group_ids"] = security_group_ids
        if self_service_portal is not None:
            self._values["self_service_portal"] = self_service_portal
        if split_tunnel is not None:
            self._values["split_tunnel"] = split_tunnel
        if tag_specifications is not None:
            self._values["tag_specifications"] = tag_specifications
        if transport_protocol is not None:
            self._values["transport_protocol"] = transport_protocol
        if vpc_id is not None:
            self._values["vpc_id"] = vpc_id
        if vpn_port is not None:
            self._values["vpn_port"] = vpn_port

    @builtins.property
    def authentication_options(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[CfnClientVpnEndpoint.ClientAuthenticationRequestProperty, aws_cdk.core.IResolvable]]]:
        '''``AWS::EC2::ClientVpnEndpoint.AuthenticationOptions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-authenticationoptions
        '''
        result = self._values.get("authentication_options")
        assert result is not None, "Required property 'authentication_options' is missing"
        return typing.cast(typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[CfnClientVpnEndpoint.ClientAuthenticationRequestProperty, aws_cdk.core.IResolvable]]], result)

    @builtins.property
    def client_cidr_block(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnEndpoint.ClientCidrBlock``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-clientcidrblock
        '''
        result = self._values.get("client_cidr_block")
        assert result is not None, "Required property 'client_cidr_block' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def connection_log_options(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.ConnectionLogOptionsProperty]:
        '''``AWS::EC2::ClientVpnEndpoint.ConnectionLogOptions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-connectionlogoptions
        '''
        result = self._values.get("connection_log_options")
        assert result is not None, "Required property 'connection_log_options' is missing"
        return typing.cast(typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.ConnectionLogOptionsProperty], result)

    @builtins.property
    def server_certificate_arn(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnEndpoint.ServerCertificateArn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-servercertificatearn
        '''
        result = self._values.get("server_certificate_arn")
        assert result is not None, "Required property 'server_certificate_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def client_connect_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.ClientConnectOptionsProperty]]:
        '''``AWS::EC2::ClientVpnEndpoint.ClientConnectOptions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-clientconnectoptions
        '''
        result = self._values.get("client_connect_options")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.ClientConnectOptionsProperty]], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::ClientVpnEndpoint.Description``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def dns_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::ClientVpnEndpoint.DnsServers``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-dnsservers
        '''
        result = self._values.get("dns_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::ClientVpnEndpoint.SecurityGroupIds``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-securitygroupids
        '''
        result = self._values.get("security_group_ids")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def self_service_portal(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::ClientVpnEndpoint.SelfServicePortal``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-selfserviceportal
        '''
        result = self._values.get("self_service_portal")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def split_tunnel(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::ClientVpnEndpoint.SplitTunnel``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-splittunnel
        '''
        result = self._values.get("split_tunnel")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.TagSpecificationProperty]]]]:
        '''``AWS::EC2::ClientVpnEndpoint.TagSpecifications``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-tagspecifications
        '''
        result = self._values.get("tag_specifications")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnClientVpnEndpoint.TagSpecificationProperty]]]], result)

    @builtins.property
    def transport_protocol(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::ClientVpnEndpoint.TransportProtocol``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-transportprotocol
        '''
        result = self._values.get("transport_protocol")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def vpc_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::ClientVpnEndpoint.VpcId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-vpcid
        '''
        result = self._values.get("vpc_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def vpn_port(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::ClientVpnEndpoint.VpnPort``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnendpoint.html#cfn-ec2-clientvpnendpoint-vpnport
        '''
        result = self._values.get("vpn_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClientVpnEndpointProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnClientVpnRoute(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnClientVpnRoute",
):
    '''A CloudFormation ``AWS::EC2::ClientVpnRoute``.

    :cloudformationResource: AWS::EC2::ClientVpnRoute
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        client_vpn_endpoint_id: builtins.str,
        destination_cidr_block: builtins.str,
        target_vpc_subnet_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::ClientVpnRoute``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param client_vpn_endpoint_id: ``AWS::EC2::ClientVpnRoute.ClientVpnEndpointId``.
        :param destination_cidr_block: ``AWS::EC2::ClientVpnRoute.DestinationCidrBlock``.
        :param target_vpc_subnet_id: ``AWS::EC2::ClientVpnRoute.TargetVpcSubnetId``.
        :param description: ``AWS::EC2::ClientVpnRoute.Description``.
        '''
        props = CfnClientVpnRouteProps(
            client_vpn_endpoint_id=client_vpn_endpoint_id,
            destination_cidr_block=destination_cidr_block,
            target_vpc_subnet_id=target_vpc_subnet_id,
            description=description,
        )

        jsii.create(CfnClientVpnRoute, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clientVpnEndpointId")
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnRoute.ClientVpnEndpointId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-clientvpnendpointid
        '''
        return typing.cast(builtins.str, jsii.get(self, "clientVpnEndpointId"))

    @client_vpn_endpoint_id.setter
    def client_vpn_endpoint_id(self, value: builtins.str) -> None:
        jsii.set(self, "clientVpnEndpointId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="destinationCidrBlock")
    def destination_cidr_block(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnRoute.DestinationCidrBlock``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-destinationcidrblock
        '''
        return typing.cast(builtins.str, jsii.get(self, "destinationCidrBlock"))

    @destination_cidr_block.setter
    def destination_cidr_block(self, value: builtins.str) -> None:
        jsii.set(self, "destinationCidrBlock", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="targetVpcSubnetId")
    def target_vpc_subnet_id(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnRoute.TargetVpcSubnetId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-targetvpcsubnetid
        '''
        return typing.cast(builtins.str, jsii.get(self, "targetVpcSubnetId"))

    @target_vpc_subnet_id.setter
    def target_vpc_subnet_id(self, value: builtins.str) -> None:
        jsii.set(self, "targetVpcSubnetId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::ClientVpnRoute.Description``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "description", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnClientVpnRouteProps",
    jsii_struct_bases=[],
    name_mapping={
        "client_vpn_endpoint_id": "clientVpnEndpointId",
        "destination_cidr_block": "destinationCidrBlock",
        "target_vpc_subnet_id": "targetVpcSubnetId",
        "description": "description",
    },
)
class CfnClientVpnRouteProps:
    def __init__(
        self,
        *,
        client_vpn_endpoint_id: builtins.str,
        destination_cidr_block: builtins.str,
        target_vpc_subnet_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::ClientVpnRoute``.

        :param client_vpn_endpoint_id: ``AWS::EC2::ClientVpnRoute.ClientVpnEndpointId``.
        :param destination_cidr_block: ``AWS::EC2::ClientVpnRoute.DestinationCidrBlock``.
        :param target_vpc_subnet_id: ``AWS::EC2::ClientVpnRoute.TargetVpcSubnetId``.
        :param description: ``AWS::EC2::ClientVpnRoute.Description``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "client_vpn_endpoint_id": client_vpn_endpoint_id,
            "destination_cidr_block": destination_cidr_block,
            "target_vpc_subnet_id": target_vpc_subnet_id,
        }
        if description is not None:
            self._values["description"] = description

    @builtins.property
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnRoute.ClientVpnEndpointId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-clientvpnendpointid
        '''
        result = self._values.get("client_vpn_endpoint_id")
        assert result is not None, "Required property 'client_vpn_endpoint_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def destination_cidr_block(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnRoute.DestinationCidrBlock``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-destinationcidrblock
        '''
        result = self._values.get("destination_cidr_block")
        assert result is not None, "Required property 'destination_cidr_block' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def target_vpc_subnet_id(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnRoute.TargetVpcSubnetId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-targetvpcsubnetid
        '''
        result = self._values.get("target_vpc_subnet_id")
        assert result is not None, "Required property 'target_vpc_subnet_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::ClientVpnRoute.Description``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpnroute.html#cfn-ec2-clientvpnroute-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClientVpnRouteProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnClientVpnTargetNetworkAssociation(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnClientVpnTargetNetworkAssociation",
):
    '''A CloudFormation ``AWS::EC2::ClientVpnTargetNetworkAssociation``.

    :cloudformationResource: AWS::EC2::ClientVpnTargetNetworkAssociation
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpntargetnetworkassociation.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        client_vpn_endpoint_id: builtins.str,
        subnet_id: builtins.str,
    ) -> None:
        '''Create a new ``AWS::EC2::ClientVpnTargetNetworkAssociation``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param client_vpn_endpoint_id: ``AWS::EC2::ClientVpnTargetNetworkAssociation.ClientVpnEndpointId``.
        :param subnet_id: ``AWS::EC2::ClientVpnTargetNetworkAssociation.SubnetId``.
        '''
        props = CfnClientVpnTargetNetworkAssociationProps(
            client_vpn_endpoint_id=client_vpn_endpoint_id, subnet_id=subnet_id
        )

        jsii.create(CfnClientVpnTargetNetworkAssociation, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clientVpnEndpointId")
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnTargetNetworkAssociation.ClientVpnEndpointId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpntargetnetworkassociation.html#cfn-ec2-clientvpntargetnetworkassociation-clientvpnendpointid
        '''
        return typing.cast(builtins.str, jsii.get(self, "clientVpnEndpointId"))

    @client_vpn_endpoint_id.setter
    def client_vpn_endpoint_id(self, value: builtins.str) -> None:
        jsii.set(self, "clientVpnEndpointId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="subnetId")
    def subnet_id(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnTargetNetworkAssociation.SubnetId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpntargetnetworkassociation.html#cfn-ec2-clientvpntargetnetworkassociation-subnetid
        '''
        return typing.cast(builtins.str, jsii.get(self, "subnetId"))

    @subnet_id.setter
    def subnet_id(self, value: builtins.str) -> None:
        jsii.set(self, "subnetId", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnClientVpnTargetNetworkAssociationProps",
    jsii_struct_bases=[],
    name_mapping={
        "client_vpn_endpoint_id": "clientVpnEndpointId",
        "subnet_id": "subnetId",
    },
)
class CfnClientVpnTargetNetworkAssociationProps:
    def __init__(
        self,
        *,
        client_vpn_endpoint_id: builtins.str,
        subnet_id: builtins.str,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::ClientVpnTargetNetworkAssociation``.

        :param client_vpn_endpoint_id: ``AWS::EC2::ClientVpnTargetNetworkAssociation.ClientVpnEndpointId``.
        :param subnet_id: ``AWS::EC2::ClientVpnTargetNetworkAssociation.SubnetId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpntargetnetworkassociation.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "client_vpn_endpoint_id": client_vpn_endpoint_id,
            "subnet_id": subnet_id,
        }

    @builtins.property
    def client_vpn_endpoint_id(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnTargetNetworkAssociation.ClientVpnEndpointId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpntargetnetworkassociation.html#cfn-ec2-clientvpntargetnetworkassociation-clientvpnendpointid
        '''
        result = self._values.get("client_vpn_endpoint_id")
        assert result is not None, "Required property 'client_vpn_endpoint_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def subnet_id(self) -> builtins.str:
        '''``AWS::EC2::ClientVpnTargetNetworkAssociation.SubnetId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-clientvpntargetnetworkassociation.html#cfn-ec2-clientvpntargetnetworkassociation-subnetid
        '''
        result = self._values.get("subnet_id")
        assert result is not None, "Required property 'subnet_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnClientVpnTargetNetworkAssociationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnCustomerGateway(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnCustomerGateway",
):
    '''A CloudFormation ``AWS::EC2::CustomerGateway``.

    :cloudformationResource: AWS::EC2::CustomerGateway
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        bgp_asn: jsii.Number,
        ip_address: builtins.str,
        type: builtins.str,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::CustomerGateway``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param bgp_asn: ``AWS::EC2::CustomerGateway.BgpAsn``.
        :param ip_address: ``AWS::EC2::CustomerGateway.IpAddress``.
        :param type: ``AWS::EC2::CustomerGateway.Type``.
        :param tags: ``AWS::EC2::CustomerGateway.Tags``.
        '''
        props = CfnCustomerGatewayProps(
            bgp_asn=bgp_asn, ip_address=ip_address, type=type, tags=tags
        )

        jsii.create(CfnCustomerGateway, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''``AWS::EC2::CustomerGateway.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html#cfn-ec2-customergateway-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="bgpAsn")
    def bgp_asn(self) -> jsii.Number:
        '''``AWS::EC2::CustomerGateway.BgpAsn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html#cfn-ec2-customergateway-bgpasn
        '''
        return typing.cast(jsii.Number, jsii.get(self, "bgpAsn"))

    @bgp_asn.setter
    def bgp_asn(self, value: jsii.Number) -> None:
        jsii.set(self, "bgpAsn", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ipAddress")
    def ip_address(self) -> builtins.str:
        '''``AWS::EC2::CustomerGateway.IpAddress``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html#cfn-ec2-customergateway-ipaddress
        '''
        return typing.cast(builtins.str, jsii.get(self, "ipAddress"))

    @ip_address.setter
    def ip_address(self, value: builtins.str) -> None:
        jsii.set(self, "ipAddress", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        '''``AWS::EC2::CustomerGateway.Type``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html#cfn-ec2-customergateway-type
        '''
        return typing.cast(builtins.str, jsii.get(self, "type"))

    @type.setter
    def type(self, value: builtins.str) -> None:
        jsii.set(self, "type", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnCustomerGatewayProps",
    jsii_struct_bases=[],
    name_mapping={
        "bgp_asn": "bgpAsn",
        "ip_address": "ipAddress",
        "type": "type",
        "tags": "tags",
    },
)
class CfnCustomerGatewayProps:
    def __init__(
        self,
        *,
        bgp_asn: jsii.Number,
        ip_address: builtins.str,
        type: builtins.str,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::CustomerGateway``.

        :param bgp_asn: ``AWS::EC2::CustomerGateway.BgpAsn``.
        :param ip_address: ``AWS::EC2::CustomerGateway.IpAddress``.
        :param type: ``AWS::EC2::CustomerGateway.Type``.
        :param tags: ``AWS::EC2::CustomerGateway.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "bgp_asn": bgp_asn,
            "ip_address": ip_address,
            "type": type,
        }
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def bgp_asn(self) -> jsii.Number:
        '''``AWS::EC2::CustomerGateway.BgpAsn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html#cfn-ec2-customergateway-bgpasn
        '''
        result = self._values.get("bgp_asn")
        assert result is not None, "Required property 'bgp_asn' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def ip_address(self) -> builtins.str:
        '''``AWS::EC2::CustomerGateway.IpAddress``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html#cfn-ec2-customergateway-ipaddress
        '''
        result = self._values.get("ip_address")
        assert result is not None, "Required property 'ip_address' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def type(self) -> builtins.str:
        '''``AWS::EC2::CustomerGateway.Type``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html#cfn-ec2-customergateway-type
        '''
        result = self._values.get("type")
        assert result is not None, "Required property 'type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''``AWS::EC2::CustomerGateway.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-customer-gateway.html#cfn-ec2-customergateway-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnCustomerGatewayProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnDHCPOptions(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnDHCPOptions",
):
    '''A CloudFormation ``AWS::EC2::DHCPOptions``.

    :cloudformationResource: AWS::EC2::DHCPOptions
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcp-options.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        domain_name: typing.Optional[builtins.str] = None,
        domain_name_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        netbios_name_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        netbios_node_type: typing.Optional[jsii.Number] = None,
        ntp_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::DHCPOptions``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param domain_name: ``AWS::EC2::DHCPOptions.DomainName``.
        :param domain_name_servers: ``AWS::EC2::DHCPOptions.DomainNameServers``.
        :param netbios_name_servers: ``AWS::EC2::DHCPOptions.NetbiosNameServers``.
        :param netbios_node_type: ``AWS::EC2::DHCPOptions.NetbiosNodeType``.
        :param ntp_servers: ``AWS::EC2::DHCPOptions.NtpServers``.
        :param tags: ``AWS::EC2::DHCPOptions.Tags``.
        '''
        props = CfnDHCPOptionsProps(
            domain_name=domain_name,
            domain_name_servers=domain_name_servers,
            netbios_name_servers=netbios_name_servers,
            netbios_node_type=netbios_node_type,
            ntp_servers=ntp_servers,
            tags=tags,
        )

        jsii.create(CfnDHCPOptions, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''``AWS::EC2::DHCPOptions.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcp-options.html#cfn-ec2-dhcpoptions-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="domainName")
    def domain_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::DHCPOptions.DomainName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcp-options.html#cfn-ec2-dhcpoptions-domainname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "domainName"))

    @domain_name.setter
    def domain_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "domainName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="domainNameServers")
    def domain_name_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::DHCPOptions.DomainNameServers``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcp-options.html#cfn-ec2-dhcpoptions-domainnameservers
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "domainNameServers"))

    @domain_name_servers.setter
    def domain_name_servers(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "domainNameServers", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="netbiosNameServers")
    def netbios_name_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::DHCPOptions.NetbiosNameServers``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcp-options.html#cfn-ec2-dhcpoptions-netbiosnameservers
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "netbiosNameServers"))

    @netbios_name_servers.setter
    def netbios_name_servers(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "netbiosNameServers", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="netbiosNodeType")
    def netbios_node_type(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::DHCPOptions.NetbiosNodeType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcp-options.html#cfn-ec2-dhcpoptions-netbiosnodetype
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "netbiosNodeType"))

    @netbios_node_type.setter
    def netbios_node_type(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "netbiosNodeType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ntpServers")
    def ntp_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::DHCPOptions.NtpServers``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcp-options.html#cfn-ec2-dhcpoptions-ntpservers
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "ntpServers"))

    @ntp_servers.setter
    def ntp_servers(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        jsii.set(self, "ntpServers", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnDHCPOptionsProps",
    jsii_struct_bases=[],
    name_mapping={
        "domain_name": "domainName",
        "domain_name_servers": "domainNameServers",
        "netbios_name_servers": "netbiosNameServers",
        "netbios_node_type": "netbiosNodeType",
        "ntp_servers": "ntpServers",
        "tags": "tags",
    },
)
class CfnDHCPOptionsProps:
    def __init__(
        self,
        *,
        domain_name: typing.Optional[builtins.str] = None,
        domain_name_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        netbios_name_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        netbios_node_type: typing.Optional[jsii.Number] = None,
        ntp_servers: typing.Optional[typing.Sequence[builtins.str]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::DHCPOptions``.

        :param domain_name: ``AWS::EC2::DHCPOptions.DomainName``.
        :param domain_name_servers: ``AWS::EC2::DHCPOptions.DomainNameServers``.
        :param netbios_name_servers: ``AWS::EC2::DHCPOptions.NetbiosNameServers``.
        :param netbios_node_type: ``AWS::EC2::DHCPOptions.NetbiosNodeType``.
        :param ntp_servers: ``AWS::EC2::DHCPOptions.NtpServers``.
        :param tags: ``AWS::EC2::DHCPOptions.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcp-options.html
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if domain_name is not None:
            self._values["domain_name"] = domain_name
        if domain_name_servers is not None:
            self._values["domain_name_servers"] = domain_name_servers
        if netbios_name_servers is not None:
            self._values["netbios_name_servers"] = netbios_name_servers
        if netbios_node_type is not None:
            self._values["netbios_node_type"] = netbios_node_type
        if ntp_servers is not None:
            self._values["ntp_servers"] = ntp_servers
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def domain_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::DHCPOptions.DomainName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcp-options.html#cfn-ec2-dhcpoptions-domainname
        '''
        result = self._values.get("domain_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def domain_name_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::DHCPOptions.DomainNameServers``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcp-options.html#cfn-ec2-dhcpoptions-domainnameservers
        '''
        result = self._values.get("domain_name_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def netbios_name_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::DHCPOptions.NetbiosNameServers``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcp-options.html#cfn-ec2-dhcpoptions-netbiosnameservers
        '''
        result = self._values.get("netbios_name_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def netbios_node_type(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::DHCPOptions.NetbiosNodeType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcp-options.html#cfn-ec2-dhcpoptions-netbiosnodetype
        '''
        result = self._values.get("netbios_node_type")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def ntp_servers(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::DHCPOptions.NtpServers``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcp-options.html#cfn-ec2-dhcpoptions-ntpservers
        '''
        result = self._values.get("ntp_servers")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''``AWS::EC2::DHCPOptions.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-dhcp-options.html#cfn-ec2-dhcpoptions-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnDHCPOptionsProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnEC2Fleet(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet",
):
    '''A CloudFormation ``AWS::EC2::EC2Fleet``.

    :cloudformationResource: AWS::EC2::EC2Fleet
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        launch_template_configs: typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty"]]],
        target_capacity_specification: typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TargetCapacitySpecificationRequestProperty"],
        excess_capacity_termination_policy: typing.Optional[builtins.str] = None,
        on_demand_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.OnDemandOptionsRequestProperty"]] = None,
        replace_unhealthy_instances: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        spot_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.SpotOptionsRequestProperty"]] = None,
        tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TagSpecificationProperty"]]]] = None,
        terminate_instances_with_expiration: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        type: typing.Optional[builtins.str] = None,
        valid_from: typing.Optional[builtins.str] = None,
        valid_until: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::EC2Fleet``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param launch_template_configs: ``AWS::EC2::EC2Fleet.LaunchTemplateConfigs``.
        :param target_capacity_specification: ``AWS::EC2::EC2Fleet.TargetCapacitySpecification``.
        :param excess_capacity_termination_policy: ``AWS::EC2::EC2Fleet.ExcessCapacityTerminationPolicy``.
        :param on_demand_options: ``AWS::EC2::EC2Fleet.OnDemandOptions``.
        :param replace_unhealthy_instances: ``AWS::EC2::EC2Fleet.ReplaceUnhealthyInstances``.
        :param spot_options: ``AWS::EC2::EC2Fleet.SpotOptions``.
        :param tag_specifications: ``AWS::EC2::EC2Fleet.TagSpecifications``.
        :param terminate_instances_with_expiration: ``AWS::EC2::EC2Fleet.TerminateInstancesWithExpiration``.
        :param type: ``AWS::EC2::EC2Fleet.Type``.
        :param valid_from: ``AWS::EC2::EC2Fleet.ValidFrom``.
        :param valid_until: ``AWS::EC2::EC2Fleet.ValidUntil``.
        '''
        props = CfnEC2FleetProps(
            launch_template_configs=launch_template_configs,
            target_capacity_specification=target_capacity_specification,
            excess_capacity_termination_policy=excess_capacity_termination_policy,
            on_demand_options=on_demand_options,
            replace_unhealthy_instances=replace_unhealthy_instances,
            spot_options=spot_options,
            tag_specifications=tag_specifications,
            terminate_instances_with_expiration=terminate_instances_with_expiration,
            type=type,
            valid_from=valid_from,
            valid_until=valid_until,
        )

        jsii.create(CfnEC2Fleet, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrFleetId")
    def attr_fleet_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: FleetId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrFleetId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="launchTemplateConfigs")
    def launch_template_configs(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty"]]]:
        '''``AWS::EC2::EC2Fleet.LaunchTemplateConfigs``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-launchtemplateconfigs
        '''
        return typing.cast(typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty"]]], jsii.get(self, "launchTemplateConfigs"))

    @launch_template_configs.setter
    def launch_template_configs(
        self,
        value: typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty"]]],
    ) -> None:
        jsii.set(self, "launchTemplateConfigs", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="targetCapacitySpecification")
    def target_capacity_specification(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TargetCapacitySpecificationRequestProperty"]:
        '''``AWS::EC2::EC2Fleet.TargetCapacitySpecification``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-targetcapacityspecification
        '''
        return typing.cast(typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TargetCapacitySpecificationRequestProperty"], jsii.get(self, "targetCapacitySpecification"))

    @target_capacity_specification.setter
    def target_capacity_specification(
        self,
        value: typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TargetCapacitySpecificationRequestProperty"],
    ) -> None:
        jsii.set(self, "targetCapacitySpecification", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="excessCapacityTerminationPolicy")
    def excess_capacity_termination_policy(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EC2Fleet.ExcessCapacityTerminationPolicy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-excesscapacityterminationpolicy
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "excessCapacityTerminationPolicy"))

    @excess_capacity_termination_policy.setter
    def excess_capacity_termination_policy(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        jsii.set(self, "excessCapacityTerminationPolicy", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="onDemandOptions")
    def on_demand_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.OnDemandOptionsRequestProperty"]]:
        '''``AWS::EC2::EC2Fleet.OnDemandOptions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-ondemandoptions
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.OnDemandOptionsRequestProperty"]], jsii.get(self, "onDemandOptions"))

    @on_demand_options.setter
    def on_demand_options(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.OnDemandOptionsRequestProperty"]],
    ) -> None:
        jsii.set(self, "onDemandOptions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="replaceUnhealthyInstances")
    def replace_unhealthy_instances(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::EC2Fleet.ReplaceUnhealthyInstances``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-replaceunhealthyinstances
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "replaceUnhealthyInstances"))

    @replace_unhealthy_instances.setter
    def replace_unhealthy_instances(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "replaceUnhealthyInstances", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="spotOptions")
    def spot_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.SpotOptionsRequestProperty"]]:
        '''``AWS::EC2::EC2Fleet.SpotOptions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-spotoptions
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.SpotOptionsRequestProperty"]], jsii.get(self, "spotOptions"))

    @spot_options.setter
    def spot_options(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.SpotOptionsRequestProperty"]],
    ) -> None:
        jsii.set(self, "spotOptions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tagSpecifications")
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TagSpecificationProperty"]]]]:
        '''``AWS::EC2::EC2Fleet.TagSpecifications``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-tagspecifications
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TagSpecificationProperty"]]]], jsii.get(self, "tagSpecifications"))

    @tag_specifications.setter
    def tag_specifications(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.TagSpecificationProperty"]]]],
    ) -> None:
        jsii.set(self, "tagSpecifications", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="terminateInstancesWithExpiration")
    def terminate_instances_with_expiration(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::EC2Fleet.TerminateInstancesWithExpiration``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-terminateinstanceswithexpiration
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "terminateInstancesWithExpiration"))

    @terminate_instances_with_expiration.setter
    def terminate_instances_with_expiration(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "terminateInstancesWithExpiration", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="type")
    def type(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EC2Fleet.Type``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-type
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "type"))

    @type.setter
    def type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "type", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="validFrom")
    def valid_from(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EC2Fleet.ValidFrom``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-validfrom
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "validFrom"))

    @valid_from.setter
    def valid_from(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "validFrom", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="validUntil")
    def valid_until(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EC2Fleet.ValidUntil``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-validuntil
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "validUntil"))

    @valid_until.setter
    def valid_until(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "validUntil", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.CapacityReservationOptionsRequestProperty",
        jsii_struct_bases=[],
        name_mapping={"usage_strategy": "usageStrategy"},
    )
    class CapacityReservationOptionsRequestProperty:
        def __init__(
            self,
            *,
            usage_strategy: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param usage_strategy: ``CfnEC2Fleet.CapacityReservationOptionsRequestProperty.UsageStrategy``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-capacityreservationoptionsrequest.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if usage_strategy is not None:
                self._values["usage_strategy"] = usage_strategy

        @builtins.property
        def usage_strategy(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.CapacityReservationOptionsRequestProperty.UsageStrategy``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-capacityreservationoptionsrequest.html#cfn-ec2-ec2fleet-capacityreservationoptionsrequest-usagestrategy
            '''
            result = self._values.get("usage_strategy")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityReservationOptionsRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "launch_template_specification": "launchTemplateSpecification",
            "overrides": "overrides",
        },
    )
    class FleetLaunchTemplateConfigRequestProperty:
        def __init__(
            self,
            *,
            launch_template_specification: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty"]] = None,
            overrides: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty"]]]] = None,
        ) -> None:
            '''
            :param launch_template_specification: ``CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty.LaunchTemplateSpecification``.
            :param overrides: ``CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty.Overrides``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateconfigrequest.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if launch_template_specification is not None:
                self._values["launch_template_specification"] = launch_template_specification
            if overrides is not None:
                self._values["overrides"] = overrides

        @builtins.property
        def launch_template_specification(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty"]]:
            '''``CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty.LaunchTemplateSpecification``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateconfigrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateconfigrequest-launchtemplatespecification
            '''
            result = self._values.get("launch_template_specification")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty"]], result)

        @builtins.property
        def overrides(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty"]]]]:
            '''``CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty.Overrides``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateconfigrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateconfigrequest-overrides
            '''
            result = self._values.get("overrides")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FleetLaunchTemplateConfigRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "availability_zone": "availabilityZone",
            "instance_type": "instanceType",
            "max_price": "maxPrice",
            "placement": "placement",
            "priority": "priority",
            "subnet_id": "subnetId",
            "weighted_capacity": "weightedCapacity",
        },
    )
    class FleetLaunchTemplateOverridesRequestProperty:
        def __init__(
            self,
            *,
            availability_zone: typing.Optional[builtins.str] = None,
            instance_type: typing.Optional[builtins.str] = None,
            max_price: typing.Optional[builtins.str] = None,
            placement: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.PlacementProperty"]] = None,
            priority: typing.Optional[jsii.Number] = None,
            subnet_id: typing.Optional[builtins.str] = None,
            weighted_capacity: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param availability_zone: ``CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty.AvailabilityZone``.
            :param instance_type: ``CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty.InstanceType``.
            :param max_price: ``CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty.MaxPrice``.
            :param placement: ``CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty.Placement``.
            :param priority: ``CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty.Priority``.
            :param subnet_id: ``CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty.SubnetId``.
            :param weighted_capacity: ``CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty.WeightedCapacity``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if availability_zone is not None:
                self._values["availability_zone"] = availability_zone
            if instance_type is not None:
                self._values["instance_type"] = instance_type
            if max_price is not None:
                self._values["max_price"] = max_price
            if placement is not None:
                self._values["placement"] = placement
            if priority is not None:
                self._values["priority"] = priority
            if subnet_id is not None:
                self._values["subnet_id"] = subnet_id
            if weighted_capacity is not None:
                self._values["weighted_capacity"] = weighted_capacity

        @builtins.property
        def availability_zone(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty.AvailabilityZone``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-availabilityzone
            '''
            result = self._values.get("availability_zone")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_type(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty.InstanceType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-instancetype
            '''
            result = self._values.get("instance_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def max_price(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty.MaxPrice``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-maxprice
            '''
            result = self._values.get("max_price")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def placement(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.PlacementProperty"]]:
            '''``CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty.Placement``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-placement
            '''
            result = self._values.get("placement")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.PlacementProperty"]], result)

        @builtins.property
        def priority(self) -> typing.Optional[jsii.Number]:
            '''``CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty.Priority``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-priority
            '''
            result = self._values.get("priority")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def subnet_id(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty.SubnetId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-subnetid
            '''
            result = self._values.get("subnet_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def weighted_capacity(self) -> typing.Optional[jsii.Number]:
            '''``CfnEC2Fleet.FleetLaunchTemplateOverridesRequestProperty.WeightedCapacity``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplateoverridesrequest-weightedcapacity
            '''
            result = self._values.get("weighted_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FleetLaunchTemplateOverridesRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "launch_template_id": "launchTemplateId",
            "launch_template_name": "launchTemplateName",
            "version": "version",
        },
    )
    class FleetLaunchTemplateSpecificationRequestProperty:
        def __init__(
            self,
            *,
            launch_template_id: typing.Optional[builtins.str] = None,
            launch_template_name: typing.Optional[builtins.str] = None,
            version: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param launch_template_id: ``CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty.LaunchTemplateId``.
            :param launch_template_name: ``CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty.LaunchTemplateName``.
            :param version: ``CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty.Version``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if launch_template_id is not None:
                self._values["launch_template_id"] = launch_template_id
            if launch_template_name is not None:
                self._values["launch_template_name"] = launch_template_name
            if version is not None:
                self._values["version"] = version

        @builtins.property
        def launch_template_id(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty.LaunchTemplateId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest-launchtemplateid
            '''
            result = self._values.get("launch_template_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def launch_template_name(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty.LaunchTemplateName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest-launchtemplatename
            '''
            result = self._values.get("launch_template_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def version(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.FleetLaunchTemplateSpecificationRequestProperty.Version``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest.html#cfn-ec2-ec2fleet-fleetlaunchtemplatespecificationrequest-version
            '''
            result = self._values.get("version")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FleetLaunchTemplateSpecificationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.OnDemandOptionsRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "allocation_strategy": "allocationStrategy",
            "capacity_reservation_options": "capacityReservationOptions",
            "max_total_price": "maxTotalPrice",
            "min_target_capacity": "minTargetCapacity",
            "single_availability_zone": "singleAvailabilityZone",
            "single_instance_type": "singleInstanceType",
        },
    )
    class OnDemandOptionsRequestProperty:
        def __init__(
            self,
            *,
            allocation_strategy: typing.Optional[builtins.str] = None,
            capacity_reservation_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.CapacityReservationOptionsRequestProperty"]] = None,
            max_total_price: typing.Optional[builtins.str] = None,
            min_target_capacity: typing.Optional[jsii.Number] = None,
            single_availability_zone: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            single_instance_type: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ) -> None:
            '''
            :param allocation_strategy: ``CfnEC2Fleet.OnDemandOptionsRequestProperty.AllocationStrategy``.
            :param capacity_reservation_options: ``CfnEC2Fleet.OnDemandOptionsRequestProperty.CapacityReservationOptions``.
            :param max_total_price: ``CfnEC2Fleet.OnDemandOptionsRequestProperty.MaxTotalPrice``.
            :param min_target_capacity: ``CfnEC2Fleet.OnDemandOptionsRequestProperty.MinTargetCapacity``.
            :param single_availability_zone: ``CfnEC2Fleet.OnDemandOptionsRequestProperty.SingleAvailabilityZone``.
            :param single_instance_type: ``CfnEC2Fleet.OnDemandOptionsRequestProperty.SingleInstanceType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if allocation_strategy is not None:
                self._values["allocation_strategy"] = allocation_strategy
            if capacity_reservation_options is not None:
                self._values["capacity_reservation_options"] = capacity_reservation_options
            if max_total_price is not None:
                self._values["max_total_price"] = max_total_price
            if min_target_capacity is not None:
                self._values["min_target_capacity"] = min_target_capacity
            if single_availability_zone is not None:
                self._values["single_availability_zone"] = single_availability_zone
            if single_instance_type is not None:
                self._values["single_instance_type"] = single_instance_type

        @builtins.property
        def allocation_strategy(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.OnDemandOptionsRequestProperty.AllocationStrategy``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-allocationstrategy
            '''
            result = self._values.get("allocation_strategy")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def capacity_reservation_options(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.CapacityReservationOptionsRequestProperty"]]:
            '''``CfnEC2Fleet.OnDemandOptionsRequestProperty.CapacityReservationOptions``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-capacityreservationoptions
            '''
            result = self._values.get("capacity_reservation_options")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnEC2Fleet.CapacityReservationOptionsRequestProperty"]], result)

        @builtins.property
        def max_total_price(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.OnDemandOptionsRequestProperty.MaxTotalPrice``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-maxtotalprice
            '''
            result = self._values.get("max_total_price")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def min_target_capacity(self) -> typing.Optional[jsii.Number]:
            '''``CfnEC2Fleet.OnDemandOptionsRequestProperty.MinTargetCapacity``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-mintargetcapacity
            '''
            result = self._values.get("min_target_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def single_availability_zone(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnEC2Fleet.OnDemandOptionsRequestProperty.SingleAvailabilityZone``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-singleavailabilityzone
            '''
            result = self._values.get("single_availability_zone")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def single_instance_type(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnEC2Fleet.OnDemandOptionsRequestProperty.SingleInstanceType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-ondemandoptionsrequest.html#cfn-ec2-ec2fleet-ondemandoptionsrequest-singleinstancetype
            '''
            result = self._values.get("single_instance_type")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "OnDemandOptionsRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.PlacementProperty",
        jsii_struct_bases=[],
        name_mapping={
            "affinity": "affinity",
            "availability_zone": "availabilityZone",
            "group_name": "groupName",
            "host_id": "hostId",
            "host_resource_group_arn": "hostResourceGroupArn",
            "partition_number": "partitionNumber",
            "spread_domain": "spreadDomain",
            "tenancy": "tenancy",
        },
    )
    class PlacementProperty:
        def __init__(
            self,
            *,
            affinity: typing.Optional[builtins.str] = None,
            availability_zone: typing.Optional[builtins.str] = None,
            group_name: typing.Optional[builtins.str] = None,
            host_id: typing.Optional[builtins.str] = None,
            host_resource_group_arn: typing.Optional[builtins.str] = None,
            partition_number: typing.Optional[jsii.Number] = None,
            spread_domain: typing.Optional[builtins.str] = None,
            tenancy: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param affinity: ``CfnEC2Fleet.PlacementProperty.Affinity``.
            :param availability_zone: ``CfnEC2Fleet.PlacementProperty.AvailabilityZone``.
            :param group_name: ``CfnEC2Fleet.PlacementProperty.GroupName``.
            :param host_id: ``CfnEC2Fleet.PlacementProperty.HostId``.
            :param host_resource_group_arn: ``CfnEC2Fleet.PlacementProperty.HostResourceGroupArn``.
            :param partition_number: ``CfnEC2Fleet.PlacementProperty.PartitionNumber``.
            :param spread_domain: ``CfnEC2Fleet.PlacementProperty.SpreadDomain``.
            :param tenancy: ``CfnEC2Fleet.PlacementProperty.Tenancy``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if affinity is not None:
                self._values["affinity"] = affinity
            if availability_zone is not None:
                self._values["availability_zone"] = availability_zone
            if group_name is not None:
                self._values["group_name"] = group_name
            if host_id is not None:
                self._values["host_id"] = host_id
            if host_resource_group_arn is not None:
                self._values["host_resource_group_arn"] = host_resource_group_arn
            if partition_number is not None:
                self._values["partition_number"] = partition_number
            if spread_domain is not None:
                self._values["spread_domain"] = spread_domain
            if tenancy is not None:
                self._values["tenancy"] = tenancy

        @builtins.property
        def affinity(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.PlacementProperty.Affinity``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-affinity
            '''
            result = self._values.get("affinity")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def availability_zone(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.PlacementProperty.AvailabilityZone``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-availabilityzone
            '''
            result = self._values.get("availability_zone")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def group_name(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.PlacementProperty.GroupName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-groupname
            '''
            result = self._values.get("group_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def host_id(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.PlacementProperty.HostId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-hostid
            '''
            result = self._values.get("host_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def host_resource_group_arn(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.PlacementProperty.HostResourceGroupArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-hostresourcegrouparn
            '''
            result = self._values.get("host_resource_group_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def partition_number(self) -> typing.Optional[jsii.Number]:
            '''``CfnEC2Fleet.PlacementProperty.PartitionNumber``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-partitionnumber
            '''
            result = self._values.get("partition_number")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def spread_domain(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.PlacementProperty.SpreadDomain``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-spreaddomain
            '''
            result = self._values.get("spread_domain")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def tenancy(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.PlacementProperty.Tenancy``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-placement.html#cfn-ec2-ec2fleet-placement-tenancy
            '''
            result = self._values.get("tenancy")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PlacementProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.SpotOptionsRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "allocation_strategy": "allocationStrategy",
            "instance_interruption_behavior": "instanceInterruptionBehavior",
            "instance_pools_to_use_count": "instancePoolsToUseCount",
            "max_total_price": "maxTotalPrice",
            "min_target_capacity": "minTargetCapacity",
            "single_availability_zone": "singleAvailabilityZone",
            "single_instance_type": "singleInstanceType",
        },
    )
    class SpotOptionsRequestProperty:
        def __init__(
            self,
            *,
            allocation_strategy: typing.Optional[builtins.str] = None,
            instance_interruption_behavior: typing.Optional[builtins.str] = None,
            instance_pools_to_use_count: typing.Optional[jsii.Number] = None,
            max_total_price: typing.Optional[builtins.str] = None,
            min_target_capacity: typing.Optional[jsii.Number] = None,
            single_availability_zone: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            single_instance_type: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ) -> None:
            '''
            :param allocation_strategy: ``CfnEC2Fleet.SpotOptionsRequestProperty.AllocationStrategy``.
            :param instance_interruption_behavior: ``CfnEC2Fleet.SpotOptionsRequestProperty.InstanceInterruptionBehavior``.
            :param instance_pools_to_use_count: ``CfnEC2Fleet.SpotOptionsRequestProperty.InstancePoolsToUseCount``.
            :param max_total_price: ``CfnEC2Fleet.SpotOptionsRequestProperty.MaxTotalPrice``.
            :param min_target_capacity: ``CfnEC2Fleet.SpotOptionsRequestProperty.MinTargetCapacity``.
            :param single_availability_zone: ``CfnEC2Fleet.SpotOptionsRequestProperty.SingleAvailabilityZone``.
            :param single_instance_type: ``CfnEC2Fleet.SpotOptionsRequestProperty.SingleInstanceType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if allocation_strategy is not None:
                self._values["allocation_strategy"] = allocation_strategy
            if instance_interruption_behavior is not None:
                self._values["instance_interruption_behavior"] = instance_interruption_behavior
            if instance_pools_to_use_count is not None:
                self._values["instance_pools_to_use_count"] = instance_pools_to_use_count
            if max_total_price is not None:
                self._values["max_total_price"] = max_total_price
            if min_target_capacity is not None:
                self._values["min_target_capacity"] = min_target_capacity
            if single_availability_zone is not None:
                self._values["single_availability_zone"] = single_availability_zone
            if single_instance_type is not None:
                self._values["single_instance_type"] = single_instance_type

        @builtins.property
        def allocation_strategy(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.SpotOptionsRequestProperty.AllocationStrategy``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-allocationstrategy
            '''
            result = self._values.get("allocation_strategy")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_interruption_behavior(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.SpotOptionsRequestProperty.InstanceInterruptionBehavior``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-instanceinterruptionbehavior
            '''
            result = self._values.get("instance_interruption_behavior")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_pools_to_use_count(self) -> typing.Optional[jsii.Number]:
            '''``CfnEC2Fleet.SpotOptionsRequestProperty.InstancePoolsToUseCount``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-instancepoolstousecount
            '''
            result = self._values.get("instance_pools_to_use_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def max_total_price(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.SpotOptionsRequestProperty.MaxTotalPrice``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-maxtotalprice
            '''
            result = self._values.get("max_total_price")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def min_target_capacity(self) -> typing.Optional[jsii.Number]:
            '''``CfnEC2Fleet.SpotOptionsRequestProperty.MinTargetCapacity``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-mintargetcapacity
            '''
            result = self._values.get("min_target_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def single_availability_zone(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnEC2Fleet.SpotOptionsRequestProperty.SingleAvailabilityZone``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-singleavailabilityzone
            '''
            result = self._values.get("single_availability_zone")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def single_instance_type(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnEC2Fleet.SpotOptionsRequestProperty.SingleInstanceType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-spotoptionsrequest.html#cfn-ec2-ec2fleet-spotoptionsrequest-singleinstancetype
            '''
            result = self._values.get("single_instance_type")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SpotOptionsRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.TagSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"resource_type": "resourceType", "tags": "tags"},
    )
    class TagSpecificationProperty:
        def __init__(
            self,
            *,
            resource_type: typing.Optional[builtins.str] = None,
            tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
        ) -> None:
            '''
            :param resource_type: ``CfnEC2Fleet.TagSpecificationProperty.ResourceType``.
            :param tags: ``CfnEC2Fleet.TagSpecificationProperty.Tags``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-tagspecification.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if resource_type is not None:
                self._values["resource_type"] = resource_type
            if tags is not None:
                self._values["tags"] = tags

        @builtins.property
        def resource_type(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.TagSpecificationProperty.ResourceType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-tagspecification.html#cfn-ec2-ec2fleet-tagspecification-resourcetype
            '''
            result = self._values.get("resource_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
            '''``CfnEC2Fleet.TagSpecificationProperty.Tags``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-tagspecification.html#cfn-ec2-ec2fleet-tagspecification-tags
            '''
            result = self._values.get("tags")
            return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnEC2Fleet.TargetCapacitySpecificationRequestProperty",
        jsii_struct_bases=[],
        name_mapping={
            "total_target_capacity": "totalTargetCapacity",
            "default_target_capacity_type": "defaultTargetCapacityType",
            "on_demand_target_capacity": "onDemandTargetCapacity",
            "spot_target_capacity": "spotTargetCapacity",
        },
    )
    class TargetCapacitySpecificationRequestProperty:
        def __init__(
            self,
            *,
            total_target_capacity: jsii.Number,
            default_target_capacity_type: typing.Optional[builtins.str] = None,
            on_demand_target_capacity: typing.Optional[jsii.Number] = None,
            spot_target_capacity: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param total_target_capacity: ``CfnEC2Fleet.TargetCapacitySpecificationRequestProperty.TotalTargetCapacity``.
            :param default_target_capacity_type: ``CfnEC2Fleet.TargetCapacitySpecificationRequestProperty.DefaultTargetCapacityType``.
            :param on_demand_target_capacity: ``CfnEC2Fleet.TargetCapacitySpecificationRequestProperty.OnDemandTargetCapacity``.
            :param spot_target_capacity: ``CfnEC2Fleet.TargetCapacitySpecificationRequestProperty.SpotTargetCapacity``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-targetcapacityspecificationrequest.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "total_target_capacity": total_target_capacity,
            }
            if default_target_capacity_type is not None:
                self._values["default_target_capacity_type"] = default_target_capacity_type
            if on_demand_target_capacity is not None:
                self._values["on_demand_target_capacity"] = on_demand_target_capacity
            if spot_target_capacity is not None:
                self._values["spot_target_capacity"] = spot_target_capacity

        @builtins.property
        def total_target_capacity(self) -> jsii.Number:
            '''``CfnEC2Fleet.TargetCapacitySpecificationRequestProperty.TotalTargetCapacity``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-targetcapacityspecificationrequest.html#cfn-ec2-ec2fleet-targetcapacityspecificationrequest-totaltargetcapacity
            '''
            result = self._values.get("total_target_capacity")
            assert result is not None, "Required property 'total_target_capacity' is missing"
            return typing.cast(jsii.Number, result)

        @builtins.property
        def default_target_capacity_type(self) -> typing.Optional[builtins.str]:
            '''``CfnEC2Fleet.TargetCapacitySpecificationRequestProperty.DefaultTargetCapacityType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-targetcapacityspecificationrequest.html#cfn-ec2-ec2fleet-targetcapacityspecificationrequest-defaulttargetcapacitytype
            '''
            result = self._values.get("default_target_capacity_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def on_demand_target_capacity(self) -> typing.Optional[jsii.Number]:
            '''``CfnEC2Fleet.TargetCapacitySpecificationRequestProperty.OnDemandTargetCapacity``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-targetcapacityspecificationrequest.html#cfn-ec2-ec2fleet-targetcapacityspecificationrequest-ondemandtargetcapacity
            '''
            result = self._values.get("on_demand_target_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def spot_target_capacity(self) -> typing.Optional[jsii.Number]:
            '''``CfnEC2Fleet.TargetCapacitySpecificationRequestProperty.SpotTargetCapacity``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-ec2fleet-targetcapacityspecificationrequest.html#cfn-ec2-ec2fleet-targetcapacityspecificationrequest-spottargetcapacity
            '''
            result = self._values.get("spot_target_capacity")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TargetCapacitySpecificationRequestProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnEC2FleetProps",
    jsii_struct_bases=[],
    name_mapping={
        "launch_template_configs": "launchTemplateConfigs",
        "target_capacity_specification": "targetCapacitySpecification",
        "excess_capacity_termination_policy": "excessCapacityTerminationPolicy",
        "on_demand_options": "onDemandOptions",
        "replace_unhealthy_instances": "replaceUnhealthyInstances",
        "spot_options": "spotOptions",
        "tag_specifications": "tagSpecifications",
        "terminate_instances_with_expiration": "terminateInstancesWithExpiration",
        "type": "type",
        "valid_from": "validFrom",
        "valid_until": "validUntil",
    },
)
class CfnEC2FleetProps:
    def __init__(
        self,
        *,
        launch_template_configs: typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty]]],
        target_capacity_specification: typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.TargetCapacitySpecificationRequestProperty],
        excess_capacity_termination_policy: typing.Optional[builtins.str] = None,
        on_demand_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.OnDemandOptionsRequestProperty]] = None,
        replace_unhealthy_instances: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        spot_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.SpotOptionsRequestProperty]] = None,
        tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.TagSpecificationProperty]]]] = None,
        terminate_instances_with_expiration: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        type: typing.Optional[builtins.str] = None,
        valid_from: typing.Optional[builtins.str] = None,
        valid_until: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::EC2Fleet``.

        :param launch_template_configs: ``AWS::EC2::EC2Fleet.LaunchTemplateConfigs``.
        :param target_capacity_specification: ``AWS::EC2::EC2Fleet.TargetCapacitySpecification``.
        :param excess_capacity_termination_policy: ``AWS::EC2::EC2Fleet.ExcessCapacityTerminationPolicy``.
        :param on_demand_options: ``AWS::EC2::EC2Fleet.OnDemandOptions``.
        :param replace_unhealthy_instances: ``AWS::EC2::EC2Fleet.ReplaceUnhealthyInstances``.
        :param spot_options: ``AWS::EC2::EC2Fleet.SpotOptions``.
        :param tag_specifications: ``AWS::EC2::EC2Fleet.TagSpecifications``.
        :param terminate_instances_with_expiration: ``AWS::EC2::EC2Fleet.TerminateInstancesWithExpiration``.
        :param type: ``AWS::EC2::EC2Fleet.Type``.
        :param valid_from: ``AWS::EC2::EC2Fleet.ValidFrom``.
        :param valid_until: ``AWS::EC2::EC2Fleet.ValidUntil``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "launch_template_configs": launch_template_configs,
            "target_capacity_specification": target_capacity_specification,
        }
        if excess_capacity_termination_policy is not None:
            self._values["excess_capacity_termination_policy"] = excess_capacity_termination_policy
        if on_demand_options is not None:
            self._values["on_demand_options"] = on_demand_options
        if replace_unhealthy_instances is not None:
            self._values["replace_unhealthy_instances"] = replace_unhealthy_instances
        if spot_options is not None:
            self._values["spot_options"] = spot_options
        if tag_specifications is not None:
            self._values["tag_specifications"] = tag_specifications
        if terminate_instances_with_expiration is not None:
            self._values["terminate_instances_with_expiration"] = terminate_instances_with_expiration
        if type is not None:
            self._values["type"] = type
        if valid_from is not None:
            self._values["valid_from"] = valid_from
        if valid_until is not None:
            self._values["valid_until"] = valid_until

    @builtins.property
    def launch_template_configs(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty]]]:
        '''``AWS::EC2::EC2Fleet.LaunchTemplateConfigs``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-launchtemplateconfigs
        '''
        result = self._values.get("launch_template_configs")
        assert result is not None, "Required property 'launch_template_configs' is missing"
        return typing.cast(typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.FleetLaunchTemplateConfigRequestProperty]]], result)

    @builtins.property
    def target_capacity_specification(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.TargetCapacitySpecificationRequestProperty]:
        '''``AWS::EC2::EC2Fleet.TargetCapacitySpecification``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-targetcapacityspecification
        '''
        result = self._values.get("target_capacity_specification")
        assert result is not None, "Required property 'target_capacity_specification' is missing"
        return typing.cast(typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.TargetCapacitySpecificationRequestProperty], result)

    @builtins.property
    def excess_capacity_termination_policy(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EC2Fleet.ExcessCapacityTerminationPolicy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-excesscapacityterminationpolicy
        '''
        result = self._values.get("excess_capacity_termination_policy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def on_demand_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.OnDemandOptionsRequestProperty]]:
        '''``AWS::EC2::EC2Fleet.OnDemandOptions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-ondemandoptions
        '''
        result = self._values.get("on_demand_options")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.OnDemandOptionsRequestProperty]], result)

    @builtins.property
    def replace_unhealthy_instances(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::EC2Fleet.ReplaceUnhealthyInstances``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-replaceunhealthyinstances
        '''
        result = self._values.get("replace_unhealthy_instances")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def spot_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.SpotOptionsRequestProperty]]:
        '''``AWS::EC2::EC2Fleet.SpotOptions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-spotoptions
        '''
        result = self._values.get("spot_options")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.SpotOptionsRequestProperty]], result)

    @builtins.property
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.TagSpecificationProperty]]]]:
        '''``AWS::EC2::EC2Fleet.TagSpecifications``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-tagspecifications
        '''
        result = self._values.get("tag_specifications")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnEC2Fleet.TagSpecificationProperty]]]], result)

    @builtins.property
    def terminate_instances_with_expiration(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::EC2Fleet.TerminateInstancesWithExpiration``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-terminateinstanceswithexpiration
        '''
        result = self._values.get("terminate_instances_with_expiration")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def type(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EC2Fleet.Type``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-type
        '''
        result = self._values.get("type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def valid_from(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EC2Fleet.ValidFrom``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-validfrom
        '''
        result = self._values.get("valid_from")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def valid_until(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EC2Fleet.ValidUntil``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-ec2fleet.html#cfn-ec2-ec2fleet-validuntil
        '''
        result = self._values.get("valid_until")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEC2FleetProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnEIP(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnEIP",
):
    '''A CloudFormation ``AWS::EC2::EIP``.

    :cloudformationResource: AWS::EC2::EIP
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        domain: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        public_ipv4_pool: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::EIP``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param domain: ``AWS::EC2::EIP.Domain``.
        :param instance_id: ``AWS::EC2::EIP.InstanceId``.
        :param public_ipv4_pool: ``AWS::EC2::EIP.PublicIpv4Pool``.
        :param tags: ``AWS::EC2::EIP.Tags``.
        '''
        props = CfnEIPProps(
            domain=domain,
            instance_id=instance_id,
            public_ipv4_pool=public_ipv4_pool,
            tags=tags,
        )

        jsii.create(CfnEIP, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrAllocationId")
    def attr_allocation_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: AllocationId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAllocationId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''``AWS::EC2::EIP.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html#cfn-ec2-eip-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="domain")
    def domain(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EIP.Domain``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html#cfn-ec2-eip-domain
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "domain"))

    @domain.setter
    def domain(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "domain", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceId")
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EIP.InstanceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html#cfn-ec2-eip-instanceid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceId"))

    @instance_id.setter
    def instance_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "instanceId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="publicIpv4Pool")
    def public_ipv4_pool(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EIP.PublicIpv4Pool``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html#cfn-ec2-eip-publicipv4pool
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "publicIpv4Pool"))

    @public_ipv4_pool.setter
    def public_ipv4_pool(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "publicIpv4Pool", value)


@jsii.implements(aws_cdk.core.IInspectable)
class CfnEIPAssociation(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnEIPAssociation",
):
    '''A CloudFormation ``AWS::EC2::EIPAssociation``.

    :cloudformationResource: AWS::EC2::EIPAssociation
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        allocation_id: typing.Optional[builtins.str] = None,
        eip: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        network_interface_id: typing.Optional[builtins.str] = None,
        private_ip_address: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::EIPAssociation``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param allocation_id: ``AWS::EC2::EIPAssociation.AllocationId``.
        :param eip: ``AWS::EC2::EIPAssociation.EIP``.
        :param instance_id: ``AWS::EC2::EIPAssociation.InstanceId``.
        :param network_interface_id: ``AWS::EC2::EIPAssociation.NetworkInterfaceId``.
        :param private_ip_address: ``AWS::EC2::EIPAssociation.PrivateIpAddress``.
        '''
        props = CfnEIPAssociationProps(
            allocation_id=allocation_id,
            eip=eip,
            instance_id=instance_id,
            network_interface_id=network_interface_id,
            private_ip_address=private_ip_address,
        )

        jsii.create(CfnEIPAssociation, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="allocationId")
    def allocation_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EIPAssociation.AllocationId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-allocationid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "allocationId"))

    @allocation_id.setter
    def allocation_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "allocationId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="eip")
    def eip(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EIPAssociation.EIP``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-eip
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "eip"))

    @eip.setter
    def eip(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "eip", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceId")
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EIPAssociation.InstanceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-instanceid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceId"))

    @instance_id.setter
    def instance_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "instanceId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkInterfaceId")
    def network_interface_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EIPAssociation.NetworkInterfaceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-networkinterfaceid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "networkInterfaceId"))

    @network_interface_id.setter
    def network_interface_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "networkInterfaceId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="privateIpAddress")
    def private_ip_address(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EIPAssociation.PrivateIpAddress``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-PrivateIpAddress
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "privateIpAddress"))

    @private_ip_address.setter
    def private_ip_address(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "privateIpAddress", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnEIPAssociationProps",
    jsii_struct_bases=[],
    name_mapping={
        "allocation_id": "allocationId",
        "eip": "eip",
        "instance_id": "instanceId",
        "network_interface_id": "networkInterfaceId",
        "private_ip_address": "privateIpAddress",
    },
)
class CfnEIPAssociationProps:
    def __init__(
        self,
        *,
        allocation_id: typing.Optional[builtins.str] = None,
        eip: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        network_interface_id: typing.Optional[builtins.str] = None,
        private_ip_address: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::EIPAssociation``.

        :param allocation_id: ``AWS::EC2::EIPAssociation.AllocationId``.
        :param eip: ``AWS::EC2::EIPAssociation.EIP``.
        :param instance_id: ``AWS::EC2::EIPAssociation.InstanceId``.
        :param network_interface_id: ``AWS::EC2::EIPAssociation.NetworkInterfaceId``.
        :param private_ip_address: ``AWS::EC2::EIPAssociation.PrivateIpAddress``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if allocation_id is not None:
            self._values["allocation_id"] = allocation_id
        if eip is not None:
            self._values["eip"] = eip
        if instance_id is not None:
            self._values["instance_id"] = instance_id
        if network_interface_id is not None:
            self._values["network_interface_id"] = network_interface_id
        if private_ip_address is not None:
            self._values["private_ip_address"] = private_ip_address

    @builtins.property
    def allocation_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EIPAssociation.AllocationId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-allocationid
        '''
        result = self._values.get("allocation_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def eip(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EIPAssociation.EIP``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-eip
        '''
        result = self._values.get("eip")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EIPAssociation.InstanceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-instanceid
        '''
        result = self._values.get("instance_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def network_interface_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EIPAssociation.NetworkInterfaceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-networkinterfaceid
        '''
        result = self._values.get("network_interface_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def private_ip_address(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EIPAssociation.PrivateIpAddress``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip-association.html#cfn-ec2-eipassociation-PrivateIpAddress
        '''
        result = self._values.get("private_ip_address")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEIPAssociationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnEIPProps",
    jsii_struct_bases=[],
    name_mapping={
        "domain": "domain",
        "instance_id": "instanceId",
        "public_ipv4_pool": "publicIpv4Pool",
        "tags": "tags",
    },
)
class CfnEIPProps:
    def __init__(
        self,
        *,
        domain: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        public_ipv4_pool: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::EIP``.

        :param domain: ``AWS::EC2::EIP.Domain``.
        :param instance_id: ``AWS::EC2::EIP.InstanceId``.
        :param public_ipv4_pool: ``AWS::EC2::EIP.PublicIpv4Pool``.
        :param tags: ``AWS::EC2::EIP.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if domain is not None:
            self._values["domain"] = domain
        if instance_id is not None:
            self._values["instance_id"] = instance_id
        if public_ipv4_pool is not None:
            self._values["public_ipv4_pool"] = public_ipv4_pool
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def domain(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EIP.Domain``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html#cfn-ec2-eip-domain
        '''
        result = self._values.get("domain")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EIP.InstanceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html#cfn-ec2-eip-instanceid
        '''
        result = self._values.get("instance_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def public_ipv4_pool(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::EIP.PublicIpv4Pool``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html#cfn-ec2-eip-publicipv4pool
        '''
        result = self._values.get("public_ipv4_pool")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''``AWS::EC2::EIP.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-eip.html#cfn-ec2-eip-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEIPProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnEgressOnlyInternetGateway(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnEgressOnlyInternetGateway",
):
    '''A CloudFormation ``AWS::EC2::EgressOnlyInternetGateway``.

    :cloudformationResource: AWS::EC2::EgressOnlyInternetGateway
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-egressonlyinternetgateway.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        vpc_id: builtins.str,
    ) -> None:
        '''Create a new ``AWS::EC2::EgressOnlyInternetGateway``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param vpc_id: ``AWS::EC2::EgressOnlyInternetGateway.VpcId``.
        '''
        props = CfnEgressOnlyInternetGatewayProps(vpc_id=vpc_id)

        jsii.create(CfnEgressOnlyInternetGateway, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="vpcId")
    def vpc_id(self) -> builtins.str:
        '''``AWS::EC2::EgressOnlyInternetGateway.VpcId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-egressonlyinternetgateway.html#cfn-ec2-egressonlyinternetgateway-vpcid
        '''
        return typing.cast(builtins.str, jsii.get(self, "vpcId"))

    @vpc_id.setter
    def vpc_id(self, value: builtins.str) -> None:
        jsii.set(self, "vpcId", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnEgressOnlyInternetGatewayProps",
    jsii_struct_bases=[],
    name_mapping={"vpc_id": "vpcId"},
)
class CfnEgressOnlyInternetGatewayProps:
    def __init__(self, *, vpc_id: builtins.str) -> None:
        '''Properties for defining a ``AWS::EC2::EgressOnlyInternetGateway``.

        :param vpc_id: ``AWS::EC2::EgressOnlyInternetGateway.VpcId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-egressonlyinternetgateway.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "vpc_id": vpc_id,
        }

    @builtins.property
    def vpc_id(self) -> builtins.str:
        '''``AWS::EC2::EgressOnlyInternetGateway.VpcId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-egressonlyinternetgateway.html#cfn-ec2-egressonlyinternetgateway-vpcid
        '''
        result = self._values.get("vpc_id")
        assert result is not None, "Required property 'vpc_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEgressOnlyInternetGatewayProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnEnclaveCertificateIamRoleAssociation(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnEnclaveCertificateIamRoleAssociation",
):
    '''A CloudFormation ``AWS::EC2::EnclaveCertificateIamRoleAssociation``.

    :cloudformationResource: AWS::EC2::EnclaveCertificateIamRoleAssociation
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-enclavecertificateiamroleassociation.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        certificate_arn: builtins.str,
        role_arn: builtins.str,
    ) -> None:
        '''Create a new ``AWS::EC2::EnclaveCertificateIamRoleAssociation``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param certificate_arn: ``AWS::EC2::EnclaveCertificateIamRoleAssociation.CertificateArn``.
        :param role_arn: ``AWS::EC2::EnclaveCertificateIamRoleAssociation.RoleArn``.
        '''
        props = CfnEnclaveCertificateIamRoleAssociationProps(
            certificate_arn=certificate_arn, role_arn=role_arn
        )

        jsii.create(CfnEnclaveCertificateIamRoleAssociation, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrCertificateS3BucketName")
    def attr_certificate_s3_bucket_name(self) -> builtins.str:
        '''
        :cloudformationAttribute: CertificateS3BucketName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCertificateS3BucketName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrCertificateS3ObjectKey")
    def attr_certificate_s3_object_key(self) -> builtins.str:
        '''
        :cloudformationAttribute: CertificateS3ObjectKey
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCertificateS3ObjectKey"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrEncryptionKmsKeyId")
    def attr_encryption_kms_key_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: EncryptionKmsKeyId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrEncryptionKmsKeyId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="certificateArn")
    def certificate_arn(self) -> builtins.str:
        '''``AWS::EC2::EnclaveCertificateIamRoleAssociation.CertificateArn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-enclavecertificateiamroleassociation.html#cfn-ec2-enclavecertificateiamroleassociation-certificatearn
        '''
        return typing.cast(builtins.str, jsii.get(self, "certificateArn"))

    @certificate_arn.setter
    def certificate_arn(self, value: builtins.str) -> None:
        jsii.set(self, "certificateArn", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> builtins.str:
        '''``AWS::EC2::EnclaveCertificateIamRoleAssociation.RoleArn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-enclavecertificateiamroleassociation.html#cfn-ec2-enclavecertificateiamroleassociation-rolearn
        '''
        return typing.cast(builtins.str, jsii.get(self, "roleArn"))

    @role_arn.setter
    def role_arn(self, value: builtins.str) -> None:
        jsii.set(self, "roleArn", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnEnclaveCertificateIamRoleAssociationProps",
    jsii_struct_bases=[],
    name_mapping={"certificate_arn": "certificateArn", "role_arn": "roleArn"},
)
class CfnEnclaveCertificateIamRoleAssociationProps:
    def __init__(
        self,
        *,
        certificate_arn: builtins.str,
        role_arn: builtins.str,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::EnclaveCertificateIamRoleAssociation``.

        :param certificate_arn: ``AWS::EC2::EnclaveCertificateIamRoleAssociation.CertificateArn``.
        :param role_arn: ``AWS::EC2::EnclaveCertificateIamRoleAssociation.RoleArn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-enclavecertificateiamroleassociation.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "certificate_arn": certificate_arn,
            "role_arn": role_arn,
        }

    @builtins.property
    def certificate_arn(self) -> builtins.str:
        '''``AWS::EC2::EnclaveCertificateIamRoleAssociation.CertificateArn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-enclavecertificateiamroleassociation.html#cfn-ec2-enclavecertificateiamroleassociation-certificatearn
        '''
        result = self._values.get("certificate_arn")
        assert result is not None, "Required property 'certificate_arn' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def role_arn(self) -> builtins.str:
        '''``AWS::EC2::EnclaveCertificateIamRoleAssociation.RoleArn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-enclavecertificateiamroleassociation.html#cfn-ec2-enclavecertificateiamroleassociation-rolearn
        '''
        result = self._values.get("role_arn")
        assert result is not None, "Required property 'role_arn' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnEnclaveCertificateIamRoleAssociationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnFlowLog(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnFlowLog",
):
    '''A CloudFormation ``AWS::EC2::FlowLog``.

    :cloudformationResource: AWS::EC2::FlowLog
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        resource_id: builtins.str,
        resource_type: builtins.str,
        traffic_type: builtins.str,
        deliver_logs_permission_arn: typing.Optional[builtins.str] = None,
        log_destination: typing.Optional[builtins.str] = None,
        log_destination_type: typing.Optional[builtins.str] = None,
        log_format: typing.Optional[builtins.str] = None,
        log_group_name: typing.Optional[builtins.str] = None,
        max_aggregation_interval: typing.Optional[jsii.Number] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::FlowLog``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param resource_id: ``AWS::EC2::FlowLog.ResourceId``.
        :param resource_type: ``AWS::EC2::FlowLog.ResourceType``.
        :param traffic_type: ``AWS::EC2::FlowLog.TrafficType``.
        :param deliver_logs_permission_arn: ``AWS::EC2::FlowLog.DeliverLogsPermissionArn``.
        :param log_destination: ``AWS::EC2::FlowLog.LogDestination``.
        :param log_destination_type: ``AWS::EC2::FlowLog.LogDestinationType``.
        :param log_format: ``AWS::EC2::FlowLog.LogFormat``.
        :param log_group_name: ``AWS::EC2::FlowLog.LogGroupName``.
        :param max_aggregation_interval: ``AWS::EC2::FlowLog.MaxAggregationInterval``.
        :param tags: ``AWS::EC2::FlowLog.Tags``.
        '''
        props = CfnFlowLogProps(
            resource_id=resource_id,
            resource_type=resource_type,
            traffic_type=traffic_type,
            deliver_logs_permission_arn=deliver_logs_permission_arn,
            log_destination=log_destination,
            log_destination_type=log_destination_type,
            log_format=log_format,
            log_group_name=log_group_name,
            max_aggregation_interval=max_aggregation_interval,
            tags=tags,
        )

        jsii.create(CfnFlowLog, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''``AWS::EC2::FlowLog.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="resourceId")
    def resource_id(self) -> builtins.str:
        '''``AWS::EC2::FlowLog.ResourceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-resourceid
        '''
        return typing.cast(builtins.str, jsii.get(self, "resourceId"))

    @resource_id.setter
    def resource_id(self, value: builtins.str) -> None:
        jsii.set(self, "resourceId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="resourceType")
    def resource_type(self) -> builtins.str:
        '''``AWS::EC2::FlowLog.ResourceType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-resourcetype
        '''
        return typing.cast(builtins.str, jsii.get(self, "resourceType"))

    @resource_type.setter
    def resource_type(self, value: builtins.str) -> None:
        jsii.set(self, "resourceType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="trafficType")
    def traffic_type(self) -> builtins.str:
        '''``AWS::EC2::FlowLog.TrafficType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-traffictype
        '''
        return typing.cast(builtins.str, jsii.get(self, "trafficType"))

    @traffic_type.setter
    def traffic_type(self, value: builtins.str) -> None:
        jsii.set(self, "trafficType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="deliverLogsPermissionArn")
    def deliver_logs_permission_arn(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::FlowLog.DeliverLogsPermissionArn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-deliverlogspermissionarn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "deliverLogsPermissionArn"))

    @deliver_logs_permission_arn.setter
    def deliver_logs_permission_arn(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "deliverLogsPermissionArn", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="logDestination")
    def log_destination(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::FlowLog.LogDestination``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logdestination
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "logDestination"))

    @log_destination.setter
    def log_destination(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "logDestination", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="logDestinationType")
    def log_destination_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::FlowLog.LogDestinationType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logdestinationtype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "logDestinationType"))

    @log_destination_type.setter
    def log_destination_type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "logDestinationType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="logFormat")
    def log_format(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::FlowLog.LogFormat``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logformat
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "logFormat"))

    @log_format.setter
    def log_format(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "logFormat", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="logGroupName")
    def log_group_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::FlowLog.LogGroupName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-loggroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "logGroupName"))

    @log_group_name.setter
    def log_group_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "logGroupName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="maxAggregationInterval")
    def max_aggregation_interval(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::FlowLog.MaxAggregationInterval``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-maxaggregationinterval
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "maxAggregationInterval"))

    @max_aggregation_interval.setter
    def max_aggregation_interval(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "maxAggregationInterval", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnFlowLogProps",
    jsii_struct_bases=[],
    name_mapping={
        "resource_id": "resourceId",
        "resource_type": "resourceType",
        "traffic_type": "trafficType",
        "deliver_logs_permission_arn": "deliverLogsPermissionArn",
        "log_destination": "logDestination",
        "log_destination_type": "logDestinationType",
        "log_format": "logFormat",
        "log_group_name": "logGroupName",
        "max_aggregation_interval": "maxAggregationInterval",
        "tags": "tags",
    },
)
class CfnFlowLogProps:
    def __init__(
        self,
        *,
        resource_id: builtins.str,
        resource_type: builtins.str,
        traffic_type: builtins.str,
        deliver_logs_permission_arn: typing.Optional[builtins.str] = None,
        log_destination: typing.Optional[builtins.str] = None,
        log_destination_type: typing.Optional[builtins.str] = None,
        log_format: typing.Optional[builtins.str] = None,
        log_group_name: typing.Optional[builtins.str] = None,
        max_aggregation_interval: typing.Optional[jsii.Number] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::FlowLog``.

        :param resource_id: ``AWS::EC2::FlowLog.ResourceId``.
        :param resource_type: ``AWS::EC2::FlowLog.ResourceType``.
        :param traffic_type: ``AWS::EC2::FlowLog.TrafficType``.
        :param deliver_logs_permission_arn: ``AWS::EC2::FlowLog.DeliverLogsPermissionArn``.
        :param log_destination: ``AWS::EC2::FlowLog.LogDestination``.
        :param log_destination_type: ``AWS::EC2::FlowLog.LogDestinationType``.
        :param log_format: ``AWS::EC2::FlowLog.LogFormat``.
        :param log_group_name: ``AWS::EC2::FlowLog.LogGroupName``.
        :param max_aggregation_interval: ``AWS::EC2::FlowLog.MaxAggregationInterval``.
        :param tags: ``AWS::EC2::FlowLog.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "resource_id": resource_id,
            "resource_type": resource_type,
            "traffic_type": traffic_type,
        }
        if deliver_logs_permission_arn is not None:
            self._values["deliver_logs_permission_arn"] = deliver_logs_permission_arn
        if log_destination is not None:
            self._values["log_destination"] = log_destination
        if log_destination_type is not None:
            self._values["log_destination_type"] = log_destination_type
        if log_format is not None:
            self._values["log_format"] = log_format
        if log_group_name is not None:
            self._values["log_group_name"] = log_group_name
        if max_aggregation_interval is not None:
            self._values["max_aggregation_interval"] = max_aggregation_interval
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def resource_id(self) -> builtins.str:
        '''``AWS::EC2::FlowLog.ResourceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-resourceid
        '''
        result = self._values.get("resource_id")
        assert result is not None, "Required property 'resource_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def resource_type(self) -> builtins.str:
        '''``AWS::EC2::FlowLog.ResourceType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-resourcetype
        '''
        result = self._values.get("resource_type")
        assert result is not None, "Required property 'resource_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def traffic_type(self) -> builtins.str:
        '''``AWS::EC2::FlowLog.TrafficType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-traffictype
        '''
        result = self._values.get("traffic_type")
        assert result is not None, "Required property 'traffic_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def deliver_logs_permission_arn(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::FlowLog.DeliverLogsPermissionArn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-deliverlogspermissionarn
        '''
        result = self._values.get("deliver_logs_permission_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_destination(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::FlowLog.LogDestination``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logdestination
        '''
        result = self._values.get("log_destination")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_destination_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::FlowLog.LogDestinationType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logdestinationtype
        '''
        result = self._values.get("log_destination_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_format(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::FlowLog.LogFormat``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-logformat
        '''
        result = self._values.get("log_format")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def log_group_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::FlowLog.LogGroupName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-loggroupname
        '''
        result = self._values.get("log_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_aggregation_interval(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::FlowLog.MaxAggregationInterval``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-maxaggregationinterval
        '''
        result = self._values.get("max_aggregation_interval")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''``AWS::EC2::FlowLog.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-flowlog.html#cfn-ec2-flowlog-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnFlowLogProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnGatewayRouteTableAssociation(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnGatewayRouteTableAssociation",
):
    '''A CloudFormation ``AWS::EC2::GatewayRouteTableAssociation``.

    :cloudformationResource: AWS::EC2::GatewayRouteTableAssociation
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-gatewayroutetableassociation.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        gateway_id: builtins.str,
        route_table_id: builtins.str,
    ) -> None:
        '''Create a new ``AWS::EC2::GatewayRouteTableAssociation``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param gateway_id: ``AWS::EC2::GatewayRouteTableAssociation.GatewayId``.
        :param route_table_id: ``AWS::EC2::GatewayRouteTableAssociation.RouteTableId``.
        '''
        props = CfnGatewayRouteTableAssociationProps(
            gateway_id=gateway_id, route_table_id=route_table_id
        )

        jsii.create(CfnGatewayRouteTableAssociation, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrAssociationId")
    def attr_association_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: AssociationId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAssociationId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="gatewayId")
    def gateway_id(self) -> builtins.str:
        '''``AWS::EC2::GatewayRouteTableAssociation.GatewayId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-gatewayroutetableassociation.html#cfn-ec2-gatewayroutetableassociation-gatewayid
        '''
        return typing.cast(builtins.str, jsii.get(self, "gatewayId"))

    @gateway_id.setter
    def gateway_id(self, value: builtins.str) -> None:
        jsii.set(self, "gatewayId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="routeTableId")
    def route_table_id(self) -> builtins.str:
        '''``AWS::EC2::GatewayRouteTableAssociation.RouteTableId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-gatewayroutetableassociation.html#cfn-ec2-gatewayroutetableassociation-routetableid
        '''
        return typing.cast(builtins.str, jsii.get(self, "routeTableId"))

    @route_table_id.setter
    def route_table_id(self, value: builtins.str) -> None:
        jsii.set(self, "routeTableId", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnGatewayRouteTableAssociationProps",
    jsii_struct_bases=[],
    name_mapping={"gateway_id": "gatewayId", "route_table_id": "routeTableId"},
)
class CfnGatewayRouteTableAssociationProps:
    def __init__(
        self,
        *,
        gateway_id: builtins.str,
        route_table_id: builtins.str,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::GatewayRouteTableAssociation``.

        :param gateway_id: ``AWS::EC2::GatewayRouteTableAssociation.GatewayId``.
        :param route_table_id: ``AWS::EC2::GatewayRouteTableAssociation.RouteTableId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-gatewayroutetableassociation.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "gateway_id": gateway_id,
            "route_table_id": route_table_id,
        }

    @builtins.property
    def gateway_id(self) -> builtins.str:
        '''``AWS::EC2::GatewayRouteTableAssociation.GatewayId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-gatewayroutetableassociation.html#cfn-ec2-gatewayroutetableassociation-gatewayid
        '''
        result = self._values.get("gateway_id")
        assert result is not None, "Required property 'gateway_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def route_table_id(self) -> builtins.str:
        '''``AWS::EC2::GatewayRouteTableAssociation.RouteTableId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-gatewayroutetableassociation.html#cfn-ec2-gatewayroutetableassociation-routetableid
        '''
        result = self._values.get("route_table_id")
        assert result is not None, "Required property 'route_table_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnGatewayRouteTableAssociationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnHost(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnHost",
):
    '''A CloudFormation ``AWS::EC2::Host``.

    :cloudformationResource: AWS::EC2::Host
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        availability_zone: builtins.str,
        instance_type: builtins.str,
        auto_placement: typing.Optional[builtins.str] = None,
        host_recovery: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::Host``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param availability_zone: ``AWS::EC2::Host.AvailabilityZone``.
        :param instance_type: ``AWS::EC2::Host.InstanceType``.
        :param auto_placement: ``AWS::EC2::Host.AutoPlacement``.
        :param host_recovery: ``AWS::EC2::Host.HostRecovery``.
        '''
        props = CfnHostProps(
            availability_zone=availability_zone,
            instance_type=instance_type,
            auto_placement=auto_placement,
            host_recovery=host_recovery,
        )

        jsii.create(CfnHost, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="availabilityZone")
    def availability_zone(self) -> builtins.str:
        '''``AWS::EC2::Host.AvailabilityZone``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-availabilityzone
        '''
        return typing.cast(builtins.str, jsii.get(self, "availabilityZone"))

    @availability_zone.setter
    def availability_zone(self, value: builtins.str) -> None:
        jsii.set(self, "availabilityZone", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceType")
    def instance_type(self) -> builtins.str:
        '''``AWS::EC2::Host.InstanceType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-instancetype
        '''
        return typing.cast(builtins.str, jsii.get(self, "instanceType"))

    @instance_type.setter
    def instance_type(self, value: builtins.str) -> None:
        jsii.set(self, "instanceType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="autoPlacement")
    def auto_placement(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Host.AutoPlacement``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-autoplacement
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "autoPlacement"))

    @auto_placement.setter
    def auto_placement(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "autoPlacement", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="hostRecovery")
    def host_recovery(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Host.HostRecovery``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-hostrecovery
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "hostRecovery"))

    @host_recovery.setter
    def host_recovery(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "hostRecovery", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnHostProps",
    jsii_struct_bases=[],
    name_mapping={
        "availability_zone": "availabilityZone",
        "instance_type": "instanceType",
        "auto_placement": "autoPlacement",
        "host_recovery": "hostRecovery",
    },
)
class CfnHostProps:
    def __init__(
        self,
        *,
        availability_zone: builtins.str,
        instance_type: builtins.str,
        auto_placement: typing.Optional[builtins.str] = None,
        host_recovery: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::Host``.

        :param availability_zone: ``AWS::EC2::Host.AvailabilityZone``.
        :param instance_type: ``AWS::EC2::Host.InstanceType``.
        :param auto_placement: ``AWS::EC2::Host.AutoPlacement``.
        :param host_recovery: ``AWS::EC2::Host.HostRecovery``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "availability_zone": availability_zone,
            "instance_type": instance_type,
        }
        if auto_placement is not None:
            self._values["auto_placement"] = auto_placement
        if host_recovery is not None:
            self._values["host_recovery"] = host_recovery

    @builtins.property
    def availability_zone(self) -> builtins.str:
        '''``AWS::EC2::Host.AvailabilityZone``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-availabilityzone
        '''
        result = self._values.get("availability_zone")
        assert result is not None, "Required property 'availability_zone' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def instance_type(self) -> builtins.str:
        '''``AWS::EC2::Host.InstanceType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-instancetype
        '''
        result = self._values.get("instance_type")
        assert result is not None, "Required property 'instance_type' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def auto_placement(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Host.AutoPlacement``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-autoplacement
        '''
        result = self._values.get("auto_placement")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def host_recovery(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Host.HostRecovery``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-host.html#cfn-ec2-host-hostrecovery
        '''
        result = self._values.get("host_recovery")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnHostProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnInstance(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnInstance",
):
    '''A CloudFormation ``AWS::EC2::Instance``.

    :cloudformationResource: AWS::EC2::Instance
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        additional_info: typing.Optional[builtins.str] = None,
        affinity: typing.Optional[builtins.str] = None,
        availability_zone: typing.Optional[builtins.str] = None,
        block_device_mappings: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.BlockDeviceMappingProperty"]]]] = None,
        cpu_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.CpuOptionsProperty"]] = None,
        credit_specification: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.CreditSpecificationProperty"]] = None,
        disable_api_termination: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ebs_optimized: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        elastic_gpu_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.ElasticGpuSpecificationProperty"]]]] = None,
        elastic_inference_accelerators: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.ElasticInferenceAcceleratorProperty"]]]] = None,
        enclave_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.EnclaveOptionsProperty"]] = None,
        hibernation_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.HibernationOptionsProperty"]] = None,
        host_id: typing.Optional[builtins.str] = None,
        host_resource_group_arn: typing.Optional[builtins.str] = None,
        iam_instance_profile: typing.Optional[builtins.str] = None,
        image_id: typing.Optional[builtins.str] = None,
        instance_initiated_shutdown_behavior: typing.Optional[builtins.str] = None,
        instance_type: typing.Optional[builtins.str] = None,
        ipv6_address_count: typing.Optional[jsii.Number] = None,
        ipv6_addresses: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.InstanceIpv6AddressProperty"]]]] = None,
        kernel_id: typing.Optional[builtins.str] = None,
        key_name: typing.Optional[builtins.str] = None,
        launch_template: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.LaunchTemplateSpecificationProperty"]] = None,
        license_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.LicenseSpecificationProperty"]]]] = None,
        monitoring: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        network_interfaces: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.NetworkInterfaceProperty"]]]] = None,
        placement_group_name: typing.Optional[builtins.str] = None,
        private_ip_address: typing.Optional[builtins.str] = None,
        ramdisk_id: typing.Optional[builtins.str] = None,
        security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        source_dest_check: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ssm_associations: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.SsmAssociationProperty"]]]] = None,
        subnet_id: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
        tenancy: typing.Optional[builtins.str] = None,
        user_data: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.VolumeProperty"]]]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::Instance``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param additional_info: ``AWS::EC2::Instance.AdditionalInfo``.
        :param affinity: ``AWS::EC2::Instance.Affinity``.
        :param availability_zone: ``AWS::EC2::Instance.AvailabilityZone``.
        :param block_device_mappings: ``AWS::EC2::Instance.BlockDeviceMappings``.
        :param cpu_options: ``AWS::EC2::Instance.CpuOptions``.
        :param credit_specification: ``AWS::EC2::Instance.CreditSpecification``.
        :param disable_api_termination: ``AWS::EC2::Instance.DisableApiTermination``.
        :param ebs_optimized: ``AWS::EC2::Instance.EbsOptimized``.
        :param elastic_gpu_specifications: ``AWS::EC2::Instance.ElasticGpuSpecifications``.
        :param elastic_inference_accelerators: ``AWS::EC2::Instance.ElasticInferenceAccelerators``.
        :param enclave_options: ``AWS::EC2::Instance.EnclaveOptions``.
        :param hibernation_options: ``AWS::EC2::Instance.HibernationOptions``.
        :param host_id: ``AWS::EC2::Instance.HostId``.
        :param host_resource_group_arn: ``AWS::EC2::Instance.HostResourceGroupArn``.
        :param iam_instance_profile: ``AWS::EC2::Instance.IamInstanceProfile``.
        :param image_id: ``AWS::EC2::Instance.ImageId``.
        :param instance_initiated_shutdown_behavior: ``AWS::EC2::Instance.InstanceInitiatedShutdownBehavior``.
        :param instance_type: ``AWS::EC2::Instance.InstanceType``.
        :param ipv6_address_count: ``AWS::EC2::Instance.Ipv6AddressCount``.
        :param ipv6_addresses: ``AWS::EC2::Instance.Ipv6Addresses``.
        :param kernel_id: ``AWS::EC2::Instance.KernelId``.
        :param key_name: ``AWS::EC2::Instance.KeyName``.
        :param launch_template: ``AWS::EC2::Instance.LaunchTemplate``.
        :param license_specifications: ``AWS::EC2::Instance.LicenseSpecifications``.
        :param monitoring: ``AWS::EC2::Instance.Monitoring``.
        :param network_interfaces: ``AWS::EC2::Instance.NetworkInterfaces``.
        :param placement_group_name: ``AWS::EC2::Instance.PlacementGroupName``.
        :param private_ip_address: ``AWS::EC2::Instance.PrivateIpAddress``.
        :param ramdisk_id: ``AWS::EC2::Instance.RamdiskId``.
        :param security_group_ids: ``AWS::EC2::Instance.SecurityGroupIds``.
        :param security_groups: ``AWS::EC2::Instance.SecurityGroups``.
        :param source_dest_check: ``AWS::EC2::Instance.SourceDestCheck``.
        :param ssm_associations: ``AWS::EC2::Instance.SsmAssociations``.
        :param subnet_id: ``AWS::EC2::Instance.SubnetId``.
        :param tags: ``AWS::EC2::Instance.Tags``.
        :param tenancy: ``AWS::EC2::Instance.Tenancy``.
        :param user_data: ``AWS::EC2::Instance.UserData``.
        :param volumes: ``AWS::EC2::Instance.Volumes``.
        '''
        props = CfnInstanceProps(
            additional_info=additional_info,
            affinity=affinity,
            availability_zone=availability_zone,
            block_device_mappings=block_device_mappings,
            cpu_options=cpu_options,
            credit_specification=credit_specification,
            disable_api_termination=disable_api_termination,
            ebs_optimized=ebs_optimized,
            elastic_gpu_specifications=elastic_gpu_specifications,
            elastic_inference_accelerators=elastic_inference_accelerators,
            enclave_options=enclave_options,
            hibernation_options=hibernation_options,
            host_id=host_id,
            host_resource_group_arn=host_resource_group_arn,
            iam_instance_profile=iam_instance_profile,
            image_id=image_id,
            instance_initiated_shutdown_behavior=instance_initiated_shutdown_behavior,
            instance_type=instance_type,
            ipv6_address_count=ipv6_address_count,
            ipv6_addresses=ipv6_addresses,
            kernel_id=kernel_id,
            key_name=key_name,
            launch_template=launch_template,
            license_specifications=license_specifications,
            monitoring=monitoring,
            network_interfaces=network_interfaces,
            placement_group_name=placement_group_name,
            private_ip_address=private_ip_address,
            ramdisk_id=ramdisk_id,
            security_group_ids=security_group_ids,
            security_groups=security_groups,
            source_dest_check=source_dest_check,
            ssm_associations=ssm_associations,
            subnet_id=subnet_id,
            tags=tags,
            tenancy=tenancy,
            user_data=user_data,
            volumes=volumes,
        )

        jsii.create(CfnInstance, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrAvailabilityZone")
    def attr_availability_zone(self) -> builtins.str:
        '''
        :cloudformationAttribute: AvailabilityZone
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrAvailabilityZone"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrPrivateDnsName")
    def attr_private_dns_name(self) -> builtins.str:
        '''
        :cloudformationAttribute: PrivateDnsName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrPrivateDnsName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrPrivateIp")
    def attr_private_ip(self) -> builtins.str:
        '''
        :cloudformationAttribute: PrivateIp
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrPrivateIp"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrPublicDnsName")
    def attr_public_dns_name(self) -> builtins.str:
        '''
        :cloudformationAttribute: PublicDnsName
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrPublicDnsName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrPublicIp")
    def attr_public_ip(self) -> builtins.str:
        '''
        :cloudformationAttribute: PublicIp
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrPublicIp"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''``AWS::EC2::Instance.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="additionalInfo")
    def additional_info(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.AdditionalInfo``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-additionalinfo
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "additionalInfo"))

    @additional_info.setter
    def additional_info(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "additionalInfo", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="affinity")
    def affinity(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.Affinity``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-affinity
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "affinity"))

    @affinity.setter
    def affinity(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "affinity", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="availabilityZone")
    def availability_zone(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.AvailabilityZone``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-availabilityzone
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "availabilityZone"))

    @availability_zone.setter
    def availability_zone(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "availabilityZone", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="blockDeviceMappings")
    def block_device_mappings(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.BlockDeviceMappingProperty"]]]]:
        '''``AWS::EC2::Instance.BlockDeviceMappings``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-blockdevicemappings
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.BlockDeviceMappingProperty"]]]], jsii.get(self, "blockDeviceMappings"))

    @block_device_mappings.setter
    def block_device_mappings(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.BlockDeviceMappingProperty"]]]],
    ) -> None:
        jsii.set(self, "blockDeviceMappings", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cpuOptions")
    def cpu_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.CpuOptionsProperty"]]:
        '''``AWS::EC2::Instance.CpuOptions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-cpuoptions
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.CpuOptionsProperty"]], jsii.get(self, "cpuOptions"))

    @cpu_options.setter
    def cpu_options(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.CpuOptionsProperty"]],
    ) -> None:
        jsii.set(self, "cpuOptions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="creditSpecification")
    def credit_specification(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.CreditSpecificationProperty"]]:
        '''``AWS::EC2::Instance.CreditSpecification``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-creditspecification
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.CreditSpecificationProperty"]], jsii.get(self, "creditSpecification"))

    @credit_specification.setter
    def credit_specification(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.CreditSpecificationProperty"]],
    ) -> None:
        jsii.set(self, "creditSpecification", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="disableApiTermination")
    def disable_api_termination(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::Instance.DisableApiTermination``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-disableapitermination
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "disableApiTermination"))

    @disable_api_termination.setter
    def disable_api_termination(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "disableApiTermination", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ebsOptimized")
    def ebs_optimized(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::Instance.EbsOptimized``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ebsoptimized
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "ebsOptimized"))

    @ebs_optimized.setter
    def ebs_optimized(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "ebsOptimized", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="elasticGpuSpecifications")
    def elastic_gpu_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.ElasticGpuSpecificationProperty"]]]]:
        '''``AWS::EC2::Instance.ElasticGpuSpecifications``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-elasticgpuspecifications
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.ElasticGpuSpecificationProperty"]]]], jsii.get(self, "elasticGpuSpecifications"))

    @elastic_gpu_specifications.setter
    def elastic_gpu_specifications(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.ElasticGpuSpecificationProperty"]]]],
    ) -> None:
        jsii.set(self, "elasticGpuSpecifications", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="elasticInferenceAccelerators")
    def elastic_inference_accelerators(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.ElasticInferenceAcceleratorProperty"]]]]:
        '''``AWS::EC2::Instance.ElasticInferenceAccelerators``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-elasticinferenceaccelerators
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.ElasticInferenceAcceleratorProperty"]]]], jsii.get(self, "elasticInferenceAccelerators"))

    @elastic_inference_accelerators.setter
    def elastic_inference_accelerators(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.ElasticInferenceAcceleratorProperty"]]]],
    ) -> None:
        jsii.set(self, "elasticInferenceAccelerators", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="enclaveOptions")
    def enclave_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.EnclaveOptionsProperty"]]:
        '''``AWS::EC2::Instance.EnclaveOptions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-enclaveoptions
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.EnclaveOptionsProperty"]], jsii.get(self, "enclaveOptions"))

    @enclave_options.setter
    def enclave_options(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.EnclaveOptionsProperty"]],
    ) -> None:
        jsii.set(self, "enclaveOptions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="hibernationOptions")
    def hibernation_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.HibernationOptionsProperty"]]:
        '''``AWS::EC2::Instance.HibernationOptions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-hibernationoptions
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.HibernationOptionsProperty"]], jsii.get(self, "hibernationOptions"))

    @hibernation_options.setter
    def hibernation_options(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.HibernationOptionsProperty"]],
    ) -> None:
        jsii.set(self, "hibernationOptions", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="hostId")
    def host_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.HostId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-hostid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "hostId"))

    @host_id.setter
    def host_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "hostId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="hostResourceGroupArn")
    def host_resource_group_arn(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.HostResourceGroupArn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-hostresourcegrouparn
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "hostResourceGroupArn"))

    @host_resource_group_arn.setter
    def host_resource_group_arn(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "hostResourceGroupArn", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="iamInstanceProfile")
    def iam_instance_profile(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.IamInstanceProfile``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-iaminstanceprofile
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "iamInstanceProfile"))

    @iam_instance_profile.setter
    def iam_instance_profile(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "iamInstanceProfile", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="imageId")
    def image_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.ImageId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-imageid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "imageId"))

    @image_id.setter
    def image_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "imageId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceInitiatedShutdownBehavior")
    def instance_initiated_shutdown_behavior(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.InstanceInitiatedShutdownBehavior``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-instanceinitiatedshutdownbehavior
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceInitiatedShutdownBehavior"))

    @instance_initiated_shutdown_behavior.setter
    def instance_initiated_shutdown_behavior(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        jsii.set(self, "instanceInitiatedShutdownBehavior", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceType")
    def instance_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.InstanceType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-instancetype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceType"))

    @instance_type.setter
    def instance_type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "instanceType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ipv6AddressCount")
    def ipv6_address_count(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::Instance.Ipv6AddressCount``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ipv6addresscount
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "ipv6AddressCount"))

    @ipv6_address_count.setter
    def ipv6_address_count(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "ipv6AddressCount", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ipv6Addresses")
    def ipv6_addresses(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.InstanceIpv6AddressProperty"]]]]:
        '''``AWS::EC2::Instance.Ipv6Addresses``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ipv6addresses
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.InstanceIpv6AddressProperty"]]]], jsii.get(self, "ipv6Addresses"))

    @ipv6_addresses.setter
    def ipv6_addresses(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.InstanceIpv6AddressProperty"]]]],
    ) -> None:
        jsii.set(self, "ipv6Addresses", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="kernelId")
    def kernel_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.KernelId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-kernelid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "kernelId"))

    @kernel_id.setter
    def kernel_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "kernelId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="keyName")
    def key_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.KeyName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-keyname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "keyName"))

    @key_name.setter
    def key_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "keyName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="launchTemplate")
    def launch_template(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.LaunchTemplateSpecificationProperty"]]:
        '''``AWS::EC2::Instance.LaunchTemplate``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-launchtemplate
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.LaunchTemplateSpecificationProperty"]], jsii.get(self, "launchTemplate"))

    @launch_template.setter
    def launch_template(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.LaunchTemplateSpecificationProperty"]],
    ) -> None:
        jsii.set(self, "launchTemplate", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="licenseSpecifications")
    def license_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.LicenseSpecificationProperty"]]]]:
        '''``AWS::EC2::Instance.LicenseSpecifications``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-licensespecifications
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.LicenseSpecificationProperty"]]]], jsii.get(self, "licenseSpecifications"))

    @license_specifications.setter
    def license_specifications(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.LicenseSpecificationProperty"]]]],
    ) -> None:
        jsii.set(self, "licenseSpecifications", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="monitoring")
    def monitoring(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::Instance.Monitoring``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-monitoring
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "monitoring"))

    @monitoring.setter
    def monitoring(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "monitoring", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkInterfaces")
    def network_interfaces(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.NetworkInterfaceProperty"]]]]:
        '''``AWS::EC2::Instance.NetworkInterfaces``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-networkinterfaces
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.NetworkInterfaceProperty"]]]], jsii.get(self, "networkInterfaces"))

    @network_interfaces.setter
    def network_interfaces(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.NetworkInterfaceProperty"]]]],
    ) -> None:
        jsii.set(self, "networkInterfaces", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="placementGroupName")
    def placement_group_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.PlacementGroupName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-placementgroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "placementGroupName"))

    @placement_group_name.setter
    def placement_group_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "placementGroupName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="privateIpAddress")
    def private_ip_address(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.PrivateIpAddress``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-privateipaddress
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "privateIpAddress"))

    @private_ip_address.setter
    def private_ip_address(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "privateIpAddress", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ramdiskId")
    def ramdisk_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.RamdiskId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ramdiskid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ramdiskId"))

    @ramdisk_id.setter
    def ramdisk_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "ramdiskId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="securityGroupIds")
    def security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::Instance.SecurityGroupIds``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-securitygroupids
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "securityGroupIds"))

    @security_group_ids.setter
    def security_group_ids(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "securityGroupIds", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="securityGroups")
    def security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::Instance.SecurityGroups``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-securitygroups
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "securityGroups"))

    @security_groups.setter
    def security_groups(
        self,
        value: typing.Optional[typing.List[builtins.str]],
    ) -> None:
        jsii.set(self, "securityGroups", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceDestCheck")
    def source_dest_check(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::Instance.SourceDestCheck``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-sourcedestcheck
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "sourceDestCheck"))

    @source_dest_check.setter
    def source_dest_check(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "sourceDestCheck", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ssmAssociations")
    def ssm_associations(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.SsmAssociationProperty"]]]]:
        '''``AWS::EC2::Instance.SsmAssociations``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ssmassociations
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.SsmAssociationProperty"]]]], jsii.get(self, "ssmAssociations"))

    @ssm_associations.setter
    def ssm_associations(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.SsmAssociationProperty"]]]],
    ) -> None:
        jsii.set(self, "ssmAssociations", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="subnetId")
    def subnet_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.SubnetId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-subnetid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "subnetId"))

    @subnet_id.setter
    def subnet_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "subnetId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tenancy")
    def tenancy(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.Tenancy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-tenancy
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "tenancy"))

    @tenancy.setter
    def tenancy(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "tenancy", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="userData")
    def user_data(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.UserData``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-userdata
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "userData"))

    @user_data.setter
    def user_data(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "userData", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="volumes")
    def volumes(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.VolumeProperty"]]]]:
        '''``AWS::EC2::Instance.Volumes``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-volumes
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.VolumeProperty"]]]], jsii.get(self, "volumes"))

    @volumes.setter
    def volumes(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.VolumeProperty"]]]],
    ) -> None:
        jsii.set(self, "volumes", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.AssociationParameterProperty",
        jsii_struct_bases=[],
        name_mapping={"key": "key", "value": "value"},
    )
    class AssociationParameterProperty:
        def __init__(
            self,
            *,
            key: builtins.str,
            value: typing.Sequence[builtins.str],
        ) -> None:
            '''
            :param key: ``CfnInstance.AssociationParameterProperty.Key``.
            :param value: ``CfnInstance.AssociationParameterProperty.Value``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-ssmassociations-associationparameters.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "key": key,
                "value": value,
            }

        @builtins.property
        def key(self) -> builtins.str:
            '''``CfnInstance.AssociationParameterProperty.Key``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-ssmassociations-associationparameters.html#cfn-ec2-instance-ssmassociations-associationparameters-key
            '''
            result = self._values.get("key")
            assert result is not None, "Required property 'key' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def value(self) -> typing.List[builtins.str]:
            '''``CfnInstance.AssociationParameterProperty.Value``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-ssmassociations-associationparameters.html#cfn-ec2-instance-ssmassociations-associationparameters-value
            '''
            result = self._values.get("value")
            assert result is not None, "Required property 'value' is missing"
            return typing.cast(typing.List[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AssociationParameterProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.BlockDeviceMappingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "device_name": "deviceName",
            "ebs": "ebs",
            "no_device": "noDevice",
            "virtual_name": "virtualName",
        },
    )
    class BlockDeviceMappingProperty:
        def __init__(
            self,
            *,
            device_name: builtins.str,
            ebs: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.EbsProperty"]] = None,
            no_device: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.NoDeviceProperty"]] = None,
            virtual_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param device_name: ``CfnInstance.BlockDeviceMappingProperty.DeviceName``.
            :param ebs: ``CfnInstance.BlockDeviceMappingProperty.Ebs``.
            :param no_device: ``CfnInstance.BlockDeviceMappingProperty.NoDevice``.
            :param virtual_name: ``CfnInstance.BlockDeviceMappingProperty.VirtualName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-mapping.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "device_name": device_name,
            }
            if ebs is not None:
                self._values["ebs"] = ebs
            if no_device is not None:
                self._values["no_device"] = no_device
            if virtual_name is not None:
                self._values["virtual_name"] = virtual_name

        @builtins.property
        def device_name(self) -> builtins.str:
            '''``CfnInstance.BlockDeviceMappingProperty.DeviceName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-mapping.html#cfn-ec2-blockdev-mapping-devicename
            '''
            result = self._values.get("device_name")
            assert result is not None, "Required property 'device_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def ebs(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.EbsProperty"]]:
            '''``CfnInstance.BlockDeviceMappingProperty.Ebs``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-mapping.html#cfn-ec2-blockdev-mapping-ebs
            '''
            result = self._values.get("ebs")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.EbsProperty"]], result)

        @builtins.property
        def no_device(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.NoDeviceProperty"]]:
            '''``CfnInstance.BlockDeviceMappingProperty.NoDevice``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-mapping.html#cfn-ec2-blockdev-mapping-nodevice
            '''
            result = self._values.get("no_device")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.NoDeviceProperty"]], result)

        @builtins.property
        def virtual_name(self) -> typing.Optional[builtins.str]:
            '''``CfnInstance.BlockDeviceMappingProperty.VirtualName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-mapping.html#cfn-ec2-blockdev-mapping-virtualname
            '''
            result = self._values.get("virtual_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BlockDeviceMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.CpuOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={"core_count": "coreCount", "threads_per_core": "threadsPerCore"},
    )
    class CpuOptionsProperty:
        def __init__(
            self,
            *,
            core_count: typing.Optional[jsii.Number] = None,
            threads_per_core: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param core_count: ``CfnInstance.CpuOptionsProperty.CoreCount``.
            :param threads_per_core: ``CfnInstance.CpuOptionsProperty.ThreadsPerCore``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-cpuoptions.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if core_count is not None:
                self._values["core_count"] = core_count
            if threads_per_core is not None:
                self._values["threads_per_core"] = threads_per_core

        @builtins.property
        def core_count(self) -> typing.Optional[jsii.Number]:
            '''``CfnInstance.CpuOptionsProperty.CoreCount``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-cpuoptions.html#cfn-ec2-instance-cpuoptions-corecount
            '''
            result = self._values.get("core_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def threads_per_core(self) -> typing.Optional[jsii.Number]:
            '''``CfnInstance.CpuOptionsProperty.ThreadsPerCore``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-cpuoptions.html#cfn-ec2-instance-cpuoptions-threadspercore
            '''
            result = self._values.get("threads_per_core")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CpuOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.CreditSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"cpu_credits": "cpuCredits"},
    )
    class CreditSpecificationProperty:
        def __init__(
            self,
            *,
            cpu_credits: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param cpu_credits: ``CfnInstance.CreditSpecificationProperty.CPUCredits``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-creditspecification.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if cpu_credits is not None:
                self._values["cpu_credits"] = cpu_credits

        @builtins.property
        def cpu_credits(self) -> typing.Optional[builtins.str]:
            '''``CfnInstance.CreditSpecificationProperty.CPUCredits``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-creditspecification.html#cfn-ec2-instance-creditspecification-cpucredits
            '''
            result = self._values.get("cpu_credits")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CreditSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.EbsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "delete_on_termination": "deleteOnTermination",
            "encrypted": "encrypted",
            "iops": "iops",
            "kms_key_id": "kmsKeyId",
            "snapshot_id": "snapshotId",
            "volume_size": "volumeSize",
            "volume_type": "volumeType",
        },
    )
    class EbsProperty:
        def __init__(
            self,
            *,
            delete_on_termination: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            encrypted: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            iops: typing.Optional[jsii.Number] = None,
            kms_key_id: typing.Optional[builtins.str] = None,
            snapshot_id: typing.Optional[builtins.str] = None,
            volume_size: typing.Optional[jsii.Number] = None,
            volume_type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param delete_on_termination: ``CfnInstance.EbsProperty.DeleteOnTermination``.
            :param encrypted: ``CfnInstance.EbsProperty.Encrypted``.
            :param iops: ``CfnInstance.EbsProperty.Iops``.
            :param kms_key_id: ``CfnInstance.EbsProperty.KmsKeyId``.
            :param snapshot_id: ``CfnInstance.EbsProperty.SnapshotId``.
            :param volume_size: ``CfnInstance.EbsProperty.VolumeSize``.
            :param volume_type: ``CfnInstance.EbsProperty.VolumeType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-template.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if delete_on_termination is not None:
                self._values["delete_on_termination"] = delete_on_termination
            if encrypted is not None:
                self._values["encrypted"] = encrypted
            if iops is not None:
                self._values["iops"] = iops
            if kms_key_id is not None:
                self._values["kms_key_id"] = kms_key_id
            if snapshot_id is not None:
                self._values["snapshot_id"] = snapshot_id
            if volume_size is not None:
                self._values["volume_size"] = volume_size
            if volume_type is not None:
                self._values["volume_type"] = volume_type

        @builtins.property
        def delete_on_termination(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnInstance.EbsProperty.DeleteOnTermination``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-template.html#cfn-ec2-blockdev-template-deleteontermination
            '''
            result = self._values.get("delete_on_termination")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def encrypted(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnInstance.EbsProperty.Encrypted``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-template.html#cfn-ec2-blockdev-template-encrypted
            '''
            result = self._values.get("encrypted")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def iops(self) -> typing.Optional[jsii.Number]:
            '''``CfnInstance.EbsProperty.Iops``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-template.html#cfn-ec2-blockdev-template-iops
            '''
            result = self._values.get("iops")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def kms_key_id(self) -> typing.Optional[builtins.str]:
            '''``CfnInstance.EbsProperty.KmsKeyId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-template.html#cfn-ec2-instance-ebs-kmskeyid
            '''
            result = self._values.get("kms_key_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def snapshot_id(self) -> typing.Optional[builtins.str]:
            '''``CfnInstance.EbsProperty.SnapshotId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-template.html#cfn-ec2-blockdev-template-snapshotid
            '''
            result = self._values.get("snapshot_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def volume_size(self) -> typing.Optional[jsii.Number]:
            '''``CfnInstance.EbsProperty.VolumeSize``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-template.html#cfn-ec2-blockdev-template-volumesize
            '''
            result = self._values.get("volume_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def volume_type(self) -> typing.Optional[builtins.str]:
            '''``CfnInstance.EbsProperty.VolumeType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-blockdev-template.html#cfn-ec2-blockdev-template-volumetype
            '''
            result = self._values.get("volume_type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EbsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.ElasticGpuSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type"},
    )
    class ElasticGpuSpecificationProperty:
        def __init__(self, *, type: builtins.str) -> None:
            '''
            :param type: ``CfnInstance.ElasticGpuSpecificationProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-elasticgpuspecification.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "type": type,
            }

        @builtins.property
        def type(self) -> builtins.str:
            '''``CfnInstance.ElasticGpuSpecificationProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-elasticgpuspecification.html#cfn-ec2-instance-elasticgpuspecification-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ElasticGpuSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.ElasticInferenceAcceleratorProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type", "count": "count"},
    )
    class ElasticInferenceAcceleratorProperty:
        def __init__(
            self,
            *,
            type: builtins.str,
            count: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param type: ``CfnInstance.ElasticInferenceAcceleratorProperty.Type``.
            :param count: ``CfnInstance.ElasticInferenceAcceleratorProperty.Count``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-elasticinferenceaccelerator.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "type": type,
            }
            if count is not None:
                self._values["count"] = count

        @builtins.property
        def type(self) -> builtins.str:
            '''``CfnInstance.ElasticInferenceAcceleratorProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-elasticinferenceaccelerator.html#cfn-ec2-instance-elasticinferenceaccelerator-type
            '''
            result = self._values.get("type")
            assert result is not None, "Required property 'type' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def count(self) -> typing.Optional[jsii.Number]:
            '''``CfnInstance.ElasticInferenceAcceleratorProperty.Count``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-elasticinferenceaccelerator.html#cfn-ec2-instance-elasticinferenceaccelerator-count
            '''
            result = self._values.get("count")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ElasticInferenceAcceleratorProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.EnclaveOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={"enabled": "enabled"},
    )
    class EnclaveOptionsProperty:
        def __init__(
            self,
            *,
            enabled: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ) -> None:
            '''
            :param enabled: ``CfnInstance.EnclaveOptionsProperty.Enabled``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-enclaveoptions.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if enabled is not None:
                self._values["enabled"] = enabled

        @builtins.property
        def enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnInstance.EnclaveOptionsProperty.Enabled``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-enclaveoptions.html#cfn-ec2-instance-enclaveoptions-enabled
            '''
            result = self._values.get("enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EnclaveOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.HibernationOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={"configured": "configured"},
    )
    class HibernationOptionsProperty:
        def __init__(
            self,
            *,
            configured: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ) -> None:
            '''
            :param configured: ``CfnInstance.HibernationOptionsProperty.Configured``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-hibernationoptions.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if configured is not None:
                self._values["configured"] = configured

        @builtins.property
        def configured(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnInstance.HibernationOptionsProperty.Configured``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-hibernationoptions.html#cfn-ec2-instance-hibernationoptions-configured
            '''
            result = self._values.get("configured")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HibernationOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.InstanceIpv6AddressProperty",
        jsii_struct_bases=[],
        name_mapping={"ipv6_address": "ipv6Address"},
    )
    class InstanceIpv6AddressProperty:
        def __init__(self, *, ipv6_address: builtins.str) -> None:
            '''
            :param ipv6_address: ``CfnInstance.InstanceIpv6AddressProperty.Ipv6Address``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-instanceipv6address.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "ipv6_address": ipv6_address,
            }

        @builtins.property
        def ipv6_address(self) -> builtins.str:
            '''``CfnInstance.InstanceIpv6AddressProperty.Ipv6Address``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-instanceipv6address.html#cfn-ec2-instance-instanceipv6address-ipv6address
            '''
            result = self._values.get("ipv6_address")
            assert result is not None, "Required property 'ipv6_address' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceIpv6AddressProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.LaunchTemplateSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "version": "version",
            "launch_template_id": "launchTemplateId",
            "launch_template_name": "launchTemplateName",
        },
    )
    class LaunchTemplateSpecificationProperty:
        def __init__(
            self,
            *,
            version: builtins.str,
            launch_template_id: typing.Optional[builtins.str] = None,
            launch_template_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param version: ``CfnInstance.LaunchTemplateSpecificationProperty.Version``.
            :param launch_template_id: ``CfnInstance.LaunchTemplateSpecificationProperty.LaunchTemplateId``.
            :param launch_template_name: ``CfnInstance.LaunchTemplateSpecificationProperty.LaunchTemplateName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-launchtemplatespecification.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "version": version,
            }
            if launch_template_id is not None:
                self._values["launch_template_id"] = launch_template_id
            if launch_template_name is not None:
                self._values["launch_template_name"] = launch_template_name

        @builtins.property
        def version(self) -> builtins.str:
            '''``CfnInstance.LaunchTemplateSpecificationProperty.Version``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-launchtemplatespecification.html#cfn-ec2-instance-launchtemplatespecification-version
            '''
            result = self._values.get("version")
            assert result is not None, "Required property 'version' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def launch_template_id(self) -> typing.Optional[builtins.str]:
            '''``CfnInstance.LaunchTemplateSpecificationProperty.LaunchTemplateId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-launchtemplatespecification.html#cfn-ec2-instance-launchtemplatespecification-launchtemplateid
            '''
            result = self._values.get("launch_template_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def launch_template_name(self) -> typing.Optional[builtins.str]:
            '''``CfnInstance.LaunchTemplateSpecificationProperty.LaunchTemplateName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-launchtemplatespecification.html#cfn-ec2-instance-launchtemplatespecification-launchtemplatename
            '''
            result = self._values.get("launch_template_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LaunchTemplateSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.LicenseSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"license_configuration_arn": "licenseConfigurationArn"},
    )
    class LicenseSpecificationProperty:
        def __init__(self, *, license_configuration_arn: builtins.str) -> None:
            '''
            :param license_configuration_arn: ``CfnInstance.LicenseSpecificationProperty.LicenseConfigurationArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-licensespecification.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "license_configuration_arn": license_configuration_arn,
            }

        @builtins.property
        def license_configuration_arn(self) -> builtins.str:
            '''``CfnInstance.LicenseSpecificationProperty.LicenseConfigurationArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-licensespecification.html#cfn-ec2-instance-licensespecification-licenseconfigurationarn
            '''
            result = self._values.get("license_configuration_arn")
            assert result is not None, "Required property 'license_configuration_arn' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LicenseSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.NetworkInterfaceProperty",
        jsii_struct_bases=[],
        name_mapping={
            "device_index": "deviceIndex",
            "associate_public_ip_address": "associatePublicIpAddress",
            "delete_on_termination": "deleteOnTermination",
            "description": "description",
            "group_set": "groupSet",
            "ipv6_address_count": "ipv6AddressCount",
            "ipv6_addresses": "ipv6Addresses",
            "network_interface_id": "networkInterfaceId",
            "private_ip_address": "privateIpAddress",
            "private_ip_addresses": "privateIpAddresses",
            "secondary_private_ip_address_count": "secondaryPrivateIpAddressCount",
            "subnet_id": "subnetId",
        },
    )
    class NetworkInterfaceProperty:
        def __init__(
            self,
            *,
            device_index: builtins.str,
            associate_public_ip_address: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            delete_on_termination: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            description: typing.Optional[builtins.str] = None,
            group_set: typing.Optional[typing.Sequence[builtins.str]] = None,
            ipv6_address_count: typing.Optional[jsii.Number] = None,
            ipv6_addresses: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.InstanceIpv6AddressProperty"]]]] = None,
            network_interface_id: typing.Optional[builtins.str] = None,
            private_ip_address: typing.Optional[builtins.str] = None,
            private_ip_addresses: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.PrivateIpAddressSpecificationProperty"]]]] = None,
            secondary_private_ip_address_count: typing.Optional[jsii.Number] = None,
            subnet_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param device_index: ``CfnInstance.NetworkInterfaceProperty.DeviceIndex``.
            :param associate_public_ip_address: ``CfnInstance.NetworkInterfaceProperty.AssociatePublicIpAddress``.
            :param delete_on_termination: ``CfnInstance.NetworkInterfaceProperty.DeleteOnTermination``.
            :param description: ``CfnInstance.NetworkInterfaceProperty.Description``.
            :param group_set: ``CfnInstance.NetworkInterfaceProperty.GroupSet``.
            :param ipv6_address_count: ``CfnInstance.NetworkInterfaceProperty.Ipv6AddressCount``.
            :param ipv6_addresses: ``CfnInstance.NetworkInterfaceProperty.Ipv6Addresses``.
            :param network_interface_id: ``CfnInstance.NetworkInterfaceProperty.NetworkInterfaceId``.
            :param private_ip_address: ``CfnInstance.NetworkInterfaceProperty.PrivateIpAddress``.
            :param private_ip_addresses: ``CfnInstance.NetworkInterfaceProperty.PrivateIpAddresses``.
            :param secondary_private_ip_address_count: ``CfnInstance.NetworkInterfaceProperty.SecondaryPrivateIpAddressCount``.
            :param subnet_id: ``CfnInstance.NetworkInterfaceProperty.SubnetId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "device_index": device_index,
            }
            if associate_public_ip_address is not None:
                self._values["associate_public_ip_address"] = associate_public_ip_address
            if delete_on_termination is not None:
                self._values["delete_on_termination"] = delete_on_termination
            if description is not None:
                self._values["description"] = description
            if group_set is not None:
                self._values["group_set"] = group_set
            if ipv6_address_count is not None:
                self._values["ipv6_address_count"] = ipv6_address_count
            if ipv6_addresses is not None:
                self._values["ipv6_addresses"] = ipv6_addresses
            if network_interface_id is not None:
                self._values["network_interface_id"] = network_interface_id
            if private_ip_address is not None:
                self._values["private_ip_address"] = private_ip_address
            if private_ip_addresses is not None:
                self._values["private_ip_addresses"] = private_ip_addresses
            if secondary_private_ip_address_count is not None:
                self._values["secondary_private_ip_address_count"] = secondary_private_ip_address_count
            if subnet_id is not None:
                self._values["subnet_id"] = subnet_id

        @builtins.property
        def device_index(self) -> builtins.str:
            '''``CfnInstance.NetworkInterfaceProperty.DeviceIndex``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-deviceindex
            '''
            result = self._values.get("device_index")
            assert result is not None, "Required property 'device_index' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def associate_public_ip_address(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnInstance.NetworkInterfaceProperty.AssociatePublicIpAddress``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-associatepubip
            '''
            result = self._values.get("associate_public_ip_address")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def delete_on_termination(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnInstance.NetworkInterfaceProperty.DeleteOnTermination``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-delete
            '''
            result = self._values.get("delete_on_termination")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            '''``CfnInstance.NetworkInterfaceProperty.Description``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-description
            '''
            result = self._values.get("description")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def group_set(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnInstance.NetworkInterfaceProperty.GroupSet``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-groupset
            '''
            result = self._values.get("group_set")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def ipv6_address_count(self) -> typing.Optional[jsii.Number]:
            '''``CfnInstance.NetworkInterfaceProperty.Ipv6AddressCount``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#cfn-ec2-instance-networkinterface-ipv6addresscount
            '''
            result = self._values.get("ipv6_address_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def ipv6_addresses(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.InstanceIpv6AddressProperty"]]]]:
            '''``CfnInstance.NetworkInterfaceProperty.Ipv6Addresses``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#cfn-ec2-instance-networkinterface-ipv6addresses
            '''
            result = self._values.get("ipv6_addresses")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.InstanceIpv6AddressProperty"]]]], result)

        @builtins.property
        def network_interface_id(self) -> typing.Optional[builtins.str]:
            '''``CfnInstance.NetworkInterfaceProperty.NetworkInterfaceId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-network-iface
            '''
            result = self._values.get("network_interface_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def private_ip_address(self) -> typing.Optional[builtins.str]:
            '''``CfnInstance.NetworkInterfaceProperty.PrivateIpAddress``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-privateipaddress
            '''
            result = self._values.get("private_ip_address")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def private_ip_addresses(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.PrivateIpAddressSpecificationProperty"]]]]:
            '''``CfnInstance.NetworkInterfaceProperty.PrivateIpAddresses``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-privateipaddresses
            '''
            result = self._values.get("private_ip_addresses")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.PrivateIpAddressSpecificationProperty"]]]], result)

        @builtins.property
        def secondary_private_ip_address_count(self) -> typing.Optional[jsii.Number]:
            '''``CfnInstance.NetworkInterfaceProperty.SecondaryPrivateIpAddressCount``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-secondprivateip
            '''
            result = self._values.get("secondary_private_ip_address_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def subnet_id(self) -> typing.Optional[builtins.str]:
            '''``CfnInstance.NetworkInterfaceProperty.SubnetId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-iface-embedded.html#aws-properties-ec2-network-iface-embedded-subnetid
            '''
            result = self._values.get("subnet_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NetworkInterfaceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.NoDeviceProperty",
        jsii_struct_bases=[],
        name_mapping={},
    )
    class NoDeviceProperty:
        def __init__(self) -> None:
            '''
            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-nodevice.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NoDeviceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.PrivateIpAddressSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"primary": "primary", "private_ip_address": "privateIpAddress"},
    )
    class PrivateIpAddressSpecificationProperty:
        def __init__(
            self,
            *,
            primary: typing.Union[builtins.bool, aws_cdk.core.IResolvable],
            private_ip_address: builtins.str,
        ) -> None:
            '''
            :param primary: ``CfnInstance.PrivateIpAddressSpecificationProperty.Primary``.
            :param private_ip_address: ``CfnInstance.PrivateIpAddressSpecificationProperty.PrivateIpAddress``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-interface-privateipspec.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "primary": primary,
                "private_ip_address": private_ip_address,
            }

        @builtins.property
        def primary(self) -> typing.Union[builtins.bool, aws_cdk.core.IResolvable]:
            '''``CfnInstance.PrivateIpAddressSpecificationProperty.Primary``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-interface-privateipspec.html#cfn-ec2-networkinterface-privateipspecification-primary
            '''
            result = self._values.get("primary")
            assert result is not None, "Required property 'primary' is missing"
            return typing.cast(typing.Union[builtins.bool, aws_cdk.core.IResolvable], result)

        @builtins.property
        def private_ip_address(self) -> builtins.str:
            '''``CfnInstance.PrivateIpAddressSpecificationProperty.PrivateIpAddress``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-interface-privateipspec.html#cfn-ec2-networkinterface-privateipspecification-privateipaddress
            '''
            result = self._values.get("private_ip_address")
            assert result is not None, "Required property 'private_ip_address' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PrivateIpAddressSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.SsmAssociationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "document_name": "documentName",
            "association_parameters": "associationParameters",
        },
    )
    class SsmAssociationProperty:
        def __init__(
            self,
            *,
            document_name: builtins.str,
            association_parameters: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.AssociationParameterProperty"]]]] = None,
        ) -> None:
            '''
            :param document_name: ``CfnInstance.SsmAssociationProperty.DocumentName``.
            :param association_parameters: ``CfnInstance.SsmAssociationProperty.AssociationParameters``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-ssmassociations.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "document_name": document_name,
            }
            if association_parameters is not None:
                self._values["association_parameters"] = association_parameters

        @builtins.property
        def document_name(self) -> builtins.str:
            '''``CfnInstance.SsmAssociationProperty.DocumentName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-ssmassociations.html#cfn-ec2-instance-ssmassociations-documentname
            '''
            result = self._values.get("document_name")
            assert result is not None, "Required property 'document_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def association_parameters(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.AssociationParameterProperty"]]]]:
            '''``CfnInstance.SsmAssociationProperty.AssociationParameters``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance-ssmassociations.html#cfn-ec2-instance-ssmassociations-associationparameters
            '''
            result = self._values.get("association_parameters")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnInstance.AssociationParameterProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SsmAssociationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnInstance.VolumeProperty",
        jsii_struct_bases=[],
        name_mapping={"device": "device", "volume_id": "volumeId"},
    )
    class VolumeProperty:
        def __init__(self, *, device: builtins.str, volume_id: builtins.str) -> None:
            '''
            :param device: ``CfnInstance.VolumeProperty.Device``.
            :param volume_id: ``CfnInstance.VolumeProperty.VolumeId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-mount-point.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "device": device,
                "volume_id": volume_id,
            }

        @builtins.property
        def device(self) -> builtins.str:
            '''``CfnInstance.VolumeProperty.Device``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-mount-point.html#cfn-ec2-mountpoint-device
            '''
            result = self._values.get("device")
            assert result is not None, "Required property 'device' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def volume_id(self) -> builtins.str:
            '''``CfnInstance.VolumeProperty.VolumeId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-mount-point.html#cfn-ec2-mountpoint-volumeid
            '''
            result = self._values.get("volume_id")
            assert result is not None, "Required property 'volume_id' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "VolumeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnInstanceProps",
    jsii_struct_bases=[],
    name_mapping={
        "additional_info": "additionalInfo",
        "affinity": "affinity",
        "availability_zone": "availabilityZone",
        "block_device_mappings": "blockDeviceMappings",
        "cpu_options": "cpuOptions",
        "credit_specification": "creditSpecification",
        "disable_api_termination": "disableApiTermination",
        "ebs_optimized": "ebsOptimized",
        "elastic_gpu_specifications": "elasticGpuSpecifications",
        "elastic_inference_accelerators": "elasticInferenceAccelerators",
        "enclave_options": "enclaveOptions",
        "hibernation_options": "hibernationOptions",
        "host_id": "hostId",
        "host_resource_group_arn": "hostResourceGroupArn",
        "iam_instance_profile": "iamInstanceProfile",
        "image_id": "imageId",
        "instance_initiated_shutdown_behavior": "instanceInitiatedShutdownBehavior",
        "instance_type": "instanceType",
        "ipv6_address_count": "ipv6AddressCount",
        "ipv6_addresses": "ipv6Addresses",
        "kernel_id": "kernelId",
        "key_name": "keyName",
        "launch_template": "launchTemplate",
        "license_specifications": "licenseSpecifications",
        "monitoring": "monitoring",
        "network_interfaces": "networkInterfaces",
        "placement_group_name": "placementGroupName",
        "private_ip_address": "privateIpAddress",
        "ramdisk_id": "ramdiskId",
        "security_group_ids": "securityGroupIds",
        "security_groups": "securityGroups",
        "source_dest_check": "sourceDestCheck",
        "ssm_associations": "ssmAssociations",
        "subnet_id": "subnetId",
        "tags": "tags",
        "tenancy": "tenancy",
        "user_data": "userData",
        "volumes": "volumes",
    },
)
class CfnInstanceProps:
    def __init__(
        self,
        *,
        additional_info: typing.Optional[builtins.str] = None,
        affinity: typing.Optional[builtins.str] = None,
        availability_zone: typing.Optional[builtins.str] = None,
        block_device_mappings: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnInstance.BlockDeviceMappingProperty]]]] = None,
        cpu_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.CpuOptionsProperty]] = None,
        credit_specification: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.CreditSpecificationProperty]] = None,
        disable_api_termination: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ebs_optimized: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        elastic_gpu_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnInstance.ElasticGpuSpecificationProperty]]]] = None,
        elastic_inference_accelerators: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnInstance.ElasticInferenceAcceleratorProperty]]]] = None,
        enclave_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.EnclaveOptionsProperty]] = None,
        hibernation_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.HibernationOptionsProperty]] = None,
        host_id: typing.Optional[builtins.str] = None,
        host_resource_group_arn: typing.Optional[builtins.str] = None,
        iam_instance_profile: typing.Optional[builtins.str] = None,
        image_id: typing.Optional[builtins.str] = None,
        instance_initiated_shutdown_behavior: typing.Optional[builtins.str] = None,
        instance_type: typing.Optional[builtins.str] = None,
        ipv6_address_count: typing.Optional[jsii.Number] = None,
        ipv6_addresses: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnInstance.InstanceIpv6AddressProperty]]]] = None,
        kernel_id: typing.Optional[builtins.str] = None,
        key_name: typing.Optional[builtins.str] = None,
        launch_template: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.LaunchTemplateSpecificationProperty]] = None,
        license_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnInstance.LicenseSpecificationProperty]]]] = None,
        monitoring: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        network_interfaces: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnInstance.NetworkInterfaceProperty]]]] = None,
        placement_group_name: typing.Optional[builtins.str] = None,
        private_ip_address: typing.Optional[builtins.str] = None,
        ramdisk_id: typing.Optional[builtins.str] = None,
        security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
        security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
        source_dest_check: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ssm_associations: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnInstance.SsmAssociationProperty]]]] = None,
        subnet_id: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
        tenancy: typing.Optional[builtins.str] = None,
        user_data: typing.Optional[builtins.str] = None,
        volumes: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnInstance.VolumeProperty]]]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::Instance``.

        :param additional_info: ``AWS::EC2::Instance.AdditionalInfo``.
        :param affinity: ``AWS::EC2::Instance.Affinity``.
        :param availability_zone: ``AWS::EC2::Instance.AvailabilityZone``.
        :param block_device_mappings: ``AWS::EC2::Instance.BlockDeviceMappings``.
        :param cpu_options: ``AWS::EC2::Instance.CpuOptions``.
        :param credit_specification: ``AWS::EC2::Instance.CreditSpecification``.
        :param disable_api_termination: ``AWS::EC2::Instance.DisableApiTermination``.
        :param ebs_optimized: ``AWS::EC2::Instance.EbsOptimized``.
        :param elastic_gpu_specifications: ``AWS::EC2::Instance.ElasticGpuSpecifications``.
        :param elastic_inference_accelerators: ``AWS::EC2::Instance.ElasticInferenceAccelerators``.
        :param enclave_options: ``AWS::EC2::Instance.EnclaveOptions``.
        :param hibernation_options: ``AWS::EC2::Instance.HibernationOptions``.
        :param host_id: ``AWS::EC2::Instance.HostId``.
        :param host_resource_group_arn: ``AWS::EC2::Instance.HostResourceGroupArn``.
        :param iam_instance_profile: ``AWS::EC2::Instance.IamInstanceProfile``.
        :param image_id: ``AWS::EC2::Instance.ImageId``.
        :param instance_initiated_shutdown_behavior: ``AWS::EC2::Instance.InstanceInitiatedShutdownBehavior``.
        :param instance_type: ``AWS::EC2::Instance.InstanceType``.
        :param ipv6_address_count: ``AWS::EC2::Instance.Ipv6AddressCount``.
        :param ipv6_addresses: ``AWS::EC2::Instance.Ipv6Addresses``.
        :param kernel_id: ``AWS::EC2::Instance.KernelId``.
        :param key_name: ``AWS::EC2::Instance.KeyName``.
        :param launch_template: ``AWS::EC2::Instance.LaunchTemplate``.
        :param license_specifications: ``AWS::EC2::Instance.LicenseSpecifications``.
        :param monitoring: ``AWS::EC2::Instance.Monitoring``.
        :param network_interfaces: ``AWS::EC2::Instance.NetworkInterfaces``.
        :param placement_group_name: ``AWS::EC2::Instance.PlacementGroupName``.
        :param private_ip_address: ``AWS::EC2::Instance.PrivateIpAddress``.
        :param ramdisk_id: ``AWS::EC2::Instance.RamdiskId``.
        :param security_group_ids: ``AWS::EC2::Instance.SecurityGroupIds``.
        :param security_groups: ``AWS::EC2::Instance.SecurityGroups``.
        :param source_dest_check: ``AWS::EC2::Instance.SourceDestCheck``.
        :param ssm_associations: ``AWS::EC2::Instance.SsmAssociations``.
        :param subnet_id: ``AWS::EC2::Instance.SubnetId``.
        :param tags: ``AWS::EC2::Instance.Tags``.
        :param tenancy: ``AWS::EC2::Instance.Tenancy``.
        :param user_data: ``AWS::EC2::Instance.UserData``.
        :param volumes: ``AWS::EC2::Instance.Volumes``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if additional_info is not None:
            self._values["additional_info"] = additional_info
        if affinity is not None:
            self._values["affinity"] = affinity
        if availability_zone is not None:
            self._values["availability_zone"] = availability_zone
        if block_device_mappings is not None:
            self._values["block_device_mappings"] = block_device_mappings
        if cpu_options is not None:
            self._values["cpu_options"] = cpu_options
        if credit_specification is not None:
            self._values["credit_specification"] = credit_specification
        if disable_api_termination is not None:
            self._values["disable_api_termination"] = disable_api_termination
        if ebs_optimized is not None:
            self._values["ebs_optimized"] = ebs_optimized
        if elastic_gpu_specifications is not None:
            self._values["elastic_gpu_specifications"] = elastic_gpu_specifications
        if elastic_inference_accelerators is not None:
            self._values["elastic_inference_accelerators"] = elastic_inference_accelerators
        if enclave_options is not None:
            self._values["enclave_options"] = enclave_options
        if hibernation_options is not None:
            self._values["hibernation_options"] = hibernation_options
        if host_id is not None:
            self._values["host_id"] = host_id
        if host_resource_group_arn is not None:
            self._values["host_resource_group_arn"] = host_resource_group_arn
        if iam_instance_profile is not None:
            self._values["iam_instance_profile"] = iam_instance_profile
        if image_id is not None:
            self._values["image_id"] = image_id
        if instance_initiated_shutdown_behavior is not None:
            self._values["instance_initiated_shutdown_behavior"] = instance_initiated_shutdown_behavior
        if instance_type is not None:
            self._values["instance_type"] = instance_type
        if ipv6_address_count is not None:
            self._values["ipv6_address_count"] = ipv6_address_count
        if ipv6_addresses is not None:
            self._values["ipv6_addresses"] = ipv6_addresses
        if kernel_id is not None:
            self._values["kernel_id"] = kernel_id
        if key_name is not None:
            self._values["key_name"] = key_name
        if launch_template is not None:
            self._values["launch_template"] = launch_template
        if license_specifications is not None:
            self._values["license_specifications"] = license_specifications
        if monitoring is not None:
            self._values["monitoring"] = monitoring
        if network_interfaces is not None:
            self._values["network_interfaces"] = network_interfaces
        if placement_group_name is not None:
            self._values["placement_group_name"] = placement_group_name
        if private_ip_address is not None:
            self._values["private_ip_address"] = private_ip_address
        if ramdisk_id is not None:
            self._values["ramdisk_id"] = ramdisk_id
        if security_group_ids is not None:
            self._values["security_group_ids"] = security_group_ids
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if source_dest_check is not None:
            self._values["source_dest_check"] = source_dest_check
        if ssm_associations is not None:
            self._values["ssm_associations"] = ssm_associations
        if subnet_id is not None:
            self._values["subnet_id"] = subnet_id
        if tags is not None:
            self._values["tags"] = tags
        if tenancy is not None:
            self._values["tenancy"] = tenancy
        if user_data is not None:
            self._values["user_data"] = user_data
        if volumes is not None:
            self._values["volumes"] = volumes

    @builtins.property
    def additional_info(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.AdditionalInfo``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-additionalinfo
        '''
        result = self._values.get("additional_info")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def affinity(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.Affinity``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-affinity
        '''
        result = self._values.get("affinity")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def availability_zone(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.AvailabilityZone``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-availabilityzone
        '''
        result = self._values.get("availability_zone")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def block_device_mappings(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.BlockDeviceMappingProperty]]]]:
        '''``AWS::EC2::Instance.BlockDeviceMappings``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-blockdevicemappings
        '''
        result = self._values.get("block_device_mappings")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.BlockDeviceMappingProperty]]]], result)

    @builtins.property
    def cpu_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.CpuOptionsProperty]]:
        '''``AWS::EC2::Instance.CpuOptions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-cpuoptions
        '''
        result = self._values.get("cpu_options")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.CpuOptionsProperty]], result)

    @builtins.property
    def credit_specification(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.CreditSpecificationProperty]]:
        '''``AWS::EC2::Instance.CreditSpecification``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-creditspecification
        '''
        result = self._values.get("credit_specification")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.CreditSpecificationProperty]], result)

    @builtins.property
    def disable_api_termination(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::Instance.DisableApiTermination``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-disableapitermination
        '''
        result = self._values.get("disable_api_termination")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def ebs_optimized(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::Instance.EbsOptimized``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ebsoptimized
        '''
        result = self._values.get("ebs_optimized")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def elastic_gpu_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.ElasticGpuSpecificationProperty]]]]:
        '''``AWS::EC2::Instance.ElasticGpuSpecifications``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-elasticgpuspecifications
        '''
        result = self._values.get("elastic_gpu_specifications")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.ElasticGpuSpecificationProperty]]]], result)

    @builtins.property
    def elastic_inference_accelerators(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.ElasticInferenceAcceleratorProperty]]]]:
        '''``AWS::EC2::Instance.ElasticInferenceAccelerators``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-elasticinferenceaccelerators
        '''
        result = self._values.get("elastic_inference_accelerators")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.ElasticInferenceAcceleratorProperty]]]], result)

    @builtins.property
    def enclave_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.EnclaveOptionsProperty]]:
        '''``AWS::EC2::Instance.EnclaveOptions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-enclaveoptions
        '''
        result = self._values.get("enclave_options")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.EnclaveOptionsProperty]], result)

    @builtins.property
    def hibernation_options(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.HibernationOptionsProperty]]:
        '''``AWS::EC2::Instance.HibernationOptions``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-hibernationoptions
        '''
        result = self._values.get("hibernation_options")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.HibernationOptionsProperty]], result)

    @builtins.property
    def host_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.HostId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-hostid
        '''
        result = self._values.get("host_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def host_resource_group_arn(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.HostResourceGroupArn``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-hostresourcegrouparn
        '''
        result = self._values.get("host_resource_group_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def iam_instance_profile(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.IamInstanceProfile``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-iaminstanceprofile
        '''
        result = self._values.get("iam_instance_profile")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def image_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.ImageId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-imageid
        '''
        result = self._values.get("image_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_initiated_shutdown_behavior(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.InstanceInitiatedShutdownBehavior``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-instanceinitiatedshutdownbehavior
        '''
        result = self._values.get("instance_initiated_shutdown_behavior")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.InstanceType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-instancetype
        '''
        result = self._values.get("instance_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ipv6_address_count(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::Instance.Ipv6AddressCount``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ipv6addresscount
        '''
        result = self._values.get("ipv6_address_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def ipv6_addresses(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.InstanceIpv6AddressProperty]]]]:
        '''``AWS::EC2::Instance.Ipv6Addresses``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ipv6addresses
        '''
        result = self._values.get("ipv6_addresses")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.InstanceIpv6AddressProperty]]]], result)

    @builtins.property
    def kernel_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.KernelId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-kernelid
        '''
        result = self._values.get("kernel_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def key_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.KeyName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-keyname
        '''
        result = self._values.get("key_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def launch_template(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.LaunchTemplateSpecificationProperty]]:
        '''``AWS::EC2::Instance.LaunchTemplate``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-launchtemplate
        '''
        result = self._values.get("launch_template")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnInstance.LaunchTemplateSpecificationProperty]], result)

    @builtins.property
    def license_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.LicenseSpecificationProperty]]]]:
        '''``AWS::EC2::Instance.LicenseSpecifications``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-licensespecifications
        '''
        result = self._values.get("license_specifications")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.LicenseSpecificationProperty]]]], result)

    @builtins.property
    def monitoring(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::Instance.Monitoring``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-monitoring
        '''
        result = self._values.get("monitoring")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def network_interfaces(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.NetworkInterfaceProperty]]]]:
        '''``AWS::EC2::Instance.NetworkInterfaces``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-networkinterfaces
        '''
        result = self._values.get("network_interfaces")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.NetworkInterfaceProperty]]]], result)

    @builtins.property
    def placement_group_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.PlacementGroupName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-placementgroupname
        '''
        result = self._values.get("placement_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def private_ip_address(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.PrivateIpAddress``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-privateipaddress
        '''
        result = self._values.get("private_ip_address")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ramdisk_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.RamdiskId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ramdiskid
        '''
        result = self._values.get("ramdisk_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::Instance.SecurityGroupIds``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-securitygroupids
        '''
        result = self._values.get("security_group_ids")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::Instance.SecurityGroups``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-securitygroups
        '''
        result = self._values.get("security_groups")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def source_dest_check(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::Instance.SourceDestCheck``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-sourcedestcheck
        '''
        result = self._values.get("source_dest_check")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def ssm_associations(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.SsmAssociationProperty]]]]:
        '''``AWS::EC2::Instance.SsmAssociations``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-ssmassociations
        '''
        result = self._values.get("ssm_associations")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.SsmAssociationProperty]]]], result)

    @builtins.property
    def subnet_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.SubnetId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-subnetid
        '''
        result = self._values.get("subnet_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''``AWS::EC2::Instance.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    @builtins.property
    def tenancy(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.Tenancy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-tenancy
        '''
        result = self._values.get("tenancy")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def user_data(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Instance.UserData``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-userdata
        '''
        result = self._values.get("user_data")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def volumes(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.VolumeProperty]]]]:
        '''``AWS::EC2::Instance.Volumes``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html#cfn-ec2-instance-volumes
        '''
        result = self._values.get("volumes")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnInstance.VolumeProperty]]]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnInstanceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnInternetGateway(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnInternetGateway",
):
    '''A CloudFormation ``AWS::EC2::InternetGateway``.

    :cloudformationResource: AWS::EC2::InternetGateway
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-internetgateway.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::InternetGateway``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param tags: ``AWS::EC2::InternetGateway.Tags``.
        '''
        props = CfnInternetGatewayProps(tags=tags)

        jsii.create(CfnInternetGateway, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''``AWS::EC2::InternetGateway.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-internetgateway.html#cfn-ec2-internetgateway-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnInternetGatewayProps",
    jsii_struct_bases=[],
    name_mapping={"tags": "tags"},
)
class CfnInternetGatewayProps:
    def __init__(
        self,
        *,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::InternetGateway``.

        :param tags: ``AWS::EC2::InternetGateway.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-internetgateway.html
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''``AWS::EC2::InternetGateway.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-internetgateway.html#cfn-ec2-internetgateway-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnInternetGatewayProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnLaunchTemplate(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate",
):
    '''A CloudFormation ``AWS::EC2::LaunchTemplate``.

    :cloudformationResource: AWS::EC2::LaunchTemplate
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        launch_template_data: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateDataProperty"]] = None,
        launch_template_name: typing.Optional[builtins.str] = None,
        tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateTagSpecificationProperty"]]]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::LaunchTemplate``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param launch_template_data: ``AWS::EC2::LaunchTemplate.LaunchTemplateData``.
        :param launch_template_name: ``AWS::EC2::LaunchTemplate.LaunchTemplateName``.
        :param tag_specifications: ``AWS::EC2::LaunchTemplate.TagSpecifications``.
        '''
        props = CfnLaunchTemplateProps(
            launch_template_data=launch_template_data,
            launch_template_name=launch_template_name,
            tag_specifications=tag_specifications,
        )

        jsii.create(CfnLaunchTemplate, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrDefaultVersionNumber")
    def attr_default_version_number(self) -> builtins.str:
        '''
        :cloudformationAttribute: DefaultVersionNumber
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrDefaultVersionNumber"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrLatestVersionNumber")
    def attr_latest_version_number(self) -> builtins.str:
        '''
        :cloudformationAttribute: LatestVersionNumber
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrLatestVersionNumber"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="launchTemplateData")
    def launch_template_data(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateDataProperty"]]:
        '''``AWS::EC2::LaunchTemplate.LaunchTemplateData``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html#cfn-ec2-launchtemplate-launchtemplatedata
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateDataProperty"]], jsii.get(self, "launchTemplateData"))

    @launch_template_data.setter
    def launch_template_data(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateDataProperty"]],
    ) -> None:
        jsii.set(self, "launchTemplateData", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="launchTemplateName")
    def launch_template_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::LaunchTemplate.LaunchTemplateName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html#cfn-ec2-launchtemplate-launchtemplatename
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "launchTemplateName"))

    @launch_template_name.setter
    def launch_template_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "launchTemplateName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tagSpecifications")
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateTagSpecificationProperty"]]]]:
        '''``AWS::EC2::LaunchTemplate.TagSpecifications``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html#cfn-ec2-launchtemplate-tagspecifications
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateTagSpecificationProperty"]]]], jsii.get(self, "tagSpecifications"))

    @tag_specifications.setter
    def tag_specifications(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateTagSpecificationProperty"]]]],
    ) -> None:
        jsii.set(self, "tagSpecifications", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.BlockDeviceMappingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "device_name": "deviceName",
            "ebs": "ebs",
            "no_device": "noDevice",
            "virtual_name": "virtualName",
        },
    )
    class BlockDeviceMappingProperty:
        def __init__(
            self,
            *,
            device_name: typing.Optional[builtins.str] = None,
            ebs: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.EbsProperty"]] = None,
            no_device: typing.Optional[builtins.str] = None,
            virtual_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param device_name: ``CfnLaunchTemplate.BlockDeviceMappingProperty.DeviceName``.
            :param ebs: ``CfnLaunchTemplate.BlockDeviceMappingProperty.Ebs``.
            :param no_device: ``CfnLaunchTemplate.BlockDeviceMappingProperty.NoDevice``.
            :param virtual_name: ``CfnLaunchTemplate.BlockDeviceMappingProperty.VirtualName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if device_name is not None:
                self._values["device_name"] = device_name
            if ebs is not None:
                self._values["ebs"] = ebs
            if no_device is not None:
                self._values["no_device"] = no_device
            if virtual_name is not None:
                self._values["virtual_name"] = virtual_name

        @builtins.property
        def device_name(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.BlockDeviceMappingProperty.DeviceName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping.html#cfn-ec2-launchtemplate-blockdevicemapping-devicename
            '''
            result = self._values.get("device_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ebs(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.EbsProperty"]]:
            '''``CfnLaunchTemplate.BlockDeviceMappingProperty.Ebs``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping.html#cfn-ec2-launchtemplate-blockdevicemapping-ebs
            '''
            result = self._values.get("ebs")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.EbsProperty"]], result)

        @builtins.property
        def no_device(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.BlockDeviceMappingProperty.NoDevice``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping.html#cfn-ec2-launchtemplate-blockdevicemapping-nodevice
            '''
            result = self._values.get("no_device")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def virtual_name(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.BlockDeviceMappingProperty.VirtualName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping.html#cfn-ec2-launchtemplate-blockdevicemapping-virtualname
            '''
            result = self._values.get("virtual_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BlockDeviceMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.CapacityReservationSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "capacity_reservation_preference": "capacityReservationPreference",
            "capacity_reservation_target": "capacityReservationTarget",
        },
    )
    class CapacityReservationSpecificationProperty:
        def __init__(
            self,
            *,
            capacity_reservation_preference: typing.Optional[builtins.str] = None,
            capacity_reservation_target: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.CapacityReservationTargetProperty"]] = None,
        ) -> None:
            '''
            :param capacity_reservation_preference: ``CfnLaunchTemplate.CapacityReservationSpecificationProperty.CapacityReservationPreference``.
            :param capacity_reservation_target: ``CfnLaunchTemplate.CapacityReservationSpecificationProperty.CapacityReservationTarget``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-capacityreservationspecification.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if capacity_reservation_preference is not None:
                self._values["capacity_reservation_preference"] = capacity_reservation_preference
            if capacity_reservation_target is not None:
                self._values["capacity_reservation_target"] = capacity_reservation_target

        @builtins.property
        def capacity_reservation_preference(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.CapacityReservationSpecificationProperty.CapacityReservationPreference``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-capacityreservationspecification.html#cfn-ec2-launchtemplate-launchtemplatedata-capacityreservationspecification-capacityreservationpreference
            '''
            result = self._values.get("capacity_reservation_preference")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def capacity_reservation_target(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.CapacityReservationTargetProperty"]]:
            '''``CfnLaunchTemplate.CapacityReservationSpecificationProperty.CapacityReservationTarget``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-capacityreservationspecification.html#cfn-ec2-launchtemplate-launchtemplatedata-capacityreservationspecification-capacityreservationtarget
            '''
            result = self._values.get("capacity_reservation_target")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.CapacityReservationTargetProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityReservationSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.CapacityReservationTargetProperty",
        jsii_struct_bases=[],
        name_mapping={
            "capacity_reservation_id": "capacityReservationId",
            "capacity_reservation_resource_group_arn": "capacityReservationResourceGroupArn",
        },
    )
    class CapacityReservationTargetProperty:
        def __init__(
            self,
            *,
            capacity_reservation_id: typing.Optional[builtins.str] = None,
            capacity_reservation_resource_group_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param capacity_reservation_id: ``CfnLaunchTemplate.CapacityReservationTargetProperty.CapacityReservationId``.
            :param capacity_reservation_resource_group_arn: ``CfnLaunchTemplate.CapacityReservationTargetProperty.CapacityReservationResourceGroupArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-capacityreservationtarget.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if capacity_reservation_id is not None:
                self._values["capacity_reservation_id"] = capacity_reservation_id
            if capacity_reservation_resource_group_arn is not None:
                self._values["capacity_reservation_resource_group_arn"] = capacity_reservation_resource_group_arn

        @builtins.property
        def capacity_reservation_id(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.CapacityReservationTargetProperty.CapacityReservationId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-capacityreservationtarget.html#cfn-ec2-launchtemplate-capacityreservationtarget-capacityreservationid
            '''
            result = self._values.get("capacity_reservation_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def capacity_reservation_resource_group_arn(
            self,
        ) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.CapacityReservationTargetProperty.CapacityReservationResourceGroupArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-capacityreservationtarget.html#cfn-ec2-launchtemplate-capacityreservationtarget-capacityreservationresourcegrouparn
            '''
            result = self._values.get("capacity_reservation_resource_group_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CapacityReservationTargetProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.CpuOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={"core_count": "coreCount", "threads_per_core": "threadsPerCore"},
    )
    class CpuOptionsProperty:
        def __init__(
            self,
            *,
            core_count: typing.Optional[jsii.Number] = None,
            threads_per_core: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param core_count: ``CfnLaunchTemplate.CpuOptionsProperty.CoreCount``.
            :param threads_per_core: ``CfnLaunchTemplate.CpuOptionsProperty.ThreadsPerCore``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-cpuoptions.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if core_count is not None:
                self._values["core_count"] = core_count
            if threads_per_core is not None:
                self._values["threads_per_core"] = threads_per_core

        @builtins.property
        def core_count(self) -> typing.Optional[jsii.Number]:
            '''``CfnLaunchTemplate.CpuOptionsProperty.CoreCount``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-cpuoptions.html#cfn-ec2-launchtemplate-launchtemplatedata-cpuoptions-corecount
            '''
            result = self._values.get("core_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def threads_per_core(self) -> typing.Optional[jsii.Number]:
            '''``CfnLaunchTemplate.CpuOptionsProperty.ThreadsPerCore``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-cpuoptions.html#cfn-ec2-launchtemplate-launchtemplatedata-cpuoptions-threadspercore
            '''
            result = self._values.get("threads_per_core")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CpuOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.CreditSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"cpu_credits": "cpuCredits"},
    )
    class CreditSpecificationProperty:
        def __init__(
            self,
            *,
            cpu_credits: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param cpu_credits: ``CfnLaunchTemplate.CreditSpecificationProperty.CpuCredits``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-creditspecification.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if cpu_credits is not None:
                self._values["cpu_credits"] = cpu_credits

        @builtins.property
        def cpu_credits(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.CreditSpecificationProperty.CpuCredits``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-creditspecification.html#cfn-ec2-launchtemplate-launchtemplatedata-creditspecification-cpucredits
            '''
            result = self._values.get("cpu_credits")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "CreditSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.EbsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "delete_on_termination": "deleteOnTermination",
            "encrypted": "encrypted",
            "iops": "iops",
            "kms_key_id": "kmsKeyId",
            "snapshot_id": "snapshotId",
            "throughput": "throughput",
            "volume_size": "volumeSize",
            "volume_type": "volumeType",
        },
    )
    class EbsProperty:
        def __init__(
            self,
            *,
            delete_on_termination: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            encrypted: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            iops: typing.Optional[jsii.Number] = None,
            kms_key_id: typing.Optional[builtins.str] = None,
            snapshot_id: typing.Optional[builtins.str] = None,
            throughput: typing.Optional[jsii.Number] = None,
            volume_size: typing.Optional[jsii.Number] = None,
            volume_type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param delete_on_termination: ``CfnLaunchTemplate.EbsProperty.DeleteOnTermination``.
            :param encrypted: ``CfnLaunchTemplate.EbsProperty.Encrypted``.
            :param iops: ``CfnLaunchTemplate.EbsProperty.Iops``.
            :param kms_key_id: ``CfnLaunchTemplate.EbsProperty.KmsKeyId``.
            :param snapshot_id: ``CfnLaunchTemplate.EbsProperty.SnapshotId``.
            :param throughput: ``CfnLaunchTemplate.EbsProperty.Throughput``.
            :param volume_size: ``CfnLaunchTemplate.EbsProperty.VolumeSize``.
            :param volume_type: ``CfnLaunchTemplate.EbsProperty.VolumeType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping-ebs.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if delete_on_termination is not None:
                self._values["delete_on_termination"] = delete_on_termination
            if encrypted is not None:
                self._values["encrypted"] = encrypted
            if iops is not None:
                self._values["iops"] = iops
            if kms_key_id is not None:
                self._values["kms_key_id"] = kms_key_id
            if snapshot_id is not None:
                self._values["snapshot_id"] = snapshot_id
            if throughput is not None:
                self._values["throughput"] = throughput
            if volume_size is not None:
                self._values["volume_size"] = volume_size
            if volume_type is not None:
                self._values["volume_type"] = volume_type

        @builtins.property
        def delete_on_termination(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnLaunchTemplate.EbsProperty.DeleteOnTermination``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping-ebs.html#cfn-ec2-launchtemplate-blockdevicemapping-ebs-deleteontermination
            '''
            result = self._values.get("delete_on_termination")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def encrypted(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnLaunchTemplate.EbsProperty.Encrypted``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping-ebs.html#cfn-ec2-launchtemplate-blockdevicemapping-ebs-encrypted
            '''
            result = self._values.get("encrypted")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def iops(self) -> typing.Optional[jsii.Number]:
            '''``CfnLaunchTemplate.EbsProperty.Iops``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping-ebs.html#cfn-ec2-launchtemplate-blockdevicemapping-ebs-iops
            '''
            result = self._values.get("iops")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def kms_key_id(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.EbsProperty.KmsKeyId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping-ebs.html#cfn-ec2-launchtemplate-blockdevicemapping-ebs-kmskeyid
            '''
            result = self._values.get("kms_key_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def snapshot_id(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.EbsProperty.SnapshotId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping-ebs.html#cfn-ec2-launchtemplate-blockdevicemapping-ebs-snapshotid
            '''
            result = self._values.get("snapshot_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def throughput(self) -> typing.Optional[jsii.Number]:
            '''``CfnLaunchTemplate.EbsProperty.Throughput``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping-ebs.html#cfn-ec2-launchtemplate-blockdevicemapping-ebs-throughput
            '''
            result = self._values.get("throughput")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def volume_size(self) -> typing.Optional[jsii.Number]:
            '''``CfnLaunchTemplate.EbsProperty.VolumeSize``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping-ebs.html#cfn-ec2-launchtemplate-blockdevicemapping-ebs-volumesize
            '''
            result = self._values.get("volume_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def volume_type(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.EbsProperty.VolumeType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-blockdevicemapping-ebs.html#cfn-ec2-launchtemplate-blockdevicemapping-ebs-volumetype
            '''
            result = self._values.get("volume_type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EbsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.ElasticGpuSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"type": "type"},
    )
    class ElasticGpuSpecificationProperty:
        def __init__(self, *, type: typing.Optional[builtins.str] = None) -> None:
            '''
            :param type: ``CfnLaunchTemplate.ElasticGpuSpecificationProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-elasticgpuspecification.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if type is not None:
                self._values["type"] = type

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.ElasticGpuSpecificationProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-elasticgpuspecification.html#cfn-ec2-launchtemplate-elasticgpuspecification-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ElasticGpuSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.EnclaveOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={"enabled": "enabled"},
    )
    class EnclaveOptionsProperty:
        def __init__(
            self,
            *,
            enabled: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ) -> None:
            '''
            :param enabled: ``CfnLaunchTemplate.EnclaveOptionsProperty.Enabled``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-enclaveoptions.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if enabled is not None:
                self._values["enabled"] = enabled

        @builtins.property
        def enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnLaunchTemplate.EnclaveOptionsProperty.Enabled``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-enclaveoptions.html#cfn-ec2-launchtemplate-launchtemplatedata-enclaveoptions-enabled
            '''
            result = self._values.get("enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EnclaveOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.HibernationOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={"configured": "configured"},
    )
    class HibernationOptionsProperty:
        def __init__(
            self,
            *,
            configured: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ) -> None:
            '''
            :param configured: ``CfnLaunchTemplate.HibernationOptionsProperty.Configured``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-hibernationoptions.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if configured is not None:
                self._values["configured"] = configured

        @builtins.property
        def configured(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnLaunchTemplate.HibernationOptionsProperty.Configured``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-hibernationoptions.html#cfn-ec2-launchtemplate-launchtemplatedata-hibernationoptions-configured
            '''
            result = self._values.get("configured")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "HibernationOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.IamInstanceProfileProperty",
        jsii_struct_bases=[],
        name_mapping={"arn": "arn", "name": "name"},
    )
    class IamInstanceProfileProperty:
        def __init__(
            self,
            *,
            arn: typing.Optional[builtins.str] = None,
            name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param arn: ``CfnLaunchTemplate.IamInstanceProfileProperty.Arn``.
            :param name: ``CfnLaunchTemplate.IamInstanceProfileProperty.Name``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-iaminstanceprofile.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if arn is not None:
                self._values["arn"] = arn
            if name is not None:
                self._values["name"] = name

        @builtins.property
        def arn(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.IamInstanceProfileProperty.Arn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-iaminstanceprofile.html#cfn-ec2-launchtemplate-launchtemplatedata-iaminstanceprofile-arn
            '''
            result = self._values.get("arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def name(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.IamInstanceProfileProperty.Name``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-iaminstanceprofile.html#cfn-ec2-launchtemplate-launchtemplatedata-iaminstanceprofile-name
            '''
            result = self._values.get("name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "IamInstanceProfileProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.InstanceMarketOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={"market_type": "marketType", "spot_options": "spotOptions"},
    )
    class InstanceMarketOptionsProperty:
        def __init__(
            self,
            *,
            market_type: typing.Optional[builtins.str] = None,
            spot_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.SpotOptionsProperty"]] = None,
        ) -> None:
            '''
            :param market_type: ``CfnLaunchTemplate.InstanceMarketOptionsProperty.MarketType``.
            :param spot_options: ``CfnLaunchTemplate.InstanceMarketOptionsProperty.SpotOptions``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-instancemarketoptions.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if market_type is not None:
                self._values["market_type"] = market_type
            if spot_options is not None:
                self._values["spot_options"] = spot_options

        @builtins.property
        def market_type(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.InstanceMarketOptionsProperty.MarketType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-instancemarketoptions.html#cfn-ec2-launchtemplate-launchtemplatedata-instancemarketoptions-markettype
            '''
            result = self._values.get("market_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def spot_options(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.SpotOptionsProperty"]]:
            '''``CfnLaunchTemplate.InstanceMarketOptionsProperty.SpotOptions``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-instancemarketoptions.html#cfn-ec2-launchtemplate-launchtemplatedata-instancemarketoptions-spotoptions
            '''
            result = self._values.get("spot_options")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.SpotOptionsProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceMarketOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.Ipv6AddProperty",
        jsii_struct_bases=[],
        name_mapping={"ipv6_address": "ipv6Address"},
    )
    class Ipv6AddProperty:
        def __init__(
            self,
            *,
            ipv6_address: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param ipv6_address: ``CfnLaunchTemplate.Ipv6AddProperty.Ipv6Address``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-ipv6add.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if ipv6_address is not None:
                self._values["ipv6_address"] = ipv6_address

        @builtins.property
        def ipv6_address(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.Ipv6AddProperty.Ipv6Address``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-ipv6add.html#cfn-ec2-launchtemplate-ipv6add-ipv6address
            '''
            result = self._values.get("ipv6_address")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "Ipv6AddProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.LaunchTemplateDataProperty",
        jsii_struct_bases=[],
        name_mapping={
            "block_device_mappings": "blockDeviceMappings",
            "capacity_reservation_specification": "capacityReservationSpecification",
            "cpu_options": "cpuOptions",
            "credit_specification": "creditSpecification",
            "disable_api_termination": "disableApiTermination",
            "ebs_optimized": "ebsOptimized",
            "elastic_gpu_specifications": "elasticGpuSpecifications",
            "elastic_inference_accelerators": "elasticInferenceAccelerators",
            "enclave_options": "enclaveOptions",
            "hibernation_options": "hibernationOptions",
            "iam_instance_profile": "iamInstanceProfile",
            "image_id": "imageId",
            "instance_initiated_shutdown_behavior": "instanceInitiatedShutdownBehavior",
            "instance_market_options": "instanceMarketOptions",
            "instance_type": "instanceType",
            "kernel_id": "kernelId",
            "key_name": "keyName",
            "license_specifications": "licenseSpecifications",
            "metadata_options": "metadataOptions",
            "monitoring": "monitoring",
            "network_interfaces": "networkInterfaces",
            "placement": "placement",
            "ram_disk_id": "ramDiskId",
            "security_group_ids": "securityGroupIds",
            "security_groups": "securityGroups",
            "tag_specifications": "tagSpecifications",
            "user_data": "userData",
        },
    )
    class LaunchTemplateDataProperty:
        def __init__(
            self,
            *,
            block_device_mappings: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.BlockDeviceMappingProperty"]]]] = None,
            capacity_reservation_specification: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.CapacityReservationSpecificationProperty"]] = None,
            cpu_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.CpuOptionsProperty"]] = None,
            credit_specification: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.CreditSpecificationProperty"]] = None,
            disable_api_termination: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            ebs_optimized: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            elastic_gpu_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.ElasticGpuSpecificationProperty"]]]] = None,
            elastic_inference_accelerators: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateElasticInferenceAcceleratorProperty"]]]] = None,
            enclave_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.EnclaveOptionsProperty"]] = None,
            hibernation_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.HibernationOptionsProperty"]] = None,
            iam_instance_profile: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.IamInstanceProfileProperty"]] = None,
            image_id: typing.Optional[builtins.str] = None,
            instance_initiated_shutdown_behavior: typing.Optional[builtins.str] = None,
            instance_market_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.InstanceMarketOptionsProperty"]] = None,
            instance_type: typing.Optional[builtins.str] = None,
            kernel_id: typing.Optional[builtins.str] = None,
            key_name: typing.Optional[builtins.str] = None,
            license_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LicenseSpecificationProperty"]]]] = None,
            metadata_options: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.MetadataOptionsProperty"]] = None,
            monitoring: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.MonitoringProperty"]] = None,
            network_interfaces: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.NetworkInterfaceProperty"]]]] = None,
            placement: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.PlacementProperty"]] = None,
            ram_disk_id: typing.Optional[builtins.str] = None,
            security_group_ids: typing.Optional[typing.Sequence[builtins.str]] = None,
            security_groups: typing.Optional[typing.Sequence[builtins.str]] = None,
            tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.TagSpecificationProperty"]]]] = None,
            user_data: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param block_device_mappings: ``CfnLaunchTemplate.LaunchTemplateDataProperty.BlockDeviceMappings``.
            :param capacity_reservation_specification: ``CfnLaunchTemplate.LaunchTemplateDataProperty.CapacityReservationSpecification``.
            :param cpu_options: ``CfnLaunchTemplate.LaunchTemplateDataProperty.CpuOptions``.
            :param credit_specification: ``CfnLaunchTemplate.LaunchTemplateDataProperty.CreditSpecification``.
            :param disable_api_termination: ``CfnLaunchTemplate.LaunchTemplateDataProperty.DisableApiTermination``.
            :param ebs_optimized: ``CfnLaunchTemplate.LaunchTemplateDataProperty.EbsOptimized``.
            :param elastic_gpu_specifications: ``CfnLaunchTemplate.LaunchTemplateDataProperty.ElasticGpuSpecifications``.
            :param elastic_inference_accelerators: ``CfnLaunchTemplate.LaunchTemplateDataProperty.ElasticInferenceAccelerators``.
            :param enclave_options: ``CfnLaunchTemplate.LaunchTemplateDataProperty.EnclaveOptions``.
            :param hibernation_options: ``CfnLaunchTemplate.LaunchTemplateDataProperty.HibernationOptions``.
            :param iam_instance_profile: ``CfnLaunchTemplate.LaunchTemplateDataProperty.IamInstanceProfile``.
            :param image_id: ``CfnLaunchTemplate.LaunchTemplateDataProperty.ImageId``.
            :param instance_initiated_shutdown_behavior: ``CfnLaunchTemplate.LaunchTemplateDataProperty.InstanceInitiatedShutdownBehavior``.
            :param instance_market_options: ``CfnLaunchTemplate.LaunchTemplateDataProperty.InstanceMarketOptions``.
            :param instance_type: ``CfnLaunchTemplate.LaunchTemplateDataProperty.InstanceType``.
            :param kernel_id: ``CfnLaunchTemplate.LaunchTemplateDataProperty.KernelId``.
            :param key_name: ``CfnLaunchTemplate.LaunchTemplateDataProperty.KeyName``.
            :param license_specifications: ``CfnLaunchTemplate.LaunchTemplateDataProperty.LicenseSpecifications``.
            :param metadata_options: ``CfnLaunchTemplate.LaunchTemplateDataProperty.MetadataOptions``.
            :param monitoring: ``CfnLaunchTemplate.LaunchTemplateDataProperty.Monitoring``.
            :param network_interfaces: ``CfnLaunchTemplate.LaunchTemplateDataProperty.NetworkInterfaces``.
            :param placement: ``CfnLaunchTemplate.LaunchTemplateDataProperty.Placement``.
            :param ram_disk_id: ``CfnLaunchTemplate.LaunchTemplateDataProperty.RamDiskId``.
            :param security_group_ids: ``CfnLaunchTemplate.LaunchTemplateDataProperty.SecurityGroupIds``.
            :param security_groups: ``CfnLaunchTemplate.LaunchTemplateDataProperty.SecurityGroups``.
            :param tag_specifications: ``CfnLaunchTemplate.LaunchTemplateDataProperty.TagSpecifications``.
            :param user_data: ``CfnLaunchTemplate.LaunchTemplateDataProperty.UserData``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if block_device_mappings is not None:
                self._values["block_device_mappings"] = block_device_mappings
            if capacity_reservation_specification is not None:
                self._values["capacity_reservation_specification"] = capacity_reservation_specification
            if cpu_options is not None:
                self._values["cpu_options"] = cpu_options
            if credit_specification is not None:
                self._values["credit_specification"] = credit_specification
            if disable_api_termination is not None:
                self._values["disable_api_termination"] = disable_api_termination
            if ebs_optimized is not None:
                self._values["ebs_optimized"] = ebs_optimized
            if elastic_gpu_specifications is not None:
                self._values["elastic_gpu_specifications"] = elastic_gpu_specifications
            if elastic_inference_accelerators is not None:
                self._values["elastic_inference_accelerators"] = elastic_inference_accelerators
            if enclave_options is not None:
                self._values["enclave_options"] = enclave_options
            if hibernation_options is not None:
                self._values["hibernation_options"] = hibernation_options
            if iam_instance_profile is not None:
                self._values["iam_instance_profile"] = iam_instance_profile
            if image_id is not None:
                self._values["image_id"] = image_id
            if instance_initiated_shutdown_behavior is not None:
                self._values["instance_initiated_shutdown_behavior"] = instance_initiated_shutdown_behavior
            if instance_market_options is not None:
                self._values["instance_market_options"] = instance_market_options
            if instance_type is not None:
                self._values["instance_type"] = instance_type
            if kernel_id is not None:
                self._values["kernel_id"] = kernel_id
            if key_name is not None:
                self._values["key_name"] = key_name
            if license_specifications is not None:
                self._values["license_specifications"] = license_specifications
            if metadata_options is not None:
                self._values["metadata_options"] = metadata_options
            if monitoring is not None:
                self._values["monitoring"] = monitoring
            if network_interfaces is not None:
                self._values["network_interfaces"] = network_interfaces
            if placement is not None:
                self._values["placement"] = placement
            if ram_disk_id is not None:
                self._values["ram_disk_id"] = ram_disk_id
            if security_group_ids is not None:
                self._values["security_group_ids"] = security_group_ids
            if security_groups is not None:
                self._values["security_groups"] = security_groups
            if tag_specifications is not None:
                self._values["tag_specifications"] = tag_specifications
            if user_data is not None:
                self._values["user_data"] = user_data

        @builtins.property
        def block_device_mappings(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.BlockDeviceMappingProperty"]]]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.BlockDeviceMappings``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-blockdevicemappings
            '''
            result = self._values.get("block_device_mappings")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.BlockDeviceMappingProperty"]]]], result)

        @builtins.property
        def capacity_reservation_specification(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.CapacityReservationSpecificationProperty"]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.CapacityReservationSpecification``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-capacityreservationspecification
            '''
            result = self._values.get("capacity_reservation_specification")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.CapacityReservationSpecificationProperty"]], result)

        @builtins.property
        def cpu_options(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.CpuOptionsProperty"]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.CpuOptions``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-cpuoptions
            '''
            result = self._values.get("cpu_options")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.CpuOptionsProperty"]], result)

        @builtins.property
        def credit_specification(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.CreditSpecificationProperty"]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.CreditSpecification``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-creditspecification
            '''
            result = self._values.get("credit_specification")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.CreditSpecificationProperty"]], result)

        @builtins.property
        def disable_api_termination(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.DisableApiTermination``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-disableapitermination
            '''
            result = self._values.get("disable_api_termination")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def ebs_optimized(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.EbsOptimized``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-ebsoptimized
            '''
            result = self._values.get("ebs_optimized")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def elastic_gpu_specifications(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.ElasticGpuSpecificationProperty"]]]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.ElasticGpuSpecifications``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-elasticgpuspecifications
            '''
            result = self._values.get("elastic_gpu_specifications")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.ElasticGpuSpecificationProperty"]]]], result)

        @builtins.property
        def elastic_inference_accelerators(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateElasticInferenceAcceleratorProperty"]]]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.ElasticInferenceAccelerators``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-elasticinferenceaccelerators
            '''
            result = self._values.get("elastic_inference_accelerators")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LaunchTemplateElasticInferenceAcceleratorProperty"]]]], result)

        @builtins.property
        def enclave_options(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.EnclaveOptionsProperty"]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.EnclaveOptions``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-enclaveoptions
            '''
            result = self._values.get("enclave_options")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.EnclaveOptionsProperty"]], result)

        @builtins.property
        def hibernation_options(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.HibernationOptionsProperty"]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.HibernationOptions``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-hibernationoptions
            '''
            result = self._values.get("hibernation_options")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.HibernationOptionsProperty"]], result)

        @builtins.property
        def iam_instance_profile(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.IamInstanceProfileProperty"]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.IamInstanceProfile``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-iaminstanceprofile
            '''
            result = self._values.get("iam_instance_profile")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.IamInstanceProfileProperty"]], result)

        @builtins.property
        def image_id(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.ImageId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-imageid
            '''
            result = self._values.get("image_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_initiated_shutdown_behavior(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.InstanceInitiatedShutdownBehavior``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-instanceinitiatedshutdownbehavior
            '''
            result = self._values.get("instance_initiated_shutdown_behavior")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_market_options(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.InstanceMarketOptionsProperty"]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.InstanceMarketOptions``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-instancemarketoptions
            '''
            result = self._values.get("instance_market_options")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.InstanceMarketOptionsProperty"]], result)

        @builtins.property
        def instance_type(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.InstanceType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-instancetype
            '''
            result = self._values.get("instance_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def kernel_id(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.KernelId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-kernelid
            '''
            result = self._values.get("kernel_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def key_name(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.KeyName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-keyname
            '''
            result = self._values.get("key_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def license_specifications(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LicenseSpecificationProperty"]]]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.LicenseSpecifications``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-licensespecifications
            '''
            result = self._values.get("license_specifications")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.LicenseSpecificationProperty"]]]], result)

        @builtins.property
        def metadata_options(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.MetadataOptionsProperty"]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.MetadataOptions``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-metadataoptions
            '''
            result = self._values.get("metadata_options")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.MetadataOptionsProperty"]], result)

        @builtins.property
        def monitoring(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.MonitoringProperty"]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.Monitoring``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-monitoring
            '''
            result = self._values.get("monitoring")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.MonitoringProperty"]], result)

        @builtins.property
        def network_interfaces(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.NetworkInterfaceProperty"]]]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.NetworkInterfaces``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-networkinterfaces
            '''
            result = self._values.get("network_interfaces")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.NetworkInterfaceProperty"]]]], result)

        @builtins.property
        def placement(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.PlacementProperty"]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.Placement``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-placement
            '''
            result = self._values.get("placement")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.PlacementProperty"]], result)

        @builtins.property
        def ram_disk_id(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.RamDiskId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-ramdiskid
            '''
            result = self._values.get("ram_disk_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def security_group_ids(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.SecurityGroupIds``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-securitygroupids
            '''
            result = self._values.get("security_group_ids")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def security_groups(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.SecurityGroups``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-securitygroups
            '''
            result = self._values.get("security_groups")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def tag_specifications(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.TagSpecificationProperty"]]]]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.TagSpecifications``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-tagspecifications
            '''
            result = self._values.get("tag_specifications")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.TagSpecificationProperty"]]]], result)

        @builtins.property
        def user_data(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.LaunchTemplateDataProperty.UserData``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata.html#cfn-ec2-launchtemplate-launchtemplatedata-userdata
            '''
            result = self._values.get("user_data")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LaunchTemplateDataProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.LaunchTemplateElasticInferenceAcceleratorProperty",
        jsii_struct_bases=[],
        name_mapping={"count": "count", "type": "type"},
    )
    class LaunchTemplateElasticInferenceAcceleratorProperty:
        def __init__(
            self,
            *,
            count: typing.Optional[jsii.Number] = None,
            type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param count: ``CfnLaunchTemplate.LaunchTemplateElasticInferenceAcceleratorProperty.Count``.
            :param type: ``CfnLaunchTemplate.LaunchTemplateElasticInferenceAcceleratorProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplateelasticinferenceaccelerator.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if count is not None:
                self._values["count"] = count
            if type is not None:
                self._values["type"] = type

        @builtins.property
        def count(self) -> typing.Optional[jsii.Number]:
            '''``CfnLaunchTemplate.LaunchTemplateElasticInferenceAcceleratorProperty.Count``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplateelasticinferenceaccelerator.html#cfn-ec2-launchtemplate-launchtemplateelasticinferenceaccelerator-count
            '''
            result = self._values.get("count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def type(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.LaunchTemplateElasticInferenceAcceleratorProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplateelasticinferenceaccelerator.html#cfn-ec2-launchtemplate-launchtemplateelasticinferenceaccelerator-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LaunchTemplateElasticInferenceAcceleratorProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.LaunchTemplateTagSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"resource_type": "resourceType", "tags": "tags"},
    )
    class LaunchTemplateTagSpecificationProperty:
        def __init__(
            self,
            *,
            resource_type: typing.Optional[builtins.str] = None,
            tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
        ) -> None:
            '''
            :param resource_type: ``CfnLaunchTemplate.LaunchTemplateTagSpecificationProperty.ResourceType``.
            :param tags: ``CfnLaunchTemplate.LaunchTemplateTagSpecificationProperty.Tags``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatetagspecification.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if resource_type is not None:
                self._values["resource_type"] = resource_type
            if tags is not None:
                self._values["tags"] = tags

        @builtins.property
        def resource_type(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.LaunchTemplateTagSpecificationProperty.ResourceType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatetagspecification.html#cfn-ec2-launchtemplate-launchtemplatetagspecification-resourcetype
            '''
            result = self._values.get("resource_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
            '''``CfnLaunchTemplate.LaunchTemplateTagSpecificationProperty.Tags``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatetagspecification.html#cfn-ec2-launchtemplate-launchtemplatetagspecification-tags
            '''
            result = self._values.get("tags")
            return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LaunchTemplateTagSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.LicenseSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"license_configuration_arn": "licenseConfigurationArn"},
    )
    class LicenseSpecificationProperty:
        def __init__(
            self,
            *,
            license_configuration_arn: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param license_configuration_arn: ``CfnLaunchTemplate.LicenseSpecificationProperty.LicenseConfigurationArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-licensespecification.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if license_configuration_arn is not None:
                self._values["license_configuration_arn"] = license_configuration_arn

        @builtins.property
        def license_configuration_arn(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.LicenseSpecificationProperty.LicenseConfigurationArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-licensespecification.html#cfn-ec2-launchtemplate-licensespecification-licenseconfigurationarn
            '''
            result = self._values.get("license_configuration_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LicenseSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.MetadataOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "http_endpoint": "httpEndpoint",
            "http_put_response_hop_limit": "httpPutResponseHopLimit",
            "http_tokens": "httpTokens",
        },
    )
    class MetadataOptionsProperty:
        def __init__(
            self,
            *,
            http_endpoint: typing.Optional[builtins.str] = None,
            http_put_response_hop_limit: typing.Optional[jsii.Number] = None,
            http_tokens: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param http_endpoint: ``CfnLaunchTemplate.MetadataOptionsProperty.HttpEndpoint``.
            :param http_put_response_hop_limit: ``CfnLaunchTemplate.MetadataOptionsProperty.HttpPutResponseHopLimit``.
            :param http_tokens: ``CfnLaunchTemplate.MetadataOptionsProperty.HttpTokens``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-metadataoptions.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if http_endpoint is not None:
                self._values["http_endpoint"] = http_endpoint
            if http_put_response_hop_limit is not None:
                self._values["http_put_response_hop_limit"] = http_put_response_hop_limit
            if http_tokens is not None:
                self._values["http_tokens"] = http_tokens

        @builtins.property
        def http_endpoint(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.MetadataOptionsProperty.HttpEndpoint``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-metadataoptions.html#cfn-ec2-launchtemplate-launchtemplatedata-metadataoptions-httpendpoint
            '''
            result = self._values.get("http_endpoint")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def http_put_response_hop_limit(self) -> typing.Optional[jsii.Number]:
            '''``CfnLaunchTemplate.MetadataOptionsProperty.HttpPutResponseHopLimit``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-metadataoptions.html#cfn-ec2-launchtemplate-launchtemplatedata-metadataoptions-httpputresponsehoplimit
            '''
            result = self._values.get("http_put_response_hop_limit")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def http_tokens(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.MetadataOptionsProperty.HttpTokens``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-metadataoptions.html#cfn-ec2-launchtemplate-launchtemplatedata-metadataoptions-httptokens
            '''
            result = self._values.get("http_tokens")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MetadataOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.MonitoringProperty",
        jsii_struct_bases=[],
        name_mapping={"enabled": "enabled"},
    )
    class MonitoringProperty:
        def __init__(
            self,
            *,
            enabled: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        ) -> None:
            '''
            :param enabled: ``CfnLaunchTemplate.MonitoringProperty.Enabled``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-monitoring.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if enabled is not None:
                self._values["enabled"] = enabled

        @builtins.property
        def enabled(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnLaunchTemplate.MonitoringProperty.Enabled``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-monitoring.html#cfn-ec2-launchtemplate-launchtemplatedata-monitoring-enabled
            '''
            result = self._values.get("enabled")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "MonitoringProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.NetworkInterfaceProperty",
        jsii_struct_bases=[],
        name_mapping={
            "associate_carrier_ip_address": "associateCarrierIpAddress",
            "associate_public_ip_address": "associatePublicIpAddress",
            "delete_on_termination": "deleteOnTermination",
            "description": "description",
            "device_index": "deviceIndex",
            "groups": "groups",
            "interface_type": "interfaceType",
            "ipv6_address_count": "ipv6AddressCount",
            "ipv6_addresses": "ipv6Addresses",
            "network_card_index": "networkCardIndex",
            "network_interface_id": "networkInterfaceId",
            "private_ip_address": "privateIpAddress",
            "private_ip_addresses": "privateIpAddresses",
            "secondary_private_ip_address_count": "secondaryPrivateIpAddressCount",
            "subnet_id": "subnetId",
        },
    )
    class NetworkInterfaceProperty:
        def __init__(
            self,
            *,
            associate_carrier_ip_address: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            associate_public_ip_address: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            delete_on_termination: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            description: typing.Optional[builtins.str] = None,
            device_index: typing.Optional[jsii.Number] = None,
            groups: typing.Optional[typing.Sequence[builtins.str]] = None,
            interface_type: typing.Optional[builtins.str] = None,
            ipv6_address_count: typing.Optional[jsii.Number] = None,
            ipv6_addresses: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.Ipv6AddProperty"]]]] = None,
            network_card_index: typing.Optional[jsii.Number] = None,
            network_interface_id: typing.Optional[builtins.str] = None,
            private_ip_address: typing.Optional[builtins.str] = None,
            private_ip_addresses: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.PrivateIpAddProperty"]]]] = None,
            secondary_private_ip_address_count: typing.Optional[jsii.Number] = None,
            subnet_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param associate_carrier_ip_address: ``CfnLaunchTemplate.NetworkInterfaceProperty.AssociateCarrierIpAddress``.
            :param associate_public_ip_address: ``CfnLaunchTemplate.NetworkInterfaceProperty.AssociatePublicIpAddress``.
            :param delete_on_termination: ``CfnLaunchTemplate.NetworkInterfaceProperty.DeleteOnTermination``.
            :param description: ``CfnLaunchTemplate.NetworkInterfaceProperty.Description``.
            :param device_index: ``CfnLaunchTemplate.NetworkInterfaceProperty.DeviceIndex``.
            :param groups: ``CfnLaunchTemplate.NetworkInterfaceProperty.Groups``.
            :param interface_type: ``CfnLaunchTemplate.NetworkInterfaceProperty.InterfaceType``.
            :param ipv6_address_count: ``CfnLaunchTemplate.NetworkInterfaceProperty.Ipv6AddressCount``.
            :param ipv6_addresses: ``CfnLaunchTemplate.NetworkInterfaceProperty.Ipv6Addresses``.
            :param network_card_index: ``CfnLaunchTemplate.NetworkInterfaceProperty.NetworkCardIndex``.
            :param network_interface_id: ``CfnLaunchTemplate.NetworkInterfaceProperty.NetworkInterfaceId``.
            :param private_ip_address: ``CfnLaunchTemplate.NetworkInterfaceProperty.PrivateIpAddress``.
            :param private_ip_addresses: ``CfnLaunchTemplate.NetworkInterfaceProperty.PrivateIpAddresses``.
            :param secondary_private_ip_address_count: ``CfnLaunchTemplate.NetworkInterfaceProperty.SecondaryPrivateIpAddressCount``.
            :param subnet_id: ``CfnLaunchTemplate.NetworkInterfaceProperty.SubnetId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-networkinterface.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if associate_carrier_ip_address is not None:
                self._values["associate_carrier_ip_address"] = associate_carrier_ip_address
            if associate_public_ip_address is not None:
                self._values["associate_public_ip_address"] = associate_public_ip_address
            if delete_on_termination is not None:
                self._values["delete_on_termination"] = delete_on_termination
            if description is not None:
                self._values["description"] = description
            if device_index is not None:
                self._values["device_index"] = device_index
            if groups is not None:
                self._values["groups"] = groups
            if interface_type is not None:
                self._values["interface_type"] = interface_type
            if ipv6_address_count is not None:
                self._values["ipv6_address_count"] = ipv6_address_count
            if ipv6_addresses is not None:
                self._values["ipv6_addresses"] = ipv6_addresses
            if network_card_index is not None:
                self._values["network_card_index"] = network_card_index
            if network_interface_id is not None:
                self._values["network_interface_id"] = network_interface_id
            if private_ip_address is not None:
                self._values["private_ip_address"] = private_ip_address
            if private_ip_addresses is not None:
                self._values["private_ip_addresses"] = private_ip_addresses
            if secondary_private_ip_address_count is not None:
                self._values["secondary_private_ip_address_count"] = secondary_private_ip_address_count
            if subnet_id is not None:
                self._values["subnet_id"] = subnet_id

        @builtins.property
        def associate_carrier_ip_address(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnLaunchTemplate.NetworkInterfaceProperty.AssociateCarrierIpAddress``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-networkinterface.html#cfn-ec2-launchtemplate-networkinterface-associatecarrieripaddress
            '''
            result = self._values.get("associate_carrier_ip_address")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def associate_public_ip_address(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnLaunchTemplate.NetworkInterfaceProperty.AssociatePublicIpAddress``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-networkinterface.html#cfn-ec2-launchtemplate-networkinterface-associatepublicipaddress
            '''
            result = self._values.get("associate_public_ip_address")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def delete_on_termination(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnLaunchTemplate.NetworkInterfaceProperty.DeleteOnTermination``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-networkinterface.html#cfn-ec2-launchtemplate-networkinterface-deleteontermination
            '''
            result = self._values.get("delete_on_termination")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.NetworkInterfaceProperty.Description``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-networkinterface.html#cfn-ec2-launchtemplate-networkinterface-description
            '''
            result = self._values.get("description")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def device_index(self) -> typing.Optional[jsii.Number]:
            '''``CfnLaunchTemplate.NetworkInterfaceProperty.DeviceIndex``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-networkinterface.html#cfn-ec2-launchtemplate-networkinterface-deviceindex
            '''
            result = self._values.get("device_index")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def groups(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnLaunchTemplate.NetworkInterfaceProperty.Groups``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-networkinterface.html#cfn-ec2-launchtemplate-networkinterface-groups
            '''
            result = self._values.get("groups")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def interface_type(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.NetworkInterfaceProperty.InterfaceType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-networkinterface.html#cfn-ec2-launchtemplate-networkinterface-interfacetype
            '''
            result = self._values.get("interface_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ipv6_address_count(self) -> typing.Optional[jsii.Number]:
            '''``CfnLaunchTemplate.NetworkInterfaceProperty.Ipv6AddressCount``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-networkinterface.html#cfn-ec2-launchtemplate-networkinterface-ipv6addresscount
            '''
            result = self._values.get("ipv6_address_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def ipv6_addresses(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.Ipv6AddProperty"]]]]:
            '''``CfnLaunchTemplate.NetworkInterfaceProperty.Ipv6Addresses``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-networkinterface.html#cfn-ec2-launchtemplate-networkinterface-ipv6addresses
            '''
            result = self._values.get("ipv6_addresses")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.Ipv6AddProperty"]]]], result)

        @builtins.property
        def network_card_index(self) -> typing.Optional[jsii.Number]:
            '''``CfnLaunchTemplate.NetworkInterfaceProperty.NetworkCardIndex``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-networkinterface.html#cfn-ec2-launchtemplate-networkinterface-networkcardindex
            '''
            result = self._values.get("network_card_index")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def network_interface_id(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.NetworkInterfaceProperty.NetworkInterfaceId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-networkinterface.html#cfn-ec2-launchtemplate-networkinterface-networkinterfaceid
            '''
            result = self._values.get("network_interface_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def private_ip_address(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.NetworkInterfaceProperty.PrivateIpAddress``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-networkinterface.html#cfn-ec2-launchtemplate-networkinterface-privateipaddress
            '''
            result = self._values.get("private_ip_address")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def private_ip_addresses(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.PrivateIpAddProperty"]]]]:
            '''``CfnLaunchTemplate.NetworkInterfaceProperty.PrivateIpAddresses``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-networkinterface.html#cfn-ec2-launchtemplate-networkinterface-privateipaddresses
            '''
            result = self._values.get("private_ip_addresses")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnLaunchTemplate.PrivateIpAddProperty"]]]], result)

        @builtins.property
        def secondary_private_ip_address_count(self) -> typing.Optional[jsii.Number]:
            '''``CfnLaunchTemplate.NetworkInterfaceProperty.SecondaryPrivateIpAddressCount``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-networkinterface.html#cfn-ec2-launchtemplate-networkinterface-secondaryprivateipaddresscount
            '''
            result = self._values.get("secondary_private_ip_address_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def subnet_id(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.NetworkInterfaceProperty.SubnetId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-networkinterface.html#cfn-ec2-launchtemplate-networkinterface-subnetid
            '''
            result = self._values.get("subnet_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "NetworkInterfaceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.PlacementProperty",
        jsii_struct_bases=[],
        name_mapping={
            "affinity": "affinity",
            "availability_zone": "availabilityZone",
            "group_name": "groupName",
            "host_id": "hostId",
            "host_resource_group_arn": "hostResourceGroupArn",
            "partition_number": "partitionNumber",
            "spread_domain": "spreadDomain",
            "tenancy": "tenancy",
        },
    )
    class PlacementProperty:
        def __init__(
            self,
            *,
            affinity: typing.Optional[builtins.str] = None,
            availability_zone: typing.Optional[builtins.str] = None,
            group_name: typing.Optional[builtins.str] = None,
            host_id: typing.Optional[builtins.str] = None,
            host_resource_group_arn: typing.Optional[builtins.str] = None,
            partition_number: typing.Optional[jsii.Number] = None,
            spread_domain: typing.Optional[builtins.str] = None,
            tenancy: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param affinity: ``CfnLaunchTemplate.PlacementProperty.Affinity``.
            :param availability_zone: ``CfnLaunchTemplate.PlacementProperty.AvailabilityZone``.
            :param group_name: ``CfnLaunchTemplate.PlacementProperty.GroupName``.
            :param host_id: ``CfnLaunchTemplate.PlacementProperty.HostId``.
            :param host_resource_group_arn: ``CfnLaunchTemplate.PlacementProperty.HostResourceGroupArn``.
            :param partition_number: ``CfnLaunchTemplate.PlacementProperty.PartitionNumber``.
            :param spread_domain: ``CfnLaunchTemplate.PlacementProperty.SpreadDomain``.
            :param tenancy: ``CfnLaunchTemplate.PlacementProperty.Tenancy``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-placement.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if affinity is not None:
                self._values["affinity"] = affinity
            if availability_zone is not None:
                self._values["availability_zone"] = availability_zone
            if group_name is not None:
                self._values["group_name"] = group_name
            if host_id is not None:
                self._values["host_id"] = host_id
            if host_resource_group_arn is not None:
                self._values["host_resource_group_arn"] = host_resource_group_arn
            if partition_number is not None:
                self._values["partition_number"] = partition_number
            if spread_domain is not None:
                self._values["spread_domain"] = spread_domain
            if tenancy is not None:
                self._values["tenancy"] = tenancy

        @builtins.property
        def affinity(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.PlacementProperty.Affinity``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-placement.html#cfn-ec2-launchtemplate-launchtemplatedata-placement-affinity
            '''
            result = self._values.get("affinity")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def availability_zone(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.PlacementProperty.AvailabilityZone``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-placement.html#cfn-ec2-launchtemplate-launchtemplatedata-placement-availabilityzone
            '''
            result = self._values.get("availability_zone")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def group_name(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.PlacementProperty.GroupName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-placement.html#cfn-ec2-launchtemplate-launchtemplatedata-placement-groupname
            '''
            result = self._values.get("group_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def host_id(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.PlacementProperty.HostId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-placement.html#cfn-ec2-launchtemplate-launchtemplatedata-placement-hostid
            '''
            result = self._values.get("host_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def host_resource_group_arn(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.PlacementProperty.HostResourceGroupArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-placement.html#cfn-ec2-launchtemplate-launchtemplatedata-placement-hostresourcegrouparn
            '''
            result = self._values.get("host_resource_group_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def partition_number(self) -> typing.Optional[jsii.Number]:
            '''``CfnLaunchTemplate.PlacementProperty.PartitionNumber``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-placement.html#cfn-ec2-launchtemplate-launchtemplatedata-placement-partitionnumber
            '''
            result = self._values.get("partition_number")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def spread_domain(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.PlacementProperty.SpreadDomain``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-placement.html#cfn-ec2-launchtemplate-launchtemplatedata-placement-spreaddomain
            '''
            result = self._values.get("spread_domain")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def tenancy(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.PlacementProperty.Tenancy``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-placement.html#cfn-ec2-launchtemplate-launchtemplatedata-placement-tenancy
            '''
            result = self._values.get("tenancy")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PlacementProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.PrivateIpAddProperty",
        jsii_struct_bases=[],
        name_mapping={"primary": "primary", "private_ip_address": "privateIpAddress"},
    )
    class PrivateIpAddProperty:
        def __init__(
            self,
            *,
            primary: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            private_ip_address: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param primary: ``CfnLaunchTemplate.PrivateIpAddProperty.Primary``.
            :param private_ip_address: ``CfnLaunchTemplate.PrivateIpAddProperty.PrivateIpAddress``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-privateipadd.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if primary is not None:
                self._values["primary"] = primary
            if private_ip_address is not None:
                self._values["private_ip_address"] = private_ip_address

        @builtins.property
        def primary(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnLaunchTemplate.PrivateIpAddProperty.Primary``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-privateipadd.html#cfn-ec2-launchtemplate-privateipadd-primary
            '''
            result = self._values.get("primary")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def private_ip_address(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.PrivateIpAddProperty.PrivateIpAddress``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-privateipadd.html#cfn-ec2-launchtemplate-privateipadd-privateipaddress
            '''
            result = self._values.get("private_ip_address")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PrivateIpAddProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.SpotOptionsProperty",
        jsii_struct_bases=[],
        name_mapping={
            "block_duration_minutes": "blockDurationMinutes",
            "instance_interruption_behavior": "instanceInterruptionBehavior",
            "max_price": "maxPrice",
            "spot_instance_type": "spotInstanceType",
            "valid_until": "validUntil",
        },
    )
    class SpotOptionsProperty:
        def __init__(
            self,
            *,
            block_duration_minutes: typing.Optional[jsii.Number] = None,
            instance_interruption_behavior: typing.Optional[builtins.str] = None,
            max_price: typing.Optional[builtins.str] = None,
            spot_instance_type: typing.Optional[builtins.str] = None,
            valid_until: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param block_duration_minutes: ``CfnLaunchTemplate.SpotOptionsProperty.BlockDurationMinutes``.
            :param instance_interruption_behavior: ``CfnLaunchTemplate.SpotOptionsProperty.InstanceInterruptionBehavior``.
            :param max_price: ``CfnLaunchTemplate.SpotOptionsProperty.MaxPrice``.
            :param spot_instance_type: ``CfnLaunchTemplate.SpotOptionsProperty.SpotInstanceType``.
            :param valid_until: ``CfnLaunchTemplate.SpotOptionsProperty.ValidUntil``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-instancemarketoptions-spotoptions.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if block_duration_minutes is not None:
                self._values["block_duration_minutes"] = block_duration_minutes
            if instance_interruption_behavior is not None:
                self._values["instance_interruption_behavior"] = instance_interruption_behavior
            if max_price is not None:
                self._values["max_price"] = max_price
            if spot_instance_type is not None:
                self._values["spot_instance_type"] = spot_instance_type
            if valid_until is not None:
                self._values["valid_until"] = valid_until

        @builtins.property
        def block_duration_minutes(self) -> typing.Optional[jsii.Number]:
            '''``CfnLaunchTemplate.SpotOptionsProperty.BlockDurationMinutes``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-instancemarketoptions-spotoptions.html#cfn-ec2-launchtemplate-launchtemplatedata-instancemarketoptions-spotoptions-blockdurationminutes
            '''
            result = self._values.get("block_duration_minutes")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def instance_interruption_behavior(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.SpotOptionsProperty.InstanceInterruptionBehavior``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-instancemarketoptions-spotoptions.html#cfn-ec2-launchtemplate-launchtemplatedata-instancemarketoptions-spotoptions-instanceinterruptionbehavior
            '''
            result = self._values.get("instance_interruption_behavior")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def max_price(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.SpotOptionsProperty.MaxPrice``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-instancemarketoptions-spotoptions.html#cfn-ec2-launchtemplate-launchtemplatedata-instancemarketoptions-spotoptions-maxprice
            '''
            result = self._values.get("max_price")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def spot_instance_type(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.SpotOptionsProperty.SpotInstanceType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-instancemarketoptions-spotoptions.html#cfn-ec2-launchtemplate-launchtemplatedata-instancemarketoptions-spotoptions-spotinstancetype
            '''
            result = self._values.get("spot_instance_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def valid_until(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.SpotOptionsProperty.ValidUntil``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-instancemarketoptions-spotoptions.html#cfn-ec2-launchtemplate-launchtemplatedata-instancemarketoptions-spotoptions-validuntil
            '''
            result = self._values.get("valid_until")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "SpotOptionsProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplate.TagSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"resource_type": "resourceType", "tags": "tags"},
    )
    class TagSpecificationProperty:
        def __init__(
            self,
            *,
            resource_type: typing.Optional[builtins.str] = None,
            tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
        ) -> None:
            '''
            :param resource_type: ``CfnLaunchTemplate.TagSpecificationProperty.ResourceType``.
            :param tags: ``CfnLaunchTemplate.TagSpecificationProperty.Tags``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-tagspecification.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if resource_type is not None:
                self._values["resource_type"] = resource_type
            if tags is not None:
                self._values["tags"] = tags

        @builtins.property
        def resource_type(self) -> typing.Optional[builtins.str]:
            '''``CfnLaunchTemplate.TagSpecificationProperty.ResourceType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-tagspecification.html#cfn-ec2-launchtemplate-tagspecification-resourcetype
            '''
            result = self._values.get("resource_type")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
            '''``CfnLaunchTemplate.TagSpecificationProperty.Tags``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-tagspecification.html#cfn-ec2-launchtemplate-tagspecification-tags
            '''
            result = self._values.get("tags")
            return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "TagSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnLaunchTemplateProps",
    jsii_struct_bases=[],
    name_mapping={
        "launch_template_data": "launchTemplateData",
        "launch_template_name": "launchTemplateName",
        "tag_specifications": "tagSpecifications",
    },
)
class CfnLaunchTemplateProps:
    def __init__(
        self,
        *,
        launch_template_data: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnLaunchTemplate.LaunchTemplateDataProperty]] = None,
        launch_template_name: typing.Optional[builtins.str] = None,
        tag_specifications: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnLaunchTemplate.LaunchTemplateTagSpecificationProperty]]]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::LaunchTemplate``.

        :param launch_template_data: ``AWS::EC2::LaunchTemplate.LaunchTemplateData``.
        :param launch_template_name: ``AWS::EC2::LaunchTemplate.LaunchTemplateName``.
        :param tag_specifications: ``AWS::EC2::LaunchTemplate.TagSpecifications``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if launch_template_data is not None:
            self._values["launch_template_data"] = launch_template_data
        if launch_template_name is not None:
            self._values["launch_template_name"] = launch_template_name
        if tag_specifications is not None:
            self._values["tag_specifications"] = tag_specifications

    @builtins.property
    def launch_template_data(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnLaunchTemplate.LaunchTemplateDataProperty]]:
        '''``AWS::EC2::LaunchTemplate.LaunchTemplateData``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html#cfn-ec2-launchtemplate-launchtemplatedata
        '''
        result = self._values.get("launch_template_data")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnLaunchTemplate.LaunchTemplateDataProperty]], result)

    @builtins.property
    def launch_template_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::LaunchTemplate.LaunchTemplateName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html#cfn-ec2-launchtemplate-launchtemplatename
        '''
        result = self._values.get("launch_template_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tag_specifications(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnLaunchTemplate.LaunchTemplateTagSpecificationProperty]]]]:
        '''``AWS::EC2::LaunchTemplate.TagSpecifications``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-launchtemplate.html#cfn-ec2-launchtemplate-tagspecifications
        '''
        result = self._values.get("tag_specifications")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnLaunchTemplate.LaunchTemplateTagSpecificationProperty]]]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnLaunchTemplateProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnLocalGatewayRoute(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnLocalGatewayRoute",
):
    '''A CloudFormation ``AWS::EC2::LocalGatewayRoute``.

    :cloudformationResource: AWS::EC2::LocalGatewayRoute
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-localgatewayroute.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        destination_cidr_block: builtins.str,
        local_gateway_route_table_id: builtins.str,
        local_gateway_virtual_interface_group_id: builtins.str,
    ) -> None:
        '''Create a new ``AWS::EC2::LocalGatewayRoute``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param destination_cidr_block: ``AWS::EC2::LocalGatewayRoute.DestinationCidrBlock``.
        :param local_gateway_route_table_id: ``AWS::EC2::LocalGatewayRoute.LocalGatewayRouteTableId``.
        :param local_gateway_virtual_interface_group_id: ``AWS::EC2::LocalGatewayRoute.LocalGatewayVirtualInterfaceGroupId``.
        '''
        props = CfnLocalGatewayRouteProps(
            destination_cidr_block=destination_cidr_block,
            local_gateway_route_table_id=local_gateway_route_table_id,
            local_gateway_virtual_interface_group_id=local_gateway_virtual_interface_group_id,
        )

        jsii.create(CfnLocalGatewayRoute, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrState")
    def attr_state(self) -> builtins.str:
        '''
        :cloudformationAttribute: State
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrState"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrType")
    def attr_type(self) -> builtins.str:
        '''
        :cloudformationAttribute: Type
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrType"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="destinationCidrBlock")
    def destination_cidr_block(self) -> builtins.str:
        '''``AWS::EC2::LocalGatewayRoute.DestinationCidrBlock``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-localgatewayroute.html#cfn-ec2-localgatewayroute-destinationcidrblock
        '''
        return typing.cast(builtins.str, jsii.get(self, "destinationCidrBlock"))

    @destination_cidr_block.setter
    def destination_cidr_block(self, value: builtins.str) -> None:
        jsii.set(self, "destinationCidrBlock", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="localGatewayRouteTableId")
    def local_gateway_route_table_id(self) -> builtins.str:
        '''``AWS::EC2::LocalGatewayRoute.LocalGatewayRouteTableId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-localgatewayroute.html#cfn-ec2-localgatewayroute-localgatewayroutetableid
        '''
        return typing.cast(builtins.str, jsii.get(self, "localGatewayRouteTableId"))

    @local_gateway_route_table_id.setter
    def local_gateway_route_table_id(self, value: builtins.str) -> None:
        jsii.set(self, "localGatewayRouteTableId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="localGatewayVirtualInterfaceGroupId")
    def local_gateway_virtual_interface_group_id(self) -> builtins.str:
        '''``AWS::EC2::LocalGatewayRoute.LocalGatewayVirtualInterfaceGroupId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-localgatewayroute.html#cfn-ec2-localgatewayroute-localgatewayvirtualinterfacegroupid
        '''
        return typing.cast(builtins.str, jsii.get(self, "localGatewayVirtualInterfaceGroupId"))

    @local_gateway_virtual_interface_group_id.setter
    def local_gateway_virtual_interface_group_id(self, value: builtins.str) -> None:
        jsii.set(self, "localGatewayVirtualInterfaceGroupId", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnLocalGatewayRouteProps",
    jsii_struct_bases=[],
    name_mapping={
        "destination_cidr_block": "destinationCidrBlock",
        "local_gateway_route_table_id": "localGatewayRouteTableId",
        "local_gateway_virtual_interface_group_id": "localGatewayVirtualInterfaceGroupId",
    },
)
class CfnLocalGatewayRouteProps:
    def __init__(
        self,
        *,
        destination_cidr_block: builtins.str,
        local_gateway_route_table_id: builtins.str,
        local_gateway_virtual_interface_group_id: builtins.str,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::LocalGatewayRoute``.

        :param destination_cidr_block: ``AWS::EC2::LocalGatewayRoute.DestinationCidrBlock``.
        :param local_gateway_route_table_id: ``AWS::EC2::LocalGatewayRoute.LocalGatewayRouteTableId``.
        :param local_gateway_virtual_interface_group_id: ``AWS::EC2::LocalGatewayRoute.LocalGatewayVirtualInterfaceGroupId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-localgatewayroute.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "destination_cidr_block": destination_cidr_block,
            "local_gateway_route_table_id": local_gateway_route_table_id,
            "local_gateway_virtual_interface_group_id": local_gateway_virtual_interface_group_id,
        }

    @builtins.property
    def destination_cidr_block(self) -> builtins.str:
        '''``AWS::EC2::LocalGatewayRoute.DestinationCidrBlock``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-localgatewayroute.html#cfn-ec2-localgatewayroute-destinationcidrblock
        '''
        result = self._values.get("destination_cidr_block")
        assert result is not None, "Required property 'destination_cidr_block' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def local_gateway_route_table_id(self) -> builtins.str:
        '''``AWS::EC2::LocalGatewayRoute.LocalGatewayRouteTableId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-localgatewayroute.html#cfn-ec2-localgatewayroute-localgatewayroutetableid
        '''
        result = self._values.get("local_gateway_route_table_id")
        assert result is not None, "Required property 'local_gateway_route_table_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def local_gateway_virtual_interface_group_id(self) -> builtins.str:
        '''``AWS::EC2::LocalGatewayRoute.LocalGatewayVirtualInterfaceGroupId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-localgatewayroute.html#cfn-ec2-localgatewayroute-localgatewayvirtualinterfacegroupid
        '''
        result = self._values.get("local_gateway_virtual_interface_group_id")
        assert result is not None, "Required property 'local_gateway_virtual_interface_group_id' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnLocalGatewayRouteProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnLocalGatewayRouteTableVPCAssociation(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnLocalGatewayRouteTableVPCAssociation",
):
    '''A CloudFormation ``AWS::EC2::LocalGatewayRouteTableVPCAssociation``.

    :cloudformationResource: AWS::EC2::LocalGatewayRouteTableVPCAssociation
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-localgatewayroutetablevpcassociation.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        local_gateway_route_table_id: builtins.str,
        vpc_id: builtins.str,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::LocalGatewayRouteTableVPCAssociation``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param local_gateway_route_table_id: ``AWS::EC2::LocalGatewayRouteTableVPCAssociation.LocalGatewayRouteTableId``.
        :param vpc_id: ``AWS::EC2::LocalGatewayRouteTableVPCAssociation.VpcId``.
        :param tags: ``AWS::EC2::LocalGatewayRouteTableVPCAssociation.Tags``.
        '''
        props = CfnLocalGatewayRouteTableVPCAssociationProps(
            local_gateway_route_table_id=local_gateway_route_table_id,
            vpc_id=vpc_id,
            tags=tags,
        )

        jsii.create(CfnLocalGatewayRouteTableVPCAssociation, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrLocalGatewayId")
    def attr_local_gateway_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: LocalGatewayId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrLocalGatewayId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrLocalGatewayRouteTableVpcAssociationId")
    def attr_local_gateway_route_table_vpc_association_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: LocalGatewayRouteTableVpcAssociationId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrLocalGatewayRouteTableVpcAssociationId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrState")
    def attr_state(self) -> builtins.str:
        '''
        :cloudformationAttribute: State
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrState"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''``AWS::EC2::LocalGatewayRouteTableVPCAssociation.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-localgatewayroutetablevpcassociation.html#cfn-ec2-localgatewayroutetablevpcassociation-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="localGatewayRouteTableId")
    def local_gateway_route_table_id(self) -> builtins.str:
        '''``AWS::EC2::LocalGatewayRouteTableVPCAssociation.LocalGatewayRouteTableId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-localgatewayroutetablevpcassociation.html#cfn-ec2-localgatewayroutetablevpcassociation-localgatewayroutetableid
        '''
        return typing.cast(builtins.str, jsii.get(self, "localGatewayRouteTableId"))

    @local_gateway_route_table_id.setter
    def local_gateway_route_table_id(self, value: builtins.str) -> None:
        jsii.set(self, "localGatewayRouteTableId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="vpcId")
    def vpc_id(self) -> builtins.str:
        '''``AWS::EC2::LocalGatewayRouteTableVPCAssociation.VpcId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-localgatewayroutetablevpcassociation.html#cfn-ec2-localgatewayroutetablevpcassociation-vpcid
        '''
        return typing.cast(builtins.str, jsii.get(self, "vpcId"))

    @vpc_id.setter
    def vpc_id(self, value: builtins.str) -> None:
        jsii.set(self, "vpcId", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnLocalGatewayRouteTableVPCAssociationProps",
    jsii_struct_bases=[],
    name_mapping={
        "local_gateway_route_table_id": "localGatewayRouteTableId",
        "vpc_id": "vpcId",
        "tags": "tags",
    },
)
class CfnLocalGatewayRouteTableVPCAssociationProps:
    def __init__(
        self,
        *,
        local_gateway_route_table_id: builtins.str,
        vpc_id: builtins.str,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::LocalGatewayRouteTableVPCAssociation``.

        :param local_gateway_route_table_id: ``AWS::EC2::LocalGatewayRouteTableVPCAssociation.LocalGatewayRouteTableId``.
        :param vpc_id: ``AWS::EC2::LocalGatewayRouteTableVPCAssociation.VpcId``.
        :param tags: ``AWS::EC2::LocalGatewayRouteTableVPCAssociation.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-localgatewayroutetablevpcassociation.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "local_gateway_route_table_id": local_gateway_route_table_id,
            "vpc_id": vpc_id,
        }
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def local_gateway_route_table_id(self) -> builtins.str:
        '''``AWS::EC2::LocalGatewayRouteTableVPCAssociation.LocalGatewayRouteTableId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-localgatewayroutetablevpcassociation.html#cfn-ec2-localgatewayroutetablevpcassociation-localgatewayroutetableid
        '''
        result = self._values.get("local_gateway_route_table_id")
        assert result is not None, "Required property 'local_gateway_route_table_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def vpc_id(self) -> builtins.str:
        '''``AWS::EC2::LocalGatewayRouteTableVPCAssociation.VpcId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-localgatewayroutetablevpcassociation.html#cfn-ec2-localgatewayroutetablevpcassociation-vpcid
        '''
        result = self._values.get("vpc_id")
        assert result is not None, "Required property 'vpc_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''``AWS::EC2::LocalGatewayRouteTableVPCAssociation.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-localgatewayroutetablevpcassociation.html#cfn-ec2-localgatewayroutetablevpcassociation-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnLocalGatewayRouteTableVPCAssociationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnNatGateway(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnNatGateway",
):
    '''A CloudFormation ``AWS::EC2::NatGateway``.

    :cloudformationResource: AWS::EC2::NatGateway
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-natgateway.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        subnet_id: builtins.str,
        allocation_id: typing.Optional[builtins.str] = None,
        connectivity_type: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::NatGateway``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param subnet_id: ``AWS::EC2::NatGateway.SubnetId``.
        :param allocation_id: ``AWS::EC2::NatGateway.AllocationId``.
        :param connectivity_type: ``AWS::EC2::NatGateway.ConnectivityType``.
        :param tags: ``AWS::EC2::NatGateway.Tags``.
        '''
        props = CfnNatGatewayProps(
            subnet_id=subnet_id,
            allocation_id=allocation_id,
            connectivity_type=connectivity_type,
            tags=tags,
        )

        jsii.create(CfnNatGateway, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''``AWS::EC2::NatGateway.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-natgateway.html#cfn-ec2-natgateway-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="subnetId")
    def subnet_id(self) -> builtins.str:
        '''``AWS::EC2::NatGateway.SubnetId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-natgateway.html#cfn-ec2-natgateway-subnetid
        '''
        return typing.cast(builtins.str, jsii.get(self, "subnetId"))

    @subnet_id.setter
    def subnet_id(self, value: builtins.str) -> None:
        jsii.set(self, "subnetId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="allocationId")
    def allocation_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NatGateway.AllocationId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-natgateway.html#cfn-ec2-natgateway-allocationid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "allocationId"))

    @allocation_id.setter
    def allocation_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "allocationId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="connectivityType")
    def connectivity_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NatGateway.ConnectivityType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-natgateway.html#cfn-ec2-natgateway-connectivitytype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "connectivityType"))

    @connectivity_type.setter
    def connectivity_type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "connectivityType", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnNatGatewayProps",
    jsii_struct_bases=[],
    name_mapping={
        "subnet_id": "subnetId",
        "allocation_id": "allocationId",
        "connectivity_type": "connectivityType",
        "tags": "tags",
    },
)
class CfnNatGatewayProps:
    def __init__(
        self,
        *,
        subnet_id: builtins.str,
        allocation_id: typing.Optional[builtins.str] = None,
        connectivity_type: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::NatGateway``.

        :param subnet_id: ``AWS::EC2::NatGateway.SubnetId``.
        :param allocation_id: ``AWS::EC2::NatGateway.AllocationId``.
        :param connectivity_type: ``AWS::EC2::NatGateway.ConnectivityType``.
        :param tags: ``AWS::EC2::NatGateway.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-natgateway.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "subnet_id": subnet_id,
        }
        if allocation_id is not None:
            self._values["allocation_id"] = allocation_id
        if connectivity_type is not None:
            self._values["connectivity_type"] = connectivity_type
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def subnet_id(self) -> builtins.str:
        '''``AWS::EC2::NatGateway.SubnetId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-natgateway.html#cfn-ec2-natgateway-subnetid
        '''
        result = self._values.get("subnet_id")
        assert result is not None, "Required property 'subnet_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def allocation_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NatGateway.AllocationId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-natgateway.html#cfn-ec2-natgateway-allocationid
        '''
        result = self._values.get("allocation_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def connectivity_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NatGateway.ConnectivityType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-natgateway.html#cfn-ec2-natgateway-connectivitytype
        '''
        result = self._values.get("connectivity_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''``AWS::EC2::NatGateway.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-natgateway.html#cfn-ec2-natgateway-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnNatGatewayProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnNetworkAcl(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnNetworkAcl",
):
    '''A CloudFormation ``AWS::EC2::NetworkAcl``.

    :cloudformationResource: AWS::EC2::NetworkAcl
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        vpc_id: builtins.str,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::NetworkAcl``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param vpc_id: ``AWS::EC2::NetworkAcl.VpcId``.
        :param tags: ``AWS::EC2::NetworkAcl.Tags``.
        '''
        props = CfnNetworkAclProps(vpc_id=vpc_id, tags=tags)

        jsii.create(CfnNetworkAcl, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''``AWS::EC2::NetworkAcl.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl.html#cfn-ec2-networkacl-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="vpcId")
    def vpc_id(self) -> builtins.str:
        '''``AWS::EC2::NetworkAcl.VpcId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl.html#cfn-ec2-networkacl-vpcid
        '''
        return typing.cast(builtins.str, jsii.get(self, "vpcId"))

    @vpc_id.setter
    def vpc_id(self, value: builtins.str) -> None:
        jsii.set(self, "vpcId", value)


@jsii.implements(aws_cdk.core.IInspectable)
class CfnNetworkAclEntry(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnNetworkAclEntry",
):
    '''A CloudFormation ``AWS::EC2::NetworkAclEntry``.

    :cloudformationResource: AWS::EC2::NetworkAclEntry
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        network_acl_id: builtins.str,
        protocol: jsii.Number,
        rule_action: builtins.str,
        rule_number: jsii.Number,
        cidr_block: typing.Optional[builtins.str] = None,
        egress: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        icmp: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkAclEntry.IcmpProperty"]] = None,
        ipv6_cidr_block: typing.Optional[builtins.str] = None,
        port_range: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkAclEntry.PortRangeProperty"]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::NetworkAclEntry``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param network_acl_id: ``AWS::EC2::NetworkAclEntry.NetworkAclId``.
        :param protocol: ``AWS::EC2::NetworkAclEntry.Protocol``.
        :param rule_action: ``AWS::EC2::NetworkAclEntry.RuleAction``.
        :param rule_number: ``AWS::EC2::NetworkAclEntry.RuleNumber``.
        :param cidr_block: ``AWS::EC2::NetworkAclEntry.CidrBlock``.
        :param egress: ``AWS::EC2::NetworkAclEntry.Egress``.
        :param icmp: ``AWS::EC2::NetworkAclEntry.Icmp``.
        :param ipv6_cidr_block: ``AWS::EC2::NetworkAclEntry.Ipv6CidrBlock``.
        :param port_range: ``AWS::EC2::NetworkAclEntry.PortRange``.
        '''
        props = CfnNetworkAclEntryProps(
            network_acl_id=network_acl_id,
            protocol=protocol,
            rule_action=rule_action,
            rule_number=rule_number,
            cidr_block=cidr_block,
            egress=egress,
            icmp=icmp,
            ipv6_cidr_block=ipv6_cidr_block,
            port_range=port_range,
        )

        jsii.create(CfnNetworkAclEntry, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkAclId")
    def network_acl_id(self) -> builtins.str:
        '''``AWS::EC2::NetworkAclEntry.NetworkAclId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-networkaclid
        '''
        return typing.cast(builtins.str, jsii.get(self, "networkAclId"))

    @network_acl_id.setter
    def network_acl_id(self, value: builtins.str) -> None:
        jsii.set(self, "networkAclId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="protocol")
    def protocol(self) -> jsii.Number:
        '''``AWS::EC2::NetworkAclEntry.Protocol``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-protocol
        '''
        return typing.cast(jsii.Number, jsii.get(self, "protocol"))

    @protocol.setter
    def protocol(self, value: jsii.Number) -> None:
        jsii.set(self, "protocol", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ruleAction")
    def rule_action(self) -> builtins.str:
        '''``AWS::EC2::NetworkAclEntry.RuleAction``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-ruleaction
        '''
        return typing.cast(builtins.str, jsii.get(self, "ruleAction"))

    @rule_action.setter
    def rule_action(self, value: builtins.str) -> None:
        jsii.set(self, "ruleAction", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ruleNumber")
    def rule_number(self) -> jsii.Number:
        '''``AWS::EC2::NetworkAclEntry.RuleNumber``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-rulenumber
        '''
        return typing.cast(jsii.Number, jsii.get(self, "ruleNumber"))

    @rule_number.setter
    def rule_number(self, value: jsii.Number) -> None:
        jsii.set(self, "ruleNumber", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cidrBlock")
    def cidr_block(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NetworkAclEntry.CidrBlock``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-cidrblock
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cidrBlock"))

    @cidr_block.setter
    def cidr_block(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "cidrBlock", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="egress")
    def egress(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::NetworkAclEntry.Egress``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-egress
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "egress"))

    @egress.setter
    def egress(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "egress", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="icmp")
    def icmp(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkAclEntry.IcmpProperty"]]:
        '''``AWS::EC2::NetworkAclEntry.Icmp``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-icmp
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkAclEntry.IcmpProperty"]], jsii.get(self, "icmp"))

    @icmp.setter
    def icmp(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkAclEntry.IcmpProperty"]],
    ) -> None:
        jsii.set(self, "icmp", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ipv6CidrBlock")
    def ipv6_cidr_block(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NetworkAclEntry.Ipv6CidrBlock``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-ipv6cidrblock
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "ipv6CidrBlock"))

    @ipv6_cidr_block.setter
    def ipv6_cidr_block(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "ipv6CidrBlock", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="portRange")
    def port_range(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkAclEntry.PortRangeProperty"]]:
        '''``AWS::EC2::NetworkAclEntry.PortRange``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-portrange
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkAclEntry.PortRangeProperty"]], jsii.get(self, "portRange"))

    @port_range.setter
    def port_range(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkAclEntry.PortRangeProperty"]],
    ) -> None:
        jsii.set(self, "portRange", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnNetworkAclEntry.IcmpProperty",
        jsii_struct_bases=[],
        name_mapping={"code": "code", "type": "type"},
    )
    class IcmpProperty:
        def __init__(
            self,
            *,
            code: typing.Optional[jsii.Number] = None,
            type: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param code: ``CfnNetworkAclEntry.IcmpProperty.Code``.
            :param type: ``CfnNetworkAclEntry.IcmpProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkaclentry-icmp.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if code is not None:
                self._values["code"] = code
            if type is not None:
                self._values["type"] = type

        @builtins.property
        def code(self) -> typing.Optional[jsii.Number]:
            '''``CfnNetworkAclEntry.IcmpProperty.Code``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkaclentry-icmp.html#cfn-ec2-networkaclentry-icmp-code
            '''
            result = self._values.get("code")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def type(self) -> typing.Optional[jsii.Number]:
            '''``CfnNetworkAclEntry.IcmpProperty.Type``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkaclentry-icmp.html#cfn-ec2-networkaclentry-icmp-type
            '''
            result = self._values.get("type")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "IcmpProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnNetworkAclEntry.PortRangeProperty",
        jsii_struct_bases=[],
        name_mapping={"from_": "from", "to": "to"},
    )
    class PortRangeProperty:
        def __init__(
            self,
            *,
            from_: typing.Optional[jsii.Number] = None,
            to: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param from_: ``CfnNetworkAclEntry.PortRangeProperty.From``.
            :param to: ``CfnNetworkAclEntry.PortRangeProperty.To``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkaclentry-portrange.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if from_ is not None:
                self._values["from_"] = from_
            if to is not None:
                self._values["to"] = to

        @builtins.property
        def from_(self) -> typing.Optional[jsii.Number]:
            '''``CfnNetworkAclEntry.PortRangeProperty.From``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkaclentry-portrange.html#cfn-ec2-networkaclentry-portrange-from
            '''
            result = self._values.get("from_")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def to(self) -> typing.Optional[jsii.Number]:
            '''``CfnNetworkAclEntry.PortRangeProperty.To``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkaclentry-portrange.html#cfn-ec2-networkaclentry-portrange-to
            '''
            result = self._values.get("to")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PortRangeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnNetworkAclEntryProps",
    jsii_struct_bases=[],
    name_mapping={
        "network_acl_id": "networkAclId",
        "protocol": "protocol",
        "rule_action": "ruleAction",
        "rule_number": "ruleNumber",
        "cidr_block": "cidrBlock",
        "egress": "egress",
        "icmp": "icmp",
        "ipv6_cidr_block": "ipv6CidrBlock",
        "port_range": "portRange",
    },
)
class CfnNetworkAclEntryProps:
    def __init__(
        self,
        *,
        network_acl_id: builtins.str,
        protocol: jsii.Number,
        rule_action: builtins.str,
        rule_number: jsii.Number,
        cidr_block: typing.Optional[builtins.str] = None,
        egress: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        icmp: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnNetworkAclEntry.IcmpProperty]] = None,
        ipv6_cidr_block: typing.Optional[builtins.str] = None,
        port_range: typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnNetworkAclEntry.PortRangeProperty]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::NetworkAclEntry``.

        :param network_acl_id: ``AWS::EC2::NetworkAclEntry.NetworkAclId``.
        :param protocol: ``AWS::EC2::NetworkAclEntry.Protocol``.
        :param rule_action: ``AWS::EC2::NetworkAclEntry.RuleAction``.
        :param rule_number: ``AWS::EC2::NetworkAclEntry.RuleNumber``.
        :param cidr_block: ``AWS::EC2::NetworkAclEntry.CidrBlock``.
        :param egress: ``AWS::EC2::NetworkAclEntry.Egress``.
        :param icmp: ``AWS::EC2::NetworkAclEntry.Icmp``.
        :param ipv6_cidr_block: ``AWS::EC2::NetworkAclEntry.Ipv6CidrBlock``.
        :param port_range: ``AWS::EC2::NetworkAclEntry.PortRange``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "network_acl_id": network_acl_id,
            "protocol": protocol,
            "rule_action": rule_action,
            "rule_number": rule_number,
        }
        if cidr_block is not None:
            self._values["cidr_block"] = cidr_block
        if egress is not None:
            self._values["egress"] = egress
        if icmp is not None:
            self._values["icmp"] = icmp
        if ipv6_cidr_block is not None:
            self._values["ipv6_cidr_block"] = ipv6_cidr_block
        if port_range is not None:
            self._values["port_range"] = port_range

    @builtins.property
    def network_acl_id(self) -> builtins.str:
        '''``AWS::EC2::NetworkAclEntry.NetworkAclId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-networkaclid
        '''
        result = self._values.get("network_acl_id")
        assert result is not None, "Required property 'network_acl_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def protocol(self) -> jsii.Number:
        '''``AWS::EC2::NetworkAclEntry.Protocol``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-protocol
        '''
        result = self._values.get("protocol")
        assert result is not None, "Required property 'protocol' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def rule_action(self) -> builtins.str:
        '''``AWS::EC2::NetworkAclEntry.RuleAction``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-ruleaction
        '''
        result = self._values.get("rule_action")
        assert result is not None, "Required property 'rule_action' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def rule_number(self) -> jsii.Number:
        '''``AWS::EC2::NetworkAclEntry.RuleNumber``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-rulenumber
        '''
        result = self._values.get("rule_number")
        assert result is not None, "Required property 'rule_number' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def cidr_block(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NetworkAclEntry.CidrBlock``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-cidrblock
        '''
        result = self._values.get("cidr_block")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def egress(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::NetworkAclEntry.Egress``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-egress
        '''
        result = self._values.get("egress")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def icmp(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnNetworkAclEntry.IcmpProperty]]:
        '''``AWS::EC2::NetworkAclEntry.Icmp``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-icmp
        '''
        result = self._values.get("icmp")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnNetworkAclEntry.IcmpProperty]], result)

    @builtins.property
    def ipv6_cidr_block(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NetworkAclEntry.Ipv6CidrBlock``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-ipv6cidrblock
        '''
        result = self._values.get("ipv6_cidr_block")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def port_range(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnNetworkAclEntry.PortRangeProperty]]:
        '''``AWS::EC2::NetworkAclEntry.PortRange``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl-entry.html#cfn-ec2-networkaclentry-portrange
        '''
        result = self._values.get("port_range")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, CfnNetworkAclEntry.PortRangeProperty]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnNetworkAclEntryProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnNetworkAclProps",
    jsii_struct_bases=[],
    name_mapping={"vpc_id": "vpcId", "tags": "tags"},
)
class CfnNetworkAclProps:
    def __init__(
        self,
        *,
        vpc_id: builtins.str,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::NetworkAcl``.

        :param vpc_id: ``AWS::EC2::NetworkAcl.VpcId``.
        :param tags: ``AWS::EC2::NetworkAcl.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "vpc_id": vpc_id,
        }
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def vpc_id(self) -> builtins.str:
        '''``AWS::EC2::NetworkAcl.VpcId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl.html#cfn-ec2-networkacl-vpcid
        '''
        result = self._values.get("vpc_id")
        assert result is not None, "Required property 'vpc_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''``AWS::EC2::NetworkAcl.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-acl.html#cfn-ec2-networkacl-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnNetworkAclProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnNetworkInsightsAnalysis(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnNetworkInsightsAnalysis",
):
    '''A CloudFormation ``AWS::EC2::NetworkInsightsAnalysis``.

    :cloudformationResource: AWS::EC2::NetworkInsightsAnalysis
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightsanalysis.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        network_insights_path_id: builtins.str,
        filter_in_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::NetworkInsightsAnalysis``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param network_insights_path_id: ``AWS::EC2::NetworkInsightsAnalysis.NetworkInsightsPathId``.
        :param filter_in_arns: ``AWS::EC2::NetworkInsightsAnalysis.FilterInArns``.
        :param tags: ``AWS::EC2::NetworkInsightsAnalysis.Tags``.
        '''
        props = CfnNetworkInsightsAnalysisProps(
            network_insights_path_id=network_insights_path_id,
            filter_in_arns=filter_in_arns,
            tags=tags,
        )

        jsii.create(CfnNetworkInsightsAnalysis, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrAlternatePathHints")
    def attr_alternate_path_hints(self) -> aws_cdk.core.IResolvable:
        '''
        :cloudformationAttribute: AlternatePathHints
        '''
        return typing.cast(aws_cdk.core.IResolvable, jsii.get(self, "attrAlternatePathHints"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrExplanations")
    def attr_explanations(self) -> aws_cdk.core.IResolvable:
        '''
        :cloudformationAttribute: Explanations
        '''
        return typing.cast(aws_cdk.core.IResolvable, jsii.get(self, "attrExplanations"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrForwardPathComponents")
    def attr_forward_path_components(self) -> aws_cdk.core.IResolvable:
        '''
        :cloudformationAttribute: ForwardPathComponents
        '''
        return typing.cast(aws_cdk.core.IResolvable, jsii.get(self, "attrForwardPathComponents"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrNetworkInsightsAnalysisArn")
    def attr_network_insights_analysis_arn(self) -> builtins.str:
        '''
        :cloudformationAttribute: NetworkInsightsAnalysisArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrNetworkInsightsAnalysisArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrNetworkInsightsAnalysisId")
    def attr_network_insights_analysis_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: NetworkInsightsAnalysisId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrNetworkInsightsAnalysisId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrNetworkPathFound")
    def attr_network_path_found(self) -> aws_cdk.core.IResolvable:
        '''
        :cloudformationAttribute: NetworkPathFound
        '''
        return typing.cast(aws_cdk.core.IResolvable, jsii.get(self, "attrNetworkPathFound"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrReturnPathComponents")
    def attr_return_path_components(self) -> aws_cdk.core.IResolvable:
        '''
        :cloudformationAttribute: ReturnPathComponents
        '''
        return typing.cast(aws_cdk.core.IResolvable, jsii.get(self, "attrReturnPathComponents"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrStartDate")
    def attr_start_date(self) -> builtins.str:
        '''
        :cloudformationAttribute: StartDate
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrStartDate"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrStatus")
    def attr_status(self) -> builtins.str:
        '''
        :cloudformationAttribute: Status
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrStatus"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrStatusMessage")
    def attr_status_message(self) -> builtins.str:
        '''
        :cloudformationAttribute: StatusMessage
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrStatusMessage"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''``AWS::EC2::NetworkInsightsAnalysis.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightsanalysis.html#cfn-ec2-networkinsightsanalysis-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkInsightsPathId")
    def network_insights_path_id(self) -> builtins.str:
        '''``AWS::EC2::NetworkInsightsAnalysis.NetworkInsightsPathId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightsanalysis.html#cfn-ec2-networkinsightsanalysis-networkinsightspathid
        '''
        return typing.cast(builtins.str, jsii.get(self, "networkInsightsPathId"))

    @network_insights_path_id.setter
    def network_insights_path_id(self, value: builtins.str) -> None:
        jsii.set(self, "networkInsightsPathId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="filterInArns")
    def filter_in_arns(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::NetworkInsightsAnalysis.FilterInArns``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightsanalysis.html#cfn-ec2-networkinsightsanalysis-filterinarns
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "filterInArns"))

    @filter_in_arns.setter
    def filter_in_arns(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        jsii.set(self, "filterInArns", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnNetworkInsightsAnalysis.AlternatePathHintProperty",
        jsii_struct_bases=[],
        name_mapping={"component_arn": "componentArn", "component_id": "componentId"},
    )
    class AlternatePathHintProperty:
        def __init__(
            self,
            *,
            component_arn: typing.Optional[builtins.str] = None,
            component_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param component_arn: ``CfnNetworkInsightsAnalysis.AlternatePathHintProperty.ComponentArn``.
            :param component_id: ``CfnNetworkInsightsAnalysis.AlternatePathHintProperty.ComponentId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-alternatepathhint.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if component_arn is not None:
                self._values["component_arn"] = component_arn
            if component_id is not None:
                self._values["component_id"] = component_id

        @builtins.property
        def component_arn(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AlternatePathHintProperty.ComponentArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-alternatepathhint.html#cfn-ec2-networkinsightsanalysis-alternatepathhint-componentarn
            '''
            result = self._values.get("component_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def component_id(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AlternatePathHintProperty.ComponentId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-alternatepathhint.html#cfn-ec2-networkinsightsanalysis-alternatepathhint-componentid
            '''
            result = self._values.get("component_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AlternatePathHintProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cidr": "cidr",
            "egress": "egress",
            "port_range": "portRange",
            "protocol": "protocol",
            "rule_action": "ruleAction",
            "rule_number": "ruleNumber",
        },
    )
    class AnalysisAclRuleProperty:
        def __init__(
            self,
            *,
            cidr: typing.Optional[builtins.str] = None,
            egress: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            port_range: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.PortRangeProperty"]] = None,
            protocol: typing.Optional[builtins.str] = None,
            rule_action: typing.Optional[builtins.str] = None,
            rule_number: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param cidr: ``CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty.Cidr``.
            :param egress: ``CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty.Egress``.
            :param port_range: ``CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty.PortRange``.
            :param protocol: ``CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty.Protocol``.
            :param rule_action: ``CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty.RuleAction``.
            :param rule_number: ``CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty.RuleNumber``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisaclrule.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if cidr is not None:
                self._values["cidr"] = cidr
            if egress is not None:
                self._values["egress"] = egress
            if port_range is not None:
                self._values["port_range"] = port_range
            if protocol is not None:
                self._values["protocol"] = protocol
            if rule_action is not None:
                self._values["rule_action"] = rule_action
            if rule_number is not None:
                self._values["rule_number"] = rule_number

        @builtins.property
        def cidr(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty.Cidr``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisaclrule.html#cfn-ec2-networkinsightsanalysis-analysisaclrule-cidr
            '''
            result = self._values.get("cidr")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def egress(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty.Egress``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisaclrule.html#cfn-ec2-networkinsightsanalysis-analysisaclrule-egress
            '''
            result = self._values.get("egress")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def port_range(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.PortRangeProperty"]]:
            '''``CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty.PortRange``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisaclrule.html#cfn-ec2-networkinsightsanalysis-analysisaclrule-portrange
            '''
            result = self._values.get("port_range")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.PortRangeProperty"]], result)

        @builtins.property
        def protocol(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty.Protocol``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisaclrule.html#cfn-ec2-networkinsightsanalysis-analysisaclrule-protocol
            '''
            result = self._values.get("protocol")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def rule_action(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty.RuleAction``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisaclrule.html#cfn-ec2-networkinsightsanalysis-analysisaclrule-ruleaction
            '''
            result = self._values.get("rule_action")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def rule_number(self) -> typing.Optional[jsii.Number]:
            '''``CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty.RuleNumber``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisaclrule.html#cfn-ec2-networkinsightsanalysis-analysisaclrule-rulenumber
            '''
            result = self._values.get("rule_number")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AnalysisAclRuleProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnNetworkInsightsAnalysis.AnalysisComponentProperty",
        jsii_struct_bases=[],
        name_mapping={"arn": "arn", "id": "id"},
    )
    class AnalysisComponentProperty:
        def __init__(
            self,
            *,
            arn: typing.Optional[builtins.str] = None,
            id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param arn: ``CfnNetworkInsightsAnalysis.AnalysisComponentProperty.Arn``.
            :param id: ``CfnNetworkInsightsAnalysis.AnalysisComponentProperty.Id``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysiscomponent.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if arn is not None:
                self._values["arn"] = arn
            if id is not None:
                self._values["id"] = id

        @builtins.property
        def arn(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisComponentProperty.Arn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysiscomponent.html#cfn-ec2-networkinsightsanalysis-analysiscomponent-arn
            '''
            result = self._values.get("arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def id(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisComponentProperty.Id``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysiscomponent.html#cfn-ec2-networkinsightsanalysis-analysiscomponent-id
            '''
            result = self._values.get("id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AnalysisComponentProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnNetworkInsightsAnalysis.AnalysisLoadBalancerListenerProperty",
        jsii_struct_bases=[],
        name_mapping={
            "instance_port": "instancePort",
            "load_balancer_port": "loadBalancerPort",
        },
    )
    class AnalysisLoadBalancerListenerProperty:
        def __init__(
            self,
            *,
            instance_port: typing.Optional[jsii.Number] = None,
            load_balancer_port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param instance_port: ``CfnNetworkInsightsAnalysis.AnalysisLoadBalancerListenerProperty.InstancePort``.
            :param load_balancer_port: ``CfnNetworkInsightsAnalysis.AnalysisLoadBalancerListenerProperty.LoadBalancerPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisloadbalancerlistener.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if instance_port is not None:
                self._values["instance_port"] = instance_port
            if load_balancer_port is not None:
                self._values["load_balancer_port"] = load_balancer_port

        @builtins.property
        def instance_port(self) -> typing.Optional[jsii.Number]:
            '''``CfnNetworkInsightsAnalysis.AnalysisLoadBalancerListenerProperty.InstancePort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisloadbalancerlistener.html#cfn-ec2-networkinsightsanalysis-analysisloadbalancerlistener-instanceport
            '''
            result = self._values.get("instance_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def load_balancer_port(self) -> typing.Optional[jsii.Number]:
            '''``CfnNetworkInsightsAnalysis.AnalysisLoadBalancerListenerProperty.LoadBalancerPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisloadbalancerlistener.html#cfn-ec2-networkinsightsanalysis-analysisloadbalancerlistener-loadbalancerport
            '''
            result = self._values.get("load_balancer_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AnalysisLoadBalancerListenerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnNetworkInsightsAnalysis.AnalysisLoadBalancerTargetProperty",
        jsii_struct_bases=[],
        name_mapping={
            "address": "address",
            "availability_zone": "availabilityZone",
            "instance": "instance",
            "port": "port",
        },
    )
    class AnalysisLoadBalancerTargetProperty:
        def __init__(
            self,
            *,
            address: typing.Optional[builtins.str] = None,
            availability_zone: typing.Optional[builtins.str] = None,
            instance: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param address: ``CfnNetworkInsightsAnalysis.AnalysisLoadBalancerTargetProperty.Address``.
            :param availability_zone: ``CfnNetworkInsightsAnalysis.AnalysisLoadBalancerTargetProperty.AvailabilityZone``.
            :param instance: ``CfnNetworkInsightsAnalysis.AnalysisLoadBalancerTargetProperty.Instance``.
            :param port: ``CfnNetworkInsightsAnalysis.AnalysisLoadBalancerTargetProperty.Port``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisloadbalancertarget.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if address is not None:
                self._values["address"] = address
            if availability_zone is not None:
                self._values["availability_zone"] = availability_zone
            if instance is not None:
                self._values["instance"] = instance
            if port is not None:
                self._values["port"] = port

        @builtins.property
        def address(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisLoadBalancerTargetProperty.Address``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisloadbalancertarget.html#cfn-ec2-networkinsightsanalysis-analysisloadbalancertarget-address
            '''
            result = self._values.get("address")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def availability_zone(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisLoadBalancerTargetProperty.AvailabilityZone``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisloadbalancertarget.html#cfn-ec2-networkinsightsanalysis-analysisloadbalancertarget-availabilityzone
            '''
            result = self._values.get("availability_zone")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.AnalysisLoadBalancerTargetProperty.Instance``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisloadbalancertarget.html#cfn-ec2-networkinsightsanalysis-analysisloadbalancertarget-instance
            '''
            result = self._values.get("instance")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''``CfnNetworkInsightsAnalysis.AnalysisLoadBalancerTargetProperty.Port``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisloadbalancertarget.html#cfn-ec2-networkinsightsanalysis-analysisloadbalancertarget-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AnalysisLoadBalancerTargetProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnNetworkInsightsAnalysis.AnalysisPacketHeaderProperty",
        jsii_struct_bases=[],
        name_mapping={
            "destination_addresses": "destinationAddresses",
            "destination_port_ranges": "destinationPortRanges",
            "protocol": "protocol",
            "source_addresses": "sourceAddresses",
            "source_port_ranges": "sourcePortRanges",
        },
    )
    class AnalysisPacketHeaderProperty:
        def __init__(
            self,
            *,
            destination_addresses: typing.Optional[typing.Sequence[builtins.str]] = None,
            destination_port_ranges: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.PortRangeProperty"]]]] = None,
            protocol: typing.Optional[builtins.str] = None,
            source_addresses: typing.Optional[typing.Sequence[builtins.str]] = None,
            source_port_ranges: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.PortRangeProperty"]]]] = None,
        ) -> None:
            '''
            :param destination_addresses: ``CfnNetworkInsightsAnalysis.AnalysisPacketHeaderProperty.DestinationAddresses``.
            :param destination_port_ranges: ``CfnNetworkInsightsAnalysis.AnalysisPacketHeaderProperty.DestinationPortRanges``.
            :param protocol: ``CfnNetworkInsightsAnalysis.AnalysisPacketHeaderProperty.Protocol``.
            :param source_addresses: ``CfnNetworkInsightsAnalysis.AnalysisPacketHeaderProperty.SourceAddresses``.
            :param source_port_ranges: ``CfnNetworkInsightsAnalysis.AnalysisPacketHeaderProperty.SourcePortRanges``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysispacketheader.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if destination_addresses is not None:
                self._values["destination_addresses"] = destination_addresses
            if destination_port_ranges is not None:
                self._values["destination_port_ranges"] = destination_port_ranges
            if protocol is not None:
                self._values["protocol"] = protocol
            if source_addresses is not None:
                self._values["source_addresses"] = source_addresses
            if source_port_ranges is not None:
                self._values["source_port_ranges"] = source_port_ranges

        @builtins.property
        def destination_addresses(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnNetworkInsightsAnalysis.AnalysisPacketHeaderProperty.DestinationAddresses``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysispacketheader.html#cfn-ec2-networkinsightsanalysis-analysispacketheader-destinationaddresses
            '''
            result = self._values.get("destination_addresses")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def destination_port_ranges(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.PortRangeProperty"]]]]:
            '''``CfnNetworkInsightsAnalysis.AnalysisPacketHeaderProperty.DestinationPortRanges``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysispacketheader.html#cfn-ec2-networkinsightsanalysis-analysispacketheader-destinationportranges
            '''
            result = self._values.get("destination_port_ranges")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.PortRangeProperty"]]]], result)

        @builtins.property
        def protocol(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisPacketHeaderProperty.Protocol``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysispacketheader.html#cfn-ec2-networkinsightsanalysis-analysispacketheader-protocol
            '''
            result = self._values.get("protocol")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def source_addresses(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnNetworkInsightsAnalysis.AnalysisPacketHeaderProperty.SourceAddresses``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysispacketheader.html#cfn-ec2-networkinsightsanalysis-analysispacketheader-sourceaddresses
            '''
            result = self._values.get("source_addresses")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def source_port_ranges(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.PortRangeProperty"]]]]:
            '''``CfnNetworkInsightsAnalysis.AnalysisPacketHeaderProperty.SourcePortRanges``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysispacketheader.html#cfn-ec2-networkinsightsanalysis-analysispacketheader-sourceportranges
            '''
            result = self._values.get("source_port_ranges")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.PortRangeProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AnalysisPacketHeaderProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty",
        jsii_struct_bases=[],
        name_mapping={
            "destination_cidr": "destinationCidr",
            "destination_prefix_list_id": "destinationPrefixListId",
            "egress_only_internet_gateway_id": "egressOnlyInternetGatewayId",
            "gateway_id": "gatewayId",
            "instance_id": "instanceId",
            "nat_gateway_id": "natGatewayId",
            "network_interface_id": "networkInterfaceId",
            "origin": "origin",
            "transit_gateway_id": "transitGatewayId",
            "vpc_peering_connection_id": "vpcPeeringConnectionId",
        },
    )
    class AnalysisRouteTableRouteProperty:
        def __init__(
            self,
            *,
            destination_cidr: typing.Optional[builtins.str] = None,
            destination_prefix_list_id: typing.Optional[builtins.str] = None,
            egress_only_internet_gateway_id: typing.Optional[builtins.str] = None,
            gateway_id: typing.Optional[builtins.str] = None,
            instance_id: typing.Optional[builtins.str] = None,
            nat_gateway_id: typing.Optional[builtins.str] = None,
            network_interface_id: typing.Optional[builtins.str] = None,
            origin: typing.Optional[builtins.str] = None,
            transit_gateway_id: typing.Optional[builtins.str] = None,
            vpc_peering_connection_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param destination_cidr: ``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.destinationCidr``.
            :param destination_prefix_list_id: ``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.destinationPrefixListId``.
            :param egress_only_internet_gateway_id: ``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.egressOnlyInternetGatewayId``.
            :param gateway_id: ``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.gatewayId``.
            :param instance_id: ``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.instanceId``.
            :param nat_gateway_id: ``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.NatGatewayId``.
            :param network_interface_id: ``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.NetworkInterfaceId``.
            :param origin: ``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.Origin``.
            :param transit_gateway_id: ``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.TransitGatewayId``.
            :param vpc_peering_connection_id: ``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.VpcPeeringConnectionId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisroutetableroute.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if destination_cidr is not None:
                self._values["destination_cidr"] = destination_cidr
            if destination_prefix_list_id is not None:
                self._values["destination_prefix_list_id"] = destination_prefix_list_id
            if egress_only_internet_gateway_id is not None:
                self._values["egress_only_internet_gateway_id"] = egress_only_internet_gateway_id
            if gateway_id is not None:
                self._values["gateway_id"] = gateway_id
            if instance_id is not None:
                self._values["instance_id"] = instance_id
            if nat_gateway_id is not None:
                self._values["nat_gateway_id"] = nat_gateway_id
            if network_interface_id is not None:
                self._values["network_interface_id"] = network_interface_id
            if origin is not None:
                self._values["origin"] = origin
            if transit_gateway_id is not None:
                self._values["transit_gateway_id"] = transit_gateway_id
            if vpc_peering_connection_id is not None:
                self._values["vpc_peering_connection_id"] = vpc_peering_connection_id

        @builtins.property
        def destination_cidr(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.destinationCidr``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisroutetableroute.html#cfn-ec2-networkinsightsanalysis-analysisroutetableroute-destinationcidr
            '''
            result = self._values.get("destination_cidr")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def destination_prefix_list_id(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.destinationPrefixListId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisroutetableroute.html#cfn-ec2-networkinsightsanalysis-analysisroutetableroute-destinationprefixlistid
            '''
            result = self._values.get("destination_prefix_list_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def egress_only_internet_gateway_id(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.egressOnlyInternetGatewayId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisroutetableroute.html#cfn-ec2-networkinsightsanalysis-analysisroutetableroute-egressonlyinternetgatewayid
            '''
            result = self._values.get("egress_only_internet_gateway_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def gateway_id(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.gatewayId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisroutetableroute.html#cfn-ec2-networkinsightsanalysis-analysisroutetableroute-gatewayid
            '''
            result = self._values.get("gateway_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def instance_id(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.instanceId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisroutetableroute.html#cfn-ec2-networkinsightsanalysis-analysisroutetableroute-instanceid
            '''
            result = self._values.get("instance_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def nat_gateway_id(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.NatGatewayId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisroutetableroute.html#cfn-ec2-networkinsightsanalysis-analysisroutetableroute-natgatewayid
            '''
            result = self._values.get("nat_gateway_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def network_interface_id(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.NetworkInterfaceId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisroutetableroute.html#cfn-ec2-networkinsightsanalysis-analysisroutetableroute-networkinterfaceid
            '''
            result = self._values.get("network_interface_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def origin(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.Origin``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisroutetableroute.html#cfn-ec2-networkinsightsanalysis-analysisroutetableroute-origin
            '''
            result = self._values.get("origin")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def transit_gateway_id(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.TransitGatewayId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisroutetableroute.html#cfn-ec2-networkinsightsanalysis-analysisroutetableroute-transitgatewayid
            '''
            result = self._values.get("transit_gateway_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def vpc_peering_connection_id(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty.VpcPeeringConnectionId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysisroutetableroute.html#cfn-ec2-networkinsightsanalysis-analysisroutetableroute-vpcpeeringconnectionid
            '''
            result = self._values.get("vpc_peering_connection_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AnalysisRouteTableRouteProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty",
        jsii_struct_bases=[],
        name_mapping={
            "cidr": "cidr",
            "direction": "direction",
            "port_range": "portRange",
            "prefix_list_id": "prefixListId",
            "protocol": "protocol",
            "security_group_id": "securityGroupId",
        },
    )
    class AnalysisSecurityGroupRuleProperty:
        def __init__(
            self,
            *,
            cidr: typing.Optional[builtins.str] = None,
            direction: typing.Optional[builtins.str] = None,
            port_range: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.PortRangeProperty"]] = None,
            prefix_list_id: typing.Optional[builtins.str] = None,
            protocol: typing.Optional[builtins.str] = None,
            security_group_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param cidr: ``CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty.Cidr``.
            :param direction: ``CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty.Direction``.
            :param port_range: ``CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty.PortRange``.
            :param prefix_list_id: ``CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty.PrefixListId``.
            :param protocol: ``CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty.Protocol``.
            :param security_group_id: ``CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty.SecurityGroupId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysissecuritygrouprule.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if cidr is not None:
                self._values["cidr"] = cidr
            if direction is not None:
                self._values["direction"] = direction
            if port_range is not None:
                self._values["port_range"] = port_range
            if prefix_list_id is not None:
                self._values["prefix_list_id"] = prefix_list_id
            if protocol is not None:
                self._values["protocol"] = protocol
            if security_group_id is not None:
                self._values["security_group_id"] = security_group_id

        @builtins.property
        def cidr(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty.Cidr``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysissecuritygrouprule.html#cfn-ec2-networkinsightsanalysis-analysissecuritygrouprule-cidr
            '''
            result = self._values.get("cidr")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def direction(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty.Direction``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysissecuritygrouprule.html#cfn-ec2-networkinsightsanalysis-analysissecuritygrouprule-direction
            '''
            result = self._values.get("direction")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def port_range(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.PortRangeProperty"]]:
            '''``CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty.PortRange``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysissecuritygrouprule.html#cfn-ec2-networkinsightsanalysis-analysissecuritygrouprule-portrange
            '''
            result = self._values.get("port_range")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.PortRangeProperty"]], result)

        @builtins.property
        def prefix_list_id(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty.PrefixListId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysissecuritygrouprule.html#cfn-ec2-networkinsightsanalysis-analysissecuritygrouprule-prefixlistid
            '''
            result = self._values.get("prefix_list_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def protocol(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty.Protocol``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysissecuritygrouprule.html#cfn-ec2-networkinsightsanalysis-analysissecuritygrouprule-protocol
            '''
            result = self._values.get("protocol")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def security_group_id(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty.SecurityGroupId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-analysissecuritygrouprule.html#cfn-ec2-networkinsightsanalysis-analysissecuritygrouprule-securitygroupid
            '''
            result = self._values.get("security_group_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "AnalysisSecurityGroupRuleProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnNetworkInsightsAnalysis.ExplanationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "acl": "acl",
            "acl_rule": "aclRule",
            "address": "address",
            "addresses": "addresses",
            "attached_to": "attachedTo",
            "availability_zones": "availabilityZones",
            "cidrs": "cidrs",
            "classic_load_balancer_listener": "classicLoadBalancerListener",
            "component": "component",
            "customer_gateway": "customerGateway",
            "destination": "destination",
            "destination_vpc": "destinationVpc",
            "direction": "direction",
            "elastic_load_balancer_listener": "elasticLoadBalancerListener",
            "explanation_code": "explanationCode",
            "ingress_route_table": "ingressRouteTable",
            "internet_gateway": "internetGateway",
            "load_balancer_arn": "loadBalancerArn",
            "load_balancer_listener_port": "loadBalancerListenerPort",
            "load_balancer_target": "loadBalancerTarget",
            "load_balancer_target_group": "loadBalancerTargetGroup",
            "load_balancer_target_groups": "loadBalancerTargetGroups",
            "load_balancer_target_port": "loadBalancerTargetPort",
            "missing_component": "missingComponent",
            "nat_gateway": "natGateway",
            "network_interface": "networkInterface",
            "packet_field": "packetField",
            "port": "port",
            "port_ranges": "portRanges",
            "prefix_list": "prefixList",
            "protocols": "protocols",
            "route_table": "routeTable",
            "route_table_route": "routeTableRoute",
            "security_group": "securityGroup",
            "security_group_rule": "securityGroupRule",
            "security_groups": "securityGroups",
            "source_vpc": "sourceVpc",
            "state": "state",
            "subnet": "subnet",
            "subnet_route_table": "subnetRouteTable",
            "vpc": "vpc",
            "vpc_endpoint": "vpcEndpoint",
            "vpc_peering_connection": "vpcPeeringConnection",
            "vpn_connection": "vpnConnection",
            "vpn_gateway": "vpnGateway",
        },
    )
    class ExplanationProperty:
        def __init__(
            self,
            *,
            acl: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            acl_rule: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty"]] = None,
            address: typing.Optional[builtins.str] = None,
            addresses: typing.Optional[typing.Sequence[builtins.str]] = None,
            attached_to: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            availability_zones: typing.Optional[typing.Sequence[builtins.str]] = None,
            cidrs: typing.Optional[typing.Sequence[builtins.str]] = None,
            classic_load_balancer_listener: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisLoadBalancerListenerProperty"]] = None,
            component: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            customer_gateway: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            destination: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            destination_vpc: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            direction: typing.Optional[builtins.str] = None,
            elastic_load_balancer_listener: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            explanation_code: typing.Optional[builtins.str] = None,
            ingress_route_table: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            internet_gateway: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            load_balancer_arn: typing.Optional[builtins.str] = None,
            load_balancer_listener_port: typing.Optional[jsii.Number] = None,
            load_balancer_target: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisLoadBalancerTargetProperty"]] = None,
            load_balancer_target_group: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            load_balancer_target_groups: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]]] = None,
            load_balancer_target_port: typing.Optional[jsii.Number] = None,
            missing_component: typing.Optional[builtins.str] = None,
            nat_gateway: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            network_interface: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            packet_field: typing.Optional[builtins.str] = None,
            port: typing.Optional[jsii.Number] = None,
            port_ranges: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.PortRangeProperty"]]]] = None,
            prefix_list: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            protocols: typing.Optional[typing.Sequence[builtins.str]] = None,
            route_table: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            route_table_route: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty"]] = None,
            security_group: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            security_group_rule: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty"]] = None,
            security_groups: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]]] = None,
            source_vpc: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            state: typing.Optional[builtins.str] = None,
            subnet: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            subnet_route_table: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            vpc: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            vpc_endpoint: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            vpc_peering_connection: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            vpn_connection: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            vpn_gateway: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
        ) -> None:
            '''
            :param acl: ``CfnNetworkInsightsAnalysis.ExplanationProperty.Acl``.
            :param acl_rule: ``CfnNetworkInsightsAnalysis.ExplanationProperty.AclRule``.
            :param address: ``CfnNetworkInsightsAnalysis.ExplanationProperty.Address``.
            :param addresses: ``CfnNetworkInsightsAnalysis.ExplanationProperty.Addresses``.
            :param attached_to: ``CfnNetworkInsightsAnalysis.ExplanationProperty.AttachedTo``.
            :param availability_zones: ``CfnNetworkInsightsAnalysis.ExplanationProperty.AvailabilityZones``.
            :param cidrs: ``CfnNetworkInsightsAnalysis.ExplanationProperty.Cidrs``.
            :param classic_load_balancer_listener: ``CfnNetworkInsightsAnalysis.ExplanationProperty.ClassicLoadBalancerListener``.
            :param component: ``CfnNetworkInsightsAnalysis.ExplanationProperty.Component``.
            :param customer_gateway: ``CfnNetworkInsightsAnalysis.ExplanationProperty.CustomerGateway``.
            :param destination: ``CfnNetworkInsightsAnalysis.ExplanationProperty.Destination``.
            :param destination_vpc: ``CfnNetworkInsightsAnalysis.ExplanationProperty.DestinationVpc``.
            :param direction: ``CfnNetworkInsightsAnalysis.ExplanationProperty.Direction``.
            :param elastic_load_balancer_listener: ``CfnNetworkInsightsAnalysis.ExplanationProperty.ElasticLoadBalancerListener``.
            :param explanation_code: ``CfnNetworkInsightsAnalysis.ExplanationProperty.ExplanationCode``.
            :param ingress_route_table: ``CfnNetworkInsightsAnalysis.ExplanationProperty.IngressRouteTable``.
            :param internet_gateway: ``CfnNetworkInsightsAnalysis.ExplanationProperty.InternetGateway``.
            :param load_balancer_arn: ``CfnNetworkInsightsAnalysis.ExplanationProperty.LoadBalancerArn``.
            :param load_balancer_listener_port: ``CfnNetworkInsightsAnalysis.ExplanationProperty.LoadBalancerListenerPort``.
            :param load_balancer_target: ``CfnNetworkInsightsAnalysis.ExplanationProperty.LoadBalancerTarget``.
            :param load_balancer_target_group: ``CfnNetworkInsightsAnalysis.ExplanationProperty.LoadBalancerTargetGroup``.
            :param load_balancer_target_groups: ``CfnNetworkInsightsAnalysis.ExplanationProperty.LoadBalancerTargetGroups``.
            :param load_balancer_target_port: ``CfnNetworkInsightsAnalysis.ExplanationProperty.LoadBalancerTargetPort``.
            :param missing_component: ``CfnNetworkInsightsAnalysis.ExplanationProperty.MissingComponent``.
            :param nat_gateway: ``CfnNetworkInsightsAnalysis.ExplanationProperty.NatGateway``.
            :param network_interface: ``CfnNetworkInsightsAnalysis.ExplanationProperty.NetworkInterface``.
            :param packet_field: ``CfnNetworkInsightsAnalysis.ExplanationProperty.PacketField``.
            :param port: ``CfnNetworkInsightsAnalysis.ExplanationProperty.Port``.
            :param port_ranges: ``CfnNetworkInsightsAnalysis.ExplanationProperty.PortRanges``.
            :param prefix_list: ``CfnNetworkInsightsAnalysis.ExplanationProperty.PrefixList``.
            :param protocols: ``CfnNetworkInsightsAnalysis.ExplanationProperty.Protocols``.
            :param route_table: ``CfnNetworkInsightsAnalysis.ExplanationProperty.RouteTable``.
            :param route_table_route: ``CfnNetworkInsightsAnalysis.ExplanationProperty.RouteTableRoute``.
            :param security_group: ``CfnNetworkInsightsAnalysis.ExplanationProperty.SecurityGroup``.
            :param security_group_rule: ``CfnNetworkInsightsAnalysis.ExplanationProperty.SecurityGroupRule``.
            :param security_groups: ``CfnNetworkInsightsAnalysis.ExplanationProperty.SecurityGroups``.
            :param source_vpc: ``CfnNetworkInsightsAnalysis.ExplanationProperty.SourceVpc``.
            :param state: ``CfnNetworkInsightsAnalysis.ExplanationProperty.State``.
            :param subnet: ``CfnNetworkInsightsAnalysis.ExplanationProperty.Subnet``.
            :param subnet_route_table: ``CfnNetworkInsightsAnalysis.ExplanationProperty.SubnetRouteTable``.
            :param vpc: ``CfnNetworkInsightsAnalysis.ExplanationProperty.Vpc``.
            :param vpc_endpoint: ``CfnNetworkInsightsAnalysis.ExplanationProperty.vpcEndpoint``.
            :param vpc_peering_connection: ``CfnNetworkInsightsAnalysis.ExplanationProperty.VpcPeeringConnection``.
            :param vpn_connection: ``CfnNetworkInsightsAnalysis.ExplanationProperty.VpnConnection``.
            :param vpn_gateway: ``CfnNetworkInsightsAnalysis.ExplanationProperty.VpnGateway``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if acl is not None:
                self._values["acl"] = acl
            if acl_rule is not None:
                self._values["acl_rule"] = acl_rule
            if address is not None:
                self._values["address"] = address
            if addresses is not None:
                self._values["addresses"] = addresses
            if attached_to is not None:
                self._values["attached_to"] = attached_to
            if availability_zones is not None:
                self._values["availability_zones"] = availability_zones
            if cidrs is not None:
                self._values["cidrs"] = cidrs
            if classic_load_balancer_listener is not None:
                self._values["classic_load_balancer_listener"] = classic_load_balancer_listener
            if component is not None:
                self._values["component"] = component
            if customer_gateway is not None:
                self._values["customer_gateway"] = customer_gateway
            if destination is not None:
                self._values["destination"] = destination
            if destination_vpc is not None:
                self._values["destination_vpc"] = destination_vpc
            if direction is not None:
                self._values["direction"] = direction
            if elastic_load_balancer_listener is not None:
                self._values["elastic_load_balancer_listener"] = elastic_load_balancer_listener
            if explanation_code is not None:
                self._values["explanation_code"] = explanation_code
            if ingress_route_table is not None:
                self._values["ingress_route_table"] = ingress_route_table
            if internet_gateway is not None:
                self._values["internet_gateway"] = internet_gateway
            if load_balancer_arn is not None:
                self._values["load_balancer_arn"] = load_balancer_arn
            if load_balancer_listener_port is not None:
                self._values["load_balancer_listener_port"] = load_balancer_listener_port
            if load_balancer_target is not None:
                self._values["load_balancer_target"] = load_balancer_target
            if load_balancer_target_group is not None:
                self._values["load_balancer_target_group"] = load_balancer_target_group
            if load_balancer_target_groups is not None:
                self._values["load_balancer_target_groups"] = load_balancer_target_groups
            if load_balancer_target_port is not None:
                self._values["load_balancer_target_port"] = load_balancer_target_port
            if missing_component is not None:
                self._values["missing_component"] = missing_component
            if nat_gateway is not None:
                self._values["nat_gateway"] = nat_gateway
            if network_interface is not None:
                self._values["network_interface"] = network_interface
            if packet_field is not None:
                self._values["packet_field"] = packet_field
            if port is not None:
                self._values["port"] = port
            if port_ranges is not None:
                self._values["port_ranges"] = port_ranges
            if prefix_list is not None:
                self._values["prefix_list"] = prefix_list
            if protocols is not None:
                self._values["protocols"] = protocols
            if route_table is not None:
                self._values["route_table"] = route_table
            if route_table_route is not None:
                self._values["route_table_route"] = route_table_route
            if security_group is not None:
                self._values["security_group"] = security_group
            if security_group_rule is not None:
                self._values["security_group_rule"] = security_group_rule
            if security_groups is not None:
                self._values["security_groups"] = security_groups
            if source_vpc is not None:
                self._values["source_vpc"] = source_vpc
            if state is not None:
                self._values["state"] = state
            if subnet is not None:
                self._values["subnet"] = subnet
            if subnet_route_table is not None:
                self._values["subnet_route_table"] = subnet_route_table
            if vpc is not None:
                self._values["vpc"] = vpc
            if vpc_endpoint is not None:
                self._values["vpc_endpoint"] = vpc_endpoint
            if vpc_peering_connection is not None:
                self._values["vpc_peering_connection"] = vpc_peering_connection
            if vpn_connection is not None:
                self._values["vpn_connection"] = vpn_connection
            if vpn_gateway is not None:
                self._values["vpn_gateway"] = vpn_gateway

        @builtins.property
        def acl(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.Acl``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-acl
            '''
            result = self._values.get("acl")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def acl_rule(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.AclRule``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-aclrule
            '''
            result = self._values.get("acl_rule")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty"]], result)

        @builtins.property
        def address(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.Address``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-address
            '''
            result = self._values.get("address")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def addresses(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.Addresses``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-addresses
            '''
            result = self._values.get("addresses")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def attached_to(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.AttachedTo``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-attachedto
            '''
            result = self._values.get("attached_to")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def availability_zones(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.AvailabilityZones``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-availabilityzones
            '''
            result = self._values.get("availability_zones")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def cidrs(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.Cidrs``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-cidrs
            '''
            result = self._values.get("cidrs")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def classic_load_balancer_listener(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisLoadBalancerListenerProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.ClassicLoadBalancerListener``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-classicloadbalancerlistener
            '''
            result = self._values.get("classic_load_balancer_listener")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisLoadBalancerListenerProperty"]], result)

        @builtins.property
        def component(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.Component``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-component
            '''
            result = self._values.get("component")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def customer_gateway(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.CustomerGateway``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-customergateway
            '''
            result = self._values.get("customer_gateway")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def destination(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.Destination``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-destination
            '''
            result = self._values.get("destination")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def destination_vpc(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.DestinationVpc``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-destinationvpc
            '''
            result = self._values.get("destination_vpc")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def direction(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.Direction``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-direction
            '''
            result = self._values.get("direction")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def elastic_load_balancer_listener(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.ElasticLoadBalancerListener``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-elasticloadbalancerlistener
            '''
            result = self._values.get("elastic_load_balancer_listener")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def explanation_code(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.ExplanationCode``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-explanationcode
            '''
            result = self._values.get("explanation_code")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def ingress_route_table(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.IngressRouteTable``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-ingressroutetable
            '''
            result = self._values.get("ingress_route_table")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def internet_gateway(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.InternetGateway``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-internetgateway
            '''
            result = self._values.get("internet_gateway")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def load_balancer_arn(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.LoadBalancerArn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-loadbalancerarn
            '''
            result = self._values.get("load_balancer_arn")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def load_balancer_listener_port(self) -> typing.Optional[jsii.Number]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.LoadBalancerListenerPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-loadbalancerlistenerport
            '''
            result = self._values.get("load_balancer_listener_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def load_balancer_target(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisLoadBalancerTargetProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.LoadBalancerTarget``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-loadbalancertarget
            '''
            result = self._values.get("load_balancer_target")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisLoadBalancerTargetProperty"]], result)

        @builtins.property
        def load_balancer_target_group(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.LoadBalancerTargetGroup``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-loadbalancertargetgroup
            '''
            result = self._values.get("load_balancer_target_group")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def load_balancer_target_groups(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.LoadBalancerTargetGroups``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-loadbalancertargetgroups
            '''
            result = self._values.get("load_balancer_target_groups")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]]], result)

        @builtins.property
        def load_balancer_target_port(self) -> typing.Optional[jsii.Number]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.LoadBalancerTargetPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-loadbalancertargetport
            '''
            result = self._values.get("load_balancer_target_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def missing_component(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.MissingComponent``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-missingcomponent
            '''
            result = self._values.get("missing_component")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def nat_gateway(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.NatGateway``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-natgateway
            '''
            result = self._values.get("nat_gateway")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def network_interface(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.NetworkInterface``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-networkinterface
            '''
            result = self._values.get("network_interface")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def packet_field(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.PacketField``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-packetfield
            '''
            result = self._values.get("packet_field")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def port(self) -> typing.Optional[jsii.Number]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.Port``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-port
            '''
            result = self._values.get("port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def port_ranges(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.PortRangeProperty"]]]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.PortRanges``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-portranges
            '''
            result = self._values.get("port_ranges")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.PortRangeProperty"]]]], result)

        @builtins.property
        def prefix_list(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.PrefixList``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-prefixlist
            '''
            result = self._values.get("prefix_list")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def protocols(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.Protocols``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-protocols
            '''
            result = self._values.get("protocols")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def route_table(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.RouteTable``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-routetable
            '''
            result = self._values.get("route_table")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def route_table_route(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.RouteTableRoute``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-routetableroute
            '''
            result = self._values.get("route_table_route")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty"]], result)

        @builtins.property
        def security_group(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.SecurityGroup``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-securitygroup
            '''
            result = self._values.get("security_group")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def security_group_rule(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.SecurityGroupRule``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-securitygrouprule
            '''
            result = self._values.get("security_group_rule")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty"]], result)

        @builtins.property
        def security_groups(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.SecurityGroups``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-securitygroups
            '''
            result = self._values.get("security_groups")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]]], result)

        @builtins.property
        def source_vpc(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.SourceVpc``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-sourcevpc
            '''
            result = self._values.get("source_vpc")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def state(self) -> typing.Optional[builtins.str]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.State``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-state
            '''
            result = self._values.get("state")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def subnet(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.Subnet``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-subnet
            '''
            result = self._values.get("subnet")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def subnet_route_table(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.SubnetRouteTable``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-subnetroutetable
            '''
            result = self._values.get("subnet_route_table")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def vpc(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.Vpc``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-vpc
            '''
            result = self._values.get("vpc")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def vpc_endpoint(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.vpcEndpoint``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-vpcendpoint
            '''
            result = self._values.get("vpc_endpoint")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def vpc_peering_connection(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.VpcPeeringConnection``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-vpcpeeringconnection
            '''
            result = self._values.get("vpc_peering_connection")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def vpn_connection(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.VpnConnection``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-vpnconnection
            '''
            result = self._values.get("vpn_connection")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def vpn_gateway(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.ExplanationProperty.VpnGateway``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-explanation.html#cfn-ec2-networkinsightsanalysis-explanation-vpngateway
            '''
            result = self._values.get("vpn_gateway")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ExplanationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnNetworkInsightsAnalysis.PathComponentProperty",
        jsii_struct_bases=[],
        name_mapping={
            "acl_rule": "aclRule",
            "component": "component",
            "destination_vpc": "destinationVpc",
            "inbound_header": "inboundHeader",
            "outbound_header": "outboundHeader",
            "route_table_route": "routeTableRoute",
            "security_group_rule": "securityGroupRule",
            "sequence_number": "sequenceNumber",
            "source_vpc": "sourceVpc",
            "subnet": "subnet",
            "vpc": "vpc",
        },
    )
    class PathComponentProperty:
        def __init__(
            self,
            *,
            acl_rule: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty"]] = None,
            component: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            destination_vpc: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            inbound_header: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisPacketHeaderProperty"]] = None,
            outbound_header: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisPacketHeaderProperty"]] = None,
            route_table_route: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty"]] = None,
            security_group_rule: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty"]] = None,
            sequence_number: typing.Optional[jsii.Number] = None,
            source_vpc: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            subnet: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
            vpc: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]] = None,
        ) -> None:
            '''
            :param acl_rule: ``CfnNetworkInsightsAnalysis.PathComponentProperty.AclRule``.
            :param component: ``CfnNetworkInsightsAnalysis.PathComponentProperty.Component``.
            :param destination_vpc: ``CfnNetworkInsightsAnalysis.PathComponentProperty.DestinationVpc``.
            :param inbound_header: ``CfnNetworkInsightsAnalysis.PathComponentProperty.InboundHeader``.
            :param outbound_header: ``CfnNetworkInsightsAnalysis.PathComponentProperty.OutboundHeader``.
            :param route_table_route: ``CfnNetworkInsightsAnalysis.PathComponentProperty.RouteTableRoute``.
            :param security_group_rule: ``CfnNetworkInsightsAnalysis.PathComponentProperty.SecurityGroupRule``.
            :param sequence_number: ``CfnNetworkInsightsAnalysis.PathComponentProperty.SequenceNumber``.
            :param source_vpc: ``CfnNetworkInsightsAnalysis.PathComponentProperty.SourceVpc``.
            :param subnet: ``CfnNetworkInsightsAnalysis.PathComponentProperty.Subnet``.
            :param vpc: ``CfnNetworkInsightsAnalysis.PathComponentProperty.Vpc``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-pathcomponent.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if acl_rule is not None:
                self._values["acl_rule"] = acl_rule
            if component is not None:
                self._values["component"] = component
            if destination_vpc is not None:
                self._values["destination_vpc"] = destination_vpc
            if inbound_header is not None:
                self._values["inbound_header"] = inbound_header
            if outbound_header is not None:
                self._values["outbound_header"] = outbound_header
            if route_table_route is not None:
                self._values["route_table_route"] = route_table_route
            if security_group_rule is not None:
                self._values["security_group_rule"] = security_group_rule
            if sequence_number is not None:
                self._values["sequence_number"] = sequence_number
            if source_vpc is not None:
                self._values["source_vpc"] = source_vpc
            if subnet is not None:
                self._values["subnet"] = subnet
            if vpc is not None:
                self._values["vpc"] = vpc

        @builtins.property
        def acl_rule(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty"]]:
            '''``CfnNetworkInsightsAnalysis.PathComponentProperty.AclRule``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-pathcomponent.html#cfn-ec2-networkinsightsanalysis-pathcomponent-aclrule
            '''
            result = self._values.get("acl_rule")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisAclRuleProperty"]], result)

        @builtins.property
        def component(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.PathComponentProperty.Component``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-pathcomponent.html#cfn-ec2-networkinsightsanalysis-pathcomponent-component
            '''
            result = self._values.get("component")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def destination_vpc(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.PathComponentProperty.DestinationVpc``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-pathcomponent.html#cfn-ec2-networkinsightsanalysis-pathcomponent-destinationvpc
            '''
            result = self._values.get("destination_vpc")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def inbound_header(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisPacketHeaderProperty"]]:
            '''``CfnNetworkInsightsAnalysis.PathComponentProperty.InboundHeader``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-pathcomponent.html#cfn-ec2-networkinsightsanalysis-pathcomponent-inboundheader
            '''
            result = self._values.get("inbound_header")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisPacketHeaderProperty"]], result)

        @builtins.property
        def outbound_header(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisPacketHeaderProperty"]]:
            '''``CfnNetworkInsightsAnalysis.PathComponentProperty.OutboundHeader``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-pathcomponent.html#cfn-ec2-networkinsightsanalysis-pathcomponent-outboundheader
            '''
            result = self._values.get("outbound_header")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisPacketHeaderProperty"]], result)

        @builtins.property
        def route_table_route(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty"]]:
            '''``CfnNetworkInsightsAnalysis.PathComponentProperty.RouteTableRoute``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-pathcomponent.html#cfn-ec2-networkinsightsanalysis-pathcomponent-routetableroute
            '''
            result = self._values.get("route_table_route")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisRouteTableRouteProperty"]], result)

        @builtins.property
        def security_group_rule(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty"]]:
            '''``CfnNetworkInsightsAnalysis.PathComponentProperty.SecurityGroupRule``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-pathcomponent.html#cfn-ec2-networkinsightsanalysis-pathcomponent-securitygrouprule
            '''
            result = self._values.get("security_group_rule")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisSecurityGroupRuleProperty"]], result)

        @builtins.property
        def sequence_number(self) -> typing.Optional[jsii.Number]:
            '''``CfnNetworkInsightsAnalysis.PathComponentProperty.SequenceNumber``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-pathcomponent.html#cfn-ec2-networkinsightsanalysis-pathcomponent-sequencenumber
            '''
            result = self._values.get("sequence_number")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def source_vpc(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.PathComponentProperty.SourceVpc``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-pathcomponent.html#cfn-ec2-networkinsightsanalysis-pathcomponent-sourcevpc
            '''
            result = self._values.get("source_vpc")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def subnet(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.PathComponentProperty.Subnet``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-pathcomponent.html#cfn-ec2-networkinsightsanalysis-pathcomponent-subnet
            '''
            result = self._values.get("subnet")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        @builtins.property
        def vpc(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]]:
            '''``CfnNetworkInsightsAnalysis.PathComponentProperty.Vpc``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-pathcomponent.html#cfn-ec2-networkinsightsanalysis-pathcomponent-vpc
            '''
            result = self._values.get("vpc")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInsightsAnalysis.AnalysisComponentProperty"]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PathComponentProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnNetworkInsightsAnalysis.PortRangeProperty",
        jsii_struct_bases=[],
        name_mapping={"from_": "from", "to": "to"},
    )
    class PortRangeProperty:
        def __init__(
            self,
            *,
            from_: typing.Optional[jsii.Number] = None,
            to: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param from_: ``CfnNetworkInsightsAnalysis.PortRangeProperty.From``.
            :param to: ``CfnNetworkInsightsAnalysis.PortRangeProperty.To``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-portrange.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if from_ is not None:
                self._values["from_"] = from_
            if to is not None:
                self._values["to"] = to

        @builtins.property
        def from_(self) -> typing.Optional[jsii.Number]:
            '''``CfnNetworkInsightsAnalysis.PortRangeProperty.From``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-portrange.html#cfn-ec2-networkinsightsanalysis-portrange-from
            '''
            result = self._values.get("from_")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def to(self) -> typing.Optional[jsii.Number]:
            '''``CfnNetworkInsightsAnalysis.PortRangeProperty.To``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinsightsanalysis-portrange.html#cfn-ec2-networkinsightsanalysis-portrange-to
            '''
            result = self._values.get("to")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PortRangeProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnNetworkInsightsAnalysisProps",
    jsii_struct_bases=[],
    name_mapping={
        "network_insights_path_id": "networkInsightsPathId",
        "filter_in_arns": "filterInArns",
        "tags": "tags",
    },
)
class CfnNetworkInsightsAnalysisProps:
    def __init__(
        self,
        *,
        network_insights_path_id: builtins.str,
        filter_in_arns: typing.Optional[typing.Sequence[builtins.str]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::NetworkInsightsAnalysis``.

        :param network_insights_path_id: ``AWS::EC2::NetworkInsightsAnalysis.NetworkInsightsPathId``.
        :param filter_in_arns: ``AWS::EC2::NetworkInsightsAnalysis.FilterInArns``.
        :param tags: ``AWS::EC2::NetworkInsightsAnalysis.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightsanalysis.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "network_insights_path_id": network_insights_path_id,
        }
        if filter_in_arns is not None:
            self._values["filter_in_arns"] = filter_in_arns
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def network_insights_path_id(self) -> builtins.str:
        '''``AWS::EC2::NetworkInsightsAnalysis.NetworkInsightsPathId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightsanalysis.html#cfn-ec2-networkinsightsanalysis-networkinsightspathid
        '''
        result = self._values.get("network_insights_path_id")
        assert result is not None, "Required property 'network_insights_path_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def filter_in_arns(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::NetworkInsightsAnalysis.FilterInArns``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightsanalysis.html#cfn-ec2-networkinsightsanalysis-filterinarns
        '''
        result = self._values.get("filter_in_arns")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''``AWS::EC2::NetworkInsightsAnalysis.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightsanalysis.html#cfn-ec2-networkinsightsanalysis-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnNetworkInsightsAnalysisProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnNetworkInsightsPath(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnNetworkInsightsPath",
):
    '''A CloudFormation ``AWS::EC2::NetworkInsightsPath``.

    :cloudformationResource: AWS::EC2::NetworkInsightsPath
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightspath.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        destination: builtins.str,
        protocol: builtins.str,
        source: builtins.str,
        destination_ip: typing.Optional[builtins.str] = None,
        destination_port: typing.Optional[jsii.Number] = None,
        source_ip: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::NetworkInsightsPath``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param destination: ``AWS::EC2::NetworkInsightsPath.Destination``.
        :param protocol: ``AWS::EC2::NetworkInsightsPath.Protocol``.
        :param source: ``AWS::EC2::NetworkInsightsPath.Source``.
        :param destination_ip: ``AWS::EC2::NetworkInsightsPath.DestinationIp``.
        :param destination_port: ``AWS::EC2::NetworkInsightsPath.DestinationPort``.
        :param source_ip: ``AWS::EC2::NetworkInsightsPath.SourceIp``.
        :param tags: ``AWS::EC2::NetworkInsightsPath.Tags``.
        '''
        props = CfnNetworkInsightsPathProps(
            destination=destination,
            protocol=protocol,
            source=source,
            destination_ip=destination_ip,
            destination_port=destination_port,
            source_ip=source_ip,
            tags=tags,
        )

        jsii.create(CfnNetworkInsightsPath, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrCreatedDate")
    def attr_created_date(self) -> builtins.str:
        '''
        :cloudformationAttribute: CreatedDate
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrCreatedDate"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrNetworkInsightsPathArn")
    def attr_network_insights_path_arn(self) -> builtins.str:
        '''
        :cloudformationAttribute: NetworkInsightsPathArn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrNetworkInsightsPathArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrNetworkInsightsPathId")
    def attr_network_insights_path_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: NetworkInsightsPathId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrNetworkInsightsPathId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''``AWS::EC2::NetworkInsightsPath.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightspath.html#cfn-ec2-networkinsightspath-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="destination")
    def destination(self) -> builtins.str:
        '''``AWS::EC2::NetworkInsightsPath.Destination``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightspath.html#cfn-ec2-networkinsightspath-destination
        '''
        return typing.cast(builtins.str, jsii.get(self, "destination"))

    @destination.setter
    def destination(self, value: builtins.str) -> None:
        jsii.set(self, "destination", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="protocol")
    def protocol(self) -> builtins.str:
        '''``AWS::EC2::NetworkInsightsPath.Protocol``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightspath.html#cfn-ec2-networkinsightspath-protocol
        '''
        return typing.cast(builtins.str, jsii.get(self, "protocol"))

    @protocol.setter
    def protocol(self, value: builtins.str) -> None:
        jsii.set(self, "protocol", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="source")
    def source(self) -> builtins.str:
        '''``AWS::EC2::NetworkInsightsPath.Source``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightspath.html#cfn-ec2-networkinsightspath-source
        '''
        return typing.cast(builtins.str, jsii.get(self, "source"))

    @source.setter
    def source(self, value: builtins.str) -> None:
        jsii.set(self, "source", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="destinationIp")
    def destination_ip(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NetworkInsightsPath.DestinationIp``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightspath.html#cfn-ec2-networkinsightspath-destinationip
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "destinationIp"))

    @destination_ip.setter
    def destination_ip(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "destinationIp", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="destinationPort")
    def destination_port(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::NetworkInsightsPath.DestinationPort``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightspath.html#cfn-ec2-networkinsightspath-destinationport
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "destinationPort"))

    @destination_port.setter
    def destination_port(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "destinationPort", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceIp")
    def source_ip(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NetworkInsightsPath.SourceIp``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightspath.html#cfn-ec2-networkinsightspath-sourceip
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceIp"))

    @source_ip.setter
    def source_ip(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "sourceIp", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnNetworkInsightsPathProps",
    jsii_struct_bases=[],
    name_mapping={
        "destination": "destination",
        "protocol": "protocol",
        "source": "source",
        "destination_ip": "destinationIp",
        "destination_port": "destinationPort",
        "source_ip": "sourceIp",
        "tags": "tags",
    },
)
class CfnNetworkInsightsPathProps:
    def __init__(
        self,
        *,
        destination: builtins.str,
        protocol: builtins.str,
        source: builtins.str,
        destination_ip: typing.Optional[builtins.str] = None,
        destination_port: typing.Optional[jsii.Number] = None,
        source_ip: typing.Optional[builtins.str] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::NetworkInsightsPath``.

        :param destination: ``AWS::EC2::NetworkInsightsPath.Destination``.
        :param protocol: ``AWS::EC2::NetworkInsightsPath.Protocol``.
        :param source: ``AWS::EC2::NetworkInsightsPath.Source``.
        :param destination_ip: ``AWS::EC2::NetworkInsightsPath.DestinationIp``.
        :param destination_port: ``AWS::EC2::NetworkInsightsPath.DestinationPort``.
        :param source_ip: ``AWS::EC2::NetworkInsightsPath.SourceIp``.
        :param tags: ``AWS::EC2::NetworkInsightsPath.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightspath.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "destination": destination,
            "protocol": protocol,
            "source": source,
        }
        if destination_ip is not None:
            self._values["destination_ip"] = destination_ip
        if destination_port is not None:
            self._values["destination_port"] = destination_port
        if source_ip is not None:
            self._values["source_ip"] = source_ip
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def destination(self) -> builtins.str:
        '''``AWS::EC2::NetworkInsightsPath.Destination``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightspath.html#cfn-ec2-networkinsightspath-destination
        '''
        result = self._values.get("destination")
        assert result is not None, "Required property 'destination' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def protocol(self) -> builtins.str:
        '''``AWS::EC2::NetworkInsightsPath.Protocol``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightspath.html#cfn-ec2-networkinsightspath-protocol
        '''
        result = self._values.get("protocol")
        assert result is not None, "Required property 'protocol' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def source(self) -> builtins.str:
        '''``AWS::EC2::NetworkInsightsPath.Source``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightspath.html#cfn-ec2-networkinsightspath-source
        '''
        result = self._values.get("source")
        assert result is not None, "Required property 'source' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def destination_ip(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NetworkInsightsPath.DestinationIp``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightspath.html#cfn-ec2-networkinsightspath-destinationip
        '''
        result = self._values.get("destination_ip")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def destination_port(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::NetworkInsightsPath.DestinationPort``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightspath.html#cfn-ec2-networkinsightspath-destinationport
        '''
        result = self._values.get("destination_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def source_ip(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NetworkInsightsPath.SourceIp``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightspath.html#cfn-ec2-networkinsightspath-sourceip
        '''
        result = self._values.get("source_ip")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''``AWS::EC2::NetworkInsightsPath.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinsightspath.html#cfn-ec2-networkinsightspath-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnNetworkInsightsPathProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnNetworkInterface(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnNetworkInterface",
):
    '''A CloudFormation ``AWS::EC2::NetworkInterface``.

    :cloudformationResource: AWS::EC2::NetworkInterface
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        subnet_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
        group_set: typing.Optional[typing.Sequence[builtins.str]] = None,
        interface_type: typing.Optional[builtins.str] = None,
        ipv6_address_count: typing.Optional[jsii.Number] = None,
        ipv6_addresses: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInterface.InstanceIpv6AddressProperty"]]]] = None,
        private_ip_address: typing.Optional[builtins.str] = None,
        private_ip_addresses: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInterface.PrivateIpAddressSpecificationProperty"]]]] = None,
        secondary_private_ip_address_count: typing.Optional[jsii.Number] = None,
        source_dest_check: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::NetworkInterface``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param subnet_id: ``AWS::EC2::NetworkInterface.SubnetId``.
        :param description: ``AWS::EC2::NetworkInterface.Description``.
        :param group_set: ``AWS::EC2::NetworkInterface.GroupSet``.
        :param interface_type: ``AWS::EC2::NetworkInterface.InterfaceType``.
        :param ipv6_address_count: ``AWS::EC2::NetworkInterface.Ipv6AddressCount``.
        :param ipv6_addresses: ``AWS::EC2::NetworkInterface.Ipv6Addresses``.
        :param private_ip_address: ``AWS::EC2::NetworkInterface.PrivateIpAddress``.
        :param private_ip_addresses: ``AWS::EC2::NetworkInterface.PrivateIpAddresses``.
        :param secondary_private_ip_address_count: ``AWS::EC2::NetworkInterface.SecondaryPrivateIpAddressCount``.
        :param source_dest_check: ``AWS::EC2::NetworkInterface.SourceDestCheck``.
        :param tags: ``AWS::EC2::NetworkInterface.Tags``.
        '''
        props = CfnNetworkInterfaceProps(
            subnet_id=subnet_id,
            description=description,
            group_set=group_set,
            interface_type=interface_type,
            ipv6_address_count=ipv6_address_count,
            ipv6_addresses=ipv6_addresses,
            private_ip_address=private_ip_address,
            private_ip_addresses=private_ip_addresses,
            secondary_private_ip_address_count=secondary_private_ip_address_count,
            source_dest_check=source_dest_check,
            tags=tags,
        )

        jsii.create(CfnNetworkInterface, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrPrimaryPrivateIpAddress")
    def attr_primary_private_ip_address(self) -> builtins.str:
        '''
        :cloudformationAttribute: PrimaryPrivateIpAddress
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrPrimaryPrivateIpAddress"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrSecondaryPrivateIpAddresses")
    def attr_secondary_private_ip_addresses(self) -> typing.List[builtins.str]:
        '''
        :cloudformationAttribute: SecondaryPrivateIpAddresses
        '''
        return typing.cast(typing.List[builtins.str], jsii.get(self, "attrSecondaryPrivateIpAddresses"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''``AWS::EC2::NetworkInterface.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-awsec2networkinterface-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="subnetId")
    def subnet_id(self) -> builtins.str:
        '''``AWS::EC2::NetworkInterface.SubnetId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-awsec2networkinterface-subnetid
        '''
        return typing.cast(builtins.str, jsii.get(self, "subnetId"))

    @subnet_id.setter
    def subnet_id(self, value: builtins.str) -> None:
        jsii.set(self, "subnetId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NetworkInterface.Description``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-awsec2networkinterface-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "description", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="groupSet")
    def group_set(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::NetworkInterface.GroupSet``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-awsec2networkinterface-groupset
        '''
        return typing.cast(typing.Optional[typing.List[builtins.str]], jsii.get(self, "groupSet"))

    @group_set.setter
    def group_set(self, value: typing.Optional[typing.List[builtins.str]]) -> None:
        jsii.set(self, "groupSet", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="interfaceType")
    def interface_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NetworkInterface.InterfaceType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-ec2-networkinterface-interfacetype
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "interfaceType"))

    @interface_type.setter
    def interface_type(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "interfaceType", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ipv6AddressCount")
    def ipv6_address_count(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::NetworkInterface.Ipv6AddressCount``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-ec2-networkinterface-ipv6addresscount
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "ipv6AddressCount"))

    @ipv6_address_count.setter
    def ipv6_address_count(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "ipv6AddressCount", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ipv6Addresses")
    def ipv6_addresses(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInterface.InstanceIpv6AddressProperty"]]]]:
        '''``AWS::EC2::NetworkInterface.Ipv6Addresses``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-ec2-networkinterface-ipv6addresses
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInterface.InstanceIpv6AddressProperty"]]]], jsii.get(self, "ipv6Addresses"))

    @ipv6_addresses.setter
    def ipv6_addresses(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInterface.InstanceIpv6AddressProperty"]]]],
    ) -> None:
        jsii.set(self, "ipv6Addresses", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="privateIpAddress")
    def private_ip_address(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NetworkInterface.PrivateIpAddress``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-awsec2networkinterface-privateipaddress
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "privateIpAddress"))

    @private_ip_address.setter
    def private_ip_address(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "privateIpAddress", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="privateIpAddresses")
    def private_ip_addresses(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInterface.PrivateIpAddressSpecificationProperty"]]]]:
        '''``AWS::EC2::NetworkInterface.PrivateIpAddresses``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-awsec2networkinterface-privateipaddresses
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInterface.PrivateIpAddressSpecificationProperty"]]]], jsii.get(self, "privateIpAddresses"))

    @private_ip_addresses.setter
    def private_ip_addresses(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnNetworkInterface.PrivateIpAddressSpecificationProperty"]]]],
    ) -> None:
        jsii.set(self, "privateIpAddresses", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="secondaryPrivateIpAddressCount")
    def secondary_private_ip_address_count(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::NetworkInterface.SecondaryPrivateIpAddressCount``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-awsec2networkinterface-secondaryprivateipcount
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "secondaryPrivateIpAddressCount"))

    @secondary_private_ip_address_count.setter
    def secondary_private_ip_address_count(
        self,
        value: typing.Optional[jsii.Number],
    ) -> None:
        jsii.set(self, "secondaryPrivateIpAddressCount", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceDestCheck")
    def source_dest_check(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::NetworkInterface.SourceDestCheck``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-awsec2networkinterface-sourcedestcheck
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "sourceDestCheck"))

    @source_dest_check.setter
    def source_dest_check(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "sourceDestCheck", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnNetworkInterface.InstanceIpv6AddressProperty",
        jsii_struct_bases=[],
        name_mapping={"ipv6_address": "ipv6Address"},
    )
    class InstanceIpv6AddressProperty:
        def __init__(self, *, ipv6_address: builtins.str) -> None:
            '''
            :param ipv6_address: ``CfnNetworkInterface.InstanceIpv6AddressProperty.Ipv6Address``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinterface-instanceipv6address.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "ipv6_address": ipv6_address,
            }

        @builtins.property
        def ipv6_address(self) -> builtins.str:
            '''``CfnNetworkInterface.InstanceIpv6AddressProperty.Ipv6Address``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-networkinterface-instanceipv6address.html#cfn-ec2-networkinterface-instanceipv6address-ipv6address
            '''
            result = self._values.get("ipv6_address")
            assert result is not None, "Required property 'ipv6_address' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceIpv6AddressProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnNetworkInterface.PrivateIpAddressSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"primary": "primary", "private_ip_address": "privateIpAddress"},
    )
    class PrivateIpAddressSpecificationProperty:
        def __init__(
            self,
            *,
            primary: typing.Union[builtins.bool, aws_cdk.core.IResolvable],
            private_ip_address: builtins.str,
        ) -> None:
            '''
            :param primary: ``CfnNetworkInterface.PrivateIpAddressSpecificationProperty.Primary``.
            :param private_ip_address: ``CfnNetworkInterface.PrivateIpAddressSpecificationProperty.PrivateIpAddress``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-interface-privateipspec.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "primary": primary,
                "private_ip_address": private_ip_address,
            }

        @builtins.property
        def primary(self) -> typing.Union[builtins.bool, aws_cdk.core.IResolvable]:
            '''``CfnNetworkInterface.PrivateIpAddressSpecificationProperty.Primary``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-interface-privateipspec.html#cfn-ec2-networkinterface-privateipspecification-primary
            '''
            result = self._values.get("primary")
            assert result is not None, "Required property 'primary' is missing"
            return typing.cast(typing.Union[builtins.bool, aws_cdk.core.IResolvable], result)

        @builtins.property
        def private_ip_address(self) -> builtins.str:
            '''``CfnNetworkInterface.PrivateIpAddressSpecificationProperty.PrivateIpAddress``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-network-interface-privateipspec.html#cfn-ec2-networkinterface-privateipspecification-privateipaddress
            '''
            result = self._values.get("private_ip_address")
            assert result is not None, "Required property 'private_ip_address' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "PrivateIpAddressSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnNetworkInterfaceAttachment(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnNetworkInterfaceAttachment",
):
    '''A CloudFormation ``AWS::EC2::NetworkInterfaceAttachment``.

    :cloudformationResource: AWS::EC2::NetworkInterfaceAttachment
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface-attachment.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        device_index: builtins.str,
        instance_id: builtins.str,
        network_interface_id: builtins.str,
        delete_on_termination: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::NetworkInterfaceAttachment``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param device_index: ``AWS::EC2::NetworkInterfaceAttachment.DeviceIndex``.
        :param instance_id: ``AWS::EC2::NetworkInterfaceAttachment.InstanceId``.
        :param network_interface_id: ``AWS::EC2::NetworkInterfaceAttachment.NetworkInterfaceId``.
        :param delete_on_termination: ``AWS::EC2::NetworkInterfaceAttachment.DeleteOnTermination``.
        '''
        props = CfnNetworkInterfaceAttachmentProps(
            device_index=device_index,
            instance_id=instance_id,
            network_interface_id=network_interface_id,
            delete_on_termination=delete_on_termination,
        )

        jsii.create(CfnNetworkInterfaceAttachment, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="deviceIndex")
    def device_index(self) -> builtins.str:
        '''``AWS::EC2::NetworkInterfaceAttachment.DeviceIndex``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface-attachment.html#cfn-ec2-network-interface-attachment-deviceindex
        '''
        return typing.cast(builtins.str, jsii.get(self, "deviceIndex"))

    @device_index.setter
    def device_index(self, value: builtins.str) -> None:
        jsii.set(self, "deviceIndex", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceId")
    def instance_id(self) -> builtins.str:
        '''``AWS::EC2::NetworkInterfaceAttachment.InstanceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface-attachment.html#cfn-ec2-network-interface-attachment-instanceid
        '''
        return typing.cast(builtins.str, jsii.get(self, "instanceId"))

    @instance_id.setter
    def instance_id(self, value: builtins.str) -> None:
        jsii.set(self, "instanceId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkInterfaceId")
    def network_interface_id(self) -> builtins.str:
        '''``AWS::EC2::NetworkInterfaceAttachment.NetworkInterfaceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface-attachment.html#cfn-ec2-network-interface-attachment-networkinterfaceid
        '''
        return typing.cast(builtins.str, jsii.get(self, "networkInterfaceId"))

    @network_interface_id.setter
    def network_interface_id(self, value: builtins.str) -> None:
        jsii.set(self, "networkInterfaceId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="deleteOnTermination")
    def delete_on_termination(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::NetworkInterfaceAttachment.DeleteOnTermination``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface-attachment.html#cfn-ec2-network-interface-attachment-deleteonterm
        '''
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], jsii.get(self, "deleteOnTermination"))

    @delete_on_termination.setter
    def delete_on_termination(
        self,
        value: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]],
    ) -> None:
        jsii.set(self, "deleteOnTermination", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnNetworkInterfaceAttachmentProps",
    jsii_struct_bases=[],
    name_mapping={
        "device_index": "deviceIndex",
        "instance_id": "instanceId",
        "network_interface_id": "networkInterfaceId",
        "delete_on_termination": "deleteOnTermination",
    },
)
class CfnNetworkInterfaceAttachmentProps:
    def __init__(
        self,
        *,
        device_index: builtins.str,
        instance_id: builtins.str,
        network_interface_id: builtins.str,
        delete_on_termination: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::NetworkInterfaceAttachment``.

        :param device_index: ``AWS::EC2::NetworkInterfaceAttachment.DeviceIndex``.
        :param instance_id: ``AWS::EC2::NetworkInterfaceAttachment.InstanceId``.
        :param network_interface_id: ``AWS::EC2::NetworkInterfaceAttachment.NetworkInterfaceId``.
        :param delete_on_termination: ``AWS::EC2::NetworkInterfaceAttachment.DeleteOnTermination``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface-attachment.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "device_index": device_index,
            "instance_id": instance_id,
            "network_interface_id": network_interface_id,
        }
        if delete_on_termination is not None:
            self._values["delete_on_termination"] = delete_on_termination

    @builtins.property
    def device_index(self) -> builtins.str:
        '''``AWS::EC2::NetworkInterfaceAttachment.DeviceIndex``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface-attachment.html#cfn-ec2-network-interface-attachment-deviceindex
        '''
        result = self._values.get("device_index")
        assert result is not None, "Required property 'device_index' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def instance_id(self) -> builtins.str:
        '''``AWS::EC2::NetworkInterfaceAttachment.InstanceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface-attachment.html#cfn-ec2-network-interface-attachment-instanceid
        '''
        result = self._values.get("instance_id")
        assert result is not None, "Required property 'instance_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def network_interface_id(self) -> builtins.str:
        '''``AWS::EC2::NetworkInterfaceAttachment.NetworkInterfaceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface-attachment.html#cfn-ec2-network-interface-attachment-networkinterfaceid
        '''
        result = self._values.get("network_interface_id")
        assert result is not None, "Required property 'network_interface_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def delete_on_termination(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::NetworkInterfaceAttachment.DeleteOnTermination``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface-attachment.html#cfn-ec2-network-interface-attachment-deleteonterm
        '''
        result = self._values.get("delete_on_termination")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnNetworkInterfaceAttachmentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnNetworkInterfacePermission(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnNetworkInterfacePermission",
):
    '''A CloudFormation ``AWS::EC2::NetworkInterfacePermission``.

    :cloudformationResource: AWS::EC2::NetworkInterfacePermission
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinterfacepermission.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        aws_account_id: builtins.str,
        network_interface_id: builtins.str,
        permission: builtins.str,
    ) -> None:
        '''Create a new ``AWS::EC2::NetworkInterfacePermission``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param aws_account_id: ``AWS::EC2::NetworkInterfacePermission.AwsAccountId``.
        :param network_interface_id: ``AWS::EC2::NetworkInterfacePermission.NetworkInterfaceId``.
        :param permission: ``AWS::EC2::NetworkInterfacePermission.Permission``.
        '''
        props = CfnNetworkInterfacePermissionProps(
            aws_account_id=aws_account_id,
            network_interface_id=network_interface_id,
            permission=permission,
        )

        jsii.create(CfnNetworkInterfacePermission, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="awsAccountId")
    def aws_account_id(self) -> builtins.str:
        '''``AWS::EC2::NetworkInterfacePermission.AwsAccountId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinterfacepermission.html#cfn-ec2-networkinterfacepermission-awsaccountid
        '''
        return typing.cast(builtins.str, jsii.get(self, "awsAccountId"))

    @aws_account_id.setter
    def aws_account_id(self, value: builtins.str) -> None:
        jsii.set(self, "awsAccountId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkInterfaceId")
    def network_interface_id(self) -> builtins.str:
        '''``AWS::EC2::NetworkInterfacePermission.NetworkInterfaceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinterfacepermission.html#cfn-ec2-networkinterfacepermission-networkinterfaceid
        '''
        return typing.cast(builtins.str, jsii.get(self, "networkInterfaceId"))

    @network_interface_id.setter
    def network_interface_id(self, value: builtins.str) -> None:
        jsii.set(self, "networkInterfaceId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="permission")
    def permission(self) -> builtins.str:
        '''``AWS::EC2::NetworkInterfacePermission.Permission``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinterfacepermission.html#cfn-ec2-networkinterfacepermission-permission
        '''
        return typing.cast(builtins.str, jsii.get(self, "permission"))

    @permission.setter
    def permission(self, value: builtins.str) -> None:
        jsii.set(self, "permission", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnNetworkInterfacePermissionProps",
    jsii_struct_bases=[],
    name_mapping={
        "aws_account_id": "awsAccountId",
        "network_interface_id": "networkInterfaceId",
        "permission": "permission",
    },
)
class CfnNetworkInterfacePermissionProps:
    def __init__(
        self,
        *,
        aws_account_id: builtins.str,
        network_interface_id: builtins.str,
        permission: builtins.str,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::NetworkInterfacePermission``.

        :param aws_account_id: ``AWS::EC2::NetworkInterfacePermission.AwsAccountId``.
        :param network_interface_id: ``AWS::EC2::NetworkInterfacePermission.NetworkInterfaceId``.
        :param permission: ``AWS::EC2::NetworkInterfacePermission.Permission``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinterfacepermission.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "aws_account_id": aws_account_id,
            "network_interface_id": network_interface_id,
            "permission": permission,
        }

    @builtins.property
    def aws_account_id(self) -> builtins.str:
        '''``AWS::EC2::NetworkInterfacePermission.AwsAccountId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinterfacepermission.html#cfn-ec2-networkinterfacepermission-awsaccountid
        '''
        result = self._values.get("aws_account_id")
        assert result is not None, "Required property 'aws_account_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def network_interface_id(self) -> builtins.str:
        '''``AWS::EC2::NetworkInterfacePermission.NetworkInterfaceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinterfacepermission.html#cfn-ec2-networkinterfacepermission-networkinterfaceid
        '''
        result = self._values.get("network_interface_id")
        assert result is not None, "Required property 'network_interface_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def permission(self) -> builtins.str:
        '''``AWS::EC2::NetworkInterfacePermission.Permission``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-networkinterfacepermission.html#cfn-ec2-networkinterfacepermission-permission
        '''
        result = self._values.get("permission")
        assert result is not None, "Required property 'permission' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnNetworkInterfacePermissionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnNetworkInterfaceProps",
    jsii_struct_bases=[],
    name_mapping={
        "subnet_id": "subnetId",
        "description": "description",
        "group_set": "groupSet",
        "interface_type": "interfaceType",
        "ipv6_address_count": "ipv6AddressCount",
        "ipv6_addresses": "ipv6Addresses",
        "private_ip_address": "privateIpAddress",
        "private_ip_addresses": "privateIpAddresses",
        "secondary_private_ip_address_count": "secondaryPrivateIpAddressCount",
        "source_dest_check": "sourceDestCheck",
        "tags": "tags",
    },
)
class CfnNetworkInterfaceProps:
    def __init__(
        self,
        *,
        subnet_id: builtins.str,
        description: typing.Optional[builtins.str] = None,
        group_set: typing.Optional[typing.Sequence[builtins.str]] = None,
        interface_type: typing.Optional[builtins.str] = None,
        ipv6_address_count: typing.Optional[jsii.Number] = None,
        ipv6_addresses: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnNetworkInterface.InstanceIpv6AddressProperty]]]] = None,
        private_ip_address: typing.Optional[builtins.str] = None,
        private_ip_addresses: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnNetworkInterface.PrivateIpAddressSpecificationProperty]]]] = None,
        secondary_private_ip_address_count: typing.Optional[jsii.Number] = None,
        source_dest_check: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::NetworkInterface``.

        :param subnet_id: ``AWS::EC2::NetworkInterface.SubnetId``.
        :param description: ``AWS::EC2::NetworkInterface.Description``.
        :param group_set: ``AWS::EC2::NetworkInterface.GroupSet``.
        :param interface_type: ``AWS::EC2::NetworkInterface.InterfaceType``.
        :param ipv6_address_count: ``AWS::EC2::NetworkInterface.Ipv6AddressCount``.
        :param ipv6_addresses: ``AWS::EC2::NetworkInterface.Ipv6Addresses``.
        :param private_ip_address: ``AWS::EC2::NetworkInterface.PrivateIpAddress``.
        :param private_ip_addresses: ``AWS::EC2::NetworkInterface.PrivateIpAddresses``.
        :param secondary_private_ip_address_count: ``AWS::EC2::NetworkInterface.SecondaryPrivateIpAddressCount``.
        :param source_dest_check: ``AWS::EC2::NetworkInterface.SourceDestCheck``.
        :param tags: ``AWS::EC2::NetworkInterface.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "subnet_id": subnet_id,
        }
        if description is not None:
            self._values["description"] = description
        if group_set is not None:
            self._values["group_set"] = group_set
        if interface_type is not None:
            self._values["interface_type"] = interface_type
        if ipv6_address_count is not None:
            self._values["ipv6_address_count"] = ipv6_address_count
        if ipv6_addresses is not None:
            self._values["ipv6_addresses"] = ipv6_addresses
        if private_ip_address is not None:
            self._values["private_ip_address"] = private_ip_address
        if private_ip_addresses is not None:
            self._values["private_ip_addresses"] = private_ip_addresses
        if secondary_private_ip_address_count is not None:
            self._values["secondary_private_ip_address_count"] = secondary_private_ip_address_count
        if source_dest_check is not None:
            self._values["source_dest_check"] = source_dest_check
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def subnet_id(self) -> builtins.str:
        '''``AWS::EC2::NetworkInterface.SubnetId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-awsec2networkinterface-subnetid
        '''
        result = self._values.get("subnet_id")
        assert result is not None, "Required property 'subnet_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NetworkInterface.Description``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-awsec2networkinterface-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def group_set(self) -> typing.Optional[typing.List[builtins.str]]:
        '''``AWS::EC2::NetworkInterface.GroupSet``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-awsec2networkinterface-groupset
        '''
        result = self._values.get("group_set")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def interface_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NetworkInterface.InterfaceType``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-ec2-networkinterface-interfacetype
        '''
        result = self._values.get("interface_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def ipv6_address_count(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::NetworkInterface.Ipv6AddressCount``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-ec2-networkinterface-ipv6addresscount
        '''
        result = self._values.get("ipv6_address_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def ipv6_addresses(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnNetworkInterface.InstanceIpv6AddressProperty]]]]:
        '''``AWS::EC2::NetworkInterface.Ipv6Addresses``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-ec2-networkinterface-ipv6addresses
        '''
        result = self._values.get("ipv6_addresses")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnNetworkInterface.InstanceIpv6AddressProperty]]]], result)

    @builtins.property
    def private_ip_address(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::NetworkInterface.PrivateIpAddress``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-awsec2networkinterface-privateipaddress
        '''
        result = self._values.get("private_ip_address")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def private_ip_addresses(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnNetworkInterface.PrivateIpAddressSpecificationProperty]]]]:
        '''``AWS::EC2::NetworkInterface.PrivateIpAddresses``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-awsec2networkinterface-privateipaddresses
        '''
        result = self._values.get("private_ip_addresses")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnNetworkInterface.PrivateIpAddressSpecificationProperty]]]], result)

    @builtins.property
    def secondary_private_ip_address_count(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::NetworkInterface.SecondaryPrivateIpAddressCount``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-awsec2networkinterface-secondaryprivateipcount
        '''
        result = self._values.get("secondary_private_ip_address_count")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def source_dest_check(
        self,
    ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
        '''``AWS::EC2::NetworkInterface.SourceDestCheck``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-awsec2networkinterface-sourcedestcheck
        '''
        result = self._values.get("source_dest_check")
        return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''``AWS::EC2::NetworkInterface.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-network-interface.html#cfn-awsec2networkinterface-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnNetworkInterfaceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnPlacementGroup(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnPlacementGroup",
):
    '''A CloudFormation ``AWS::EC2::PlacementGroup``.

    :cloudformationResource: AWS::EC2::PlacementGroup
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-placementgroup.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        strategy: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::PlacementGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param strategy: ``AWS::EC2::PlacementGroup.Strategy``.
        '''
        props = CfnPlacementGroupProps(strategy=strategy)

        jsii.create(CfnPlacementGroup, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="strategy")
    def strategy(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::PlacementGroup.Strategy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-placementgroup.html#cfn-ec2-placementgroup-strategy
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "strategy"))

    @strategy.setter
    def strategy(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "strategy", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnPlacementGroupProps",
    jsii_struct_bases=[],
    name_mapping={"strategy": "strategy"},
)
class CfnPlacementGroupProps:
    def __init__(self, *, strategy: typing.Optional[builtins.str] = None) -> None:
        '''Properties for defining a ``AWS::EC2::PlacementGroup``.

        :param strategy: ``AWS::EC2::PlacementGroup.Strategy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-placementgroup.html
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if strategy is not None:
            self._values["strategy"] = strategy

    @builtins.property
    def strategy(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::PlacementGroup.Strategy``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-placementgroup.html#cfn-ec2-placementgroup-strategy
        '''
        result = self._values.get("strategy")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnPlacementGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnPrefixList(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnPrefixList",
):
    '''A CloudFormation ``AWS::EC2::PrefixList``.

    :cloudformationResource: AWS::EC2::PrefixList
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-prefixlist.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        address_family: builtins.str,
        max_entries: jsii.Number,
        prefix_list_name: builtins.str,
        entries: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnPrefixList.EntryProperty"]]]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::PrefixList``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param address_family: ``AWS::EC2::PrefixList.AddressFamily``.
        :param max_entries: ``AWS::EC2::PrefixList.MaxEntries``.
        :param prefix_list_name: ``AWS::EC2::PrefixList.PrefixListName``.
        :param entries: ``AWS::EC2::PrefixList.Entries``.
        :param tags: ``AWS::EC2::PrefixList.Tags``.
        '''
        props = CfnPrefixListProps(
            address_family=address_family,
            max_entries=max_entries,
            prefix_list_name=prefix_list_name,
            entries=entries,
            tags=tags,
        )

        jsii.create(CfnPrefixList, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrArn")
    def attr_arn(self) -> builtins.str:
        '''
        :cloudformationAttribute: Arn
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrOwnerId")
    def attr_owner_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: OwnerId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrOwnerId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrPrefixListId")
    def attr_prefix_list_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: PrefixListId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrPrefixListId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrVersion")
    def attr_version(self) -> jsii.Number:
        '''
        :cloudformationAttribute: Version
        '''
        return typing.cast(jsii.Number, jsii.get(self, "attrVersion"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''``AWS::EC2::PrefixList.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-prefixlist.html#cfn-ec2-prefixlist-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="addressFamily")
    def address_family(self) -> builtins.str:
        '''``AWS::EC2::PrefixList.AddressFamily``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-prefixlist.html#cfn-ec2-prefixlist-addressfamily
        '''
        return typing.cast(builtins.str, jsii.get(self, "addressFamily"))

    @address_family.setter
    def address_family(self, value: builtins.str) -> None:
        jsii.set(self, "addressFamily", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="maxEntries")
    def max_entries(self) -> jsii.Number:
        '''``AWS::EC2::PrefixList.MaxEntries``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-prefixlist.html#cfn-ec2-prefixlist-maxentries
        '''
        return typing.cast(jsii.Number, jsii.get(self, "maxEntries"))

    @max_entries.setter
    def max_entries(self, value: jsii.Number) -> None:
        jsii.set(self, "maxEntries", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="prefixListName")
    def prefix_list_name(self) -> builtins.str:
        '''``AWS::EC2::PrefixList.PrefixListName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-prefixlist.html#cfn-ec2-prefixlist-prefixlistname
        '''
        return typing.cast(builtins.str, jsii.get(self, "prefixListName"))

    @prefix_list_name.setter
    def prefix_list_name(self, value: builtins.str) -> None:
        jsii.set(self, "prefixListName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="entries")
    def entries(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnPrefixList.EntryProperty"]]]]:
        '''``AWS::EC2::PrefixList.Entries``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-prefixlist.html#cfn-ec2-prefixlist-entries
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnPrefixList.EntryProperty"]]]], jsii.get(self, "entries"))

    @entries.setter
    def entries(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnPrefixList.EntryProperty"]]]],
    ) -> None:
        jsii.set(self, "entries", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnPrefixList.EntryProperty",
        jsii_struct_bases=[],
        name_mapping={"cidr": "cidr", "description": "description"},
    )
    class EntryProperty:
        def __init__(
            self,
            *,
            cidr: builtins.str,
            description: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param cidr: ``CfnPrefixList.EntryProperty.Cidr``.
            :param description: ``CfnPrefixList.EntryProperty.Description``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-prefixlist-entry.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "cidr": cidr,
            }
            if description is not None:
                self._values["description"] = description

        @builtins.property
        def cidr(self) -> builtins.str:
            '''``CfnPrefixList.EntryProperty.Cidr``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-prefixlist-entry.html#cfn-ec2-prefixlist-entry-cidr
            '''
            result = self._values.get("cidr")
            assert result is not None, "Required property 'cidr' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            '''``CfnPrefixList.EntryProperty.Description``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-prefixlist-entry.html#cfn-ec2-prefixlist-entry-description
            '''
            result = self._values.get("description")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EntryProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnPrefixListProps",
    jsii_struct_bases=[],
    name_mapping={
        "address_family": "addressFamily",
        "max_entries": "maxEntries",
        "prefix_list_name": "prefixListName",
        "entries": "entries",
        "tags": "tags",
    },
)
class CfnPrefixListProps:
    def __init__(
        self,
        *,
        address_family: builtins.str,
        max_entries: jsii.Number,
        prefix_list_name: builtins.str,
        entries: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnPrefixList.EntryProperty]]]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::PrefixList``.

        :param address_family: ``AWS::EC2::PrefixList.AddressFamily``.
        :param max_entries: ``AWS::EC2::PrefixList.MaxEntries``.
        :param prefix_list_name: ``AWS::EC2::PrefixList.PrefixListName``.
        :param entries: ``AWS::EC2::PrefixList.Entries``.
        :param tags: ``AWS::EC2::PrefixList.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-prefixlist.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "address_family": address_family,
            "max_entries": max_entries,
            "prefix_list_name": prefix_list_name,
        }
        if entries is not None:
            self._values["entries"] = entries
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def address_family(self) -> builtins.str:
        '''``AWS::EC2::PrefixList.AddressFamily``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-prefixlist.html#cfn-ec2-prefixlist-addressfamily
        '''
        result = self._values.get("address_family")
        assert result is not None, "Required property 'address_family' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def max_entries(self) -> jsii.Number:
        '''``AWS::EC2::PrefixList.MaxEntries``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-prefixlist.html#cfn-ec2-prefixlist-maxentries
        '''
        result = self._values.get("max_entries")
        assert result is not None, "Required property 'max_entries' is missing"
        return typing.cast(jsii.Number, result)

    @builtins.property
    def prefix_list_name(self) -> builtins.str:
        '''``AWS::EC2::PrefixList.PrefixListName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-prefixlist.html#cfn-ec2-prefixlist-prefixlistname
        '''
        result = self._values.get("prefix_list_name")
        assert result is not None, "Required property 'prefix_list_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def entries(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnPrefixList.EntryProperty]]]]:
        '''``AWS::EC2::PrefixList.Entries``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-prefixlist.html#cfn-ec2-prefixlist-entries
        '''
        result = self._values.get("entries")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnPrefixList.EntryProperty]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''``AWS::EC2::PrefixList.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-prefixlist.html#cfn-ec2-prefixlist-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnPrefixListProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnRoute(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnRoute",
):
    '''A CloudFormation ``AWS::EC2::Route``.

    :cloudformationResource: AWS::EC2::Route
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        route_table_id: builtins.str,
        carrier_gateway_id: typing.Optional[builtins.str] = None,
        destination_cidr_block: typing.Optional[builtins.str] = None,
        destination_ipv6_cidr_block: typing.Optional[builtins.str] = None,
        egress_only_internet_gateway_id: typing.Optional[builtins.str] = None,
        gateway_id: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        local_gateway_id: typing.Optional[builtins.str] = None,
        nat_gateway_id: typing.Optional[builtins.str] = None,
        network_interface_id: typing.Optional[builtins.str] = None,
        transit_gateway_id: typing.Optional[builtins.str] = None,
        vpc_endpoint_id: typing.Optional[builtins.str] = None,
        vpc_peering_connection_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::Route``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param route_table_id: ``AWS::EC2::Route.RouteTableId``.
        :param carrier_gateway_id: ``AWS::EC2::Route.CarrierGatewayId``.
        :param destination_cidr_block: ``AWS::EC2::Route.DestinationCidrBlock``.
        :param destination_ipv6_cidr_block: ``AWS::EC2::Route.DestinationIpv6CidrBlock``.
        :param egress_only_internet_gateway_id: ``AWS::EC2::Route.EgressOnlyInternetGatewayId``.
        :param gateway_id: ``AWS::EC2::Route.GatewayId``.
        :param instance_id: ``AWS::EC2::Route.InstanceId``.
        :param local_gateway_id: ``AWS::EC2::Route.LocalGatewayId``.
        :param nat_gateway_id: ``AWS::EC2::Route.NatGatewayId``.
        :param network_interface_id: ``AWS::EC2::Route.NetworkInterfaceId``.
        :param transit_gateway_id: ``AWS::EC2::Route.TransitGatewayId``.
        :param vpc_endpoint_id: ``AWS::EC2::Route.VpcEndpointId``.
        :param vpc_peering_connection_id: ``AWS::EC2::Route.VpcPeeringConnectionId``.
        '''
        props = CfnRouteProps(
            route_table_id=route_table_id,
            carrier_gateway_id=carrier_gateway_id,
            destination_cidr_block=destination_cidr_block,
            destination_ipv6_cidr_block=destination_ipv6_cidr_block,
            egress_only_internet_gateway_id=egress_only_internet_gateway_id,
            gateway_id=gateway_id,
            instance_id=instance_id,
            local_gateway_id=local_gateway_id,
            nat_gateway_id=nat_gateway_id,
            network_interface_id=network_interface_id,
            transit_gateway_id=transit_gateway_id,
            vpc_endpoint_id=vpc_endpoint_id,
            vpc_peering_connection_id=vpc_peering_connection_id,
        )

        jsii.create(CfnRoute, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="routeTableId")
    def route_table_id(self) -> builtins.str:
        '''``AWS::EC2::Route.RouteTableId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-routetableid
        '''
        return typing.cast(builtins.str, jsii.get(self, "routeTableId"))

    @route_table_id.setter
    def route_table_id(self, value: builtins.str) -> None:
        jsii.set(self, "routeTableId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="carrierGatewayId")
    def carrier_gateway_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.CarrierGatewayId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-carriergatewayid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "carrierGatewayId"))

    @carrier_gateway_id.setter
    def carrier_gateway_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "carrierGatewayId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="destinationCidrBlock")
    def destination_cidr_block(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.DestinationCidrBlock``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-destinationcidrblock
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "destinationCidrBlock"))

    @destination_cidr_block.setter
    def destination_cidr_block(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "destinationCidrBlock", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="destinationIpv6CidrBlock")
    def destination_ipv6_cidr_block(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.DestinationIpv6CidrBlock``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-destinationipv6cidrblock
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "destinationIpv6CidrBlock"))

    @destination_ipv6_cidr_block.setter
    def destination_ipv6_cidr_block(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "destinationIpv6CidrBlock", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="egressOnlyInternetGatewayId")
    def egress_only_internet_gateway_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.EgressOnlyInternetGatewayId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-egressonlyinternetgatewayid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "egressOnlyInternetGatewayId"))

    @egress_only_internet_gateway_id.setter
    def egress_only_internet_gateway_id(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        jsii.set(self, "egressOnlyInternetGatewayId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="gatewayId")
    def gateway_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.GatewayId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-gatewayid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "gatewayId"))

    @gateway_id.setter
    def gateway_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "gatewayId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceId")
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.InstanceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-instanceid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "instanceId"))

    @instance_id.setter
    def instance_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "instanceId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="localGatewayId")
    def local_gateway_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.LocalGatewayId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-localgatewayid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "localGatewayId"))

    @local_gateway_id.setter
    def local_gateway_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "localGatewayId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="natGatewayId")
    def nat_gateway_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.NatGatewayId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-natgatewayid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "natGatewayId"))

    @nat_gateway_id.setter
    def nat_gateway_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "natGatewayId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkInterfaceId")
    def network_interface_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.NetworkInterfaceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-networkinterfaceid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "networkInterfaceId"))

    @network_interface_id.setter
    def network_interface_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "networkInterfaceId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="transitGatewayId")
    def transit_gateway_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.TransitGatewayId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-transitgatewayid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "transitGatewayId"))

    @transit_gateway_id.setter
    def transit_gateway_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "transitGatewayId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="vpcEndpointId")
    def vpc_endpoint_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.VpcEndpointId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-vpcendpointid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "vpcEndpointId"))

    @vpc_endpoint_id.setter
    def vpc_endpoint_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "vpcEndpointId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="vpcPeeringConnectionId")
    def vpc_peering_connection_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.VpcPeeringConnectionId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-vpcpeeringconnectionid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "vpcPeeringConnectionId"))

    @vpc_peering_connection_id.setter
    def vpc_peering_connection_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "vpcPeeringConnectionId", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnRouteProps",
    jsii_struct_bases=[],
    name_mapping={
        "route_table_id": "routeTableId",
        "carrier_gateway_id": "carrierGatewayId",
        "destination_cidr_block": "destinationCidrBlock",
        "destination_ipv6_cidr_block": "destinationIpv6CidrBlock",
        "egress_only_internet_gateway_id": "egressOnlyInternetGatewayId",
        "gateway_id": "gatewayId",
        "instance_id": "instanceId",
        "local_gateway_id": "localGatewayId",
        "nat_gateway_id": "natGatewayId",
        "network_interface_id": "networkInterfaceId",
        "transit_gateway_id": "transitGatewayId",
        "vpc_endpoint_id": "vpcEndpointId",
        "vpc_peering_connection_id": "vpcPeeringConnectionId",
    },
)
class CfnRouteProps:
    def __init__(
        self,
        *,
        route_table_id: builtins.str,
        carrier_gateway_id: typing.Optional[builtins.str] = None,
        destination_cidr_block: typing.Optional[builtins.str] = None,
        destination_ipv6_cidr_block: typing.Optional[builtins.str] = None,
        egress_only_internet_gateway_id: typing.Optional[builtins.str] = None,
        gateway_id: typing.Optional[builtins.str] = None,
        instance_id: typing.Optional[builtins.str] = None,
        local_gateway_id: typing.Optional[builtins.str] = None,
        nat_gateway_id: typing.Optional[builtins.str] = None,
        network_interface_id: typing.Optional[builtins.str] = None,
        transit_gateway_id: typing.Optional[builtins.str] = None,
        vpc_endpoint_id: typing.Optional[builtins.str] = None,
        vpc_peering_connection_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::Route``.

        :param route_table_id: ``AWS::EC2::Route.RouteTableId``.
        :param carrier_gateway_id: ``AWS::EC2::Route.CarrierGatewayId``.
        :param destination_cidr_block: ``AWS::EC2::Route.DestinationCidrBlock``.
        :param destination_ipv6_cidr_block: ``AWS::EC2::Route.DestinationIpv6CidrBlock``.
        :param egress_only_internet_gateway_id: ``AWS::EC2::Route.EgressOnlyInternetGatewayId``.
        :param gateway_id: ``AWS::EC2::Route.GatewayId``.
        :param instance_id: ``AWS::EC2::Route.InstanceId``.
        :param local_gateway_id: ``AWS::EC2::Route.LocalGatewayId``.
        :param nat_gateway_id: ``AWS::EC2::Route.NatGatewayId``.
        :param network_interface_id: ``AWS::EC2::Route.NetworkInterfaceId``.
        :param transit_gateway_id: ``AWS::EC2::Route.TransitGatewayId``.
        :param vpc_endpoint_id: ``AWS::EC2::Route.VpcEndpointId``.
        :param vpc_peering_connection_id: ``AWS::EC2::Route.VpcPeeringConnectionId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "route_table_id": route_table_id,
        }
        if carrier_gateway_id is not None:
            self._values["carrier_gateway_id"] = carrier_gateway_id
        if destination_cidr_block is not None:
            self._values["destination_cidr_block"] = destination_cidr_block
        if destination_ipv6_cidr_block is not None:
            self._values["destination_ipv6_cidr_block"] = destination_ipv6_cidr_block
        if egress_only_internet_gateway_id is not None:
            self._values["egress_only_internet_gateway_id"] = egress_only_internet_gateway_id
        if gateway_id is not None:
            self._values["gateway_id"] = gateway_id
        if instance_id is not None:
            self._values["instance_id"] = instance_id
        if local_gateway_id is not None:
            self._values["local_gateway_id"] = local_gateway_id
        if nat_gateway_id is not None:
            self._values["nat_gateway_id"] = nat_gateway_id
        if network_interface_id is not None:
            self._values["network_interface_id"] = network_interface_id
        if transit_gateway_id is not None:
            self._values["transit_gateway_id"] = transit_gateway_id
        if vpc_endpoint_id is not None:
            self._values["vpc_endpoint_id"] = vpc_endpoint_id
        if vpc_peering_connection_id is not None:
            self._values["vpc_peering_connection_id"] = vpc_peering_connection_id

    @builtins.property
    def route_table_id(self) -> builtins.str:
        '''``AWS::EC2::Route.RouteTableId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-routetableid
        '''
        result = self._values.get("route_table_id")
        assert result is not None, "Required property 'route_table_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def carrier_gateway_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.CarrierGatewayId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-carriergatewayid
        '''
        result = self._values.get("carrier_gateway_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def destination_cidr_block(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.DestinationCidrBlock``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-destinationcidrblock
        '''
        result = self._values.get("destination_cidr_block")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def destination_ipv6_cidr_block(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.DestinationIpv6CidrBlock``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-destinationipv6cidrblock
        '''
        result = self._values.get("destination_ipv6_cidr_block")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def egress_only_internet_gateway_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.EgressOnlyInternetGatewayId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-egressonlyinternetgatewayid
        '''
        result = self._values.get("egress_only_internet_gateway_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def gateway_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.GatewayId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-gatewayid
        '''
        result = self._values.get("gateway_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def instance_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.InstanceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-instanceid
        '''
        result = self._values.get("instance_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def local_gateway_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.LocalGatewayId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-localgatewayid
        '''
        result = self._values.get("local_gateway_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def nat_gateway_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.NatGatewayId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-natgatewayid
        '''
        result = self._values.get("nat_gateway_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def network_interface_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.NetworkInterfaceId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-networkinterfaceid
        '''
        result = self._values.get("network_interface_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def transit_gateway_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.TransitGatewayId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-transitgatewayid
        '''
        result = self._values.get("transit_gateway_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def vpc_endpoint_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.VpcEndpointId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-vpcendpointid
        '''
        result = self._values.get("vpc_endpoint_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def vpc_peering_connection_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::Route.VpcPeeringConnectionId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route.html#cfn-ec2-route-vpcpeeringconnectionid
        '''
        result = self._values.get("vpc_peering_connection_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnRouteProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnRouteTable(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnRouteTable",
):
    '''A CloudFormation ``AWS::EC2::RouteTable``.

    :cloudformationResource: AWS::EC2::RouteTable
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route-table.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        vpc_id: builtins.str,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::RouteTable``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param vpc_id: ``AWS::EC2::RouteTable.VpcId``.
        :param tags: ``AWS::EC2::RouteTable.Tags``.
        '''
        props = CfnRouteTableProps(vpc_id=vpc_id, tags=tags)

        jsii.create(CfnRouteTable, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''``AWS::EC2::RouteTable.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route-table.html#cfn-ec2-routetable-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="vpcId")
    def vpc_id(self) -> builtins.str:
        '''``AWS::EC2::RouteTable.VpcId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route-table.html#cfn-ec2-routetable-vpcid
        '''
        return typing.cast(builtins.str, jsii.get(self, "vpcId"))

    @vpc_id.setter
    def vpc_id(self, value: builtins.str) -> None:
        jsii.set(self, "vpcId", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnRouteTableProps",
    jsii_struct_bases=[],
    name_mapping={"vpc_id": "vpcId", "tags": "tags"},
)
class CfnRouteTableProps:
    def __init__(
        self,
        *,
        vpc_id: builtins.str,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::RouteTable``.

        :param vpc_id: ``AWS::EC2::RouteTable.VpcId``.
        :param tags: ``AWS::EC2::RouteTable.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route-table.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "vpc_id": vpc_id,
        }
        if tags is not None:
            self._values["tags"] = tags

    @builtins.property
    def vpc_id(self) -> builtins.str:
        '''``AWS::EC2::RouteTable.VpcId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route-table.html#cfn-ec2-routetable-vpcid
        '''
        result = self._values.get("vpc_id")
        assert result is not None, "Required property 'vpc_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''``AWS::EC2::RouteTable.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-route-table.html#cfn-ec2-routetable-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnRouteTableProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnSecurityGroup(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnSecurityGroup",
):
    '''A CloudFormation ``AWS::EC2::SecurityGroup``.

    :cloudformationResource: AWS::EC2::SecurityGroup
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        group_description: builtins.str,
        group_name: typing.Optional[builtins.str] = None,
        security_group_egress: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnSecurityGroup.EgressProperty"]]]] = None,
        security_group_ingress: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnSecurityGroup.IngressProperty"]]]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
        vpc_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::SecurityGroup``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param group_description: ``AWS::EC2::SecurityGroup.GroupDescription``.
        :param group_name: ``AWS::EC2::SecurityGroup.GroupName``.
        :param security_group_egress: ``AWS::EC2::SecurityGroup.SecurityGroupEgress``.
        :param security_group_ingress: ``AWS::EC2::SecurityGroup.SecurityGroupIngress``.
        :param tags: ``AWS::EC2::SecurityGroup.Tags``.
        :param vpc_id: ``AWS::EC2::SecurityGroup.VpcId``.
        '''
        props = CfnSecurityGroupProps(
            group_description=group_description,
            group_name=group_name,
            security_group_egress=security_group_egress,
            security_group_ingress=security_group_ingress,
            tags=tags,
            vpc_id=vpc_id,
        )

        jsii.create(CfnSecurityGroup, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrGroupId")
    def attr_group_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: GroupId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrGroupId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrVpcId")
    def attr_vpc_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: VpcId
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrVpcId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> aws_cdk.core.TagManager:
        '''``AWS::EC2::SecurityGroup.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html#cfn-ec2-securitygroup-tags
        '''
        return typing.cast(aws_cdk.core.TagManager, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="groupDescription")
    def group_description(self) -> builtins.str:
        '''``AWS::EC2::SecurityGroup.GroupDescription``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html#cfn-ec2-securitygroup-groupdescription
        '''
        return typing.cast(builtins.str, jsii.get(self, "groupDescription"))

    @group_description.setter
    def group_description(self, value: builtins.str) -> None:
        jsii.set(self, "groupDescription", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="groupName")
    def group_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroup.GroupName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html#cfn-ec2-securitygroup-groupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "groupName"))

    @group_name.setter
    def group_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "groupName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="securityGroupEgress")
    def security_group_egress(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnSecurityGroup.EgressProperty"]]]]:
        '''``AWS::EC2::SecurityGroup.SecurityGroupEgress``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html#cfn-ec2-securitygroup-securitygroupegress
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnSecurityGroup.EgressProperty"]]]], jsii.get(self, "securityGroupEgress"))

    @security_group_egress.setter
    def security_group_egress(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnSecurityGroup.EgressProperty"]]]],
    ) -> None:
        jsii.set(self, "securityGroupEgress", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="securityGroupIngress")
    def security_group_ingress(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnSecurityGroup.IngressProperty"]]]]:
        '''``AWS::EC2::SecurityGroup.SecurityGroupIngress``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html#cfn-ec2-securitygroup-securitygroupingress
        '''
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnSecurityGroup.IngressProperty"]]]], jsii.get(self, "securityGroupIngress"))

    @security_group_ingress.setter
    def security_group_ingress(
        self,
        value: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnSecurityGroup.IngressProperty"]]]],
    ) -> None:
        jsii.set(self, "securityGroupIngress", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="vpcId")
    def vpc_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroup.VpcId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html#cfn-ec2-securitygroup-vpcid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "vpcId"))

    @vpc_id.setter
    def vpc_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "vpcId", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnSecurityGroup.EgressProperty",
        jsii_struct_bases=[],
        name_mapping={
            "ip_protocol": "ipProtocol",
            "cidr_ip": "cidrIp",
            "cidr_ipv6": "cidrIpv6",
            "description": "description",
            "destination_prefix_list_id": "destinationPrefixListId",
            "destination_security_group_id": "destinationSecurityGroupId",
            "from_port": "fromPort",
            "to_port": "toPort",
        },
    )
    class EgressProperty:
        def __init__(
            self,
            *,
            ip_protocol: builtins.str,
            cidr_ip: typing.Optional[builtins.str] = None,
            cidr_ipv6: typing.Optional[builtins.str] = None,
            description: typing.Optional[builtins.str] = None,
            destination_prefix_list_id: typing.Optional[builtins.str] = None,
            destination_security_group_id: typing.Optional[builtins.str] = None,
            from_port: typing.Optional[jsii.Number] = None,
            to_port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param ip_protocol: ``CfnSecurityGroup.EgressProperty.IpProtocol``.
            :param cidr_ip: ``CfnSecurityGroup.EgressProperty.CidrIp``.
            :param cidr_ipv6: ``CfnSecurityGroup.EgressProperty.CidrIpv6``.
            :param description: ``CfnSecurityGroup.EgressProperty.Description``.
            :param destination_prefix_list_id: ``CfnSecurityGroup.EgressProperty.DestinationPrefixListId``.
            :param destination_security_group_id: ``CfnSecurityGroup.EgressProperty.DestinationSecurityGroupId``.
            :param from_port: ``CfnSecurityGroup.EgressProperty.FromPort``.
            :param to_port: ``CfnSecurityGroup.EgressProperty.ToPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "ip_protocol": ip_protocol,
            }
            if cidr_ip is not None:
                self._values["cidr_ip"] = cidr_ip
            if cidr_ipv6 is not None:
                self._values["cidr_ipv6"] = cidr_ipv6
            if description is not None:
                self._values["description"] = description
            if destination_prefix_list_id is not None:
                self._values["destination_prefix_list_id"] = destination_prefix_list_id
            if destination_security_group_id is not None:
                self._values["destination_security_group_id"] = destination_security_group_id
            if from_port is not None:
                self._values["from_port"] = from_port
            if to_port is not None:
                self._values["to_port"] = to_port

        @builtins.property
        def ip_protocol(self) -> builtins.str:
            '''``CfnSecurityGroup.EgressProperty.IpProtocol``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-security-group-rule-ipprotocol
            '''
            result = self._values.get("ip_protocol")
            assert result is not None, "Required property 'ip_protocol' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def cidr_ip(self) -> typing.Optional[builtins.str]:
            '''``CfnSecurityGroup.EgressProperty.CidrIp``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-security-group-rule-cidrip
            '''
            result = self._values.get("cidr_ip")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def cidr_ipv6(self) -> typing.Optional[builtins.str]:
            '''``CfnSecurityGroup.EgressProperty.CidrIpv6``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-security-group-rule-cidripv6
            '''
            result = self._values.get("cidr_ipv6")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            '''``CfnSecurityGroup.EgressProperty.Description``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-security-group-rule-description
            '''
            result = self._values.get("description")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def destination_prefix_list_id(self) -> typing.Optional[builtins.str]:
            '''``CfnSecurityGroup.EgressProperty.DestinationPrefixListId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-security-group-rule-destinationprefixlistid
            '''
            result = self._values.get("destination_prefix_list_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def destination_security_group_id(self) -> typing.Optional[builtins.str]:
            '''``CfnSecurityGroup.EgressProperty.DestinationSecurityGroupId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-security-group-rule-destsecgroupid
            '''
            result = self._values.get("destination_security_group_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def from_port(self) -> typing.Optional[jsii.Number]:
            '''``CfnSecurityGroup.EgressProperty.FromPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-security-group-rule-fromport
            '''
            result = self._values.get("from_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def to_port(self) -> typing.Optional[jsii.Number]:
            '''``CfnSecurityGroup.EgressProperty.ToPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-security-group-rule-toport
            '''
            result = self._values.get("to_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EgressProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnSecurityGroup.IngressProperty",
        jsii_struct_bases=[],
        name_mapping={
            "ip_protocol": "ipProtocol",
            "cidr_ip": "cidrIp",
            "cidr_ipv6": "cidrIpv6",
            "description": "description",
            "from_port": "fromPort",
            "source_prefix_list_id": "sourcePrefixListId",
            "source_security_group_id": "sourceSecurityGroupId",
            "source_security_group_name": "sourceSecurityGroupName",
            "source_security_group_owner_id": "sourceSecurityGroupOwnerId",
            "to_port": "toPort",
        },
    )
    class IngressProperty:
        def __init__(
            self,
            *,
            ip_protocol: builtins.str,
            cidr_ip: typing.Optional[builtins.str] = None,
            cidr_ipv6: typing.Optional[builtins.str] = None,
            description: typing.Optional[builtins.str] = None,
            from_port: typing.Optional[jsii.Number] = None,
            source_prefix_list_id: typing.Optional[builtins.str] = None,
            source_security_group_id: typing.Optional[builtins.str] = None,
            source_security_group_name: typing.Optional[builtins.str] = None,
            source_security_group_owner_id: typing.Optional[builtins.str] = None,
            to_port: typing.Optional[jsii.Number] = None,
        ) -> None:
            '''
            :param ip_protocol: ``CfnSecurityGroup.IngressProperty.IpProtocol``.
            :param cidr_ip: ``CfnSecurityGroup.IngressProperty.CidrIp``.
            :param cidr_ipv6: ``CfnSecurityGroup.IngressProperty.CidrIpv6``.
            :param description: ``CfnSecurityGroup.IngressProperty.Description``.
            :param from_port: ``CfnSecurityGroup.IngressProperty.FromPort``.
            :param source_prefix_list_id: ``CfnSecurityGroup.IngressProperty.SourcePrefixListId``.
            :param source_security_group_id: ``CfnSecurityGroup.IngressProperty.SourceSecurityGroupId``.
            :param source_security_group_name: ``CfnSecurityGroup.IngressProperty.SourceSecurityGroupName``.
            :param source_security_group_owner_id: ``CfnSecurityGroup.IngressProperty.SourceSecurityGroupOwnerId``.
            :param to_port: ``CfnSecurityGroup.IngressProperty.ToPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "ip_protocol": ip_protocol,
            }
            if cidr_ip is not None:
                self._values["cidr_ip"] = cidr_ip
            if cidr_ipv6 is not None:
                self._values["cidr_ipv6"] = cidr_ipv6
            if description is not None:
                self._values["description"] = description
            if from_port is not None:
                self._values["from_port"] = from_port
            if source_prefix_list_id is not None:
                self._values["source_prefix_list_id"] = source_prefix_list_id
            if source_security_group_id is not None:
                self._values["source_security_group_id"] = source_security_group_id
            if source_security_group_name is not None:
                self._values["source_security_group_name"] = source_security_group_name
            if source_security_group_owner_id is not None:
                self._values["source_security_group_owner_id"] = source_security_group_owner_id
            if to_port is not None:
                self._values["to_port"] = to_port

        @builtins.property
        def ip_protocol(self) -> builtins.str:
            '''``CfnSecurityGroup.IngressProperty.IpProtocol``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-security-group-rule-ipprotocol
            '''
            result = self._values.get("ip_protocol")
            assert result is not None, "Required property 'ip_protocol' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def cidr_ip(self) -> typing.Optional[builtins.str]:
            '''``CfnSecurityGroup.IngressProperty.CidrIp``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-security-group-rule-cidrip
            '''
            result = self._values.get("cidr_ip")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def cidr_ipv6(self) -> typing.Optional[builtins.str]:
            '''``CfnSecurityGroup.IngressProperty.CidrIpv6``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-security-group-rule-cidripv6
            '''
            result = self._values.get("cidr_ipv6")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            '''``CfnSecurityGroup.IngressProperty.Description``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-security-group-rule-description
            '''
            result = self._values.get("description")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def from_port(self) -> typing.Optional[jsii.Number]:
            '''``CfnSecurityGroup.IngressProperty.FromPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-security-group-rule-fromport
            '''
            result = self._values.get("from_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def source_prefix_list_id(self) -> typing.Optional[builtins.str]:
            '''``CfnSecurityGroup.IngressProperty.SourcePrefixListId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-securitygroup-ingress-sourceprefixlistid
            '''
            result = self._values.get("source_prefix_list_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def source_security_group_id(self) -> typing.Optional[builtins.str]:
            '''``CfnSecurityGroup.IngressProperty.SourceSecurityGroupId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-security-group-rule-sourcesecuritygroupid
            '''
            result = self._values.get("source_security_group_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def source_security_group_name(self) -> typing.Optional[builtins.str]:
            '''``CfnSecurityGroup.IngressProperty.SourceSecurityGroupName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-security-group-rule-sourcesecuritygroupname
            '''
            result = self._values.get("source_security_group_name")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def source_security_group_owner_id(self) -> typing.Optional[builtins.str]:
            '''``CfnSecurityGroup.IngressProperty.SourceSecurityGroupOwnerId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-security-group-rule-sourcesecuritygroupownerid
            '''
            result = self._values.get("source_security_group_owner_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def to_port(self) -> typing.Optional[jsii.Number]:
            '''``CfnSecurityGroup.IngressProperty.ToPort``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-rule.html#cfn-ec2-security-group-rule-toport
            '''
            result = self._values.get("to_port")
            return typing.cast(typing.Optional[jsii.Number], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "IngressProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnSecurityGroupEgress(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnSecurityGroupEgress",
):
    '''A CloudFormation ``AWS::EC2::SecurityGroupEgress``.

    :cloudformationResource: AWS::EC2::SecurityGroupEgress
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        group_id: builtins.str,
        ip_protocol: builtins.str,
        cidr_ip: typing.Optional[builtins.str] = None,
        cidr_ipv6: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        destination_prefix_list_id: typing.Optional[builtins.str] = None,
        destination_security_group_id: typing.Optional[builtins.str] = None,
        from_port: typing.Optional[jsii.Number] = None,
        to_port: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::SecurityGroupEgress``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param group_id: ``AWS::EC2::SecurityGroupEgress.GroupId``.
        :param ip_protocol: ``AWS::EC2::SecurityGroupEgress.IpProtocol``.
        :param cidr_ip: ``AWS::EC2::SecurityGroupEgress.CidrIp``.
        :param cidr_ipv6: ``AWS::EC2::SecurityGroupEgress.CidrIpv6``.
        :param description: ``AWS::EC2::SecurityGroupEgress.Description``.
        :param destination_prefix_list_id: ``AWS::EC2::SecurityGroupEgress.DestinationPrefixListId``.
        :param destination_security_group_id: ``AWS::EC2::SecurityGroupEgress.DestinationSecurityGroupId``.
        :param from_port: ``AWS::EC2::SecurityGroupEgress.FromPort``.
        :param to_port: ``AWS::EC2::SecurityGroupEgress.ToPort``.
        '''
        props = CfnSecurityGroupEgressProps(
            group_id=group_id,
            ip_protocol=ip_protocol,
            cidr_ip=cidr_ip,
            cidr_ipv6=cidr_ipv6,
            description=description,
            destination_prefix_list_id=destination_prefix_list_id,
            destination_security_group_id=destination_security_group_id,
            from_port=from_port,
            to_port=to_port,
        )

        jsii.create(CfnSecurityGroupEgress, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="groupId")
    def group_id(self) -> builtins.str:
        '''``AWS::EC2::SecurityGroupEgress.GroupId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-groupid
        '''
        return typing.cast(builtins.str, jsii.get(self, "groupId"))

    @group_id.setter
    def group_id(self, value: builtins.str) -> None:
        jsii.set(self, "groupId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ipProtocol")
    def ip_protocol(self) -> builtins.str:
        '''``AWS::EC2::SecurityGroupEgress.IpProtocol``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-ipprotocol
        '''
        return typing.cast(builtins.str, jsii.get(self, "ipProtocol"))

    @ip_protocol.setter
    def ip_protocol(self, value: builtins.str) -> None:
        jsii.set(self, "ipProtocol", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cidrIp")
    def cidr_ip(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupEgress.CidrIp``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-cidrip
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cidrIp"))

    @cidr_ip.setter
    def cidr_ip(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "cidrIp", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cidrIpv6")
    def cidr_ipv6(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupEgress.CidrIpv6``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-cidripv6
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cidrIpv6"))

    @cidr_ipv6.setter
    def cidr_ipv6(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "cidrIpv6", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupEgress.Description``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "description", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="destinationPrefixListId")
    def destination_prefix_list_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupEgress.DestinationPrefixListId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-destinationprefixlistid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "destinationPrefixListId"))

    @destination_prefix_list_id.setter
    def destination_prefix_list_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "destinationPrefixListId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="destinationSecurityGroupId")
    def destination_security_group_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupEgress.DestinationSecurityGroupId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-destinationsecuritygroupid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "destinationSecurityGroupId"))

    @destination_security_group_id.setter
    def destination_security_group_id(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        jsii.set(self, "destinationSecurityGroupId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="fromPort")
    def from_port(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::SecurityGroupEgress.FromPort``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-fromport
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "fromPort"))

    @from_port.setter
    def from_port(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "fromPort", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="toPort")
    def to_port(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::SecurityGroupEgress.ToPort``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-toport
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "toPort"))

    @to_port.setter
    def to_port(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "toPort", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnSecurityGroupEgressProps",
    jsii_struct_bases=[],
    name_mapping={
        "group_id": "groupId",
        "ip_protocol": "ipProtocol",
        "cidr_ip": "cidrIp",
        "cidr_ipv6": "cidrIpv6",
        "description": "description",
        "destination_prefix_list_id": "destinationPrefixListId",
        "destination_security_group_id": "destinationSecurityGroupId",
        "from_port": "fromPort",
        "to_port": "toPort",
    },
)
class CfnSecurityGroupEgressProps:
    def __init__(
        self,
        *,
        group_id: builtins.str,
        ip_protocol: builtins.str,
        cidr_ip: typing.Optional[builtins.str] = None,
        cidr_ipv6: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        destination_prefix_list_id: typing.Optional[builtins.str] = None,
        destination_security_group_id: typing.Optional[builtins.str] = None,
        from_port: typing.Optional[jsii.Number] = None,
        to_port: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::SecurityGroupEgress``.

        :param group_id: ``AWS::EC2::SecurityGroupEgress.GroupId``.
        :param ip_protocol: ``AWS::EC2::SecurityGroupEgress.IpProtocol``.
        :param cidr_ip: ``AWS::EC2::SecurityGroupEgress.CidrIp``.
        :param cidr_ipv6: ``AWS::EC2::SecurityGroupEgress.CidrIpv6``.
        :param description: ``AWS::EC2::SecurityGroupEgress.Description``.
        :param destination_prefix_list_id: ``AWS::EC2::SecurityGroupEgress.DestinationPrefixListId``.
        :param destination_security_group_id: ``AWS::EC2::SecurityGroupEgress.DestinationSecurityGroupId``.
        :param from_port: ``AWS::EC2::SecurityGroupEgress.FromPort``.
        :param to_port: ``AWS::EC2::SecurityGroupEgress.ToPort``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "group_id": group_id,
            "ip_protocol": ip_protocol,
        }
        if cidr_ip is not None:
            self._values["cidr_ip"] = cidr_ip
        if cidr_ipv6 is not None:
            self._values["cidr_ipv6"] = cidr_ipv6
        if description is not None:
            self._values["description"] = description
        if destination_prefix_list_id is not None:
            self._values["destination_prefix_list_id"] = destination_prefix_list_id
        if destination_security_group_id is not None:
            self._values["destination_security_group_id"] = destination_security_group_id
        if from_port is not None:
            self._values["from_port"] = from_port
        if to_port is not None:
            self._values["to_port"] = to_port

    @builtins.property
    def group_id(self) -> builtins.str:
        '''``AWS::EC2::SecurityGroupEgress.GroupId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-groupid
        '''
        result = self._values.get("group_id")
        assert result is not None, "Required property 'group_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def ip_protocol(self) -> builtins.str:
        '''``AWS::EC2::SecurityGroupEgress.IpProtocol``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-ipprotocol
        '''
        result = self._values.get("ip_protocol")
        assert result is not None, "Required property 'ip_protocol' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def cidr_ip(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupEgress.CidrIp``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-cidrip
        '''
        result = self._values.get("cidr_ip")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cidr_ipv6(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupEgress.CidrIpv6``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-cidripv6
        '''
        result = self._values.get("cidr_ipv6")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupEgress.Description``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def destination_prefix_list_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupEgress.DestinationPrefixListId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-destinationprefixlistid
        '''
        result = self._values.get("destination_prefix_list_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def destination_security_group_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupEgress.DestinationSecurityGroupId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-destinationsecuritygroupid
        '''
        result = self._values.get("destination_security_group_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def from_port(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::SecurityGroupEgress.FromPort``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-fromport
        '''
        result = self._values.get("from_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def to_port(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::SecurityGroupEgress.ToPort``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-security-group-egress.html#cfn-ec2-securitygroupegress-toport
        '''
        result = self._values.get("to_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnSecurityGroupEgressProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnSecurityGroupIngress(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnSecurityGroupIngress",
):
    '''A CloudFormation ``AWS::EC2::SecurityGroupIngress``.

    :cloudformationResource: AWS::EC2::SecurityGroupIngress
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        ip_protocol: builtins.str,
        cidr_ip: typing.Optional[builtins.str] = None,
        cidr_ipv6: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        from_port: typing.Optional[jsii.Number] = None,
        group_id: typing.Optional[builtins.str] = None,
        group_name: typing.Optional[builtins.str] = None,
        source_prefix_list_id: typing.Optional[builtins.str] = None,
        source_security_group_id: typing.Optional[builtins.str] = None,
        source_security_group_name: typing.Optional[builtins.str] = None,
        source_security_group_owner_id: typing.Optional[builtins.str] = None,
        to_port: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Create a new ``AWS::EC2::SecurityGroupIngress``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param ip_protocol: ``AWS::EC2::SecurityGroupIngress.IpProtocol``.
        :param cidr_ip: ``AWS::EC2::SecurityGroupIngress.CidrIp``.
        :param cidr_ipv6: ``AWS::EC2::SecurityGroupIngress.CidrIpv6``.
        :param description: ``AWS::EC2::SecurityGroupIngress.Description``.
        :param from_port: ``AWS::EC2::SecurityGroupIngress.FromPort``.
        :param group_id: ``AWS::EC2::SecurityGroupIngress.GroupId``.
        :param group_name: ``AWS::EC2::SecurityGroupIngress.GroupName``.
        :param source_prefix_list_id: ``AWS::EC2::SecurityGroupIngress.SourcePrefixListId``.
        :param source_security_group_id: ``AWS::EC2::SecurityGroupIngress.SourceSecurityGroupId``.
        :param source_security_group_name: ``AWS::EC2::SecurityGroupIngress.SourceSecurityGroupName``.
        :param source_security_group_owner_id: ``AWS::EC2::SecurityGroupIngress.SourceSecurityGroupOwnerId``.
        :param to_port: ``AWS::EC2::SecurityGroupIngress.ToPort``.
        '''
        props = CfnSecurityGroupIngressProps(
            ip_protocol=ip_protocol,
            cidr_ip=cidr_ip,
            cidr_ipv6=cidr_ipv6,
            description=description,
            from_port=from_port,
            group_id=group_id,
            group_name=group_name,
            source_prefix_list_id=source_prefix_list_id,
            source_security_group_id=source_security_group_id,
            source_security_group_name=source_security_group_name,
            source_security_group_owner_id=source_security_group_owner_id,
            to_port=to_port,
        )

        jsii.create(CfnSecurityGroupIngress, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ipProtocol")
    def ip_protocol(self) -> builtins.str:
        '''``AWS::EC2::SecurityGroupIngress.IpProtocol``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-ipprotocol
        '''
        return typing.cast(builtins.str, jsii.get(self, "ipProtocol"))

    @ip_protocol.setter
    def ip_protocol(self, value: builtins.str) -> None:
        jsii.set(self, "ipProtocol", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cidrIp")
    def cidr_ip(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.CidrIp``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-cidrip
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cidrIp"))

    @cidr_ip.setter
    def cidr_ip(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "cidrIp", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cidrIpv6")
    def cidr_ipv6(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.CidrIpv6``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-cidripv6
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "cidrIpv6"))

    @cidr_ipv6.setter
    def cidr_ipv6(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "cidrIpv6", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.Description``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-description
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))

    @description.setter
    def description(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "description", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="fromPort")
    def from_port(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::SecurityGroupIngress.FromPort``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-fromport
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "fromPort"))

    @from_port.setter
    def from_port(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "fromPort", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="groupId")
    def group_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.GroupId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-groupid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "groupId"))

    @group_id.setter
    def group_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "groupId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="groupName")
    def group_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.GroupName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-groupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "groupName"))

    @group_name.setter
    def group_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "groupName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourcePrefixListId")
    def source_prefix_list_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.SourcePrefixListId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-securitygroupingress-sourceprefixlistid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourcePrefixListId"))

    @source_prefix_list_id.setter
    def source_prefix_list_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "sourcePrefixListId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceSecurityGroupId")
    def source_security_group_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.SourceSecurityGroupId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-sourcesecuritygroupid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceSecurityGroupId"))

    @source_security_group_id.setter
    def source_security_group_id(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "sourceSecurityGroupId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceSecurityGroupName")
    def source_security_group_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.SourceSecurityGroupName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-sourcesecuritygroupname
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceSecurityGroupName"))

    @source_security_group_name.setter
    def source_security_group_name(self, value: typing.Optional[builtins.str]) -> None:
        jsii.set(self, "sourceSecurityGroupName", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceSecurityGroupOwnerId")
    def source_security_group_owner_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.SourceSecurityGroupOwnerId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-sourcesecuritygroupownerid
        '''
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "sourceSecurityGroupOwnerId"))

    @source_security_group_owner_id.setter
    def source_security_group_owner_id(
        self,
        value: typing.Optional[builtins.str],
    ) -> None:
        jsii.set(self, "sourceSecurityGroupOwnerId", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="toPort")
    def to_port(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::SecurityGroupIngress.ToPort``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-toport
        '''
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "toPort"))

    @to_port.setter
    def to_port(self, value: typing.Optional[jsii.Number]) -> None:
        jsii.set(self, "toPort", value)


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnSecurityGroupIngressProps",
    jsii_struct_bases=[],
    name_mapping={
        "ip_protocol": "ipProtocol",
        "cidr_ip": "cidrIp",
        "cidr_ipv6": "cidrIpv6",
        "description": "description",
        "from_port": "fromPort",
        "group_id": "groupId",
        "group_name": "groupName",
        "source_prefix_list_id": "sourcePrefixListId",
        "source_security_group_id": "sourceSecurityGroupId",
        "source_security_group_name": "sourceSecurityGroupName",
        "source_security_group_owner_id": "sourceSecurityGroupOwnerId",
        "to_port": "toPort",
    },
)
class CfnSecurityGroupIngressProps:
    def __init__(
        self,
        *,
        ip_protocol: builtins.str,
        cidr_ip: typing.Optional[builtins.str] = None,
        cidr_ipv6: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        from_port: typing.Optional[jsii.Number] = None,
        group_id: typing.Optional[builtins.str] = None,
        group_name: typing.Optional[builtins.str] = None,
        source_prefix_list_id: typing.Optional[builtins.str] = None,
        source_security_group_id: typing.Optional[builtins.str] = None,
        source_security_group_name: typing.Optional[builtins.str] = None,
        source_security_group_owner_id: typing.Optional[builtins.str] = None,
        to_port: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::SecurityGroupIngress``.

        :param ip_protocol: ``AWS::EC2::SecurityGroupIngress.IpProtocol``.
        :param cidr_ip: ``AWS::EC2::SecurityGroupIngress.CidrIp``.
        :param cidr_ipv6: ``AWS::EC2::SecurityGroupIngress.CidrIpv6``.
        :param description: ``AWS::EC2::SecurityGroupIngress.Description``.
        :param from_port: ``AWS::EC2::SecurityGroupIngress.FromPort``.
        :param group_id: ``AWS::EC2::SecurityGroupIngress.GroupId``.
        :param group_name: ``AWS::EC2::SecurityGroupIngress.GroupName``.
        :param source_prefix_list_id: ``AWS::EC2::SecurityGroupIngress.SourcePrefixListId``.
        :param source_security_group_id: ``AWS::EC2::SecurityGroupIngress.SourceSecurityGroupId``.
        :param source_security_group_name: ``AWS::EC2::SecurityGroupIngress.SourceSecurityGroupName``.
        :param source_security_group_owner_id: ``AWS::EC2::SecurityGroupIngress.SourceSecurityGroupOwnerId``.
        :param to_port: ``AWS::EC2::SecurityGroupIngress.ToPort``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "ip_protocol": ip_protocol,
        }
        if cidr_ip is not None:
            self._values["cidr_ip"] = cidr_ip
        if cidr_ipv6 is not None:
            self._values["cidr_ipv6"] = cidr_ipv6
        if description is not None:
            self._values["description"] = description
        if from_port is not None:
            self._values["from_port"] = from_port
        if group_id is not None:
            self._values["group_id"] = group_id
        if group_name is not None:
            self._values["group_name"] = group_name
        if source_prefix_list_id is not None:
            self._values["source_prefix_list_id"] = source_prefix_list_id
        if source_security_group_id is not None:
            self._values["source_security_group_id"] = source_security_group_id
        if source_security_group_name is not None:
            self._values["source_security_group_name"] = source_security_group_name
        if source_security_group_owner_id is not None:
            self._values["source_security_group_owner_id"] = source_security_group_owner_id
        if to_port is not None:
            self._values["to_port"] = to_port

    @builtins.property
    def ip_protocol(self) -> builtins.str:
        '''``AWS::EC2::SecurityGroupIngress.IpProtocol``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-ipprotocol
        '''
        result = self._values.get("ip_protocol")
        assert result is not None, "Required property 'ip_protocol' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def cidr_ip(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.CidrIp``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-cidrip
        '''
        result = self._values.get("cidr_ip")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def cidr_ipv6(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.CidrIpv6``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-cidripv6
        '''
        result = self._values.get("cidr_ipv6")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.Description``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-description
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def from_port(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::SecurityGroupIngress.FromPort``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-fromport
        '''
        result = self._values.get("from_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def group_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.GroupId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-groupid
        '''
        result = self._values.get("group_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def group_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.GroupName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-groupname
        '''
        result = self._values.get("group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_prefix_list_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.SourcePrefixListId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-securitygroupingress-sourceprefixlistid
        '''
        result = self._values.get("source_prefix_list_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_security_group_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.SourceSecurityGroupId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-sourcesecuritygroupid
        '''
        result = self._values.get("source_security_group_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_security_group_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.SourceSecurityGroupName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-sourcesecuritygroupname
        '''
        result = self._values.get("source_security_group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def source_security_group_owner_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroupIngress.SourceSecurityGroupOwnerId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-sourcesecuritygroupownerid
        '''
        result = self._values.get("source_security_group_owner_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def to_port(self) -> typing.Optional[jsii.Number]:
        '''``AWS::EC2::SecurityGroupIngress.ToPort``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group-ingress.html#cfn-ec2-security-group-ingress-toport
        '''
        result = self._values.get("to_port")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnSecurityGroupIngressProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@aws-cdk/aws-ec2.CfnSecurityGroupProps",
    jsii_struct_bases=[],
    name_mapping={
        "group_description": "groupDescription",
        "group_name": "groupName",
        "security_group_egress": "securityGroupEgress",
        "security_group_ingress": "securityGroupIngress",
        "tags": "tags",
        "vpc_id": "vpcId",
    },
)
class CfnSecurityGroupProps:
    def __init__(
        self,
        *,
        group_description: builtins.str,
        group_name: typing.Optional[builtins.str] = None,
        security_group_egress: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnSecurityGroup.EgressProperty]]]] = None,
        security_group_ingress: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, CfnSecurityGroup.IngressProperty]]]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.core.CfnTag]] = None,
        vpc_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for defining a ``AWS::EC2::SecurityGroup``.

        :param group_description: ``AWS::EC2::SecurityGroup.GroupDescription``.
        :param group_name: ``AWS::EC2::SecurityGroup.GroupName``.
        :param security_group_egress: ``AWS::EC2::SecurityGroup.SecurityGroupEgress``.
        :param security_group_ingress: ``AWS::EC2::SecurityGroup.SecurityGroupIngress``.
        :param tags: ``AWS::EC2::SecurityGroup.Tags``.
        :param vpc_id: ``AWS::EC2::SecurityGroup.VpcId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "group_description": group_description,
        }
        if group_name is not None:
            self._values["group_name"] = group_name
        if security_group_egress is not None:
            self._values["security_group_egress"] = security_group_egress
        if security_group_ingress is not None:
            self._values["security_group_ingress"] = security_group_ingress
        if tags is not None:
            self._values["tags"] = tags
        if vpc_id is not None:
            self._values["vpc_id"] = vpc_id

    @builtins.property
    def group_description(self) -> builtins.str:
        '''``AWS::EC2::SecurityGroup.GroupDescription``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html#cfn-ec2-securitygroup-groupdescription
        '''
        result = self._values.get("group_description")
        assert result is not None, "Required property 'group_description' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def group_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroup.GroupName``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html#cfn-ec2-securitygroup-groupname
        '''
        result = self._values.get("group_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def security_group_egress(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnSecurityGroup.EgressProperty]]]]:
        '''``AWS::EC2::SecurityGroup.SecurityGroupEgress``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html#cfn-ec2-securitygroup-securitygroupegress
        '''
        result = self._values.get("security_group_egress")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnSecurityGroup.EgressProperty]]]], result)

    @builtins.property
    def security_group_ingress(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnSecurityGroup.IngressProperty]]]]:
        '''``AWS::EC2::SecurityGroup.SecurityGroupIngress``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html#cfn-ec2-securitygroup-securitygroupingress
        '''
        result = self._values.get("security_group_ingress")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, CfnSecurityGroup.IngressProperty]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.core.CfnTag]]:
        '''``AWS::EC2::SecurityGroup.Tags``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html#cfn-ec2-securitygroup-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.core.CfnTag]], result)

    @builtins.property
    def vpc_id(self) -> typing.Optional[builtins.str]:
        '''``AWS::EC2::SecurityGroup.VpcId``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-security-group.html#cfn-ec2-securitygroup-vpcid
        '''
        result = self._values.get("vpc_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CfnSecurityGroupProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(aws_cdk.core.IInspectable)
class CfnSpotFleet(
    aws_cdk.core.CfnResource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@aws-cdk/aws-ec2.CfnSpotFleet",
):
    '''A CloudFormation ``AWS::EC2::SpotFleet``.

    :cloudformationResource: AWS::EC2::SpotFleet
    :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-spotfleet.html
    '''

    def __init__(
        self,
        scope: aws_cdk.core.Construct,
        id: builtins.str,
        *,
        spot_fleet_request_config_data: typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.SpotFleetRequestConfigDataProperty"],
    ) -> None:
        '''Create a new ``AWS::EC2::SpotFleet``.

        :param scope: - scope in which this resource is defined.
        :param id: - scoped id of the resource.
        :param spot_fleet_request_config_data: ``AWS::EC2::SpotFleet.SpotFleetRequestConfigData``.
        '''
        props = CfnSpotFleetProps(
            spot_fleet_request_config_data=spot_fleet_request_config_data
        )

        jsii.create(CfnSpotFleet, self, [scope, id, props])

    @jsii.member(jsii_name="inspect")
    def inspect(self, inspector: aws_cdk.core.TreeInspector) -> None:
        '''Examines the CloudFormation resource and discloses attributes.

        :param inspector: - tree inspector to collect and process attributes.
        '''
        return typing.cast(None, jsii.invoke(self, "inspect", [inspector]))

    @jsii.member(jsii_name="renderProperties")
    def _render_properties(
        self,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param props: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "renderProperties", [props]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="CFN_RESOURCE_TYPE_NAME")
    def CFN_RESOURCE_TYPE_NAME(cls) -> builtins.str:
        '''The CloudFormation resource type name for this resource class.'''
        return typing.cast(builtins.str, jsii.sget(cls, "CFN_RESOURCE_TYPE_NAME"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="attrId")
    def attr_id(self) -> builtins.str:
        '''
        :cloudformationAttribute: Id
        '''
        return typing.cast(builtins.str, jsii.get(self, "attrId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cfnProperties")
    def _cfn_properties(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "cfnProperties"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="spotFleetRequestConfigData")
    def spot_fleet_request_config_data(
        self,
    ) -> typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.SpotFleetRequestConfigDataProperty"]:
        '''``AWS::EC2::SpotFleet.SpotFleetRequestConfigData``.

        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-spotfleet.html#cfn-ec2-spotfleet-spotfleetrequestconfigdata
        '''
        return typing.cast(typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.SpotFleetRequestConfigDataProperty"], jsii.get(self, "spotFleetRequestConfigData"))

    @spot_fleet_request_config_data.setter
    def spot_fleet_request_config_data(
        self,
        value: typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.SpotFleetRequestConfigDataProperty"],
    ) -> None:
        jsii.set(self, "spotFleetRequestConfigData", value)

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnSpotFleet.BlockDeviceMappingProperty",
        jsii_struct_bases=[],
        name_mapping={
            "device_name": "deviceName",
            "ebs": "ebs",
            "no_device": "noDevice",
            "virtual_name": "virtualName",
        },
    )
    class BlockDeviceMappingProperty:
        def __init__(
            self,
            *,
            device_name: builtins.str,
            ebs: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.EbsBlockDeviceProperty"]] = None,
            no_device: typing.Optional[builtins.str] = None,
            virtual_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param device_name: ``CfnSpotFleet.BlockDeviceMappingProperty.DeviceName``.
            :param ebs: ``CfnSpotFleet.BlockDeviceMappingProperty.Ebs``.
            :param no_device: ``CfnSpotFleet.BlockDeviceMappingProperty.NoDevice``.
            :param virtual_name: ``CfnSpotFleet.BlockDeviceMappingProperty.VirtualName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-blockdevicemapping.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "device_name": device_name,
            }
            if ebs is not None:
                self._values["ebs"] = ebs
            if no_device is not None:
                self._values["no_device"] = no_device
            if virtual_name is not None:
                self._values["virtual_name"] = virtual_name

        @builtins.property
        def device_name(self) -> builtins.str:
            '''``CfnSpotFleet.BlockDeviceMappingProperty.DeviceName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-blockdevicemapping.html#cfn-ec2-spotfleet-blockdevicemapping-devicename
            '''
            result = self._values.get("device_name")
            assert result is not None, "Required property 'device_name' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def ebs(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.EbsBlockDeviceProperty"]]:
            '''``CfnSpotFleet.BlockDeviceMappingProperty.Ebs``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-blockdevicemapping.html#cfn-ec2-spotfleet-blockdevicemapping-ebs
            '''
            result = self._values.get("ebs")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.EbsBlockDeviceProperty"]], result)

        @builtins.property
        def no_device(self) -> typing.Optional[builtins.str]:
            '''``CfnSpotFleet.BlockDeviceMappingProperty.NoDevice``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-blockdevicemapping.html#cfn-ec2-spotfleet-blockdevicemapping-nodevice
            '''
            result = self._values.get("no_device")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def virtual_name(self) -> typing.Optional[builtins.str]:
            '''``CfnSpotFleet.BlockDeviceMappingProperty.VirtualName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-blockdevicemapping.html#cfn-ec2-spotfleet-blockdevicemapping-virtualname
            '''
            result = self._values.get("virtual_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "BlockDeviceMappingProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnSpotFleet.ClassicLoadBalancerProperty",
        jsii_struct_bases=[],
        name_mapping={"name": "name"},
    )
    class ClassicLoadBalancerProperty:
        def __init__(self, *, name: builtins.str) -> None:
            '''
            :param name: ``CfnSpotFleet.ClassicLoadBalancerProperty.Name``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-classicloadbalancer.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "name": name,
            }

        @builtins.property
        def name(self) -> builtins.str:
            '''``CfnSpotFleet.ClassicLoadBalancerProperty.Name``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-classicloadbalancer.html#cfn-ec2-spotfleet-classicloadbalancer-name
            '''
            result = self._values.get("name")
            assert result is not None, "Required property 'name' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClassicLoadBalancerProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnSpotFleet.ClassicLoadBalancersConfigProperty",
        jsii_struct_bases=[],
        name_mapping={"classic_load_balancers": "classicLoadBalancers"},
    )
    class ClassicLoadBalancersConfigProperty:
        def __init__(
            self,
            *,
            classic_load_balancers: typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.ClassicLoadBalancerProperty"]]],
        ) -> None:
            '''
            :param classic_load_balancers: ``CfnSpotFleet.ClassicLoadBalancersConfigProperty.ClassicLoadBalancers``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-classicloadbalancersconfig.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "classic_load_balancers": classic_load_balancers,
            }

        @builtins.property
        def classic_load_balancers(
            self,
        ) -> typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.ClassicLoadBalancerProperty"]]]:
            '''``CfnSpotFleet.ClassicLoadBalancersConfigProperty.ClassicLoadBalancers``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-classicloadbalancersconfig.html#cfn-ec2-spotfleet-classicloadbalancersconfig-classicloadbalancers
            '''
            result = self._values.get("classic_load_balancers")
            assert result is not None, "Required property 'classic_load_balancers' is missing"
            return typing.cast(typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.ClassicLoadBalancerProperty"]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "ClassicLoadBalancersConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnSpotFleet.EbsBlockDeviceProperty",
        jsii_struct_bases=[],
        name_mapping={
            "delete_on_termination": "deleteOnTermination",
            "encrypted": "encrypted",
            "iops": "iops",
            "snapshot_id": "snapshotId",
            "volume_size": "volumeSize",
            "volume_type": "volumeType",
        },
    )
    class EbsBlockDeviceProperty:
        def __init__(
            self,
            *,
            delete_on_termination: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            encrypted: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            iops: typing.Optional[jsii.Number] = None,
            snapshot_id: typing.Optional[builtins.str] = None,
            volume_size: typing.Optional[jsii.Number] = None,
            volume_type: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param delete_on_termination: ``CfnSpotFleet.EbsBlockDeviceProperty.DeleteOnTermination``.
            :param encrypted: ``CfnSpotFleet.EbsBlockDeviceProperty.Encrypted``.
            :param iops: ``CfnSpotFleet.EbsBlockDeviceProperty.Iops``.
            :param snapshot_id: ``CfnSpotFleet.EbsBlockDeviceProperty.SnapshotId``.
            :param volume_size: ``CfnSpotFleet.EbsBlockDeviceProperty.VolumeSize``.
            :param volume_type: ``CfnSpotFleet.EbsBlockDeviceProperty.VolumeType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-ebsblockdevice.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if delete_on_termination is not None:
                self._values["delete_on_termination"] = delete_on_termination
            if encrypted is not None:
                self._values["encrypted"] = encrypted
            if iops is not None:
                self._values["iops"] = iops
            if snapshot_id is not None:
                self._values["snapshot_id"] = snapshot_id
            if volume_size is not None:
                self._values["volume_size"] = volume_size
            if volume_type is not None:
                self._values["volume_type"] = volume_type

        @builtins.property
        def delete_on_termination(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnSpotFleet.EbsBlockDeviceProperty.DeleteOnTermination``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-ebsblockdevice.html#cfn-ec2-spotfleet-ebsblockdevice-deleteontermination
            '''
            result = self._values.get("delete_on_termination")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def encrypted(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnSpotFleet.EbsBlockDeviceProperty.Encrypted``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-ebsblockdevice.html#cfn-ec2-spotfleet-ebsblockdevice-encrypted
            '''
            result = self._values.get("encrypted")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def iops(self) -> typing.Optional[jsii.Number]:
            '''``CfnSpotFleet.EbsBlockDeviceProperty.Iops``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-ebsblockdevice.html#cfn-ec2-spotfleet-ebsblockdevice-iops
            '''
            result = self._values.get("iops")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def snapshot_id(self) -> typing.Optional[builtins.str]:
            '''``CfnSpotFleet.EbsBlockDeviceProperty.SnapshotId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-ebsblockdevice.html#cfn-ec2-spotfleet-ebsblockdevice-snapshotid
            '''
            result = self._values.get("snapshot_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def volume_size(self) -> typing.Optional[jsii.Number]:
            '''``CfnSpotFleet.EbsBlockDeviceProperty.VolumeSize``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-ebsblockdevice.html#cfn-ec2-spotfleet-ebsblockdevice-volumesize
            '''
            result = self._values.get("volume_size")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def volume_type(self) -> typing.Optional[builtins.str]:
            '''``CfnSpotFleet.EbsBlockDeviceProperty.VolumeType``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-ebsblockdevice.html#cfn-ec2-spotfleet-ebsblockdevice-volumetype
            '''
            result = self._values.get("volume_type")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "EbsBlockDeviceProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnSpotFleet.FleetLaunchTemplateSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "version": "version",
            "launch_template_id": "launchTemplateId",
            "launch_template_name": "launchTemplateName",
        },
    )
    class FleetLaunchTemplateSpecificationProperty:
        def __init__(
            self,
            *,
            version: builtins.str,
            launch_template_id: typing.Optional[builtins.str] = None,
            launch_template_name: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param version: ``CfnSpotFleet.FleetLaunchTemplateSpecificationProperty.Version``.
            :param launch_template_id: ``CfnSpotFleet.FleetLaunchTemplateSpecificationProperty.LaunchTemplateId``.
            :param launch_template_name: ``CfnSpotFleet.FleetLaunchTemplateSpecificationProperty.LaunchTemplateName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-fleetlaunchtemplatespecification.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "version": version,
            }
            if launch_template_id is not None:
                self._values["launch_template_id"] = launch_template_id
            if launch_template_name is not None:
                self._values["launch_template_name"] = launch_template_name

        @builtins.property
        def version(self) -> builtins.str:
            '''``CfnSpotFleet.FleetLaunchTemplateSpecificationProperty.Version``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-fleetlaunchtemplatespecification.html#cfn-ec2-spotfleet-fleetlaunchtemplatespecification-version
            '''
            result = self._values.get("version")
            assert result is not None, "Required property 'version' is missing"
            return typing.cast(builtins.str, result)

        @builtins.property
        def launch_template_id(self) -> typing.Optional[builtins.str]:
            '''``CfnSpotFleet.FleetLaunchTemplateSpecificationProperty.LaunchTemplateId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-fleetlaunchtemplatespecification.html#cfn-ec2-spotfleet-fleetlaunchtemplatespecification-launchtemplateid
            '''
            result = self._values.get("launch_template_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def launch_template_name(self) -> typing.Optional[builtins.str]:
            '''``CfnSpotFleet.FleetLaunchTemplateSpecificationProperty.LaunchTemplateName``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-fleetlaunchtemplatespecification.html#cfn-ec2-spotfleet-fleetlaunchtemplatespecification-launchtemplatename
            '''
            result = self._values.get("launch_template_name")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "FleetLaunchTemplateSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnSpotFleet.GroupIdentifierProperty",
        jsii_struct_bases=[],
        name_mapping={"group_id": "groupId"},
    )
    class GroupIdentifierProperty:
        def __init__(self, *, group_id: builtins.str) -> None:
            '''
            :param group_id: ``CfnSpotFleet.GroupIdentifierProperty.GroupId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-groupidentifier.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "group_id": group_id,
            }

        @builtins.property
        def group_id(self) -> builtins.str:
            '''``CfnSpotFleet.GroupIdentifierProperty.GroupId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-groupidentifier.html#cfn-ec2-spotfleet-groupidentifier-groupid
            '''
            result = self._values.get("group_id")
            assert result is not None, "Required property 'group_id' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "GroupIdentifierProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnSpotFleet.IamInstanceProfileSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={"arn": "arn"},
    )
    class IamInstanceProfileSpecificationProperty:
        def __init__(self, *, arn: typing.Optional[builtins.str] = None) -> None:
            '''
            :param arn: ``CfnSpotFleet.IamInstanceProfileSpecificationProperty.Arn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-iaminstanceprofilespecification.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if arn is not None:
                self._values["arn"] = arn

        @builtins.property
        def arn(self) -> typing.Optional[builtins.str]:
            '''``CfnSpotFleet.IamInstanceProfileSpecificationProperty.Arn``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-iaminstanceprofilespecification.html#cfn-ec2-spotfleet-iaminstanceprofilespecification-arn
            '''
            result = self._values.get("arn")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "IamInstanceProfileSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnSpotFleet.InstanceIpv6AddressProperty",
        jsii_struct_bases=[],
        name_mapping={"ipv6_address": "ipv6Address"},
    )
    class InstanceIpv6AddressProperty:
        def __init__(self, *, ipv6_address: builtins.str) -> None:
            '''
            :param ipv6_address: ``CfnSpotFleet.InstanceIpv6AddressProperty.Ipv6Address``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-instanceipv6address.html
            '''
            self._values: typing.Dict[str, typing.Any] = {
                "ipv6_address": ipv6_address,
            }

        @builtins.property
        def ipv6_address(self) -> builtins.str:
            '''``CfnSpotFleet.InstanceIpv6AddressProperty.Ipv6Address``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-instanceipv6address.html#cfn-ec2-spotfleet-instanceipv6address-ipv6address
            '''
            result = self._values.get("ipv6_address")
            assert result is not None, "Required property 'ipv6_address' is missing"
            return typing.cast(builtins.str, result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceIpv6AddressProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty",
        jsii_struct_bases=[],
        name_mapping={
            "associate_public_ip_address": "associatePublicIpAddress",
            "delete_on_termination": "deleteOnTermination",
            "description": "description",
            "device_index": "deviceIndex",
            "groups": "groups",
            "ipv6_address_count": "ipv6AddressCount",
            "ipv6_addresses": "ipv6Addresses",
            "network_interface_id": "networkInterfaceId",
            "private_ip_addresses": "privateIpAddresses",
            "secondary_private_ip_address_count": "secondaryPrivateIpAddressCount",
            "subnet_id": "subnetId",
        },
    )
    class InstanceNetworkInterfaceSpecificationProperty:
        def __init__(
            self,
            *,
            associate_public_ip_address: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            delete_on_termination: typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]] = None,
            description: typing.Optional[builtins.str] = None,
            device_index: typing.Optional[jsii.Number] = None,
            groups: typing.Optional[typing.Sequence[builtins.str]] = None,
            ipv6_address_count: typing.Optional[jsii.Number] = None,
            ipv6_addresses: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.InstanceIpv6AddressProperty"]]]] = None,
            network_interface_id: typing.Optional[builtins.str] = None,
            private_ip_addresses: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.PrivateIpAddressSpecificationProperty"]]]] = None,
            secondary_private_ip_address_count: typing.Optional[jsii.Number] = None,
            subnet_id: typing.Optional[builtins.str] = None,
        ) -> None:
            '''
            :param associate_public_ip_address: ``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.AssociatePublicIpAddress``.
            :param delete_on_termination: ``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.DeleteOnTermination``.
            :param description: ``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.Description``.
            :param device_index: ``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.DeviceIndex``.
            :param groups: ``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.Groups``.
            :param ipv6_address_count: ``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.Ipv6AddressCount``.
            :param ipv6_addresses: ``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.Ipv6Addresses``.
            :param network_interface_id: ``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.NetworkInterfaceId``.
            :param private_ip_addresses: ``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.PrivateIpAddresses``.
            :param secondary_private_ip_address_count: ``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.SecondaryPrivateIpAddressCount``.
            :param subnet_id: ``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.SubnetId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-instancenetworkinterfacespecification.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if associate_public_ip_address is not None:
                self._values["associate_public_ip_address"] = associate_public_ip_address
            if delete_on_termination is not None:
                self._values["delete_on_termination"] = delete_on_termination
            if description is not None:
                self._values["description"] = description
            if device_index is not None:
                self._values["device_index"] = device_index
            if groups is not None:
                self._values["groups"] = groups
            if ipv6_address_count is not None:
                self._values["ipv6_address_count"] = ipv6_address_count
            if ipv6_addresses is not None:
                self._values["ipv6_addresses"] = ipv6_addresses
            if network_interface_id is not None:
                self._values["network_interface_id"] = network_interface_id
            if private_ip_addresses is not None:
                self._values["private_ip_addresses"] = private_ip_addresses
            if secondary_private_ip_address_count is not None:
                self._values["secondary_private_ip_address_count"] = secondary_private_ip_address_count
            if subnet_id is not None:
                self._values["subnet_id"] = subnet_id

        @builtins.property
        def associate_public_ip_address(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.AssociatePublicIpAddress``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-instancenetworkinterfacespecification.html#cfn-ec2-spotfleet-instancenetworkinterfacespecification-associatepublicipaddress
            '''
            result = self._values.get("associate_public_ip_address")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def delete_on_termination(
            self,
        ) -> typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]]:
            '''``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.DeleteOnTermination``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-instancenetworkinterfacespecification.html#cfn-ec2-spotfleet-instancenetworkinterfacespecification-deleteontermination
            '''
            result = self._values.get("delete_on_termination")
            return typing.cast(typing.Optional[typing.Union[builtins.bool, aws_cdk.core.IResolvable]], result)

        @builtins.property
        def description(self) -> typing.Optional[builtins.str]:
            '''``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.Description``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-instancenetworkinterfacespecification.html#cfn-ec2-spotfleet-instancenetworkinterfacespecification-description
            '''
            result = self._values.get("description")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def device_index(self) -> typing.Optional[jsii.Number]:
            '''``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.DeviceIndex``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-instancenetworkinterfacespecification.html#cfn-ec2-spotfleet-instancenetworkinterfacespecification-deviceindex
            '''
            result = self._values.get("device_index")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def groups(self) -> typing.Optional[typing.List[builtins.str]]:
            '''``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.Groups``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-instancenetworkinterfacespecification.html#cfn-ec2-spotfleet-instancenetworkinterfacespecification-groups
            '''
            result = self._values.get("groups")
            return typing.cast(typing.Optional[typing.List[builtins.str]], result)

        @builtins.property
        def ipv6_address_count(self) -> typing.Optional[jsii.Number]:
            '''``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.Ipv6AddressCount``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-instancenetworkinterfacespecification.html#cfn-ec2-spotfleet-instancenetworkinterfacespecification-ipv6addresscount
            '''
            result = self._values.get("ipv6_address_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def ipv6_addresses(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.InstanceIpv6AddressProperty"]]]]:
            '''``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.Ipv6Addresses``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-instancenetworkinterfacespecification.html#cfn-ec2-spotfleet-instancenetworkinterfacespecification-ipv6addresses
            '''
            result = self._values.get("ipv6_addresses")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.InstanceIpv6AddressProperty"]]]], result)

        @builtins.property
        def network_interface_id(self) -> typing.Optional[builtins.str]:
            '''``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.NetworkInterfaceId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-instancenetworkinterfacespecification.html#cfn-ec2-spotfleet-instancenetworkinterfacespecification-networkinterfaceid
            '''
            result = self._values.get("network_interface_id")
            return typing.cast(typing.Optional[builtins.str], result)

        @builtins.property
        def private_ip_addresses(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.PrivateIpAddressSpecificationProperty"]]]]:
            '''``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.PrivateIpAddresses``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-instancenetworkinterfacespecification.html#cfn-ec2-spotfleet-instancenetworkinterfacespecification-privateipaddresses
            '''
            result = self._values.get("private_ip_addresses")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.PrivateIpAddressSpecificationProperty"]]]], result)

        @builtins.property
        def secondary_private_ip_address_count(self) -> typing.Optional[jsii.Number]:
            '''``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.SecondaryPrivateIpAddressCount``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-instancenetworkinterfacespecification.html#cfn-ec2-spotfleet-instancenetworkinterfacespecification-secondaryprivateipaddresscount
            '''
            result = self._values.get("secondary_private_ip_address_count")
            return typing.cast(typing.Optional[jsii.Number], result)

        @builtins.property
        def subnet_id(self) -> typing.Optional[builtins.str]:
            '''``CfnSpotFleet.InstanceNetworkInterfaceSpecificationProperty.SubnetId``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-instancenetworkinterfacespecification.html#cfn-ec2-spotfleet-instancenetworkinterfacespecification-subnetid
            '''
            result = self._values.get("subnet_id")
            return typing.cast(typing.Optional[builtins.str], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "InstanceNetworkInterfaceSpecificationProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnSpotFleet.LaunchTemplateConfigProperty",
        jsii_struct_bases=[],
        name_mapping={
            "launch_template_specification": "launchTemplateSpecification",
            "overrides": "overrides",
        },
    )
    class LaunchTemplateConfigProperty:
        def __init__(
            self,
            *,
            launch_template_specification: typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.FleetLaunchTemplateSpecificationProperty"]] = None,
            overrides: typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.Sequence[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.LaunchTemplateOverridesProperty"]]]] = None,
        ) -> None:
            '''
            :param launch_template_specification: ``CfnSpotFleet.LaunchTemplateConfigProperty.LaunchTemplateSpecification``.
            :param overrides: ``CfnSpotFleet.LaunchTemplateConfigProperty.Overrides``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-launchtemplateconfig.html
            '''
            self._values: typing.Dict[str, typing.Any] = {}
            if launch_template_specification is not None:
                self._values["launch_template_specification"] = launch_template_specification
            if overrides is not None:
                self._values["overrides"] = overrides

        @builtins.property
        def launch_template_specification(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.FleetLaunchTemplateSpecificationProperty"]]:
            '''``CfnSpotFleet.LaunchTemplateConfigProperty.LaunchTemplateSpecification``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-launchtemplateconfig.html#cfn-ec2-spotfleet-launchtemplateconfig-launchtemplatespecification
            '''
            result = self._values.get("launch_template_specification")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.FleetLaunchTemplateSpecificationProperty"]], result)

        @builtins.property
        def overrides(
            self,
        ) -> typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.LaunchTemplateOverridesProperty"]]]]:
            '''``CfnSpotFleet.LaunchTemplateConfigProperty.Overrides``.

            :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-spotfleet-launchtemplateconfig.html#cfn-ec2-spotfleet-launchtemplateconfig-overrides
            '''
            result = self._values.get("overrides")
            return typing.cast(typing.Optional[typing.Union[aws_cdk.core.IResolvable, typing.List[typing.Union[aws_cdk.core.IResolvable, "CfnSpotFleet.LaunchTemplateOverridesProperty"]]]], result)

        def __eq__(self, rhs: typing.Any) -> builtins.bool:
            return isinstance(rhs, self.__class__) and rhs._values == self._values

        def __ne__(self, rhs: typing.Any) -> builtins.bool:
            return not (rhs == self)

        def __repr__(self) -> str:
            return "LaunchTemplateConfigProperty(%s)" % ", ".join(
                k + "=" + repr(v) for k, v in self._values.items()
            )

    @jsii.data_type(
        jsii_type="@aws-cdk/aws-ec2.CfnSpotFleet.LaunchTemplateOverridesProperty",
        jsii_struct_bases=[],
        name_mapping={
            "availability_zone": "availabilityZone",
            "instance_type": "instanceType",
            "spot_price": "spotPrice",
            "subnet_id": "subnetId",
            "weighted_capacity": "weightedCapacity",
        },
    )
    class LaunchTemplateOverridesProperty:
        def __init__(
            self,
            *,
     