"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClientVpnEndpoint = exports.ClientVpnUserBasedAuthentication = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const logs = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const client_vpn_authorization_rule_1 = require("./client-vpn-authorization-rule");
const client_vpn_route_1 = require("./client-vpn-route");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const network_util_1 = require("./network-util");
const security_group_1 = require("./security-group");
/**
 * User-based authentication for a client VPN endpoint.
 *
 * @stability stable
 */
class ClientVpnUserBasedAuthentication {
    /**
     * Active Directory authentication.
     *
     * @stability stable
     */
    static activeDirectory(directoryId) {
        return new ActiveDirectoryAuthentication(directoryId);
    }
    /**
     * Federated authentication.
     *
     * @stability stable
     */
    static federated(samlProvider, selfServiceSamlProvider) {
        return new FederatedAuthentication(samlProvider, selfServiceSamlProvider);
    }
}
exports.ClientVpnUserBasedAuthentication = ClientVpnUserBasedAuthentication;
_a = JSII_RTTI_SYMBOL_1;
ClientVpnUserBasedAuthentication[_a] = { fqn: "@aws-cdk/aws-ec2.ClientVpnUserBasedAuthentication", version: "1.108.1" };
/**
 * Active Directory authentication
 */
class ActiveDirectoryAuthentication extends ClientVpnUserBasedAuthentication {
    constructor(directoryId) {
        super();
        this.directoryId = directoryId;
    }
    render() {
        return {
            type: 'directory-service-authentication',
            activeDirectory: { directoryId: this.directoryId },
        };
    }
}
/**
 * Federated authentication
 */
class FederatedAuthentication extends ClientVpnUserBasedAuthentication {
    constructor(samlProvider, selfServiceSamlProvider) {
        super();
        this.samlProvider = samlProvider;
        this.selfServiceSamlProvider = selfServiceSamlProvider;
    }
    render() {
        var _c;
        return {
            type: 'federated-authentication',
            federatedAuthentication: {
                samlProviderArn: this.samlProvider.samlProviderArn,
                selfServiceSamlProviderArn: (_c = this.selfServiceSamlProvider) === null || _c === void 0 ? void 0 : _c.samlProviderArn,
            },
        };
    }
}
/**
 * A client VPN connnection.
 *
 * @stability stable
 */
class ClientVpnEndpoint extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h;
        super(scope, id);
        this._targetNetworksAssociated = new core_1.ConcreteDependable();
        if (!core_1.Token.isUnresolved(props.vpc.vpcCidrBlock)) {
            const clientCidr = new network_util_1.CidrBlock(props.cidr);
            const vpcCidr = new network_util_1.CidrBlock(props.vpc.vpcCidrBlock);
            if (vpcCidr.containsCidr(clientCidr)) {
                throw new Error('The client CIDR cannot overlap with the local CIDR of the VPC');
            }
        }
        if (props.dnsServers && props.dnsServers.length > 2) {
            throw new Error('A client VPN endpoint can have up to two DNS servers');
        }
        if (props.logging == false && (props.logGroup || props.logStream)) {
            throw new Error('Cannot specify `logGroup` or `logStream` when logging is disabled');
        }
        if (props.clientConnectionHandler
            && !core_1.Token.isUnresolved(props.clientConnectionHandler.functionName)
            && !props.clientConnectionHandler.functionName.startsWith('AWSClientVPN-')) {
            throw new Error('The name of the Lambda function must begin with the `AWSClientVPN-` prefix');
        }
        const logging = (_c = props.logging) !== null && _c !== void 0 ? _c : true;
        const logGroup = logging
            ? (_d = props.logGroup) !== null && _d !== void 0 ? _d : new logs.LogGroup(this, 'LogGroup') : undefined;
        const securityGroups = (_e = props.securityGroups) !== null && _e !== void 0 ? _e : [new security_group_1.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
            })];
        this.connections = new connections_1.Connections({ securityGroups });
        const endpoint = new ec2_generated_1.CfnClientVpnEndpoint(this, 'Resource', {
            authenticationOptions: renderAuthenticationOptions(props.clientCertificateArn, props.userBasedAuthentication),
            clientCidrBlock: props.cidr,
            clientConnectOptions: props.clientConnectionHandler
                ? {
                    enabled: true,
                    lambdaFunctionArn: props.clientConnectionHandler.functionArn,
                }
                : undefined,
            connectionLogOptions: {
                enabled: logging,
                cloudwatchLogGroup: logGroup === null || logGroup === void 0 ? void 0 : logGroup.logGroupName,
                cloudwatchLogStream: (_f = props.logStream) === null || _f === void 0 ? void 0 : _f.logStreamName,
            },
            description: props.description,
            dnsServers: props.dnsServers,
            securityGroupIds: securityGroups.map(s => s.securityGroupId),
            selfServicePortal: booleanToEnabledDisabled(props.selfServicePortal),
            serverCertificateArn: props.serverCertificateArn,
            splitTunnel: props.splitTunnel,
            transportProtocol: props.transportProtocol,
            vpcId: props.vpc.vpcId,
            vpnPort: props.port,
        });
        this.endpointId = endpoint.ref;
        if (props.userBasedAuthentication && ((_g = props.selfServicePortal) !== null && _g !== void 0 ? _g : true)) {
            // Output self-service portal URL
            new core_1.CfnOutput(this, 'SelfServicePortalUrl', {
                value: `https://self-service.clientvpn.amazonaws.com/endpoints/${this.endpointId}`,
            });
        }
        // Associate subnets
        const subnetIds = props.vpc.selectSubnets(props.vpcSubnets).subnetIds;
        if (core_1.Token.isUnresolved(subnetIds)) {
            throw new Error('Cannot associate subnets when VPC are imported from parameters or exports containing lists of subnet IDs.');
        }
        for (const [idx, subnetId] of Object.entries(subnetIds)) {
            this._targetNetworksAssociated.add(new ec2_generated_1.CfnClientVpnTargetNetworkAssociation(this, `Association${idx}`, {
                clientVpnEndpointId: this.endpointId,
                subnetId,
            }));
        }
        this.targetNetworksAssociated = this._targetNetworksAssociated;
        if ((_h = props.authorizeAllUsersToVpcCidr) !== null && _h !== void 0 ? _h : true) {
            this.addAuthorizationRule('AuthorizeAll', {
                cidr: props.vpc.vpcCidrBlock,
            });
        }
    }
    /**
     * Import an existing client VPN endpoint.
     *
     * @stability stable
     */
    static fromEndpointAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.endpointId = attrs.endpointId;
                this.connections = new connections_1.Connections({ securityGroups: attrs.securityGroups });
                this.targetNetworksAssociated = new core_1.ConcreteDependable();
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an authorization rule to this endpoint.
     *
     * @stability stable
     */
    addAuthorizationRule(id, props) {
        return new client_vpn_authorization_rule_1.ClientVpnAuthorizationRule(this, id, {
            ...props,
            clientVpnEndoint: this,
        });
    }
    /**
     * Adds a route to this endpoint.
     *
     * @stability stable
     */
    addRoute(id, props) {
        return new client_vpn_route_1.ClientVpnRoute(this, id, {
            ...props,
            clientVpnEndoint: this,
        });
    }
}
exports.ClientVpnEndpoint = ClientVpnEndpoint;
_b = JSII_RTTI_SYMBOL_1;
ClientVpnEndpoint[_b] = { fqn: "@aws-cdk/aws-ec2.ClientVpnEndpoint", version: "1.108.1" };
function renderAuthenticationOptions(clientCertificateArn, userBasedAuthentication) {
    const authenticationOptions = [];
    if (clientCertificateArn) {
        authenticationOptions.push({
            type: 'certificate-authentication',
            mutualAuthentication: {
                clientRootCertificateChainArn: clientCertificateArn,
            },
        });
    }
    if (userBasedAuthentication) {
        authenticationOptions.push(userBasedAuthentication.render());
    }
    if (authenticationOptions.length === 0) {
        throw new Error('A client VPN endpoint must use at least one authentication option');
    }
    return authenticationOptions;
}
function booleanToEnabledDisabled(val) {
    switch (val) {
        case undefined:
            return undefined;
        case true:
            return 'enabled';
        case false:
            return 'disabled';
    }
}
//# sourceMappingURL=data:application/json;base64,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