# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pika_pydantic']

package_data = \
{'': ['*']}

install_requires = \
['aenum>=3.1.11,<4.0.0', 'pika>=1.2.1,<2.0.0', 'pydantic>=1.9.1,<2.0.0']

setup_kwargs = {
    'name': 'pika-pydantic',
    'version': '0.1.3',
    'description': 'An opinionated Python implementation of the Producer-Consumer Pattern using RabbitMQ on top of the Python pika and pydantic libraries.',
    'long_description': '# Pika-Pydantic\n\nAn opinionated Python implementation of the Producer-Consumer Pattern using RabbitMQ on top of `pika` and `pydantic`.\n\n# Introduction\n\nThis `pika_pydantic` library is a thin wrapper on top of the `pika` and `pydantic` libraries that makes it quick and easy to create **Producer-Consumer** workers that interface with a **RabbitMQ** message queue. For more information of why this library was created, see the Backstory section below in the documentation.\n\nI was inspired in many ways by what Sebastian created with `FastAPI` by building on top of good existing libraries. `pika_pydantic` attempts to follow that method in its own much simpler way for the **asynchronous Producer-Consumer pattern using RabbitMQ**.\n\nIf you are creating a long chain of Producers and Consumers then the `pika_pydantic` library can save quite a lot of boilerplate code and potential errors.\n\n# Installation\n\nTo install the `pika_pydantic` package using pip\n\n    pip install pika-pydantic\n\nOr using poetry\n\n    poetry add pika-pydantic\n\n# Dependencies - requires RabbitMQ\n\nIn addition you need to have a **RabbitMQ** instance up and running that can receive and route the messages.\n\nIf you have some familiarity with Docker, the easiest method is to spin up a docker container running RabbitMQ and use that as your message service. The `docker-compose-rabbit.yml` provides a simple `docker-compose` configuration script for this.\n\nAlternatively, you can install RabbitMQ natively on your development machine or server, or link to a hosted RabbitMQ instance. More details on RabbitMQ installation can be found on [the official RabbitMQ documentation](https://www.rabbitmq.com/#getstarted)\n\n# Quickstart\n\nThis simple example creates a simple message Producer-Consumer that passes around a message object.\n\n## Create the pika connection\n\nFirst we create a pika connection to the RabbitMQ system\n\n```python\nimport pika\n\nparameters = pika.URLParameters("amqp://guest:guest@localhost:5672/")\nconnection = pika.BlockingConnection(parameters)\n```\n\nThis creates a normal `pika` blocking connection.  \nThe [`pika` documentation can be found here](https://pika.readthedocs.io/en/stable/)\n\n## Pika-pydantic Channel vs pika Channel\n\nNow we deviate from the standard `pika` method. Instead of using `connection.channel()` or similar to create a `pika.BlockingChannel` we use the `pika_pydantic.BlockingChannel` object instead. This object also initialises queues and adds various other useful methods on top of the standard `pika.BlockingChannel` object.\n\nBut before we do that we need to define the data validation and the queues that will constrain and validate our Producers and Consumers.\n\n## Defining data models\n\nWe want to pass around a message data object that has a title, and text.\nThis data model is defined using the `pika_pydantic.BaseModel` which is a wrapper around the standard `pydantic.BaseModel`\n\n```python\nimport pika_pydantic\n\nclass MyMessage(pika_pydantic.BaseModel):\n    """A message"""\n    title: str\n    text: str\n```\n\n> `pika_pydantic.BaseModel` objects are pydantic.BaseModel objects with some additional elements for encoding and decoding the objects for RabbitMQ. See the [pydantic documentation](https://pydantic-docs.helpmanual.io/usage/models/) for more details.\n\n## Defining queues\n\nWe also define the single message queue we will use in this example by definding an `pika_pydantic.Queues` enum. The name on the left defines the Enum but also the RabbitMQ queue name. The value on the right defines the data model to use for validation.\n\n```python\nclass MyQueues(pika_pydantic.Queues):\n    MESSAGE = MyMessage\n```\n\nThis object is the master that defines the valid queues and the corresponding data that all Producers and Consumers must use. Add more elements to this enum as you add queues and data models.\n\n> `pika_pydantic.Queues` objects are a Python `enum.Enum` class. The RabbitMQ queue name will be set to the same as the enum name (on the left), and the value on the right is the `pika_pydantic.BaseModel` data model object that all Producers and Consumers on this queue need to use.\n\n## Initialise the Channel\n\nNow we can initialise the channel and we pass it the `pika.connection` and the `pika_pydantic.Queues` enum we just defined.\n\n```python\nchannel = pika_pydantic.BlockingChannel(connection=connection, queues=MyQueues)\n```\n\n> `pika_pydantic.BlockingChannel` is a `pika.BlockingChannel` object with some additional methods attached that allow simpler creation of Consumers (`listen()`) and Producers (`send()`)\n\nThis object declares all the queues, and validates the message data on each queue and does the necessary encoding and decoding of the data for Consumers and Producers.\n\n## Create a Consumer\n\nTo create a new Consumer for this message queue we use the new `channel.listen(queue, callback)` method. This validates the inputs and does the decoding needed for that particular queue. We define a callback as in pika and add the consumer to the channel.\n\n```python\ndef callback(channel, method, frame, data: MyMessage):\n    print(f"Received message with title ({data.title}) and text ({data.text}).")\n\nchannel.listen(queue=MyQueues.MESSAGE, callback=callback, auto_ack=True)\n```\n\n## Create a Producer\n\nTo create a Producer we use the new `channel.send(queue, data)` method. This takes the data object and does all the validation and encoding needed to pass it to the RabbitMQ queue.\n\n```python\nmessage = MyMessage(title="Important", text="Remember to feed the dog")\nchannel.send(queue=MyQueues.MESSAGE, data=message)\n```\n\n## Start it running\n\nAs with standard pika, the channel can start polling so that the defined Consumers start listening for messages on their queue.\n\n```python\nchannel.start_consuming()\n```\n\nOr to not block the thread and process the messages currently in the queue we can use\n\n```python\nconnection.process_data_events(time_limit=None)\n```\n\n# Other examples\n\nThe `examples` folder provides further examples and a suggested project folder structure that reuses the `pika_pydantic` elements across multiple Consumers and Producers.\n\n# The backstory\n\n## Asynchronous messaging\n\nGood code structure generally separates concerns (jobs) between different modules. Microservices takes this one step further and separates jobs into different deployable systems that interact with each other.\n\nThese different systems are interfaced through various APIs, usually called from one system to another in realtime.\n\nBut some jobs are long lasting or resource hungry and this is where we can use asynchronous interfaces between the different systems.\n\nThere is a lot of interest currently in Kafka as a system for managing these asynchronous jobs. But for most projects a simpler message queue such as RabbitMQ will do the job. It provides a way to pass data and a job onto another system, and that other system will pick up the job when it has resources to do so.\n\n## The Producer-Consumer pattern\n\nFor many purposes a system does some work and prepares some data. It then passes this on as a job for the next system element to work on when it has resources available. This is the **Producer-Consumer Pattern**.\n\nIn a bit more detail\n\n- A Producer completes some job, often resulting in some data artifact to be passed to the next stage.\n- The Producer publishes this data to a message queue\n- The next job is a Consumer of this message queue. When it has resources available, it picks up the message and the published data and then does it work.\n- This Consumer may itself also be a Producer publishing it\'s data to a different message queue for the next Consumer in the chain to take forward.\n\n## RabbitMQ and the Python `pika` library\n\nIn the Python world there are good libraries for this, most notably is the `pika` library that interfaces with a RabbitMQ message queue. `pika` is relatively simple and very flexible.\n\nBut for my needs I wanted to use stricter software development principles and the flexibility of `pika` too flexible. Specifically:\n\n- My system has many Consumers and many Publishers. I wanted to be able to define the `pika` boilerplate code to set up the connection, the queues and the channel in one central place for all the different jobs.\n- I also wanted to restrict my Consumers and Publishers to only valid queues, so to do this I wanted to define the valid queues in an Enum to reduce strange bugs.\n- I wanted to ensure that each Producer sending data sends the data in the right format and each Consumer picks up the data and validates to the same format. For this the `pydantic` library is very helpful to constrain the Producer and Consumer data to be passed. This is how the `fastapi` library ensures data being passed around that API is validated and structured correctly. I wanted to use this pattern.\n\n## Contributing\n\nIf you find this useful, consider adding a star and contributing.\n\nCurrently this only uses the `pika.BlockingChannel` implementation.\n\n## Tests\n\nWhen running tests, a RabbitMQ instance needs to be up and running on your machine as the tests do live tests using that RabbitMQ.\n\nIf using docker, you can spin up a RabbitMQ instance for testing using\n\n    docker-compose -f docker-compose-rabbit.yml up\n\nThe environment variable `PIKA_URL` can be overwritten to point to your test RabbitMQ instance.\n\nThen run tests use **pytest**\n\n    pytest\n',
    'author': 'Matt Gosden',
    'author_email': 'mdgosden@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/ttamg/pika-pydantic',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
