"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RunnerImageComponent = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_image_builder_1 = require("./aws-image-builder");
const common_1 = require("../common");
/**
 * Components are used to build runner images. They can run commands in the image, copy files into the image, and run some Docker commands.
 */
class RunnerImageComponent {
    /**
     * Define a custom component that can run commands in the image, copy files into the image, and run some Docker commands.
     *
     * The order of operations is (1) assets (2) commands (3) docker commands.
     *
     * Use this to customize the image for the runner.
     *
     * **WARNING:** Docker commands are not guaranteed to be included before the next component
     */
    static custom(props) {
        return new class extends RunnerImageComponent {
            get name() {
                if (props.name && !props.name.match(/[a-zA-Z0-9\-]/)) {
                    throw new Error(`Invalid component name: ${props.name}. Name must only contain alphanumeric characters and dashes.`);
                }
                return `Custom-${props.name ?? 'Undefined'}`;
            }
            getCommands(_os, _architecture) {
                return props.commands ?? [];
            }
            getAssets(_os, _architecture) {
                return props.assets ?? [];
            }
            getDockerCommands(_os, _architecture) {
                return props.dockerCommands ?? [];
            }
        }();
    }
    /**
     * A component to install the required packages for the runner.
     */
    static requiredPackages() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'RequiredPackages';
            }
            getCommands(os, architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU)) {
                    let archUrl;
                    if (architecture.is(common_1.Architecture.X86_64)) {
                        archUrl = 'amd64';
                    }
                    else if (architecture.is(common_1.Architecture.ARM64)) {
                        archUrl = 'arm64';
                    }
                    else {
                        throw new Error(`Unsupported architecture for required packages: ${architecture.name}`);
                    }
                    return [
                        'apt-get update',
                        'DEBIAN_FRONTEND=noninteractive apt-get upgrade -y',
                        'DEBIAN_FRONTEND=noninteractive apt-get install -y curl sudo jq bash zip unzip iptables software-properties-common ca-certificates',
                        `curl -sfLo /tmp/amazon-cloudwatch-agent.deb https://s3.amazonaws.com/amazoncloudwatch-agent/ubuntu/${archUrl}/latest/amazon-cloudwatch-agent.deb`,
                        'dpkg -i -E /tmp/amazon-cloudwatch-agent.deb',
                        'rm /tmp/amazon-cloudwatch-agent.deb',
                    ];
                }
                else if (os.is(common_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'yum update -y',
                        'yum install -y jq tar gzip bzip2 which binutils zip unzip sudo shadow-utils',
                    ];
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    return [
                        'Start-Process msiexec.exe -Wait -ArgumentList \'/i https://s3.amazonaws.com/amazoncloudwatch-agent/windows/amd64/latest/amazon-cloudwatch-agent.msi /qn\'',
                    ];
                }
                throw new Error(`Unsupported OS for required packages: ${os.name}`);
            }
        };
    }
    /**
     * A component to prepare the required runner user.
     */
    static runnerUser() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'RunnerUser';
            }
            getCommands(os, _architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU)) {
                    return [
                        'addgroup runner',
                        'adduser --system --disabled-password --home /home/runner --ingroup runner runner',
                        'usermod -aG sudo runner',
                        'echo "%sudo   ALL=(ALL:ALL) NOPASSWD: ALL" > /etc/sudoers.d/runner',
                    ];
                }
                else if (os.is(common_1.Os.LINUX_AMAZON_2)) {
                    return [
                        '/usr/sbin/groupadd runner',
                        '/usr/sbin/useradd --system --shell /usr/sbin/nologin --home-dir /home/runner --gid runner runner',
                        'mkdir -p /home/runner',
                        'chown runner /home/runner',
                        'echo "%runner   ALL=(ALL:ALL) NOPASSWD: ALL" > /etc/sudoers.d/runner',
                    ];
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    return [];
                }
                throw new Error(`Unsupported OS for runner user: ${os.name}`);
            }
        };
    }
    /**
     * A component to install the AWS CLI.
     */
    static awsCli() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'AwsCli';
            }
            getCommands(os, architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU) || os.is(common_1.Os.LINUX_AMAZON_2)) {
                    let archUrl;
                    if (architecture.is(common_1.Architecture.X86_64)) {
                        archUrl = 'x86_64';
                    }
                    else if (architecture.is(common_1.Architecture.ARM64)) {
                        archUrl = 'aarch64';
                    }
                    else {
                        throw new Error(`Unsupported architecture for awscli: ${architecture.name}`);
                    }
                    return [
                        `curl -fsSL "https://awscli.amazonaws.com/awscli-exe-linux-${archUrl}.zip" -o awscliv2.zip`,
                        'unzip -q awscliv2.zip',
                        './aws/install',
                        'rm -rf awscliv2.zip aws',
                    ];
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    return [
                        'Start-Process msiexec.exe -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for awscli: ${os.name}/${architecture.name}`);
            }
        }();
    }
    /**
     * A component to install the GitHub CLI.
     */
    static githubCli() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'GithubCli';
            }
            getCommands(os, architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU)) {
                    return [
                        'curl -fsSL https://cli.github.com/packages/githubcli-archive-keyring.gpg | sudo dd of=/usr/share/keyrings/githubcli-archive-keyring.gpg',
                        'echo "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/githubcli-archive-keyring.gpg] ' +
                            '  https://cli.github.com/packages stable main" | sudo tee /etc/apt/sources.list.d/github-cli.list > /dev/null',
                        'apt-get update',
                        'DEBIAN_FRONTEND=noninteractive apt-get install -y gh',
                    ];
                }
                else if (os.is(common_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'curl -fsSSL https://cli.github.com/packages/rpm/gh-cli.repo -o /etc/yum.repos.d/gh-cli.repo',
                        'yum install -y gh',
                    ];
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    return [
                        'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/cli/cli/releases/latest > $Env:TEMP\\latest-gh',
                        '$LatestUrl = Get-Content $Env:TEMP\\latest-gh',
                        '$GH_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                        'Invoke-WebRequest -UseBasicParsing -Uri "https://github.com/cli/cli/releases/download/v${GH_VERSION}/gh_${GH_VERSION}_windows_amd64.msi" -OutFile gh.msi',
                        'Start-Process msiexec.exe -Wait -ArgumentList \'/i gh.msi /qn\'',
                        'del gh.msi',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for github cli: ${os.name}/${architecture.name}`);
            }
        }();
    }
    /**
     * A component to install the GitHub CLI.
     */
    static git() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'Git';
            }
            getCommands(os, architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU)) {
                    return [
                        'add-apt-repository ppa:git-core/ppa',
                        'apt-get update',
                        'DEBIAN_FRONTEND=noninteractive apt-get install -y git',
                    ];
                }
                else if (os.is(common_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'yum install -y git',
                    ];
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    return [
                        'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/git-for-windows/git/releases/latest > $Env:TEMP\\latest-git',
                        '$LatestUrl = Get-Content $Env:TEMP\\latest-git',
                        '$GIT_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                        '$GIT_VERSION_SHORT = ($GIT_VERSION -Split \'.windows.\')[0]',
                        '$GIT_REVISION = ($GIT_VERSION -Split \'.windows.\')[1]',
                        'If ($GIT_REVISION -gt 1) {$GIT_VERSION_SHORT = "$GIT_VERSION_SHORT.$GIT_REVISION"}',
                        'Invoke-WebRequest -UseBasicParsing -Uri https://github.com/git-for-windows/git/releases/download/v${GIT_VERSION}/Git-${GIT_VERSION_SHORT}-64-bit.exe -OutFile git-setup.exe',
                        'Start-Process git-setup.exe -Wait -ArgumentList \'/VERYSILENT\'',
                        'del git-setup.exe',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for git: ${os.name}/${architecture.name}`);
            }
        }();
    }
    /**
     * A component to install the GitHub Actions Runner. This is the actual executable that connects to GitHub to ask for jobs and then execute them.
     *
     * @param runnerVersion The version of the runner to install. Usually you would set this to latest.
     */
    static githubRunner(runnerVersion) {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'GithubRunner';
            }
            getCommands(os, architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU) || os.is(common_1.Os.LINUX_AMAZON_2)) {
                    let versionCommand;
                    if (runnerVersion.is(common_1.RunnerVersion.latest())) {
                        versionCommand = 'RUNNER_VERSION=`curl -w "%{redirect_url}" -fsS https://github.com/actions/runner/releases/latest | grep -oE "[^/v]+$"`';
                    }
                    else {
                        versionCommand = `RUNNER_VERSION='${runnerVersion.version}'`;
                    }
                    let archUrl;
                    if (architecture.is(common_1.Architecture.X86_64)) {
                        archUrl = 'x64';
                    }
                    else if (architecture.is(common_1.Architecture.ARM64)) {
                        archUrl = 'arm64';
                    }
                    else {
                        throw new Error(`Unsupported architecture for GitHub Runner: ${architecture.name}`);
                    }
                    let commands = [
                        versionCommand,
                        `curl -fsSLO "https://github.com/actions/runner/releases/download/v\${RUNNER_VERSION}/actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz"`,
                        `tar -C /home/runner -xzf "actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz"`,
                        `rm actions-runner-linux-${archUrl}-\${RUNNER_VERSION}.tar.gz`,
                        `echo -n ${runnerVersion.version} > /home/runner/RUNNER_VERSION`,
                    ];
                    if (os.is(common_1.Os.LINUX_UBUNTU)) {
                        commands.push('/home/runner/bin/installdependencies.sh');
                    }
                    else if (os.is(common_1.Os.LINUX_AMAZON_2)) {
                        commands.push('yum install -y openssl-libs krb5-libs zlib libicu60');
                    }
                    return commands;
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    let runnerCommands;
                    if (runnerVersion.is(common_1.RunnerVersion.latest())) {
                        runnerCommands = [
                            'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/actions/runner/releases/latest > $Env:TEMP\\latest-gha',
                            '$LatestUrl = Get-Content $Env:TEMP\\latest-gha',
                            '$RUNNER_VERSION = ($LatestUrl -Split \'/\')[-1].substring(1)',
                        ];
                    }
                    else {
                        runnerCommands = [`$RUNNER_VERSION = '${runnerVersion.version}'`];
                    }
                    return runnerCommands.concat([
                        'Invoke-WebRequest -UseBasicParsing -Uri "https://github.com/actions/runner/releases/download/v${RUNNER_VERSION}/actions-runner-win-x64-${RUNNER_VERSION}.zip" -OutFile actions.zip',
                        'Expand-Archive actions.zip -DestinationPath C:\\actions',
                        'del actions.zip',
                        `echo ${runnerVersion.version} | Out-File -Encoding ASCII -NoNewline C:\\actions\\RUNNER_VERSION`,
                    ]);
                }
                throw new Error(`Unknown os/architecture combo for github runner: ${os.name}/${architecture.name}`);
            }
            getDockerCommands(_os, _architecture) {
                return [
                    `ENV RUNNER_VERSION=${runnerVersion.version}`,
                ];
            }
        }();
    }
    /**
     * A component to install Docker. On Windows this installs Docker Desktop.
     */
    static docker() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'Docker';
            }
            getCommands(os, architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU)) {
                    return [
                        'curl -fsSL https://download.docker.com/linux/ubuntu/gpg | sudo gpg --dearmor -o /usr/share/keyrings/docker.gpg',
                        'echo ' +
                            '  "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/docker.gpg] https://download.docker.com/linux/ubuntu ' +
                            '  $(lsb_release -cs) stable" | sudo tee /etc/apt/sources.list.d/docker.list > /dev/null',
                        'apt-get update',
                        'DEBIAN_FRONTEND=noninteractive apt-get install -y docker-ce docker-ce-cli containerd.io docker-compose-plugin',
                        'usermod -aG docker runner',
                        'ln -s /usr/libexec/docker/cli-plugins/docker-compose /usr/bin/docker-compose',
                    ];
                }
                else if (os.is(common_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'yum install -y docker',
                    ];
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    return [
                        'Invoke-WebRequest -UseBasicParsing -Uri https://desktop.docker.com/win/main/amd64/Docker%20Desktop%20Installer.exe -OutFile docker-setup.exe',
                        'Start-Process \'docker-setup.exe\' -Wait -ArgumentList \'/install --quiet --accept-license\'',
                        'del docker-setup.exe',
                        'cmd /c curl -w "%{redirect_url}" -fsS https://github.com/docker/compose/releases/latest > $Env:TEMP\\latest-docker-compose',
                        '$LatestUrl = Get-Content $Env:TEMP\\latest-docker-compose',
                        '$LatestDockerCompose = ($LatestUrl -Split \'/\')[-1]',
                        'Invoke-WebRequest -UseBasicParsing -Uri  "https://github.com/docker/compose/releases/download/${LatestDockerCompose}/docker-compose-Windows-x86_64.exe" -OutFile $Env:ProgramFiles\\Docker\\docker-compose.exe',
                        'copy $Env:ProgramFiles\\Docker\\docker-compose.exe $Env:ProgramFiles\\Docker\\cli-plugins\\docker-compose.exe',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for docker: ${os.name}/${architecture.name}`);
            }
        }();
    }
    /**
     * A component to install Docker-in-Docker.
     */
    static dockerInDocker() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'Docker-in-Docker';
            }
            getCommands(os, architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU) || os.is(common_1.Os.LINUX_AMAZON_2)) {
                    let archUrl;
                    if (architecture.is(common_1.Architecture.X86_64)) {
                        archUrl = 'x86_64';
                    }
                    else if (architecture.is(common_1.Architecture.ARM64)) {
                        archUrl = 'aarch64';
                    }
                    else {
                        throw new Error(`Unsupported architecture for Docker-in-Docker: ${architecture.name}`);
                    }
                    return [
                        'DOCKER_CHANNEL="stable"',
                        'DIND_COMMIT="42b1175eda071c0e9121e1d64345928384a93df1"',
                        'DOCKER_VERSION="20.10.18"',
                        'DOCKER_COMPOSE_VERSION="2.11.0"',
                        `curl -fsSL "https://download.docker.com/linux/static/\${DOCKER_CHANNEL}/${archUrl}/docker-\${DOCKER_VERSION}.tgz" -o docker.tgz`,
                        'tar --strip-components 1 -C /usr/local/bin/ -xzf docker.tgz',
                        'rm docker.tgz',
                        '# set up subuid/subgid so that "--userns-remap=default" works out-of-the box',
                        'addgroup dockremap',
                        'useradd -g dockremap dockremap',
                        'echo \'dockremap:165536:65536\' >> /etc/subuid',
                        'echo \'dockremap:165536:65536\' >> /etc/subgid',
                        'curl -fsSL "https://raw.githubusercontent.com/docker/docker/${DIND_COMMIT}/hack/dind" -o /usr/local/bin/dind',
                        `curl -fsSL https://github.com/docker/compose/releases/download/v\${DOCKER_COMPOSE_VERSION}/docker-compose-linux-${archUrl} -o /usr/local/bin/docker-compose`,
                        'mkdir -p /home/runner/.docker/cli-plugins && ln -s /usr/local/bin/docker-compose /home/runner/.docker/cli-plugins/docker-compose',
                        'chown -R runner /home/runner/.docker',
                        'chmod +x /usr/local/bin/dind /usr/local/bin/docker-compose',
                        'addgroup docker && usermod -aG docker runner',
                    ];
                }
                throw new Error(`Unknown os/architecture combo for Docker-in-Docker: ${os.name}/${architecture.name}`);
            }
        }();
    }
    /**
     * A component to add a trusted certificate authority. This can be used to support GitHub Enterprise Server with self-signed certificate.
     *
     * @param source path to certificate file in PEM format
     * @param name unique certificate name to be used on runner file system
     */
    static extraCertificates(source, name) {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = `Extra-Certificates-${name}`;
            }
            getCommands(os, architecture) {
                if (!name.match(/^[a-zA-Z0-9_-]+$/)) {
                    throw new Error(`Invalid certificate name: ${name}. Name must only contain alphanumeric characters, dashes and underscores.`);
                }
                if (os.is(common_1.Os.LINUX_UBUNTU)) {
                    return [
                        'update-ca-certificates',
                    ];
                }
                else if (os.is(common_1.Os.LINUX_AMAZON_2)) {
                    return [
                        'update-ca-trust',
                    ];
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    return [
                        `Import-Certificate -FilePath C:\\${name}.crt -CertStoreLocation Cert:\\LocalMachine\\Root`,
                        `Remove-Item C:\\${name}.crt`,
                    ];
                }
                throw new Error(`Unknown os/architecture combo for extra certificates: ${os.name}/${architecture.name}`);
            }
            getAssets(os, _architecture) {
                if (os.is(common_1.Os.LINUX_UBUNTU)) {
                    return [
                        { source, target: `/usr/local/share/ca-certificates/${name}.crt` },
                    ];
                }
                else if (os.is(common_1.Os.LINUX_AMAZON_2)) {
                    return [
                        { source, target: `/etc/pki/ca-trust/source/anchors/${name}.crt` },
                    ];
                }
                else if (os.is(common_1.Os.WINDOWS)) {
                    return [
                        { source, target: `C:\\${name}.crt` },
                    ];
                }
                throw new Error(`Unsupported OS for extra certificates: ${os.name}`);
            }
        }();
    }
    /**
     * A component to set up the required Lambda entrypoint for Lambda runners.
     */
    static lambdaEntrypoint() {
        return new class extends RunnerImageComponent {
            constructor() {
                super(...arguments);
                this.name = 'Lambda-Entrypoint';
            }
            getCommands(os, _architecture) {
                if (!os.is(common_1.Os.LINUX_AMAZON_2) && !os.is(common_1.Os.LINUX_UBUNTU)) {
                    throw new Error(`Unsupported OS for Lambda entrypoint: ${os.name}`);
                }
                return [];
            }
            getAssets(_os, _architecture) {
                return [
                    {
                        source: path.join(__dirname, '..', 'docker-images', 'lambda', 'linux-x64', 'runner.js'),
                        target: '${LAMBDA_TASK_ROOT}/runner.js',
                    },
                    {
                        source: path.join(__dirname, '..', 'docker-images', 'lambda', 'linux-x64', 'runner.sh'),
                        target: '${LAMBDA_TASK_ROOT}/runner.sh',
                    },
                ];
            }
            getDockerCommands(_os, _architecture) {
                return [
                    'WORKDIR ${LAMBDA_TASK_ROOT}',
                    'CMD ["runner.handler"]',
                ];
            }
        };
    }
    /**
     * Returns assets to copy into the built image. Can be used to copy files into the image.
     */
    getAssets(_os, _architecture) {
        return [];
    }
    /**
     * Returns Docker commands to run to in built image. Can be used to add commands like `VOLUME`, `ENTRYPOINT`, `CMD`, etc.
     *
     * Docker commands are added after assets and normal commands.
     */
    getDockerCommands(_os, _architecture) {
        return [];
    }
    /**
     * Convert component to an AWS Image Builder component.
     *
     * @internal
     */
    _asAwsImageBuilderComponent(scope, id, os, architecture) {
        let platform;
        if (os.is(common_1.Os.LINUX_UBUNTU) || os.is(common_1.Os.LINUX_AMAZON_2)) {
            platform = 'Linux';
        }
        else if (os.is(common_1.Os.WINDOWS)) {
            platform = 'Windows';
        }
        else {
            throw new Error(`Unknown os/architecture combo for image builder component: ${os.name}/${architecture.name}`);
        }
        return new aws_image_builder_1.ImageBuilderComponent(scope, id, {
            platform: platform,
            commands: this.getCommands(os, architecture),
            assets: this.getAssets(os, architecture).map((asset, index) => {
                return {
                    asset: new aws_cdk_lib_1.aws_s3_assets.Asset(scope, `${id} asset ${index}`, { path: asset.source }),
                    path: asset.target,
                };
            }),
            displayName: id,
            description: id,
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
RunnerImageComponent[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.RunnerImageComponent", version: "0.9.0" };
exports.RunnerImageComponent = RunnerImageComponent;
//# sourceMappingURL=data:application/json;base64,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