"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RunnerImageBuilderBase = exports.RunnerImageBuilderType = exports.uniqueImageBuilderName = void 0;
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
/**
 * @internal
 */
function uniqueImageBuilderName(scope) {
    return cdk.Names.uniqueResourceName(scope, {
        maxLength: 126,
        separator: '-',
        allowedSpecialCharacters: '_-',
    });
}
exports.uniqueImageBuilderName = uniqueImageBuilderName;
var RunnerImageBuilderType;
(function (RunnerImageBuilderType) {
    /**
     * Build runner images using AWS CodeBuild.
     *
     * Faster than AWS Image Builder, but can only be used to build Linux Docker images.
     */
    RunnerImageBuilderType["CODE_BUILD"] = "CodeBuild";
    /**
     * Build runner images using AWS Image Builder.
     *
     * Slower than CodeBuild, but can be used to build any type of image including AMIs and Windows images.
     */
    RunnerImageBuilderType["AWS_IMAGE_BUILDER"] = "AwsImageBuilder";
})(RunnerImageBuilderType = exports.RunnerImageBuilderType || (exports.RunnerImageBuilderType = {}));
/**
 * @internal
 */
class RunnerImageBuilderBase extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.components = [];
        if (props?.components) {
            this.components.push(...props.components);
        }
    }
    /**
     * Add a component to the image builder. The component will be added to the end of the list of components.
     *
     * @param component component to add
     */
    addComponent(component) {
        this.components.push(component);
    }
    /**
     * Remove a component from the image builder. Removal is done by component name. Multiple components with the same name will all be removed.
     *
     * @param component component to remove
     */
    removeComponent(component) {
        this.components = this.components.filter(c => c.name !== component.name);
    }
}
exports.RunnerImageBuilderBase = RunnerImageBuilderBase;
//# sourceMappingURL=data:application/json;base64,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