"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeBuildImageBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
const build_image_function_1 = require("../../lambdas/build-image-function");
const utils_1 = require("../../utils");
const common_1 = require("../common");
/**
 * An image builder that uses CodeBuild to build Docker images pre-baked with all the GitHub Actions runner requirements. Builders can be used with runner providers.
 *
 * Each builder re-runs automatically at a set interval to make sure the images contain the latest versions of everything.
 *
 * You can create an instance of this construct to customize the image used to spin-up runners. Each provider has its own requirements for what an image should do. That's why they each provide their own Dockerfile.
 *
 * For example, to set a specific runner version, rebuild the image every 2 weeks, and add a few packages for the Fargate provider, use:
 *
 * ```
 * const builder = new CodeBuildImageBuilder(this, 'Builder', {
 *     dockerfilePath: FargateProvider.LINUX_X64_DOCKERFILE_PATH,
 *     runnerVersion: RunnerVersion.specific('2.293.0'),
 *     rebuildInterval: Duration.days(14),
 * });
 * builder.setBuildArg('EXTRA_PACKAGES', 'nginx xz-utils');
 * new FargateRunner(this, 'Fargate provider', {
 *     label: 'customized-fargate',
 *     imageBuilder: builder,
 * });
 * ```
 *
 * @deprecated use RunnerImageBuilder
 */
class CodeBuildImageBuilder extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.preBuild = [];
        this.postBuild = [];
        this.buildArgs = new Map();
        this.policyStatements = [];
        this.secondaryAssets = new Map();
        if (props.subnetSelection?.subnetType == aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_ISOLATED) {
            aws_cdk_lib_1.Annotations.of(this).addWarning('Private isolated subnets cannot pull from public ECR and VPC endpoint is not supported yet. ' +
                'See https://github.com/aws/containers-roadmap/issues/1160');
        }
        // set platform
        this.architecture = props.architecture ?? common_1.Architecture.X86_64;
        this.os = props.os ?? common_1.Os.LINUX;
        // create repository that only keeps one tag
        this.repository = new aws_cdk_lib_1.aws_ecr.Repository(this, 'Repository', {
            imageScanOnPush: true,
            imageTagMutability: aws_ecr_1.TagMutability.MUTABLE,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            lifecycleRules: [
                {
                    description: 'Remove untagged images that have been replaced by CodeBuild',
                    tagStatus: aws_ecr_1.TagStatus.UNTAGGED,
                    maxImageAge: aws_cdk_lib_1.Duration.days(1),
                },
            ],
        });
        // upload Dockerfile to S3 as an asset
        this.dockerfile = new aws_cdk_lib_1.aws_s3_assets.Asset(this, 'Dockerfile', {
            path: props.dockerfilePath,
        });
        // choose build image
        this.buildImage = props?.buildImage ?? this.getBuildImage();
    }
    /**
     * Uploads a folder to the build server at a given folder name.
     *
     * @param sourcePath path to source directory
     * @param destName name of destination folder
     */
    addFiles(sourcePath, destName) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        const asset = new aws_cdk_lib_1.aws_s3_assets.Asset(this, destName, { path: sourcePath });
        this.secondaryAssets.set(destName, asset);
        this.preBuild.push(`rm -rf "${destName}" && cp -r "$CODEBUILD_SRC_DIR_${destName}" "${destName}"`); // symlinks don't work with docker
    }
    /**
     * Adds a command that runs before `docker build`.
     *
     * @param command command to add
     */
    addPreBuildCommand(command) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.preBuild.push(command);
    }
    /**
     * Adds a command that runs after `docker build` and `docker push`.
     *
     * @param command command to add
     */
    addPostBuildCommand(command) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.postBuild.push(command);
    }
    /**
     * Adds a build argument for Docker. See the documentation for the Dockerfile you're using for a list of supported build arguments.
     *
     * @param name build argument name
     * @param value build argument value
     */
    setBuildArg(name, value) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.buildArgs.set(name, value);
    }
    /**
     * Add a policy statement to the builder to access resources required to the image build.
     *
     * @param statement IAM policy statement
     */
    addPolicyStatement(statement) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.policyStatements.push(statement);
    }
    /**
     * Add extra trusted certificates. This helps deal with self-signed certificates for GitHub Enterprise Server.
     *
     * All first party Dockerfiles support this. Others may not.
     *
     * @param path path to directory containing a file called certs.pem containing all the required certificates
     */
    addExtraCertificates(path) {
        if (this.boundImage) {
            throw new Error('Image is already bound. Use this method before passing the builder to a runner provider.');
        }
        this.addFiles(path, 'extra_certs');
    }
    /**
     * Called by IRunnerProvider to finalize settings and create the image builder.
     */
    bindDockerImage() {
        if (this.boundImage) {
            return this.boundImage;
        }
        // log group for the image builds
        const logGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
            retention: this.props.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH,
            removalPolicy: this.props.logRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        // generate buildSpec
        const buildSpec = this.getBuildSpec(this.repository, logGroup, this.props.runnerVersion);
        // create CodeBuild project that builds Dockerfile and pushes to repository
        const project = new aws_cdk_lib_1.aws_codebuild.Project(this, 'CodeBuild', {
            description: `Build docker image for self-hosted GitHub runner ${this.node.path} (${this.os.name}/${this.architecture.name})`,
            buildSpec: aws_cdk_lib_1.aws_codebuild.BuildSpec.fromObject(buildSpec),
            source: aws_cdk_lib_1.aws_codebuild.Source.s3({
                bucket: this.dockerfile.bucket,
                path: this.dockerfile.s3ObjectKey,
            }),
            vpc: this.props.vpc,
            securityGroups: this.props.securityGroup ? [this.props.securityGroup] : undefined,
            subnetSelection: this.props.subnetSelection,
            timeout: this.props.timeout ?? aws_cdk_lib_1.Duration.hours(1),
            environment: {
                buildImage: this.buildImage,
                computeType: this.props.computeType ?? aws_codebuild_1.ComputeType.SMALL,
                privileged: true,
            },
            logging: {
                cloudWatch: {
                    logGroup,
                },
            },
        });
        // permissions
        this.repository.grantPullPush(project);
        this.policyStatements.forEach(project.addToRolePolicy);
        // call CodeBuild during deployment and delete all images from repository during destruction
        const cr = this.customResource(project);
        // rebuild image on a schedule
        this.rebuildImageOnSchedule(project, this.props.rebuildInterval);
        for (const [assetPath, asset] of this.secondaryAssets.entries()) {
            project.addSecondarySource(aws_cdk_lib_1.aws_codebuild.Source.s3({
                identifier: assetPath,
                bucket: asset.bucket,
                path: asset.s3ObjectKey,
            }));
        }
        this.boundImage = {
            imageRepository: aws_cdk_lib_1.aws_ecr.Repository.fromRepositoryAttributes(this, 'Dependable Image', {
                // There are simpler ways to get name and ARN, but we want an image object that depends on the custom resource.
                // We want whoever is using this image to automatically wait for CodeBuild to start and finish through the custom resource.
                repositoryName: cr.getAttString('Name'),
                repositoryArn: cr.ref,
            }),
            imageTag: 'latest',
            architecture: this.architecture,
            os: this.os,
            logGroup,
            runnerVersion: this.props.runnerVersion ?? common_1.RunnerVersion.latest(),
        };
        return this.boundImage;
    }
    getBuildImage() {
        if (this.os.is(common_1.Os.LINUX)) {
            if (this.architecture.is(common_1.Architecture.X86_64)) {
                return aws_cdk_lib_1.aws_codebuild.LinuxBuildImage.STANDARD_6_0;
            }
            else if (this.architecture.is(common_1.Architecture.ARM64)) {
                return aws_cdk_lib_1.aws_codebuild.LinuxArmBuildImage.AMAZON_LINUX_2_STANDARD_2_0;
            }
        }
        if (this.os.is(common_1.Os.WINDOWS)) {
            throw new Error('CodeBuild cannot be used to build Windows Docker images https://github.com/docker-library/docker/issues/49');
        }
        throw new Error(`Unable to find CodeBuild image for ${this.os.name}/${this.architecture.name}`);
    }
    getBuildSpec(repository, logGroup, runnerVersion) {
        // don't forget to change BUILDSPEC_VERSION when the buildSpec changes, and you want to trigger a rebuild on deploy
        let buildArgs = '';
        for (const [name, value] of this.buildArgs.entries()) {
            buildArgs += ` --build-arg "${name}"="${value}"`;
        }
        buildArgs += ` --build-arg RUNNER_VERSION="${runnerVersion ? runnerVersion.version : common_1.RunnerVersion.latest().version}"`;
        const thisStack = cdk.Stack.of(this);
        return {
            version: '0.2',
            env: {
                variables: {
                    REPO_ARN: repository.repositoryArn,
                    REPO_URI: repository.repositoryUri,
                    STACK_ID: 'unspecified',
                    REQUEST_ID: 'unspecified',
                    LOGICAL_RESOURCE_ID: 'unspecified',
                    RESPONSE_URL: 'unspecified',
                    RUNNER_VERSION: runnerVersion ? runnerVersion.version : common_1.RunnerVersion.latest().version,
                },
            },
            phases: {
                pre_build: {
                    commands: this.preBuild.concat([
                        'mkdir -p extra_certs',
                        `aws ecr get-login-password --region "$AWS_DEFAULT_REGION" | docker login --username AWS --password-stdin ${thisStack.account}.dkr.ecr.${thisStack.region}.amazonaws.com`,
                    ]),
                },
                build: {
                    commands: [
                        `docker build . -t "$REPO_URI" ${buildArgs}`,
                        'docker push "$REPO_URI"',
                    ],
                },
                post_build: {
                    commands: this.postBuild.concat([
                        'STATUS="SUCCESS"',
                        'if [ $CODEBUILD_BUILD_SUCCEEDING -ne 1 ]; then STATUS="FAILED"; fi',
                        'cat <<EOF > /tmp/payload.json\n' +
                            '{\n' +
                            '  "StackId": "$STACK_ID",\n' +
                            '  "RequestId": "$REQUEST_ID",\n' +
                            '  "LogicalResourceId": "$LOGICAL_RESOURCE_ID",\n' +
                            '  "PhysicalResourceId": "$REPO_ARN",\n' +
                            '  "Status": "$STATUS",\n' +
                            `  "Reason": "See logs in ${logGroup.logGroupName}/$CODEBUILD_LOG_PATH (deploy again with \'cdk deploy -R\' or logRemovalPolicy=RemovalPolicy.RETAIN if they are already deleted)",\n` +
                            `  "Data": {"Name": "${repository.repositoryName}"}\n` +
                            '}\n' +
                            'EOF',
                        'if [ "$RESPONSE_URL" != "unspecified" ]; then jq . /tmp/payload.json; curl -fsSL -X PUT -H "Content-Type:" -d "@/tmp/payload.json" "$RESPONSE_URL"; fi',
                    ]),
                },
            },
        };
    }
    customResource(project) {
        const crHandler = (0, utils_1.singletonLambda)(build_image_function_1.BuildImageFunction, this, 'build-image', {
            description: 'Custom resource handler that triggers CodeBuild to build runner images, and cleans-up images on deletion',
            timeout: cdk.Duration.minutes(3),
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
        });
        const policy = new aws_cdk_lib_1.aws_iam.Policy(this, 'CR Policy', {
            statements: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['codebuild:StartBuild'],
                    resources: [project.projectArn],
                }),
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['ecr:BatchDeleteImage', 'ecr:ListImages'],
                    resources: [this.repository.repositoryArn],
                }),
            ],
        });
        crHandler.role?.attachInlinePolicy(policy);
        const cr = new aws_cdk_lib_1.CustomResource(this, 'Builder', {
            serviceToken: crHandler.functionArn,
            resourceType: 'Custom::ImageBuilder',
            properties: {
                RepoName: this.repository.repositoryName,
                ProjectName: project.projectName,
                // We include a hash so the image is built immediately on changes, and we don't have to wait for its scheduled build.
                // This also helps make sure the changes are good. If they have a bug, the deployment will fail instead of just the scheduled build.
                BuildHash: this.hashBuildSettings(),
            },
        });
        // add dependencies to make sure resources are there when we need them
        cr.node.addDependency(project);
        cr.node.addDependency(policy);
        cr.node.addDependency(crHandler);
        return cr;
    }
    /**
     * Return hash of all settings that can affect the result image so we can trigger the build when it changes.
     * @private
     */
    hashBuildSettings() {
        // main Dockerfile
        let components = [this.dockerfile.assetHash];
        // all additional files
        for (const [name, asset] of this.secondaryAssets.entries()) {
            components.push(name);
            components.push(asset.assetHash);
        }
        // buildspec.yml version
        components.push(`v${CodeBuildImageBuilder.BUILDSPEC_VERSION}`);
        // runner version
        components.push(this.props.runnerVersion?.version ?? common_1.RunnerVersion.latest().version);
        // user commands
        components = components.concat(this.preBuild);
        components = components.concat(this.postBuild);
        for (const [name, value] of this.buildArgs.entries()) {
            components.push(name);
            components.push(value);
        }
        // hash it
        const all = components.join('-');
        return crypto.createHash('md5').update(all).digest('hex');
    }
    rebuildImageOnSchedule(project, rebuildInterval) {
        rebuildInterval = rebuildInterval ?? aws_cdk_lib_1.Duration.days(7);
        if (rebuildInterval.toMilliseconds() != 0) {
            const scheduleRule = new aws_cdk_lib_1.aws_events.Rule(this, 'Build Schedule', {
                description: `Rebuild runner image for ${this.repository.repositoryName}`,
                schedule: aws_cdk_lib_1.aws_events.Schedule.rate(rebuildInterval),
            });
            scheduleRule.addTarget(new aws_cdk_lib_1.aws_events_targets.CodeBuildProject(project));
        }
    }
    get connections() {
        return new aws_cdk_lib_1.aws_ec2.Connections({
            securityGroups: this.props.securityGroup ? [this.props.securityGroup] : [],
        });
    }
    bindAmi() {
        throw new Error('CodeBuildImageBuilder does not support building AMIs');
    }
}
_a = JSII_RTTI_SYMBOL_1;
CodeBuildImageBuilder[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.CodeBuildImageBuilder", version: "0.9.0" };
/**
 * Bump this number every time the buildspec or any important setting of the project changes. It will force a rebuild of the image.
 * @private
 */
CodeBuildImageBuilder.BUILDSPEC_VERSION = 2;
exports.CodeBuildImageBuilder = CodeBuildImageBuilder;
//# sourceMappingURL=data:application/json;base64,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