"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ImageBuilderBase = void 0;
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const common_1 = require("../../../common");
const common_2 = require("../../common");
/**
 * @internal
 */
class ImageBuilderBase extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.components = [];
        // arch
        this.architecture = props?.architecture ?? common_1.Architecture.X86_64;
        if (!this.architecture.isIn(props.supportedArchitectures)) {
            throw new Error(`Unsupported architecture: ${this.architecture.name}. Consider CodeBuild for faster image builds.`);
        }
        // os
        this.os = props?.os ?? common_1.Os.LINUX;
        if (!this.os.isIn(props.supportedOs)) {
            throw new Error(`Unsupported OS: ${this.os.name}.`);
        }
        // platform
        if (this.os.is(common_1.Os.WINDOWS)) {
            this.platform = 'Windows';
        }
        else if (this.os.is(common_1.Os.LINUX) || this.os.is(common_1.Os.LINUX_UBUNTU) || this.os.is(common_1.Os.LINUX_UBUNTU)) {
            this.platform = 'Linux';
        }
        else {
            throw new Error(`Unsupported OS: ${this.os.name}.`);
        }
        // builder options
        this.rebuildInterval = props?.rebuildInterval ?? cdk.Duration.days(7);
        // vpc settings
        if (props?.vpc) {
            this.vpc = props.vpc;
            this.subnetId = props.vpc.selectSubnets(props.subnetSelection).subnetIds[0];
        }
        else {
            this.vpc = aws_cdk_lib_1.aws_ec2.Vpc.fromLookup(this, 'Default VPC', { isDefault: true });
        }
        if (props?.securityGroups) {
            this.securityGroups = props.securityGroups;
        }
        else {
            this.securityGroups = [new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'SG', { vpc: this.vpc })];
        }
        // instance type
        this.instanceType = props?.instanceType ?? aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.M5, aws_cdk_lib_1.aws_ec2.InstanceSize.LARGE);
        if (!this.architecture.instanceTypeMatch(this.instanceType)) {
            throw new Error(`Builder architecture (${this.architecture.name}) doesn't match selected instance type (${this.instanceType} / ${this.instanceType.architecture})`);
        }
        // log settings
        this.logRetention = props?.logRetention ?? aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH;
        this.logRemovalPolicy = props?.logRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.DESTROY;
        // runner version
        this.runnerVersion = props?.runnerVersion ?? common_1.RunnerVersion.latest();
        // description
        this.description = `Build ${props.imageTypeName} for GitHub Actions runner ${this.node.path} (${this.os.name}/${this.architecture.name})`;
    }
    createLog(recipeName) {
        return new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Log', {
            logGroupName: `/aws/imagebuilder/${recipeName}`,
            retention: this.logRetention,
            removalPolicy: this.logRemovalPolicy,
        });
    }
    createInfrastructure(managedPolicies) {
        let role = new aws_cdk_lib_1.aws_iam.Role(this, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: managedPolicies,
        });
        for (const component of this.components) {
            component.grantAssetsRead(role);
        }
        return new aws_cdk_lib_1.aws_imagebuilder.CfnInfrastructureConfiguration(this, 'Infrastructure', {
            name: (0, common_2.uniqueImageBuilderName)(this),
            description: this.description,
            subnetId: this.subnetId,
            securityGroupIds: this.securityGroups.map(sg => sg.securityGroupId),
            instanceTypes: [this.instanceType.toString()],
            instanceProfileName: new aws_cdk_lib_1.aws_iam.CfnInstanceProfile(this, 'Instance Profile', {
                roles: [
                    role.roleName,
                ],
            }).ref,
        });
    }
    createImage(infra, dist, log, imageRecipeArn, containerRecipeArn) {
        const image = new aws_cdk_lib_1.aws_imagebuilder.CfnImage(this, 'Image', {
            infrastructureConfigurationArn: infra.attrArn,
            distributionConfigurationArn: dist.attrArn,
            imageRecipeArn,
            containerRecipeArn,
            imageTestsConfiguration: {
                imageTestsEnabled: false,
            },
        });
        image.node.addDependency(infra);
        image.node.addDependency(log);
        return image;
    }
    createPipeline(infra, dist, log, imageRecipeArn, containerRecipeArn) {
        let scheduleOptions;
        if (this.rebuildInterval.toDays() > 0) {
            scheduleOptions = {
                scheduleExpression: aws_cdk_lib_1.aws_events.Schedule.rate(this.rebuildInterval).expressionString,
                pipelineExecutionStartCondition: 'EXPRESSION_MATCH_ONLY',
            };
        }
        const pipeline = new aws_cdk_lib_1.aws_imagebuilder.CfnImagePipeline(this, 'Pipeline', {
            name: (0, common_2.uniqueImageBuilderName)(this),
            description: this.description,
            infrastructureConfigurationArn: infra.attrArn,
            distributionConfigurationArn: dist.attrArn,
            imageRecipeArn,
            containerRecipeArn,
            schedule: scheduleOptions,
            imageTestsConfiguration: {
                imageTestsEnabled: false,
            },
        });
        pipeline.node.addDependency(infra);
        pipeline.node.addDependency(log);
        return pipeline;
    }
    /**
     * The network connections associated with this resource.
     */
    get connections() {
        return new aws_cdk_lib_1.aws_ec2.Connections({ securityGroups: this.securityGroups });
    }
}
exports.ImageBuilderBase = ImageBuilderBase;
//# sourceMappingURL=data:application/json;base64,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