"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable import/no-extraneous-dependencies */
const auth_app_1 = require("@octokit/auth-app");
const core_1 = require("@octokit/core");
const AWS = require("aws-sdk");
const github_1 = require("./github");
const helpers_1 = require("./helpers");
const cfn = new AWS.CloudFormation();
const ec2 = new AWS.EC2();
const ecr = new AWS.ECR();
const sf = new AWS.StepFunctions();
function secretArnToUrl(arn) {
    const parts = arn.split(':'); // arn:aws:secretsmanager:us-east-1:12345678:secret:secret-name-REVISION
    const region = parts[3];
    const fullName = parts[6];
    const name = fullName.slice(0, fullName.lastIndexOf('-'));
    return `https://${region}.console.aws.amazon.com/secretsmanager/home?region=${region}#!/secret?name=${name}`;
}
function lambdaArnToUrl(arn) {
    const parts = arn.split(':'); // arn:aws:lambda:us-east-1:12345678:function:name-XYZ
    const region = parts[3];
    const name = parts[6];
    return `https://${region}.console.aws.amazon.com/lambda/home?region=${region}#/functions/${name}?tab=monitoring`;
}
function stepFunctionArnToUrl(arn) {
    const parts = arn.split(':'); // arn:aws:states:us-east-1:12345678:stateMachine:name-XYZ
    const region = parts[3];
    return `https://${region}.console.aws.amazon.com/states/home?region=${region}#/statemachines/view/${arn}`;
}
async function generateProvidersStatus(stack, logicalId) {
    const resource = await cfn.describeStackResource({ StackName: stack, LogicalResourceId: logicalId }).promise();
    const providers = JSON.parse(resource.StackResourceDetail?.Metadata ?? '{}').providers;
    if (!providers) {
        return {};
    }
    return Promise.all(providers.map(async (p) => {
        // add ECR data, if image is from ECR
        if (p.image?.imageRepository?.match(/[0-9]+\.dkr\.ecr\.[a-z0-9\-]+\.amazonaws\.com\/.+/)) {
            const tags = await ecr.describeImages({
                repositoryName: p.image.imageRepository.split('/')[1],
                filter: {
                    tagStatus: 'TAGGED',
                },
                maxResults: 1,
            }).promise();
            if (tags.imageDetails && tags.imageDetails?.length >= 1) {
                p.image.latestImage = {
                    tags: tags.imageDetails[0].imageTags,
                    digest: tags.imageDetails[0].imageDigest,
                    date: tags.imageDetails[0].imagePushedAt,
                };
            }
        }
        // add AMI data, if image is AMI
        if (p.ami?.launchTemplate) {
            const versions = await ec2.describeLaunchTemplateVersions({
                LaunchTemplateId: p.ami.launchTemplate,
                Versions: ['$Default'],
            }).promise();
            if (versions.LaunchTemplateVersions && versions.LaunchTemplateVersions.length >= 1) {
                p.ami.latestAmi = versions.LaunchTemplateVersions[0].LaunchTemplateData?.ImageId;
            }
        }
        return p;
    }));
}
exports.handler = async function () {
    // confirm required environment variables
    if (!process.env.WEBHOOK_SECRET_ARN || !process.env.GITHUB_SECRET_ARN || !process.env.GITHUB_PRIVATE_KEY_SECRET_ARN || !process.env.LOGICAL_ID ||
        !process.env.WEBHOOK_HANDLER_ARN || !process.env.STEP_FUNCTION_ARN || !process.env.SETUP_SECRET_ARN || !process.env.SETUP_FUNCTION_URL ||
        !process.env.STACK_NAME) {
        throw new Error('Missing environment variables');
    }
    // base status
    const status = {
        github: {
            setup: {
                status: 'Unknown',
                url: '',
                secretArn: process.env.SETUP_SECRET_ARN,
                secretUrl: secretArnToUrl(process.env.SETUP_SECRET_ARN),
            },
            domain: 'Unknown',
            webhook: {
                url: process.env.WEBHOOK_URL,
                status: 'Unable to check',
                secretArn: process.env.WEBHOOK_SECRET_ARN,
                secretUrl: secretArnToUrl(process.env.WEBHOOK_SECRET_ARN),
            },
            auth: {
                type: 'Unknown',
                status: 'Unknown',
                secretArn: process.env.GITHUB_SECRET_ARN,
                secretUrl: secretArnToUrl(process.env.GITHUB_SECRET_ARN),
                privateKeySecretArn: process.env.GITHUB_PRIVATE_KEY_SECRET_ARN,
                privateKeySecretUrl: secretArnToUrl(process.env.GITHUB_PRIVATE_KEY_SECRET_ARN),
                app: {
                    id: '',
                    url: '',
                    installations: [],
                },
                personalAuthToken: '',
            },
        },
        providers: await generateProvidersStatus(process.env.STACK_NAME, process.env.LOGICAL_ID),
        troubleshooting: {
            webhookHandlerArn: process.env.WEBHOOK_HANDLER_ARN,
            webhookHandlerUrl: lambdaArnToUrl(process.env.WEBHOOK_HANDLER_ARN),
            stepFunctionArn: process.env.STEP_FUNCTION_ARN,
            stepFunctionUrl: stepFunctionArnToUrl(process.env.STEP_FUNCTION_ARN),
            stepFunctionLogGroup: process.env.STEP_FUNCTION_LOG_GROUP,
            recentRuns: [],
        },
    };
    // setup url
    const setupToken = (await (0, helpers_1.getSecretJsonValue)(process.env.SETUP_SECRET_ARN)).token;
    if (setupToken) {
        status.github.setup.status = 'Pending';
        status.github.setup.url = `${process.env.SETUP_FUNCTION_URL}?token=${setupToken}`;
    }
    else {
        status.github.setup.status = 'Complete';
    }
    // list last 10 executions and their status
    try {
        const executions = await sf.listExecutions({
            stateMachineArn: process.env.STEP_FUNCTION_ARN,
            maxResults: 10,
        }).promise();
        for (const execution of executions.executions) {
            const executionDetails = await sf.describeExecution({
                executionArn: execution.executionArn,
            }).promise();
            const input = JSON.parse(executionDetails.input || '{}');
            status.troubleshooting.recentRuns.push({
                executionArn: execution.executionArn,
                status: execution.status,
                owner: input.owner,
                repo: input.repo,
                runId: input.runId,
            });
        }
    }
    catch (e) {
        status.troubleshooting.recentRuns.push({ status: `Error getting executions: ${e}` });
    }
    // get secrets
    let githubSecrets;
    try {
        githubSecrets = await (0, helpers_1.getSecretJsonValue)(process.env.GITHUB_SECRET_ARN);
    }
    catch (e) {
        status.github.auth.status = `Unable to read secret: ${e}`;
        return status;
    }
    let privateKey;
    try {
        privateKey = await (0, helpers_1.getSecretValue)(process.env.GITHUB_PRIVATE_KEY_SECRET_ARN);
    }
    catch (e) {
        status.github.auth.status = `Unable to read private key secret: ${e}`;
        return status;
    }
    // calculate base url
    let baseUrl = (0, github_1.baseUrlFromDomain)(githubSecrets.domain);
    status.github.domain = githubSecrets.domain;
    if (githubSecrets.personalAuthToken) {
        // try authenticating with personal authentication token
        status.github.auth.type = 'Personal Auth Token';
        status.github.auth.personalAuthToken = '*redacted*';
        let octokit;
        try {
            octokit = new core_1.Octokit({ baseUrl, auth: githubSecrets.personalAuthToken });
        }
        catch (e) {
            status.github.auth.status = `Unable to authenticate using personal auth token: ${e}`;
            return status;
        }
        try {
            const user = await octokit.request('GET /user');
            status.github.auth.personalAuthToken = `username: ${user.data.login}`;
        }
        catch (e) {
            status.github.auth.status = `Unable to call /user with personal auth token: ${e}`;
            return status;
        }
        status.github.auth.status = 'OK';
        status.github.webhook.status = 'Unable to verify automatically';
    }
    else {
        // try authenticating with GitHub app
        status.github.auth.type = 'GitHub App';
        status.github.auth.app.id = githubSecrets.appId;
        let appOctokit;
        try {
            appOctokit = new core_1.Octokit({
                baseUrl,
                authStrategy: auth_app_1.createAppAuth,
                auth: {
                    appId: githubSecrets.appId,
                    privateKey: privateKey,
                },
            });
        }
        catch (e) {
            status.github.auth.status = `Unable to authenticate app: ${e}`;
            return status;
        }
        // get app url
        try {
            const app = (await appOctokit.request('GET /app')).data;
            status.github.auth.app.url = app.html_url;
        }
        catch (e) {
            status.github.auth.status = `Unable to get app details: ${e}`;
            return status;
        }
        // list all app installations
        try {
            const installations = (await appOctokit.request('GET /app/installations')).data;
            for (const installation of installations) {
                let installationDetails = {
                    id: installation.id,
                    url: installation.html_url,
                    status: 'Unable to query',
                    repositories: [],
                };
                let token;
                try {
                    token = (await appOctokit.auth({
                        type: 'installation',
                        installationId: installation.id,
                    })).token;
                }
                catch (e) {
                    installationDetails.status = `Unable to authenticate app installation: ${e}`;
                    continue;
                }
                let octokit;
                try {
                    octokit = new core_1.Octokit({ baseUrl, auth: token });
                }
                catch (e) {
                    installationDetails.status = `Unable to authenticate using app: ${e}`;
                    continue;
                }
                try {
                    const repositories = (await octokit.request('GET /installation/repositories')).data.repositories;
                    for (const repo of repositories) {
                        installationDetails.repositories.push(repo.full_name);
                    }
                }
                catch (e) {
                    installationDetails.status = `Unable to authenticate using installation token: ${e}`;
                    continue;
                }
                installationDetails.status = 'OK';
                status.github.auth.app.installations.push(installationDetails);
            }
        }
        catch (e) {
            status.github.auth.status = 'Unable to list app installations';
            return status;
        }
        status.github.auth.status = 'OK';
        // check webhook config
        try {
            const response = await appOctokit.request('GET /app/hook/config', {});
            if (response.data.url !== process.env.WEBHOOK_URL) {
                status.github.webhook.status = 'GitHub has wrong webhook URL configured';
            }
            else {
                // TODO check secret by doing a dummy delivery? force apply secret?
                status.github.webhook.status = 'OK (note that secret cannot be checked automatically)';
            }
        }
        catch (e) {
            status.github.webhook.status = `Unable to check app configuration: ${e}`;
            return status;
        }
    }
    return status;
};
//# sourceMappingURL=data:application/json;base64,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