"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const github_1 = require("./github");
async function getRunnerId(octokit, owner, repo, name, idleOnly) {
    let page = 1;
    while (true) {
        const runners = await octokit.request('GET /repos/{owner}/{repo}/actions/runners?per_page=100&page={page}', {
            page: page,
            owner: owner,
            repo: repo,
        });
        if (runners.data.runners.length == 0) {
            return;
        }
        for (const runner of runners.data.runners) {
            if (runner.name == name) {
                if (idleOnly) {
                    if (!runner.busy) {
                        return runner.id;
                    }
                    else {
                        console.log('Runner is busy, no need to delete.');
                        return;
                    }
                }
                return runner.id;
            }
        }
        page++;
    }
}
class RunnerBusy extends Error {
    constructor(msg) {
        super(msg);
        this.name = 'RunnerBusy';
        Object.setPrototypeOf(this, RunnerBusy.prototype);
    }
}
exports.handler = async function (event) {
    const { octokit } = await (0, github_1.getOctokit)(event.installationId);
    // find runner id
    const runnerId = await getRunnerId(octokit, event.owner, event.repo, event.runnerName, event.idleOnly);
    if (!runnerId) {
        console.error(`Unable to find runner id for ${event.owner}/${event.repo}:${event.runnerName}`);
        return;
    }
    console.log(`Runner ${event.runnerName} has id #${runnerId}`);
    // delete runner (it usually gets deleted by ./run.sh, but it stopped prematurely if we're here).
    // it seems like runners are automatically removed after a timeout, if they first accepted a job.
    // we try removing it anyway for cases where a job wasn't accepted, and just in case it wasn't removed.
    // repos have a limited number of self-hosted runners, so we can't leave dead ones behind.
    try {
        await octokit.rest.actions.deleteSelfHostedRunnerFromRepo({
            owner: event.owner,
            repo: event.repo,
            runner_id: runnerId,
        });
    }
    catch (e) {
        const reqError = e;
        if (reqError.message.includes('is still running a job')) {
            throw new RunnerBusy(reqError.message);
        }
        else {
            throw e;
        }
    }
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVsZXRlLXJ1bm5lci5sYW1iZGEuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvbGFtYmRhcy9kZWxldGUtcnVubmVyLmxhbWJkYS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUVBLHFDQUFzQztBQU90QyxLQUFLLFVBQVUsV0FBVyxDQUFDLE9BQVksRUFBRSxLQUFhLEVBQUUsSUFBWSxFQUFFLElBQVksRUFBRSxRQUFpQjtJQUNuRyxJQUFJLElBQUksR0FBRyxDQUFDLENBQUM7SUFDYixPQUFPLElBQUksRUFBRTtRQUNYLE1BQU0sT0FBTyxHQUFHLE1BQU0sT0FBTyxDQUFDLE9BQU8sQ0FBQyxvRUFBb0UsRUFBRTtZQUMxRyxJQUFJLEVBQUUsSUFBSTtZQUNWLEtBQUssRUFBRSxLQUFLO1lBQ1osSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUM7UUFFSCxJQUFJLE9BQU8sQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sSUFBSSxDQUFDLEVBQUU7WUFDcEMsT0FBTztTQUNSO1FBRUQsS0FBSyxNQUFNLE1BQU0sSUFBSSxPQUFPLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUN6QyxJQUFJLE1BQU0sQ0FBQyxJQUFJLElBQUksSUFBSSxFQUFFO2dCQUN2QixJQUFJLFFBQVEsRUFBRTtvQkFDWixJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRTt3QkFDaEIsT0FBTyxNQUFNLENBQUMsRUFBRSxDQUFDO3FCQUNsQjt5QkFBTTt3QkFDTCxPQUFPLENBQUMsR0FBRyxDQUFDLG9DQUFvQyxDQUFDLENBQUM7d0JBQ2xELE9BQU87cUJBQ1I7aUJBQ0Y7Z0JBQ0QsT0FBTyxNQUFNLENBQUMsRUFBRSxDQUFDO2FBQ2xCO1NBQ0Y7UUFFRCxJQUFJLEVBQUUsQ0FBQztLQUNSO0FBQ0gsQ0FBQztBQUVELE1BQU0sVUFBVyxTQUFRLEtBQUs7SUFDNUIsWUFBWSxHQUFXO1FBQ3JCLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUNYLElBQUksQ0FBQyxJQUFJLEdBQUcsWUFBWSxDQUFDO1FBQ3pCLE1BQU0sQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLFVBQVUsQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUNwRCxDQUFDO0NBQ0Y7QUFFRCxPQUFPLENBQUMsT0FBTyxHQUFHLEtBQUssV0FBVyxLQUF3QjtJQUN4RCxNQUFNLEVBQUUsT0FBTyxFQUFFLEdBQUcsTUFBTSxJQUFBLG1CQUFVLEVBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBRTNELGlCQUFpQjtJQUNqQixNQUFNLFFBQVEsR0FBRyxNQUFNLFdBQVcsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxVQUFVLEVBQUUsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ3ZHLElBQUksQ0FBQyxRQUFRLEVBQUU7UUFDYixPQUFPLENBQUMsS0FBSyxDQUFDLGdDQUFnQyxLQUFLLENBQUMsS0FBSyxJQUFJLEtBQUssQ0FBQyxJQUFJLElBQUksS0FBSyxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUM7UUFDL0YsT0FBTztLQUNSO0lBRUQsT0FBTyxDQUFDLEdBQUcsQ0FBQyxVQUFVLEtBQUssQ0FBQyxVQUFVLFlBQVksUUFBUSxFQUFFLENBQUMsQ0FBQztJQUU5RCxpR0FBaUc7SUFDakcsaUdBQWlHO0lBQ2pHLHVHQUF1RztJQUN2RywwRkFBMEY7SUFDMUYsSUFBSTtRQUNGLE1BQU0sT0FBTyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsOEJBQThCLENBQUM7WUFDeEQsS0FBSyxFQUFFLEtBQUssQ0FBQyxLQUFLO1lBQ2xCLElBQUksRUFBRSxLQUFLLENBQUMsSUFBSTtZQUNoQixTQUFTLEVBQUUsUUFBUTtTQUNwQixDQUFDLENBQUM7S0FDSjtJQUFDLE9BQU8sQ0FBQyxFQUFFO1FBQ1YsTUFBTSxRQUFRLEdBQWlCLENBQUMsQ0FBQztRQUNqQyxJQUFJLFFBQVEsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDLHdCQUF3QixDQUFDLEVBQUU7WUFDdkQsTUFBTSxJQUFJLFVBQVUsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUM7U0FDeEM7YUFBTTtZQUNMLE1BQU0sQ0FBQyxDQUFDO1NBQ1Q7S0FDRjtBQUNILENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8vIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBpbXBvcnQvbm8tZXh0cmFuZW91cy1kZXBlbmRlbmNpZXNcbmltcG9ydCB7IFJlcXVlc3RFcnJvciB9IGZyb20gJ0BvY3Rva2l0L3JlcXVlc3QtZXJyb3InO1xuaW1wb3J0IHsgZ2V0T2N0b2tpdCB9IGZyb20gJy4vZ2l0aHViJztcbmltcG9ydCB7IFN0ZXBGdW5jdGlvbkxhbWJkYUlucHV0IH0gZnJvbSAnLi9oZWxwZXJzJztcblxuaW50ZXJmYWNlIERlbGV0ZVJ1bm5lcklucHV0IGV4dGVuZHMgU3RlcEZ1bmN0aW9uTGFtYmRhSW5wdXQge1xuICByZWFkb25seSBpZGxlT25seTogYm9vbGVhbjtcbn1cblxuYXN5bmMgZnVuY3Rpb24gZ2V0UnVubmVySWQob2N0b2tpdDogYW55LCBvd25lcjogc3RyaW5nLCByZXBvOiBzdHJpbmcsIG5hbWU6IHN0cmluZywgaWRsZU9ubHk6IGJvb2xlYW4pIHtcbiAgbGV0IHBhZ2UgPSAxO1xuICB3aGlsZSAodHJ1ZSkge1xuICAgIGNvbnN0IHJ1bm5lcnMgPSBhd2FpdCBvY3Rva2l0LnJlcXVlc3QoJ0dFVCAvcmVwb3Mve293bmVyfS97cmVwb30vYWN0aW9ucy9ydW5uZXJzP3Blcl9wYWdlPTEwMCZwYWdlPXtwYWdlfScsIHtcbiAgICAgIHBhZ2U6IHBhZ2UsXG4gICAgICBvd25lcjogb3duZXIsXG4gICAgICByZXBvOiByZXBvLFxuICAgIH0pO1xuXG4gICAgaWYgKHJ1bm5lcnMuZGF0YS5ydW5uZXJzLmxlbmd0aCA9PSAwKSB7XG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgZm9yIChjb25zdCBydW5uZXIgb2YgcnVubmVycy5kYXRhLnJ1bm5lcnMpIHtcbiAgICAgIGlmIChydW5uZXIubmFtZSA9PSBuYW1lKSB7XG4gICAgICAgIGlmIChpZGxlT25seSkge1xuICAgICAgICAgIGlmICghcnVubmVyLmJ1c3kpIHtcbiAgICAgICAgICAgIHJldHVybiBydW5uZXIuaWQ7XG4gICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgIGNvbnNvbGUubG9nKCdSdW5uZXIgaXMgYnVzeSwgbm8gbmVlZCB0byBkZWxldGUuJyk7XG4gICAgICAgICAgICByZXR1cm47XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIHJldHVybiBydW5uZXIuaWQ7XG4gICAgICB9XG4gICAgfVxuXG4gICAgcGFnZSsrO1xuICB9XG59XG5cbmNsYXNzIFJ1bm5lckJ1c3kgZXh0ZW5kcyBFcnJvciB7XG4gIGNvbnN0cnVjdG9yKG1zZzogc3RyaW5nKSB7XG4gICAgc3VwZXIobXNnKTtcbiAgICB0aGlzLm5hbWUgPSAnUnVubmVyQnVzeSc7XG4gICAgT2JqZWN0LnNldFByb3RvdHlwZU9mKHRoaXMsIFJ1bm5lckJ1c3kucHJvdG90eXBlKTtcbiAgfVxufVxuXG5leHBvcnRzLmhhbmRsZXIgPSBhc3luYyBmdW5jdGlvbiAoZXZlbnQ6IERlbGV0ZVJ1bm5lcklucHV0KSB7XG4gIGNvbnN0IHsgb2N0b2tpdCB9ID0gYXdhaXQgZ2V0T2N0b2tpdChldmVudC5pbnN0YWxsYXRpb25JZCk7XG5cbiAgLy8gZmluZCBydW5uZXIgaWRcbiAgY29uc3QgcnVubmVySWQgPSBhd2FpdCBnZXRSdW5uZXJJZChvY3Rva2l0LCBldmVudC5vd25lciwgZXZlbnQucmVwbywgZXZlbnQucnVubmVyTmFtZSwgZXZlbnQuaWRsZU9ubHkpO1xuICBpZiAoIXJ1bm5lcklkKSB7XG4gICAgY29uc29sZS5lcnJvcihgVW5hYmxlIHRvIGZpbmQgcnVubmVyIGlkIGZvciAke2V2ZW50Lm93bmVyfS8ke2V2ZW50LnJlcG99OiR7ZXZlbnQucnVubmVyTmFtZX1gKTtcbiAgICByZXR1cm47XG4gIH1cblxuICBjb25zb2xlLmxvZyhgUnVubmVyICR7ZXZlbnQucnVubmVyTmFtZX0gaGFzIGlkICMke3J1bm5lcklkfWApO1xuXG4gIC8vIGRlbGV0ZSBydW5uZXIgKGl0IHVzdWFsbHkgZ2V0cyBkZWxldGVkIGJ5IC4vcnVuLnNoLCBidXQgaXQgc3RvcHBlZCBwcmVtYXR1cmVseSBpZiB3ZSdyZSBoZXJlKS5cbiAgLy8gaXQgc2VlbXMgbGlrZSBydW5uZXJzIGFyZSBhdXRvbWF0aWNhbGx5IHJlbW92ZWQgYWZ0ZXIgYSB0aW1lb3V0LCBpZiB0aGV5IGZpcnN0IGFjY2VwdGVkIGEgam9iLlxuICAvLyB3ZSB0cnkgcmVtb3ZpbmcgaXQgYW55d2F5IGZvciBjYXNlcyB3aGVyZSBhIGpvYiB3YXNuJ3QgYWNjZXB0ZWQsIGFuZCBqdXN0IGluIGNhc2UgaXQgd2Fzbid0IHJlbW92ZWQuXG4gIC8vIHJlcG9zIGhhdmUgYSBsaW1pdGVkIG51bWJlciBvZiBzZWxmLWhvc3RlZCBydW5uZXJzLCBzbyB3ZSBjYW4ndCBsZWF2ZSBkZWFkIG9uZXMgYmVoaW5kLlxuICB0cnkge1xuICAgIGF3YWl0IG9jdG9raXQucmVzdC5hY3Rpb25zLmRlbGV0ZVNlbGZIb3N0ZWRSdW5uZXJGcm9tUmVwbyh7XG4gICAgICBvd25lcjogZXZlbnQub3duZXIsXG4gICAgICByZXBvOiBldmVudC5yZXBvLFxuICAgICAgcnVubmVyX2lkOiBydW5uZXJJZCxcbiAgICB9KTtcbiAgfSBjYXRjaCAoZSkge1xuICAgIGNvbnN0IHJlcUVycm9yID0gPFJlcXVlc3RFcnJvcj5lO1xuICAgIGlmIChyZXFFcnJvci5tZXNzYWdlLmluY2x1ZGVzKCdpcyBzdGlsbCBydW5uaW5nIGEgam9iJykpIHtcbiAgICAgIHRocm93IG5ldyBSdW5uZXJCdXN5KHJlcUVycm9yLm1lc3NhZ2UpO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aHJvdyBlO1xuICAgIH1cbiAgfVxufTtcbiJdfQ==