"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable-next-line import/no-extraneous-dependencies */
const AWS = require("aws-sdk");
const helpers_1 = require("./helpers");
const ec2 = new AWS.EC2();
async function deleteAmis(launchTemplateId, stackName, builderName, deleteAll) {
    // this runs daily and images are built once a week, so there shouldn't be a need for pagination
    const images = await ec2.describeImages({
        Owners: ['self'],
        Filters: [
            {
                Name: 'tag:GitHubRunners:Stack',
                Values: [stackName],
            },
            {
                Name: 'tag:GitHubRunners:Builder',
                Values: [builderName],
            },
        ],
    }).promise();
    let imagesToDelete = images.Images ?? [];
    console.log(`Found ${imagesToDelete.length} AMIs`);
    console.log(JSON.stringify(imagesToDelete.map(i => i.ImageId)));
    if (!deleteAll) {
        // get launch template information to filter out the active image
        const launchTemplates = await ec2.describeLaunchTemplateVersions({
            LaunchTemplateId: launchTemplateId,
            Versions: ['$Default'],
        }).promise();
        if (!launchTemplates.LaunchTemplateVersions) {
            console.error(`Unable to describe launch template ${launchTemplateId}`);
            return;
        }
        const launchTemplate = launchTemplates.LaunchTemplateVersions[0];
        // non-active images
        imagesToDelete = imagesToDelete.filter(i => i.ImageId != launchTemplate.LaunchTemplateData?.ImageId);
        // images older than two days to avoid race conditions where an image is created while we're cleaning up
        imagesToDelete = imagesToDelete.filter(i => i.CreationDate && Date.parse(i.CreationDate) < (Date.now() - 1000 * 60 * 60 * 48));
        console.log(`${imagesToDelete.length} AMIs left after filtering by date and excluding AMI used by launch template`);
    }
    // delete all that we found
    for (const image of imagesToDelete) {
        if (!image.ImageId) {
            console.warn(`No image id? ${JSON.stringify(image)}`);
            continue;
        }
        console.log(`Deregistering ${image.ImageId}`);
        await ec2.deregisterImage({
            ImageId: image.ImageId,
        }).promise();
        for (const blockMapping of image.BlockDeviceMappings ?? []) {
            if (blockMapping.Ebs?.SnapshotId) {
                console.log(`Deleting ${blockMapping.Ebs.SnapshotId}`);
                await ec2.deleteSnapshot({
                    SnapshotId: blockMapping.Ebs.SnapshotId,
                }).promise();
            }
        }
    }
}
/* eslint-disable @typescript-eslint/no-require-imports, import/no-extraneous-dependencies */
exports.handler = async function (event, context) {
    try {
        console.log(JSON.stringify({ ...event, ResponseURL: '...' }));
        switch (event.RequestType) {
            case 'Scheduled':
                await deleteAmis(event.LaunchTemplateId, event.StackName, event.BuilderName, false);
                return;
            case 'Create':
            case 'Update':
                await (0, helpers_1.customResourceRespond)(event, 'SUCCESS', 'OK', 'DeleteAmis', {});
                break;
            case 'Delete':
                await deleteAmis('', event.ResourceProperties.StackName, event.ResourceProperties.BuilderName, true);
                await (0, helpers_1.customResourceRespond)(event, 'SUCCESS', 'OK', event.PhysicalResourceId, {});
                break;
        }
    }
    catch (e) {
        console.error(e);
        if (event.RequestType != 'Scheduled') {
            await (0, helpers_1.customResourceRespond)(event, 'FAILED', e.message || 'Internal Error', context.logStreamName, {});
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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